/**
******************************************************************************
* @file    py32e407xC.h
* @brief   CMSIS Cortex-M4 Device Peripheral Access Layer Header File.
*          This file contains all the peripheral register's definitions, bits
*          definitions and memory mapping for PY32E4xx devices.
* @version v1.0.0
*
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

#ifndef __PY32E407xC_H
#define __PY32E407xC_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
* @brief Configuration of the Cortex-M4 Processor and Core Peripherals
*/
#define __CM4_REV                 0x0001U  /*!< Core revision r0p1                            */
#define __MPU_PRESENT             1U       /*!< PY32E4xx provides an MPU                      */
#define __NVIC_PRIO_BITS          3U       /*!< PY32E4xx uses 3 Bits for the Priority Levels  */
#define __Vendor_SysTickConfig    0U       /*!< Set to 1 if different SysTick Config is used  */
#define __FPU_PRESENT             1U       /*!< FPU present                                   */

/**
 * @brief PY32E4xx Interrupt Number Definition, according to the selected device
 *        in @ref Library_configuration_section
 */
typedef enum
{
    /******  Cortex-M4 Processor Exceptions Numbers **************************************************************/
    NonMaskableInt_IRQn         = -14,    /*!< 2 Non Maskable Interrupt                                          */
    HardFault_IRQn              = -13,    /*!< 3 Cortex-M4 Hard Fault Interrupt                                  */
    MemoryManagement_IRQn       = -12,    /*!< 4 Cortex-M4 Memory Management Interrupt                           */
    BusFault_IRQn               = -11,    /*!< 5 Cortex-M4 Bus Fault Interrupt                                   */
    UsageFault_IRQn             = -10,    /*!< 6 Cortex-M4 Usage Fault Interrupt                                 */
    SVCall_IRQn                 = -5,     /*!< 11 Cortex-M4 SV Call Interrupt                                    */
    DebugMonitor_IRQn           = -4,     /*!< 12 Cortex-M4 Debug Monitor Interrupt                              */
    PendSV_IRQn                 = -2,     /*!< 14 Cortex-M4 Pend SV Interrupt                                    */
    SysTick_IRQn                = -1,     /*!< 15 Cortex-M4 System Tick Interrupt                                */
    WWDG_IRQn                   = 0,      /*!< Window WatchDog Interrupt                                         */
    PVD_IRQn                    = 1,      /*!< PVD through EXTI Line detection Interrupt                         */
    TAMPER_IRQn                 = 2,      /*!< Tamper Interrupt                                                  */
    RTC_IRQn                    = 3,      /*!< RTC global Interrupt                                              */
    FLASH_IRQn                  = 4,      /*!< FLASH global Interrupt                                            */
    RCC_CTC_IRQn                = 5,      /*!< RCC and CTC global Interrupt                                      */
    EXTI0_IRQn                  = 6,      /*!< EXTI Line0 Interrupt                                              */
    EXTI1_IRQn                  = 7,      /*!< EXTI Line1 Interrupt                                              */
    EXTI2_IRQn                  = 8,      /*!< EXTI Line2 Interrupt                                              */
    EXTI3_IRQn                  = 9,      /*!< EXTI Line3 Interrupt                                              */
    EXTI4_IRQn                  = 10,     /*!< EXTI Line4 Interrupt                                              */
    DMA1_Channel1_IRQn          = 11,     /*!< DMA1 Channel 1 global Interrupt                                   */
    DMA1_Channel2_IRQn          = 12,     /*!< DMA1 Channel 2 global Interrupt                                   */
    DMA1_Channel3_IRQn          = 13,     /*!< DMA1 Channel 3 global Interrupt                                   */
    DMA1_Channel4_IRQn          = 14,     /*!< DMA1 Channel 4 global Interrupt                                   */
    DMA1_Channel5_IRQn          = 15,     /*!< DMA1 Channel 5 global Interrupt                                   */
    DMA1_Channel6_IRQn          = 16,     /*!< DMA1 Channel 6 global Interrupt                                   */
    DMA2_Channel6_IRQn          = 17,     /*!< DMA2 Channel 6 global Interrupt                                   */
    ADC1_2_IRQn                 = 18,     /*!< ADC1 and ADC2 global Interrupt                                    */
    OTG1_FS_WKUP_IRQn           = 19,     /*!< USB OTG1 FS Wakeup through EXTI line interrupt                    */
    CAN1_INT0_IRQn              = 20,     /*!< CAN1 Interrupt 0                                                  */
    OTG1_FS_IRQn                = 21,     /*!< USB OTG1 FS global Interrupt                                      */
    TIM18_IRQn                  = 22,     /*!< TIM18 global Interrupt                                            */
    EXTI9_5_IRQn                = 23,     /*!< External Line[9:5] Interrupts                                     */
    TIM1_BRK_TIM9_IRQn          = 24,     /*!< TIM1 Break Interrupt and TIM9 global Interrupt                    */
    TIM1_UP_TIM10_IRQn          = 25,     /*!< TIM1 Update Interrupt and TIM10 global Interrupt                  */
    TIM1_TRG_COM_TIM11_IRQn     = 26,     /*!< TIM1 Trigger and Commutation Interrupt and TIM11 global interrupt */
    TIM1_CC_IRQn                = 27,     /*!< TIM1 Capture Compare Interrupt                                    */
    TIM2_IRQn                   = 28,     /*!< TIM2 global Interrupt                                             */
    TIM3_IRQn                   = 29,     /*!< TIM3 global Interrupt                                             */
    TIM4_IRQn                   = 30,     /*!< TIM4 global Interrupt                                             */
    I2C1_EV_IRQn                = 31,     /*!< I2C1 Event Interrupt                                              */
    I2C1_ER_IRQn                = 32,     /*!< I2C1 Error Interrupt                                              */
    I2C2_EV_IRQn                = 33,     /*!< I2C2 Event Interrupt                                              */
    I2C2_ER_IRQn                = 34,     /*!< I2C2 Error Interrupt                                              */
    SPI1_IRQn                   = 35,     /*!< SPI1 global Interrupt                                             */
    SPI2_IRQn                   = 36,     /*!< SPI2 global Interrupt                                             */
    USART1_IRQn                 = 37,     /*!< USART1 global Interrupt                                           */
    USART2_IRQn                 = 38,     /*!< USART2 global Interrupt                                           */
    USART3_IRQn                 = 39,     /*!< USART3 global Interrupt                                           */
    EXTI15_10_IRQn              = 40,     /*!< External Line[15:10] Interrupts                                   */
    RTC_Alarm_IRQn              = 41,     /*!< RTC Alarm through EXTI Line Interrupt                             */
    TIM19_IRQn                  = 42,     /*!< TIM19 global Interrupt                                            */
    TIM8_BRK_TIM12_IRQn         = 43,     /*!< TIM8 Break Interrupt and TIM12 global Interrupt                   */
    TIM8_UP_TIM13_IRQn          = 44,     /*!< TIM8 Update Interrupt and TIM13 global Interrupt                  */
    TIM8_TRG_COM_TIM14_IRQn     = 45,     /*!< TIM8 Trigger and Commutation Interrupt and TIM14 global interrupt */
    TIM8_CC_IRQn                = 46,     /*!< TIM8 Capture Compare Interrupt                                    */
    ADC3_IRQn                   = 47,     /*!< ADC3 global Interrupt                                             */
    ESMC_IRQn                   = 48,     /*!< ESMC global Interrupt                                             */
    SDIO_IRQn                   = 49,     /*!< SDIO global Interrupt                                             */
    TIM5_IRQn                   = 50,     /*!< TIM5 global Interrupt                                             */
    SPI3_IRQn                   = 51,     /*!< SPI3 global Interrupt                                             */
    UART1_IRQn                  = 52,     /*!< UART1 global Interrupt                                            */
    LPUART_IRQn                 = 53,     /*!< LPUART global Interrupt                                           */
    TIM6_IRQn                   = 54,     /*!< TIM6 global Interrupt                                             */
    TIM7_IRQn                   = 55,     /*!< TIM7 global Interrupt                                             */
    DMA2_Channel1_IRQn          = 56,     /*!< DMA2 Channel 1 global Interrupt                                   */
    DMA2_Channel2_IRQn          = 57,     /*!< DMA2 Channel 2 global Interrupt                                   */
    DMA2_Channel3_IRQn          = 58,     /*!< DMA2 Channel 3 global Interrupt                                   */
    DMA2_Channel4_5_IRQn        = 59,     /*!< DMA2 Channel 4 and Channel 5 global Interrupt                     */
    TIM15_IRQn                  = 60,     /*!< TIM15 global Interrupt                                            */
    TIM16_IRQn                  = 61,     /*!< TIM16 global Interrupt                                            */
    TIM17_IRQn                  = 62,     /*!< TIM17 global Interrupt                                            */
    I2C3_EV_IRQn                = 63,     /*!< I2C3 Event Interrupt                                              */
    I2C3_ER_IRQn                = 64,     /*!< I2C3 Error Interrupt                                              */
    I2C4_EV_IRQn                = 65,     /*!< I2C4 Event Interrupt                                              */
    I2C4_ER_IRQn                = 66,     /*!< I2C4 Error Interrupt                                              */
    ETH_IRQn                    = 67,     /*!< Ethernet global Interrupt                                         */
    ETH_WKUP_IRQn               = 68,     /*!< Ethernet Wakeup through EXTI line Interrupt                       */
    RNG_IRQn                    = 69,     /*!< Rng global interrupt                                              */
    AES_IRQn                    = 70,     /*!< AES global interrupt                                              */
    CORDIC_IRQn                 = 71,     /*!< CORDIC global interrupt                                           */
    DAC_IRQn                    = 72,     /*!< DAC global Interrupts                                             */
    COMP1_IRQn                  = 73,     /*!< COMP1 global Interrupt                                            */
    COMP2_IRQn                  = 74,     /*!< COMP2 global Interrupt                                            */
    COMP3_IRQn                  = 75,     /*!< COMP3 global Interrupt                                            */
    COMP4_IRQn                  = 76,     /*!< COMP4 global Interrupt                                            */
    UART2_IRQn                  = 77,     /*!< UART2 global Interrupt                                            */
    UART3_IRQn                  = 78,     /*!< UART3 global Interrupt                                            */
    LPTIM_IRQn                  = 80,     /*!< LPTIM global Interrupts                                           */
    OTG2_FS_WKUP_IRQn           = 81,     /*!< USB OTG2 FS Wakeup through EXTI line interrupt                    */
    OTG2_FS_IRQn                = 82,     /*!< USB OTG2 FS global Interrupt                                      */
    CAN1_INT1_IRQn              = 83,     /*!< CAN1 Interrupt 1                                                  */
    CAN2_INT0_IRQn              = 84,     /*!< CAN2 Interrupt 0                                                  */
    CAN2_INT1_IRQn              = 85,     /*!< CAN2 Interrupt 1                                                  */
} IRQn_Type;

#include "core_cm4.h"             /* Cortex-M4 processor and core peripherals */
#include "system_py32e4xx.h"
#include <stdint.h>

/**
* @brief ADC Registers
  */
typedef struct
{
   __IO uint32_t ISR;              /*!< ADC ISR Register,                   Address offset: 0x00  */
   __IO uint32_t IER;              /*!< ADC IER Register,                   Address offset: 0x04  */
   __IO uint32_t CR;               /*!< ADC CR Register,                    Address offset: 0x08  */
   __IO uint32_t CFGR;             /*!< ADC CFGR Register,                  Address offset: 0x0C  */
   __IO uint32_t CFGR2;            /*!< ADC CFGR2 Register,                 Address offset: 0x10  */
   __IO uint32_t SMPR1;            /*!< ADC SMPR1 Register,                 Address offset: 0x14  */
   __IO uint32_t SMPR2;            /*!< ADC SMPR2 Register,                 Address offset: 0x18  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t TR1;              /*!< ADC TR1 Register,                   Address offset: 0x20  */
   __IO uint32_t TR2;              /*!< ADC TR2 Register,                   Address offset: 0x24  */
   __IO uint32_t TR3;              /*!< ADC TR3 Register,                   Address offset: 0x28  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t SQR1;             /*!< ADC SQR1 Register,                  Address offset: 0x30  */
   __IO uint32_t SQR2;             /*!< ADC SQR2 Register,                  Address offset: 0x34  */
   __IO uint32_t SQR3;             /*!< ADC SQR3 Register,                  Address offset: 0x38  */
   __IO uint32_t SQR4;             /*!< ADC SQR4 Register,                  Address offset: 0x3C  */
   __IO uint32_t DR;               /*!< ADC DR Register,                    Address offset: 0x40  */
   __IO uint32_t RESERVED3[2];      
   __IO uint32_t JSQR;             /*!< ADC JSQR Register,                  Address offset: 0x4C  */
   __IO uint32_t RESERVED4[4];      
   __IO uint32_t OFR1;             /*!< ADC OFR1 Register,                  Address offset: 0x60  */
   __IO uint32_t OFR2;             /*!< ADC OFR2 Register,                  Address offset: 0x64  */
   __IO uint32_t OFR3;             /*!< ADC OFR3 Register,                  Address offset: 0x68  */
   __IO uint32_t OFR4;             /*!< ADC OFR4 Register,                  Address offset: 0x6C  */
   __IO uint32_t RESERVED5[4];      
   __IO uint32_t JDR1;             /*!< ADC JDR1 Register,                  Address offset: 0x80  */
   __IO uint32_t JDR2;             /*!< ADC JDR2 Register,                  Address offset: 0x84  */
   __IO uint32_t JDR3;             /*!< ADC JDR3 Register,                  Address offset: 0x88  */
   __IO uint32_t JDR4;             /*!< ADC JDR4 Register,                  Address offset: 0x8C  */
   __IO uint32_t RESERVED6[4];      
   __IO uint32_t AWD2CR;           /*!< ADC AWD2CR Register,                Address offset: 0xA0  */
   __IO uint32_t AWD3CR;           /*!< ADC AWD3CR Register,                Address offset: 0xA4  */
   __IO uint32_t RESERVED7[2];      
   __IO uint32_t DIFSEL;           /*!< ADC DIFSEL Register,                Address offset: 0xB0  */
   __IO uint32_t CALFACT;          /*!< ADC CALFACT Register,               Address offset: 0xB4  */
   __IO uint32_t RESERVED8[2];      
   __IO uint32_t GCOMP;            /*!< ADC GCOMP Register,                 Address offset: 0xC0  */
} ADC_TypeDef;

typedef struct
{
  __IO uint32_t CCR;          /*!< ADC common configuration register,     Address offset: 0x300 + 0x08 */
} ADC_Common_TypeDef;

/**
* @brief AES Registers
*/
typedef struct
{
   __IO uint32_t MSG_CFG;          /*!< AES MSG_CFG Register,               Address offset: 0x00  */
   __IO uint32_t CTX_CFG;          /*!< AES CTX_CFG Register,               Address offset: 0x4  */
   __IO uint32_t MSG_TOTAL_BYTES;  /*!< AES MSG_TOTAL_BYTES Register,       Address offset: 0x8  */
   __IO uint32_t MSG_AAD_BYTES;    /*!< AES MSG_AAD_BYTES Register,         Address offset: 0xC  */
   __IO uint32_t GCM_AAD_INFO;     /*!< AES GCM_AAD_INFO Register,          Address offset: 0x10  */
   __IO uint32_t RESERVED1[3];      
   __IO uint32_t CTX_KEY[8];       /*!< AES CTX_KEY Register,               Address offset: 0x20  */
   __IO uint32_t CTX_CBC_KEY[4];   /*!< AES CTX_CBC_KEY Register,           Address offset: 0x40  */
   __IO uint32_t CTX_CTR[4];       /*!< AES CTX_CTR Register,               Address offset: 0x50  */
   __IO uint32_t CTX_IV[4];        /*!< AES CTX_IV Register,                Address offset: 0x60  */
   __IO uint32_t CTX_MAC[4];       /*!< AES CTX_MAC0 Register,              Address offset: 0x70  */
   __IO uint32_t INGRESS_FIFO;     /*!< AES INGRESS_FIFO Register,          Address offset: 0x80  */
   __IO uint32_t ING_DBCFG;        /*!< AES ING_DBCFG Register,             Address offset: 0x84  */
   __IO uint32_t DMA_ENG_LEN;      /*!< AES DMA_ENG_LEN Register,           Address offset: 0x88  */
   __IO uint32_t ENG_DBCFG;        /*!< AES ENG_DBCFG Register,             Address offset: 0x8C  */
   __IO uint32_t DMA_ING_LEN;      /*!< AES DMA_ING_LEN Register,           Address offset: 0x90  */
   __IO uint32_t INGF_STATUS;      /*!< AES INGF_STATUS Register,           Address offset: 0x94  */
   __IO uint32_t ENGRESS_FIFO;     /*!< AES ENGRESS_FIFO Register,          Address offset: 0x98  */
   __IO uint32_t ENGFIFO_STATUS;   /*!< AES ENGFIFO_STATUS Register,        Address offset: 0x9C  */
   __IO uint32_t ING_DMA_DONE;     /*!< AES ING_DMA_DONE Register,          Address offset: 0xA0  */
   __IO uint32_t ENG_DMA_DONE;     /*!< AES ENG_DMA_DONE Register,          Address offset: 0xA4  */
   __IO uint32_t DONE_STATUS;      /*!< AES DONE_STATUS Register,           Address offset: 0xA8  */
} AES_TypeDef;

/**
* @brief BKP Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< BKP CR Register,                    Address offset: 0x0  */
   __IO uint32_t CSR;              /*!< BKP CSR Register,                   Address offset: 0x4  */
   __IO uint32_t RTCCR;            /*!< BKP RTCCR Register,                 Address offset: 0x8  */
   __IO uint32_t RESERVED1[29];     
   __IO uint32_t DR0;              /*!< BKP DR0 Register,                   Address offset: 0x80  */
   __IO uint32_t DR1;              /*!< BKP DR1 Register,                   Address offset: 0x84  */
   __IO uint32_t DR2;              /*!< BKP DR2 Register,                   Address offset: 0x88  */
   __IO uint32_t DR3;              /*!< BKP DR3 Register,                   Address offset: 0x8C  */
   __IO uint32_t DR4;              /*!< BKP DR4 Register,                   Address offset: 0x90  */
   __IO uint32_t DR5;              /*!< BKP DR5 Register,                   Address offset: 0x94  */
   __IO uint32_t DR6;              /*!< BKP DR6 Register,                   Address offset: 0x98  */
   __IO uint32_t DR7;              /*!< BKP DR7 Register,                   Address offset: 0x9C  */
   __IO uint32_t DR8;              /*!< BKP DR8 Register,                   Address offset: 0xA0  */
   __IO uint32_t DR9;              /*!< BKP DR9 Register,                   Address offset: 0xA4  */
   __IO uint32_t DR10;             /*!< BKP DR10 Register,                  Address offset: 0xA8  */
   __IO uint32_t DR11;             /*!< BKP DR11 Register,                  Address offset: 0xAC  */
   __IO uint32_t DR12;             /*!< BKP DR12 Register,                  Address offset: 0xB0  */
   __IO uint32_t DR13;             /*!< BKP DR13 Register,                  Address offset: 0xB4  */
   __IO uint32_t DR14;             /*!< BKP DR14 Register,                  Address offset: 0xB8  */
   __IO uint32_t DR15;             /*!< BKP DR15 Register,                  Address offset: 0xBC  */
   __IO uint32_t DR16;             /*!< BKP DR16 Register,                  Address offset: 0xC0  */
   __IO uint32_t DR17;             /*!< BKP DR17 Register,                  Address offset: 0xC4  */
   __IO uint32_t DR18;             /*!< BKP DR18 Register,                  Address offset: 0xC8  */
   __IO uint32_t DR19;             /*!< BKP DR19 Register,                  Address offset: 0xCC  */
   __IO uint32_t DR20;             /*!< BKP DR20 Register,                  Address offset: 0xD0  */
   __IO uint32_t DR21;             /*!< BKP DR21 Register,                  Address offset: 0xD4  */
   __IO uint32_t DR22;             /*!< BKP DR22 Register,                  Address offset: 0xD8  */
   __IO uint32_t DR23;             /*!< BKP DR23 Register,                  Address offset: 0xDC  */
   __IO uint32_t DR24;             /*!< BKP DR24 Register,                  Address offset: 0xE0  */
   __IO uint32_t DR25;             /*!< BKP DR25 Register,                  Address offset: 0xE4  */
   __IO uint32_t DR26;             /*!< BKP DR26 Register,                  Address offset: 0xE8  */
   __IO uint32_t DR27;             /*!< BKP DR27 Register,                  Address offset: 0xEC  */
   __IO uint32_t DR28;             /*!< BKP DR28 Register,                  Address offset: 0xF0  */
   __IO uint32_t DR29;             /*!< BKP DR29 Register,                  Address offset: 0xF4  */
   __IO uint32_t DR30;             /*!< BKP DR30 Register,                  Address offset: 0xF8  */
   __IO uint32_t DR31;             /*!< BKP DR31 Register,                  Address offset: 0xFC  */
}BKP_TypeDef;

/**
* @brief FDCAN Registers
*/
typedef struct
{
   __IO uint32_t RESERVED1;          
   __IO uint32_t ENDN;             /*!< FDCAN ENDN Register,                Address offset: 0x04  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t DBTP;             /*!< FDCAN DBTP Register,                Address offset: 0x0C  */
   __IO uint32_t TEST;             /*!< FDCAN TEST Register,                Address offset: 0x010  */
   __IO uint32_t RWD;              /*!< FDCAN RWD Register,                 Address offset: 0x14  */
   __IO uint32_t CCCR;             /*!< FDCAN CCCR Register,                Address offset: 0x18  */
   __IO uint32_t NBTP;             /*!< FDCAN NBTP Register,                Address offset: 0x1C  */
   __IO uint32_t TSCC;             /*!< FDCAN TSCC Register,                Address offset: 0x20  */
   __IO uint32_t TSCV;             /*!< FDCAN TSCV Register,                Address offset: 0x24  */
   __IO uint32_t TOCC;             /*!< FDCAN TOCC Register,                Address offset: 0x28  */
   __IO uint32_t TOCV;             /*!< FDCAN TOCV Register,                Address offset: 0x2C  */
   __IO uint32_t RESERVED3[4];      
   __IO uint32_t ECR;              /*!< FDCAN ECR Register,                 Address offset: 0x40  */
   __IO uint32_t PSR;              /*!< FDCAN PSR Register,                 Address offset: 0x44  */
   __IO uint32_t TDCR;             /*!< FDCAN TDCR Register,                Address offset: 0x48  */
   __IO uint32_t RESERVED4;         
   __IO uint32_t IR;               /*!< FDCAN IR Register,                  Address offset: 0x50  */
   __IO uint32_t IE;               /*!< FDCAN IE Register,                  Address offset: 0x54  */
   __IO uint32_t ILS;              /*!< FDCAN ILS Register,                 Address offset: 0x58  */
   __IO uint32_t ILE;              /*!< FDCAN ILE Register,                 Address offset: 0x5C  */
   __IO uint32_t RESERVED5[8];      
   __IO uint32_t GFC;              /*!< FDCAN GFC Register,                 Address offset: 0x80  */
   __IO uint32_t SIDFC;            /*!< FDCAN SIDFC Register,               Address offset: 0x84  */
   __IO uint32_t XIDFC;            /*!< FDCAN XIDFC Register,               Address offset: 0x88  */
   __IO uint32_t RESERVED6;         
   __IO uint32_t XIDAM;            /*!< FDCAN XIDAM Register,               Address offset: 0x90  */
   __IO uint32_t HPMS;             /*!< FDCAN HPMS Register,                Address offset: 0x94  */
   __IO uint32_t NDAT1;            /*!< FDCAN NDAT1 Register,               Address offset: 0x98  */
   __IO uint32_t NDAT2;            /*!< FDCAN NDAT2 Register,               Address offset: 0x9C  */
   __IO uint32_t RXF0C;            /*!< FDCAN RXF0C Register,               Address offset: 0xA0  */
   __IO uint32_t RXF0S;            /*!< FDCAN RXF0S Register,               Address offset: 0xA4  */
   __IO uint32_t RXF0A;            /*!< FDCAN RXF0A Register,               Address offset: 0xA8  */
   __IO uint32_t RXBC;             /*!< FDCAN RXBC Register,                Address offset: 0xAC  */
   __IO uint32_t RXF1C;            /*!< FDCAN RXF1C Register,               Address offset: 0xB0  */
   __IO uint32_t RXF1S;            /*!< FDCAN RXF1S Register,               Address offset: 0xB4  */
   __IO uint32_t RXF1A;            /*!< FDCAN RXF1A Register,               Address offset: 0xB8  */
   __IO uint32_t RXESC;            /*!< FDCAN RXESC Register,               Address offset: 0xBC  */
   __IO uint32_t TXBC;             /*!< FDCAN TXBC Register,                Address offset: 0xC0  */
   __IO uint32_t TXFQS;            /*!< FDCAN TXFQS Register,               Address offset: 0xC4  */
   __IO uint32_t TXESC;            /*!< FDCAN TXESC Register,               Address offset: 0xC8  */
   __IO uint32_t TXBRP;            /*!< FDCAN TXBRP Register,               Address offset: 0xCC  */
   __IO uint32_t TXBAR;            /*!< FDCAN TXBAR Register,               Address offset: 0xD0  */
   __IO uint32_t TXBCR;            /*!< FDCAN TXBCR Register,               Address offset: 0xD4  */
   __IO uint32_t TXBTO;            /*!< FDCAN TXBTO Register,               Address offset: 0xD8  */
   __IO uint32_t TXBCF;            /*!< FDCAN TXBCF Register,               Address offset: 0xDC  */
   __IO uint32_t TXBTIE;           /*!< FDCAN TXBTIE Register,              Address offset: 0xE0  */
   __IO uint32_t TXBCIE;           /*!< FDCAN TXBCIE Register,              Address offset: 0xE4  */
   __IO uint32_t RESERVED7[2];      
   __IO uint32_t TXEFC;            /*!< FDCAN TXEFC Register,               Address offset: 0xF0  */
   __IO uint32_t TXEFS;            /*!< FDCAN TXEFS Register,               Address offset: 0xF4  */
   __IO uint32_t TXEFA;            /*!< FDCAN TXEFA Register,               Address offset: 0xF8  */
} FDCAN_GlobalTypeDef;



/**
  * @brief Comparator
  */
typedef struct
{
  __IO uint32_t CSR;              /*!< COMP control and status register,           Address offset: 0x00 */
  __IO uint32_t FR;               /*!< COMP filter register,                       Address offset: 0x04 */
} COMP_TypeDef;

typedef struct
{
  __IO uint32_t CSR_ODD;       /*!< COMP control and status register located in register of comparator instance 1, used for bits common to several COMP instances,  Address offset: 0x00 */
  __IO uint32_t FR_ODD;
  uint32_t RESERVED0[2];        /*Reserved*/
  __IO uint32_t CSR_EVEN;      /*!< COMP control and status register located in register of comparator instance 2, used for bits common to several COMP instances, Address offset: 0x010 */
  __IO uint32_t FR_EVEN;
} COMP_Common_TypeDef;

/**
* @brief CRC Registers
*/
typedef struct
{
   __IO uint32_t DR;               /*!< CRC DR Register,                    Address offset: 0x00  */
   __IO uint32_t IDR;              /*!< CRC IDR Register,                   Address offset: 0x04  */
   __IO uint32_t CR;               /*!< CRC CR Register,                    Address offset: 0x08  */
   __IO uint32_t RESERVED1;          
   __IO uint32_t INIT;             /*!< CRC INIT Register,                  Address offset: 0x10  */
   __IO uint32_t POL;              /*!< CRC POL Register,                   Address offset: 0x14  */
} CRC_TypeDef;


/**
* @brief CTC Registers
*/
typedef struct
{
   __IO uint32_t CTL0;             /*!< CTC CTL0 Register,                  Address offset: 0x00  */
   __IO uint32_t CTL1;             /*!< CTC CTL1 Register,                  Address offset: 0x04  */
   __IO uint32_t SR;               /*!< CTC SR Register,                    Address offset: 0x08  */
   __IO uint32_t INTC;             /*!< CTC INTC Register,                  Address offset: 0x0C  */
}CTC_TypeDef;

/**
* @brief CORDIC Registers
*/
typedef struct
{
   __IO uint32_t CSR;              /*!< CORDIC CSR Register,                Address offset: 0x00  */
   __IO uint32_t WDATA;            /*!< CORDIC WDATA Register,              Address offset: 0x04  */
   __IO uint32_t RDATA;            /*!< CORDIC RDATA Register,              Address offset: 0x08  */
} CORDIC_TypeDef;

/**
* @brief DAC Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< DAC CR Register,                    Address offset: 0x00  */
   __IO uint32_t SWTRIGR;          /*!< DAC SWTRIGR Register,               Address offset: 0x04  */
   __IO uint32_t DHR12R1;          /*!< DAC DHR12R1 Register,               Address offset: 0x08  */
   __IO uint32_t DHR12L1;          /*!< DAC DHR12L1 Register,               Address offset: 0x0C  */
   __IO uint32_t DHR8R1;           /*!< DAC DHR8R1 Register,                Address offset: 0x10  */
   __IO uint32_t DHR12R2;          /*!< DAC DHR12R2 Register,               Address offset: 0x14  */
   __IO uint32_t DHR12L2;          /*!< DAC DHR12L2 Register,               Address offset: 0x18  */
   __IO uint32_t DHR8R2;           /*!< DAC DHR8R2 Register,                Address offset: 0x1C  */
   __IO uint32_t DHR12RD;          /*!< DAC DHR12RD Register,               Address offset: 0x20  */
   __IO uint32_t DHR12LD;          /*!< DAC DHR12LD Register,               Address offset: 0x24  */
   __IO uint32_t DHR8RD;           /*!< DAC DHR8RD Register,                Address offset: 0x28  */
   __IO uint32_t DOR1;             /*!< DAC DOR1 Register,                  Address offset: 0x2C  */
   __IO uint32_t DOR2;             /*!< DAC DOR2 Register,                  Address offset: 0x30  */
   __IO uint32_t SR;               /*!< DAC SR Register,                    Address offset: 0x34  */
   __IO uint32_t OC;               /*!< DAC OC Register,                    Address offset: 0x38  */
} DAC_TypeDef;


/**
* @brief DBGMCU Registers
*/
typedef struct
{
    __IO uint32_t IDCODE;      /*!< DBGMCU desc IDCODE, Address offset: 0x00 */
    __IO uint32_t CR1;         /*!< DBGMCU desc CR1,    Address offset: 0x04 */
    __IO uint32_t CR2;         /*!< DBGMCU desc CR2,    Address offset: 0x04 */
}DBGMCU_TypeDef;

/**
* @brief DMA Controller Registers
*/
typedef struct
{
   __IO uint32_t EN;               /*!< DMA EN Register,                    Address offset: 0x00  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t CH_EN;            /*!< DMA CH_EN Register,                 Address offset: 0x08  */
   __IO uint32_t RESERVED2[167];    
   __IO uint32_t STATUSHALFBLOCK;  /*!< DMA STATUSHALFBLOCK Register,       Address offset: 0x2A8  */
   __IO uint32_t RESERVED3;        
   __IO uint32_t MASKHALFBLOCK;    /*!< DMA MASKHALFBLOCK Register,         Address offset: 0x2B0  */
   __IO uint32_t RESERVED4;        
   __IO uint32_t CLEARHALFBLOCK;   /*!< DMA CLEARHALFBLOCK Register,        Address offset: 0x2B8  */
   __IO uint32_t RESERVED5[11];
   __IO uint32_t STATUSTFR;        /*!< DMA STATUSTFR Register,             Address offset: 0x2E8  */
   __IO uint32_t RESERVED6;        
   __IO uint32_t STATUSBLOCK;      /*!< DMA STATUSBLOCK Register,           Address offset: 0x2F0  */
   __IO uint32_t RESERVED7;        
   __IO uint32_t STATUSSRCTRAN;    /*!< DMA STATUSSRCTRAN Register,         Address offset: 0x2F8  */
   __IO uint32_t RESERVED8;        
   __IO uint32_t STATUSDSTTRAN;    /*!< DMA STATUSDSTTRAN Register,         Address offset: 0x300  */
   __IO uint32_t RESERVED9;        
   __IO uint32_t STATUSERR;        /*!< DMA STATUSERR Register,             Address offset: 0x308  */
   __IO uint32_t RESERVED10;        
   __IO uint32_t MASKTFR;          /*!< DMA MASKTFR Register,               Address offset: 0x310  */
   __IO uint32_t RESERVED11;        
   __IO uint32_t MASKBLOCK;        /*!< DMA MASKBLOCK Register,             Address offset: 0x318  */
   __IO uint32_t RESERVED12;        
   __IO uint32_t MASKSRCTRAN;      /*!< DMA MASKSRCTRAN Register,           Address offset: 0x320  */
   __IO uint32_t RESERVED13;        
   __IO uint32_t MASKDSTTRAN;      /*!< DMA MASKDSTTRAN Register,           Address offset: 0x328  */
   __IO uint32_t RESERVED14;        
   __IO uint32_t MASKERR;          /*!< DMA MASKERR Register,               Address offset: 0x330  */
   __IO uint32_t RESERVED15;        
   __IO uint32_t CLEARTFR;         /*!< DMA CLEARTFR Register,              Address offset: 0x338  */
   __IO uint32_t RESERVED16;        
   __IO uint32_t CLEARBLOCK;       /*!< DMA CLEARBLOCK Register,            Address offset: 0x340  */
   __IO uint32_t RESERVED17;        
   __IO uint32_t CLEARSRCTRAN;     /*!< DMA CLEARSRCTRAN Register,          Address offset: 0x348  */
   __IO uint32_t RESERVED18;        
   __IO uint32_t CLEARDSTTRAN;     /*!< DMA CLEARDSTTRAN Register,          Address offset: 0x350  */
   __IO uint32_t RESERVED19;        
   __IO uint32_t CLEARERR;         /*!< DMA CLEARERR Register,              Address offset: 0x358  */
   __IO uint32_t RESERVED20;        
   __IO uint32_t STATUSINT;        /*!< DMA STATUSINT Register,             Address offset: 0x360  */
   __IO uint32_t RESERVED21;        
   __IO uint32_t REQSRC;           /*!< DMA REQSRC Register,                Address offset: 0x368  */
   __IO uint32_t RESERVED22;        
   __IO uint32_t REQDST;           /*!< DMA REQDST Register,                Address offset: 0x370  */
   __IO uint32_t RESERVED23;        
   __IO uint32_t SGLRQSRC;         /*!< DMA SGLRQSRC Register,              Address offset: 0x378  */
   __IO uint32_t RESERVED24;        
   __IO uint32_t SGLRQDST;         /*!< DMA SGLRQDST Register,              Address offset: 0x380  */
} DMA_TypeDef;

typedef struct
{
   __IO uint32_t CH_CTRLL;        /*!< DMA CH_CTRLLx Register,             Address offset: 0x10  */
   __IO uint32_t CH_CTRLH;        /*!< DMA CH_CTRLHx Register,             Address offset: 0x14  */
   __IO uint32_t CH_CFGL;         /*!< DMA CH_CFGLx Register,              Address offset: 0x18  */
   __IO uint32_t CH_CFGH;         /*!< DMA CH_CFGHx Register,              Address offset: 0x1C  */
   __IO uint32_t SA;              /*!< DMA SAx Register,                   Address offset: 0x20  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t DA;              /*!< DMA DAx Register,                   Address offset: 0x28  */
} DMA_Channel_TypeDef;

/**
* @brief ETH Registers
*/

typedef struct
{
   __IO uint32_t MACCR;            /*!< ETH MACCR Register,                 Address offset: 0x0000  */
   __IO uint32_t MACFFR;           /*!< ETH MACFFR Register,                Address offset: 0x0004  */
   __IO uint32_t MACHTHR;          /*!< ETH MACHTHR Register,               Address offset: 0x0008  */
   __IO uint32_t MACHTLR;          /*!< ETH MACHTLR Register,               Address offset: 0x000C  */
   __IO uint32_t MACMIIAR;         /*!< ETH MACMIIAR Register,              Address offset: 0x0010  */
   __IO uint32_t MACMIIDR;         /*!< ETH MACMIIDR Register,              Address offset: 0x0014  */
   __IO uint32_t MACFCR;           /*!< ETH MACFCR Register,                Address offset: 0x0018  */
   __IO uint32_t MACVLANTR;        /*!< ETH MACVLANTR Register,             Address offset: 0x001C  */
   __IO uint32_t RESERVED1[2];      
   __IO uint32_t MACRWUFFR;        /*!< ETH MACRWUFFR Register,             Address offset: 0x0028  */
   __IO uint32_t MACPMTCSR;        /*!< ETH MACPMTCSR Register,             Address offset: 0x002C  */
   __IO uint32_t RESERVED2[2];      
   __IO uint32_t MACSR;            /*!< ETH MACSR Register,                 Address offset: 0x0038  */
   __IO uint32_t MACIMR;           /*!< ETH MACIMR Register,                Address offset: 0x003C  */     
   __IO uint32_t MACA0HR;          /*!< ETH MACA0HR Register,               Address offset: 0x0040  */
   __IO uint32_t MACA0LR;          /*!< ETH MACA0LR Register,               Address offset: 0x0044  */
   __IO uint32_t MACA1HR;          /*!< ETH MACA1HR Register,               Address offset: 0x0048  */
   __IO uint32_t MACA1LR;          /*!< ETH MACA1LR Register,               Address offset: 0x004C  */
   __IO uint32_t MACA2HR;          /*!< ETH MACA2HR Register,               Address offset: 0x0050  */
   __IO uint32_t MACA2LR;          /*!< ETH MACA2LR Register,               Address offset: 0x0054  */
   __IO uint32_t MACA3HR;          /*!< ETH MACA3HR Register,               Address offset: 0x0058  */
   __IO uint32_t MACA3LR;          /*!< ETH MACA3LR Register,               Address offset: 0x005C  */
   __IO uint32_t RESERVED5[40];     
   __IO uint32_t MMCCR;            /*!< ETH MMCCR Register,                 Address offset: 0x0100  */
   __IO uint32_t MMCRIR;           /*!< ETH MMCRIR Register,                Address offset: 0x0104  */
   __IO uint32_t MMCTIR;           /*!< ETH MMCTIR Register,                Address offset: 0x0108  */
   __IO uint32_t MMCRIMR;          /*!< ETH MMCRIMR Register,               Address offset: 0x010C  */
   __IO uint32_t MMCTIMR;          /*!< ETH MMCTIMR Register,               Address offset: 0x0110  */
   __IO uint32_t RESERVED6[14];     
   __IO uint32_t MMCTGFSCCR;       /*!< ETH MMCTGFSCCR Register,            Address offset: 0x014C  */
   __IO uint32_t MMCTGFMSCCR;      /*!< ETH MMCTGFMSCCR Register,           Address offset: 0x0150  */
   __IO uint32_t RESERVED7[5];      
   __IO uint32_t MMCTGFCR;         /*!< ETH MMCTGFCR Register,              Address offset: 0x0168  */
   __IO uint32_t RESERVED8[10];     
   __IO uint32_t MMCRFCECR;        /*!< ETH MMCRFCECR Register,             Address offset: 0x0194  */
   __IO uint32_t MMCRFAECR;        /*!< ETH MMCRFAECR Register,             Address offset: 0x0198  */
   __IO uint32_t RESERVED9[10];     
   __IO uint32_t MMCRGUFCR;        /*!< ETH MMCRGUFCR Register,             Address offset: 0x01C4  */
   __IO uint32_t RESERVED10[334];   
   __IO uint32_t PTPTSCR;          /*!< ETH PTPTSCR Register,               Address offset: 0x0700  */
   __IO uint32_t PTPSSIR;          /*!< ETH PTPSSIR Register,               Address offset: 0x0704  */
   __IO uint32_t PTPTSHR;          /*!< ETH PTPTSHR Register,               Address offset: 0x0708  */
   __IO uint32_t PTPTSLR;          /*!< ETH PTPTSLR Register,               Address offset: 0x070C  */
   __IO uint32_t PTPTSHUR;         /*!< ETH PTPTSHUR Register,              Address offset: 0x0710  */
   __IO uint32_t PTPTSLUR;         /*!< ETH PTPTSLUR Register,              Address offset: 0x0714  */
   __IO uint32_t PTPTSAR;          /*!< ETH PTPTSAR Register,               Address offset: 0x0718  */
   __IO uint32_t PTPTTHR;          /*!< ETH PTPTTHR Register,               Address offset: 0x071C  */
   __IO uint32_t PTPTTLR;          /*!< ETH PTPTTLR Register,               Address offset: 0x0720  */
   __IO uint32_t RESERVED11[567];   
   __IO uint32_t DMABMR;           /*!< ETH DMABMR Register,                Address offset: 0x1000  */
   __IO uint32_t DMATPDR;          /*!< ETH DMATPDR Register,               Address offset: 0x1004  */
   __IO uint32_t DMARPDR;          /*!< ETH DMARPDR Register,               Address offset: 0x1008  */
   __IO uint32_t DMARDLAR;         /*!< ETH DMARDLAR Register,              Address offset: 0x100C  */
   __IO uint32_t DMATDLAR;         /*!< ETH DMATDLAR Register,              Address offset: 0x1010  */
   __IO uint32_t DMASR;            /*!< ETH DMASR Register,                 Address offset: 0x1014  */
   __IO uint32_t DMAOMR;           /*!< ETH DMAOMR Register,                Address offset: 0x1018  */
   __IO uint32_t DMAIER;           /*!< ETH DMAIER Register,                Address offset: 0x101C  */
   __IO uint32_t DMAMFBOCR;        /*!< ETH DMAMFBOCR Register,             Address offset: 0x1020  */
   __IO uint32_t DMARIWT;          /*!< ETH DMARIWT Register,               Address offset: 0x1024  */
   __IO uint32_t RESERVED12[8];     
   __IO uint32_t DMACHTDR;         /*!< ETH DMACHTDR Register,              Address offset: 0x1048  */
   __IO uint32_t DMACHRDR;         /*!< ETH DMACHRDR Register,              Address offset: 0x104C  */
   __IO uint32_t DMACHTBAR;        /*!< ETH DMACHTBAR Register,             Address offset: 0x1050  */
   __IO uint32_t DMACHRBAR;        /*!< ETH DMACHRBAR Register,             Address offset: 0x1054  */
} ETH_TypeDef;

/**
* @brief EXTI Registers
*/
typedef struct
{
   __IO uint32_t IMR1;             /*!< EXTI IMR1 Register,                 Address offset: 0x00  */
   __IO uint32_t EMR1;             /*!< EXTI EMR1 Register,                 Address offset: 0x04  */
   __IO uint32_t RTSR;             /*!< EXTI RTSR Register,                 Address offset: 0x08  */
   __IO uint32_t FTSR;             /*!< EXTI FTSR Register,                 Address offset: 0x0C  */
   __IO uint32_t SWIER;            /*!< EXTI SWIER Register,                Address offset: 0x10  */
   __IO uint32_t PR;               /*!< EXTI PR Register,                   Address offset: 0x14  */
   __IO uint32_t IMR2;             /*!< EXTI IMR2 Register,                 Address offset: 0x18  */
   __IO uint32_t EMR2;             /*!< EXTI EMR2 Register,                 Address offset: 0x1C  */
} EXTI_TypeDef;

/**
* @brief FMC Registers
*/
typedef struct
{
   __IO uint32_t ACR;              /*!< FLASH ACR Register,                 Address offset: 0x00  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t KEYR;             /*!< FLASH KEYR Register,                Address offset: 0x08  */
   __IO uint32_t OPTKEYR;          /*!< FLASH OPTKEYR Register,             Address offset: 0x0C  */
   __IO uint32_t SR;               /*!< FLASH SR Register,                  Address offset: 0x10  */
   __IO uint32_t CR;               /*!< FLASH CR Register,                  Address offset: 0x14  */
   __IO uint32_t ECCR;             /*!< FLASH ECCR Register,                Address offset: 0x18  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t OPTR1;            /*!< FLASH OPTR1 Register,               Address offset: 0x20  */
   __IO uint32_t OPTR2;            /*!< FLASH OPTR2 Register,               Address offset: 0x24  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t WRPR;             /*!< FLASH WRPR Register,                Address offset: 0x2C  */
   __IO uint32_t PCROP0SR;         /*!< FLASH PCROP0SR Register,            Address offset: 0x30  */
   __IO uint32_t PCROP0ER;         /*!< FLASH PCROP0ER Register,            Address offset: 0x34  */
   __IO uint32_t PCROP1SR;         /*!< FLASH PCROP1SR Register,            Address offset: 0x38  */
   __IO uint32_t PCROP1ER;         /*!< FLASH PCROP1ER Register,            Address offset: 0x3C  */
} FLASH_TypeDef;




/**
* @brief General Purpose I/O
*/
typedef struct
{
    __IO uint32_t MODER;       /*!< GPIO port mode register,               Address offset: 0x00      */
    __IO uint32_t OTYPER;      /*!< GPIO port output type register,        Address offset: 0x04      */
    __IO uint32_t OSPEEDR;     /*!< GPIO port output speed register,       Address offset: 0x08      */
    __IO uint32_t PUPDR;       /*!< GPIO port pull-up/pull-down register,  Address offset: 0x0C      */
    __IO uint32_t IDR;         /*!< GPIO port input data register,         Address offset: 0x10      */
    __IO uint32_t ODR;         /*!< GPIO port output data register,        Address offset: 0x14      */
    __IO uint32_t BSRR;        /*!< GPIO port bit set/reset  register,     Address offset: 0x18      */
    __IO uint32_t LCKR;        /*!< GPIO port configuration lock register, Address offset: 0x1C      */
    __IO uint32_t AFR[2];      /*!< GPIO alternate function registers,     Address offset: 0x20-0x24 */
    __IO uint32_t BRR;         /*!< GPIO Bit Reset register,               Address offset: 0x28      */
} GPIO_TypeDef;

/**
* @brief I2C Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< I2C CR1 Register,                   Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< I2C CR2 Register,                   Address offset: 0x04  */
   __IO uint32_t OAR1;             /*!< I2C OAR1 Register,                  Address offset: 0x08  */
   __IO uint32_t OAR2;             /*!< I2C OAR2 Register,                  Address offset: 0x0C  */
   __IO uint32_t DR;               /*!< I2C DR Register,                    Address offset: 0x10  */
   __IO uint32_t SR1;              /*!< I2C SR1 Register,                   Address offset: 0x14  */
   __IO uint32_t SR2;              /*!< I2C SR2 Register,                   Address offset: 0x18  */
   __IO uint32_t CCR;              /*!< I2C CCR Register,                   Address offset: 0x1C  */
   __IO uint32_t TRISE;            /*!< I2C TRISE Register,                 Address offset: 0x20  */
   __IO uint32_t TIMEOUTR;         /*!< I2C TIMEOUTR Register,              Address offset: 0x24  */
} I2C_TypeDef;
/**
* @brief IWDG Registers
*/
typedef struct
{
   __IO uint32_t KR;               /*!< IWDG KR Register,                   Address offset: 0x0  */
   __IO uint32_t PR;               /*!< IWDG PR Register,                   Address offset: 0x4  */
   __IO uint32_t RLR;              /*!< IWDG RLR Register,                  Address offset: 0x8  */
   __IO uint32_t SR;               /*!< IWDG SR Register,                   Address offset: 0xC  */
}IWDG_TypeDef;

/**
  * @brief LCDC registers
  */
typedef struct
{
  __IO uint32_t DR;          /*!< LCD desc DR;  , Address offset: 0x04 */
  __IO uint32_t CMDR;        /*!< LCD desc CMDR;, Address offset: 0x08 */
  __IO uint32_t WCFGR;       /*!< LCD desc WCFGR, Address offset: 0x0C */
  __IO uint32_t RCFGR;       /*!< LCD desc RCFGR, Address offset: 0x10 */
} LCDC_TypeDef;

typedef struct
{
  __IO uint32_t CSR;          /*!< LCDC common configuration and status register,   Address offset: 0x00 */
} LCDC_Common_TypeDef;

/**
* @brief LPTIM Registers
*/
typedef struct
{
   __IO uint32_t ISR;              /*!< LPTIM ISR Register,                 Address offset: 0x00  */
   __IO uint32_t ICR;              /*!< LPTIM ICR Register,                 Address offset: 0x04  */
   __IO uint32_t IER;              /*!< LPTIM IER Register,                 Address offset: 0x08  */
   __IO uint32_t CFGR;             /*!< LPTIM CFGR Register,                Address offset: 0x0C  */
   __IO uint32_t CR;               /*!< LPTIM CR Register,                  Address offset: 0x10  */
   __IO uint32_t CMP;              /*!< LPTIM CMP Register,                 Address offset: 0x14  */
   __IO uint32_t ARR;              /*!< LPTIM ARR Register,                 Address offset: 0x18  */
   __IO uint32_t CNT;              /*!< LPTIM CNT Register,                 Address offset: 0x1C  */
   __IO uint32_t OR;               /*!< LPTIM OR Register,                  Address offset: 0x20  */
} LPTIM_TypeDef;


/**
  * @brief Comparator
  */
typedef struct
{
  __IO uint32_t CSR;            /*!< OPA output enable register,            Address offset: 0x00 */
  __IO uint32_t RESERVED[2];    
  __IO uint32_t TCMR;           /*!< OPA control register,                  Address offset: 0x0C */
} OPA_TypeDef;

/**
* @brief PWR Registers
*/
typedef struct
{
    __IO uint32_t CR1;         /*!< PWR desc CR1, Address offset: 0x00 */
    __IO uint32_t CR2;         /*!< PWR desc CR2, Address offset: 0x04 */
    __IO uint32_t CR3;         /*!< PWR desc CR3, Address offset: 0x08 */
    __IO uint32_t CR4;         /*!< PWR desc CR4, Address offset: 0x0C */
    __IO uint32_t SR;          /*!< PWR desc SR,  Address offset: 0x10 */
    __IO uint32_t SCR;         /*!< PWR desc SCR, Address offset: 0x14 */
}PWR_TypeDef;

/**
* @brief ESMC Registers
*/
typedef struct
{
   __IO uint8_t CR;                /*!< ESMC CR Register,                   Address offset: 0x0  */
   __IO uint8_t CR2;               /*!< ESMC CR2 Register,                  Address offset: 0x1  */
   __IO uint8_t TCR;               /*!< ESMC TCR Register,                  Address offset: 0x2  */
   __IO uint8_t BAUD;              /*!< ESMC BAUD Register,                 Address offset: 0x3  */
   __IO uint8_t SFCR;              /*!< ESMC SFCR Register,                 Address offset: 0x4  */
   __IO uint8_t SOCR;              /*!< ESMC SOCR Register,                 Address offset: 0x5  */
   __IO uint8_t DCR;               /*!< ESMC DCR Register,                  Address offset: 0x6  */
   __IO uint8_t CR3;               /*!< ESMC CR3 Register,                  Address offset: 0x7  */
   __IO uint32_t ADDR24;           /*!< ESMC ADDR24 Register,               Address offset: 0x8  */
   __IO uint32_t ADDR32;           /*!< ESMC ADDR32 Register,               Address offset: 0xC  */
   __IO uint32_t DATA;             /*!< ESMC DATA Register,                 Address offset: 0x10  */
   __IO uint8_t SR;                /*!< ESMC SR Register,                   Address offset: 0x14  */
   __IO uint8_t SR1;               /*!< ESMC SR1 Register,                  Address offset: 0x15  */
   __IO uint8_t RXSTAT;            /*!< ESMC RXSTAT Register,               Address offset: 0x16  */
   __IO uint8_t RESERVED1;          
   __IO uint8_t IFR1;              /*!< ESMC IFR1 Register,                 Address offset: 0x18  */
   __IO uint8_t IER1;              /*!< ESMC IER1 Register,                 Address offset: 0x19  */
   __IO uint8_t IFR2;              /*!< ESMC IFR2 Register,                 Address offset: 0x1A  */
   __IO uint8_t IER2;              /*!< ESMC IER2 Register,                 Address offset: 0x1B  */
   __IO uint8_t STRR;              /*!< ESMC STRR Register,                 Address offset: 0x1C  */
   __IO uint8_t RESERVED2[3];       
   __IO uint8_t XSFCR;             /*!< ESMC XSFCR Register,                Address offset: 0x20  */
   __IO uint8_t XSOCR;             /*!< ESMC XSOCR Register,                Address offset: 0x21  */
   __IO uint8_t XDCR;              /*!< ESMC XDCR Register,                 Address offset: 0x22  */
   __IO uint8_t XCR3;              /*!< ESMC XCR3 Register,                 Address offset: 0x23  */
   __IO uint8_t XMODE;             /*!< ESMC XMODE Register,                Address offset: 0x24  */
   __IO uint8_t XMODE_WE;          /*!< ESMC XMODE_WE Register,             Address offset: 0x25  */
   __IO uint8_t XSSOCR;            /*!< ESMC XSSOCR Register,               Address offset: 0x26  */
   __IO uint8_t XSSOCR_WE;         /*!< ESMC XSSOCR_WE Register,            Address offset: 0x27  */
   __IO uint8_t XSFCR_WE;          /*!< ESMC XSFCR_WE Register,             Address offset: 0x28  */
   __IO uint8_t XSOCR_WE;          /*!< ESMC XSOCR_WE Register,             Address offset: 0x29  */
   __IO uint8_t XDCR_WE;           /*!< ESMC XDCR_WE Register,              Address offset: 0x2A  */
   __IO uint8_t XCR3_WE;           /*!< ESMC XCR3_WE Register,              Address offset: 0x2B  */
   __IO uint32_t BCR;              /*!< ESMC BCR Register,                  Address offset: 0x2C  */
   __IO uint8_t RXTOUT;            /*!< ESMC RXTOUT Register,               Address offset: 0x30  */
   __IO uint8_t XSTRR;             /*!< ESMC XSTRR Register,                Address offset: 0x31  */
} ESMC_TypeDef;

/**
* @brief RCC Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< RCC CR Register,                    Address offset: 0x00  */
   __IO uint32_t CFGR;             /*!< RCC CFGR Register,                  Address offset: 0x04  */
   __IO uint32_t CIR;              /*!< RCC CIR Register,                   Address offset: 0x08  */
   __IO uint32_t RESERVED1[7];      
   __IO uint32_t AHB1RSTR;         /*!< RCC AHB1RSTR Register,              Address offset: 0x28  */
   __IO uint32_t AHB2RSTR;         /*!< RCC AHB2RSTR Register,              Address offset: 0x2C  */
   __IO uint32_t RESERVED2[2];      
   __IO uint32_t APB1RSTR1;        /*!< RCC APB1RSTR1 Register,             Address offset: 0x38  */
   __IO uint32_t APB1RSTR2;        /*!< RCC APB1RSTR2 Register,             Address offset: 0x3C  */
   __IO uint32_t APB2RSTR;         /*!< RCC APB2RSTR Register,              Address offset: 0x40  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t AHB1ENR;          /*!< RCC AHB1ENR Register,               Address offset: 0x48  */
   __IO uint32_t AHB2ENR;          /*!< RCC AHB2ENR Register,               Address offset: 0x4C  */
   __IO uint32_t RESERVED4[2];      
   __IO uint32_t APB1ENR1;         /*!< RCC APB1ENR1 Register,              Address offset: 0x58  */
   __IO uint32_t APB1ENR2;         /*!< RCC APB1ENR2 Register,              Address offset: 0x5C  */
   __IO uint32_t APB2ENR;          /*!< RCC APB2ENR Register,               Address offset: 0x60  */
   __IO uint32_t RESERVED5;         
   __IO uint32_t CCIPR;            /*!< RCC CCIPR Register,                 Address offset: 0x68  */
   __IO uint32_t RESERVED6;         
   __IO uint32_t BDCR;             /*!< RCC BDCR Register,                  Address offset: 0x70  */
   __IO uint32_t CSR;              /*!< RCC CSR Register,                   Address offset: 0x74  */
   __IO uint32_t CFGR1;            /*!< RCC CFGR1 Register,                 Address offset: 0x78  */
   __IO uint32_t CFGR2;            /*!< RCC CFGR2 Register,                 Address offset: 0x7C  */
   __IO uint32_t CFGR3;            /*!< RCC CFGR3 Register,                 Address offset: 0x80  */
} RCC_TypeDef;


/**
* @brief RNG Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< RNG CR Register,                    Address offset: 0x00  */
   __IO uint32_t LFSR;             /*!< RNG LFSR Register,                  Address offset: 0x04  */
   __IO uint32_t SR;               /*!< RNG SR Register,                    Address offset: 0x08  */
   __IO uint32_t DR;               /*!< RNG DR Register,                    Address offset: 0x0C  */
} RNG_TypeDef;

/**
* @brief RTC Registers
*/
typedef struct
{
   __IO uint32_t CRH;              /*!< RTC CRH Register,                   Address offset: 0x00  */
   __IO uint32_t CRL;              /*!< RTC CRL Register,                   Address offset: 0x04  */
   __IO uint32_t PRLH;             /*!< RTC PRLH Register,                  Address offset: 0x08  */
   __IO uint32_t PRLL;             /*!< RTC PRLL Register,                  Address offset: 0x0C  */
   __IO uint32_t DIVH;             /*!< RTC DIVH Register,                  Address offset: 0x10  */
   __IO uint32_t DIVL;             /*!< RTC DIVL Register,                  Address offset: 0x14  */
   __IO uint32_t CNTH;             /*!< RTC CNTH Register,                  Address offset: 0x18  */
   __IO uint32_t CNTL;             /*!< RTC CNTL Register,                  Address offset: 0x1C  */
   __IO uint32_t ALRH;             /*!< RTC ALRH Register,                  Address offset: 0x20  */
   __IO uint32_t ALRL;             /*!< RTC ALRL Register,                  Address offset: 0x24  */
} RTC_TypeDef;


/**
* @brief SDIO Registers
*/
typedef struct
{
    __IO uint32_t POWER;       /*!< SDIO desc POWER,    Address offset: 0x00 */
    __IO uint32_t CLKCR;       /*!< SDIO desc CLKCR,    Address offset: 0x04 */
    __IO uint32_t ARG;         /*!< SDIO desc ARG,      Address offset: 0x08 */
    __IO uint32_t CMD;         /*!< SDIO desc CMD,      Address offset: 0x0C */
    __IO uint32_t RESPCMD;     /*!< SDIO desc RESPCMD,  Address offset: 0x10 */
    __IO uint32_t RESP0;       /*!< SDIO desc RESP0,    Address offset: 0x14 */
    __IO uint32_t RESP1;       /*!< SDIO desc RESP1,    Address offset: 0x18 */
    __IO uint32_t RESP2;       /*!< SDIO desc RESP2,    Address offset: 0x1C */
    __IO uint32_t RESP3;       /*!< SDIO desc RESP3,    Address offset: 0x20 */
    __IO uint32_t TMOUT;       /*!< SDIO desc TMOUT,    Address offset: 0x24 */
    __IO uint32_t BLKSIZ;      /*!< SDIO desc BLKSIZ,   Address offset: 0x28 */
    __IO uint32_t DLEN;        /*!< SDIO desc DLEN,     Address offset: 0x2C */
    __IO uint32_t CTRL;        /*!< SDIO desc CTRL,     Address offset: 0x30 */
    __IO uint32_t STATUS;      /*!< SDIO desc STATUS,   Address offset: 0x34 */
    __IO uint32_t INTSTS;      /*!< SDIO desc INTSTS,   Address offset: 0x38 */
    __IO uint32_t INTMASK;     /*!< SDIO desc INTMASK,  Address offset: 0x3C */
    __IO uint32_t FIFOTH;      /*!< SDIO desc FIFOTH,   Address offset: 0x40 */
    __IO uint32_t TCBCNT;      /*!< SDIO desc TCBCNT,   Address offset: 0x44 */
    __IO uint32_t TBBCNT;      /*!< SDIO desc TBBCNT,   Address offset: 0x48 */
    uint8_t RESERVED19[436];   /*!< Reserved, Address offset: 0x4C - 0x1FF */
    __IO uint32_t FIFODATA;    /*!< SDIO desc FIFODATA, Address offset: 0x200 */
}SDIO_TypeDef;

/**
* @brief SPI Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< SPI CR1 Register,                   Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< SPI CR2 Register,                   Address offset: 0x04  */
   __IO uint32_t SR;               /*!< SPI SR Register,                    Address offset: 0x08  */
   __IO uint32_t DR;               /*!< SPI DR Register,                    Address offset: 0x0C  */
   __IO uint32_t CRCPR;            /*!< SPI CRCPR Register,                 Address offset: 0x10  */
   __IO uint32_t RXCRCR;           /*!< SPI RXCRCR Register,                Address offset: 0x14  */
   __IO uint32_t TXCRCR;           /*!< SPI TXCRCR Register,                Address offset: 0x18  */
   __IO uint32_t I2SCFGR;          /*!< SPI I2SCFGR Register,               Address offset: 0x1C  */
   __IO uint32_t I2SPR;            /*!< SPI I2SPR Register,                 Address offset: 0x20  */
} SPI_TypeDef;




/**
* @brief SYSCFG Registers
*/
typedef struct
{
   __IO uint32_t CFGR[5];          /*!< SYSCFG configuration register,   Address offset: 0x00 ~ 0x13 */
   __IO uint32_t EXTICR[4];        /*!< SYSCFG external interrupt configuration registers,   Address offset: 0x14-0x23 */
   __IO uint32_t PAENS;            /*!< SYSCFG desc PAENS,   Address offset: 0x24 */
   __IO uint32_t PBENS;            /*!< SYSCFG desc PBENS,   Address offset: 0x28 */
   __IO uint32_t PCENS;            /*!< SYSCFG desc PCENS,   Address offset: 0x2C */
   __IO uint32_t PDENS;            /*!< SYSCFG desc PDENS,   Address offset: 0x30 */
   __IO uint32_t PEENS;            /*!< SYSCFG desc PEENS,   Address offset: 0x34 */
   __IO uint32_t PFENS;            /*!< SYSCFG desc PFENS,   Address offset: 0x38 */
   __IO uint32_t RESERVED1;         
   __IO uint32_t PAANA2ENR;        /*!< PA ANA2ENR Register,                Address offset: 0x40  */
   __IO uint32_t PBANA2ENR;        /*!< PB ANA2ENR Register,                Address offset: 0x44  */
   __IO uint32_t PCANA2ENR;        /*!< PC ANA2ENR Register,                Address offset: 0x48  */
   __IO uint32_t PDANA2ENR;        /*!< PD ANA2ENR Register,                Address offset: 0x4C  */
   __IO uint32_t PEANA2ENR;        /*!< PE ANA2ENR Register,                Address offset: 0x50  */
   __IO uint32_t PVDINANA2ENR;     /*!< PVDIN ANA2ENR Register,             Address offset: 0x54  */
   __IO uint32_t RESERVED2[6];          
   __IO uint32_t SCSR;             /*!< SYSCFG SCSR Register,               Address offset: 0x70  */
   __IO uint32_t SWPR;             /*!< SYSCFG SWPR Register,               Address offset: 0x74  */
   __IO uint32_t SKR;              /*!< SYSCFG SKR Register,                Address offset: 0x78  */
}SYSCFG_TypeDef;

/**
* @brief TIM Registers
*/
typedef struct
{
    __IO uint32_t CR1;         /*!< TIM desc CR1,          Address offset: 0x00 */
    __IO uint32_t CR2;         /*!< TIM desc CR2,          Address offset: 0x04 */
    __IO uint32_t SMCR;        /*!< TIM desc SMCR,         Address offset: 0x08 */
    __IO uint32_t DIER;        /*!< TIM desc DIER,         Address offset: 0x0C */
    __IO uint32_t SR;          /*!< TIM desc SR,           Address offset: 0x10 */
    __IO uint32_t EGR;         /*!< TIM desc EGR,          Address offset: 0x14 */
    __IO uint32_t CCMR1;       /*!< TIM desc CCMR1:OUTPUT, Address offset: 0x18 */
    __IO uint32_t CCMR2;       /*!< TIM desc CCMR2:OUTPUT, Address offset: 0x1C */
    __IO uint32_t CCER;        /*!< TIM desc CCER,         Address offset: 0x20 */
    __IO uint32_t CNT;         /*!< TIM desc CNT,          Address offset: 0x24 */
    __IO uint32_t PSC;         /*!< TIM desc PSC,          Address offset: 0x28 */
    __IO uint32_t ARR;         /*!< TIM desc ARR,          Address offset: 0x2C */
    __IO uint32_t RCR;         /*!< TIM desc RCR,          Address offset: 0x30 */
    __IO uint32_t CCR1;        /*!< TIM desc CCR1,         Address offset: 0x34 */
    __IO uint32_t CCR2;        /*!< TIM desc CCR2,         Address offset: 0x38 */
    __IO uint32_t CCR3;        /*!< TIM desc CCR3,         Address offset: 0x3C */
    __IO uint32_t CCR4;        /*!< TIM desc CCR4,         Address offset: 0x40 */
    __IO uint32_t BDTR;        /*!< TIM desc BDTR,         Address offset: 0x44 */
    __IO uint32_t CCR5;        /*!< TIM desc CCR5,         Address offset: 0x48 */
    __IO uint32_t CCR6;        /*!< TIM desc CCR6,         Address offset: 0x4C */
    __IO uint32_t CCMR3;       /*!< TIM desc CCMR3:OUTPUT, Address offset: 0x50 */
    __IO uint32_t DTR2;        /*!< TIM desc DTR,          Address offset: 0x54 */
    __IO uint32_t ECR;         /*!< TIM desc ECR,          Address offset: 0x58 */
    __IO uint32_t TISEL;       /*!< TIM desc TISEL,        Address offset: 0x5C */
    __IO uint32_t AF1;         /*!< TIM AF1 Register,      Address offset: 0x60 */
    __IO uint32_t AF2;         /*!< TIM AF2 Register,      Address offset: 0x64 */
    __IO uint32_t OR1;         /*!< TIM OR1 Register,      Address offset: 0x68 */
         uint32_t RESERVED0[220];/*!< Reserved,              Address offset: 0x6C */
    __IO uint32_t DCR;         /*!< TIM desc DCR,          Address offset: 0x3DC */
    __IO uint32_t DMAR;        /*!< TIM desc DMAR,         Address offset: 0x3E0 */
}TIM_TypeDef;

/**
* @brief USART Registers
*/
typedef struct
{
   __IO uint32_t SR;               /*!< USART SR Register,                  Address offset: 0x00  */
   __IO uint32_t DR;               /*!< USART DR Register,                  Address offset: 0x04  */
   __IO uint32_t BRR;              /*!< USART BRR Register,                 Address offset: 0x08  */
   __IO uint32_t CR1;              /*!< USART CR1 Register,                 Address offset: 0x0C  */
   __IO uint32_t CR2;              /*!< USART CR2 Register,                 Address offset: 0x10  */
   __IO uint32_t CR3;              /*!< USART CR3 Register,                 Address offset: 0x14  */
   __IO uint32_t GTPR;             /*!< USART GTPR Register,                Address offset: 0x18  */
   __IO uint32_t RTOR;             /*!< USART RTOR Register,                Address offset: 0x1C  */
} USART_TypeDef;



/**
  * @brief Universal Synchronous Asynchronous Receiver Transmitter
  */
typedef struct
{
  __IO uint32_t DR;          /*!< UART DR Register,           Address offset: 0x00  */
  __IO uint32_t BRR;         /*!< UART BRR register,          Address offset: 0x04  */
  __IO uint32_t RESERVED1[2];   
  __IO uint32_t SR;          /*!< UART SR  register 2,        Address offset: 0x10  */
  __IO uint32_t CR1;         /*!< UART Control register 1,    Address offset: 0x14  */
  __IO uint32_t CR2;         /*!< UART Control register 2,    Address offset: 0x18  */
  __IO uint32_t CR3;         /*!< UART Control register 3,    Address offset: 0x1C  */
  __IO uint32_t RAR;         /*!< UART RAR register,          Address offset: 0x20  */
  __IO uint32_t TAR;         /*!< UART TAR register,          Address offset: 0x24  */
  __IO uint32_t BRRF;        /*!< UART BRRF register,         Address offset: 0x28  */
} UART_TypeDef;

/**
* @brief LPUART Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< LPUART CR1 Register,                Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< LPUART CR2 Register,                Address offset: 0x04  */
   __IO uint32_t CR3;              /*!< LPUART CR3 Register,                Address offset: 0x08  */
   __IO uint32_t BRR;              /*!< LPUART BRR Register,                Address offset: 0x0C  */
   __IO uint32_t RESERVED1[2];      
   __IO uint32_t RQR;              /*!< LPUART RQR Register,                Address offset: 0x18  */
   __IO uint32_t ISR;              /*!< LPUART ISR Register,                Address offset: 0x1C  */
   __IO uint32_t ICR;              /*!< LPUART ICR Register,                Address offset: 0x20  */
   __IO uint32_t RDR;              /*!< LPUART RDR Register,                Address offset: 0x24  */
   __IO uint32_t TDR;              /*!< LPUART TDR Register,                Address offset: 0x28  */
   __IO uint32_t PRESC;            /*!< LPUART PRESC Register,              Address offset: 0x2C  */
} LPUART_TypeDef;


/** 
  * @brief __USB_OTG_Core_register
  */

typedef struct
{
  __IO uint32_t GOTGCTL;              /*!<  USB_OTG Control and Status Register       Address offset: 000h */
  __IO uint32_t GOTGINT;              /*!<  USB_OTG Interrupt Register                Address offset: 004h */
  __IO uint32_t GAHBCFG;              /*!<  Core AHB Configuration Register           Address offset: 008h */
  __IO uint32_t GUSBCFG;              /*!<  Core USB Configuration Register           Address offset: 00Ch */
  __IO uint32_t GRSTCTL;              /*!<  Core Reset Register                       Address offset: 010h */
  __IO uint32_t GINTSTS;              /*!<  Core Interrupt Register                   Address offset: 014h */
  __IO uint32_t GINTMSK;              /*!<  Core Interrupt Mask Register              Address offset: 018h */
  __IO uint32_t GRXSTSR;              /*!<  Receive Sts Q Read Register               Address offset: 01Ch */
  __IO uint32_t GRXSTSP;              /*!<  Receive Sts Q Read & POP Register         Address offset: 020h */
  __IO uint32_t GRXFSIZ;              /*!< Receive FIFO Size Register                 Address offset: 024h */
  __IO uint32_t DIEPTXF0_HNPTXFSIZ;   /*!<  EP0 / Non Periodic Tx FIFO Size Register  Address offset: 028h */
  __IO uint32_t HNPTXSTS;             /*!<  Non Periodic Tx FIFO/Queue Sts reg        Address offset: 02Ch */
  uint32_t Reserved30[2];             /*!< Reserved 030h*/
  __IO uint32_t GCCFG;                /*!< General Purpose IO Register                Address offset: 038h */
  __IO uint32_t CID;                  /*!< User ID Register                           Address offset: 03Ch */
  uint32_t  Reserved40[48];           /*!< Reserved 040h-0FFh */
  __IO uint32_t HPTXFSIZ;             /*!< Host Periodic Tx FIFO Size Reg             Address offset: 100h */
  __IO uint32_t DIEPTXF[0x0F];        /*!< dev Periodic Transmit FIFO                 Address offset: 0x104 */
} USB_OTG_GlobalTypeDef;

/** 
  * @brief __device_Registers
  */

typedef struct 
{
  __IO uint32_t DCFG;                 /*!< dev Configuration Register                 Address offset: 800h*/
  __IO uint32_t DCTL;                 /*!< dev Control Register                       Address offset: 804h*/
  __IO uint32_t DSTS;                 /*!< dev Status Register (RO)                   Address offset: 808h*/
  uint32_t Reserved0C;                /*!< Reserved 80Ch*/
  __IO uint32_t DIEPMSK;              /*!< dev IN Endpoint Mask                       Address offset: 810h*/
  __IO uint32_t DOEPMSK;              /*!< dev OUT Endpoint Mask                      Address offset: 814h*/
  __IO uint32_t DAINT;                /*!< dev All Endpoints Itr Reg                  Address offset: 818h*/
  __IO uint32_t DAINTMSK;             /*!< dev All Endpoints Itr Mask                 Address offset: 81Ch*/
  uint32_t Reserved20;                /*!< Reserved 820h                                                  */
  uint32_t Reserved9;                 /*!< Reserved 824h                                                  */
  __IO uint32_t DVBUSDIS;             /*!< dev VBUS discharge Register                Address offset: 828h*/
  __IO uint32_t DVBUSPULSE;           /*!< dev VBUS Pulse Register                    Address offset: 82Ch*/
  uint32_t Reserved;                  /*!< Reserved 830h                                                  */
  __IO uint32_t DIEPEMPMSK;           /*!< dev empty msk                              Address offset: 834h*/
} USB_OTG_DeviceTypeDef;

/** 
  * @brief __IN_Endpoint-Specific_Register
  */

typedef struct 
{
  __IO uint32_t DIEPCTL;              /*!< dev IN Endpoint Control Reg                900h + (ep_num * 20h) + 00h*/
  uint32_t Reserved04;                /*!< Reserved                                   900h + (ep_num * 20h) + 04h*/
  __IO uint32_t DIEPINT;              /*!< dev IN Endpoint Itr Reg                    900h + (ep_num * 20h) + 08h*/
  uint32_t Reserved0C;                /*!< Reserved                                   900h + (ep_num * 20h) + 0Ch*/
  __IO uint32_t DIEPTSIZ;             /*!< IN Endpoint Txfer Size                     900h + (ep_num * 20h) + 10h*/
  uint32_t Reserved14;                /*!< Reserved                                   900h + (ep_num * 20h) + 14h*/
  __IO uint32_t DTXFSTS;              /*!< IN Endpoint Tx FIFO Status Reg             900h + (ep_num * 20h) + 18h*/
  uint32_t Reserved1C;                /*!< Reserved                                   900h + (ep_num * 20h) + 1Ch */
} USB_OTG_INEndpointTypeDef;

/** 
  * @brief __OUT_Endpoint-Specific_Registers
  */

typedef struct 
{
  __IO uint32_t DOEPCTL;              /*!< dev OUT Endpoint Control Reg               B00h + (ep_num * 20h) + 00h*/
  uint32_t Reserved04;                /*!< Reserved                                   B00h + (ep_num * 20h) + 04h*/
  __IO uint32_t DOEPINT;              /*!< dev OUT Endpoint Itr Reg                   B00h + (ep_num * 20h) + 08h*/
  uint32_t Reserved0C;                /*!< Reserved                                   B00h + (ep_num * 20h) + 0Ch*/
  __IO uint32_t DOEPTSIZ;             /*!< dev OUT Endpoint Txfer Size                B00h + (ep_num * 20h) + 10h*/
  uint32_t Reserved14[3];             /*!< Reserved                                   B00h + (ep_num * 20h) + 14h - B00h + (ep_num * 20h) + 1Ch*/
} USB_OTG_OUTEndpointTypeDef;

/** 
  * @brief __Host_Mode_Register_Structures
  */

typedef struct 
{
  __IO uint32_t HCFG;                 /*!< Host Configuration Register                Address offset: 400h*/
  __IO uint32_t HFIR;                 /*!< Host Frame Interval Register               Address offset: 404h*/
  __IO uint32_t HFNUM;                /*!< Host Frame Nbr/Frame Remaining             Address offset: 408h*/
  uint32_t Reserved40C;               /*!< Reserved                                   Address offset: 40Ch*/
  __IO uint32_t HPTXSTS;              /*!< Host Periodic Tx FIFO/ Queue Status        Address offset: 410h*/
  __IO uint32_t HAINT;                /*!< Host All Channels Interrupt Register       Address offset: 414h*/
  __IO uint32_t HAINTMSK;             /*!< Host All Channels Interrupt Mask           Address offset: 418h*/
} USB_OTG_HostTypeDef;

/** 
  * @brief __Host_Channel_Specific_Registers
  */

typedef struct
{
  __IO uint32_t HCCHAR;               /*!< Host Channel Characteristics Register      500h + (ch_num * 20h) + 00h*/
  uint32_t Reserved04;                /*!< Reserved                                   500h + (ch_num * 20h) + 04h*/  
  __IO uint32_t HCINT;                /*!< Host Channel Interrupt Register            500h + (ch_num * 20h) + 08h*/
  __IO uint32_t HCINTMSK;             /*!< Host Channel Interrupt Mask Register       500h + (ch_num * 20h) + 0Ch*/
  __IO uint32_t HCTSIZ;               /*!< Host Channel Transfer Size Register        500h + (ch_num * 20h) + 10h*/
  uint32_t Reserved14[3];             /*!< Reserved                                   500h + (ch_num * 20h) + 14h - 500h + (ch_num * 20h) + 1Ch*/
} USB_OTG_HostChannelTypeDef;

/**
* @brief WWDG Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< WWDR CR Register,                   Address offset: 0x0  */
   __IO uint32_t CFR;              /*!< WWDG CFR Register,                  Address offset: 0x4  */
   __IO uint32_t SR;               /*!< WWDG SR Register,                   Address offset: 0x8  */
}WWDG_TypeDef;

/*!< Peripheral memory map */
#define FLASH_BASE            (0x08000000UL)                         /*!< FLASH base address */
#define FLASH_END             (0x0803FFFFUL)                         /*!< FLASH end address */
#define FLASH_SIZE            (FLASH_END - FLASH_BASE + 1)
#define FLASH_PAGE_SIZE       (0x00000200UL)                         /*!< FLASH Page Size, 512 Bytes */
#define FLASH_PAGE_NB         (FLASH_SIZE / FLASH_PAGE_SIZE)
#define FLASH_SECTOR_SIZE     (0x00001000UL)                         /*!< FLASH Sector Size, 4096 Bytes */
#define FLASH_SECTOR_NB       (FLASH_SIZE / FLASH_SECTOR_SIZE)
#define FLASH_BLOCK_SIZE      (0x00010000UL)                         /*!< FLASH Block  Size, 65536 Bytes */
#define FLASH_BLOCK_NB        (FLASH_SIZE / FLASH_BLOCK_SIZE)
#define SRAM1_BASE            (0x20000000UL)                         /*!< SRAM1(up to 64 KB) base address */
#define SRAM2_BASE            (0x20010000UL)                         /*!< SRAM2(16 KB) base address */
#define CCMSRAM_BASE          (0x10000000UL)                         /*!< CCMSRAM(32 KB) base address */
#define PERIPH_BASE           (0x40000000UL)                         /*!< Peripheral base address */
#define SRAM1_BB_BASE         (0x22000000UL)                         /*!< SRAM1(64 KB) base address in the bit-band region */
#define SRAM2_BB_BASE         (0x22200000UL)                         /*!< SRAM2(16 KB) base address in the bit-band region */
#define CCMSRAM_BB_BASE       (0x22280000UL)                         /*!< CCMSRAM(32 KB) base address in the bit-band region */
#define PERIPH_BB_BASE        (0x42000000UL)                         /*!< Peripheral base address in the bit-band region */
#define OB_BASE               (0x1FFF7400UL)                         /*!< Flash Option Bytes base address */
#define UID_BASE              (0x1FFF7E00UL)                         /*!< Unique device ID register base address */
#define OTP_BASE              (0x1FFF7000UL)

/* Legacy defines */
#define SRAM_BASE             SRAM1_BASE
#define SRAM_BB_BASE          SRAM1_BB_BASE

#define SRAM1_SIZE_MAX        (0x00010000UL) /*!< maximum SRAM1 size (up to 64 KBytes) */
#define SRAM2_SIZE            (0x00004000UL) /*!< SRAM2 size (16 KBytes) */
#define CCMSRAM_SIZE          (0x00008000UL) /*!< CCMSRAM size (32 KBytes) */

#define APB1PERIPH_BASE                        PERIPH_BASE
#define APB2PERIPH_BASE                        (PERIPH_BASE + 0x00010000UL)

#define AHBPERIPH_BASE                         (0xA0000000UL)
#define AHB1PERIPH_BASE                        (PERIPH_BASE + 0x00020000UL)
#define AHB2PERIPH_BASE                        (PERIPH_BASE + 0x08000000UL)

/*!< APB1 peripherals */
#define TIM2_BASE                              (0x40000000UL)
#define TIM3_BASE                              (0x40000400UL)
#define TIM4_BASE                              (0x40000800UL)
#define TIM5_BASE                              (0x40000C00UL)
#define TIM6_BASE                              (0x40001000UL)
#define TIM7_BASE                              (0x40001400UL)
#define TIM12_BASE                             (0x40001800UL)
#define TIM13_BASE                             (0x40001C00UL)
#define TIM14_BASE                             (0x40002000UL)
#define TIM18_BASE                             (0x40002400UL)
#define RTC_BASE                               (0x40002800UL)
#define WWDG_BASE                              (0x40002C00UL)
#define IWDG_BASE                              (0x40003000UL)
#define SPI2_BASE                              (0x40003800UL)
#define SPI3_BASE                              (0x40003C00UL)
#define USART2_BASE                            (0x40004400UL)
#define USART3_BASE                            (0x40004800UL)
#define UART1_BASE                             (0x40004C00UL)
#define UART2_BASE                             (0x40005000UL)
#define I2C1_BASE                              (0x40005400UL)
#define I2C2_BASE                              (0x40005800UL)
#define UART3_BASE                             (0x40005C00UL)
#define BKP_BASE                               (0x40006C00UL)
#define PWR_BASE                               (0x40007000UL)
#define DAC1_BASE                              (0x40007400UL)
#define I2C3_BASE                              (0x40007800UL)
#define LPTIM1_BASE                            (0x40007C00UL)
#define LPUART_BASE                            (0x40008000UL)
#define I2C4_BASE                              (0x40008400UL)
#define SRAMCAN_BASE                           (0x4000AC00UL)
#define CTC_BASE                               (0x4000C800UL)
#define FDCAN1_BASE                            (0x4000CC00UL)
#define FDCAN2_BASE                            (0x4000E000UL)

/*!< APB2 peripherals */
#define SYSCFG_BASE                            (0x40010000UL)
#define EXTI_BASE                              (0x40010400UL)
#define ADC1_BASE                              (0x40012400UL)
#define ADC_COMMON_BASE                        (ADC1_BASE + 0x00000308UL)
#define ADC2_BASE                              (0x40012800UL)
#define TIM1_BASE                              (0x40012C00UL)
#define SPI1_BASE                              (0x40013000UL)
#define TIM8_BASE                              (0x40013400UL)
#define USART1_BASE                            (0x40013800UL)
#define ADC3_BASE                              (0x40013C00UL)
#define TIM9_BASE                              (0x40014C00UL)
#define TIM10_BASE                             (0x40015000UL)
#define TIM11_BASE                             (0x40015400UL)
#define TIM15_BASE                             (0x40015800UL)
#define TIM16_BASE                             (0x40015C00UL)
#define TIM17_BASE                             (0x40016000UL)
#define TIM19_BASE                             (0x40016400UL)
#define RNG_BASE                               (0x40016800UL)
#define COMP1_BASE                             (0x40016C00UL)
#define COMP2_BASE                             (0x40016C10UL)
#define COMP3_BASE                             (0x40016C20UL)
#define COMP4_BASE                             (0x40016C30UL)
#define OPA_BASE                               (0x40017000UL)
#define OPA1_BASE                              (0x40017000UL)
#define OPA2_BASE                              (0x40017004UL)
#define OPA3_BASE                              (0x40017008UL)
#define LCDC0123_COMMON_BASE                   (0x40017400UL)
#define LCDC0_BASE                             (0x40017404UL)
#define LCDC1_BASE                             (0x40017414UL)
#define LCDC2_BASE                             (0x40017424UL)
#define LCDC3_BASE                             (0x40017434UL)

/*!< AHB1 peripherals */
#define DMA1_BASE                              (AHB1PERIPH_BASE + 0x00000000UL)
#define DMA2_BASE                              (AHB1PERIPH_BASE + 0x00000400UL)
#define RCC_BASE                               (0x40021000UL)
#define FLASH_R_BASE                           (AHB1PERIPH_BASE + 0x2000) /*!< Flash registers base address */
#define CRC_BASE                               (0x40023000UL)

#define DMA1_Channel1_BASE                     (AHB1PERIPH_BASE + 0x00000010UL)
#define DMA1_Channel2_BASE                     (AHB1PERIPH_BASE + 0x00000068UL)
#define DMA1_Channel3_BASE                     (AHB1PERIPH_BASE + 0x000000C0UL)
#define DMA1_Channel4_BASE                     (AHB1PERIPH_BASE + 0x00000118UL)
#define DMA1_Channel5_BASE                     (AHB1PERIPH_BASE + 0x00000170UL)
#define DMA1_Channel6_BASE                     (AHB1PERIPH_BASE + 0x000001C8UL)
#define DMA2_Channel1_BASE                     (AHB1PERIPH_BASE + 0x00000410UL)
#define DMA2_Channel2_BASE                     (AHB1PERIPH_BASE + 0x00000468UL)
#define DMA2_Channel3_BASE                     (AHB1PERIPH_BASE + 0x000004C0UL)
#define DMA2_Channel4_BASE                     (AHB1PERIPH_BASE + 0x00000518UL)
#define DMA2_Channel5_BASE                     (AHB1PERIPH_BASE + 0x00000570UL)
#define DMA2_Channel6_BASE                     (AHB1PERIPH_BASE + 0x000005C8UL)

/*!< AHB2 peripherals */
#define CORDIC_BASE                            (0x4002C000UL)
#define GPIOA_BASE                             (0x48000000UL)
#define GPIOB_BASE                             (0x48000400UL)
#define GPIOC_BASE                             (0x48000800UL)
#define GPIOD_BASE                             (0x48000C00UL)
#define GPIOE_BASE                             (0x48001000UL)
#define GPIOF_BASE                             (0x48001400UL)
#define SDIO_BASE                              (0x48002000UL)
#define AES_BASE                               (0x48002400UL)
#define ETH_BASE                               (0x480C0000UL)
#define ESMC_BASE                              (AHBPERIPH_BASE + 0x00001000UL)
#define USB1_OTG_FS_PERIPH_BASE                (0x48040000UL)
#define USB2_OTG_FS_PERIPH_BASE                (0x48080000UL)
#define ETH_MAC_BASE                           (ETH_BASE)
#define ETH_MMC_BASE                           (ETH_BASE + 0x00000100UL)
#define ETH_PTP_BASE                           (ETH_BASE + 0x00000700UL)
#define ETH_DMA_BASE                           (ETH_BASE + 0x00001000UL)

/*!< USB registers address offset*/
#define USB_OTG_GLOBAL_BASE                    (0x00000000UL)
#define USB_OTG_DEVICE_BASE                    (0x00000800UL)
#define USB_OTG_IN_ENDPOINT_BASE               (0x00000900UL)
#define USB_OTG_OUT_ENDPOINT_BASE              (0x00000B00UL)
#define USB_OTG_EP_REG_SIZE                    (0x00000020UL)
#define USB_OTG_HOST_BASE                      (0x00000400UL)
#define USB_OTG_HOST_PORT_BASE                 (0x00000440UL)
#define USB_OTG_HOST_CHANNEL_BASE              (0x00000500UL)
#define USB_OTG_HOST_CHANNEL_SIZE              (0x00000020UL)
#define USB_OTG_PCGCCTL_BASE                   (0x00000E00UL)
#define USB_OTG_FIFO_BASE                      (0x00001000UL)
#define USB_OTG_FIFO_SIZE                      (0x00001000UL)

/* Debug MCU registers base address */
#define DBGMCU_BASE                            (0xE0042000UL)

#define ADC1                                   ((ADC_TypeDef *) ADC1_BASE)
#define ADC_COMMON                             ((ADC_Common_TypeDef *) ADC_COMMON_BASE)
#define ADC2                                   ((ADC_TypeDef *) ADC2_BASE)
#define ADC3                                   ((ADC_TypeDef *) ADC3_BASE)
#define AES                                    ((AES_TypeDef *) AES_BASE)
#define BKP                                    ((BKP_TypeDef *) BKP_BASE)
#define FDCAN1                                 ((FDCAN_GlobalTypeDef *) FDCAN1_BASE)
#define FDCAN2                                 ((FDCAN_GlobalTypeDef *) FDCAN2_BASE)
#define COMP1                                  ((COMP_TypeDef *) COMP1_BASE)
#define COMP2                                  ((COMP_TypeDef *) COMP2_BASE)
#define COMP3                                  ((COMP_TypeDef *) COMP3_BASE)
#define COMP4                                  ((COMP_TypeDef *) COMP4_BASE)
#define COMP12_COMMON                          ((COMP_Common_TypeDef *) COMP1_BASE)
#define COMP34_COMMON                          ((COMP_Common_TypeDef *) COMP3_BASE)
#define CORDIC                                 ((CORDIC_TypeDef *) CORDIC_BASE)
#define CRC                                    ((CRC_TypeDef *) CRC_BASE)
#define CTC                                    ((CTC_TypeDef *) CTC_BASE)
#define DAC1                                   ((DAC_TypeDef *) DAC1_BASE)
#define DBGMCU                                 ((DBGMCU_TypeDef *) DBGMCU_BASE)

#define DMA1                                   ((DMA_TypeDef *)DMA1_BASE)
#define DMA2                                   ((DMA_TypeDef *)DMA2_BASE)
#define DMA1_Channel1                          ((DMA_Channel_TypeDef *)DMA1_Channel1_BASE)
#define DMA1_Channel2                          ((DMA_Channel_TypeDef *)DMA1_Channel2_BASE)
#define DMA1_Channel3                          ((DMA_Channel_TypeDef *)DMA1_Channel3_BASE)
#define DMA1_Channel4                          ((DMA_Channel_TypeDef *)DMA1_Channel4_BASE)
#define DMA1_Channel5                          ((DMA_Channel_TypeDef *)DMA1_Channel5_BASE)
#define DMA1_Channel6                          ((DMA_Channel_TypeDef *)DMA1_Channel6_BASE)
#define DMA2_Channel1                          ((DMA_Channel_TypeDef *)DMA2_Channel1_BASE)
#define DMA2_Channel2                          ((DMA_Channel_TypeDef *)DMA2_Channel2_BASE)
#define DMA2_Channel3                          ((DMA_Channel_TypeDef *)DMA2_Channel3_BASE)
#define DMA2_Channel4                          ((DMA_Channel_TypeDef *)DMA2_Channel4_BASE)
#define DMA2_Channel5                          ((DMA_Channel_TypeDef *)DMA2_Channel5_BASE)
#define DMA2_Channel6                          ((DMA_Channel_TypeDef *)DMA2_Channel6_BASE)
#define ETH                                    ((ETH_TypeDef *) ETH_BASE)
#define EXTI                                   ((EXTI_TypeDef *) EXTI_BASE)
#define ESMC                                   ((ESMC_TypeDef *) ESMC_BASE)
#define FLASH                                  ((FLASH_TypeDef *) FLASH_R_BASE)
#define OPA                                    ((OPA_TypeDef *) OPA_BASE)
#define OPA1                                   ((OPA_TypeDef *) OPA1_BASE)
#define OPA2                                   ((OPA_TypeDef *) OPA2_BASE)
#define OPA3                                   ((OPA_TypeDef *) OPA3_BASE)
#define GPIOA                                  ((GPIO_TypeDef *) GPIOA_BASE)
#define GPIOB                                  ((GPIO_TypeDef *) GPIOB_BASE)
#define GPIOC                                  ((GPIO_TypeDef *) GPIOC_BASE)
#define GPIOD                                  ((GPIO_TypeDef *) GPIOD_BASE)
#define GPIOE                                  ((GPIO_TypeDef *) GPIOE_BASE)
#define GPIOF                                  ((GPIO_TypeDef *) GPIOF_BASE)
#define I2C1                                   ((I2C_TypeDef *) I2C1_BASE)
#define I2C2                                   ((I2C_TypeDef *) I2C2_BASE)
#define I2C3                                   ((I2C_TypeDef *) I2C3_BASE)
#define I2C4                                   ((I2C_TypeDef *) I2C4_BASE)
#define IWDG                                   ((IWDG_TypeDef *) IWDG_BASE)
#define LCDC0123_COMMON                        ((LCDC_Common_TypeDef *) LCDC0123_COMMON_BASE)
#define LCDC0                                  ((LCDC_TypeDef *) LCDC0_BASE)
#define LCDC1                                  ((LCDC_TypeDef *) LCDC1_BASE)
#define LCDC2                                  ((LCDC_TypeDef *) LCDC2_BASE)
#define LCDC3                                  ((LCDC_TypeDef *) LCDC3_BASE)
#define LPTIM                                  ((LPTIM_TypeDef *) LPTIM_BASE)
#define LPTIM1                                 ((LPTIM_TypeDef *) LPTIM1_BASE)
#define LPUART                                 ((LPUART_TypeDef *) LPUART_BASE)
#define PWR                                    ((PWR_TypeDef *) PWR_BASE)
#define RCC                                    ((RCC_TypeDef *) RCC_BASE)
#define RNG                                    ((RNG_TypeDef *) RNG_BASE)
#define RTC                                    ((RTC_TypeDef *) RTC_BASE)
#define SDIO                                   ((SDIO_TypeDef *) SDIO_BASE)
#define SPI1                                   ((SPI_TypeDef *) SPI1_BASE)
#define SPI2                                   ((SPI_TypeDef *) SPI2_BASE)
#define SPI3                                   ((SPI_TypeDef *) SPI3_BASE)
#define SYSCFG                                 ((SYSCFG_TypeDef *) SYSCFG_BASE)
#define TIM1                                   ((TIM_TypeDef *) TIM1_BASE)
#define TIM2                                   ((TIM_TypeDef *) TIM2_BASE)
#define TIM3                                   ((TIM_TypeDef *) TIM3_BASE)
#define TIM4                                   ((TIM_TypeDef *) TIM4_BASE)
#define TIM5                                   ((TIM_TypeDef *) TIM5_BASE)
#define TIM6                                   ((TIM_TypeDef *) TIM6_BASE)
#define TIM7                                   ((TIM_TypeDef *) TIM7_BASE)
#define TIM8                                   ((TIM_TypeDef *) TIM8_BASE)
#define TIM9                                   ((TIM_TypeDef *) TIM9_BASE)
#define TIM10                                  ((TIM_TypeDef *) TIM10_BASE)
#define TIM11                                  ((TIM_TypeDef *) TIM11_BASE)
#define TIM12                                  ((TIM_TypeDef *) TIM12_BASE)
#define TIM13                                  ((TIM_TypeDef *) TIM13_BASE)
#define TIM14                                  ((TIM_TypeDef *) TIM14_BASE)
#define TIM15                                  ((TIM_TypeDef *) TIM15_BASE)
#define TIM16                                  ((TIM_TypeDef *) TIM16_BASE)
#define TIM17                                  ((TIM_TypeDef *) TIM17_BASE)
#define TIM18                                  ((TIM_TypeDef *) TIM18_BASE)
#define TIM19                                  ((TIM_TypeDef *) TIM19_BASE)
#define USART1                                 ((USART_TypeDef *) USART1_BASE)
#define USART2                                 ((USART_TypeDef *) USART2_BASE)
#define USART3                                 ((USART_TypeDef *) USART3_BASE)
#define UART1                                  ((UART_TypeDef *) UART1_BASE)
#define UART2                                  ((UART_TypeDef *) UART2_BASE)
#define UART3                                  ((UART_TypeDef *) UART3_BASE)
#define WWDG                                   ((WWDG_TypeDef *) WWDG_BASE)

#define USB1_OTG_FS                            ((USB_OTG_GlobalTypeDef *)USB1_OTG_FS_PERIPH_BASE)
#define USB2_OTG_FS                            ((USB_OTG_GlobalTypeDef *)USB2_OTG_FS_PERIPH_BASE)

/********************************************************************************************************************/
/********************************* ADC ******************************************************************************/
/********************************************************************************************************************/
/*
 * @brief Specific device feature definitions (not present on all devices in the mcu series)
 */
#define ADC_MULTIMODE_SUPPORT                          /*!< ADC feature available only on specific devices: multimode available on devices with several ADC instances */

/********************************* Bit definition for ADC_ISR register **********************************************/
#define ADC_ISR_EOSMP_Pos                         (1U)
#define ADC_ISR_EOSMP_Msk                         (0x1UL<<ADC_ISR_EOSMP_Pos)                        /*!< 0x00000002 */
#define ADC_ISR_EOSMP                             ADC_ISR_EOSMP_Msk                                 
#define ADC_ISR_EOC_Pos                           (2U)
#define ADC_ISR_EOC_Msk                           (0x1UL<<ADC_ISR_EOC_Pos)                          /*!< 0x00000004 */
#define ADC_ISR_EOC                               ADC_ISR_EOC_Msk                                   
#define ADC_ISR_EOS_Pos                           (3U)
#define ADC_ISR_EOS_Msk                           (0x1UL<<ADC_ISR_EOS_Pos)                          /*!< 0x00000008 */
#define ADC_ISR_EOS                               ADC_ISR_EOS_Msk                                   
#define ADC_ISR_JEOC_Pos                          (5U)
#define ADC_ISR_JEOC_Msk                          (0x1UL<<ADC_ISR_JEOC_Pos)                         /*!< 0x00000020 */
#define ADC_ISR_JEOC                              ADC_ISR_JEOC_Msk                                  
#define ADC_ISR_JEOS_Pos                          (6U)
#define ADC_ISR_JEOS_Msk                          (0x1UL<<ADC_ISR_JEOS_Pos)                         /*!< 0x00000040 */
#define ADC_ISR_JEOS                              ADC_ISR_JEOS_Msk                                  
#define ADC_ISR_AWD1_Pos                          (7U)
#define ADC_ISR_AWD1_Msk                          (0x1UL<<ADC_ISR_AWD1_Pos)                         /*!< 0x00000080 */
#define ADC_ISR_AWD1                              ADC_ISR_AWD1_Msk                                  
#define ADC_ISR_AWD2_Pos                          (8U)
#define ADC_ISR_AWD2_Msk                          (0x1UL<<ADC_ISR_AWD2_Pos)                         /*!< 0x00000100 */
#define ADC_ISR_AWD2                              ADC_ISR_AWD2_Msk                                  
#define ADC_ISR_AWD3_Pos                          (9U)
#define ADC_ISR_AWD3_Msk                          (0x1UL<<ADC_ISR_AWD3_Pos)                         /*!< 0x00000200 */
#define ADC_ISR_AWD3                              ADC_ISR_AWD3_Msk                                  
#define ADC_ISR_JSTRT_Pos                         (11U)
#define ADC_ISR_JSTRT_Msk                         (0x1UL<<ADC_ISR_JSTRT_Pos)                        /*!< 0x00000800 */
#define ADC_ISR_JSTRT                             ADC_ISR_JSTRT_Msk                                 
#define ADC_ISR_STRT_Pos                          (12U)
#define ADC_ISR_STRT_Msk                          (0x1UL<<ADC_ISR_STRT_Pos)                         /*!< 0x00001000 */
#define ADC_ISR_STRT                              ADC_ISR_STRT_Msk                                  

/********************************* Bit definition for ADC_IER register **********************************************/
#define ADC_IER_EOSMPIE_Pos                       (1U)
#define ADC_IER_EOSMPIE_Msk                       (0x1UL<<ADC_IER_EOSMPIE_Pos)                      /*!< 0x00000002 */
#define ADC_IER_EOSMPIE                           ADC_IER_EOSMPIE_Msk                               
#define ADC_IER_EOCIE_Pos                         (2U)
#define ADC_IER_EOCIE_Msk                         (0x1UL<<ADC_IER_EOCIE_Pos)                        /*!< 0x00000004 */
#define ADC_IER_EOCIE                             ADC_IER_EOCIE_Msk                                 
#define ADC_IER_EOSIE_Pos                         (3U)
#define ADC_IER_EOSIE_Msk                         (0x1UL<<ADC_IER_EOSIE_Pos)                        /*!< 0x00000008 */
#define ADC_IER_EOSIE                             ADC_IER_EOSIE_Msk                                 
#define ADC_IER_JEOCIE_Pos                        (5U)
#define ADC_IER_JEOCIE_Msk                        (0x1UL<<ADC_IER_JEOCIE_Pos)                       /*!< 0x00000020 */
#define ADC_IER_JEOCIE                            ADC_IER_JEOCIE_Msk                                
#define ADC_IER_JEOSIE_Pos                        (6U)
#define ADC_IER_JEOSIE_Msk                        (0x1UL<<ADC_IER_JEOSIE_Pos)                       /*!< 0x00000040 */
#define ADC_IER_JEOSIE                            ADC_IER_JEOSIE_Msk                                
#define ADC_IER_AWD1IE_Pos                        (7U)
#define ADC_IER_AWD1IE_Msk                        (0x1UL<<ADC_IER_AWD1IE_Pos)                       /*!< 0x00000080 */
#define ADC_IER_AWD1IE                            ADC_IER_AWD1IE_Msk                                
#define ADC_IER_AWD2IE_Pos                        (8U)
#define ADC_IER_AWD2IE_Msk                        (0x1UL<<ADC_IER_AWD2IE_Pos)                       /*!< 0x00000100 */
#define ADC_IER_AWD2IE                            ADC_IER_AWD2IE_Msk                                
#define ADC_IER_AWD3IE_Pos                        (9U)
#define ADC_IER_AWD3IE_Msk                        (0x1UL<<ADC_IER_AWD3IE_Pos)                       /*!< 0x00000200 */
#define ADC_IER_AWD3IE                            ADC_IER_AWD3IE_Msk                                

/********************************* Bit definition for ADC_CR register ***********************************************/
#define ADC_CR_ADEN_Pos                           (0U)
#define ADC_CR_ADEN_Msk                           (0x1UL<<ADC_CR_ADEN_Pos)                          /*!< 0x00000001 */
#define ADC_CR_ADEN                               ADC_CR_ADEN_Msk                                   
#define ADC_CR_ADSTART_Pos                        (2U)
#define ADC_CR_ADSTART_Msk                        (0x1UL<<ADC_CR_ADSTART_Pos)                       /*!< 0x00000004 */
#define ADC_CR_ADSTART                            ADC_CR_ADSTART_Msk                                
#define ADC_CR_JADSTART_Pos                       (3U)
#define ADC_CR_JADSTART_Msk                       (0x1UL<<ADC_CR_JADSTART_Pos)                      /*!< 0x00000008 */
#define ADC_CR_JADSTART                           ADC_CR_JADSTART_Msk                               
#define ADC_CR_ADSTP_Pos                          (4U)
#define ADC_CR_ADSTP_Msk                          (0x1UL<<ADC_CR_ADSTP_Pos)                         /*!< 0x00000010 */
#define ADC_CR_ADSTP                              ADC_CR_ADSTP_Msk                                  
#define ADC_CR_RSTCAL_Pos                         (27U)
#define ADC_CR_RSTCAL_Msk                         (0x1UL<<ADC_CR_RSTCAL_Pos)                        /*!< 0x08000000 */
#define ADC_CR_RSTCAL                             ADC_CR_RSTCAL_Msk                                 
#define ADC_CR_ADCAL_Pos                          (31U)
#define ADC_CR_ADCAL_Msk                          (0x1UL<<ADC_CR_ADCAL_Pos)                         /*!< 0x80000000 */
#define ADC_CR_ADCAL                              ADC_CR_ADCAL_Msk                                  

/********************************* Bit definition for ADC_CFGR register *********************************************/
#define ADC_CFGR_DMAEN_Pos                        (0U)
#define ADC_CFGR_DMAEN_Msk                        (0x1UL<<ADC_CFGR_DMAEN_Pos)                       /*!< 0x00000001 */
#define ADC_CFGR_DMAEN                            ADC_CFGR_DMAEN_Msk                                
#define ADC_CFGR_SCAN_Pos                         (2U)
#define ADC_CFGR_SCAN_Msk                         (0x1UL<<ADC_CFGR_SCAN_Pos)                        /*!< 0x00000004 */
#define ADC_CFGR_SCAN                             ADC_CFGR_SCAN_Msk                                 
#define ADC_CFGR_RES_Pos                          (3U)
#define ADC_CFGR_RES_Msk                          (0x3UL<<ADC_CFGR_RES_Pos)                         /*!< 0x00000018 */
#define ADC_CFGR_RES                              ADC_CFGR_RES_Msk
#define ADC_CFGR_RES_0                            (0x1UL<<ADC_CFGR_RES_Pos)                         /*!< 0x00000008 */
#define ADC_CFGR_RES_1                            (0x2UL<<ADC_CFGR_RES_Pos)                         /*!< 0x00000010 */
#define ADC_CFGR_EXTSEL_Pos                       (5U)
#define ADC_CFGR_EXTSEL_Msk                       (0x1FUL<<ADC_CFGR_EXTSEL_Pos)                     /*!< 0x000003E0 */
#define ADC_CFGR_EXTSEL                           ADC_CFGR_EXTSEL_Msk
#define ADC_CFGR_EXTSEL_0                         (0x1UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000020 */
#define ADC_CFGR_EXTSEL_1                         (0x2UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000040 */
#define ADC_CFGR_EXTSEL_2                         (0x4UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000080 */
#define ADC_CFGR_EXTSEL_3                         (0x8UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000100 */
#define ADC_CFGR_EXTSEL_4                         (0x10UL<<ADC_CFGR_EXTSEL_Pos)                     /*!< 0x00000200 */
#define ADC_CFGR_EXTEN_Pos                        (10U)
#define ADC_CFGR_EXTEN_Msk                        (0x3UL<<ADC_CFGR_EXTEN_Pos)                       /*!< 0x00000C00 */
#define ADC_CFGR_EXTEN                            ADC_CFGR_EXTEN_Msk
#define ADC_CFGR_EXTEN_0                          (0x1UL<<ADC_CFGR_EXTEN_Pos)                       /*!< 0x00000400 */
#define ADC_CFGR_EXTEN_1                          (0x2UL<<ADC_CFGR_EXTEN_Pos)                       /*!< 0x00000800 */
#define ADC_CFGR_CONT_Pos                         (13U)
#define ADC_CFGR_CONT_Msk                         (0x1UL<<ADC_CFGR_CONT_Pos)                        /*!< 0x00002000 */
#define ADC_CFGR_CONT                             ADC_CFGR_CONT_Msk                                 
#define ADC_CFGR_AUTDLY_Pos                       (14U)
#define ADC_CFGR_AUTDLY_Msk                       (0x1UL<<ADC_CFGR_AUTDLY_Pos)                      /*!< 0x00004000 */
#define ADC_CFGR_AUTDLY                           ADC_CFGR_AUTDLY_Msk                               
#define ADC_CFGR_ALIGN_Pos                        (15U)
#define ADC_CFGR_ALIGN_Msk                        (0x1UL<<ADC_CFGR_ALIGN_Pos)                       /*!< 0x00008000 */
#define ADC_CFGR_ALIGN                            ADC_CFGR_ALIGN_Msk                                
#define ADC_CFGR_DISCEN_Pos                       (16U)
#define ADC_CFGR_DISCEN_Msk                       (0x1UL<<ADC_CFGR_DISCEN_Pos)                      /*!< 0x00010000 */
#define ADC_CFGR_DISCEN                           ADC_CFGR_DISCEN_Msk                               
#define ADC_CFGR_DISCNUM_Pos                      (17U)
#define ADC_CFGR_DISCNUM_Msk                      (0x7UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x000E0000 */
#define ADC_CFGR_DISCNUM                          ADC_CFGR_DISCNUM_Msk
#define ADC_CFGR_DISCNUM_0                        (0x1UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x00020000 */
#define ADC_CFGR_DISCNUM_1                        (0x2UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x00040000 */
#define ADC_CFGR_DISCNUM_2                        (0x4UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x00080000 */
#define ADC_CFGR_JDISCEN_Pos                      (20U)
#define ADC_CFGR_JDISCEN_Msk                      (0x1UL<<ADC_CFGR_JDISCEN_Pos)                     /*!< 0x00100000 */
#define ADC_CFGR_JDISCEN                          ADC_CFGR_JDISCEN_Msk                              
#define ADC_CFGR_AWD1SGL_Pos                      (22U)
#define ADC_CFGR_AWD1SGL_Msk                      (0x1UL<<ADC_CFGR_AWD1SGL_Pos)                     /*!< 0x00400000 */
#define ADC_CFGR_AWD1SGL                          ADC_CFGR_AWD1SGL_Msk                              
#define ADC_CFGR_AWD1EN_Pos                       (23U)
#define ADC_CFGR_AWD1EN_Msk                       (0x1UL<<ADC_CFGR_AWD1EN_Pos)                      /*!< 0x00800000 */
#define ADC_CFGR_AWD1EN                           ADC_CFGR_AWD1EN_Msk                               
#define ADC_CFGR_JAWD1EN_Pos                      (24U)
#define ADC_CFGR_JAWD1EN_Msk                      (0x1UL<<ADC_CFGR_JAWD1EN_Pos)                     /*!< 0x01000000 */
#define ADC_CFGR_JAWD1EN                          ADC_CFGR_JAWD1EN_Msk                              
#define ADC_CFGR_JAUTO_Pos                        (25U)
#define ADC_CFGR_JAUTO_Msk                        (0x1UL<<ADC_CFGR_JAUTO_Pos)                       /*!< 0x02000000 */
#define ADC_CFGR_JAUTO                            ADC_CFGR_JAUTO_Msk                                
#define ADC_CFGR_AWD1CH_Pos                       (26U)
#define ADC_CFGR_AWD1CH_Msk                       (0x1FUL<<ADC_CFGR_AWD1CH_Pos)                     /*!< 0x7C000000 */
#define ADC_CFGR_AWD1CH                           ADC_CFGR_AWD1CH_Msk
#define ADC_CFGR_AWD1CH_0                         (0x1UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x04000000 */
#define ADC_CFGR_AWD1CH_1                         (0x2UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x08000000 */
#define ADC_CFGR_AWD1CH_2                         (0x4UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x10000000 */
#define ADC_CFGR_AWD1CH_3                         (0x8UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x20000000 */
#define ADC_CFGR_AWD1CH_4                         (0x10UL<<ADC_CFGR_AWD1CH_Pos)                     /*!< 0x40000000 */

/********************************* Bit definition for ADC_CFGR2 register ********************************************/
#define ADC_CFGR2_ROVSE_Pos                       (0U)
#define ADC_CFGR2_ROVSE_Msk                       (0x1UL<<ADC_CFGR2_ROVSE_Pos)                      /*!< 0x00000001 */
#define ADC_CFGR2_ROVSE                           ADC_CFGR2_ROVSE_Msk                               
#define ADC_CFGR2_JOVSE_Pos                       (1U)
#define ADC_CFGR2_JOVSE_Msk                       (0x1UL<<ADC_CFGR2_JOVSE_Pos)                      /*!< 0x00000002 */
#define ADC_CFGR2_JOVSE                           ADC_CFGR2_JOVSE_Msk                               
#define ADC_CFGR2_OVSR_Pos                        (2U)
#define ADC_CFGR2_OVSR_Msk                        (0x7UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x0000001C */
#define ADC_CFGR2_OVSR                            ADC_CFGR2_OVSR_Msk
#define ADC_CFGR2_OVSR_0                          (0x1UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x00000004 */
#define ADC_CFGR2_OVSR_1                          (0x2UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x00000008 */
#define ADC_CFGR2_OVSR_2                          (0x4UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x00000010 */
#define ADC_CFGR2_OVSS_Pos                        (5U)
#define ADC_CFGR2_OVSS_Msk                        (0xFUL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x000001E0 */
#define ADC_CFGR2_OVSS                            ADC_CFGR2_OVSS_Msk
#define ADC_CFGR2_OVSS_0                          (0x1UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000020 */
#define ADC_CFGR2_OVSS_1                          (0x2UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000040 */
#define ADC_CFGR2_OVSS_2                          (0x4UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000080 */
#define ADC_CFGR2_OVSS_3                          (0x8UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000100 */
#define ADC_CFGR2_TROVS_Pos                       (9U)
#define ADC_CFGR2_TROVS_Msk                       (0x1UL<<ADC_CFGR2_TROVS_Pos)                      /*!< 0x00000200 */
#define ADC_CFGR2_TROVS                           ADC_CFGR2_TROVS_Msk                               
#define ADC_CFGR2_ROVSM_Pos                       (10U)
#define ADC_CFGR2_ROVSM_Msk                       (0x1UL<<ADC_CFGR2_ROVSM_Pos)                      /*!< 0x00000400 */
#define ADC_CFGR2_ROVSM                           ADC_CFGR2_ROVSM_Msk                               
#define ADC_CFGR2_GCOMP_Pos                       (16U)
#define ADC_CFGR2_GCOMP_Msk                       (0x1UL<<ADC_CFGR2_GCOMP_Pos)                      /*!< 0x00010000 */
#define ADC_CFGR2_GCOMP                           ADC_CFGR2_GCOMP_Msk                               
#define ADC_CFGR2_CALNUM_Pos                      (28U)
#define ADC_CFGR2_CALNUM_Msk                      (0x7UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x70000000 */
#define ADC_CFGR2_CALNUM                          ADC_CFGR2_CALNUM_Msk
#define ADC_CFGR2_CALNUM_0                        (0x1UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x10000000 */
#define ADC_CFGR2_CALNUM_1                        (0x2UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x20000000 */
#define ADC_CFGR2_CALNUM_2                        (0x4UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x40000000 */

/********************************* Bit definition for ADC_SMPR1 register ********************************************/
#define ADC_SMPR1_SMP0_Pos                        (0U)
#define ADC_SMPR1_SMP0_Msk                        (0x7UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000007 */
#define ADC_SMPR1_SMP0                            ADC_SMPR1_SMP0_Msk
#define ADC_SMPR1_SMP0_0                          (0x1UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000001 */
#define ADC_SMPR1_SMP0_1                          (0x2UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000002 */
#define ADC_SMPR1_SMP0_2                          (0x4UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000004 */
#define ADC_SMPR1_SMP1_Pos                        (3U)
#define ADC_SMPR1_SMP1_Msk                        (0x7UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000038 */
#define ADC_SMPR1_SMP1                            ADC_SMPR1_SMP1_Msk
#define ADC_SMPR1_SMP1_0                          (0x1UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000008 */
#define ADC_SMPR1_SMP1_1                          (0x2UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000010 */
#define ADC_SMPR1_SMP1_2                          (0x4UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000020 */
#define ADC_SMPR1_SMP2_Pos                        (6U)
#define ADC_SMPR1_SMP2_Msk                        (0x7UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x000001C0 */
#define ADC_SMPR1_SMP2                            ADC_SMPR1_SMP2_Msk
#define ADC_SMPR1_SMP2_0                          (0x1UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x00000040 */
#define ADC_SMPR1_SMP2_1                          (0x2UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x00000080 */
#define ADC_SMPR1_SMP2_2                          (0x4UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x00000100 */
#define ADC_SMPR1_SMP3_Pos                        (9U)
#define ADC_SMPR1_SMP3_Msk                        (0x7UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000E00 */
#define ADC_SMPR1_SMP3                            ADC_SMPR1_SMP3_Msk
#define ADC_SMPR1_SMP3_0                          (0x1UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000200 */
#define ADC_SMPR1_SMP3_1                          (0x2UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000400 */
#define ADC_SMPR1_SMP3_2                          (0x4UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000800 */
#define ADC_SMPR1_SMP4_Pos                        (12U)
#define ADC_SMPR1_SMP4_Msk                        (0x7UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00007000 */
#define ADC_SMPR1_SMP4                            ADC_SMPR1_SMP4_Msk
#define ADC_SMPR1_SMP4_0                          (0x1UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00001000 */
#define ADC_SMPR1_SMP4_1                          (0x2UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00002000 */
#define ADC_SMPR1_SMP4_2                          (0x4UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00004000 */
#define ADC_SMPR1_SMP5_Pos                        (15U)
#define ADC_SMPR1_SMP5_Msk                        (0x7UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00038000 */
#define ADC_SMPR1_SMP5                            ADC_SMPR1_SMP5_Msk
#define ADC_SMPR1_SMP5_0                          (0x1UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00008000 */
#define ADC_SMPR1_SMP5_1                          (0x2UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00010000 */
#define ADC_SMPR1_SMP5_2                          (0x4UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00020000 */
#define ADC_SMPR1_SMP6_Pos                        (18U)
#define ADC_SMPR1_SMP6_Msk                        (0x7UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x001C0000 */
#define ADC_SMPR1_SMP6                            ADC_SMPR1_SMP6_Msk
#define ADC_SMPR1_SMP6_0                          (0x1UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x00040000 */
#define ADC_SMPR1_SMP6_1                          (0x2UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x00080000 */
#define ADC_SMPR1_SMP6_2                          (0x4UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x00100000 */
#define ADC_SMPR1_SMP7_Pos                        (21U)
#define ADC_SMPR1_SMP7_Msk                        (0x7UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00E00000 */
#define ADC_SMPR1_SMP7                            ADC_SMPR1_SMP7_Msk
#define ADC_SMPR1_SMP7_0                          (0x1UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00200000 */
#define ADC_SMPR1_SMP7_1                          (0x2UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00400000 */
#define ADC_SMPR1_SMP7_2                          (0x4UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00800000 */
#define ADC_SMPR1_SMP8_Pos                        (24U)
#define ADC_SMPR1_SMP8_Msk                        (0x7UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x07000000 */
#define ADC_SMPR1_SMP8                            ADC_SMPR1_SMP8_Msk
#define ADC_SMPR1_SMP8_0                          (0x1UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x01000000 */
#define ADC_SMPR1_SMP8_1                          (0x2UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x02000000 */
#define ADC_SMPR1_SMP8_2                          (0x4UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x04000000 */
#define ADC_SMPR1_SMP9_Pos                        (27U)
#define ADC_SMPR1_SMP9_Msk                        (0x7UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x38000000 */
#define ADC_SMPR1_SMP9                            ADC_SMPR1_SMP9_Msk
#define ADC_SMPR1_SMP9_0                          (0x1UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x08000000 */
#define ADC_SMPR1_SMP9_1                          (0x2UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x10000000 */
#define ADC_SMPR1_SMP9_2                          (0x4UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x20000000 */
#define ADC_SMPR1_SMPPLUS_Pos                     (31U)
#define ADC_SMPR1_SMPPLUS_Msk                     (0x1UL << ADC_SMPR1_SMPPLUS_Pos)                  /*!< 0x80000000 */
#define ADC_SMPR1_SMPPLUS                         ADC_SMPR1_SMPPLUS_Msk                             /*!< ADC channels sampling time additional setting */

/********************************* Bit definition for ADC_SMPR2 register ********************************************/
#define ADC_SMPR2_SMP10_Pos                       (0U)
#define ADC_SMPR2_SMP10_Msk                       (0x7UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000007 */
#define ADC_SMPR2_SMP10                           ADC_SMPR2_SMP10_Msk
#define ADC_SMPR2_SMP10_0                         (0x1UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000001 */
#define ADC_SMPR2_SMP10_1                         (0x2UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000002 */
#define ADC_SMPR2_SMP10_2                         (0x4UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000004 */
#define ADC_SMPR2_SMP11_Pos                       (3U)
#define ADC_SMPR2_SMP11_Msk                       (0x7UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000038 */
#define ADC_SMPR2_SMP11                           ADC_SMPR2_SMP11_Msk
#define ADC_SMPR2_SMP11_0                         (0x1UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000008 */
#define ADC_SMPR2_SMP11_1                         (0x2UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000010 */
#define ADC_SMPR2_SMP11_2                         (0x4UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000020 */
#define ADC_SMPR2_SMP12_Pos                       (6U)
#define ADC_SMPR2_SMP12_Msk                       (0x7UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x000001C0 */
#define ADC_SMPR2_SMP12                           ADC_SMPR2_SMP12_Msk
#define ADC_SMPR2_SMP12_0                         (0x1UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x00000040 */
#define ADC_SMPR2_SMP12_1                         (0x2UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x00000080 */
#define ADC_SMPR2_SMP12_2                         (0x4UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x00000100 */
#define ADC_SMPR2_SMP13_Pos                       (9U)
#define ADC_SMPR2_SMP13_Msk                       (0x7UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000E00 */
#define ADC_SMPR2_SMP13                           ADC_SMPR2_SMP13_Msk
#define ADC_SMPR2_SMP13_0                         (0x1UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000200 */
#define ADC_SMPR2_SMP13_1                         (0x2UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000400 */
#define ADC_SMPR2_SMP13_2                         (0x4UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000800 */
#define ADC_SMPR2_SMP14_Pos                       (12U)
#define ADC_SMPR2_SMP14_Msk                       (0x7UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00007000 */
#define ADC_SMPR2_SMP14                           ADC_SMPR2_SMP14_Msk
#define ADC_SMPR2_SMP14_0                         (0x1UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00001000 */
#define ADC_SMPR2_SMP14_1                         (0x2UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00002000 */
#define ADC_SMPR2_SMP14_2                         (0x4UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00004000 */
#define ADC_SMPR2_SMP15_Pos                       (15U)
#define ADC_SMPR2_SMP15_Msk                       (0x7UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00038000 */
#define ADC_SMPR2_SMP15                           ADC_SMPR2_SMP15_Msk
#define ADC_SMPR2_SMP15_0                         (0x1UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00008000 */
#define ADC_SMPR2_SMP15_1                         (0x2UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00010000 */
#define ADC_SMPR2_SMP15_2                         (0x4UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00020000 */
#define ADC_SMPR2_SMP16_Pos                       (18U)
#define ADC_SMPR2_SMP16_Msk                       (0x7UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x001C0000 */
#define ADC_SMPR2_SMP16                           ADC_SMPR2_SMP16_Msk
#define ADC_SMPR2_SMP16_0                         (0x1UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x00040000 */
#define ADC_SMPR2_SMP16_1                         (0x2UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x00080000 */
#define ADC_SMPR2_SMP16_2                         (0x4UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x00100000 */
#define ADC_SMPR2_SMP17_Pos                       (21U)
#define ADC_SMPR2_SMP17_Msk                       (0x7UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00E00000 */
#define ADC_SMPR2_SMP17                           ADC_SMPR2_SMP17_Msk
#define ADC_SMPR2_SMP17_0                         (0x1UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00200000 */
#define ADC_SMPR2_SMP17_1                         (0x2UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00400000 */
#define ADC_SMPR2_SMP17_2                         (0x4UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00800000 */
#define ADC_SMPR2_SMP18_Pos                       (24U)
#define ADC_SMPR2_SMP18_Msk                       (0x7UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x07000000 */
#define ADC_SMPR2_SMP18                           ADC_SMPR2_SMP18_Msk
#define ADC_SMPR2_SMP18_0                         (0x1UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x01000000 */
#define ADC_SMPR2_SMP18_1                         (0x2UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x02000000 */
#define ADC_SMPR2_SMP18_2                         (0x4UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x04000000 */
#define ADC_SMPR2_SMP19_Pos                       (27U)
#define ADC_SMPR2_SMP19_Msk                       (0x7UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x38000000 */
#define ADC_SMPR2_SMP19                           ADC_SMPR2_SMP19_Msk
#define ADC_SMPR2_SMP19_0                         (0x1UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x08000000 */
#define ADC_SMPR2_SMP19_1                         (0x2UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x10000000 */
#define ADC_SMPR2_SMP19_2                         (0x4UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x20000000 */

/********************************* Bit definition for ADC_TR1 register **********************************************/
#define ADC_TR1_LT1_Pos                           (0U)
#define ADC_TR1_LT1_Msk                           (0xFFFUL<<ADC_TR1_LT1_Pos)                        /*!< 0x00000FFF */
#define ADC_TR1_LT1                               ADC_TR1_LT1_Msk
#define ADC_TR1_AWDFILT_Pos                       (12U)
#define ADC_TR1_AWDFILT_Msk                       (0x7UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00007000 */
#define ADC_TR1_AWDFILT                           ADC_TR1_AWDFILT_Msk
#define ADC_TR1_AWDFILT_0                         (0x1UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00001000 */
#define ADC_TR1_AWDFILT_1                         (0x2UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00002000 */
#define ADC_TR1_AWDFILT_2                         (0x4UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00004000 */
#define ADC_TR1_HT1_Pos                           (16U)
#define ADC_TR1_HT1_Msk                           (0xFFFUL<<ADC_TR1_HT1_Pos)                        /*!< 0x0FFF0000 */
#define ADC_TR1_HT1                               ADC_TR1_HT1_Msk

/********************************* Bit definition for ADC_TR2 register **********************************************/
#define ADC_TR2_LT2_Pos                           (0U)
#define ADC_TR2_LT2_Msk                           (0xFFUL<<ADC_TR2_LT2_Pos)                         /*!< 0x000000FF */
#define ADC_TR2_LT2                               ADC_TR2_LT2_Msk
#define ADC_TR2_HT2_Pos                           (16U)
#define ADC_TR2_HT2_Msk                           (0xFFUL<<ADC_TR2_HT2_Pos)                         /*!< 0x00FF0000 */
#define ADC_TR2_HT2                               ADC_TR2_HT2_Msk

/********************************* Bit definition for ADC_TR3 register **********************************************/
#define ADC_TR3_LT3_Pos                           (0U)
#define ADC_TR3_LT3_Msk                           (0xFFUL<<ADC_TR3_LT3_Pos)                         /*!< 0x000000FF */
#define ADC_TR3_LT3                               ADC_TR3_LT3_Msk
#define ADC_TR3_HT3_Pos                           (16U)
#define ADC_TR3_HT3_Msk                           (0xFFUL<<ADC_TR3_HT3_Pos)                         /*!< 0x00FF0000 */
#define ADC_TR3_HT3                               ADC_TR3_HT3_Msk

/********************************* Bit definition for ADC_SQR1 register *********************************************/
#define ADC_SQR1_L_Pos                            (0U)
#define ADC_SQR1_L_Msk                            (0xFUL<<ADC_SQR1_L_Pos)                           /*!< 0x0000000F */
#define ADC_SQR1_L                                ADC_SQR1_L_Msk
#define ADC_SQR1_L_0                              (0x1UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000001 */
#define ADC_SQR1_L_1                              (0x2UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000002 */
#define ADC_SQR1_L_2                              (0x4UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000004 */
#define ADC_SQR1_L_3                              (0x8UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000008 */
#define ADC_SQR1_SQ1_Pos                          (6U)
#define ADC_SQR1_SQ1_Msk                          (0x1FUL<<ADC_SQR1_SQ1_Pos)                        /*!< 0x000007C0 */
#define ADC_SQR1_SQ1                              ADC_SQR1_SQ1_Msk
#define ADC_SQR1_SQ1_0                            (0x1UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000040 */
#define ADC_SQR1_SQ1_1                            (0x2UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000080 */
#define ADC_SQR1_SQ1_2                            (0x4UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000100 */
#define ADC_SQR1_SQ1_3                            (0x8UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000200 */
#define ADC_SQR1_SQ1_4                            (0x10UL<<ADC_SQR1_SQ1_Pos)                        /*!< 0x00000400 */
#define ADC_SQR1_SQ2_Pos                          (12U)
#define ADC_SQR1_SQ2_Msk                          (0x1FUL<<ADC_SQR1_SQ2_Pos)                        /*!< 0x0001F000 */
#define ADC_SQR1_SQ2                              ADC_SQR1_SQ2_Msk
#define ADC_SQR1_SQ2_0                            (0x1UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00001000 */
#define ADC_SQR1_SQ2_1                            (0x2UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00002000 */
#define ADC_SQR1_SQ2_2                            (0x4UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00004000 */
#define ADC_SQR1_SQ2_3                            (0x8UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00008000 */
#define ADC_SQR1_SQ2_4                            (0x10UL<<ADC_SQR1_SQ2_Pos)                        /*!< 0x00010000 */
#define ADC_SQR1_SQ3_Pos                          (18U)
#define ADC_SQR1_SQ3_Msk                          (0x1FUL<<ADC_SQR1_SQ3_Pos)                        /*!< 0x007C0000 */
#define ADC_SQR1_SQ3                              ADC_SQR1_SQ3_Msk
#define ADC_SQR1_SQ3_0                            (0x1UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00040000 */
#define ADC_SQR1_SQ3_1                            (0x2UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00080000 */
#define ADC_SQR1_SQ3_2                            (0x4UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00100000 */
#define ADC_SQR1_SQ3_3                            (0x8UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00200000 */
#define ADC_SQR1_SQ3_4                            (0x10UL<<ADC_SQR1_SQ3_Pos)                        /*!< 0x00400000 */
#define ADC_SQR1_SQ4_Pos                          (24U)
#define ADC_SQR1_SQ4_Msk                          (0x1FUL<<ADC_SQR1_SQ4_Pos)                        /*!< 0x1F000000 */
#define ADC_SQR1_SQ4                              ADC_SQR1_SQ4_Msk
#define ADC_SQR1_SQ4_0                            (0x1UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x01000000 */
#define ADC_SQR1_SQ4_1                            (0x2UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x02000000 */
#define ADC_SQR1_SQ4_2                            (0x4UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x04000000 */
#define ADC_SQR1_SQ4_3                            (0x8UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x08000000 */
#define ADC_SQR1_SQ4_4                            (0x10UL<<ADC_SQR1_SQ4_Pos)                        /*!< 0x10000000 */

/********************************* Bit definition for ADC_SQR2 register *********************************************/
#define ADC_SQR2_SQ5_Pos                          (0U)
#define ADC_SQR2_SQ5_Msk                          (0x1FUL<<ADC_SQR2_SQ5_Pos)                        /*!< 0x0000001F */
#define ADC_SQR2_SQ5                              ADC_SQR2_SQ5_Msk
#define ADC_SQR2_SQ5_0                            (0x1UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000001 */
#define ADC_SQR2_SQ5_1                            (0x2UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000002 */
#define ADC_SQR2_SQ5_2                            (0x4UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000004 */
#define ADC_SQR2_SQ5_3                            (0x8UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000008 */
#define ADC_SQR2_SQ5_4                            (0x10UL<<ADC_SQR2_SQ5_Pos)                        /*!< 0x00000010 */
#define ADC_SQR2_SQ6_Pos                          (6U)
#define ADC_SQR2_SQ6_Msk                          (0x1FUL<<ADC_SQR2_SQ6_Pos)                        /*!< 0x000007C0 */
#define ADC_SQR2_SQ6                              ADC_SQR2_SQ6_Msk
#define ADC_SQR2_SQ6_0                            (0x1UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000040 */
#define ADC_SQR2_SQ6_1                            (0x2UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000080 */
#define ADC_SQR2_SQ6_2                            (0x4UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000100 */
#define ADC_SQR2_SQ6_3                            (0x8UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000200 */
#define ADC_SQR2_SQ6_4                            (0x10UL<<ADC_SQR2_SQ6_Pos)                        /*!< 0x00000400 */
#define ADC_SQR2_SQ7_Pos                          (12U)
#define ADC_SQR2_SQ7_Msk                          (0x1FUL<<ADC_SQR2_SQ7_Pos)                        /*!< 0x0001F000 */
#define ADC_SQR2_SQ7                              ADC_SQR2_SQ7_Msk
#define ADC_SQR2_SQ7_0                            (0x1UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00001000 */
#define ADC_SQR2_SQ7_1                            (0x2UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00002000 */
#define ADC_SQR2_SQ7_2                            (0x4UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00004000 */
#define ADC_SQR2_SQ7_3                            (0x8UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00008000 */
#define ADC_SQR2_SQ7_4                            (0x10UL<<ADC_SQR2_SQ7_Pos)                        /*!< 0x00010000 */
#define ADC_SQR2_SQ8_Pos                          (18U)
#define ADC_SQR2_SQ8_Msk                          (0x1FUL<<ADC_SQR2_SQ8_Pos)                        /*!< 0x007C0000 */
#define ADC_SQR2_SQ8                              ADC_SQR2_SQ8_Msk
#define ADC_SQR2_SQ8_0                            (0x1UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00040000 */
#define ADC_SQR2_SQ8_1                            (0x2UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00080000 */
#define ADC_SQR2_SQ8_2                            (0x4UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00100000 */
#define ADC_SQR2_SQ8_3                            (0x8UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00200000 */
#define ADC_SQR2_SQ8_4                            (0x10UL<<ADC_SQR2_SQ8_Pos)                        /*!< 0x00400000 */
#define ADC_SQR2_SQ9_Pos                          (24U)
#define ADC_SQR2_SQ9_Msk                          (0x1FUL<<ADC_SQR2_SQ9_Pos)                        /*!< 0x1F000000 */
#define ADC_SQR2_SQ9                              ADC_SQR2_SQ9_Msk
#define ADC_SQR2_SQ9_0                            (0x1UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x01000000 */
#define ADC_SQR2_SQ9_1                            (0x2UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x02000000 */
#define ADC_SQR2_SQ9_2                            (0x4UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x04000000 */
#define ADC_SQR2_SQ9_3                            (0x8UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x08000000 */
#define ADC_SQR2_SQ9_4                            (0x10UL<<ADC_SQR2_SQ9_Pos)                        /*!< 0x10000000 */

/********************************* Bit definition for ADC_SQR3 register *********************************************/
#define ADC_SQR3_SQ10_Pos                         (0U)
#define ADC_SQR3_SQ10_Msk                         (0x1FUL<<ADC_SQR3_SQ10_Pos)                       /*!< 0x0000001F */
#define ADC_SQR3_SQ10                             ADC_SQR3_SQ10_Msk
#define ADC_SQR3_SQ10_0                           (0x1UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000001 */
#define ADC_SQR3_SQ10_1                           (0x2UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000002 */
#define ADC_SQR3_SQ10_2                           (0x4UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000004 */
#define ADC_SQR3_SQ10_3                           (0x8UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000008 */
#define ADC_SQR3_SQ10_4                           (0x10UL<<ADC_SQR3_SQ10_Pos)                       /*!< 0x00000010 */
#define ADC_SQR3_SQ11_Pos                         (6U)
#define ADC_SQR3_SQ11_Msk                         (0x1FUL<<ADC_SQR3_SQ11_Pos)                       /*!< 0x000007C0 */
#define ADC_SQR3_SQ11                             ADC_SQR3_SQ11_Msk
#define ADC_SQR3_SQ11_0                           (0x1UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000040 */
#define ADC_SQR3_SQ11_1                           (0x2UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000080 */
#define ADC_SQR3_SQ11_2                           (0x4UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000100 */
#define ADC_SQR3_SQ11_3                           (0x8UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000200 */
#define ADC_SQR3_SQ11_4                           (0x10UL<<ADC_SQR3_SQ11_Pos)                       /*!< 0x00000400 */
#define ADC_SQR3_SQ12_Pos                         (12U)
#define ADC_SQR3_SQ12_Msk                         (0x1FUL<<ADC_SQR3_SQ12_Pos)                       /*!< 0x0001F000 */
#define ADC_SQR3_SQ12                             ADC_SQR3_SQ12_Msk
#define ADC_SQR3_SQ12_0                           (0x1UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00001000 */
#define ADC_SQR3_SQ12_1                           (0x2UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00002000 */
#define ADC_SQR3_SQ12_2                           (0x4UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00004000 */
#define ADC_SQR3_SQ12_3                           (0x8UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00008000 */
#define ADC_SQR3_SQ12_4                           (0x10UL<<ADC_SQR3_SQ12_Pos)                       /*!< 0x00010000 */
#define ADC_SQR3_SQ13_Pos                         (18U)
#define ADC_SQR3_SQ13_Msk                         (0x1FUL<<ADC_SQR3_SQ13_Pos)                       /*!< 0x007C0000 */
#define ADC_SQR3_SQ13                             ADC_SQR3_SQ13_Msk
#define ADC_SQR3_SQ13_0                           (0x1UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00040000 */
#define ADC_SQR3_SQ13_1                           (0x2UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00080000 */
#define ADC_SQR3_SQ13_2                           (0x4UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00100000 */
#define ADC_SQR3_SQ13_3                           (0x8UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00200000 */
#define ADC_SQR3_SQ13_4                           (0x10UL<<ADC_SQR3_SQ13_Pos)                       /*!< 0x00400000 */
#define ADC_SQR3_SQ14_Pos                         (24U)
#define ADC_SQR3_SQ14_Msk                         (0x1FUL<<ADC_SQR3_SQ14_Pos)                       /*!< 0x1F000000 */
#define ADC_SQR3_SQ14                             ADC_SQR3_SQ14_Msk
#define ADC_SQR3_SQ14_0                           (0x1UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x01000000 */
#define ADC_SQR3_SQ14_1                           (0x2UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x02000000 */
#define ADC_SQR3_SQ14_2                           (0x4UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x04000000 */
#define ADC_SQR3_SQ14_3                           (0x8UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x08000000 */
#define ADC_SQR3_SQ14_4                           (0x10UL<<ADC_SQR3_SQ14_Pos)                       /*!< 0x10000000 */

/********************************* Bit definition for ADC_SQR4 register *********************************************/
#define ADC_SQR4_SQ15_Pos                         (0U)
#define ADC_SQR4_SQ15_Msk                         (0x1FUL<<ADC_SQR4_SQ15_Pos)                       /*!< 0x0000001F */
#define ADC_SQR4_SQ15                             ADC_SQR4_SQ15_Msk
#define ADC_SQR4_SQ15_0                           (0x1UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000001 */
#define ADC_SQR4_SQ15_1                           (0x2UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000002 */
#define ADC_SQR4_SQ15_2                           (0x4UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000004 */
#define ADC_SQR4_SQ15_3                           (0x8UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000008 */
#define ADC_SQR4_SQ15_4                           (0x10UL<<ADC_SQR4_SQ15_Pos)                       /*!< 0x00000010 */
#define ADC_SQR4_SQ16_Pos                         (6U)
#define ADC_SQR4_SQ16_Msk                         (0x1FUL<<ADC_SQR4_SQ16_Pos)                       /*!< 0x000007C0 */
#define ADC_SQR4_SQ16                             ADC_SQR4_SQ16_Msk
#define ADC_SQR4_SQ16_0                           (0x1UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000040 */
#define ADC_SQR4_SQ16_1                           (0x2UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000080 */
#define ADC_SQR4_SQ16_2                           (0x4UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000100 */
#define ADC_SQR4_SQ16_3                           (0x8UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000200 */
#define ADC_SQR4_SQ16_4                           (0x10UL<<ADC_SQR4_SQ16_Pos)                       /*!< 0x00000400 */

/********************************* Bit definition for ADC_DR register ***********************************************/
#define ADC_DR_RDATA_Pos                          (0U)
#define ADC_DR_RDATA_Msk                          (0xFFFFUL<<ADC_DR_RDATA_Pos)                      /*!< 0x0000FFFF */
#define ADC_DR_RDATA                              ADC_DR_RDATA_Msk
#define ADC_DR_ADC2DATA_Pos                       (16U)
#define ADC_DR_ADC2DATA_Msk                       (0xFFFFUL<<ADC_DR_ADC2DATA_Pos)                   /*!< 0xFFFF0000 */
#define ADC_DR_ADC2DATA                           ADC_DR_ADC2DATA_Msk

/********************************* Bit definition for ADC_JSQR register *********************************************/
#define ADC_JSQR_JL_Pos                           (0U)
#define ADC_JSQR_JL_Msk                           (0x3UL<<ADC_JSQR_JL_Pos)                          /*!< 0x00000003 */
#define ADC_JSQR_JL                               ADC_JSQR_JL_Msk
#define ADC_JSQR_JL_0                             (0x1UL<<ADC_JSQR_JL_Pos)                          /*!< 0x00000001 */
#define ADC_JSQR_JL_1                             (0x2UL<<ADC_JSQR_JL_Pos)                          /*!< 0x00000002 */
#define ADC_JSQR_JEXTSEL_Pos                      (2U)
#define ADC_JSQR_JEXTSEL_Msk                      (0x1FUL<<ADC_JSQR_JEXTSEL_Pos)                    /*!< 0x0000007C */
#define ADC_JSQR_JEXTSEL                          ADC_JSQR_JEXTSEL_Msk
#define ADC_JSQR_JEXTSEL_0                        (0x1UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000004 */
#define ADC_JSQR_JEXTSEL_1                        (0x2UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000008 */
#define ADC_JSQR_JEXTSEL_2                        (0x4UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000010 */
#define ADC_JSQR_JEXTSEL_3                        (0x8UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000020 */
#define ADC_JSQR_JEXTSEL_4                        (0x10UL<<ADC_JSQR_JEXTSEL_Pos)                    /*!< 0x00000040 */
#define ADC_JSQR_JEXTEN_Pos                       (7U)
#define ADC_JSQR_JEXTEN_Msk                       (0x3UL<<ADC_JSQR_JEXTEN_Pos)                      /*!< 0x00000180 */
#define ADC_JSQR_JEXTEN                           ADC_JSQR_JEXTEN_Msk
#define ADC_JSQR_JEXTEN_0                         (0x1UL<<ADC_JSQR_JEXTEN_Pos)                      /*!< 0x00000080 */
#define ADC_JSQR_JEXTEN_1                         (0x2UL<<ADC_JSQR_JEXTEN_Pos)                      /*!< 0x00000100 */
#define ADC_JSQR_JSQ1_Pos                         (9U)
#define ADC_JSQR_JSQ1_Msk                         (0x1FUL<<ADC_JSQR_JSQ1_Pos)                       /*!< 0x00003E00 */
#define ADC_JSQR_JSQ1                             ADC_JSQR_JSQ1_Msk
#define ADC_JSQR_JSQ1_0                           (0x1UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000200 */
#define ADC_JSQR_JSQ1_1                           (0x2UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000400 */
#define ADC_JSQR_JSQ1_2                           (0x4UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000800 */
#define ADC_JSQR_JSQ1_3                           (0x8UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00001000 */
#define ADC_JSQR_JSQ1_4                           (0x10UL<<ADC_JSQR_JSQ1_Pos)                       /*!< 0x00002000 */
#define ADC_JSQR_JSQ2_Pos                         (15U)
#define ADC_JSQR_JSQ2_Msk                         (0x1FUL<<ADC_JSQR_JSQ2_Pos)                       /*!< 0x000F8000 */
#define ADC_JSQR_JSQ2                             ADC_JSQR_JSQ2_Msk
#define ADC_JSQR_JSQ2_0                           (0x1UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00008000 */
#define ADC_JSQR_JSQ2_1                           (0x2UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00010000 */
#define ADC_JSQR_JSQ2_2                           (0x4UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00020000 */
#define ADC_JSQR_JSQ2_3                           (0x8UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00040000 */
#define ADC_JSQR_JSQ2_4                           (0x10UL<<ADC_JSQR_JSQ2_Pos)                       /*!< 0x00080000 */
#define ADC_JSQR_JSQ3_Pos                         (21U)
#define ADC_JSQR_JSQ3_Msk                         (0x1FUL<<ADC_JSQR_JSQ3_Pos)                       /*!< 0x03E00000 */
#define ADC_JSQR_JSQ3                             ADC_JSQR_JSQ3_Msk
#define ADC_JSQR_JSQ3_0                           (0x1UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00200000 */
#define ADC_JSQR_JSQ3_1                           (0x2UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00400000 */
#define ADC_JSQR_JSQ3_2                           (0x4UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00800000 */
#define ADC_JSQR_JSQ3_3                           (0x8UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x01000000 */
#define ADC_JSQR_JSQ3_4                           (0x10UL<<ADC_JSQR_JSQ3_Pos)                       /*!< 0x02000000 */
#define ADC_JSQR_JSQ4_Pos                         (27U)
#define ADC_JSQR_JSQ4_Msk                         (0x1FUL<<ADC_JSQR_JSQ4_Pos)                       /*!< 0xF8000000 */
#define ADC_JSQR_JSQ4                             ADC_JSQR_JSQ4_Msk
#define ADC_JSQR_JSQ4_0                           (0x1UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x08000000 */
#define ADC_JSQR_JSQ4_1                           (0x2UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x10000000 */
#define ADC_JSQR_JSQ4_2                           (0x4UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x20000000 */
#define ADC_JSQR_JSQ4_3                           (0x8UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x40000000 */
#define ADC_JSQR_JSQ4_4                           (0x10UL<<ADC_JSQR_JSQ4_Pos)                       /*!< 0x80000000 */

/********************************* Bit definition for ADC_OFR1 register *********************************************/
#define ADC_OFR1_OFFSET1_Pos                      (0U)
#define ADC_OFR1_OFFSET1_Msk                      (0xFFFUL<<ADC_OFR1_OFFSET1_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR1_OFFSET1                          ADC_OFR1_OFFSET1_Msk
#define ADC_OFR1_OFFSETPOS_Pos                    (24U)
#define ADC_OFR1_OFFSETPOS_Msk                    (0x1UL<<ADC_OFR1_OFFSETPOS_Pos)                   /*!< 0x01000000 */
#define ADC_OFR1_OFFSETPOS                        ADC_OFR1_OFFSETPOS_Msk                            
#define ADC_OFR1_SATEN_Pos                        (25U)
#define ADC_OFR1_SATEN_Msk                        (0x1UL<<ADC_OFR1_SATEN_Pos)                       /*!< 0x02000000 */
#define ADC_OFR1_SATEN                            ADC_OFR1_SATEN_Msk                                
#define ADC_OFR1_OFFSET1_CH_Pos                   (26U)
#define ADC_OFR1_OFFSET1_CH_Msk                   (0x1FUL<<ADC_OFR1_OFFSET1_CH_Pos)                 /*!< 0x7C000000 */
#define ADC_OFR1_OFFSET1_CH                       ADC_OFR1_OFFSET1_CH_Msk
#define ADC_OFR1_OFFSET1_CH_0                     (0x1UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR1_OFFSET1_CH_1                     (0x2UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR1_OFFSET1_CH_2                     (0x4UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR1_OFFSET1_CH_3                     (0x8UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x20000000 */
#define ADC_OFR1_OFFSET1_CH_4                     (0x10UL<<ADC_OFR1_OFFSET1_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR1_OFFSET1_EN_Pos                   (31U)
#define ADC_OFR1_OFFSET1_EN_Msk                   (0x1UL<<ADC_OFR1_OFFSET1_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR1_OFFSET1_EN                       ADC_OFR1_OFFSET1_EN_Msk                           

/********************************* Bit definition for ADC_OFR2 register *********************************************/
#define ADC_OFR2_OFFSET2_Pos                      (0U)
#define ADC_OFR2_OFFSET2_Msk                      (0xFFFUL<<ADC_OFR2_OFFSET2_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR2_OFFSET2                          ADC_OFR2_OFFSET2_Msk
#define ADC_OFR2_OFFSET2_CH_Pos                   (26U)
#define ADC_OFR2_OFFSET2_CH_Msk                   (0x1FUL<<ADC_OFR2_OFFSET2_CH_Pos)                 /*!< 0x7C000000 */
#define ADC_OFR2_OFFSET2_CH                       ADC_OFR2_OFFSET2_CH_Msk
#define ADC_OFR2_OFFSET2_CH_0                     (0x1UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR2_OFFSET2_CH_1                     (0x2UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR2_OFFSET2_CH_2                     (0x4UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR2_OFFSET2_CH_3                     (0x8UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x20000000 */
#define ADC_OFR2_OFFSET2_CH_4                     (0x10UL<<ADC_OFR2_OFFSET2_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR2_OFFSET2_EN_Pos                   (31U)
#define ADC_OFR2_OFFSET2_EN_Msk                   (0x1UL<<ADC_OFR2_OFFSET2_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR2_OFFSET2_EN                       ADC_OFR2_OFFSET2_EN_Msk                           

/********************************* Bit definition for ADC_OFR3 register *********************************************/
#define ADC_OFR3_OFFSET3_Pos                      (0U)
#define ADC_OFR3_OFFSET3_Msk                      (0xFFFUL<<ADC_OFR3_OFFSET3_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR3_OFFSET3                          ADC_OFR3_OFFSET3_Msk
#define ADC_OFR3_OFFSET3_CH_Pos                   (26U)
#define ADC_OFR3_OFFSET3_CH_Msk                   (0x1FUL<<ADC_OFR3_OFFSET3_CH_Pos)                 /*!< 0x7C000000 */
#define ADC_OFR3_OFFSET3_CH                       ADC_OFR3_OFFSET3_CH_Msk
#define ADC_OFR3_OFFSET3_CH_0                     (0x1UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR3_OFFSET3_CH_1                     (0x2UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR3_OFFSET3_CH_2                     (0x4UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR3_OFFSET3_CH_3                     (0x8UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x20000000 */
#define ADC_OFR3_OFFSET3_CH_4                     (0x10UL<<ADC_OFR3_OFFSET3_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR3_OFFSET3_EN_Pos                   (31U)
#define ADC_OFR3_OFFSET3_EN_Msk                   (0x1UL<<ADC_OFR3_OFFSET3_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR3_OFFSET3_EN                       ADC_OFR3_OFFSET3_EN_Msk                           

/********************************* Bit definition for ADC_OFR4 register *********************************************/
#define ADC_OFR4_OFFSET4_Pos                      (0U)
#define ADC_OFR4_OFFSET4_Msk                      (0xFFFUL<<ADC_OFR4_OFFSET4_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR4_OFFSET4                          ADC_OFR4_OFFSET4_Msk
#define ADC_OFR4_OFFSET4_CH_Pos                   (26U)
#define ADC_OFR4_OFFSET4_CH_Msk                   (0x1FUL<<ADC_OFR4_OFFSET4_CH_Pos)                 /*!< 0x7C000000 */
#define ADC_OFR4_OFFSET4_CH                       ADC_OFR4_OFFSET4_CH_Msk
#define ADC_OFR4_OFFSET4_CH_0                     (0x1UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR4_OFFSET4_CH_1                     (0x2UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR4_OFFSET4_CH_2                     (0x4UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR4_OFFSET4_CH_3                     (0x8UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x20000000 */
#define ADC_OFR4_OFFSET4_CH_4                     (0x10UL<<ADC_OFR4_OFFSET4_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR4_OFFSET4_EN_Pos                   (31U)
#define ADC_OFR4_OFFSET4_EN_Msk                   (0x1UL<<ADC_OFR4_OFFSET4_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR4_OFFSET4_EN                       ADC_OFR4_OFFSET4_EN_Msk                           

/********************************* Bit definition for ADC_JDR1 register *********************************************/
#define ADC_JDR1_JDATA_Pos                        (0U)
#define ADC_JDR1_JDATA_Msk                        (0xFFFFUL<<ADC_JDR1_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR1_JDATA                            ADC_JDR1_JDATA_Msk

/********************************* Bit definition for ADC_JDR2 register *********************************************/
#define ADC_JDR2_JDATA_Pos                        (0U)
#define ADC_JDR2_JDATA_Msk                        (0xFFFFUL<<ADC_JDR2_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR2_JDATA                            ADC_JDR2_JDATA_Msk

/********************************* Bit definition for ADC_JDR3 register *********************************************/
#define ADC_JDR3_JDATA_Pos                        (0U)
#define ADC_JDR3_JDATA_Msk                        (0xFFFFUL<<ADC_JDR3_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR3_JDATA                            ADC_JDR3_JDATA_Msk

/********************************* Bit definition for ADC_JDR4 register *********************************************/
#define ADC_JDR4_JDATA_Pos                        (0U)
#define ADC_JDR4_JDATA_Msk                        (0xFFFFUL<<ADC_JDR4_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR4_JDATA                            ADC_JDR4_JDATA_Msk

/********************************* Bit definition for ADC_AWD2CR register *******************************************/
#define ADC_AWD2CR_AWD2CH_Pos                     (0U)
#define ADC_AWD2CR_AWD2CH_Msk                     (0xFFFFFUL<<ADC_AWD2CR_AWD2CH_Pos)                /*!< 0x000FFFFF */
#define ADC_AWD2CR_AWD2CH                         ADC_AWD2CR_AWD2CH_Msk
#define ADC_AWD2CR_AWD2CH_0                       (0x00001UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000001 */
#define ADC_AWD2CR_AWD2CH_1                       (0x00002UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000002 */
#define ADC_AWD2CR_AWD2CH_2                       (0x00004UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000004 */
#define ADC_AWD2CR_AWD2CH_3                       (0x00008UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000008 */
#define ADC_AWD2CR_AWD2CH_4                       (0x00010UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000010 */
#define ADC_AWD2CR_AWD2CH_5                       (0x00020UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000020 */
#define ADC_AWD2CR_AWD2CH_6                       (0x00040UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000040 */
#define ADC_AWD2CR_AWD2CH_7                       (0x00080UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000080 */
#define ADC_AWD2CR_AWD2CH_8                       (0x00100UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000100 */
#define ADC_AWD2CR_AWD2CH_9                       (0x00200UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000200 */
#define ADC_AWD2CR_AWD2CH_10                      (0x00400UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000400 */
#define ADC_AWD2CR_AWD2CH_11                      (0x00800UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00000800 */
#define ADC_AWD2CR_AWD2CH_12                      (0x01000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00001000 */
#define ADC_AWD2CR_AWD2CH_13                      (0x02000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00002000 */
#define ADC_AWD2CR_AWD2CH_14                      (0x04000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00004000 */
#define ADC_AWD2CR_AWD2CH_15                      (0x08000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00008000 */
#define ADC_AWD2CR_AWD2CH_16                      (0x10000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00010000 */
#define ADC_AWD2CR_AWD2CH_17                      (0x20000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00020000 */
#define ADC_AWD2CR_AWD2CH_18                      (0x40000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00040000 */
#define ADC_AWD2CR_AWD2CH_19                      (0x80000UL << ADC_AWD2CR_AWD2CH_Pos)              /*!< 0x00080000 */

/********************************* Bit definition for ADC_AWD3CR register *******************************************/
#define ADC_AWD3CR_AWD3CH_Pos                     (0U)
#define ADC_AWD3CR_AWD3CH_Msk                     (0xFFFFFUL<<ADC_AWD3CR_AWD3CH_Pos)                /*!< 0x000FFFFF */
#define ADC_AWD3CR_AWD3CH                         ADC_AWD3CR_AWD3CH_Msk
#define ADC_AWD3CR_AWD3CH_0                       (0x00001UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000001 */
#define ADC_AWD3CR_AWD3CH_1                       (0x00002UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000002 */
#define ADC_AWD3CR_AWD3CH_2                       (0x00004UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000004 */
#define ADC_AWD3CR_AWD3CH_3                       (0x00008UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000008 */
#define ADC_AWD3CR_AWD3CH_4                       (0x00010UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000010 */
#define ADC_AWD3CR_AWD3CH_5                       (0x00020UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000020 */
#define ADC_AWD3CR_AWD3CH_6                       (0x00040UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000040 */
#define ADC_AWD3CR_AWD3CH_7                       (0x00080UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000080 */
#define ADC_AWD3CR_AWD3CH_8                       (0x00100UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000100 */
#define ADC_AWD3CR_AWD3CH_9                       (0x00200UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000200 */
#define ADC_AWD3CR_AWD3CH_10                      (0x00400UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000400 */
#define ADC_AWD3CR_AWD3CH_11                      (0x00800UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00000800 */
#define ADC_AWD3CR_AWD3CH_12                      (0x01000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00001000 */
#define ADC_AWD3CR_AWD3CH_13                      (0x02000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00002000 */
#define ADC_AWD3CR_AWD3CH_14                      (0x04000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00004000 */
#define ADC_AWD3CR_AWD3CH_15                      (0x08000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00008000 */
#define ADC_AWD3CR_AWD3CH_16                      (0x10000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00010000 */
#define ADC_AWD3CR_AWD3CH_17                      (0x20000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00020000 */
#define ADC_AWD3CR_AWD3CH_18                      (0x40000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00040000 */
#define ADC_AWD3CR_AWD3CH_19                      (0x80000UL << ADC_AWD3CR_AWD3CH_Pos)              /*!< 0x00080000 */

/********************************* Bit definition for ADC_DIFSEL register *******************************************/
#define ADC_DIFSEL_DIFSEL_Pos                     (0U)
#define ADC_DIFSEL_DIFSEL_Msk                     (0xFFFFFUL<<ADC_DIFSEL_DIFSEL_Pos)                /*!< 0x000FFFFF */
#define ADC_DIFSEL_DIFSEL                         ADC_DIFSEL_DIFSEL_Msk

/********************************* Bit definition for ADC_CALFACT register ******************************************/
#define ADC_CALFACT_OFFSUC_Pos                    (20U)
#define ADC_CALFACT_OFFSUC_Msk                    (0x1UL<<ADC_CALFACT_OFFSUC_Pos)                   /*!< 0x00100000 */
#define ADC_CALFACT_OFFSUC                        ADC_CALFACT_OFFSUC_Msk                            
#define ADC_CALFACT_CAPSUC_Pos                    (21U)
#define ADC_CALFACT_CAPSUC_Msk                    (0x1UL<<ADC_CALFACT_CAPSUC_Pos)                   /*!< 0x00200000 */
#define ADC_CALFACT_CAPSUC                        ADC_CALFACT_CAPSUC_Msk                            
#define ADC_CALFACT_CALON_Pos                     (22U)
#define ADC_CALFACT_CALON_Msk                     (0x1UL<<ADC_CALFACT_CALON_Pos)                    /*!< 0x00400000 */
#define ADC_CALFACT_CALON                         ADC_CALFACT_CALON_Msk                             

/********************************* Bit definition for ADC_GCOMP register ********************************************/
#define ADC_GCOMP_GCOMPCOEFF_Pos                  (0U)
#define ADC_GCOMP_GCOMPCOEFF_Msk                  (0x3FFFUL<<ADC_GCOMP_GCOMPCOEFF_Pos)              /*!< 0x00003FFF */
#define ADC_GCOMP_GCOMPCOEFF                      ADC_GCOMP_GCOMPCOEFF_Msk

/********************************* Bit definition for ADC_CCR register *********************************************/
#define ADC_CCR_DUAL_Pos                         (0U)
#define ADC_CCR_DUAL_Msk                         (0x1FUL<<ADC_CCR_DUAL_Pos)                       /*!< 0x0000001F */
#define ADC_CCR_DUAL                             ADC_CCR_DUAL_Msk
#define ADC_CCR_DUAL_0                           (0x1UL<<ADC_CCR_DUAL_Pos)                        /*!< 0x00000001 */
#define ADC_CCR_DUAL_1                           (0x2UL<<ADC_CCR_DUAL_Pos)                        /*!< 0x00000002 */
#define ADC_CCR_DUAL_2                           (0x4UL<<ADC_CCR_DUAL_Pos)                        /*!< 0x00000004 */
#define ADC_CCR_DUAL_3                           (0x8UL<<ADC_CCR_DUAL_Pos)                        /*!< 0x00000008 */
#define ADC_CCR_DUAL_4                           (0x10UL<<ADC_CCR_DUAL_Pos)                       /*!< 0x00000010 */
#define ADC_CCR_VREFBUFEN_Pos                    (5U)
#define ADC_CCR_VREFBUFEN_Msk                    (0x1UL<<ADC_CCR_VREFBUFEN_Pos)                   /*!< 0x00000020 */
#define ADC_CCR_VREFBUFEN                        ADC_CCR_VREFBUFEN_Msk                            
#define ADC_CCR_VREFBUFSEL_Pos                   (6U)
#define ADC_CCR_VREFBUFSEL_Msk                   (0x3UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x000000C0 */
#define ADC_CCR_VREFBUFSEL                       ADC_CCR_VREFBUFSEL_Msk
#define ADC_CCR_VREFBUFSEL_0                     (0x1UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x00000040 */
#define ADC_CCR_VREFBUFSEL_1                     (0x2UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x00000080 */
#define ADC_CCR_VREFEN_Pos                       (22U)
#define ADC_CCR_VREFEN_Msk                       (0x1UL<<ADC_CCR_VREFEN_Pos)                      /*!< 0x00400000 */
#define ADC_CCR_VREFEN                           ADC_CCR_VREFEN_Msk                               
#define ADC_CCR_VSENSESEL_Pos                    (23U)
#define ADC_CCR_VSENSESEL_Msk                    (0x1UL<<ADC_CCR_VSENSESEL_Pos)                   /*!< 0x00800000 */
#define ADC_CCR_VSENSESEL                        ADC_CCR_VSENSESEL_Msk                            
#define ADC_CCR_VBATSEL_Pos                      (24U)
#define ADC_CCR_VBATSEL_Msk                      (0x1UL<<ADC_CCR_VBATSEL_Pos)                     /*!< 0x01000000 */
#define ADC_CCR_VBATSEL                          ADC_CCR_VBATSEL_Msk                              
#define ADC_CCR_PWRMODE_Pos                      (25U)
#define ADC_CCR_PWRMODE_Msk                      (0x7UL<<ADC_CCR_PWRMODE_Pos)                     /*!< 0x0E000000 */
#define ADC_CCR_PWRMODE                          ADC_CCR_PWRMODE_Msk
#define ADC_CCR_PWRMODE_0                        (0x1UL<<ADC_CCR_PWRMODE_Pos)                     /*!< 0x02000000 */
#define ADC_CCR_PWRMODE_1                        (0x2UL<<ADC_CCR_PWRMODE_Pos)                     /*!< 0x04000000 */
#define ADC_CCR_PWRMODE_2                        (0x4UL<<ADC_CCR_PWRMODE_Pos)                     /*!< 0x08000000 */
#define ADC_CCR_VREFSEL_Pos                      (28U)
#define ADC_CCR_VREFSEL_Msk                      (0x1UL<<ADC_CCR_VREFSEL_Pos)                     /*!< 0x10000000 */
#define ADC_CCR_VREFSEL                          ADC_CCR_VREFSEL_Msk                              

/********************************************************************************************************************/
/********************************* AES ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for AES_MSG_CFG register ******************************************/
#define AES_MSG_CFG_AES_GO_Pos                    (0U)
#define AES_MSG_CFG_AES_GO_Msk                    (0x1UL<<AES_MSG_CFG_AES_GO_Pos)                   /*!< 0x00000001 */
#define AES_MSG_CFG_AES_GO                        AES_MSG_CFG_AES_GO_Msk                            
#define AES_MSG_CFG_MSG_END_Pos                   (1U)
#define AES_MSG_CFG_MSG_END_Msk                   (0x1UL<<AES_MSG_CFG_MSG_END_Pos)                  /*!< 0x00000002 */
#define AES_MSG_CFG_MSG_END                       AES_MSG_CFG_MSG_END_Msk                           
#define AES_MSG_CFG_MSG_BEGIN_Pos                 (2U)
#define AES_MSG_CFG_MSG_BEGIN_Msk                 (0x1UL<<AES_MSG_CFG_MSG_BEGIN_Pos)                /*!< 0x00000004 */
#define AES_MSG_CFG_MSG_BEGIN                     AES_MSG_CFG_MSG_BEGIN_Msk                         
#define AES_MSG_CFG_ALGODIR_Pos                   (3U)
#define AES_MSG_CFG_ALGODIR_Msk                   (0x1UL<<AES_MSG_CFG_ALGODIR_Pos)                  /*!< 0x00000008 */
#define AES_MSG_CFG_ALGODIR                       AES_MSG_CFG_ALGODIR_Msk                           
#define AES_MSG_CFG_ALG_MODE_Pos                  (4U)
#define AES_MSG_CFG_ALG_MODE_Msk                  (0xFUL<<AES_MSG_CFG_ALG_MODE_Pos)                 /*!< 0x000000F0 */
#define AES_MSG_CFG_ALG_MODE                      AES_MSG_CFG_ALG_MODE_Msk
#define AES_MSG_CFG_ALG_MODE_0                    (0x1UL<<AES_MSG_CFG_ALG_MODE_Pos)                 /*!< 0x00000010 */
#define AES_MSG_CFG_ALG_MODE_1                    (0x2UL<<AES_MSG_CFG_ALG_MODE_Pos)                 /*!< 0x00000020 */
#define AES_MSG_CFG_ALG_MODE_2                    (0x4UL<<AES_MSG_CFG_ALG_MODE_Pos)                 /*!< 0x00000040 */
#define AES_MSG_CFG_ALG_MODE_3                    (0x8UL<<AES_MSG_CFG_ALG_MODE_Pos)                 /*!< 0x00000080 */
#define AES_MSG_CFG_MAC_LEN_Pos                   (8U)
#define AES_MSG_CFG_MAC_LEN_Msk                   (0xFUL<<AES_MSG_CFG_MAC_LEN_Pos)                  /*!< 0x00000F00 */
#define AES_MSG_CFG_MAC_LEN                       AES_MSG_CFG_MAC_LEN_Msk
#define AES_MSG_CFG_MAC_LEN_0                     (0x1UL<<AES_MSG_CFG_MAC_LEN_Pos)                  /*!< 0x00000100 */
#define AES_MSG_CFG_MAC_LEN_1                     (0x2UL<<AES_MSG_CFG_MAC_LEN_Pos)                  /*!< 0x00000200 */
#define AES_MSG_CFG_MAC_LEN_2                     (0x4UL<<AES_MSG_CFG_MAC_LEN_Pos)                  /*!< 0x00000400 */
#define AES_MSG_CFG_MAC_LEN_3                     (0x8UL<<AES_MSG_CFG_MAC_LEN_Pos)                  /*!< 0x00000800 */
#define AES_MSG_CFG_KEY_SIZE_Pos                  (12U)
#define AES_MSG_CFG_KEY_SIZE_Msk                  (0x3UL<<AES_MSG_CFG_KEY_SIZE_Pos)                 /*!< 0x00003000 */
#define AES_MSG_CFG_KEY_SIZE                      AES_MSG_CFG_KEY_SIZE_Msk
#define AES_MSG_CFG_KEY_SIZE_0                    (0x1UL<<AES_MSG_CFG_KEY_SIZE_Pos)                 /*!< 0x00001000 */
#define AES_MSG_CFG_KEY_SIZE_1                    (0x2UL<<AES_MSG_CFG_KEY_SIZE_Pos)                 /*!< 0x00002000 */

/********************************* Bit definition for AES_CTX_CFG register ******************************************/
#define AES_CTX_CFG_CTX_RET_Pos                   (0U)
#define AES_CTX_CFG_CTX_RET_Msk                   (0x1UL<<AES_CTX_CFG_CTX_RET_Pos)                  /*!< 0x00000001 */
#define AES_CTX_CFG_CTX_RET                       AES_CTX_CFG_CTX_RET_Msk                           
#define AES_CTX_CFG_CTX_STR_Pos                   (1U)
#define AES_CTX_CFG_CTX_STR_Msk                   (0x1UL<<AES_CTX_CFG_CTX_STR_Pos)                  /*!< 0x00000002 */
#define AES_CTX_CFG_CTX_STR                       AES_CTX_CFG_CTX_STR_Msk                           
#define AES_CTX_CFG_INV_KEY_RET_Pos               (2U)
#define AES_CTX_CFG_INV_KEY_RET_Msk               (0x1UL<<AES_CTX_CFG_INV_KEY_RET_Pos)              /*!< 0x00000004 */
#define AES_CTX_CFG_INV_KEY_RET                   AES_CTX_CFG_INV_KEY_RET_Msk                       
#define AES_CTX_CFG_INV_KEY_STR_Pos               (3U)
#define AES_CTX_CFG_INV_KEY_STR_Msk               (0x1UL<<AES_CTX_CFG_INV_KEY_STR_Pos)              /*!< 0x00000008 */
#define AES_CTX_CFG_INV_KEY_STR                   AES_CTX_CFG_INV_KEY_STR_Msk                       
#define AES_CTX_CFG_CTX_INDEX_Pos                 (4U)
#define AES_CTX_CFG_CTX_INDEX_Msk                 (0xFFFUL<<AES_CTX_CFG_CTX_INDEX_Pos)              /*!< 0x0000FFF0 */
#define AES_CTX_CFG_CTX_INDEX                     AES_CTX_CFG_CTX_INDEX_Msk

/********************************* Bit definition for AES_MSG_TOTAL_BYTES register **********************************/
#define AES_MSG_TOTAL_BYTES_TOTAL_BYTES_Pos       (0U)
#define AES_MSG_TOTAL_BYTES_TOTAL_BYTES_Msk       (0xFFFFFFFUL<<AES_MSG_TOTAL_BYTES_TOTAL_BYTES_Pos)/*!< 0x0FFFFFFF */
#define AES_MSG_TOTAL_BYTES_TOTAL_BYTES           AES_MSG_TOTAL_BYTES_TOTAL_BYTES_Msk

/********************************* Bit definition for AES_MSG_AAD_BYTES register ************************************/
#define AES_MSG_AAD_BYTES_MSG_LEN_Pos             (0U)
#define AES_MSG_AAD_BYTES_MSG_LEN_Msk             (0xFFFFUL<<AES_MSG_AAD_BYTES_MSG_LEN_Pos)         /*!< 0x0000FFFF */
#define AES_MSG_AAD_BYTES_MSG_LEN                 AES_MSG_AAD_BYTES_MSG_LEN_Msk
#define AES_MSG_AAD_BYTES_ADD_LEN_Pos             (16U)
#define AES_MSG_AAD_BYTES_ADD_LEN_Msk             (0xFFFFUL<<AES_MSG_AAD_BYTES_ADD_LEN_Pos)         /*!< 0xFFFF0000 */
#define AES_MSG_AAD_BYTES_ADD_LEN                 AES_MSG_AAD_BYTES_ADD_LEN_Msk

/********************************* Bit definition for AES_GCM_AAD_INFO register *************************************/
#define AES_GCM_AAD_INFO_ADD_LEN_TOT_Pos          (0U)
#define AES_GCM_AAD_INFO_ADD_LEN_TOT_Msk          (0xFFFFFFFUL<<AES_GCM_AAD_INFO_ADD_LEN_TOT_Pos)   /*!< 0x0FFFFFFF */
#define AES_GCM_AAD_INFO_ADD_LEN_TOT              AES_GCM_AAD_INFO_ADD_LEN_TOT_Msk

/********************************* Bit definition for AES_CTX_KEY0 register *****************************************/
#define AES_CTX_KEY0_KEY0_Pos                     (0U)
#define AES_CTX_KEY0_KEY0_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY0_KEY0_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY0_KEY0                         AES_CTX_KEY0_KEY0_Msk

/********************************* Bit definition for AES_CTX_KEY1 register *****************************************/
#define AES_CTX_KEY1_KEY1_Pos                     (0U)
#define AES_CTX_KEY1_KEY1_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY1_KEY1_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY1_KEY1                         AES_CTX_KEY1_KEY1_Msk

/********************************* Bit definition for AES_CTX_KEY2 register *****************************************/
#define AES_CTX_KEY2_KEY2_Pos                     (0U)
#define AES_CTX_KEY2_KEY2_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY2_KEY2_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY2_KEY2                         AES_CTX_KEY2_KEY2_Msk

/********************************* Bit definition for AES_CTX_KEY3 register *****************************************/
#define AES_CTX_KEY3_KEY3_Pos                     (0U)
#define AES_CTX_KEY3_KEY3_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY3_KEY3_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY3_KEY3                         AES_CTX_KEY3_KEY3_Msk

/********************************* Bit definition for AES_CTX_KEY4 register *****************************************/
#define AES_CTX_KEY4_KEY4_Pos                     (0U)
#define AES_CTX_KEY4_KEY4_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY4_KEY4_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY4_KEY4                         AES_CTX_KEY4_KEY4_Msk

/********************************* Bit definition for AES_CTX_KEY5 register *****************************************/
#define AES_CTX_KEY5_KEY5_Pos                     (0U)
#define AES_CTX_KEY5_KEY5_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY5_KEY5_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY5_KEY5                         AES_CTX_KEY5_KEY5_Msk

/********************************* Bit definition for AES_CTX_KEY6 register *****************************************/
#define AES_CTX_KEY6_KEY6_Pos                     (0U)
#define AES_CTX_KEY6_KEY6_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY6_KEY6_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY6_KEY6                         AES_CTX_KEY6_KEY6_Msk

/********************************* Bit definition for AES_CTX_KEY7 register *****************************************/
#define AES_CTX_KEY7_KEY7_Pos                     (0U)
#define AES_CTX_KEY7_KEY7_Msk                     (0xFFFFFFFFUL<<AES_CTX_KEY7_KEY7_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_KEY7_KEY7                         AES_CTX_KEY7_KEY7_Msk

/********************************* Bit definition for AES_CTX_CBC_KEY0 register *************************************/
#define AES_CTX_CBC_KEY0_CBC_KEY0_Pos             (0U)
#define AES_CTX_CBC_KEY0_CBC_KEY0_Msk             (0xFFFFFFFFUL<<AES_CTX_CBC_KEY0_CBC_KEY0_Pos)     /*!< 0xFFFFFFFF */
#define AES_CTX_CBC_KEY0_CBC_KEY0                 AES_CTX_CBC_KEY0_CBC_KEY0_Msk

/********************************* Bit definition for AES_CTX_CBC_KEY1 register *************************************/
#define AES_CTX_CBC_KEY1_CBC_KEY1_Pos             (0U)
#define AES_CTX_CBC_KEY1_CBC_KEY1_Msk             (0xFFFFFFFFUL<<AES_CTX_CBC_KEY1_CBC_KEY1_Pos)     /*!< 0xFFFFFFFF */
#define AES_CTX_CBC_KEY1_CBC_KEY1                 AES_CTX_CBC_KEY1_CBC_KEY1_Msk

/********************************* Bit definition for AES_CTX_CBC_KEY2 register *************************************/
#define AES_CTX_CBC_KEY2_CBC_KEY2_Pos             (0U)
#define AES_CTX_CBC_KEY2_CBC_KEY2_Msk             (0xFFFFFFFFUL<<AES_CTX_CBC_KEY2_CBC_KEY2_Pos)     /*!< 0xFFFFFFFF */
#define AES_CTX_CBC_KEY2_CBC_KEY2                 AES_CTX_CBC_KEY2_CBC_KEY2_Msk

/********************************* Bit definition for AES_CTX_CBC_KEY3 register *************************************/
#define AES_CTX_CBC_KEY3_CBC_KEY3_Pos             (0U)
#define AES_CTX_CBC_KEY3_CBC_KEY3_Msk             (0xFFFFFFFFUL<<AES_CTX_CBC_KEY3_CBC_KEY3_Pos)     /*!< 0xFFFFFFFF */
#define AES_CTX_CBC_KEY3_CBC_KEY3                 AES_CTX_CBC_KEY3_CBC_KEY3_Msk

/********************************* Bit definition for AES_CTX_CTR0 register *****************************************/
#define AES_CTX_CTR0_CCM_CTR0_Pos                 (0U)
#define AES_CTX_CTR0_CCM_CTR0_Msk                 (0xFFFFFFFFUL<<AES_CTX_CTR0_CCM_CTR0_Pos)         /*!< 0xFFFFFFFF */
#define AES_CTX_CTR0_CCM_CTR0                     AES_CTX_CTR0_CCM_CTR0_Msk

/********************************* Bit definition for AES_CTX_CTR1 register *****************************************/
#define AES_CTX_CTR1_CCM_CTR1_Pos                 (0U)
#define AES_CTX_CTR1_CCM_CTR1_Msk                 (0xFFFFFFFFUL<<AES_CTX_CTR1_CCM_CTR1_Pos)         /*!< 0xFFFFFFFF */
#define AES_CTX_CTR1_CCM_CTR1                     AES_CTX_CTR1_CCM_CTR1_Msk

/********************************* Bit definition for AES_CTX_CTR2 register *****************************************/
#define AES_CTX_CTR2_CCM_CTR2_Pos                 (0U)
#define AES_CTX_CTR2_CCM_CTR2_Msk                 (0xFFFFFFFFUL<<AES_CTX_CTR2_CCM_CTR2_Pos)         /*!< 0xFFFFFFFF */
#define AES_CTX_CTR2_CCM_CTR2                     AES_CTX_CTR2_CCM_CTR2_Msk

/********************************* Bit definition for AES_CTX_CTR3 register *****************************************/
#define AES_CTX_CTR3_CCM_CTR3_Pos                 (0U)
#define AES_CTX_CTR3_CCM_CTR3_Msk                 (0xFFFFFFFFUL<<AES_CTX_CTR3_CCM_CTR3_Pos)         /*!< 0xFFFFFFFF */
#define AES_CTX_CTR3_CCM_CTR3                     AES_CTX_CTR3_CCM_CTR3_Msk

/********************************* Bit definition for AES_CTX_IV0 register ******************************************/
#define AES_CTX_IV0_IV0_Pos                       (0U)
#define AES_CTX_IV0_IV0_Msk                       (0xFFFFFFFFUL<<AES_CTX_IV0_IV0_Pos)               /*!< 0xFFFFFFFF */
#define AES_CTX_IV0_IV0                           AES_CTX_IV0_IV0_Msk

/********************************* Bit definition for AES_CTX_IV1 register ******************************************/
#define AES_CTX_IV1_IV1_Pos                       (0U)
#define AES_CTX_IV1_IV1_Msk                       (0xFFFFFFFFUL<<AES_CTX_IV1_IV1_Pos)               /*!< 0xFFFFFFFF */
#define AES_CTX_IV1_IV1                           AES_CTX_IV1_IV1_Msk

/********************************* Bit definition for AES_CTX_IV2 register ******************************************/
#define AES_CTX_IV2_IV2_Pos                       (0U)
#define AES_CTX_IV2_IV2_Msk                       (0xFFFFFFFFUL<<AES_CTX_IV2_IV2_Pos)               /*!< 0xFFFFFFFF */
#define AES_CTX_IV2_IV2                           AES_CTX_IV2_IV2_Msk

/********************************* Bit definition for AES_CTX_IV3 register ******************************************/
#define AES_CTX_IV3_IV3_Pos                       (0U)
#define AES_CTX_IV3_IV3_Msk                       (0xFFFFFFFFUL<<AES_CTX_IV3_IV3_Pos)               /*!< 0xFFFFFFFF */
#define AES_CTX_IV3_IV3                           AES_CTX_IV3_IV3_Msk

/********************************* Bit definition for AES_CTX_MAC0 register *****************************************/
#define AES_CTX_MAC0_MAC0_Pos                     (0U)
#define AES_CTX_MAC0_MAC0_Msk                     (0xFFFFFFFFUL<<AES_CTX_MAC0_MAC0_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_MAC0_MAC0                         AES_CTX_MAC0_MAC0_Msk

/********************************* Bit definition for AES_CTX_MAC1 register *****************************************/
#define AES_CTX_MAC1_MAC1_Pos                     (0U)
#define AES_CTX_MAC1_MAC1_Msk                     (0xFFFFFFFFUL<<AES_CTX_MAC1_MAC1_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_MAC1_MAC1                         AES_CTX_MAC1_MAC1_Msk

/********************************* Bit definition for AES_CTX_MAC2 register *****************************************/
#define AES_CTX_MAC2_MAC2_Pos                     (0U)
#define AES_CTX_MAC2_MAC2_Msk                     (0xFFFFFFFFUL<<AES_CTX_MAC2_MAC2_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_MAC2_MAC2                         AES_CTX_MAC2_MAC2_Msk

/********************************* Bit definition for AES_CTX_MAC3 register *****************************************/
#define AES_CTX_MAC3_MAC3_Pos                     (0U)
#define AES_CTX_MAC3_MAC3_Msk                     (0xFFFFFFFFUL<<AES_CTX_MAC3_MAC3_Pos)             /*!< 0xFFFFFFFF */
#define AES_CTX_MAC3_MAC3                         AES_CTX_MAC3_MAC3_Msk

/********************************* Bit definition for AES_INGRESS_FIFO register *************************************/
#define AES_INGRESS_FIFO_INGRESS_FIFO_Pos         (0U)
#define AES_INGRESS_FIFO_INGRESS_FIFO_Msk         (0xFFFFFFFFUL<<AES_INGRESS_FIFO_INGRESS_FIFO_Pos) /*!< 0xFFFFFFFF */
#define AES_INGRESS_FIFO_INGRESS_FIFO             AES_INGRESS_FIFO_INGRESS_FIFO_Msk

/********************************* Bit definition for AES_ING_DBCFG register ****************************************/
#define AES_ING_DBCFG_DST_TR_WIDTH_Pos            (0U)
#define AES_ING_DBCFG_DST_TR_WIDTH_Msk            (0x7UL<<AES_ING_DBCFG_DST_TR_WIDTH_Pos)           /*!< 0x00000007 */
#define AES_ING_DBCFG_DST_TR_WIDTH                AES_ING_DBCFG_DST_TR_WIDTH_Msk
#define AES_ING_DBCFG_DST_TR_WIDTH_0              (0x1UL<<AES_ING_DBCFG_DST_TR_WIDTH_Pos)           /*!< 0x00000001 */
#define AES_ING_DBCFG_DST_TR_WIDTH_1              (0x2UL<<AES_ING_DBCFG_DST_TR_WIDTH_Pos)           /*!< 0x00000002 */
#define AES_ING_DBCFG_DST_TR_WIDTH_2              (0x4UL<<AES_ING_DBCFG_DST_TR_WIDTH_Pos)           /*!< 0x00000004 */
#define AES_ING_DBCFG_DST_MSIZE_Pos               (3U)
#define AES_ING_DBCFG_DST_MSIZE_Msk               (0x7UL<<AES_ING_DBCFG_DST_MSIZE_Pos)              /*!< 0x00000038 */
#define AES_ING_DBCFG_DST_MSIZE                   AES_ING_DBCFG_DST_MSIZE_Msk
#define AES_ING_DBCFG_DST_MSIZE_0                 (0x1UL<<AES_ING_DBCFG_DST_MSIZE_Pos)              /*!< 0x00000008 */
#define AES_ING_DBCFG_DST_MSIZE_1                 (0x2UL<<AES_ING_DBCFG_DST_MSIZE_Pos)              /*!< 0x00000010 */
#define AES_ING_DBCFG_DST_MSIZE_2                 (0x4UL<<AES_ING_DBCFG_DST_MSIZE_Pos)              /*!< 0x00000020 */
#define AES_ING_DBCFG_DMA_EN_Pos                  (30U)
#define AES_ING_DBCFG_DMA_EN_Msk                  (0x1UL<<AES_ING_DBCFG_DMA_EN_Pos)                 /*!< 0x40000000 */
#define AES_ING_DBCFG_DMA_EN                      AES_ING_DBCFG_DMA_EN_Msk                          
#define AES_ING_DBCFG_PLAINTEXT_REQ_Pos           (31U)
#define AES_ING_DBCFG_PLAINTEXT_REQ_Msk           (0x1UL<<AES_ING_DBCFG_PLAINTEXT_REQ_Pos)          /*!< 0x80000000 */
#define AES_ING_DBCFG_PLAINTEXT_REQ               AES_ING_DBCFG_PLAINTEXT_REQ_Msk                   

/********************************* Bit definition for AES_DMA_ENG_LEN register **************************************/
#define AES_DMA_ENG_LEN_ENGRESS_DMA_DATA_LENGTH_Pos  (0U)
#define AES_DMA_ENG_LEN_ENGRESS_DMA_DATA_LENGTH_Msk  (0xFFFFFFFFUL<<AES_DMA_ENG_LEN_ENGRESS_DMA_DATA_LENGTH_Pos)/*!< 0xFFFFFFFF */
#define AES_DMA_ENG_LEN_ENGRESS_DMA_DATA_LENGTH       AES_DMA_ENG_LEN_ENGRESS_DMA_DATA_LENGTH_Msk

/********************************* Bit definition for AES_ENG_DBCFG register ****************************************/
#define AES_ENG_DBCFG_SRC_TR_WIDTH_Pos            (0U)
#define AES_ENG_DBCFG_SRC_TR_WIDTH_Msk            (0x7UL<<AES_ENG_DBCFG_SRC_TR_WIDTH_Pos)           /*!< 0x00000007 */
#define AES_ENG_DBCFG_SRC_TR_WIDTH                AES_ENG_DBCFG_SRC_TR_WIDTH_Msk
#define AES_ENG_DBCFG_SRC_TR_WIDTH_0              (0x1UL<<AES_ENG_DBCFG_SRC_TR_WIDTH_Pos)           /*!< 0x00000001 */
#define AES_ENG_DBCFG_SRC_TR_WIDTH_1              (0x2UL<<AES_ENG_DBCFG_SRC_TR_WIDTH_Pos)           /*!< 0x00000002 */
#define AES_ENG_DBCFG_SRC_TR_WIDTH_2              (0x4UL<<AES_ENG_DBCFG_SRC_TR_WIDTH_Pos)           /*!< 0x00000004 */
#define AES_ENG_DBCFG_SRC_MSIZE_Pos               (3U)
#define AES_ENG_DBCFG_SRC_MSIZE_Msk               (0x7UL<<AES_ENG_DBCFG_SRC_MSIZE_Pos)              /*!< 0x00000038 */
#define AES_ENG_DBCFG_SRC_MSIZE                   AES_ENG_DBCFG_SRC_MSIZE_Msk
#define AES_ENG_DBCFG_SRC_MSIZE_0                 (0x1UL<<AES_ENG_DBCFG_SRC_MSIZE_Pos)              /*!< 0x00000008 */
#define AES_ENG_DBCFG_SRC_MSIZE_1                 (0x2UL<<AES_ENG_DBCFG_SRC_MSIZE_Pos)              /*!< 0x00000010 */
#define AES_ENG_DBCFG_SRC_MSIZE_2                 (0x4UL<<AES_ENG_DBCFG_SRC_MSIZE_Pos)              /*!< 0x00000020 */
#define AES_ENG_DBCFG_DMA_EN_Pos                  (30U)
#define AES_ENG_DBCFG_DMA_EN_Msk                  (0x1UL<<AES_ENG_DBCFG_DMA_EN_Pos)                 /*!< 0x40000000 */
#define AES_ENG_DBCFG_DMA_EN                      AES_ENG_DBCFG_DMA_EN_Msk                          
#define AES_ENG_DBCFG_CIPERTEXT_REQ_Pos           (31U)
#define AES_ENG_DBCFG_CIPERTEXT_REQ_Msk           (0x1UL<<AES_ENG_DBCFG_CIPERTEXT_REQ_Pos)          /*!< 0x80000000 */
#define AES_ENG_DBCFG_CIPERTEXT_REQ               AES_ENG_DBCFG_CIPERTEXT_REQ_Msk                   

/********************************* Bit definition for AES_DMA_ING_LEN register **************************************/
#define AES_DMA_ING_LEN_INGRESS_DMA_DATA_LENGTH_Pos (0U)
#define AES_DMA_ING_LEN_INGRESS_DMA_DATA_LENGTH_Msk (0xFFFFFFFFUL<<AES_DMA_ING_LEN_INGRESS_DMA_DATA_LENGTH_Pos)/*!< 0xFFFFFFFF */
#define AES_DMA_ING_LEN_INGRESS_DMA_DATA_LENGTH      AES_DMA_ING_LEN_INGRESS_DMA_DATA_LENGTH_Msk

/********************************* Bit definition for AES_INGF_STATUS register **********************************/
#define AES_INGF_STATUS_ING_EMPTY_Pos             (0U)
#define AES_INGF_STATUS_ING_EMPTY_Msk             (0x1UL<<AES_INGF_STATUS_ING_EMPTY_Pos)            /*!< 0x00000001 */
#define AES_INGF_STATUS_ING_EMPTY                 AES_INGF_STATUS_ING_EMPTY_Msk                     
#define AES_INGF_STATUS_ING_FULL_Pos              (1U)
#define AES_INGF_STATUS_ING_FULL_Msk              (0x1UL<<AES_INGF_STATUS_ING_FULL_Pos)             /*!< 0x00000002 */
#define AES_INGF_STATUS_ING_FULL                  AES_INGF_STATUS_ING_FULL_Msk                      

/********************************* Bit definition for AES_ENGRESS_FIFO register *************************************/
#define AES_ENGRESS_FIFO_INGRESS_FIFO_Pos         (0U)
#define AES_ENGRESS_FIFO_INGRESS_FIFO_Msk         (0xFFFFFFFFUL<<AES_ENGRESS_FIFO_INGRESS_FIFO_Pos) /*!< 0xFFFFFFFF */
#define AES_ENGRESS_FIFO_INGRESS_FIFO             AES_ENGRESS_FIFO_INGRESS_FIFO_Msk

/********************************* Bit definition for AES_ENGFIFO_STATUS register ***********************************/
#define AES_ENGFIFO_STATUS_ENG_EMPTY_Pos          (0U)
#define AES_ENGFIFO_STATUS_ENG_EMPTY_Msk          (0x1UL<<AES_ENGFIFO_STATUS_ENG_EMPTY_Pos)         /*!< 0x00000001 */
#define AES_ENGFIFO_STATUS_ENG_EMPTY              AES_ENGFIFO_STATUS_ENG_EMPTY_Msk                  
#define AES_ENGFIFO_STATUS_ENG_FULL_Pos           (1U)
#define AES_ENGFIFO_STATUS_ENG_FULL_Msk           (0x1UL<<AES_ENGFIFO_STATUS_ENG_FULL_Pos)          /*!< 0x00000002 */
#define AES_ENGFIFO_STATUS_ENG_FULL               AES_ENGFIFO_STATUS_ENG_FULL_Msk                   

/********************************* Bit definition for AES_ING_DMA_DONE register *************************************/
#define AES_ING_DMA_DONE_DMA_INGRESS_DONE_Pos     (0U)
#define AES_ING_DMA_DONE_DMA_INGRESS_DONE_Msk     (0x1UL<<AES_ING_DMA_DONE_DMA_INGRESS_DONE_Pos)    /*!< 0x00000001 */
#define AES_ING_DMA_DONE_DMA_INGRESS_DONE         AES_ING_DMA_DONE_DMA_INGRESS_DONE_Msk             

/********************************* Bit definition for AES_ENG_DMA_DONE register *************************************/
#define AES_ENG_DMA_DONE_DMA_NGRESS_DONE_Pos      (0U)
#define AES_ENG_DMA_DONE_DMA_NGRESS_DONE_Msk      (0x1UL<<AES_ENG_DMA_DONE_DMA_NGRESS_DONE_Pos)     /*!< 0x00000001 */
#define AES_ENG_DMA_DONE_DMA_NGRESS_DONE          AES_ENG_DMA_DONE_DMA_NGRESS_DONE_Msk              

/********************************* Bit definition for AES_DONE_STATUS register **************************************/
#define AES_DONE_STATUS_AES_DONE_Pos              (0U)
#define AES_DONE_STATUS_AES_DONE_Msk              (0x1UL<<AES_DONE_STATUS_AES_DONE_Pos)             /*!< 0x00000001 */
#define AES_DONE_STATUS_AES_DONE                  AES_DONE_STATUS_AES_DONE_Msk                      
#define AES_DONE_STATUS_MAC_VAILD_Pos             (1U)
#define AES_DONE_STATUS_MAC_VAILD_Msk             (0x1UL<<AES_DONE_STATUS_MAC_VAILD_Pos)            /*!< 0x00000002 */
#define AES_DONE_STATUS_MAC_VAILD                 AES_DONE_STATUS_MAC_VAILD_Msk                     

/********************************************************************************************************************/
/********************************* BKP ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for BKP_CR register ***********************************************/
#define BKP_CR_TPE_Pos                            (0U)
#define BKP_CR_TPE_Msk                            (0x1UL<<BKP_CR_TPE_Pos)                           /*!< 0x00000001 */
#define BKP_CR_TPE                                BKP_CR_TPE_Msk                                    /*!< TAMP enable */
#define BKP_CR_TPAL_Pos                           (1U)
#define BKP_CR_TPAL_Msk                           (0x1UL<<BKP_CR_TPAL_Pos)                          /*!< 0x00000002 */
#define BKP_CR_TPAL                               BKP_CR_TPAL_Msk                                   /*!< TAMPER pin interrupt enable */

/********************************* Bit definition for BKP_CSR register **********************************************/
#define BKP_CSR_CTE_Pos                           (0U)
#define BKP_CSR_CTE_Msk                           (0x1UL<<BKP_CSR_CTE_Pos)                          /*!< 0x00000001 */
#define BKP_CSR_CTE                               BKP_CSR_CTE_Msk                                   /*!< clear TAMP event */
#define BKP_CSR_CTI_Pos                           (1U)
#define BKP_CSR_CTI_Msk                           (0x1UL<<BKP_CSR_CTI_Pos)                          /*!< 0x00000002 */
#define BKP_CSR_CTI                               BKP_CSR_CTI_Msk                                   /*!< clear TAMP interrupt */
#define BKP_CSR_TPIE_Pos                          (2U)
#define BKP_CSR_TPIE_Msk                          (0x1UL<<BKP_CSR_TPIE_Pos)                         /*!< 0x00000004 */
#define BKP_CSR_TPIE                              BKP_CSR_TPIE_Msk                                  /*!< TAMP interrupt enable */
#define BKP_CSR_TEF_Pos                           (8U)
#define BKP_CSR_TEF_Msk                           (0x1UL<<BKP_CSR_TEF_Pos)                          /*!< 0x00000100 */
#define BKP_CSR_TEF                               BKP_CSR_TEF_Msk                                   /*!< TAMP event flag */
#define BKP_CSR_TIF_Pos                           (9U)
#define BKP_CSR_TIF_Msk                           (0x1UL<<BKP_CSR_TIF_Pos)                          /*!< 0x00000200 */
#define BKP_CSR_TIF                               BKP_CSR_TIF_Msk                                   /*!< TAMP interrupt flag */

/********************************* Bit definition for BKP_RTCCR register ********************************************/
#define BKP_RTCCR_CAL_Pos                         (0U)
#define BKP_RTCCR_CAL_Msk                         (0x7FUL<<BKP_RTCCR_CAL_Pos)                       /*!< 0x0000007F */
#define BKP_RTCCR_CAL                             BKP_RTCCR_CAL_Msk                                 /*!< RTC calibration value */
#define BKP_RTCCR_CAL_0                           (0x1UL << BKP_RTCCR_CAL_Pos)                      /*!< 0x00000001 */
#define BKP_RTCCR_CAL_1                           (0x2UL << BKP_RTCCR_CAL_Pos)                      /*!< 0x00000002 */
#define BKP_RTCCR_CAL_2                           (0x4UL << BKP_RTCCR_CAL_Pos)                      /*!< 0x00000004 */
#define BKP_RTCCR_CAL_3                           (0x8UL << BKP_RTCCR_CAL_Pos)                      /*!< 0x00000008 */
#define BKP_RTCCR_CAL_4                           (0x10UL << BKP_RTCCR_CAL_Pos)                     /*!< 0x00000010 */
#define BKP_RTCCR_CAL_5                           (0x20UL << BKP_RTCCR_CAL_Pos)                     /*!< 0x00000020 */
#define BKP_RTCCR_CAL_6                           (0x40UL << BKP_RTCCR_CAL_Pos)                     /*!< 0x00000040 */

#define BKP_RTCCR_CCO_Pos                         (7U)
#define BKP_RTCCR_CCO_Msk                         (0x1UL<<BKP_RTCCR_CCO_Pos)                        /*!< 0x00000080 */
#define BKP_RTCCR_CCO                             BKP_RTCCR_CCO_Msk                                 /*!< Calibration clock output */
#define BKP_RTCCR_ASOE_Pos                        (8U)
#define BKP_RTCCR_ASOE_Msk                        (0x1UL<<BKP_RTCCR_ASOE_Pos)                       /*!< 0x00000100 */
#define BKP_RTCCR_ASOE                            BKP_RTCCR_ASOE_Msk                                /*!< Alarm or second output enable */
#define BKP_RTCCR_ASOS_Pos                        (9U)
#define BKP_RTCCR_ASOS_Msk                        (0x1UL<<BKP_RTCCR_ASOS_Pos)                       /*!< 0x00000200 */
#define BKP_RTCCR_ASOS                            BKP_RTCCR_ASOS_Msk                                /*!< Alarm or second output selection */

/********************************* Bit definition for BKP_DR0 register **********************************************/
#define BKP_DR0_D_Pos                             (0U)
#define BKP_DR0_D_Msk                             (0xFFFFFFFFUL<<BKP_DR0_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR0_D                                 BKP_DR0_D_Msk                                     /*!< bakeup data word0 */

/********************************* Bit definition for BKP_DR1 register **********************************************/
#define BKP_DR1_D_Pos                             (0U)
#define BKP_DR1_D_Msk                             (0xFFFFFFFFUL<<BKP_DR1_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR1_D                                 BKP_DR1_D_Msk                                     /*!< bakeup data word1 */

/********************************* Bit definition for BKP_DR2 register **********************************************/
#define BKP_DR2_D_Pos                             (0U)
#define BKP_DR2_D_Msk                             (0xFFFFFFFFUL<<BKP_DR2_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR2_D                                 BKP_DR2_D_Msk                                     /*!< bakeup data word2 */

/********************************* Bit definition for BKP_DR3 register **********************************************/
#define BKP_DR3_D_Pos                             (0U)
#define BKP_DR3_D_Msk                             (0xFFFFFFFFUL<<BKP_DR3_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR3_D                                 BKP_DR3_D_Msk                                     /*!< bakeup data word3 */

/********************************* Bit definition for BKP_DR4 register **********************************************/
#define BKP_DR4_D_Pos                             (0U)
#define BKP_DR4_D_Msk                             (0xFFFFFFFFUL<<BKP_DR4_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR4_D                                 BKP_DR4_D_Msk                                     /*!< bakeup data word4 */

/********************************* Bit definition for BKP_DR5 register **********************************************/
#define BKP_DR5_D_Pos                             (0U)
#define BKP_DR5_D_Msk                             (0xFFFFFFFFUL<<BKP_DR5_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR5_D                                 BKP_DR5_D_Msk                                     /*!< bakeup data word5 */

/********************************* Bit definition for BKP_DR6 register **********************************************/
#define BKP_DR6_D_Pos                             (0U)
#define BKP_DR6_D_Msk                             (0xFFFFFFFFUL<<BKP_DR6_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR6_D                                 BKP_DR6_D_Msk                                     /*!< bakeup data word6 */

/********************************* Bit definition for BKP_DR7 register **********************************************/
#define BKP_DR7_D_Pos                             (0U)
#define BKP_DR7_D_Msk                             (0xFFFFFFFFUL<<BKP_DR7_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR7_D                                 BKP_DR7_D_Msk                                     /*!< bakeup data word7 */

/********************************* Bit definition for BKP_DR8 register **********************************************/
#define BKP_DR8_D_Pos                             (0U)
#define BKP_DR8_D_Msk                             (0xFFFFFFFFUL<<BKP_DR8_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR8_D                                 BKP_DR8_D_Msk                                     /*!< bakeup data word8 */

/********************************* Bit definition for BKP_DR9 register **********************************************/
#define BKP_DR9_D_Pos                             (0U)
#define BKP_DR9_D_Msk                             (0xFFFFFFFFUL<<BKP_DR9_D_Pos)                     /*!< 0xFFFFFFFF */
#define BKP_DR9_D                                 BKP_DR9_D_Msk                                     /*!< bakeup data word9 */

/********************************* Bit definition for BKP_DR10 register *********************************************/
#define BKP_DR10_D_Pos                            (0U)
#define BKP_DR10_D_Msk                            (0xFFFFFFFFUL<<BKP_DR10_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR10_D                                BKP_DR10_D_Msk                                    /*!< bakeup data word10 */

/********************************* Bit definition for BKP_DR11 register *********************************************/
#define BKP_DR11_D_Pos                            (0U)
#define BKP_DR11_D_Msk                            (0xFFFFFFFFUL<<BKP_DR11_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR11_D                                BKP_DR11_D_Msk                                    /*!< bakeup data word11 */

/********************************* Bit definition for BKP_DR12 register *********************************************/
#define BKP_DR12_D_Pos                            (0U)
#define BKP_DR12_D_Msk                            (0xFFFFFFFFUL<<BKP_DR12_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR12_D                                BKP_DR12_D_Msk                                    /*!< bakeup data word12 */

/********************************* Bit definition for BKP_DR13 register *********************************************/
#define BKP_DR13_D_Pos                            (0U)
#define BKP_DR13_D_Msk                            (0xFFFFFFFFUL<<BKP_DR13_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR13_D                                BKP_DR13_D_Msk                                    /*!< bakeup data word13 */

/********************************* Bit definition for BKP_DR14 register *********************************************/
#define BKP_DR14_D_Pos                            (0U)
#define BKP_DR14_D_Msk                            (0xFFFFFFFFUL<<BKP_DR14_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR14_D                                BKP_DR14_D_Msk                                    /*!< bakeup data word14 */

/********************************* Bit definition for BKP_DR15 register *********************************************/
#define BKP_DR15_D_Pos                            (0U)
#define BKP_DR15_D_Msk                            (0xFFFFFFFFUL<<BKP_DR15_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR15_D                                BKP_DR15_D_Msk                                    /*!< bakeup data word15 */

/********************************* Bit definition for BKP_DR16 register *********************************************/
#define BKP_DR16_D_Pos                            (0U)
#define BKP_DR16_D_Msk                            (0xFFFFFFFFUL<<BKP_DR16_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR16_D                                BKP_DR16_D_Msk                                    /*!< bakeup data word16 */

/********************************* Bit definition for BKP_DR17 register *********************************************/
#define BKP_DR17_D_Pos                            (0U)
#define BKP_DR17_D_Msk                            (0xFFFFFFFFUL<<BKP_DR17_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR17_D                                BKP_DR17_D_Msk                                    /*!< bakeup data word17 */

/********************************* Bit definition for BKP_DR18 register *********************************************/
#define BKP_DR18_D_Pos                            (0U)
#define BKP_DR18_D_Msk                            (0xFFFFFFFFUL<<BKP_DR18_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR18_D                                BKP_DR18_D_Msk                                    /*!< bakeup data word18 */

/********************************* Bit definition for BKP_DR19 register *********************************************/
#define BKP_DR19_D_Pos                            (0U)
#define BKP_DR19_D_Msk                            (0xFFFFFFFFUL<<BKP_DR19_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR19_D                                BKP_DR19_D_Msk                                    /*!< bakeup data word19 */

/********************************* Bit definition for BKP_DR20 register *********************************************/
#define BKP_DR20_D_Pos                            (0U)
#define BKP_DR20_D_Msk                            (0xFFFFFFFFUL<<BKP_DR20_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR20_D                                BKP_DR20_D_Msk                                    /*!< bakeup data word20 */

/********************************* Bit definition for BKP_DR21 register *********************************************/
#define BKP_DR21_D_Pos                            (0U)
#define BKP_DR21_D_Msk                            (0xFFFFFFFFUL<<BKP_DR21_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR21_D                                BKP_DR21_D_Msk                                    /*!< bakeup data word21 */

/********************************* Bit definition for BKP_DR22 register *********************************************/
#define BKP_DR22_D_Pos                            (0U)
#define BKP_DR22_D_Msk                            (0xFFFFFFFFUL<<BKP_DR22_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR22_D                                BKP_DR22_D_Msk                                    /*!< bakeup data word22 */

/********************************* Bit definition for BKP_DR23 register *********************************************/
#define BKP_DR23_D_Pos                            (0U)
#define BKP_DR23_D_Msk                            (0xFFFFFFFFUL<<BKP_DR23_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR23_D                                BKP_DR23_D_Msk                                    /*!< bakeup data word23 */

/********************************* Bit definition for BKP_DR24 register *********************************************/
#define BKP_DR24_D_Pos                            (0U)
#define BKP_DR24_D_Msk                            (0xFFFFFFFFUL<<BKP_DR24_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR24_D                                BKP_DR24_D_Msk                                    /*!< bakeup data word24 */

/********************************* Bit definition for BKP_DR25 register *********************************************/
#define BKP_DR25_D_Pos                            (0U)
#define BKP_DR25_D_Msk                            (0xFFFFFFFFUL<<BKP_DR25_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR25_D                                BKP_DR25_D_Msk                                    /*!< bakeup data word25 */

/********************************* Bit definition for BKP_DR26 register *********************************************/
#define BKP_DR26_D_Pos                            (0U)
#define BKP_DR26_D_Msk                            (0xFFFFFFFFUL<<BKP_DR26_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR26_D                                BKP_DR26_D_Msk                                    /*!< bakeup data word26 */

/********************************* Bit definition for BKP_DR27 register *********************************************/
#define BKP_DR27_D_Pos                            (0U)
#define BKP_DR27_D_Msk                            (0xFFFFFFFFUL<<BKP_DR27_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR27_D                                BKP_DR27_D_Msk                                    /*!< bakeup data word27 */

/********************************* Bit definition for BKP_DR28 register *********************************************/
#define BKP_DR28_D_Pos                            (0U)
#define BKP_DR28_D_Msk                            (0xFFFFFFFFUL<<BKP_DR28_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR28_D                                BKP_DR28_D_Msk                                    /*!< bakeup data word28 */

/********************************* Bit definition for BKP_DR29 register *********************************************/
#define BKP_DR29_D_Pos                            (0U)
#define BKP_DR29_D_Msk                            (0xFFFFFFFFUL<<BKP_DR29_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR29_D                                BKP_DR29_D_Msk                                    /*!< bakeup data word29 */

/********************************* Bit definition for BKP_DR30 register *********************************************/
#define BKP_DR30_D_Pos                            (0U)
#define BKP_DR30_D_Msk                            (0xFFFFFFFFUL<<BKP_DR30_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR30_D                                BKP_DR30_D_Msk                                    /*!< bakeup data word30 */

/********************************* Bit definition for BKP_DR31 register *********************************************/
#define BKP_DR31_D_Pos                            (0U)
#define BKP_DR31_D_Msk                            (0xFFFFFFFFUL<<BKP_DR31_D_Pos)                    /*!< 0xFFFFFFFF */
#define BKP_DR31_D                                BKP_DR31_D_Msk                                    /*!< bakeup data word31 */

#define RTC_BKP_NUMBER 31

/****************************************************************************/
/*                                                                          */
/*                      Analog Comparators (COMP)                           */
/*                                                                          */
/****************************************************************************/
/**********************  Bit definition for COMP_CSR register  **************/
#define COMP_CSR_EN_Pos            (0U)
#define COMP_CSR_EN_Msk            (0x1UL << COMP_CSR_EN_Pos)                  /*!< 0x00000001 */
#define COMP_CSR_EN                COMP_CSR_EN_Msk                             /*!< Comparator enable */
#define COMP_CSR_COMP1_EN          COMP_CSR_EN
#define COMP_CSR_COMP2_EN          COMP_CSR_EN
#define COMP_CSR_COMP3_EN          COMP_CSR_EN
#define COMP_CSR_COMP4_EN          COMP_CSR_EN

#define COMP_CSR_VSEL_Pos          (1U)
#define COMP_CSR_VSEL_Msk          (0x3UL << COMP_CSR_VSEL_Pos)                /*!< 0x00000006 */
#define COMP_CSR_VSEL              COMP_CSR_VSEL_Msk                           /*!< VREFCMP DIV SOURCE voltage select */
#define COMP_CSR_VSEL_0            (0x1UL << COMP_CSR_VSEL_Pos)                /*!< 0x00000002 */
#define COMP_CSR_VSEL_1            (0x2UL << COMP_CSR_VSEL_Pos)                /*!< 0x00000004 */

#define COMP_CSR_INNSEL_Pos        (3U)
#define COMP_CSR_INNSEL_Msk        (0x7UL << COMP_CSR_INNSEL_Pos)              /*!< 0x00000038 */
#define COMP_CSR_INNSEL            COMP_CSR_INNSEL_Msk                         /*!< desc INNSEL */
#define COMP_CSR_INNSEL_0          (0x1UL << COMP_CSR_INNSEL_Pos)              /*!< 0x00000004 */
#define COMP_CSR_INNSEL_1          (0x2UL << COMP_CSR_INNSEL_Pos)              /*!< 0x00000008 */
#define COMP_CSR_INNSEL_2          (0x4UL << COMP_CSR_INNSEL_Pos)              /*!< 0x00000010 */

#define COMP_CSR_INPSEL_Pos        (9U)
#define COMP_CSR_INPSEL_Msk        (0x1UL << COMP_CSR_INPSEL_Pos)              /*!< 0x00000200 */
#define COMP_CSR_INPSEL            COMP_CSR_INPSEL_Msk                         /*!< Comparator plus minus selection */

#define COMP_CSR_WINMODE_Pos       (11U)
#define COMP_CSR_WINMODE_Msk       (0x1UL << COMP_CSR_WINMODE_Pos)             /*!< 0x00000800 */
#define COMP_CSR_WINMODE           COMP_CSR_WINMODE_Msk                        /*!< Pair of comparators window mode. Bit intended to be used with COMP common instance (COMP_Common_TypeDef) */

#define COMP_CSR_BLANKSEL_Pos      (12U)
#define COMP_CSR_BLANKSEL_Msk      (0x7UL << COMP_CSR_BLANKSEL_Pos)           /*!< 0x00000800 */
#define COMP_CSR_BLANKSEL          COMP_CSR_BLANKSEL_Msk                      /*!< Pair of comparators window mode. Bit intended to be used with COMP common instance (COMP_Common_TypeDef) */
#define COMP_CSR_BLANKSEL_0        (0x1UL << COMP_CSR_BLANKSEL_Pos)             /*!< 0x00040000 */
#define COMP_CSR_BLANKSEL_1        (0x2UL << COMP_CSR_BLANKSEL_Pos)             /*!< 0x00080000 */ 
#define COMP_CSR_BLANKSEL_2        (0x4UL << COMP_CSR_BLANKSEL_Pos)             /*!< 0x00080000 */ 

#define COMP_CSR_POLARITY_Pos      (15U)
#define COMP_CSR_POLARITY_Msk      (0x1UL << COMP_CSR_POLARITY_Pos)            /*!< 0x00008000 */
#define COMP_CSR_POLARITY          COMP_CSR_POLARITY_Msk                       /*!< Comparator output polarity */

#define COMP_CSR_HYST_Pos          (16U)
#define COMP_CSR_HYST_Msk          (0x1UL << COMP_CSR_HYST_Pos)                /*!< 0x00010000 */
#define COMP_CSR_HYST              COMP_CSR_HYST_Msk                           /*!< Comparator hysteresis enable */
/*DEFINE FOR COMP1 CSR_HYST*/
#define COMP1_CSR_HYST_Pos         (16U)
#define COMP1_CSR_HYST_Msk         (0x1UL << COMP1_CSR_HYST_Pos)                /*!< 0x00010000 */
#define COMP1_CSR_HYST             COMP1_CSR_HYST_Msk                           /*!< Comparator1 hysteresis enable */
/*DEFINE FOR COMP2 CSR_HYST*/
#define COMP2_CSR_HYST_Pos         (1U)
#define COMP2_CSR_HYST_Msk         (0x1UL << COMP2_CSR_HYST_Pos)                /*!< 0x00000002 */
#define COMP2_CSR_HYST             COMP2_CSR_HYST_Msk                           /*!< Comparator2 hysteresis enable */
/*DEFINE FOR COMP3 CSR_HYST*/
#define COMP3_CSR_HYST_Pos         (1U)
#define COMP3_CSR_HYST_Msk         (0x1UL << COMP3_CSR_HYST_Pos)                /*!< 0x00000002 */
#define COMP3_CSR_HYST             COMP3_CSR_HYST_Msk                           /*!< Comparator3 hysteresis enable */
/*DEFINE FOR COMP4 CSR_HYST*/
#define COMP4_CSR_HYST_Pos         (1U)
#define COMP4_CSR_HYST_Msk         (0x1UL << COMP4_CSR_HYST_Pos)                /*!< 0x00000002 */
#define COMP4_CSR_HYST             COMP4_CSR_HYST_Msk                           /*!< Comparator4 hysteresis enable */

#define COMP_CSR_PWRMODE_Pos       (18U)
#define COMP_CSR_PWRMODE_Msk       (0x3UL << COMP_CSR_PWRMODE_Pos)             /*!< 0x000C0000 */
#define COMP_CSR_PWRMODE           COMP_CSR_PWRMODE_Msk                        /*!< Comparator power mode */
#define COMP_CSR_PWRMODE_0         (0x1UL << COMP_CSR_PWRMODE_Pos)             /*!< 0x00040000 */
#define COMP_CSR_PWRMODE_1         (0x2UL << COMP_CSR_PWRMODE_Pos)             /*!< 0x00080000 */

#define COMP_CSR_VCDIV_Pos         (20U)
#define COMP_CSR_VCDIV_Msk         (0x3FUL << COMP_CSR_VCDIV_Pos)              /*!< 0x03F00000 */
#define COMP_CSR_VCDIV             COMP_CSR_VCDIV_Msk                          /*!< VCDIV[25:20] bits (desc VCDIV) */
#define COMP_CSR_VCDIV_0           (0x1UL << COMP_CSR_VCDIV_Pos)               /*!< 0x00100000 */
#define COMP_CSR_VCDIV_1           (0x2UL << COMP_CSR_VCDIV_Pos)               /*!< 0x00200000 */
#define COMP_CSR_VCDIV_2           (0x4UL << COMP_CSR_VCDIV_Pos)               /*!< 0x00400000 */
#define COMP_CSR_VCDIV_3           (0x8UL << COMP_CSR_VCDIV_Pos)               /*!< 0x00800000 */
#define COMP_CSR_VCDIV_4           (0x10UL << COMP_CSR_VCDIV_Pos)              /*!< 0x01000000 */
#define COMP_CSR_VCDIV_5           (0x20UL << COMP_CSR_VCDIV_Pos)              /*!< 0x02000000 */

#define COMP_CSR_VCDIV_EN_Pos      (26U)
#define COMP_CSR_VCDIV_EN_Msk      (0x1UL << COMP_CSR_VCDIV_EN_Pos)            /*!< 0x04000000 */
#define COMP_CSR_VCDIV_EN          COMP_CSR_VCDIV_EN_Msk                       /*!< desc VCDIV_EN */

#define COMP_CSR_VCSEL_Pos         (27U)
#define COMP_CSR_VCSEL_Msk         (0x1UL << COMP_CSR_VCSEL_Pos)               /*!< 0x08000000 */
#define COMP_CSR_VCSEL             COMP_CSR_VCSEL_Msk                          /*!< desc VCSEL */

#define COMP_CSR_TIM_EXTI_OUT_SEL_Pos         (29U)
#define COMP_CSR_TIM_EXTI_OUT_SEL_Msk         (0x1UL << COMP_CSR_TIM_EXTI_OUT_SEL_Pos)               /*!< 0x02000000 */
#define COMP_CSR_TIM_EXTI_OUT_SEL             COMP_CSR_TIM_EXTI_OUT_SEL_Msk                          /*!< COMP output to TIM/EXTI selection */

#define COMP_CSR_COMP_OUT_Pos       (30U)
#define COMP_CSR_COMP_OUT_Msk       (0x1UL << COMP_CSR_COMP_OUT_Pos)               /*!< 0x40000000 */
#define COMP_CSR_COMP_OUT           COMP_CSR_COMP_OUT_Msk                          /*!< COMP output status */

#define COMP_CSR_LOCK_Pos         (31U)
#define COMP_CSR_LOCK_Msk         (0x1UL << COMP_CSR_LOCK_Pos)               /*!< 0x40000000 */
#define COMP_CSR_LOCK             COMP_CSR_LOCK_Msk                          /*!< desc COMP_OUT */

/**********************  Bit definition for COMP_FR register  ***************/
#define COMP_FR_FLTEN_Pos          (0U)
#define COMP_FR_FLTEN_Msk          (0x1UL << COMP_FR_FLTEN_Pos)                /*!< 0x00000001 */
#define COMP_FR_FLTEN              COMP_FR_FLTEN_Msk                           /*!< Comparator filter enable */
#define COMP_FR_FLTCNT_Pos         (16U)
#define COMP_FR_FLTCNT_Msk         (0xFFFFUL << COMP_FR_FLTCNT_Pos)            /*!< 0xFFFF0000 */
#define COMP_FR_FLTCNT             COMP_FR_FLTCNT_Msk                          /*!< Comparator filter counter */


/********************************************************************************************************************/
/********************************* CORDIC ***************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for CORDIC_CSR register *******************************************/
#define CORDIC_CSR_FUNC_Pos                       (0U)
#define CORDIC_CSR_FUNC_Msk                       (0xFUL<<CORDIC_CSR_FUNC_Pos)                      /*!< 0x0000000F */
#define CORDIC_CSR_FUNC                           CORDIC_CSR_FUNC_Msk                               /*!<  Function */
#define CORDIC_CSR_FUNC_0                         (0x1UL<<CORDIC_CSR_FUNC_Pos)                      /*!< 0x00000001 */
#define CORDIC_CSR_FUNC_1                         (0x2UL<<CORDIC_CSR_FUNC_Pos)                      /*!< 0x00000002 */
#define CORDIC_CSR_FUNC_2                         (0x4UL<<CORDIC_CSR_FUNC_Pos)                      /*!< 0x00000004 */
#define CORDIC_CSR_FUNC_3                         (0x8UL<<CORDIC_CSR_FUNC_Pos)                      /*!< 0x00000008 */
#define CORDIC_CSR_PRECISION_Pos                  (4U)
#define CORDIC_CSR_PRECISION_Msk                  (0xFUL<<CORDIC_CSR_PRECISION_Pos)                 /*!< 0x000000F0 */
#define CORDIC_CSR_PRECISION                      CORDIC_CSR_PRECISION_Msk                          /*!<  Precision required (number of iterations) */
#define CORDIC_CSR_PRECISION_0                    (0x1UL<<CORDIC_CSR_PRECISION_Pos)                 /*!< 0x00000010 */
#define CORDIC_CSR_PRECISION_1                    (0x2UL<<CORDIC_CSR_PRECISION_Pos)                 /*!< 0x00000020 */
#define CORDIC_CSR_PRECISION_2                    (0x4UL<<CORDIC_CSR_PRECISION_Pos)                 /*!< 0x00000040 */
#define CORDIC_CSR_PRECISION_3                    (0x8UL<<CORDIC_CSR_PRECISION_Pos)                 /*!< 0x00000080 */
#define CORDIC_CSR_SCALE_Pos                      (8U)
#define CORDIC_CSR_SCALE_Msk                      (0x7UL<<CORDIC_CSR_SCALE_Pos)                     /*!< 0x00000700 */
#define CORDIC_CSR_SCALE                          CORDIC_CSR_SCALE_Msk                              /*!<  Scaling factor */
#define CORDIC_CSR_SCALE_0                        (0x1UL<<CORDIC_CSR_SCALE_Pos)                     /*!< 0x00000100 */
#define CORDIC_CSR_SCALE_1                        (0x2UL<<CORDIC_CSR_SCALE_Pos)                     /*!< 0x00000200 */
#define CORDIC_CSR_SCALE_2                        (0x4UL<<CORDIC_CSR_SCALE_Pos)                     /*!< 0x00000400 */
#define CORDIC_CSR_IEN_Pos                        (16U)
#define CORDIC_CSR_IEN_Msk                        (0x1UL<<CORDIC_CSR_IEN_Pos)                       /*!< 0x00010000 */
#define CORDIC_CSR_IEN                            CORDIC_CSR_IEN_Msk                                /*!<  Enable interrupt */
#define CORDIC_CSR_DMAREN_Pos                     (17U)
#define CORDIC_CSR_DMAREN_Msk                     (0x1UL<<CORDIC_CSR_DMAREN_Pos)                    /*!< 0x00020000 */
#define CORDIC_CSR_DMAREN                         CORDIC_CSR_DMAREN_Msk                             /*!< Enable DMA read channel */
#define CORDIC_CSR_DMAWEN_Pos                     (18U)
#define CORDIC_CSR_DMAWEN_Msk                     (0x1UL<<CORDIC_CSR_DMAWEN_Pos)                    /*!< 0x00040000 */
#define CORDIC_CSR_DMAWEN                         CORDIC_CSR_DMAWEN_Msk                             /*!< Enable DMA write channel */
#define CORDIC_CSR_NRES_Pos                       (19U)
#define CORDIC_CSR_NRES_Msk                       (0x1UL<<CORDIC_CSR_NRES_Pos)                      /*!< 0x00080000 */
#define CORDIC_CSR_NRES                           CORDIC_CSR_NRES_Msk                               /*!<  Number of results in the CORDIC_RDATA register */
#define CORDIC_CSR_NARGS_Pos                      (20U)
#define CORDIC_CSR_NARGS_Msk                      (0x1UL<<CORDIC_CSR_NARGS_Pos)                     /*!< 0x00100000 */
#define CORDIC_CSR_NARGS                          CORDIC_CSR_NARGS_Msk                              /*!<  Number of arguments expected by the CORDIC_WDATA register  */
#define CORDIC_CSR_RESSIZE_Pos                    (21U)
#define CORDIC_CSR_RESSIZE_Msk                    (0x1UL<<CORDIC_CSR_RESSIZE_Pos)                   /*!< 0x00200000 */
#define CORDIC_CSR_RESSIZE                        CORDIC_CSR_RESSIZE_Msk                            /*!< Width of input data */
#define CORDIC_CSR_ARGSIZE_Pos                    (22U)
#define CORDIC_CSR_ARGSIZE_Msk                    (0x1UL<<CORDIC_CSR_ARGSIZE_Pos)                   /*!< 0x00400000 */
#define CORDIC_CSR_ARGSIZE                        CORDIC_CSR_ARGSIZE_Msk                            /*!< Width of input data */
#define CORDIC_CSR_RRDY_Pos                       (31U)
#define CORDIC_CSR_RRDY_Msk                       (0x1UL<<CORDIC_CSR_RRDY_Pos)                      /*!< 0x80000000 */
#define CORDIC_CSR_RRDY                           CORDIC_CSR_RRDY_Msk                               /*!< Result ready flag */

/********************************* Bit definition for CORDIC_WDATA register *****************************************/
#define CORDIC_WDATA_ARG_Pos                      (16U)
#define CORDIC_WDATA_ARG_Msk                      (0xFFFFUL<<CORDIC_WDATA_ARG_Pos)                  /*!< 0xFFFF0000 */
#define CORDIC_WDATA_ARG                          CORDIC_WDATA_ARG_Msk                              /*!<  Function input arguments */

/********************************* Bit definition for CORDIC_RDATA register *****************************************/
#define CORDIC_RDATA_RES_Pos                      (16U)
#define CORDIC_RDATA_RES_Msk                      (0xFFFFUL<<CORDIC_RDATA_RES_Pos)                  /*!< 0xFFFF0000 */
#define CORDIC_RDATA_RES                          CORDIC_RDATA_RES_Msk                              /*!<  Function result */


/********************************************************************************************************************/
/********************************* CRC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for CRC_DR register ***********************************************/
#define CRC_DR_DR_Pos                             (0U)
#define CRC_DR_DR_Msk                             (0xFFFFFFFFUL<<CRC_DR_DR_Pos)                     /*!< 0xFFFFFFFF */
#define CRC_DR_DR                                 CRC_DR_DR_Msk                                     /*!< CRC data */

/********************************* Bit definition for CRC_IDR register **********************************************/
#define CRC_IDR_IDR_Pos                           (0U)
#define CRC_IDR_IDR_Msk                           (0xFFFFFFFFUL<<CRC_IDR_IDR_Pos)                   /*!< 0xFFFFFFFF */
#define CRC_IDR_IDR                               CRC_IDR_IDR_Msk                                   /*!< CRC independent data */

/********************************* Bit definition for CRC_CR register ***********************************************/
#define CRC_CR_RESET_Pos                          (0U)
#define CRC_CR_RESET_Msk                          (0x1UL<<CRC_CR_RESET_Pos)                         /*!< 0x00000001 */
#define CRC_CR_RESET                              CRC_CR_RESET_Msk                                  /*!< CRC software reset */
#define CRC_CR_POLYSIZE_Pos                       (3U)
#define CRC_CR_POLYSIZE_Msk                       (0x3UL<<CRC_CR_POLYSIZE_Pos)                      /*!< 0x00000018 */
#define CRC_CR_POLYSIZE                           CRC_CR_POLYSIZE_Msk                               /*!< Polynomial size */
#define CRC_CR_POLYSIZE_0                         (0x1UL<<CRC_CR_POLYSIZE_Pos)                      /*!< 0x00000008 */
#define CRC_CR_POLYSIZE_1                         (0x2UL<<CRC_CR_POLYSIZE_Pos)                      /*!< 0x00000010 */
#define CRC_CR_REV_IN_Pos                         (5U)
#define CRC_CR_REV_IN_Msk                         (0x3UL<<CRC_CR_REV_IN_Pos)                        /*!< 0x00000060 */
#define CRC_CR_REV_IN                             CRC_CR_REV_IN_Msk                                 /*!< Reverse input data */
#define CRC_CR_REV_IN_0                           (0x1UL<<CRC_CR_REV_IN_Pos)                        /*!< 0x00000020 */
#define CRC_CR_REV_IN_1                           (0x2UL<<CRC_CR_REV_IN_Pos)                        /*!< 0x00000040 */
#define CRC_CR_REV_OUT_Pos                        (7U)
#define CRC_CR_REV_OUT_Msk                        (0x1UL<<CRC_CR_REV_OUT_Pos)                       /*!< 0x00000080 */
#define CRC_CR_REV_OUT                            CRC_CR_REV_OUT_Msk                                /*!< Reverse output data */

/********************************* Bit definition for CRC_INIT register *********************************************/
#define CRC_INIT_CRC_INIT_Pos                     (0U)
#define CRC_INIT_CRC_INIT_Msk                     (0xFFFFFFFFUL<<CRC_INIT_CRC_INIT_Pos)             /*!< 0xFFFFFFFF */
#define CRC_INIT_CRC_INIT                         CRC_INIT_CRC_INIT_Msk                             /*!< initial value */

/********************************* Bit definition for CRC_POL register **********************************************/
#define CRC_POL_CRC_POL_Pos                       (0U)
#define CRC_POL_CRC_POL_Msk                       (0xFFFFFFFFUL<<CRC_POL_CRC_POL_Pos)               /*!< 0xFFFFFFFF */
#define CRC_POL_CRC_POL                           CRC_POL_CRC_POL_Msk                               /*!< polynomial  */

/********************************************************************************************************************/
/********************************* CTC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for CTC_CTL0 register *********************************************/
#define CTC_CTL0_CKOKIE_Pos                       (0U)
#define CTC_CTL0_CKOKIE_Msk                       (0x1UL<<CTC_CTL0_CKOKIE_Pos)                      /*!< 0x00000001 */
#define CTC_CTL0_CKOKIE                           CTC_CTL0_CKOKIE_Msk                               /*!< Clock calibration completion interrupt enable */
#define CTC_CTL0_CKWARNIE_Pos                     (1U)
#define CTC_CTL0_CKWARNIE_Msk                     (0x1UL<<CTC_CTL0_CKWARNIE_Pos)                    /*!< 0x00000002 */
#define CTC_CTL0_CKWARNIE                         CTC_CTL0_CKWARNIE_Msk                             /*!< Clock calibration warning interrupt enable */
#define CTC_CTL0_ERRIE_Pos                        (2U)
#define CTC_CTL0_ERRIE_Msk                        (0x1UL<<CTC_CTL0_ERRIE_Pos)                       /*!< 0x00000004 */
#define CTC_CTL0_ERRIE                            CTC_CTL0_ERRIE_Msk                                /*!< Error interrupt enable */
#define CTC_CTL0_EREFIE_Pos                       (3U)
#define CTC_CTL0_EREFIE_Msk                       (0x1UL<<CTC_CTL0_EREFIE_Pos)                      /*!< 0x00000008 */
#define CTC_CTL0_EREFIE                           CTC_CTL0_EREFIE_Msk                               /*!< Expected reference signal interrupt enable */
#define CTC_CTL0_CNTEN_Pos                        (5U)
#define CTC_CTL0_CNTEN_Msk                        (0x1UL<<CTC_CTL0_CNTEN_Pos)                       /*!< 0x00000020 */
#define CTC_CTL0_CNTEN                            CTC_CTL0_CNTEN_Msk                                /*!< CTC count enable */
#define CTC_CTL0_AUTOTRIM_Pos                     (6U)
#define CTC_CTL0_AUTOTRIM_Msk                     (0x1UL<<CTC_CTL0_AUTOTRIM_Pos)                    /*!< 0x00000040 */
#define CTC_CTL0_AUTOTRIM                         CTC_CTL0_AUTOTRIM_Msk                             /*!< Automatic trim mode */
#define CTC_CTL0_SWREFPUL_Pos                     (7U)
#define CTC_CTL0_SWREFPUL_Msk                     (0x1UL<<CTC_CTL0_SWREFPUL_Pos)                    /*!< 0x00000080 */
#define CTC_CTL0_SWREFPUL                         CTC_CTL0_SWREFPUL_Msk                             /*!< Software generates synchronous reference signal pulses */
#define CTC_CTL0_TRIMVALUE_Pos                    (8U)
#define CTC_CTL0_TRIMVALUE_Msk                    (0x7FUL<<CTC_CTL0_TRIMVALUE_Pos)                  /*!< 0x00007F00 */
#define CTC_CTL0_TRIMVALUE                        CTC_CTL0_TRIMVALUE_Msk                            /*!< Trim Value */
#define CTC_CTL0_TRIMVALUE_0                      (0x1UL << CTC_CTL0_TRIMVALUE_Pos)                 /*!< 0x00000100 */
#define CTC_CTL0_TRIMVALUE_1                      (0x2UL << CTC_CTL0_TRIMVALUE_Pos)                 /*!< 0x00000200 */
#define CTC_CTL0_TRIMVALUE_2                      (0x4UL << CTC_CTL0_TRIMVALUE_Pos)                 /*!< 0x00000400 */
#define CTC_CTL0_TRIMVALUE_3                      (0x8UL << CTC_CTL0_TRIMVALUE_Pos)                 /*!< 0x00000800 */
#define CTC_CTL0_TRIMVALUE_4                      (0x10UL << CTC_CTL0_TRIMVALUE_Pos)                /*!< 0x00001000 */
#define CTC_CTL0_TRIMVALUE_5                      (0x20UL << CTC_CTL0_TRIMVALUE_Pos)                /*!< 0x00002000 */
#define CTC_CTL0_TRIMVALUE_6                      (0x40UL << CTC_CTL0_TRIMVALUE_Pos)                /*!< 0x00004000 */

/********************************* Bit definition for CTC_CTL1 register *********************************************/
#define CTC_CTL1_RLVALUE_Pos                      (0U)
#define CTC_CTL1_RLVALUE_Msk                      (0xFFFFUL<<CTC_CTL1_RLVALUE_Pos)                  /*!< 0x0000FFFF */
#define CTC_CTL1_RLVALUE                          CTC_CTL1_RLVALUE_Msk                              /*!< CTC counter reload value */
#define CTC_CTL1_CKLIM_Pos                        (16U)
#define CTC_CTL1_CKLIM_Msk                        (0xFFUL<<CTC_CTL1_CKLIM_Pos)                      /*!< 0x00FF0000 */
#define CTC_CTL1_CKLIM                            CTC_CTL1_CKLIM_Msk                                /*!< Clock calibration time base limit */
#define CTC_CTL1_REFPSC_Pos                       (24U)
#define CTC_CTL1_REFPSC_Msk                       (0x7UL<<CTC_CTL1_REFPSC_Pos)                      /*!< 0x07000000 */
#define CTC_CTL1_REFPSC                           CTC_CTL1_REFPSC_Msk                               /*!< Reference signal source pre division */
#define CTC_CTL1_REFPSC_0                         (0x1UL<<CTC_CTL1_REFPSC_Pos)                      /*!< 0x01000000 */
#define CTC_CTL1_REFPSC_1                         (0x2UL<<CTC_CTL1_REFPSC_Pos)                      /*!< 0x02000000 */
#define CTC_CTL1_REFPSC_2                         (0x4UL<<CTC_CTL1_REFPSC_Pos)                      /*!< 0x04000000 */
#define CTC_CTL1_REFSEL_Pos                       (28U)
#define CTC_CTL1_REFSEL_Msk                       (0x3UL<<CTC_CTL1_REFSEL_Pos)                      /*!< 0x30000000 */
#define CTC_CTL1_REFSEL                           CTC_CTL1_REFSEL_Msk                               /*!< Selection of reference signal source */
#define CTC_CTL1_REFSEL_0                         (0x1UL<<CTC_CTL1_REFSEL_Pos)                      /*!< 0x10000000 */
#define CTC_CTL1_REFSEL_1                         (0x2UL<<CTC_CTL1_REFSEL_Pos)                      /*!< 0x20000000 */
#define CTC_CTL1_REFPOL_Pos                       (31U)
#define CTC_CTL1_REFPOL_Msk                       (0x1UL<<CTC_CTL1_REFPOL_Pos)                      /*!< 0x80000000 */
#define CTC_CTL1_REFPOL                           CTC_CTL1_REFPOL_Msk                               /*!< Reference signal source polarity */

/********************************* Bit definition for CTC_SR register ***********************************************/
#define CTC_SR_CKOKIF_Pos                         (0U)
#define CTC_SR_CKOKIF_Msk                         (0x1UL<<CTC_SR_CKOKIF_Pos)                        /*!< 0x00000001 */
#define CTC_SR_CKOKIF                             CTC_SR_CKOKIF_Msk                                 /*!< Clock calibration successful interrupt flag bit */
#define CTC_SR_CKWARNIF_Pos                       (1U)
#define CTC_SR_CKWARNIF_Msk                       (0x1UL<<CTC_SR_CKWARNIF_Pos)                      /*!< 0x00000002 */
#define CTC_SR_CKWARNIF                           CTC_SR_CKWARNIF_Msk                               /*!< Clock calibration warning interrupt flag bit */
#define CTC_SR_ERRIF_Pos                          (2U)
#define CTC_SR_ERRIF_Msk                          (0x1UL<<CTC_SR_ERRIF_Pos)                         /*!< 0x00000004 */
#define CTC_SR_ERRIF                              CTC_SR_ERRIF_Msk                                  /*!< Error interrupt flag bit */
#define CTC_SR_EREFIF_Pos                         (3U)
#define CTC_SR_EREFIF_Msk                         (0x1UL<<CTC_SR_EREFIF_Pos)                        /*!< 0x00000008 */
#define CTC_SR_EREFIF                             CTC_SR_EREFIF_Msk                                 /*!< Expected reference interrupt flag bit */
#define CTC_SR_CKERR_Pos                          (8U)
#define CTC_SR_CKERR_Msk                          (0x1UL<<CTC_SR_CKERR_Pos)                         /*!< 0x00000100 */
#define CTC_SR_CKERR                              CTC_SR_CKERR_Msk                                  /*!< Clock calibration error bit */
#define CTC_SR_REFMISS_Pos                        (9U)
#define CTC_SR_REFMISS_Msk                        (0x1UL<<CTC_SR_REFMISS_Pos)                       /*!< 0x00000200 */
#define CTC_SR_REFMISS                            CTC_SR_REFMISS_Msk                                /*!< Loss of synchronous reference pulse signal */
#define CTC_SR_TRIMERR_Pos                        (10U)
#define CTC_SR_TRIMERR_Msk                        (0x1UL<<CTC_SR_TRIMERR_Pos)                       /*!< 0x00000400 */
#define CTC_SR_TRIMERR                            CTC_SR_TRIMERR_Msk                                /*!< Calibration value error bit */
#define CTC_SR_REFDIR_Pos                         (15U)
#define CTC_SR_REFDIR_Msk                         (0x1UL<<CTC_SR_REFDIR_Pos)                        /*!< 0x00008000 */
#define CTC_SR_REFDIR                             CTC_SR_REFDIR_Msk                                 /*!< CTC calibration clock counting direction */
#define CTC_SR_REFCAP_Pos                         (16U)
#define CTC_SR_REFCAP_Msk                         (0xFFFFUL<<CTC_SR_REFCAP_Pos)                     /*!< 0xFFFF0000 */
#define CTC_SR_REFCAP                             CTC_SR_REFCAP_Msk                                 /*!< CTC counter capture value */

/********************************* Bit definition for CTC_INTC register *********************************************/
#define CTC_INTC_CKOKIC_Pos                       (0U)
#define CTC_INTC_CKOKIC_Msk                       (0x1UL<<CTC_INTC_CKOKIC_Pos)                      /*!< 0x00000001 */
#define CTC_INTC_CKOKIC                           CTC_INTC_CKOKIC_Msk                               /*!< CKOKIF interrupt clear bit */
#define CTC_INTC_CKWARNIC_Pos                     (1U)
#define CTC_INTC_CKWARNIC_Msk                     (0x1UL<<CTC_INTC_CKWARNIC_Pos)                    /*!< 0x00000002 */
#define CTC_INTC_CKWARNIC                         CTC_INTC_CKWARNIC_Msk                             /*!< CKWARNIF interrupt clear bit */
#define CTC_INTC_ERRIC_Pos                        (2U)
#define CTC_INTC_ERRIC_Msk                        (0x1UL<<CTC_INTC_ERRIC_Pos)                       /*!< 0x00000004 */
#define CTC_INTC_ERRIC                            CTC_INTC_ERRIC_Msk                                /*!< ERRIF interrupt clear bit */
#define CTC_INTC_EREFIC_Pos                       (3U)
#define CTC_INTC_EREFIC_Msk                       (0x1UL<<CTC_INTC_EREFIC_Pos)                      /*!< 0x00000008 */
#define CTC_INTC_EREFIC                           CTC_INTC_EREFIC_Msk                               /*!< EREFIF interrupt clear bit */

/********************************************************************************************************************/
/********************************* DAC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for DAC_CR register ***********************************************/
#define DAC_CR_EN1_Pos                            (0U)
#define DAC_CR_EN1_Msk                            (0x1UL<<DAC_CR_EN1_Pos)                           /*!< 0x00000001 */
#define DAC_CR_EN1                                DAC_CR_EN1_Msk                                    
#define DAC_CR_BOFF1_Pos                          (1U)
#define DAC_CR_BOFF1_Msk                          (0x1UL<<DAC_CR_BOFF1_Pos)                         /*!< 0x00000002 */
#define DAC_CR_BOFF1                              DAC_CR_BOFF1_Msk                                  
#define DAC_CR_TEN1_Pos                           (2U)
#define DAC_CR_TEN1_Msk                           (0x1UL<<DAC_CR_TEN1_Pos)                          /*!< 0x00000004 */
#define DAC_CR_TEN1                               DAC_CR_TEN1_Msk                                   
#define DAC_CR_TSEL1_Pos                          (3U)
#define DAC_CR_TSEL1_Msk                          (0xFUL<<DAC_CR_TSEL1_Pos)                         /*!< 0x00000078 */
#define DAC_CR_TSEL1                              DAC_CR_TSEL1_Msk
#define DAC_CR_TSEL1_0                            (0x1UL<<DAC_CR_TSEL1_Pos)                         /*!< 0x00000008 */
#define DAC_CR_TSEL1_1                            (0x2UL<<DAC_CR_TSEL1_Pos)                         /*!< 0x00000010 */
#define DAC_CR_TSEL1_2                            (0x4UL<<DAC_CR_TSEL1_Pos)                         /*!< 0x00000020 */
#define DAC_CR_TSEL1_3                            (0x8UL<<DAC_CR_TSEL1_Pos)                         /*!< 0x00000040 */
#define DAC_CR_WAVE1_Pos                          (7U)
#define DAC_CR_WAVE1_Msk                          (0x3UL<<DAC_CR_WAVE1_Pos)                         /*!< 0x00000180 */
#define DAC_CR_WAVE1                              DAC_CR_WAVE1_Msk
#define DAC_CR_WAVE1_0                            (0x1UL<<DAC_CR_WAVE1_Pos)                         /*!< 0x00000080 */
#define DAC_CR_WAVE1_1                            (0x2UL<<DAC_CR_WAVE1_Pos)                         /*!< 0x00000100 */
#define DAC_CR_MAMP1_Pos                          (9U)
#define DAC_CR_MAMP1_Msk                          (0xFUL<<DAC_CR_MAMP1_Pos)                         /*!< 0x00001E00 */
#define DAC_CR_MAMP1                              DAC_CR_MAMP1_Msk
#define DAC_CR_MAMP1_0                            (0x1UL<<DAC_CR_MAMP1_Pos)                         /*!< 0x00000200 */
#define DAC_CR_MAMP1_1                            (0x2UL<<DAC_CR_MAMP1_Pos)                         /*!< 0x00000400 */
#define DAC_CR_MAMP1_2                            (0x4UL<<DAC_CR_MAMP1_Pos)                         /*!< 0x00000800 */
#define DAC_CR_MAMP1_3                            (0x8UL<<DAC_CR_MAMP1_Pos)                         /*!< 0x00001000 */
#define DAC_CR_DMAEN1_Pos                         (13U)
#define DAC_CR_DMAEN1_Msk                         (0x1UL<<DAC_CR_DMAEN1_Pos)                        /*!< 0x00002000 */
#define DAC_CR_DMAEN1                             DAC_CR_DMAEN1_Msk                                 
#define DAC_CR_DMAUDRIE1_Pos                      (14U)
#define DAC_CR_DMAUDRIE1_Msk                      (0x1UL<<DAC_CR_DMAUDRIE1_Pos)                     /*!< 0x00004000 */
#define DAC_CR_DMAUDRIE1                          DAC_CR_DMAUDRIE1_Msk                              
#define DAC_CR_DAC1CEN_Pos                        (15U)
#define DAC_CR_DAC1CEN_Msk                        (0x1UL<<DAC_CR_DAC1CEN_Pos)                       /*!< 0x00008000 */
#define DAC_CR_DAC1CEN                            DAC_CR_DAC1CEN_Msk                                
#define DAC_CR_EN2_Pos                            (16U)
#define DAC_CR_EN2_Msk                            (0x1UL<<DAC_CR_EN2_Pos)                           /*!< 0x00010000 */
#define DAC_CR_EN2                                DAC_CR_EN2_Msk                                    
#define DAC_CR_BOFF2_Pos                          (17U)
#define DAC_CR_BOFF2_Msk                          (0x1UL<<DAC_CR_BOFF2_Pos)                         /*!< 0x00020000 */
#define DAC_CR_BOFF2                              DAC_CR_BOFF2_Msk                                  
#define DAC_CR_TEN2_Pos                           (18U)
#define DAC_CR_TEN2_Msk                           (0x1UL<<DAC_CR_TEN2_Pos)                          /*!< 0x00040000 */
#define DAC_CR_TEN2                               DAC_CR_TEN2_Msk                                   
#define DAC_CR_TSEL2_Pos                          (19U)
#define DAC_CR_TSEL2_Msk                          (0xFUL<<DAC_CR_TSEL2_Pos)                         /*!< 0x00780000 */
#define DAC_CR_TSEL2                              DAC_CR_TSEL2_Msk
#define DAC_CR_TSEL2_0                            (0x1UL<<DAC_CR_TSEL2_Pos)                         /*!< 0x00080000 */
#define DAC_CR_TSEL2_1                            (0x2UL<<DAC_CR_TSEL2_Pos)                         /*!< 0x00100000 */
#define DAC_CR_TSEL2_2                            (0x4UL<<DAC_CR_TSEL2_Pos)                         /*!< 0x00200000 */
#define DAC_CR_TSEL2_3                            (0x8UL<<DAC_CR_TSEL2_Pos)                         /*!< 0x00400000 */
#define DAC_CR_WAVE2_Pos                          (23U)
#define DAC_CR_WAVE2_Msk                          (0x3UL<<DAC_CR_WAVE2_Pos)                         /*!< 0x01800000 */
#define DAC_CR_WAVE2                              DAC_CR_WAVE2_Msk
#define DAC_CR_WAVE2_0                            (0x1UL<<DAC_CR_WAVE2_Pos)                         /*!< 0x00800000 */
#define DAC_CR_WAVE2_1                            (0x2UL<<DAC_CR_WAVE2_Pos)                         /*!< 0x01000000 */
#define DAC_CR_MAMP2_Pos                          (25U)
#define DAC_CR_MAMP2_Msk                          (0xFUL<<DAC_CR_MAMP2_Pos)                         /*!< 0x1E000000 */
#define DAC_CR_MAMP2                              DAC_CR_MAMP2_Msk
#define DAC_CR_MAMP2_0                            (0x1UL<<DAC_CR_MAMP2_Pos)                         /*!< 0x02000000 */
#define DAC_CR_MAMP2_1                            (0x2UL<<DAC_CR_MAMP2_Pos)                         /*!< 0x04000000 */
#define DAC_CR_MAMP2_2                            (0x4UL<<DAC_CR_MAMP2_Pos)                         /*!< 0x08000000 */
#define DAC_CR_MAMP2_3                            (0x8UL<<DAC_CR_MAMP2_Pos)                         /*!< 0x10000000 */
#define DAC_CR_DMAEN2_Pos                         (29U)
#define DAC_CR_DMAEN2_Msk                         (0x1UL<<DAC_CR_DMAEN2_Pos)                        /*!< 0x20000000 */
#define DAC_CR_DMAEN2                             DAC_CR_DMAEN2_Msk                                 
#define DAC_CR_DMAUDRIE2_Pos                      (30U)
#define DAC_CR_DMAUDRIE2_Msk                      (0x1UL<<DAC_CR_DMAUDRIE2_Pos)                     /*!< 0x40000000 */
#define DAC_CR_DMAUDRIE2                          DAC_CR_DMAUDRIE2_Msk                              
#define DAC_CR_DAC2CEN_Pos                        (31U)
#define DAC_CR_DAC2CEN_Msk                        (0x1UL<<DAC_CR_DAC2CEN_Pos)                       /*!< 0x80000000 */
#define DAC_CR_DAC2CEN                            DAC_CR_DAC2CEN_Msk                                

/********************************* Bit definition for DAC_SWTRIGR register ******************************************/
#define DAC_SWTRIGR_SWTRIG1_Pos                   (0U)
#define DAC_SWTRIGR_SWTRIG1_Msk                   (0x1UL<<DAC_SWTRIGR_SWTRIG1_Pos)                  /*!< 0x00000001 */
#define DAC_SWTRIGR_SWTRIG1                       DAC_SWTRIGR_SWTRIG1_Msk                           
#define DAC_SWTRIGR_SWTRIG2_Pos                   (1U)
#define DAC_SWTRIGR_SWTRIG2_Msk                   (0x1UL<<DAC_SWTRIGR_SWTRIG2_Pos)                  /*!< 0x00000002 */
#define DAC_SWTRIGR_SWTRIG2                       DAC_SWTRIGR_SWTRIG2_Msk                           

/********************************* Bit definition for DAC_DHR12R1 register ******************************************/
#define DAC_DHR12R1_DACC1DHR_Pos                  (0U)
#define DAC_DHR12R1_DACC1DHR_Msk                  (0xFFFUL<<DAC_DHR12R1_DACC1DHR_Pos)               /*!< 0x00000FFF */
#define DAC_DHR12R1_DACC1DHR                      DAC_DHR12R1_DACC1DHR_Msk

/********************************* Bit definition for DAC_DHR12L1 register ******************************************/
#define DAC_DHR12L1_DACC1DHR_Pos                  (4U)
#define DAC_DHR12L1_DACC1DHR_Msk                  (0xFFFUL<<DAC_DHR12L1_DACC1DHR_Pos)               /*!< 0x0000FFF0 */
#define DAC_DHR12L1_DACC1DHR                      DAC_DHR12L1_DACC1DHR_Msk

/********************************* Bit definition for DAC_DHR8R1 register *******************************************/
#define DAC_DHR8R1_DACC1DHR_Pos                   (0U)
#define DAC_DHR8R1_DACC1DHR_Msk                   (0xFFUL<<DAC_DHR8R1_DACC1DHR_Pos)                 /*!< 0x000000FF */
#define DAC_DHR8R1_DACC1DHR                       DAC_DHR8R1_DACC1DHR_Msk

/********************************* Bit definition for DAC_DHR12R2 register ******************************************/
#define DAC_DHR12R2_DACC2DHR_Pos                  (0U)
#define DAC_DHR12R2_DACC2DHR_Msk                  (0xFFFUL<<DAC_DHR12R2_DACC2DHR_Pos)               /*!< 0x00000FFF */
#define DAC_DHR12R2_DACC2DHR                      DAC_DHR12R2_DACC2DHR_Msk

/********************************* Bit definition for DAC_DHR12L2 register ******************************************/
#define DAC_DHR12L2_DACC2DHR_Pos                  (4U)
#define DAC_DHR12L2_DACC2DHR_Msk                  (0xFFFUL<<DAC_DHR12L2_DACC2DHR_Pos)               /*!< 0x0000FFF0 */
#define DAC_DHR12L2_DACC2DHR                      DAC_DHR12L2_DACC2DHR_Msk

/********************************* Bit definition for DAC_DHR8R2 register *******************************************/
#define DAC_DHR8R2_DACC2DHR_Pos                   (0U)
#define DAC_DHR8R2_DACC2DHR_Msk                   (0xFFUL<<DAC_DHR8R2_DACC2DHR_Pos)                 /*!< 0x000000FF */
#define DAC_DHR8R2_DACC2DHR                       DAC_DHR8R2_DACC2DHR_Msk

/********************************* Bit definition for DAC_DHR12RD register ******************************************/
#define DAC_DHR12RD_DACC1DHR_Pos                  (0U)
#define DAC_DHR12RD_DACC1DHR_Msk                  (0xFFFUL<<DAC_DHR12RD_DACC1DHR_Pos)               /*!< 0x00000FFF */
#define DAC_DHR12RD_DACC1DHR                      DAC_DHR12RD_DACC1DHR_Msk
#define DAC_DHR12RD_DACC2DHR_Pos                  (16U)
#define DAC_DHR12RD_DACC2DHR_Msk                  (0xFFFUL<<DAC_DHR12RD_DACC2DHR_Pos)               /*!< 0x0FFF0000 */
#define DAC_DHR12RD_DACC2DHR                      DAC_DHR12RD_DACC2DHR_Msk

/********************************* Bit definition for DAC_DHR12LD register ******************************************/
#define DAC_DHR12LD_DACC1DHR_Pos                  (4U)
#define DAC_DHR12LD_DACC1DHR_Msk                  (0xFFFUL<<DAC_DHR12LD_DACC1DHR_Pos)               /*!< 0x0000FFF0 */
#define DAC_DHR12LD_DACC1DHR                      DAC_DHR12LD_DACC1DHR_Msk                          /*!< DACC1DHR[15:4] bits (desc DACC1DHR) */
#define DAC_DHR12LD_DACC2DHR_Pos                  (20U)
#define DAC_DHR12LD_DACC2DHR_Msk                  (0xFFFUL<<DAC_DHR12LD_DACC2DHR_Pos)               /*!< 0xFFF00000 */
#define DAC_DHR12LD_DACC2DHR                      DAC_DHR12LD_DACC2DHR_Msk                          /*!< DACC2DHR[31:20] bits (desc DACC2DHR) */

/********************************* Bit definition for DAC_DHR8RD register *******************************************/
#define DAC_DHR8RD_DACC1DHR_Pos                   (0U)
#define DAC_DHR8RD_DACC1DHR_Msk                   (0xFFUL<<DAC_DHR8RD_DACC1DHR_Pos)                 /*!< 0x000000FF */
#define DAC_DHR8RD_DACC1DHR                       DAC_DHR8RD_DACC1DHR_Msk
#define DAC_DHR8RD_DACC2DHR_Pos                   (8U)
#define DAC_DHR8RD_DACC2DHR_Msk                   (0xFFUL<<DAC_DHR8RD_DACC2DHR_Pos)                 /*!< 0x0000FF00 */
#define DAC_DHR8RD_DACC2DHR                       DAC_DHR8RD_DACC2DHR_Msk

/********************************* Bit definition for DAC_DOR1 register *********************************************/
#define DAC_DOR1_DACC1DOR_Pos                     (0U)
#define DAC_DOR1_DACC1DOR_Msk                     (0xFFFUL<<DAC_DOR1_DACC1DOR_Pos)                  /*!< 0x00000FFF */
#define DAC_DOR1_DACC1DOR                         DAC_DOR1_DACC1DOR_Msk

/********************************* Bit definition for DAC_DOR2 register *********************************************/
#define DAC_DOR2_DACC2DOR_Pos                     (0U)
#define DAC_DOR2_DACC2DOR_Msk                     (0xFFFUL<<DAC_DOR2_DACC2DOR_Pos)                  /*!< 0x00000FFF */
#define DAC_DOR2_DACC2DOR                         DAC_DOR2_DACC2DOR_Msk

/********************************* Bit definition for DAC_SR register ***********************************************/
#define DAC_SR_DMAUDR1_Pos                        (13U)
#define DAC_SR_DMAUDR1_Msk                        (0x1UL<<DAC_SR_DMAUDR1_Pos)                       /*!< 0x00002000 */
#define DAC_SR_DMAUDR1                            DAC_SR_DMAUDR1_Msk                                
#define DAC_SR_DMAUDR2_Pos                        (29U)
#define DAC_SR_DMAUDR2_Msk                        (0x1UL<<DAC_SR_DMAUDR2_Pos)                       /*!< 0x20000000 */
#define DAC_SR_DMAUDR2                            DAC_SR_DMAUDR2_Msk                                

/********************************* Bit definition for DAC_OC register ***********************************************/
#define DAC_OC_INTEREN1_Pos                       (0U)
#define DAC_OC_INTEREN1_Msk                       (0x1UL<<DAC_OC_INTEREN1_Pos)                      /*!< 0x00000001 */
#define DAC_OC_INTEREN1                           DAC_OC_INTEREN1_Msk                               
#define DAC_OC_INTEREN2_Pos                       (1U)
#define DAC_OC_INTEREN2_Msk                       (0x1UL<<DAC_OC_INTEREN2_Pos)                      /*!< 0x00000002 */
#define DAC_OC_INTEREN2                           DAC_OC_INTEREN2_Msk                               

/********************************************************************************************************************/
/********************************* DBG ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for DBGMCU_IDCODE register ****************************************/
#define DBGMCU_IDCODE_REV_ID_Pos                   (0U)
#define DBGMCU_IDCODE_REV_ID_Msk                   (0xFFFFFFFFUL << DBGMCU_IDCODE_REV_ID_Pos)          /*!< 0xFFFFFFFF */
#define DBGMCU_IDCODE_REV_ID                       DBGMCU_IDCODE_REV_ID_Msk                            /*!< REV_ID[31:0] bits (desc REV_ID) */

/********************************* Bit definition for DBGMCU_CR1 register *******************************************/
#define DBGMCU_CR1_DBG_SLEEP_Pos                   (0U)
#define DBGMCU_CR1_DBG_SLEEP_Msk                   (0x1UL << DBGMCU_CR1_DBG_SLEEP_Pos)                /*!< 0x00000001 */
#define DBGMCU_CR1_DBG_SLEEP                       DBGMCU_CR1_DBG_SLEEP_Msk                           /*!< desc DBG_SLEEP */
#define DBGMCU_CR1_DBG_STOP_Pos                    (1U)
#define DBGMCU_CR1_DBG_STOP_Msk                    (0x1UL << DBGMCU_CR1_DBG_STOP_Pos)                 /*!< 0x00000002 */
#define DBGMCU_CR1_DBG_STOP                        DBGMCU_CR1_DBG_STOP_Msk                            /*!< desc DBG_STOP */
#define DBGMCU_CR1_DBG_STDBY_Pos                   (2U)
#define DBGMCU_CR1_DBG_STDBY_Msk                   (0x1UL << DBGMCU_CR1_DBG_STDBY_Pos)                /*!< 0x00000004 */
#define DBGMCU_CR1_DBG_STDBY                       DBGMCU_CR1_DBG_STDBY_Msk                           /*!< desc DBG_STDBY */
#define DBGMCU_CR1_DBG_IWDG_STOP_Pos               (8U)
#define DBGMCU_CR1_DBG_IWDG_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_IWDG_STOP_Pos)            /*!< 0x00000100 */
#define DBGMCU_CR1_DBG_IWDG_STOP                   DBGMCU_CR1_DBG_IWDG_STOP_Msk                       /*!< desc DBG_IWDG_STOP */
#define DBGMCU_CR1_DBG_WWDG_STOP_Pos               (9U)
#define DBGMCU_CR1_DBG_WWDG_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_WWDG_STOP_Pos)            /*!< 0x00000200 */
#define DBGMCU_CR1_DBG_WWDG_STOP                   DBGMCU_CR1_DBG_WWDG_STOP_Msk                       /*!< desc DBG_WWDG_STOP */
#define DBGMCU_CR1_DBG_TIM1_STOP_Pos               (10U)
#define DBGMCU_CR1_DBG_TIM1_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM1_STOP_Pos)            /*!< 0x00000400 */
#define DBGMCU_CR1_DBG_TIM1_STOP                   DBGMCU_CR1_DBG_TIM1_STOP_Msk                       /*!< desc DBG_TIM1_STOP */
#define DBGMCU_CR1_DBG_TIM2_STOP_Pos               (11U)
#define DBGMCU_CR1_DBG_TIM2_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM2_STOP_Pos)            /*!< 0x00000800 */
#define DBGMCU_CR1_DBG_TIM2_STOP                   DBGMCU_CR1_DBG_TIM2_STOP_Msk                       /*!< desc DBG_TIM2_STOP */
#define DBGMCU_CR1_DBG_TIM3_STOP_Pos               (12U)
#define DBGMCU_CR1_DBG_TIM3_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM3_STOP_Pos)            /*!< 0x00001000 */
#define DBGMCU_CR1_DBG_TIM3_STOP                   DBGMCU_CR1_DBG_TIM3_STOP_Msk                       /*!< desc DBG_TIM3_STOP */
#define DBGMCU_CR1_DBG_TIM4_STOP_Pos               (13U)
#define DBGMCU_CR1_DBG_TIM4_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM4_STOP_Pos)            /*!< 0x00002000 */
#define DBGMCU_CR1_DBG_TIM4_STOP                   DBGMCU_CR1_DBG_TIM4_STOP_Msk                       /*!< desc DBG_TIM4_STOP */
#define DBGMCU_CR1_DBG_CAN1_STOP_Pos               (14U)
#define DBGMCU_CR1_DBG_CAN1_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_CAN1_STOP_Pos)            /*!< 0x00004000 */
#define DBGMCU_CR1_DBG_CAN1_STOP                   DBGMCU_CR1_DBG_CAN1_STOP_Msk                       /*!< desc DBG_CAN_STOP */
#define DBGMCU_CR1_DBG_I2C1_SMBUS_TIMEOUT_Pos      (15U)
#define DBGMCU_CR1_DBG_I2C1_SMBUS_TIMEOUT_Msk      (0x1UL << DBGMCU_CR1_DBG_I2C1_SMBUS_TIMEOUT_Pos)   /*!< 0x00008000 */
#define DBGMCU_CR1_DBG_I2C1_SMBUS_TIMEOUT          DBGMCU_CR1_DBG_I2C1_SMBUS_TIMEOUT_Msk              /*!< desc DBG_I2C1_SMBUS_TIMEOUT */
#define DBGMCU_CR1_DBG_I2C2_SMBUS_TIMEOUT_Pos      (16U)
#define DBGMCU_CR1_DBG_I2C2_SMBUS_TIMEOUT_Msk      (0x1UL << DBGMCU_CR1_DBG_I2C2_SMBUS_TIMEOUT_Pos)   /*!< 0x00010000 */
#define DBGMCU_CR1_DBG_I2C2_SMBUS_TIMEOUT          DBGMCU_CR1_DBG_I2C2_SMBUS_TIMEOUT_Msk              /*!< desc DBG_I2C2_SMBUS_TIMEOUT */
#define DBGMCU_CR1_DBG_TIM8_STOP_Pos               (17U)
#define DBGMCU_CR1_DBG_TIM8_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM8_STOP_Pos)            /*!< 0x00020000 */
#define DBGMCU_CR1_DBG_TIM8_STOP                   DBGMCU_CR1_DBG_TIM8_STOP_Msk                       /*!< desc DBG_TIM8_STOP */
#define DBGMCU_CR1_DBG_TIM5_STOP_Pos               (18U)
#define DBGMCU_CR1_DBG_TIM5_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM5_STOP_Pos)            /*!< 0x00040000 */
#define DBGMCU_CR1_DBG_TIM5_STOP                   DBGMCU_CR1_DBG_TIM5_STOP_Msk                       /*!< desc DBG_TIM5_STOP */
#define DBGMCU_CR1_DBG_TIM6_STOP_Pos               (19U)
#define DBGMCU_CR1_DBG_TIM6_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM6_STOP_Pos)            /*!< 0x00080000 */
#define DBGMCU_CR1_DBG_TIM6_STOP                   DBGMCU_CR1_DBG_TIM6_STOP_Msk                       /*!< desc DBG_TIM6_STOP */
#define DBGMCU_CR1_DBG_TIM7_STOP_Pos               (20U)
#define DBGMCU_CR1_DBG_TIM7_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM7_STOP_Pos)            /*!< 0x00100000 */
#define DBGMCU_CR1_DBG_TIM7_STOP                   DBGMCU_CR1_DBG_TIM7_STOP_Msk                       /*!< desc DBG_TIM7_STOP */
#define DBGMCU_CR1_DBG_TIM12_STOP_Pos              (25U)
#define DBGMCU_CR1_DBG_TIM12_STOP_Msk              (0x1UL << DBGMCU_CR1_DBG_TIM12_STOP_Pos)           /*!< 0x02000000 */
#define DBGMCU_CR1_DBG_TIM12_STOP                  DBGMCU_CR1_DBG_TIM12_STOP_Msk                      /*!< desc DBG_TIM12_STOP */
#define DBGMCU_CR1_DBG_TIM13_STOP_Pos              (26U)
#define DBGMCU_CR1_DBG_TIM13_STOP_Msk              (0x1UL << DBGMCU_CR1_DBG_TIM13_STOP_Pos)           /*!< 0x04000000 */
#define DBGMCU_CR1_DBG_TIM13_STOP                  DBGMCU_CR1_DBG_TIM13_STOP_Msk                      /*!< desc DBG_TIM13_STOP */
#define DBGMCU_CR1_DBG_TIM14_STOP_Pos              (27U)
#define DBGMCU_CR1_DBG_TIM14_STOP_Msk              (0x1UL << DBGMCU_CR1_DBG_TIM14_STOP_Pos)           /*!< 0x08000000 */
#define DBGMCU_CR1_DBG_TIM14_STOP                  DBGMCU_CR1_DBG_TIM14_STOP_Msk                      /*!< desc DBG_TIM14_STOP */
#define DBGMCU_CR1_DBG_TIM9_STOP_Pos               (28U)
#define DBGMCU_CR1_DBG_TIM9_STOP_Msk               (0x1UL << DBGMCU_CR1_DBG_TIM9_STOP_Pos)            /*!< 0x10000000 */
#define DBGMCU_CR1_DBG_TIM9_STOP                   DBGMCU_CR1_DBG_TIM9_STOP_Msk                       /*!< desc DBG_TIM9_STOP */
#define DBGMCU_CR1_DBG_TIM10_STOP_Pos              (29U)
#define DBGMCU_CR1_DBG_TIM10_STOP_Msk              (0x1UL << DBGMCU_CR1_DBG_TIM10_STOP_Pos)           /*!< 0x20000000 */
#define DBGMCU_CR1_DBG_TIM10_STOP                  DBGMCU_CR1_DBG_TIM10_STOP_Msk                      /*!< desc DBG_TIM10_STOP */
#define DBGMCU_CR1_DBG_TIM11_STOP_Pos              (30U)
#define DBGMCU_CR1_DBG_TIM11_STOP_Msk              (0x1UL << DBGMCU_CR1_DBG_TIM11_STOP_Pos)           /*!< 0x40000000 */
#define DBGMCU_CR1_DBG_TIM11_STOP                  DBGMCU_CR1_DBG_TIM11_STOP_Msk                      /*!< desc DBG_TIM11_STOP */

/********************************* Bit definition for DBGMCU_CR2 register *******************************************/
#define DBGMCU_CR2_DBG_I2C3_SMBUS_TIMEOUT_Pos      (0U)
#define DBGMCU_CR2_DBG_I2C3_SMBUS_TIMEOUT_Msk      (0x1UL << DBGMCU_CR2_DBG_I2C3_SMBUS_TIMEOUT_Pos)   /*!< 0x00000001 */
#define DBGMCU_CR2_DBG_I2C3_SMBUS_TIMEOUT          DBGMCU_CR2_DBG_I2C3_SMBUS_TIMEOUT_Msk              /*!< desc DBG_I2C3_SMBUS_TIMEOUT */

#define DBGMCU_CR2_DBG_I2C4_SMBUS_TIMEOUT_Pos      (1U)
#define DBGMCU_CR2_DBG_I2C4_SMBUS_TIMEOUT_Msk      (0x1UL << DBGMCU_CR2_DBG_I2C4_SMBUS_TIMEOUT_Pos)   /*!< 0x00000002 */
#define DBGMCU_CR2_DBG_I2C4_SMBUS_TIMEOUT          DBGMCU_CR2_DBG_I2C4_SMBUS_TIMEOUT_Msk              /*!< desc DBG_I2C4_SMBUS_TIMEOUT */

#define DBGMCU_CR2_DBG_CAN2_STOP_Pos               (2U)
#define DBGMCU_CR2_DBG_CAN2_STOP_Msk               (0x1UL << DBGMCU_CR2_DBG_CAN2_STOP_Pos)            /*!< 0x00000004 */
#define DBGMCU_CR2_DBG_CAN2_STOP                   DBGMCU_CR2_DBG_CAN2_STOP_Msk                       /*!< desc DBG_CAN2_STOP */

#define DBGMCU_CR2_DBG_TIM15_STOP_Pos              (3U)
#define DBGMCU_CR2_DBG_TIM15_STOP_Msk              (0x1UL << DBGMCU_CR2_DBG_TIM15_STOP_Pos)           /*!< 0x00000008 */
#define DBGMCU_CR2_DBG_TIM15_STOP                  DBGMCU_CR2_DBG_TIM15_STOP_Msk                      /*!< desc DBG_TIM15_STOP */

#define DBGMCU_CR2_DBG_TIM16_STOP_Pos              (4U)
#define DBGMCU_CR2_DBG_TIM16_STOP_Msk              (0x1UL << DBGMCU_CR2_DBG_TIM16_STOP_Pos)           /*!< 0x00000010 */
#define DBGMCU_CR2_DBG_TIM16_STOP                  DBGMCU_CR2_DBG_TIM16_STOP_Msk                      /*!< desc DBG_TIM16_STOP */

#define DBGMCU_CR2_DBG_TIM17_STOP_Pos              (5U)
#define DBGMCU_CR2_DBG_TIM17_STOP_Msk              (0x1UL << DBGMCU_CR2_DBG_TIM17_STOP_Pos)           /*!< 0x00000020 */
#define DBGMCU_CR2_DBG_TIM17_STOP                  DBGMCU_CR2_DBG_TIM17_STOP_Msk                      /*!< desc DBG_TIM17_STOP */

#define DBGMCU_CR2_DBG_TIM18_STOP_Pos              (6U)
#define DBGMCU_CR2_DBG_TIM18_STOP_Msk              (0x1UL << DBGMCU_CR2_DBG_TIM18_STOP_Pos)           /*!< 0x00000040 */
#define DBGMCU_CR2_DBG_TIM18_STOP                  DBGMCU_CR2_DBG_TIM18_STOP_Msk                      /*!< desc DBG_TIM18_STOP */

#define DBGMCU_CR2_DBG_TIM19_STOP_Pos              (7U)
#define DBGMCU_CR2_DBG_TIM19_STOP_Msk              (0x1UL << DBGMCU_CR2_DBG_TIM19_STOP_Pos)           /*!< 0x00000080 */
#define DBGMCU_CR2_DBG_TIM19_STOP                  DBGMCU_CR2_DBG_TIM19_STOP_Msk                      /*!< desc DBG_TIM19_STOP */

#define DBGMCU_CR2_DBG_LPTIM_STOP_Pos              (8U)
#define DBGMCU_CR2_DBG_LPTIM_STOP_Msk              (0x1UL << DBGMCU_CR2_DBG_LPTIM_STOP_Pos)           /*!< 0x00000100 */
#define DBGMCU_CR2_DBG_LPTIM_STOP                  DBGMCU_CR2_DBG_LPTIM_STOP_Msk                      /*!< desc DBG_LPTIM_STOP */

/********************************************************************************************************************/
/********************************* DMA *************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for DMA_EN register ***********************************************/
#define DMA_EN_DMA_EN_Pos                         (0U)
#define DMA_EN_DMA_EN_Msk                         (0x1UL<<DMA_EN_DMA_EN_Pos)                        /*!< 0x00000001 */
#define DMA_EN_DMA_EN                             DMA_EN_DMA_EN_Msk                                 /*!< DMA Enable bit */

/********************************* Bit definition for DMA_CH_EN register ********************************************/
#define DMA_CH_EN_CH_EN_Pos                       (0U)
#define DMA_CH_EN_CH_EN_Msk                       (0x3FUL<<DMA_CH_EN_CH_EN_Pos)                     /*!< 0x0000003F */
#define DMA_CH_EN_CH_EN                           DMA_CH_EN_CH_EN_Msk                               /*!< Channel enable */
#define DMA_CH_EN_CH_EN_0                         (0x1UL<<DMA_CH_EN_CH_EN_Pos)                      /*!< 0x00000001 */
#define DMA_CH_EN_CH_EN_1                         (0x2UL<<DMA_CH_EN_CH_EN_Pos)                      /*!< 0x00000002 */
#define DMA_CH_EN_CH_EN_2                         (0x4UL<<DMA_CH_EN_CH_EN_Pos)                      /*!< 0x00000004 */
#define DMA_CH_EN_CH_EN_3                         (0x8UL<<DMA_CH_EN_CH_EN_Pos)                      /*!< 0x00000008 */
#define DMA_CH_EN_CH_EN_4                         (0x10UL<<DMA_CH_EN_CH_EN_Pos)                     /*!< 0x00000010 */
#define DMA_CH_EN_CH_EN_5                         (0x20UL<<DMA_CH_EN_CH_EN_Pos)                     /*!< 0x00000020 */
#define DMA_CH_EN_CH_EN_WE_Pos                    (8U)
#define DMA_CH_EN_CH_EN_WE_Msk                    (0x3FUL<<DMA_CH_EN_CH_EN_WE_Pos)                  /*!< 0x00003F00 */
#define DMA_CH_EN_CH_EN_WE                        DMA_CH_EN_CH_EN_WE_Msk                            /*!< Channel enable register */
#define DMA_CH_EN_CH_EN_WE_0                      (0x1UL<<DMA_CH_EN_CH_EN_WE_Pos)                   /*!< 0x00000100 */
#define DMA_CH_EN_CH_EN_WE_1                      (0x2UL<<DMA_CH_EN_CH_EN_WE_Pos)                   /*!< 0x00000200 */
#define DMA_CH_EN_CH_EN_WE_2                      (0x4UL<<DMA_CH_EN_CH_EN_WE_Pos)                   /*!< 0x00000400 */
#define DMA_CH_EN_CH_EN_WE_3                      (0x8UL<<DMA_CH_EN_CH_EN_WE_Pos)                   /*!< 0x00000800 */
#define DMA_CH_EN_CH_EN_WE_4                      (0x10UL<<DMA_CH_EN_CH_EN_WE_Pos)                  /*!< 0x00001000 */
#define DMA_CH_EN_CH_EN_WE_5                      (0x20UL<<DMA_CH_EN_CH_EN_WE_Pos)                  /*!< 0x00002000 */

/********************************* Bit definition for DMA_CH_CTRLL register ****************************************/
#define DMA_CH_CTRLL_INT_EN_Pos                  (0U)
#define DMA_CH_CTRLL_INT_EN_Msk                  (0x1UL<<DMA_CH_CTRLL_INT_EN_Pos)                 /*!< 0x00000001 */
#define DMA_CH_CTRLL_INT_EN                      DMA_CH_CTRLL_INT_EN_Msk                          /*!< Interrupt Enable Bit */
#define DMA_CH_CTRLL_DST_TR_WIDTH_Pos            (1U)
#define DMA_CH_CTRLL_DST_TR_WIDTH_Msk            (0x7UL<<DMA_CH_CTRLL_DST_TR_WIDTH_Pos)           /*!< 0x0000000E */
#define DMA_CH_CTRLL_DST_TR_WIDTH                DMA_CH_CTRLL_DST_TR_WIDTH_Msk                    /*!< Destination Transfer Width */
#define DMA_CH_CTRLL_DST_TR_WIDTH_0              (0x1UL<<DMA_CH_CTRLL_DST_TR_WIDTH_Pos)           /*!< 0x00000002 */
#define DMA_CH_CTRLL_DST_TR_WIDTH_1              (0x2UL<<DMA_CH_CTRLL_DST_TR_WIDTH_Pos)           /*!< 0x00000004 */
#define DMA_CH_CTRLL_DST_TR_WIDTH_2              (0x4UL<<DMA_CH_CTRLL_DST_TR_WIDTH_Pos)           /*!< 0x00000008 */
#define DMA_CH_CTRLL_SRC_TR_WIDTH_Pos            (4U)
#define DMA_CH_CTRLL_SRC_TR_WIDTH_Msk            (0x7UL<<DMA_CH_CTRLL_SRC_TR_WIDTH_Pos)           /*!< 0x00000070 */
#define DMA_CH_CTRLL_SRC_TR_WIDTH                DMA_CH_CTRLL_SRC_TR_WIDTH_Msk                    /*!< Source Transfer Width */
#define DMA_CH_CTRLL_SRC_TR_WIDTH_0              (0x1UL<<DMA_CH_CTRLL_SRC_TR_WIDTH_Pos)           /*!< 0x00000010 */
#define DMA_CH_CTRLL_SRC_TR_WIDTH_1              (0x2UL<<DMA_CH_CTRLL_SRC_TR_WIDTH_Pos)           /*!< 0x00000020 */
#define DMA_CH_CTRLL_SRC_TR_WIDTH_2              (0x4UL<<DMA_CH_CTRLL_SRC_TR_WIDTH_Pos)           /*!< 0x00000040 */
#define DMA_CH_CTRLL_DINC_Pos                    (7U)
#define DMA_CH_CTRLL_DINC_Msk                    (0x3UL<<DMA_CH_CTRLL_DINC_Pos)                   /*!< 0x00000180 */
#define DMA_CH_CTRLL_DINC                        DMA_CH_CTRLL_DINC_Msk                            /*!< Destination Address Increment */
#define DMA_CH_CTRLL_DINC_0                      (0x1UL<<DMA_CH_CTRLL_DINC_Pos)                   /*!< 0x00000080 */
#define DMA_CH_CTRLL_DINC_1                      (0x2UL<<DMA_CH_CTRLL_DINC_Pos)                   /*!< 0x00000100 */
#define DMA_CH_CTRLL_SINC_Pos                    (9U)
#define DMA_CH_CTRLL_SINC_Msk                    (0x3UL<<DMA_CH_CTRLL_SINC_Pos)                   /*!< 0x00000600 */
#define DMA_CH_CTRLL_SINC                        DMA_CH_CTRLL_SINC_Msk                            /*!< Source Address Increment */
#define DMA_CH_CTRLL_SINC_0                      (0x1UL<<DMA_CH_CTRLL_SINC_Pos)                   /*!< 0x00000200 */
#define DMA_CH_CTRLL_SINC_1                      (0x2UL<<DMA_CH_CTRLL_SINC_Pos)                   /*!< 0x00000400 */
#define DMA_CH_CTRLL_DST_MSIZE_Pos               (11U)
#define DMA_CH_CTRLL_DST_MSIZE_Msk               (0x7UL<<DMA_CH_CTRLL_DST_MSIZE_Pos)              /*!< 0x00003800 */
#define DMA_CH_CTRLL_DST_MSIZE                   DMA_CH_CTRLL_DST_MSIZE_Msk                       /*!< Destination  Burst Transaction Length */
#define DMA_CH_CTRLL_DST_MSIZE_0                 (0x1UL<<DMA_CH_CTRLL_DST_MSIZE_Pos)              /*!< 0x00000800 */
#define DMA_CH_CTRLL_DST_MSIZE_1                 (0x2UL<<DMA_CH_CTRLL_DST_MSIZE_Pos)              /*!< 0x00001000 */
#define DMA_CH_CTRLL_DST_MSIZE_2                 (0x4UL<<DMA_CH_CTRLL_DST_MSIZE_Pos)              /*!< 0x00002000 */
#define DMA_CH_CTRLL_SRC_MSIZE_Pos               (14U)
#define DMA_CH_CTRLL_SRC_MSIZE_Msk               (0x7UL<<DMA_CH_CTRLL_SRC_MSIZE_Pos)              /*!< 0x0001C000 */
#define DMA_CH_CTRLL_SRC_MSIZE                   DMA_CH_CTRLL_SRC_MSIZE_Msk                       /*!< Source Burst Transaction Length */
#define DMA_CH_CTRLL_SRC_MSIZE_0                 (0x1UL<<DMA_CH_CTRLL_SRC_MSIZE_Pos)              /*!< 0x00004000 */
#define DMA_CH_CTRLL_SRC_MSIZE_1                 (0x2UL<<DMA_CH_CTRLL_SRC_MSIZE_Pos)              /*!< 0x00008000 */
#define DMA_CH_CTRLL_SRC_MSIZE_2                 (0x4UL<<DMA_CH_CTRLL_SRC_MSIZE_Pos)              /*!< 0x00010000 */
#define DMA_CH_CTRLL_TT_Pos                      (20U)
#define DMA_CH_CTRLL_TT_Msk                      (0x3UL<<DMA_CH_CTRLL_TT_Pos)                     /*!< 0x00300000 */
#define DMA_CH_CTRLL_TT                          DMA_CH_CTRLL_TT_Msk                              /*!< Transfer Type */
#define DMA_CH_CTRLL_TT_0                        (0x1UL<<DMA_CH_CTRLL_TT_Pos)                     /*!< 0x00100000 */
#define DMA_CH_CTRLL_TT_1                        (0x2UL<<DMA_CH_CTRLL_TT_Pos)                     /*!< 0x00200000 */

/********************************* Bit definition for DMA_CH_CTRLH register ****************************************/
#define DMA_CH_CTRLH_BLOCK_TS_Pos                (0U)
#define DMA_CH_CTRLH_BLOCK_TS_Msk                (0xFFFUL<<DMA_CH_CTRLH_BLOCK_TS_Pos)             /*!< 0x00000FFF */
#define DMA_CH_CTRLH_BLOCK_TS                    DMA_CH_CTRLH_BLOCK_TS_Msk                        /*!< Block Transfer Size. */

/********************************* Bit definition for DMA_CH_CFGL register *****************************************/
#define DMA_CH_CFGL_INT_SUSP_Pos                 (4U)
#define DMA_CH_CFGL_INT_SUSP_Msk                 (0x1UL<<DMA_CH_CFGL_INT_SUSP_Pos)                /*!< 0x00000010 */
#define DMA_CH_CFGL_INT_SUSP                     DMA_CH_CFGL_INT_SUSP_Msk                         /*!< Interrupt suspend */
#define DMA_CH_CFGL_CH_PRIOR_Pos                 (5U)
#define DMA_CH_CFGL_CH_PRIOR_Msk                 (0x7UL<<DMA_CH_CFGL_CH_PRIOR_Pos)                /*!< 0x000000E0 */
#define DMA_CH_CFGL_CH_PRIOR                     DMA_CH_CFGL_CH_PRIOR_Msk                         /*!< Channel Priority */
#define DMA_CH_CFGL_CH_PRIOR_0                   (0x1UL<<DMA_CH_CFGL_CH_PRIOR_Pos)                /*!< 0x00000020 */
#define DMA_CH_CFGL_CH_PRIOR_1                   (0x2UL<<DMA_CH_CFGL_CH_PRIOR_Pos)                /*!< 0x00000040 */
#define DMA_CH_CFGL_CH_PRIOR_2                   (0x4UL<<DMA_CH_CFGL_CH_PRIOR_Pos)                /*!< 0x00000080 */
#define DMA_CH_CFGL_CH_SUSP_Pos                  (8U)
#define DMA_CH_CFGL_CH_SUSP_Msk                  (0x1UL<<DMA_CH_CFGL_CH_SUSP_Pos)                 /*!< 0x00000100 */
#define DMA_CH_CFGL_CH_SUSP                      DMA_CH_CFGL_CH_SUSP_Msk                          /*!< Channel Suspend */
#define DMA_CH_CFGL_FIFO_EMPTY_Pos               (9U)
#define DMA_CH_CFGL_FIFO_EMPTY_Msk               (0x1UL<<DMA_CH_CFGL_FIFO_EMPTY_Pos)              /*!< 0x00000200 */
#define DMA_CH_CFGL_FIFO_EMPTY                   DMA_CH_CFGL_FIFO_EMPTY_Msk                       /*!< Channel FIFO status */
#define DMA_CH_CFGL_HS_SEL_DST_Pos               (10U)
#define DMA_CH_CFGL_HS_SEL_DST_Msk               (0x1UL<<DMA_CH_CFGL_HS_SEL_DST_Pos)              /*!< 0x00000400 */
#define DMA_CH_CFGL_HS_SEL_DST                   DMA_CH_CFGL_HS_SEL_DST_Msk                       /*!< Destination Software or Hardware Handshaking Select  */
#define DMA_CH_CFGL_HS_SEL_SRC_Pos               (11U)
#define DMA_CH_CFGL_HS_SEL_SRC_Msk               (0x1UL<<DMA_CH_CFGL_HS_SEL_SRC_Pos)              /*!< 0x00000800 */
#define DMA_CH_CFGL_HS_SEL_SRC                   DMA_CH_CFGL_HS_SEL_SRC_Msk                       /*!< Source Software or Hardware Handshaking Select  */
#define DMA_CH_CFGL_MAX_ABRST_Pos                (20U)
#define DMA_CH_CFGL_MAX_ABRST_Msk                (0x3FFUL<<DMA_CH_CFGL_MAX_ABRST_Pos)             /*!< 0x3FF00000 */
#define DMA_CH_CFGL_MAX_ABRST                    DMA_CH_CFGL_MAX_ABRST_Msk                        /*!< Maximum AMBA Burst Length.  */
#define DMA_CH_CFGL_RELOAD_SRC_Pos               (30U)
#define DMA_CH_CFGL_RELOAD_SRC_Msk               (0x1UL<<DMA_CH_CFGL_RELOAD_SRC_Pos)              /*!< 0x40000000 */
#define DMA_CH_CFGL_RELOAD_SRC                   DMA_CH_CFGL_RELOAD_SRC_Msk                       /*!< Automatic Source Reload */
#define DMA_CH_CFGL_RELOAD_DST_Pos               (31U)
#define DMA_CH_CFGL_RELOAD_DST_Msk               (0x1UL<<DMA_CH_CFGL_RELOAD_DST_Pos)              /*!< 0x80000000 */
#define DMA_CH_CFGL_RELOAD_DST                   DMA_CH_CFGL_RELOAD_DST_Msk                       /*!< Automatic Destination Reload */

/********************************* Bit definition for DMA_CH_CFGH register *****************************************/
#define DMA_CH_CFGH_FIFO_MODE_Pos                (1U)
#define DMA_CH_CFGH_FIFO_MODE_Msk                (0x1UL<<DMA_CH_CFGH_FIFO_MODE_Pos)               /*!< 0x00000002 */
#define DMA_CH_CFGH_FIFO_MODE                    DMA_CH_CFGH_FIFO_MODE_Msk                        /*!< FIFO Mode Select */
#define DMA_CH_CFGH_SRC_PER_Pos                  (7U)
#define DMA_CH_CFGH_SRC_PER_Msk                  (0x7UL<<DMA_CH_CFGH_SRC_PER_Pos)                 /*!< 0x00000380 */
#define DMA_CH_CFGH_SRC_PER                      DMA_CH_CFGH_SRC_PER_Msk                          /*!< Source hardware interface */
#define DMA_CH_CFGH_SRC_PER_0                    (0x1UL<<DMA_CH_CFGH_SRC_PER_Pos)                 /*!< 0x00000080 */
#define DMA_CH_CFGH_SRC_PER_1                    (0x2UL<<DMA_CH_CFGH_SRC_PER_Pos)                 /*!< 0x00000100 */
#define DMA_CH_CFGH_SRC_PER_2                    (0x4UL<<DMA_CH_CFGH_SRC_PER_Pos)                 /*!< 0x00000200 */
#define DMA_CH_CFGH_DST_PER_Pos                  (11U)
#define DMA_CH_CFGH_DST_PER_Msk                  (0x7UL<<DMA_CH_CFGH_DST_PER_Pos)                 /*!< 0x00003800 */
#define DMA_CH_CFGH_DST_PER                      DMA_CH_CFGH_DST_PER_Msk                          /*!< Destination hardware interface */
#define DMA_CH_CFGH_DST_PER_0                    (0x1UL<<DMA_CH_CFGH_DST_PER_Pos)                 /*!< 0x00000800 */
#define DMA_CH_CFGH_DST_PER_1                    (0x2UL<<DMA_CH_CFGH_DST_PER_Pos)                 /*!< 0x00001000 */
#define DMA_CH_CFGH_DST_PER_2                    (0x4UL<<DMA_CH_CFGH_DST_PER_Pos)                 /*!< 0x00002000 */

/********************************* Bit definition for DMA_SA register **********************************************/
#define DMA_SA_SA_Pos                            (0U)
#define DMA_SA_SA_Msk                            (0xFFFFFFFFUL<<DMA_SA_SA_Pos)                    /*!< 0xFFFFFFFF */
#define DMA_SA_SA                                DMA_SA_SA_Msk                                    /*!< Source Address for Channel 0 */

/********************************* Bit definition for DMA_DA register **********************************************/
#define DMA_DA_DA_Pos                            (0U)
#define DMA_DA_DA_Msk                            (0xFFFFFFFFUL<<DMA_DA_DA_Pos)                    /*!< 0xFFFFFFFF */
#define DMA_DA_DA                                DMA_DA_DA_Msk                                    /*!< Destination Address Register for Channel0 */

/********************************* Bit definition for DMA_STATUSHALFBLOCK register **********************************/
#define DMA_STATUSHALFBLOCK_STATUS_Pos            (0U)
#define DMA_STATUSHALFBLOCK_STATUS_Msk            (0x3FUL<<DMA_STATUSHALFBLOCK_STATUS_Pos)          /*!< 0x0000003F */
#define DMA_STATUSHALFBLOCK_STATUS                DMA_STATUSHALFBLOCK_STATUS_Msk                    /*!< Status for IntHalfBlock Interrupt */
#define DMA_STATUSHALFBLOCK_STATUS_0              (0x1UL<<DMA_STATUSHALFBLOCK_STATUS_Pos)           /*!< 0x00000001 */
#define DMA_STATUSHALFBLOCK_STATUS_1              (0x2UL<<DMA_STATUSHALFBLOCK_STATUS_Pos)           /*!< 0x00000002 */
#define DMA_STATUSHALFBLOCK_STATUS_2              (0x4UL<<DMA_STATUSHALFBLOCK_STATUS_Pos)           /*!< 0x00000004 */
#define DMA_STATUSHALFBLOCK_STATUS_3              (0x8UL<<DMA_STATUSHALFBLOCK_STATUS_Pos)           /*!< 0x00000008 */
#define DMA_STATUSHALFBLOCK_STATUS_4              (0x10UL<<DMA_STATUSHALFBLOCK_STATUS_Pos)          /*!< 0x00000010 */
#define DMA_STATUSHALFBLOCK_STATUS_5              (0x20UL<<DMA_STATUSHALFBLOCK_STATUS_Pos)          /*!< 0x00000020 */

/********************************* Bit definition for DMA_MASKHALFBLOCK register ************************************/
#define DMA_MASKHALFBLOCK_INT_MASK_Pos            (0U)
#define DMA_MASKHALFBLOCK_INT_MASK_Msk            (0x3FUL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)          /*!< 0x0000003F */
#define DMA_MASKHALFBLOCK_INT_MASK                DMA_MASKHALFBLOCK_INT_MASK_Msk                    /*!< Mask for IntHalfBlock Interrupt */
#define DMA_MASKHALFBLOCK_INT_MASK_0              (0x1UL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)           /*!< 0x00000001 */
#define DMA_MASKHALFBLOCK_INT_MASK_1              (0x2UL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)           /*!< 0x00000002 */
#define DMA_MASKHALFBLOCK_INT_MASK_2              (0x4UL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)           /*!< 0x00000004 */
#define DMA_MASKHALFBLOCK_INT_MASK_3              (0x8UL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)           /*!< 0x00000008 */
#define DMA_MASKHALFBLOCK_INT_MASK_4              (0x10UL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)          /*!< 0x00000010 */
#define DMA_MASKHALFBLOCK_INT_MASK_5              (0x20UL<<DMA_MASKHALFBLOCK_INT_MASK_Pos)          /*!< 0x00000020 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_Pos         (8U)
#define DMA_MASKHALFBLOCK_INT_MASK_WE_Msk         (0x3FUL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)       /*!< 0x00003F00 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE             DMA_MASKHALFBLOCK_INT_MASK_WE_Msk                 /*!< Interrupt Mask Write Enable */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_0           (0x1UL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)        /*!< 0x00000100 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_1           (0x2UL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)        /*!< 0x00000200 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_2           (0x4UL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)        /*!< 0x00000400 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_3           (0x8UL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)        /*!< 0x00000800 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_4           (0x10UL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)       /*!< 0x00001000 */
#define DMA_MASKHALFBLOCK_INT_MASK_WE_5           (0x20UL<<DMA_MASKHALFBLOCK_INT_MASK_WE_Pos)       /*!< 0x00002000 */

/********************************* Bit definition for DMA_CLEARHALFBLOCK register ***********************************/
#define DMA_CLEARHALFBLOCK_CLEAR_Pos              (0U)
#define DMA_CLEARHALFBLOCK_CLEAR_Msk              (0x3FUL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)            /*!< 0x0000003F */
#define DMA_CLEARHALFBLOCK_CLEAR                  DMA_CLEARHALFBLOCK_CLEAR_Msk                      /*!< Clear for IntHalfBlock Interrupt */
#define DMA_CLEARHALFBLOCK_CLEAR_0                (0x1UL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)             /*!< 0x00000001 */
#define DMA_CLEARHALFBLOCK_CLEAR_1                (0x2UL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)             /*!< 0x00000002 */
#define DMA_CLEARHALFBLOCK_CLEAR_2                (0x4UL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)             /*!< 0x00000004 */
#define DMA_CLEARHALFBLOCK_CLEAR_3                (0x8UL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)             /*!< 0x00000008 */
#define DMA_CLEARHALFBLOCK_CLEAR_4                (0x10UL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)            /*!< 0x00000010 */
#define DMA_CLEARHALFBLOCK_CLEAR_5                (0x20UL<<DMA_CLEARHALFBLOCK_CLEAR_Pos)            /*!< 0x00000020 */

/********************************* Bit definition for DMA_STATUSTFR register ****************************************/
#define DMA_STATUSTFR_STATUS_Pos                  (0U)
#define DMA_STATUSTFR_STATUS_Msk                  (0x3FUL<<DMA_STATUSTFR_STATUS_Pos)                /*!< 0x0000003F */
#define DMA_STATUSTFR_STATUS                      DMA_STATUSTFR_STATUS_Msk                          /*!< Status for IntTfr Interrupt */
#define DMA_STATUSTFR_STATUS_0                    (0x1UL<<DMA_STATUSTFR_STATUS_Pos)                 /*!< 0x00000001 */
#define DMA_STATUSTFR_STATUS_1                    (0x2UL<<DMA_STATUSTFR_STATUS_Pos)                 /*!< 0x00000002 */
#define DMA_STATUSTFR_STATUS_2                    (0x4UL<<DMA_STATUSTFR_STATUS_Pos)                 /*!< 0x00000004 */
#define DMA_STATUSTFR_STATUS_3                    (0x8UL<<DMA_STATUSTFR_STATUS_Pos)                 /*!< 0x00000008 */
#define DMA_STATUSTFR_STATUS_4                    (0x10UL<<DMA_STATUSTFR_STATUS_Pos)                /*!< 0x00000010 */
#define DMA_STATUSTFR_STATUS_5                    (0x20UL<<DMA_STATUSTFR_STATUS_Pos)                /*!< 0x00000020 */

/********************************* Bit definition for DMA_STATUSBLOCK register **************************************/
#define DMA_STATUSBLOCK_STATUS_Pos                (0U)
#define DMA_STATUSBLOCK_STATUS_Msk                (0x3FUL<<DMA_STATUSBLOCK_STATUS_Pos)              /*!< 0x0000003F */
#define DMA_STATUSBLOCK_STATUS                    DMA_STATUSBLOCK_STATUS_Msk                        /*!< Status for IntBlock Interrupt */
#define DMA_STATUSBLOCK_STATUS_0                  (0x1UL<<DMA_STATUSBLOCK_STATUS_Pos)               /*!< 0x00000001 */
#define DMA_STATUSBLOCK_STATUS_1                  (0x2UL<<DMA_STATUSBLOCK_STATUS_Pos)               /*!< 0x00000002 */
#define DMA_STATUSBLOCK_STATUS_2                  (0x4UL<<DMA_STATUSBLOCK_STATUS_Pos)               /*!< 0x00000004 */
#define DMA_STATUSBLOCK_STATUS_3                  (0x8UL<<DMA_STATUSBLOCK_STATUS_Pos)               /*!< 0x00000008 */
#define DMA_STATUSBLOCK_STATUS_4                  (0x10UL<<DMA_STATUSBLOCK_STATUS_Pos)              /*!< 0x00000010 */
#define DMA_STATUSBLOCK_STATUS_5                  (0x20UL<<DMA_STATUSBLOCK_STATUS_Pos)              /*!< 0x00000020 */

/********************************* Bit definition for DMA_STATUSSRCTRAN register ************************************/
#define DMA_STATUSSRCTRAN_STATUS_Pos              (0U)
#define DMA_STATUSSRCTRAN_STATUS_Msk              (0x3FUL<<DMA_STATUSSRCTRAN_STATUS_Pos)            /*!< 0x0000003F */
#define DMA_STATUSSRCTRAN_STATUS                  DMA_STATUSSRCTRAN_STATUS_Msk                      /*!< Status for IntSrcTran Interrupt */
#define DMA_STATUSSRCTRAN_STATUS_0                (0x1UL<<DMA_STATUSSRCTRAN_STATUS_Pos)             /*!< 0x00000001 */
#define DMA_STATUSSRCTRAN_STATUS_1                (0x2UL<<DMA_STATUSSRCTRAN_STATUS_Pos)             /*!< 0x00000002 */
#define DMA_STATUSSRCTRAN_STATUS_2                (0x4UL<<DMA_STATUSSRCTRAN_STATUS_Pos)             /*!< 0x00000004 */
#define DMA_STATUSSRCTRAN_STATUS_3                (0x8UL<<DMA_STATUSSRCTRAN_STATUS_Pos)             /*!< 0x00000008 */
#define DMA_STATUSSRCTRAN_STATUS_4                (0x10UL<<DMA_STATUSSRCTRAN_STATUS_Pos)            /*!< 0x00000010 */
#define DMA_STATUSSRCTRAN_STATUS_5                (0x20UL<<DMA_STATUSSRCTRAN_STATUS_Pos)            /*!< 0x00000020 */

/********************************* Bit definition for DMA_STATUSDSTTRAN register ************************************/
#define DMA_STATUSDSTTRAN_STATUS_Pos              (0U)
#define DMA_STATUSDSTTRAN_STATUS_Msk              (0x3FUL<<DMA_STATUSDSTTRAN_STATUS_Pos)            /*!< 0x0000003F */
#define DMA_STATUSDSTTRAN_STATUS                  DMA_STATUSDSTTRAN_STATUS_Msk                      /*!< Status for IntDstTran Interrupt */
#define DMA_STATUSDSTTRAN_STATUS_0                (0x1UL<<DMA_STATUSDSTTRAN_STATUS_Pos)             /*!< 0x00000001 */
#define DMA_STATUSDSTTRAN_STATUS_1                (0x2UL<<DMA_STATUSDSTTRAN_STATUS_Pos)             /*!< 0x00000002 */
#define DMA_STATUSDSTTRAN_STATUS_2                (0x4UL<<DMA_STATUSDSTTRAN_STATUS_Pos)             /*!< 0x00000004 */
#define DMA_STATUSDSTTRAN_STATUS_3                (0x8UL<<DMA_STATUSDSTTRAN_STATUS_Pos)             /*!< 0x00000008 */
#define DMA_STATUSDSTTRAN_STATUS_4                (0x10UL<<DMA_STATUSDSTTRAN_STATUS_Pos)            /*!< 0x00000010 */
#define DMA_STATUSDSTTRAN_STATUS_5                (0x20UL<<DMA_STATUSDSTTRAN_STATUS_Pos)            /*!< 0x00000020 */

/********************************* Bit definition for DMA_STATUSERR register ****************************************/
#define DMA_STATUSERR_STATUS_Pos                  (0U)
#define DMA_STATUSERR_STATUS_Msk                  (0x3FUL<<DMA_STATUSERR_STATUS_Pos)                /*!< 0x0000003F */
#define DMA_STATUSERR_STATUS                      DMA_STATUSERR_STATUS_Msk                          /*!< Status for IntErr Interrupt */
#define DMA_STATUSERR_STATUS_0                    (0x1UL<<DMA_STATUSERR_STATUS_Pos)                 /*!< 0x00000001 */
#define DMA_STATUSERR_STATUS_1                    (0x2UL<<DMA_STATUSERR_STATUS_Pos)                 /*!< 0x00000002 */
#define DMA_STATUSERR_STATUS_2                    (0x4UL<<DMA_STATUSERR_STATUS_Pos)                 /*!< 0x00000004 */
#define DMA_STATUSERR_STATUS_3                    (0x8UL<<DMA_STATUSERR_STATUS_Pos)                 /*!< 0x00000008 */
#define DMA_STATUSERR_STATUS_4                    (0x10UL<<DMA_STATUSERR_STATUS_Pos)                /*!< 0x00000010 */
#define DMA_STATUSERR_STATUS_5                    (0x20UL<<DMA_STATUSERR_STATUS_Pos)                /*!< 0x00000020 */

/********************************* Bit definition for DMA_MASKTFR register ******************************************/
#define DMA_MASKTFR_INT_MASK_Pos                  (0U)
#define DMA_MASKTFR_INT_MASK_Msk                  (0x3FUL<<DMA_MASKTFR_INT_MASK_Pos)                /*!< 0x0000003F */
#define DMA_MASKTFR_INT_MASK                      DMA_MASKTFR_INT_MASK_Msk                          /*!< Mask for IntTfr Interrupt */
#define DMA_MASKTFR_INT_MASK_0                    (0x1UL<<DMA_MASKTFR_INT_MASK_Pos)                 /*!< 0x00000001 */
#define DMA_MASKTFR_INT_MASK_1                    (0x2UL<<DMA_MASKTFR_INT_MASK_Pos)                 /*!< 0x00000002 */
#define DMA_MASKTFR_INT_MASK_2                    (0x4UL<<DMA_MASKTFR_INT_MASK_Pos)                 /*!< 0x00000004 */
#define DMA_MASKTFR_INT_MASK_3                    (0x8UL<<DMA_MASKTFR_INT_MASK_Pos)                 /*!< 0x00000008 */
#define DMA_MASKTFR_INT_MASK_4                    (0x10UL<<DMA_MASKTFR_INT_MASK_Pos)                /*!< 0x00000010 */
#define DMA_MASKTFR_INT_MASK_5                    (0x20UL<<DMA_MASKTFR_INT_MASK_Pos)                /*!< 0x00000020 */
#define DMA_MASKTFR_INT_MASK_WE_Pos               (8U)
#define DMA_MASKTFR_INT_MASK_WE_Msk               (0x3FUL<<DMA_MASKTFR_INT_MASK_WE_Pos)             /*!< 0x00003F00 */
#define DMA_MASKTFR_INT_MASK_WE                   DMA_MASKTFR_INT_MASK_WE_Msk                       /*!< Interrupt Mask Write Enable */
#define DMA_MASKTFR_INT_MASK_WE_0                 (0x1UL<<DMA_MASKTFR_INT_MASK_WE_Pos)              /*!< 0x00000100 */
#define DMA_MASKTFR_INT_MASK_WE_1                 (0x2UL<<DMA_MASKTFR_INT_MASK_WE_Pos)              /*!< 0x00000200 */
#define DMA_MASKTFR_INT_MASK_WE_2                 (0x4UL<<DMA_MASKTFR_INT_MASK_WE_Pos)              /*!< 0x00000400 */
#define DMA_MASKTFR_INT_MASK_WE_3                 (0x8UL<<DMA_MASKTFR_INT_MASK_WE_Pos)              /*!< 0x00000800 */
#define DMA_MASKTFR_INT_MASK_WE_4                 (0x10UL<<DMA_MASKTFR_INT_MASK_WE_Pos)             /*!< 0x00001000 */
#define DMA_MASKTFR_INT_MASK_WE_5                 (0x20UL<<DMA_MASKTFR_INT_MASK_WE_Pos)             /*!< 0x00002000 */

/********************************* Bit definition for DMA_MASKBLOCK register ****************************************/
#define DMA_MASKBLOCK_INT_MASK_Pos                (0U)
#define DMA_MASKBLOCK_INT_MASK_Msk                (0x3FUL<<DMA_MASKBLOCK_INT_MASK_Pos)              /*!< 0x0000003F */
#define DMA_MASKBLOCK_INT_MASK                    DMA_MASKBLOCK_INT_MASK_Msk                        /*!< Mask for IntBlock Interrupt */
#define DMA_MASKBLOCK_INT_MASK_0                  (0x1UL<<DMA_MASKBLOCK_INT_MASK_Pos)               /*!< 0x00000001 */
#define DMA_MASKBLOCK_INT_MASK_1                  (0x2UL<<DMA_MASKBLOCK_INT_MASK_Pos)               /*!< 0x00000002 */
#define DMA_MASKBLOCK_INT_MASK_2                  (0x4UL<<DMA_MASKBLOCK_INT_MASK_Pos)               /*!< 0x00000004 */
#define DMA_MASKBLOCK_INT_MASK_3                  (0x8UL<<DMA_MASKBLOCK_INT_MASK_Pos)               /*!< 0x00000008 */
#define DMA_MASKBLOCK_INT_MASK_4                  (0x10UL<<DMA_MASKBLOCK_INT_MASK_Pos)              /*!< 0x00000010 */
#define DMA_MASKBLOCK_INT_MASK_5                  (0x20UL<<DMA_MASKBLOCK_INT_MASK_Pos)              /*!< 0x00000020 */
#define DMA_MASKBLOCK_INT_MASK_WE_Pos             (8U)
#define DMA_MASKBLOCK_INT_MASK_WE_Msk             (0x3FUL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)           /*!< 0x00003F00 */
#define DMA_MASKBLOCK_INT_MASK_WE                 DMA_MASKBLOCK_INT_MASK_WE_Msk                     /*!< Interrupt Mask Write Enable */
#define DMA_MASKBLOCK_INT_MASK_WE_0               (0x1UL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)            /*!< 0x00000100 */
#define DMA_MASKBLOCK_INT_MASK_WE_1               (0x2UL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)            /*!< 0x00000200 */
#define DMA_MASKBLOCK_INT_MASK_WE_2               (0x4UL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)            /*!< 0x00000400 */
#define DMA_MASKBLOCK_INT_MASK_WE_3               (0x8UL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)            /*!< 0x00000800 */
#define DMA_MASKBLOCK_INT_MASK_WE_4               (0x10UL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)           /*!< 0x00001000 */
#define DMA_MASKBLOCK_INT_MASK_WE_5               (0x20UL<<DMA_MASKBLOCK_INT_MASK_WE_Pos)           /*!< 0x00002000 */

/********************************* Bit definition for DMA_MASKSRCTRAN register **************************************/
#define DMA_MASKSRCTRAN_INT_MASK_Pos              (0U)
#define DMA_MASKSRCTRAN_INT_MASK_Msk              (0x3FUL<<DMA_MASKSRCTRAN_INT_MASK_Pos)            /*!< 0x0000003F */
#define DMA_MASKSRCTRAN_INT_MASK                  DMA_MASKSRCTRAN_INT_MASK_Msk                      /*!< Mask for IntSrcTran Interrupt */
#define DMA_MASKSRCTRAN_INT_MASK_0                (0x1UL<<DMA_MASKSRCTRAN_INT_MASK_Pos)             /*!< 0x00000001 */
#define DMA_MASKSRCTRAN_INT_MASK_1                (0x2UL<<DMA_MASKSRCTRAN_INT_MASK_Pos)             /*!< 0x00000002 */
#define DMA_MASKSRCTRAN_INT_MASK_2                (0x4UL<<DMA_MASKSRCTRAN_INT_MASK_Pos)             /*!< 0x00000004 */
#define DMA_MASKSRCTRAN_INT_MASK_3                (0x8UL<<DMA_MASKSRCTRAN_INT_MASK_Pos)             /*!< 0x00000008 */
#define DMA_MASKSRCTRAN_INT_MASK_4                (0x10UL<<DMA_MASKSRCTRAN_INT_MASK_Pos)            /*!< 0x00000010 */
#define DMA_MASKSRCTRAN_INT_MASK_5                (0x20UL<<DMA_MASKSRCTRAN_INT_MASK_Pos)            /*!< 0x00000020 */
#define DMA_MASKSRCTRAN_INT_MASK_WE_Pos           (8U)
#define DMA_MASKSRCTRAN_INT_MASK_WE_Msk           (0x3FUL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)         /*!< 0x00003F00 */
#define DMA_MASKSRCTRAN_INT_MASK_WE               DMA_MASKSRCTRAN_INT_MASK_WE_Msk                   /*!< Interrupt Mask Write Enable */
#define DMA_MASKSRCTRAN_INT_MASK_WE_0             (0x1UL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)          /*!< 0x00000100 */
#define DMA_MASKSRCTRAN_INT_MASK_WE_1             (0x2UL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)          /*!< 0x00000200 */
#define DMA_MASKSRCTRAN_INT_MASK_WE_2             (0x4UL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)          /*!< 0x00000400 */
#define DMA_MASKSRCTRAN_INT_MASK_WE_3             (0x8UL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)          /*!< 0x00000800 */
#define DMA_MASKSRCTRAN_INT_MASK_WE_4             (0x10UL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)         /*!< 0x00001000 */
#define DMA_MASKSRCTRAN_INT_MASK_WE_5             (0x20UL<<DMA_MASKSRCTRAN_INT_MASK_WE_Pos)         /*!< 0x00002000 */

/********************************* Bit definition for DMA_MASKDSTTRAN register **************************************/
#define DMA_MASKDSTTRAN_INT_MASK_Pos              (0U)
#define DMA_MASKDSTTRAN_INT_MASK_Msk              (0x3FUL<<DMA_MASKDSTTRAN_INT_MASK_Pos)            /*!< 0x0000003F */
#define DMA_MASKDSTTRAN_INT_MASK                  DMA_MASKDSTTRAN_INT_MASK_Msk                      /*!< Mask for IntDstTran Interrupt */
#define DMA_MASKDSTTRAN_INT_MASK_0                (0x1UL<<DMA_MASKDSTTRAN_INT_MASK_Pos)             /*!< 0x00000001 */
#define DMA_MASKDSTTRAN_INT_MASK_1                (0x2UL<<DMA_MASKDSTTRAN_INT_MASK_Pos)             /*!< 0x00000002 */
#define DMA_MASKDSTTRAN_INT_MASK_2                (0x4UL<<DMA_MASKDSTTRAN_INT_MASK_Pos)             /*!< 0x00000004 */
#define DMA_MASKDSTTRAN_INT_MASK_3                (0x8UL<<DMA_MASKDSTTRAN_INT_MASK_Pos)             /*!< 0x00000008 */
#define DMA_MASKDSTTRAN_INT_MASK_4                (0x10UL<<DMA_MASKDSTTRAN_INT_MASK_Pos)            /*!< 0x00000010 */
#define DMA_MASKDSTTRAN_INT_MASK_5                (0x20UL<<DMA_MASKDSTTRAN_INT_MASK_Pos)            /*!< 0x00000020 */
#define DMA_MASKDSTTRAN_INT_MASK_WE_Pos           (8U)
#define DMA_MASKDSTTRAN_INT_MASK_WE_Msk           (0x3FUL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)         /*!< 0x00003F00 */
#define DMA_MASKDSTTRAN_INT_MASK_WE               DMA_MASKDSTTRAN_INT_MASK_WE_Msk                   /*!< Interrupt Mask Write Enable */
#define DMA_MASKDSTTRAN_INT_MASK_WE_0             (0x1UL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)          /*!< 0x00000100 */
#define DMA_MASKDSTTRAN_INT_MASK_WE_1             (0x2UL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)          /*!< 0x00000200 */
#define DMA_MASKDSTTRAN_INT_MASK_WE_2             (0x4UL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)          /*!< 0x00000400 */
#define DMA_MASKDSTTRAN_INT_MASK_WE_3             (0x8UL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)          /*!< 0x00000800 */
#define DMA_MASKDSTTRAN_INT_MASK_WE_4             (0x10UL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)         /*!< 0x00001000 */
#define DMA_MASKDSTTRAN_INT_MASK_WE_5             (0x20UL<<DMA_MASKDSTTRAN_INT_MASK_WE_Pos)         /*!< 0x00002000 */

/********************************* Bit definition for DMA_MASKERR register ******************************************/
#define DMA_MASKERR_INT_MASK_Pos                  (0U)
#define DMA_MASKERR_INT_MASK_Msk                  (0x3FUL<<DMA_MASKERR_INT_MASK_Pos)                /*!< 0x0000003F */
#define DMA_MASKERR_INT_MASK                      DMA_MASKERR_INT_MASK_Msk                          /*!< Mask for IntErr Interrupt */
#define DMA_MASKERR_INT_MASK_0                    (0x1UL<<DMA_MASKERR_INT_MASK_Pos)                 /*!< 0x00000001 */
#define DMA_MASKERR_INT_MASK_1                    (0x2UL<<DMA_MASKERR_INT_MASK_Pos)                 /*!< 0x00000002 */
#define DMA_MASKERR_INT_MASK_2                    (0x4UL<<DMA_MASKERR_INT_MASK_Pos)                 /*!< 0x00000004 */
#define DMA_MASKERR_INT_MASK_3                    (0x8UL<<DMA_MASKERR_INT_MASK_Pos)                 /*!< 0x00000008 */
#define DMA_MASKERR_INT_MASK_4                    (0x10UL<<DMA_MASKERR_INT_MASK_Pos)                /*!< 0x00000010 */
#define DMA_MASKERR_INT_MASK_5                    (0x20UL<<DMA_MASKERR_INT_MASK_Pos)                /*!< 0x00000020 */
#define DMA_MASKERR_INT_MASK_WE_Pos               (8U)
#define DMA_MASKERR_INT_MASK_WE_Msk               (0x3FUL<<DMA_MASKERR_INT_MASK_WE_Pos)             /*!< 0x00003F00 */
#define DMA_MASKERR_INT_MASK_WE                   DMA_MASKERR_INT_MASK_WE_Msk                       /*!< Interrupt Mask Write Enable */
#define DMA_MASKERR_INT_MASK_WE_0                 (0x1UL<<DMA_MASKERR_INT_MASK_WE_Pos)              /*!< 0x00000100 */
#define DMA_MASKERR_INT_MASK_WE_1                 (0x2UL<<DMA_MASKERR_INT_MASK_WE_Pos)              /*!< 0x00000200 */
#define DMA_MASKERR_INT_MASK_WE_2                 (0x4UL<<DMA_MASKERR_INT_MASK_WE_Pos)              /*!< 0x00000400 */
#define DMA_MASKERR_INT_MASK_WE_3                 (0x8UL<<DMA_MASKERR_INT_MASK_WE_Pos)              /*!< 0x00000800 */
#define DMA_MASKERR_INT_MASK_WE_4                 (0x10UL<<DMA_MASKERR_INT_MASK_WE_Pos)             /*!< 0x00001000 */
#define DMA_MASKERR_INT_MASK_WE_5                 (0x20UL<<DMA_MASKERR_INT_MASK_WE_Pos)             /*!< 0x00002000 */

/********************************* Bit definition for DMA_CLEARTFR register *****************************************/
#define DMA_CLEARTFR_CLEAR_Pos                    (0U)
#define DMA_CLEARTFR_CLEAR_Msk                    (0x3FUL<<DMA_CLEARTFR_CLEAR_Pos)                  /*!< 0x0000003F */
#define DMA_CLEARTFR_CLEAR                        DMA_CLEARTFR_CLEAR_Msk                            /*!< Clear for IntTfr Interrupt */
#define DMA_CLEARTFR_CLEAR_0                      (0x1UL<<DMA_CLEARTFR_CLEAR_Pos)                   /*!< 0x00000001 */
#define DMA_CLEARTFR_CLEAR_1                      (0x2UL<<DMA_CLEARTFR_CLEAR_Pos)                   /*!< 0x00000002 */
#define DMA_CLEARTFR_CLEAR_2                      (0x4UL<<DMA_CLEARTFR_CLEAR_Pos)                   /*!< 0x00000004 */
#define DMA_CLEARTFR_CLEAR_3                      (0x8UL<<DMA_CLEARTFR_CLEAR_Pos)                   /*!< 0x00000008 */
#define DMA_CLEARTFR_CLEAR_4                      (0x10UL<<DMA_CLEARTFR_CLEAR_Pos)                  /*!< 0x00000010 */
#define DMA_CLEARTFR_CLEAR_5                      (0x20UL<<DMA_CLEARTFR_CLEAR_Pos)                  /*!< 0x00000020 */

/********************************* Bit definition for DMA_CLEARBLOCK register ***************************************/
#define DMA_CLEARBLOCK_CLEAR_Pos                  (0U)
#define DMA_CLEARBLOCK_CLEAR_Msk                  (0x3FUL<<DMA_CLEARBLOCK_CLEAR_Pos)                /*!< 0x0000003F */
#define DMA_CLEARBLOCK_CLEAR                      DMA_CLEARBLOCK_CLEAR_Msk                          /*!< Clear for IntBlock Interrupt */
#define DMA_CLEARBLOCK_CLEAR_0                    (0x1UL<<DMA_CLEARBLOCK_CLEAR_Pos)                 /*!< 0x00000001 */
#define DMA_CLEARBLOCK_CLEAR_1                    (0x2UL<<DMA_CLEARBLOCK_CLEAR_Pos)                 /*!< 0x00000002 */
#define DMA_CLEARBLOCK_CLEAR_2                    (0x4UL<<DMA_CLEARBLOCK_CLEAR_Pos)                 /*!< 0x00000004 */
#define DMA_CLEARBLOCK_CLEAR_3                    (0x8UL<<DMA_CLEARBLOCK_CLEAR_Pos)                 /*!< 0x00000008 */
#define DMA_CLEARBLOCK_CLEAR_4                    (0x10UL<<DMA_CLEARBLOCK_CLEAR_Pos)                /*!< 0x00000010 */
#define DMA_CLEARBLOCK_CLEAR_5                    (0x20UL<<DMA_CLEARBLOCK_CLEAR_Pos)                /*!< 0x00000020 */

/********************************* Bit definition for DMA_CLEARSRCTRAN register *************************************/
#define DMA_CLEARSRCTRAN_CLEAR_Pos                (0U)
#define DMA_CLEARSRCTRAN_CLEAR_Msk                (0x3FUL<<DMA_CLEARSRCTRAN_CLEAR_Pos)              /*!< 0x0000003F */
#define DMA_CLEARSRCTRAN_CLEAR                    DMA_CLEARSRCTRAN_CLEAR_Msk                        /*!< Clear for IntSrcTran Interrupt */
#define DMA_CLEARSRCTRAN_CLEAR_0                  (0x1UL<<DMA_CLEARSRCTRAN_CLEAR_Pos)               /*!< 0x00000001 */
#define DMA_CLEARSRCTRAN_CLEAR_1                  (0x2UL<<DMA_CLEARSRCTRAN_CLEAR_Pos)               /*!< 0x00000002 */
#define DMA_CLEARSRCTRAN_CLEAR_2                  (0x4UL<<DMA_CLEARSRCTRAN_CLEAR_Pos)               /*!< 0x00000004 */
#define DMA_CLEARSRCTRAN_CLEAR_3                  (0x8UL<<DMA_CLEARSRCTRAN_CLEAR_Pos)               /*!< 0x00000008 */
#define DMA_CLEARSRCTRAN_CLEAR_4                  (0x10UL<<DMA_CLEARSRCTRAN_CLEAR_Pos)              /*!< 0x00000010 */
#define DMA_CLEARSRCTRAN_CLEAR_5                  (0x20UL<<DMA_CLEARSRCTRAN_CLEAR_Pos)              /*!< 0x00000020 */

/********************************* Bit definition for DMA_CLEARDSTTRAN register *************************************/
#define DMA_CLEARDSTTRAN_CLEAR_Pos                (0U)
#define DMA_CLEARDSTTRAN_CLEAR_Msk                (0x3FUL<<DMA_CLEARDSTTRAN_CLEAR_Pos)              /*!< 0x0000003F */
#define DMA_CLEARDSTTRAN_CLEAR                    DMA_CLEARDSTTRAN_CLEAR_Msk                        /*!< Clear for IntDstTran Interrupt */
#define DMA_CLEARDSTTRAN_CLEAR_0                  (0x1UL<<DMA_CLEARDSTTRAN_CLEAR_Pos)               /*!< 0x00000001 */
#define DMA_CLEARDSTTRAN_CLEAR_1                  (0x2UL<<DMA_CLEARDSTTRAN_CLEAR_Pos)               /*!< 0x00000002 */
#define DMA_CLEARDSTTRAN_CLEAR_2                  (0x4UL<<DMA_CLEARDSTTRAN_CLEAR_Pos)               /*!< 0x00000004 */
#define DMA_CLEARDSTTRAN_CLEAR_3                  (0x8UL<<DMA_CLEARDSTTRAN_CLEAR_Pos)               /*!< 0x00000008 */
#define DMA_CLEARDSTTRAN_CLEAR_4                  (0x10UL<<DMA_CLEARDSTTRAN_CLEAR_Pos)              /*!< 0x00000010 */
#define DMA_CLEARDSTTRAN_CLEAR_5                  (0x20UL<<DMA_CLEARDSTTRAN_CLEAR_Pos)              /*!< 0x00000020 */

/********************************* Bit definition for DMA_CLEARERR register *****************************************/
#define DMA_CLEARERR_CLEAR_Pos                    (0U)
#define DMA_CLEARERR_CLEAR_Msk                    (0x3FUL<<DMA_CLEARERR_CLEAR_Pos)                  /*!< 0x0000003F */
#define DMA_CLEARERR_CLEAR                        DMA_CLEARERR_CLEAR_Msk                            /*!< Clear for IntErr Interrupt */
#define DMA_CLEARERR_CLEAR_0                      (0x1UL<<DMA_CLEARERR_CLEAR_Pos)                   /*!< 0x00000001 */
#define DMA_CLEARERR_CLEAR_1                      (0x2UL<<DMA_CLEARERR_CLEAR_Pos)                   /*!< 0x00000002 */
#define DMA_CLEARERR_CLEAR_2                      (0x4UL<<DMA_CLEARERR_CLEAR_Pos)                   /*!< 0x00000004 */
#define DMA_CLEARERR_CLEAR_3                      (0x8UL<<DMA_CLEARERR_CLEAR_Pos)                   /*!< 0x00000008 */
#define DMA_CLEARERR_CLEAR_4                      (0x10UL<<DMA_CLEARERR_CLEAR_Pos)                  /*!< 0x00000010 */
#define DMA_CLEARERR_CLEAR_5                      (0x20UL<<DMA_CLEARERR_CLEAR_Pos)                  /*!< 0x00000020 */

/********************************* Bit definition for DMA_STATUSINT register ****************************************/
#define DMA_STATUSINT_TFR_Pos                     (0U)
#define DMA_STATUSINT_TFR_Msk                     (0x1UL<<DMA_STATUSINT_TFR_Pos)                    /*!< 0x00000001 */
#define DMA_STATUSINT_TFR                         DMA_STATUSINT_TFR_Msk                             /*!< OR of the contents of StatusTfr */
#define DMA_STATUSINT_BLOCK_Pos                   (1U)
#define DMA_STATUSINT_BLOCK_Msk                   (0x1UL<<DMA_STATUSINT_BLOCK_Pos)                  /*!< 0x00000002 */
#define DMA_STATUSINT_BLOCK                       DMA_STATUSINT_BLOCK_Msk                           /*!< OR of the contents of StatusBlock */
#define DMA_STATUSINT_SRCT_Pos                    (2U)
#define DMA_STATUSINT_SRCT_Msk                    (0x1UL<<DMA_STATUSINT_SRCT_Pos)                   /*!< 0x00000004 */
#define DMA_STATUSINT_SRCT                        DMA_STATUSINT_SRCT_Msk                            /*!< OR of the contents of StatusSrcTran */
#define DMA_STATUSINT_DSTT_Pos                    (3U)
#define DMA_STATUSINT_DSTT_Msk                    (0x1UL<<DMA_STATUSINT_DSTT_Pos)                   /*!< 0x00000008 */
#define DMA_STATUSINT_DSTT                        DMA_STATUSINT_DSTT_Msk                            /*!< OR of the contents of StatusDstTran */
#define DMA_STATUSINT_ERR_Pos                     (4U)
#define DMA_STATUSINT_ERR_Msk                     (0x1UL<<DMA_STATUSINT_ERR_Pos)                    /*!< 0x00000010 */
#define DMA_STATUSINT_ERR                         DMA_STATUSINT_ERR_Msk                             /*!< OR of the contents of StatusErr */
#define DMA_STATUSINT_HALFBLOCK_Pos               (5U)
#define DMA_STATUSINT_HALFBLOCK_Msk               (0x1UL<<DMA_STATUSINT_HALFBLOCK_Pos)              /*!< 0x00000020 */
#define DMA_STATUSINT_HALFBLOCK                   DMA_STATUSINT_HALFBLOCK_Msk                       /*!< OR of the contents of StatusHalfBlock */

/********************************* Bit definition for DMA_REQSRC register *******************************************/
#define DMA_REQSRC_SRC_REQ_Pos                    (0U)
#define DMA_REQSRC_SRC_REQ_Msk                    (0x3FUL<<DMA_REQSRC_SRC_REQ_Pos)                  /*!< 0x0000003F */
#define DMA_REQSRC_SRC_REQ                        DMA_REQSRC_SRC_REQ_Msk                            /*!< Source Software Transaction Request */
#define DMA_REQSRC_SRC_REQ_0                      (0x1UL<<DMA_REQSRC_SRC_REQ_Pos)                   /*!< 0x00000001 */
#define DMA_REQSRC_SRC_REQ_1                      (0x2UL<<DMA_REQSRC_SRC_REQ_Pos)                   /*!< 0x00000002 */
#define DMA_REQSRC_SRC_REQ_2                      (0x4UL<<DMA_REQSRC_SRC_REQ_Pos)                   /*!< 0x00000004 */
#define DMA_REQSRC_SRC_REQ_3                      (0x8UL<<DMA_REQSRC_SRC_REQ_Pos)                   /*!< 0x00000008 */
#define DMA_REQSRC_SRC_REQ_4                      (0x10UL<<DMA_REQSRC_SRC_REQ_Pos)                  /*!< 0x00000010 */
#define DMA_REQSRC_SRC_REQ_5                      (0x20UL<<DMA_REQSRC_SRC_REQ_Pos)                  /*!< 0x00000020 */
#define DMA_REQSRC_SRC_REQ_WE_Pos                 (8U)
#define DMA_REQSRC_SRC_REQ_WE_Msk                 (0x3FUL<<DMA_REQSRC_SRC_REQ_WE_Pos)               /*!< 0x00003F00 */
#define DMA_REQSRC_SRC_REQ_WE                     DMA_REQSRC_SRC_REQ_WE_Msk                         /*!< Source Software Transaction Request write enable */
#define DMA_REQSRC_SRC_REQ_WE_0                   (0x1UL<<DMA_REQSRC_SRC_REQ_WE_Pos)                /*!< 0x00000100 */
#define DMA_REQSRC_SRC_REQ_WE_1                   (0x2UL<<DMA_REQSRC_SRC_REQ_WE_Pos)                /*!< 0x00000200 */
#define DMA_REQSRC_SRC_REQ_WE_2                   (0x4UL<<DMA_REQSRC_SRC_REQ_WE_Pos)                /*!< 0x00000400 */
#define DMA_REQSRC_SRC_REQ_WE_3                   (0x8UL<<DMA_REQSRC_SRC_REQ_WE_Pos)                /*!< 0x00000800 */
#define DMA_REQSRC_SRC_REQ_WE_4                   (0x10UL<<DMA_REQSRC_SRC_REQ_WE_Pos)               /*!< 0x00001000 */
#define DMA_REQSRC_SRC_REQ_WE_5                   (0x20UL<<DMA_REQSRC_SRC_REQ_WE_Pos)               /*!< 0x00002000 */

/********************************* Bit definition for DMA_REQDST register *******************************************/
#define DMA_REQDST_DST_REQ_Pos                    (0U)
#define DMA_REQDST_DST_REQ_Msk                    (0x3FUL<<DMA_REQDST_DST_REQ_Pos)                  /*!< 0x0000003F */
#define DMA_REQDST_DST_REQ                        DMA_REQDST_DST_REQ_Msk                            /*!< Destination Software Transaction Request */
#define DMA_REQDST_DST_REQ_0                      (0x1UL<<DMA_REQDST_DST_REQ_Pos)                   /*!< 0x00000001 */
#define DMA_REQDST_DST_REQ_1                      (0x2UL<<DMA_REQDST_DST_REQ_Pos)                   /*!< 0x00000002 */
#define DMA_REQDST_DST_REQ_2                      (0x4UL<<DMA_REQDST_DST_REQ_Pos)                   /*!< 0x00000004 */
#define DMA_REQDST_DST_REQ_3                      (0x8UL<<DMA_REQDST_DST_REQ_Pos)                   /*!< 0x00000008 */
#define DMA_REQDST_DST_REQ_4                      (0x10UL<<DMA_REQDST_DST_REQ_Pos)                  /*!< 0x00000010 */
#define DMA_REQDST_DST_REQ_5                      (0x20UL<<DMA_REQDST_DST_REQ_Pos)                  /*!< 0x00000020 */
#define DMA_REQDST_DST_REQ_WE_Pos                 (8U)
#define DMA_REQDST_DST_REQ_WE_Msk                 (0x3FUL<<DMA_REQDST_DST_REQ_WE_Pos)               /*!< 0x00003F00 */
#define DMA_REQDST_DST_REQ_WE                     DMA_REQDST_DST_REQ_WE_Msk                         /*!< Destination Software Transaction Request write enable */
#define DMA_REQDST_DST_REQ_WE_0                   (0x1UL<<DMA_REQDST_DST_REQ_WE_Pos)                /*!< 0x00000100 */
#define DMA_REQDST_DST_REQ_WE_1                   (0x2UL<<DMA_REQDST_DST_REQ_WE_Pos)                /*!< 0x00000200 */
#define DMA_REQDST_DST_REQ_WE_2                   (0x4UL<<DMA_REQDST_DST_REQ_WE_Pos)                /*!< 0x00000400 */
#define DMA_REQDST_DST_REQ_WE_3                   (0x8UL<<DMA_REQDST_DST_REQ_WE_Pos)                /*!< 0x00000800 */
#define DMA_REQDST_DST_REQ_WE_4                   (0x10UL<<DMA_REQDST_DST_REQ_WE_Pos)               /*!< 0x00001000 */
#define DMA_REQDST_DST_REQ_WE_5                   (0x20UL<<DMA_REQDST_DST_REQ_WE_Pos)               /*!< 0x00002000 */

/********************************* Bit definition for DMA_SGLRQSRC register *****************************************/
#define DMA_SGLRQSRC_SRC_SGLREQ_Pos               (0U)
#define DMA_SGLRQSRC_SRC_SGLREQ_Msk               (0x3FUL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)             /*!< 0x0000003F */
#define DMA_SGLRQSRC_SRC_SGLREQ                   DMA_SGLRQSRC_SRC_SGLREQ_Msk                       /*!< Source Single Transaction Request */
#define DMA_SGLRQSRC_SRC_SGLREQ_0                 (0x1UL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)              /*!< 0x00000001 */
#define DMA_SGLRQSRC_SRC_SGLREQ_1                 (0x2UL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)              /*!< 0x00000002 */
#define DMA_SGLRQSRC_SRC_SGLREQ_2                 (0x4UL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)              /*!< 0x00000004 */
#define DMA_SGLRQSRC_SRC_SGLREQ_3                 (0x8UL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)              /*!< 0x00000008 */
#define DMA_SGLRQSRC_SRC_SGLREQ_4                 (0x10UL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)             /*!< 0x00000010 */
#define DMA_SGLRQSRC_SRC_SGLREQ_5                 (0x20UL<<DMA_SGLRQSRC_SRC_SGLREQ_Pos)             /*!< 0x00000020 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos            (8U)
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_Msk            (0x3FUL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)          /*!< 0x00003F00 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE                DMA_SGLRQSRC_SRC_SGLREQ_WE_Msk                    /*!< Source Single Transaction Request write enable */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_0              (0x1UL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)           /*!< 0x00000100 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_1              (0x2UL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)           /*!< 0x00000200 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_2              (0x4UL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)           /*!< 0x00000400 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_3              (0x8UL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)           /*!< 0x00000800 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_4              (0x10UL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)          /*!< 0x00001000 */
#define DMA_SGLRQSRC_SRC_SGLREQ_WE_5              (0x20UL<<DMA_SGLRQSRC_SRC_SGLREQ_WE_Pos)          /*!< 0x00002000 */

/********************************* Bit definition for DMA_SGLRQDST register *****************************************/
#define DMA_SGLRQDST_DST_SGLREQ_Pos               (0U)
#define DMA_SGLRQDST_DST_SGLREQ_Msk               (0x3FUL<<DMA_SGLRQDST_DST_SGLREQ_Pos)             /*!< 0x0000003F */
#define DMA_SGLRQDST_DST_SGLREQ                   DMA_SGLRQDST_DST_SGLREQ_Msk                       /*!< Destination Single Transaction Request */
#define DMA_SGLRQDST_DST_SGLREQ_0                 (0x1UL<<DMA_SGLRQDST_DST_SGLREQ_Pos)              /*!< 0x00000001 */
#define DMA_SGLRQDST_DST_SGLREQ_1                 (0x2UL<<DMA_SGLRQDST_DST_SGLREQ_Pos)              /*!< 0x00000002 */
#define DMA_SGLRQDST_DST_SGLREQ_2                 (0x4UL<<DMA_SGLRQDST_DST_SGLREQ_Pos)              /*!< 0x00000004 */
#define DMA_SGLRQDST_DST_SGLREQ_3                 (0x8UL<<DMA_SGLRQDST_DST_SGLREQ_Pos)              /*!< 0x00000008 */
#define DMA_SGLRQDST_DST_SGLREQ_4                 (0x10UL<<DMA_SGLRQDST_DST_SGLREQ_Pos)             /*!< 0x00000010 */
#define DMA_SGLRQDST_DST_SGLREQ_5                 (0x20UL<<DMA_SGLRQDST_DST_SGLREQ_Pos)             /*!< 0x00000020 */
#define DMA_SGLRQDST_DST_SGLREQ_WE_Pos            (8U)
#define DMA_SGLRQDST_DST_SGLREQ_WE_Msk            (0x3FUL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)          /*!< 0x00003F00 */
#define DMA_SGLRQDST_DST_SGLREQ_WE                DMA_SGLRQDST_DST_SGLREQ_WE_Msk                    /*!< Destination Single Transaction Request write enable */
#define DMA_SGLRQDST_DST_SGLREQ_WE_0              (0x1UL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)           /*!< 0x00000100 */
#define DMA_SGLRQDST_DST_SGLREQ_WE_1              (0x2UL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)           /*!< 0x00000200 */
#define DMA_SGLRQDST_DST_SGLREQ_WE_2              (0x4UL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)           /*!< 0x00000400 */
#define DMA_SGLRQDST_DST_SGLREQ_WE_3              (0x8UL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)           /*!< 0x00000800 */
#define DMA_SGLRQDST_DST_SGLREQ_WE_4              (0x10UL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)          /*!< 0x00001000 */
#define DMA_SGLRQDST_DST_SGLREQ_WE_5              (0x20UL<<DMA_SGLRQDST_DST_SGLREQ_WE_Pos)          /*!< 0x00002000 */

/********************************************************************************************************************/
/********************************* ETH ******************************************************************************/
/********************************************************************************************************************/
/* Bit definition for Ethernet MAC Control Register register */
#define ETH_MACCR_WD_Pos                       (23U)                           
#define ETH_MACCR_WD_Msk                       (0x1UL << ETH_MACCR_WD_Pos)      /*!< 0x00800000 */
#define ETH_MACCR_WD                           ETH_MACCR_WD_Msk                /* Watchdog disable */
#define ETH_MACCR_JD_Pos                       (22U)                           
#define ETH_MACCR_JD_Msk                       (0x1UL << ETH_MACCR_JD_Pos)      /*!< 0x00400000 */
#define ETH_MACCR_JD                           ETH_MACCR_JD_Msk                /* Jabber disable */
#define ETH_MACCR_IFG_Pos                      (17U)                           
#define ETH_MACCR_IFG_Msk                      (0x7UL << ETH_MACCR_IFG_Pos)     /*!< 0x000E0000 */
#define ETH_MACCR_IFG                          ETH_MACCR_IFG_Msk               /* Inter-frame gap */
#define ETH_MACCR_IFG_96Bit                    0x00000000U                     /* Minimum IFG between frames during transmission is 96Bit */
#define ETH_MACCR_IFG_88Bit                    0x00020000U                     /* Minimum IFG between frames during transmission is 88Bit */
#define ETH_MACCR_IFG_80Bit                    0x00040000U                     /* Minimum IFG between frames during transmission is 80Bit */
#define ETH_MACCR_IFG_72Bit                    0x00060000U                     /* Minimum IFG between frames during transmission is 72Bit */
#define ETH_MACCR_IFG_64Bit                    0x00080000U                     /* Minimum IFG between frames during transmission is 64Bit */        
#define ETH_MACCR_IFG_56Bit                    0x000A0000U                     /* Minimum IFG between frames during transmission is 56Bit */
#define ETH_MACCR_IFG_48Bit                    0x000C0000U                     /* Minimum IFG between frames during transmission is 48Bit */
#define ETH_MACCR_IFG_40Bit                    0x000E0000U                     /* Minimum IFG between frames during transmission is 40Bit */              
#define ETH_MACCR_CSD_Pos                      (16U)                           
#define ETH_MACCR_CSD_Msk                      (0x1UL << ETH_MACCR_CSD_Pos)     /*!< 0x00010000 */
#define ETH_MACCR_CSD                          ETH_MACCR_CSD_Msk               /* Carrier sense disable (during transmission) */
#define ETH_MACCR_FES_Pos                      (14U)                           
#define ETH_MACCR_FES_Msk                      (0x1UL << ETH_MACCR_FES_Pos)     /*!< 0x00004000 */
#define ETH_MACCR_FES                          ETH_MACCR_FES_Msk               /* Fast ethernet speed */
#define ETH_MACCR_ROD_Pos                      (13U)                           
#define ETH_MACCR_ROD_Msk                      (0x1UL << ETH_MACCR_ROD_Pos)     /*!< 0x00002000 */
#define ETH_MACCR_ROD                          ETH_MACCR_ROD_Msk               /* Receive own disable */
#define ETH_MACCR_LM_Pos                       (12U)                           
#define ETH_MACCR_LM_Msk                       (0x1UL << ETH_MACCR_LM_Pos)      /*!< 0x00001000 */
#define ETH_MACCR_LM                           ETH_MACCR_LM_Msk                /* loopback mode */
#define ETH_MACCR_DM_Pos                       (11U)                           
#define ETH_MACCR_DM_Msk                       (0x1UL << ETH_MACCR_DM_Pos)      /*!< 0x00000800 */
#define ETH_MACCR_DM                           ETH_MACCR_DM_Msk                /* Duplex mode */
#define ETH_MACCR_IPCO_Pos                     (10U)                           
#define ETH_MACCR_IPCO_Msk                     (0x1UL << ETH_MACCR_IPCO_Pos)    /*!< 0x00000400 */
#define ETH_MACCR_IPCO                         ETH_MACCR_IPCO_Msk              /* IP Checksum offload */
#define ETH_MACCR_RD_Pos                       (9U)                            
#define ETH_MACCR_RD_Msk                       (0x1UL << ETH_MACCR_RD_Pos)      /*!< 0x00000200 */
#define ETH_MACCR_RD                           ETH_MACCR_RD_Msk                /* Retry disable */
#define ETH_MACCR_APCS_Pos                     (7U)                            
#define ETH_MACCR_APCS_Msk                     (0x1UL << ETH_MACCR_APCS_Pos)    /*!< 0x00000080 */
#define ETH_MACCR_APCS                         ETH_MACCR_APCS_Msk              /* Automatic Pad/CRC stripping */
#define ETH_MACCR_BL_Pos                       (5U)                            
#define ETH_MACCR_BL_Msk                       (0x3UL << ETH_MACCR_BL_Pos)      /*!< 0x00000060 */
#define ETH_MACCR_BL                           ETH_MACCR_BL_Msk                /* Back-off limit: random integer number (r) of slot time delays before rescheduling
                                                       a transmission attempt during retries after a collision: 0 =< r <2^k */
#define ETH_MACCR_BL_10                        0x00000000U                     /* k = min (n, 10) */
#define ETH_MACCR_BL_8                         0x00000020U                     /* k = min (n, 8) */
#define ETH_MACCR_BL_4                         0x00000040U                     /* k = min (n, 4) */
#define ETH_MACCR_BL_1                         0x00000060U                     /* k = min (n, 1) */ 
#define ETH_MACCR_DC_Pos                       (4U)                            
#define ETH_MACCR_DC_Msk                       (0x1UL << ETH_MACCR_DC_Pos)      /*!< 0x00000010 */
#define ETH_MACCR_DC                           ETH_MACCR_DC_Msk                /* Defferal check */
#define ETH_MACCR_TE_Pos                       (3U)                            
#define ETH_MACCR_TE_Msk                       (0x1UL << ETH_MACCR_TE_Pos)      /*!< 0x00000008 */
#define ETH_MACCR_TE                           ETH_MACCR_TE_Msk                /* Transmitter enable */
#define ETH_MACCR_RE_Pos                       (2U)                            
#define ETH_MACCR_RE_Msk                       (0x1UL << ETH_MACCR_RE_Pos)      /*!< 0x00000004 */
#define ETH_MACCR_RE                           ETH_MACCR_RE_Msk                /* Receiver enable */
#define ETH_MACCR_PRELEN_Pos                   (0U)                            
#define ETH_MACCR_PRELEN_Msk                   (0x3UL << ETH_MACCR_PRELEN_Pos)  /*!< 0x00000003 */
#define ETH_MACCR_PRELEN                       ETH_MACCR_PRELEN_Msk            /* Preamble Length for Transmit frames */
#define ETH_MACCR_PRELEN_0                     (0x1UL << ETH_MACCR_PRELEN_Pos)  /*!< 0x00000001 */
#define ETH_MACCR_PRELEN_1                     (0x2UL << ETH_MACCR_PRELEN_Pos)  /*!< 0x00000002 */

/* Bit definition for Ethernet MAC Frame Filter Register */
#define ETH_MACFFR_RA_Pos                      (31U)                           
#define ETH_MACFFR_RA_Msk                      (0x1UL << ETH_MACFFR_RA_Pos)     /*!< 0x80000000 */
#define ETH_MACFFR_RA                          ETH_MACFFR_RA_Msk               /* Receive all */ 
#define ETH_MACFFR_VTFE_Pos                    (16U)                           
#define ETH_MACFFR_VTFE_Msk                    (0x1UL << ETH_MACFFR_VTFE_Pos)   /*!< 0x00010000 */
#define ETH_MACFFR_VTFE                        ETH_MACFFR_VTFE_Msk             /* VLAN Tag Filter Enable */ 
#define ETH_MACFFR_HPF_Pos                     (10U)                           
#define ETH_MACFFR_HPF_Msk                     (0x1UL << ETH_MACFFR_HPF_Pos)    /*!< 0x00000400 */
#define ETH_MACFFR_HPF                         ETH_MACFFR_HPF_Msk              /* Hash or perfect filter */ 
#define ETH_MACFFR_SAF_Pos                     (9U)                            
#define ETH_MACFFR_SAF_Msk                     (0x1UL << ETH_MACFFR_SAF_Pos)    /*!< 0x00000200 */
#define ETH_MACFFR_SAF                         ETH_MACFFR_SAF_Msk              /* Source address filter enable */ 
#define ETH_MACFFR_SAIF_Pos                    (8U)                            
#define ETH_MACFFR_SAIF_Msk                    (0x1UL << ETH_MACFFR_SAIF_Pos)   /*!< 0x00000100 */
#define ETH_MACFFR_SAIF                        ETH_MACFFR_SAIF_Msk             /* SA inverse filtering */ 
#define ETH_MACFFR_PCF_Pos                     (6U)                            
#define ETH_MACFFR_PCF_Msk                     (0x3UL << ETH_MACFFR_PCF_Pos)    /*!< 0x000000C0 */
#define ETH_MACFFR_PCF                         ETH_MACFFR_PCF_Msk              /* Pass control frames: 3 cases */
#define ETH_MACFFR_PCF_BlockAll_Pos            (6U)                            
#define ETH_MACFFR_PCF_BlockAll_Msk            (0x1UL << ETH_MACFFR_PCF_BlockAll_Pos) /*!< 0x00000040 */
#define ETH_MACFFR_PCF_BlockAll                ETH_MACFFR_PCF_BlockAll_Msk     /* MAC filters all control frames from reaching the application */
#define ETH_MACFFR_PCF_ForwardAll_Pos          (7U)                            
#define ETH_MACFFR_PCF_ForwardAll_Msk          (0x1UL << ETH_MACFFR_PCF_ForwardAll_Pos) /*!< 0x00000080 */
#define ETH_MACFFR_PCF_ForwardAll              ETH_MACFFR_PCF_ForwardAll_Msk   /* MAC forwards all control frames to application even if they fail the Address Filter */
#define ETH_MACFFR_PCF_ForwardPassedAddrFilter_Pos (6U)                        
#define ETH_MACFFR_PCF_ForwardPassedAddrFilter_Msk (0x3UL << ETH_MACFFR_PCF_ForwardPassedAddrFilter_Pos) /*!< 0x000000C0 */
#define ETH_MACFFR_PCF_ForwardPassedAddrFilter ETH_MACFFR_PCF_ForwardPassedAddrFilter_Msk /* MAC forwards control frames that pass the Address Filter. */ 
#define ETH_MACFFR_BFD_Pos                     (5U)                            
#define ETH_MACFFR_BFD_Msk                     (0x1UL << ETH_MACFFR_BFD_Pos)    /*!< 0x00000020 */
#define ETH_MACFFR_BFD                         ETH_MACFFR_BFD_Msk              /* Broadcast frame disable */ 
#define ETH_MACFFR_PAM_Pos                     (4U)                            
#define ETH_MACFFR_PAM_Msk                     (0x1UL << ETH_MACFFR_PAM_Pos)    /*!< 0x00000010 */
#define ETH_MACFFR_PAM                         ETH_MACFFR_PAM_Msk              /* Pass all mutlicast */ 
#define ETH_MACFFR_DAIF_Pos                    (3U)                            
#define ETH_MACFFR_DAIF_Msk                    (0x1UL << ETH_MACFFR_DAIF_Pos)   /*!< 0x00000008 */
#define ETH_MACFFR_DAIF                        ETH_MACFFR_DAIF_Msk             /* DA Inverse filtering */ 
#define ETH_MACFFR_HM_Pos                      (2U)                            
#define ETH_MACFFR_HM_Msk                      (0x1UL << ETH_MACFFR_HM_Pos)     /*!< 0x00000004 */
#define ETH_MACFFR_HM                          ETH_MACFFR_HM_Msk               /* Hash multicast */ 
#define ETH_MACFFR_HU_Pos                      (1U)                            
#define ETH_MACFFR_HU_Msk                      (0x1UL << ETH_MACFFR_HU_Pos)     /*!< 0x00000002 */
#define ETH_MACFFR_HU                          ETH_MACFFR_HU_Msk               /* Hash unicast */
#define ETH_MACFFR_PM_Pos                      (0U)                            
#define ETH_MACFFR_PM_Msk                      (0x1UL << ETH_MACFFR_PM_Pos)     /*!< 0x00000001 */
#define ETH_MACFFR_PM                          ETH_MACFFR_PM_Msk               /* Promiscuous mode */

/* Bit definition for Ethernet MAC Hash Table High Register */
#define ETH_MACHTHR_HTH_Pos                    (0U)                            
#define ETH_MACHTHR_HTH_Msk                    (0xFFFFFFFFUL << ETH_MACHTHR_HTH_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACHTHR_HTH                        ETH_MACHTHR_HTH_Msk             /* Hash table high */

/* Bit definition for Ethernet MAC Hash Table Low Register */
#define ETH_MACHTLR_HTL_Pos                    (0U)                            
#define ETH_MACHTLR_HTL_Msk                    (0xFFFFFFFFUL << ETH_MACHTLR_HTL_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACHTLR_HTL                        ETH_MACHTLR_HTL_Msk             /* Hash table low */

/* Bit definition for Ethernet MAC MII Address Register */
#define ETH_MACMIIAR_PA_Pos                    (11U)                           
#define ETH_MACMIIAR_PA_Msk                    (0x1FUL << ETH_MACMIIAR_PA_Pos)  /*!< 0x0000F800 */
#define ETH_MACMIIAR_PA                        ETH_MACMIIAR_PA_Msk             /* Physical layer address */ 
#define ETH_MACMIIAR_MR_Pos                    (6U)                            
#define ETH_MACMIIAR_MR_Msk                    (0x1FUL << ETH_MACMIIAR_MR_Pos)  /*!< 0x000007C0 */
#define ETH_MACMIIAR_MR                        ETH_MACMIIAR_MR_Msk             /* MII register in the selected PHY */ 
#define ETH_MACMIIAR_CR_Pos                    (2U)                            
#define ETH_MACMIIAR_CR_Msk                    (0xFUL << ETH_MACMIIAR_CR_Pos)   /*!< 0x0000003C */
#define ETH_MACMIIAR_CR                        ETH_MACMIIAR_CR_Msk             /* CR clock range: 14 cases */
#define ETH_MACMIIAR_CR_0                      (0x1UL << ETH_MACMIIAR_CR_Pos)   /*!< 0x00000004 */
#define ETH_MACMIIAR_CR_1                      (0x2UL << ETH_MACMIIAR_CR_Pos)   /*!< 0x00000008 */
#define ETH_MACMIIAR_CR_2                      (0x4UL << ETH_MACMIIAR_CR_Pos)   /*!< 0x00000010 */
#define ETH_MACMIIAR_CR_3                      (0x8UL << ETH_MACMIIAR_CR_Pos)   /*!< 0x00000020 */ 

#define ETH_MACMIIAR_CR_DIV42                  0x00000000U                     /* HCLK:60-100 MHz; MDC clock= HCLK/42 */
#define ETH_MACMIIAR_CR_DIV62_Pos              (2U)
#define ETH_MACMIIAR_CR_DIV62_Msk              (0x1UL << ETH_MACMIIAR_CR_DIV62_Pos) /*!< 0x00000004 */
#define ETH_MACMIIAR_CR_DIV62                  ETH_MACMIIAR_CR_DIV62_Msk       /* HCLK:100-150 MHz; MDC clock= HCLK/62 */
#define ETH_MACMIIAR_CR_DIV16_Pos              (3U)                            
#define ETH_MACMIIAR_CR_DIV16_Msk              (0x1UL << ETH_MACMIIAR_CR_DIV16_Pos) /*!< 0x00000008 */
#define ETH_MACMIIAR_CR_DIV16                  ETH_MACMIIAR_CR_DIV16_Msk       /* HCLK:20-35 MHz; MDC clock= HCLK/16 */
#define ETH_MACMIIAR_CR_DIV26_Pos              (2U)                            
#define ETH_MACMIIAR_CR_DIV26_Msk              (0x3UL << ETH_MACMIIAR_CR_DIV26_Pos) /*!< 0x0000000C */
#define ETH_MACMIIAR_CR_DIV26                  ETH_MACMIIAR_CR_DIV26_Msk       /* HCLK:35-60 MHz; MDC clock= HCLK/26 */
#define ETH_MACMIIAR_CR_DIV102_Pos             (4U)
#define ETH_MACMIIAR_CR_DIV102_Msk             (0x1UL << ETH_MACMIIAR_CR_DIV102_Pos) /*!< 0x00000010 */
#define ETH_MACMIIAR_CR_DIV102                 ETH_MACMIIAR_CR_DIV102_Msk       /* HCLK:150-250 MHz; MDC clock= HCLK/102 */
#define ETH_MACMIIAR_CR_DIV104_Pos             (2U)
#define ETH_MACMIIAR_CR_DIV104_Msk             (0x5UL << ETH_MACMIIAR_CR_DIV104_Pos) /*!< 0x00000014 */
#define ETH_MACMIIAR_CR_DIV104                 ETH_MACMIIAR_CR_DIV104_Msk       /* HCLK:250-300 MHz; MDC clock= HCLK/104 */
#define ETH_MACMIIAR_CR_DIV4                   ETH_MACMIIAR_CR_3                                                                   /* MDC clock= HCLK/4  */
#define ETH_MACMIIAR_CR_DIV6                   (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_0)                                               /* MDC clock= HCLK/6  */
#define ETH_MACMIIAR_CR_DIV8                   (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_1)                                               /* MDC clock= HCLK/8  */
#define ETH_MACMIIAR_CR_DIV10                  (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_1|ETH_MACMIIAR_CR_0)                             /* MDC clock= HCLK/10 */
#define ETH_MACMIIAR_CR_DIV12                  (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_2)                                               /* MDC clock= HCLK/12 */
#define ETH_MACMIIAR_CR_DIV14                  (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_2|ETH_MACMIIAR_CR_0)                             /* MDC clock= HCLK/14 */
//#define ETH_MACMIIAR_CR_DIV16                  (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_2|ETH_MACMIIAR_CR_1)                             /* MDC clock= HCLK/16 */
#define ETH_MACMIIAR_CR_DIV18                  (ETH_MACMIIAR_CR_3|ETH_MACMIIAR_CR_2|ETH_MACMIIAR_CR_1|ETH_MACMIIAR_CR_0)           /* MDC clock= HCLK/18 */

#define ETH_MACMIIAR_MW_Pos                    (1U)                            
#define ETH_MACMIIAR_MW_Msk                    (0x1UL << ETH_MACMIIAR_MW_Pos)   /*!< 0x00000002 */
#define ETH_MACMIIAR_MW                        ETH_MACMIIAR_MW_Msk             /* MII write */ 
#define ETH_MACMIIAR_MB_Pos                    (0U)                            
#define ETH_MACMIIAR_MB_Msk                    (0x1UL << ETH_MACMIIAR_MB_Pos)   /*!< 0x00000001 */
#define ETH_MACMIIAR_MB                        ETH_MACMIIAR_MB_Msk             /* MII busy */ 
  
/* Bit definition for Ethernet MAC MII Data Register */
#define ETH_MACMIIDR_MD_Pos                    (0U)                            
#define ETH_MACMIIDR_MD_Msk                    (0xFFFFUL << ETH_MACMIIDR_MD_Pos) /*!< 0x0000FFFF */
#define ETH_MACMIIDR_MD                        ETH_MACMIIDR_MD_Msk             /* MII data: read/write data from/to PHY */

/* Bit definition for Ethernet MAC Flow Control Register */
#define ETH_MACFCR_PT_Pos                      (16U)                           
#define ETH_MACFCR_PT_Msk                      (0xFFFFUL << ETH_MACFCR_PT_Pos)  /*!< 0xFFFF0000 */
#define ETH_MACFCR_PT                          ETH_MACFCR_PT_Msk               /* Pause time */
#define ETH_MACFCR_ZQPD_Pos                    (7U)                            
#define ETH_MACFCR_ZQPD_Msk                    (0x1UL << ETH_MACFCR_ZQPD_Pos)   /*!< 0x00000080 */
#define ETH_MACFCR_ZQPD                        ETH_MACFCR_ZQPD_Msk             /* Zero-quanta pause disable */
#define ETH_MACFCR_PLT_Pos                     (4U)                            
#define ETH_MACFCR_PLT_Msk                     (0x3UL << ETH_MACFCR_PLT_Pos)    /*!< 0x00000030 */
#define ETH_MACFCR_PLT                         ETH_MACFCR_PLT_Msk              /* Pause low threshold: 4 cases */
#define ETH_MACFCR_PLT_Minus4                  0x00000000U                     /* Pause time minus 4 slot times */
#define ETH_MACFCR_PLT_Minus28_Pos             (4U)                            
#define ETH_MACFCR_PLT_Minus28_Msk             (0x1UL << ETH_MACFCR_PLT_Minus28_Pos) /*!< 0x00000010 */
#define ETH_MACFCR_PLT_Minus28                 ETH_MACFCR_PLT_Minus28_Msk      /* Pause time minus 28 slot times */
#define ETH_MACFCR_PLT_Minus144_Pos            (5U)                            
#define ETH_MACFCR_PLT_Minus144_Msk            (0x1UL << ETH_MACFCR_PLT_Minus144_Pos) /*!< 0x00000020 */
#define ETH_MACFCR_PLT_Minus144                ETH_MACFCR_PLT_Minus144_Msk     /* Pause time minus 144 slot times */
#define ETH_MACFCR_PLT_Minus256_Pos            (4U)                            
#define ETH_MACFCR_PLT_Minus256_Msk            (0x3UL << ETH_MACFCR_PLT_Minus256_Pos) /*!< 0x00000030 */
#define ETH_MACFCR_PLT_Minus256                ETH_MACFCR_PLT_Minus256_Msk     /* Pause time minus 256 slot times */      
#define ETH_MACFCR_UPFD_Pos                    (3U)                            
#define ETH_MACFCR_UPFD_Msk                    (0x1UL << ETH_MACFCR_UPFD_Pos)   /*!< 0x00000008 */
#define ETH_MACFCR_UPFD                        ETH_MACFCR_UPFD_Msk             /* Unicast pause frame detect */
#define ETH_MACFCR_RFCE_Pos                    (2U)                            
#define ETH_MACFCR_RFCE_Msk                    (0x1UL << ETH_MACFCR_RFCE_Pos)   /*!< 0x00000004 */
#define ETH_MACFCR_RFCE                        ETH_MACFCR_RFCE_Msk             /* Receive flow control enable */
#define ETH_MACFCR_TFCE_Pos                    (1U)                            
#define ETH_MACFCR_TFCE_Msk                    (0x1UL << ETH_MACFCR_TFCE_Pos)   /*!< 0x00000002 */
#define ETH_MACFCR_TFCE                        ETH_MACFCR_TFCE_Msk             /* Transmit flow control enable */
#define ETH_MACFCR_FCBBPA_Pos                  (0U)                            
#define ETH_MACFCR_FCBBPA_Msk                  (0x1UL << ETH_MACFCR_FCBBPA_Pos) /*!< 0x00000001 */
#define ETH_MACFCR_FCBBPA                      ETH_MACFCR_FCBBPA_Msk           /* Flow control busy/backpressure activate */

/* Bit definition for Ethernet MAC VLAN Tag Register */
#define ETH_MACVLANTR_VLANTC_Pos               (16U)                           
#define ETH_MACVLANTR_VLANTC_Msk               (0x1UL << ETH_MACVLANTR_VLANTC_Pos) /*!< 0x00010000 */
#define ETH_MACVLANTR_VLANTC                   ETH_MACVLANTR_VLANTC_Msk        /* 12-bit VLAN tag comparison */
#define ETH_MACVLANTR_VLANTI_Pos               (0U)                            
#define ETH_MACVLANTR_VLANTI_Msk               (0xFFFFUL << ETH_MACVLANTR_VLANTI_Pos) /*!< 0x0000FFFF */
#define ETH_MACVLANTR_VLANTI                   ETH_MACVLANTR_VLANTI_Msk        /* VLAN tag identifier (for receive frames) */

/* Bit definition for Ethernet MAC Remote Wake-UpFrame Filter Register */ 
#define ETH_MACRWUFFR_D_Pos                    (0U)                            
#define ETH_MACRWUFFR_D_Msk                    (0xFFFFFFFFUL << ETH_MACRWUFFR_D_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACRWUFFR_D                        ETH_MACRWUFFR_D_Msk             /* Wake-up frame filter register data */
/* Eight sequential Writes to this address (offset 0x28) will write all Wake-UpFrame Filter Registers.
   Eight sequential Reads from this address (offset 0x28) will read all Wake-UpFrame Filter Registers. */
/* Wake-UpFrame Filter Reg0 : Filter 0 Byte Mask
   Wake-UpFrame Filter Reg1 : Filter 1 Byte Mask
   Wake-UpFrame Filter Reg2 : Filter 2 Byte Mask
   Wake-UpFrame Filter Reg3 : Filter 3 Byte Mask
   Wake-UpFrame Filter Reg4 : RSVD - Filter3 Command - RSVD - Filter2 Command - 
                              RSVD - Filter1 Command - RSVD - Filter0 Command
   Wake-UpFrame Filter Re5 : Filter3 Offset - Filter2 Offset - Filter1 Offset - Filter0 Offset
   Wake-UpFrame Filter Re6 : Filter1 CRC16 - Filter0 CRC16
   Wake-UpFrame Filter Re7 : Filter3 CRC16 - Filter2 CRC16 */

/* Bit definition for Ethernet MAC PMT Control and Status Register */ 
#define ETH_MACPMTCSR_WFFRPR_Pos               (31U)                           
#define ETH_MACPMTCSR_WFFRPR_Msk               (0x1UL << ETH_MACPMTCSR_WFFRPR_Pos) /*!< 0x80000000 */
#define ETH_MACPMTCSR_WFFRPR                   ETH_MACPMTCSR_WFFRPR_Msk        /* Wake-Up Frame Filter Register Pointer Reset */
#define ETH_MACPMTCSR_RWKPTR_Pos               (24U)
#define ETH_MACPMTCSR_RWKPTR_Msk               (0x1FUL<<ETH_MACPMTCSR_RWKPTR_Pos)               /*!< 0x1F000000 */
#define ETH_MACPMTCSR_RWKPTR                   ETH_MACPMTCSR_RWKPTR_Msk
#define ETH_MACPMTCSR_RWKPTR_0                 (0x1UL<<ETH_MACPMTCSR_RWKPTR_Pos)                /*!< 0x01000000 */
#define ETH_MACPMTCSR_RWKPTR_1                 (0x2UL<<ETH_MACPMTCSR_RWKPTR_Pos)                /*!< 0x02000000 */
#define ETH_MACPMTCSR_RWKPTR_2                 (0x4UL<<ETH_MACPMTCSR_RWKPTR_Pos)                /*!< 0x04000000 */
#define ETH_MACPMTCSR_RWKPTR_3                 (0x8UL<<ETH_MACPMTCSR_RWKPTR_Pos)                /*!< 0x08000000 */
#define ETH_MACPMTCSR_RWKPTR_4                 (0x10UL<<ETH_MACPMTCSR_RWKPTR_Pos)               /*!< 0x10000000 */
#define ETH_MACPMTCSR_GU_Pos                   (9U)                            
#define ETH_MACPMTCSR_GU_Msk                   (0x1UL << ETH_MACPMTCSR_GU_Pos)  /*!< 0x00000200 */
#define ETH_MACPMTCSR_GU                       ETH_MACPMTCSR_GU_Msk            /* Global Unicast */
#define ETH_MACPMTCSR_WFR_Pos                  (6U)                            
#define ETH_MACPMTCSR_WFR_Msk                  (0x1UL << ETH_MACPMTCSR_WFR_Pos) /*!< 0x00000040 */
#define ETH_MACPMTCSR_WFR                      ETH_MACPMTCSR_WFR_Msk           /* Wake-Up Frame Received */
#define ETH_MACPMTCSR_MPR_Pos                  (5U)                            
#define ETH_MACPMTCSR_MPR_Msk                  (0x1UL << ETH_MACPMTCSR_MPR_Pos) /*!< 0x00000020 */
#define ETH_MACPMTCSR_MPR                      ETH_MACPMTCSR_MPR_Msk           /* Magic Packet Received */
#define ETH_MACPMTCSR_WFE_Pos                  (2U)                            
#define ETH_MACPMTCSR_WFE_Msk                  (0x1UL << ETH_MACPMTCSR_WFE_Pos) /*!< 0x00000004 */
#define ETH_MACPMTCSR_WFE                      ETH_MACPMTCSR_WFE_Msk           /* Wake-Up Frame Enable */
#define ETH_MACPMTCSR_MPE_Pos                  (1U)                            
#define ETH_MACPMTCSR_MPE_Msk                  (0x1UL << ETH_MACPMTCSR_MPE_Pos) /*!< 0x00000002 */
#define ETH_MACPMTCSR_MPE                      ETH_MACPMTCSR_MPE_Msk           /* Magic Packet Enable */
#define ETH_MACPMTCSR_PD_Pos                   (0U)                            
#define ETH_MACPMTCSR_PD_Msk                   (0x1UL << ETH_MACPMTCSR_PD_Pos)  /*!< 0x00000001 */
#define ETH_MACPMTCSR_PD                       ETH_MACPMTCSR_PD_Msk            /* Power Down */

/* Bit definition for Ethernet MAC Status Register */
#define ETH_MACSR_TSTS_Pos                     (9U)                            
#define ETH_MACSR_TSTS_Msk                     (0x1UL << ETH_MACSR_TSTS_Pos)    /*!< 0x00000200 */
#define ETH_MACSR_TSTS                         ETH_MACSR_TSTS_Msk              /* Time stamp trigger status */
#define ETH_MACSR_MMCTS_Pos                    (6U)                            
#define ETH_MACSR_MMCTS_Msk                    (0x1UL << ETH_MACSR_MMCTS_Pos)   /*!< 0x00000040 */
#define ETH_MACSR_MMCTS                        ETH_MACSR_MMCTS_Msk             /* MMC transmit status */
#define ETH_MACSR_MMMCRS_Pos                   (5U)                            
#define ETH_MACSR_MMMCRS_Msk                   (0x1UL << ETH_MACSR_MMMCRS_Pos)  /*!< 0x00000020 */
#define ETH_MACSR_MMMCRS                       ETH_MACSR_MMMCRS_Msk            /* MMC receive status */
#define ETH_MACSR_MMCS_Pos                     (4U)                            
#define ETH_MACSR_MMCS_Msk                     (0x1UL << ETH_MACSR_MMCS_Pos)    /*!< 0x00000010 */
#define ETH_MACSR_MMCS                         ETH_MACSR_MMCS_Msk              /* MMC status */
#define ETH_MACSR_PMTS_Pos                     (3U)                            
#define ETH_MACSR_PMTS_Msk                     (0x1UL << ETH_MACSR_PMTS_Pos)    /*!< 0x00000008 */
#define ETH_MACSR_PMTS                         ETH_MACSR_PMTS_Msk              /* PMT status */

/* Bit definition for Ethernet MAC Interrupt Mask Register */
#define ETH_MACIMR_TSTIM_Pos                   (9U)                            
#define ETH_MACIMR_TSTIM_Msk                   (0x1UL << ETH_MACIMR_TSTIM_Pos)  /*!< 0x00000200 */
#define ETH_MACIMR_TSTIM                       ETH_MACIMR_TSTIM_Msk            /* Time stamp trigger interrupt mask */
#define ETH_MACIMR_PMTIM_Pos                   (3U)                            
#define ETH_MACIMR_PMTIM_Msk                   (0x1UL << ETH_MACIMR_PMTIM_Pos)  /*!< 0x00000008 */
#define ETH_MACIMR_PMTIM                       ETH_MACIMR_PMTIM_Msk            /* PMT interrupt mask */

/* Bit definition for Ethernet MAC Address0 High Register */
#define ETH_MACA0HR_MACA0H_Pos                 (0U)                            
#define ETH_MACA0HR_MACA0H_Msk                 (0xFFFFUL << ETH_MACA0HR_MACA0H_Pos) /*!< 0x0000FFFF */
#define ETH_MACA0HR_MACA0H                     ETH_MACA0HR_MACA0H_Msk          /* MAC address0 high */

/* Bit definition for Ethernet MAC Address0 Low Register */
#define ETH_MACA0LR_MACA0L_Pos                 (0U)                            
#define ETH_MACA0LR_MACA0L_Msk                 (0xFFFFFFFFUL << ETH_MACA0LR_MACA0L_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACA0LR_MACA0L                     ETH_MACA0LR_MACA0L_Msk          /* MAC address0 low */

/* Bit definition for Ethernet MAC Address1 High Register */
#define ETH_MACA1HR_AE_Pos                     (31U)                           
#define ETH_MACA1HR_AE_Msk                     (0x1UL << ETH_MACA1HR_AE_Pos)    /*!< 0x80000000 */
#define ETH_MACA1HR_AE                         ETH_MACA1HR_AE_Msk              /* Address enable */
#define ETH_MACA1HR_SA_Pos                     (30U)                           
#define ETH_MACA1HR_SA_Msk                     (0x1UL << ETH_MACA1HR_SA_Pos)    /*!< 0x40000000 */
#define ETH_MACA1HR_SA                         ETH_MACA1HR_SA_Msk              /* Source address */
#define ETH_MACA1HR_MBC_Pos                    (24U)                           
#define ETH_MACA1HR_MBC_Msk                    (0x3FUL << ETH_MACA1HR_MBC_Pos)  /*!< 0x3F000000 */
#define ETH_MACA1HR_MBC                        ETH_MACA1HR_MBC_Msk             /* Mask byte control: bits to mask for comparison of the MAC Address bytes */
#define ETH_MACA1HR_MBC_HBits15_8              0x20000000U                     /* Mask MAC Address high reg bits [15:8] */
#define ETH_MACA1HR_MBC_HBits7_0               0x10000000U                     /* Mask MAC Address high reg bits [7:0] */
#define ETH_MACA1HR_MBC_LBits31_24             0x08000000U                     /* Mask MAC Address low reg bits [31:24] */
#define ETH_MACA1HR_MBC_LBits23_16             0x04000000U                     /* Mask MAC Address low reg bits [23:16] */
#define ETH_MACA1HR_MBC_LBits15_8              0x02000000U                     /* Mask MAC Address low reg bits [15:8] */
#define ETH_MACA1HR_MBC_LBits7_0               0x01000000U                     /* Mask MAC Address low reg bits [7:0] */ 
#define ETH_MACA1HR_MACA1H_Pos                 (0U)                            
#define ETH_MACA1HR_MACA1H_Msk                 (0xFFFFUL << ETH_MACA1HR_MACA1H_Pos) /*!< 0x0000FFFF */
#define ETH_MACA1HR_MACA1H                     ETH_MACA1HR_MACA1H_Msk          /* MAC address1 high */

/* Bit definition for Ethernet MAC Address1 Low Register */
#define ETH_MACA1LR_MACA1L_Pos                 (0U)                            
#define ETH_MACA1LR_MACA1L_Msk                 (0xFFFFFFFFUL << ETH_MACA1LR_MACA1L_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACA1LR_MACA1L                     ETH_MACA1LR_MACA1L_Msk          /* MAC address1 low */

/* Bit definition for Ethernet MAC Address2 High Register */
#define ETH_MACA2HR_AE_Pos                     (31U)                           
#define ETH_MACA2HR_AE_Msk                     (0x1UL << ETH_MACA2HR_AE_Pos)    /*!< 0x80000000 */
#define ETH_MACA2HR_AE                         ETH_MACA2HR_AE_Msk              /* Address enable */
#define ETH_MACA2HR_SA_Pos                     (30U)                           
#define ETH_MACA2HR_SA_Msk                     (0x1UL << ETH_MACA2HR_SA_Pos)    /*!< 0x40000000 */
#define ETH_MACA2HR_SA                         ETH_MACA2HR_SA_Msk              /* Source address */
#define ETH_MACA2HR_MBC_Pos                    (24U)                           
#define ETH_MACA2HR_MBC_Msk                    (0x3FUL << ETH_MACA2HR_MBC_Pos)  /*!< 0x3F000000 */
#define ETH_MACA2HR_MBC                        ETH_MACA2HR_MBC_Msk             /* Mask byte control */
#define ETH_MACA2HR_MBC_HBits15_8              0x20000000U                     /* Mask MAC Address high reg bits [15:8] */
#define ETH_MACA2HR_MBC_HBits7_0               0x10000000U                     /* Mask MAC Address high reg bits [7:0] */
#define ETH_MACA2HR_MBC_LBits31_24             0x08000000U                     /* Mask MAC Address low reg bits [31:24] */
#define ETH_MACA2HR_MBC_LBits23_16             0x04000000U                     /* Mask MAC Address low reg bits [23:16] */
#define ETH_MACA2HR_MBC_LBits15_8              0x02000000U                     /* Mask MAC Address low reg bits [15:8] */
#define ETH_MACA2HR_MBC_LBits7_0               0x01000000U                     /* Mask MAC Address low reg bits [70] */
#define ETH_MACA2HR_MACA2H_Pos                 (0U)                            
#define ETH_MACA2HR_MACA2H_Msk                 (0xFFFFUL << ETH_MACA2HR_MACA2H_Pos) /*!< 0x0000FFFF */
#define ETH_MACA2HR_MACA2H                     ETH_MACA2HR_MACA2H_Msk          /* MAC address1 high */

/* Bit definition for Ethernet MAC Address2 Low Register */
#define ETH_MACA2LR_MACA2L_Pos                 (0U)                            
#define ETH_MACA2LR_MACA2L_Msk                 (0xFFFFFFFFUL << ETH_MACA2LR_MACA2L_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACA2LR_MACA2L                     ETH_MACA2LR_MACA2L_Msk          /* MAC address2 low */

/* Bit definition for Ethernet MAC Address3 High Register */
#define ETH_MACA3HR_AE_Pos                     (31U)                           
#define ETH_MACA3HR_AE_Msk                     (0x1UL << ETH_MACA3HR_AE_Pos)    /*!< 0x80000000 */
#define ETH_MACA3HR_AE                         ETH_MACA3HR_AE_Msk              /* Address enable */
#define ETH_MACA3HR_SA_Pos                     (30U)                           
#define ETH_MACA3HR_SA_Msk                     (0x1UL << ETH_MACA3HR_SA_Pos)    /*!< 0x40000000 */
#define ETH_MACA3HR_SA                         ETH_MACA3HR_SA_Msk              /* Source address */
#define ETH_MACA3HR_MBC_Pos                    (24U)                           
#define ETH_MACA3HR_MBC_Msk                    (0x3FUL << ETH_MACA3HR_MBC_Pos)  /*!< 0x3F000000 */
#define ETH_MACA3HR_MBC                        ETH_MACA3HR_MBC_Msk             /* Mask byte control */
#define ETH_MACA3HR_MBC_HBits15_8              0x20000000U                     /* Mask MAC Address high reg bits [15:8] */
#define ETH_MACA3HR_MBC_HBits7_0               0x10000000U                     /* Mask MAC Address high reg bits [7:0] */
#define ETH_MACA3HR_MBC_LBits31_24             0x08000000U                     /* Mask MAC Address low reg bits [31:24] */
#define ETH_MACA3HR_MBC_LBits23_16             0x04000000U                     /* Mask MAC Address low reg bits [23:16] */
#define ETH_MACA3HR_MBC_LBits15_8              0x02000000U                     /* Mask MAC Address low reg bits [15:8] */
#define ETH_MACA3HR_MBC_LBits7_0               0x01000000U                     /* Mask MAC Address low reg bits [70] */
#define ETH_MACA3HR_MACA3H_Pos                 (0U)                            
#define ETH_MACA3HR_MACA3H_Msk                 (0xFFFFUL << ETH_MACA3HR_MACA3H_Pos) /*!< 0x0000FFFF */
#define ETH_MACA3HR_MACA3H                     ETH_MACA3HR_MACA3H_Msk          /* MAC address3 high */

/* Bit definition for Ethernet MAC Address3 Low Register */
#define ETH_MACA3LR_MACA3L_Pos                 (0U)                            
#define ETH_MACA3LR_MACA3L_Msk                 (0xFFFFFFFFUL << ETH_MACA3LR_MACA3L_Pos) /*!< 0xFFFFFFFF */
#define ETH_MACA3LR_MACA3L                     ETH_MACA3LR_MACA3L_Msk          /* MAC address3 low */

/******************************************************************************/
/*                Ethernet MMC Registers bits definition                      */
/******************************************************************************/

/* Bit definition for Ethernet MMC Contol Register */
#define ETH_MMCCR_MCF_Pos                      (3U)                            
#define ETH_MMCCR_MCF_Msk                      (0x1UL << ETH_MMCCR_MCF_Pos)     /*!< 0x00000008 */
#define ETH_MMCCR_MCF                          ETH_MMCCR_MCF_Msk               /* MMC Counter Freeze */
#define ETH_MMCCR_ROR_Pos                      (2U)                            
#define ETH_MMCCR_ROR_Msk                      (0x1UL << ETH_MMCCR_ROR_Pos)     /*!< 0x00000004 */
#define ETH_MMCCR_ROR                          ETH_MMCCR_ROR_Msk               /* Reset on Read */
#define ETH_MMCCR_CSR_Pos                      (1U)                            
#define ETH_MMCCR_CSR_Msk                      (0x1UL << ETH_MMCCR_CSR_Pos)     /*!< 0x00000002 */
#define ETH_MMCCR_CSR                          ETH_MMCCR_CSR_Msk               /* Counter Stop Rollover */
#define ETH_MMCCR_CR_Pos                       (0U)                            
#define ETH_MMCCR_CR_Msk                       (0x1UL << ETH_MMCCR_CR_Pos)      /*!< 0x00000001 */
#define ETH_MMCCR_CR                           ETH_MMCCR_CR_Msk                /* Counters Reset */

/* Bit definition for Ethernet MMC Receive Interrupt Register */
#define ETH_MMCRIR_RGUFS_Pos                   (17U)                           
#define ETH_MMCRIR_RGUFS_Msk                   (0x1UL << ETH_MMCRIR_RGUFS_Pos)  /*!< 0x00020000 */
#define ETH_MMCRIR_RGUFS                       ETH_MMCRIR_RGUFS_Msk            /* Set when Rx good unicast frames counter reaches half the maximum value */
#define ETH_MMCRIR_RFAES_Pos                   (6U)                            
#define ETH_MMCRIR_RFAES_Msk                   (0x1UL << ETH_MMCRIR_RFAES_Pos)  /*!< 0x00000040 */
#define ETH_MMCRIR_RFAES                       ETH_MMCRIR_RFAES_Msk            /* Set when Rx alignment error counter reaches half the maximum value */
#define ETH_MMCRIR_RFCES_Pos                   (5U)                            
#define ETH_MMCRIR_RFCES_Msk                   (0x1UL << ETH_MMCRIR_RFCES_Pos)  /*!< 0x00000020 */
#define ETH_MMCRIR_RFCES                       ETH_MMCRIR_RFCES_Msk            /* Set when Rx crc error counter reaches half the maximum value */

/* Bit definition for Ethernet MMC Transmit Interrupt Register */
#define ETH_MMCTIR_TGFS_Pos                    (21U)                           
#define ETH_MMCTIR_TGFS_Msk                    (0x1UL << ETH_MMCTIR_TGFS_Pos)   /*!< 0x00200000 */
#define ETH_MMCTIR_TGFS                        ETH_MMCTIR_TGFS_Msk             /* Set when Tx good frame count counter reaches half the maximum value */
#define ETH_MMCTIR_TGFMSCS_Pos                 (15U)                           
#define ETH_MMCTIR_TGFMSCS_Msk                 (0x1UL << ETH_MMCTIR_TGFMSCS_Pos) /*!< 0x00008000 */
#define ETH_MMCTIR_TGFMSCS                     ETH_MMCTIR_TGFMSCS_Msk          /* Set when Tx good multi col counter reaches half the maximum value */
#define ETH_MMCTIR_TGFSCS_Pos                  (14U)                           
#define ETH_MMCTIR_TGFSCS_Msk                  (0x1UL << ETH_MMCTIR_TGFSCS_Pos) /*!< 0x00004000 */
#define ETH_MMCTIR_TGFSCS                      ETH_MMCTIR_TGFSCS_Msk           /* Set when Tx good single col counter reaches half the maximum value */

/* Bit definition for Ethernet MMC Receive Interrupt Mask Register */
#define ETH_MMCRIMR_RGUFM_Pos                  (17U)                           
#define ETH_MMCRIMR_RGUFM_Msk                  (0x1UL << ETH_MMCRIMR_RGUFM_Pos) /*!< 0x00020000 */
#define ETH_MMCRIMR_RGUFM                      ETH_MMCRIMR_RGUFM_Msk           /* Mask the interrupt when Rx good unicast frames counter reaches half the maximum value */
#define ETH_MMCRIMR_RFAEM_Pos                  (6U)                            
#define ETH_MMCRIMR_RFAEM_Msk                  (0x1UL << ETH_MMCRIMR_RFAEM_Pos) /*!< 0x00000040 */
#define ETH_MMCRIMR_RFAEM                      ETH_MMCRIMR_RFAEM_Msk           /* Mask the interrupt when when Rx alignment error counter reaches half the maximum value */
#define ETH_MMCRIMR_RFCEM_Pos                  (5U)                            
#define ETH_MMCRIMR_RFCEM_Msk                  (0x1UL << ETH_MMCRIMR_RFCEM_Pos) /*!< 0x00000020 */
#define ETH_MMCRIMR_RFCEM                      ETH_MMCRIMR_RFCEM_Msk           /* Mask the interrupt when Rx crc error counter reaches half the maximum value */

/* Bit definition for Ethernet MMC Transmit Interrupt Mask Register */
#define ETH_MMCTIMR_TGFM_Pos                   (21U)                           
#define ETH_MMCTIMR_TGFM_Msk                   (0x1UL << ETH_MMCTIMR_TGFM_Pos)  /*!< 0x00200000 */
#define ETH_MMCTIMR_TGFM                       ETH_MMCTIMR_TGFM_Msk            /* Mask the interrupt when Tx good frame count counter reaches half the maximum value */
#define ETH_MMCTIMR_TGFMSCM_Pos                (15U)                           
#define ETH_MMCTIMR_TGFMSCM_Msk                (0x1UL << ETH_MMCTIMR_TGFMSCM_Pos) /*!< 0x00008000 */
#define ETH_MMCTIMR_TGFMSCM                    ETH_MMCTIMR_TGFMSCM_Msk         /* Mask the interrupt when Tx good multi col counter reaches half the maximum value */
#define ETH_MMCTIMR_TGFSCM_Pos                 (14U)                           
#define ETH_MMCTIMR_TGFSCM_Msk                 (0x1UL << ETH_MMCTIMR_TGFSCM_Pos) /*!< 0x00004000 */
#define ETH_MMCTIMR_TGFSCM                     ETH_MMCTIMR_TGFSCM_Msk          /* Mask the interrupt when Tx good single col counter reaches half the maximum value */

/* Bit definition for Ethernet MMC Transmitted Good Frames after Single Collision Counter Register */
#define ETH_MMCTGFSCCR_TGFSCC_Pos              (0U)                            
#define ETH_MMCTGFSCCR_TGFSCC_Msk              (0xFFFFFFFFUL << ETH_MMCTGFSCCR_TGFSCC_Pos) /*!< 0xFFFFFFFF */
#define ETH_MMCTGFSCCR_TGFSCC                  ETH_MMCTGFSCCR_TGFSCC_Msk       /* Number of successfully transmitted frames after a single collision in Half-duplex mode. */

/* Bit definition for Ethernet MMC Transmitted Good Frames after More than a Single Collision Counter Register */
#define ETH_MMCTGFMSCCR_TGFMSCC_Pos            (0U)                            
#define ETH_MMCTGFMSCCR_TGFMSCC_Msk            (0xFFFFFFFFUL << ETH_MMCTGFMSCCR_TGFMSCC_Pos) /*!< 0xFFFFFFFF */
#define ETH_MMCTGFMSCCR_TGFMSCC                ETH_MMCTGFMSCCR_TGFMSCC_Msk     /* Number of successfully transmitted frames after more than a single collision in Half-duplex mode. */

/* Bit definition for Ethernet MMC Transmitted Good Frames Counter Register */
#define ETH_MMCTGFCR_TGFC_Pos                  (0U)                            
#define ETH_MMCTGFCR_TGFC_Msk                  (0xFFFFFFFFUL << ETH_MMCTGFCR_TGFC_Pos) /*!< 0xFFFFFFFF */
#define ETH_MMCTGFCR_TGFC                      ETH_MMCTGFCR_TGFC_Msk           /* Number of good frames transmitted. */

/* Bit definition for Ethernet MMC Received Frames with CRC Error Counter Register */
#define ETH_MMCRFCECR_RFCEC_Pos                (0U)                            
#define ETH_MMCRFCECR_RFCEC_Msk                (0xFFFFFFFFUL << ETH_MMCRFCECR_RFCEC_Pos) /*!< 0xFFFFFFFF */
#define ETH_MMCRFCECR_RFCEC                    ETH_MMCRFCECR_RFCEC_Msk         /* Number of frames received with CRC error. */

/* Bit definition for Ethernet MMC Received Frames with Alignement Error Counter Register */
#define ETH_MMCRFAECR_RFAEC_Pos                (0U)                            
#define ETH_MMCRFAECR_RFAEC_Msk                (0xFFFFFFFFUL << ETH_MMCRFAECR_RFAEC_Pos) /*!< 0xFFFFFFFF */
#define ETH_MMCRFAECR_RFAEC                    ETH_MMCRFAECR_RFAEC_Msk         /* Number of frames received with alignment (dribble) error */

/* Bit definition for Ethernet MMC Received Good Unicast Frames Counter Register */
#define ETH_MMCRGUFCR_RGUFC_Pos                (0U)                            
#define ETH_MMCRGUFCR_RGUFC_Msk                (0xFFFFFFFFUL << ETH_MMCRGUFCR_RGUFC_Pos) /*!< 0xFFFFFFFF */
#define ETH_MMCRGUFCR_RGUFC                    ETH_MMCRGUFCR_RGUFC_Msk         /* Number of good unicast frames received. */

/******************************************************************************/
/*               Ethernet PTP Registers bits definition                       */
/******************************************************************************/

/* Bit definition for Ethernet PTP Time Stamp Contol Register */
#define ETH_PTPTSCR_TSARU_Pos                  (5U)                            
#define ETH_PTPTSCR_TSARU_Msk                  (0x1UL << ETH_PTPTSCR_TSARU_Pos) /*!< 0x00000020 */
#define ETH_PTPTSCR_TSARU                      ETH_PTPTSCR_TSARU_Msk           /* Addend register update */
#define ETH_PTPTSCR_TSITE_Pos                  (4U)                            
#define ETH_PTPTSCR_TSITE_Msk                  (0x1UL << ETH_PTPTSCR_TSITE_Pos) /*!< 0x00000010 */
#define ETH_PTPTSCR_TSITE                      ETH_PTPTSCR_TSITE_Msk           /* Time stamp interrupt trigger enable */
#define ETH_PTPTSCR_TSSTU_Pos                  (3U)                            
#define ETH_PTPTSCR_TSSTU_Msk                  (0x1UL << ETH_PTPTSCR_TSSTU_Pos) /*!< 0x00000008 */
#define ETH_PTPTSCR_TSSTU                      ETH_PTPTSCR_TSSTU_Msk           /* Time stamp update */
#define ETH_PTPTSCR_TSSTI_Pos                  (2U)                            
#define ETH_PTPTSCR_TSSTI_Msk                  (0x1UL << ETH_PTPTSCR_TSSTI_Pos) /*!< 0x00000004 */
#define ETH_PTPTSCR_TSSTI                      ETH_PTPTSCR_TSSTI_Msk           /* Time stamp initialize */
#define ETH_PTPTSCR_TSFCU_Pos                  (1U)                            
#define ETH_PTPTSCR_TSFCU_Msk                  (0x1UL << ETH_PTPTSCR_TSFCU_Pos) /*!< 0x00000002 */
#define ETH_PTPTSCR_TSFCU                      ETH_PTPTSCR_TSFCU_Msk           /* Time stamp fine or coarse update */
#define ETH_PTPTSCR_TSE_Pos                    (0U)                            
#define ETH_PTPTSCR_TSE_Msk                    (0x1UL << ETH_PTPTSCR_TSE_Pos)   /*!< 0x00000001 */
#define ETH_PTPTSCR_TSE                        ETH_PTPTSCR_TSE_Msk             /* Time stamp enable */

/* Bit definition for Ethernet PTP Sub-Second Increment Register */
#define ETH_PTPSSIR_STSSI_Pos                  (0U)                            
#define ETH_PTPSSIR_STSSI_Msk                  (0xFFUL << ETH_PTPSSIR_STSSI_Pos) /*!< 0x000000FF */
#define ETH_PTPSSIR_STSSI                      ETH_PTPSSIR_STSSI_Msk           /* System time Sub-second increment value */

/* Bit definition for Ethernet PTP Time Stamp High Register */
#define ETH_PTPTSHR_STS_Pos                    (0U)                            
#define ETH_PTPTSHR_STS_Msk                    (0xFFFFFFFFUL << ETH_PTPTSHR_STS_Pos) /*!< 0xFFFFFFFF */
#define ETH_PTPTSHR_STS                        ETH_PTPTSHR_STS_Msk             /* System Time second */

/* Bit definition for Ethernet PTP Time Stamp Low Register */
#define ETH_PTPTSLR_STPNS_Pos                  (31U)                           
#define ETH_PTPTSLR_STPNS_Msk                  (0x1UL << ETH_PTPTSLR_STPNS_Pos) /*!< 0x80000000 */
#define ETH_PTPTSLR_STPNS                      ETH_PTPTSLR_STPNS_Msk           /* System Time Positive or negative time */
#define ETH_PTPTSLR_STSS_Pos                   (0U)                            
#define ETH_PTPTSLR_STSS_Msk                   (0x7FFFFFFFUL << ETH_PTPTSLR_STSS_Pos) /*!< 0x7FFFFFFF */
#define ETH_PTPTSLR_STSS                       ETH_PTPTSLR_STSS_Msk            /* System Time sub-seconds */

/* Bit definition for Ethernet PTP Time Stamp High Update Register */
#define ETH_PTPTSHUR_TSUS_Pos                  (0U)                            
#define ETH_PTPTSHUR_TSUS_Msk                  (0xFFFFFFFFUL << ETH_PTPTSHUR_TSUS_Pos) /*!< 0xFFFFFFFF */
#define ETH_PTPTSHUR_TSUS                      ETH_PTPTSHUR_TSUS_Msk           /* Time stamp update seconds */

/* Bit definition for Ethernet PTP Time Stamp Low Update Register */
#define ETH_PTPTSLUR_TSUPNS_Pos                (31U)                           
#define ETH_PTPTSLUR_TSUPNS_Msk                (0x1UL << ETH_PTPTSLUR_TSUPNS_Pos) /*!< 0x80000000 */
#define ETH_PTPTSLUR_TSUPNS                    ETH_PTPTSLUR_TSUPNS_Msk         /* Time stamp update Positive or negative time */
#define ETH_PTPTSLUR_TSUSS_Pos                 (0U)                            
#define ETH_PTPTSLUR_TSUSS_Msk                 (0x7FFFFFFFUL << ETH_PTPTSLUR_TSUSS_Pos) /*!< 0x7FFFFFFF */
#define ETH_PTPTSLUR_TSUSS                     ETH_PTPTSLUR_TSUSS_Msk          /* Time stamp update sub-seconds */

/* Bit definition for Ethernet PTP Time Stamp Addend Register */
#define ETH_PTPTSAR_TSA_Pos                    (0U)                            
#define ETH_PTPTSAR_TSA_Msk                    (0xFFFFFFFFUL << ETH_PTPTSAR_TSA_Pos) /*!< 0xFFFFFFFF */
#define ETH_PTPTSAR_TSA                        ETH_PTPTSAR_TSA_Msk             /* Time stamp addend */

/* Bit definition for Ethernet PTP Target Time High Register */
#define ETH_PTPTTHR_TTSH_Pos                   (0U)                            
#define ETH_PTPTTHR_TTSH_Msk                   (0xFFFFFFFFUL << ETH_PTPTTHR_TTSH_Pos) /*!< 0xFFFFFFFF */
#define ETH_PTPTTHR_TTSH                       ETH_PTPTTHR_TTSH_Msk            /* Target time stamp high */

/* Bit definition for Ethernet PTP Target Time Low Register */
#define ETH_PTPTTLR_TTSL_Pos                   (0U)                            
#define ETH_PTPTTLR_TTSL_Msk                   (0xFFFFFFFFUL << ETH_PTPTTLR_TTSL_Pos) /*!< 0xFFFFFFFF */
#define ETH_PTPTTLR_TTSL                       ETH_PTPTTLR_TTSL_Msk            /* Target time stamp low */

/******************************************************************************/
/*                 Ethernet DMA Registers bits definition                     */
/******************************************************************************/

/* Bit definition for Ethernet DMA Bus Mode Register */
#define ETH_DMABMR_AAB_Pos                     (25U)                           
#define ETH_DMABMR_AAB_Msk                     (0x1UL << ETH_DMABMR_AAB_Pos)    /*!< 0x02000000 */
#define ETH_DMABMR_AAB                         ETH_DMABMR_AAB_Msk              /* Address-Aligned beats */
#define ETH_DMABMR_FPM_Pos                     (24U)                           
#define ETH_DMABMR_FPM_Msk                     (0x1UL << ETH_DMABMR_FPM_Pos)    /*!< 0x01000000 */
#define ETH_DMABMR_FPM                         ETH_DMABMR_FPM_Msk              /* 4xPBL mode */
#define ETH_DMABMR_USP_Pos                     (23U)                           
#define ETH_DMABMR_USP_Msk                     (0x1UL << ETH_DMABMR_USP_Pos)    /*!< 0x00800000 */
#define ETH_DMABMR_USP                         ETH_DMABMR_USP_Msk              /* Use separate PBL */
#define ETH_DMABMR_RDP_Pos                     (17U)                           
#define ETH_DMABMR_RDP_Msk                     (0x3FUL << ETH_DMABMR_RDP_Pos)   /*!< 0x007E0000 */
#define ETH_DMABMR_RDP                         ETH_DMABMR_RDP_Msk              /* RxDMA PBL */
#define ETH_DMABMR_RDP_1Beat                   0x00020000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 1 */
#define ETH_DMABMR_RDP_2Beat                   0x00040000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 2 */
#define ETH_DMABMR_RDP_4Beat                   0x00080000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 4 */
#define ETH_DMABMR_RDP_8Beat                   0x00100000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 8 */
#define ETH_DMABMR_RDP_16Beat                  0x00200000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 16 */
#define ETH_DMABMR_RDP_32Beat                  0x00400000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 32 */                
#define ETH_DMABMR_RDP_4xPBL_4Beat             0x01020000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 4 */
#define ETH_DMABMR_RDP_4xPBL_8Beat             0x01040000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 8 */
#define ETH_DMABMR_RDP_4xPBL_16Beat            0x01080000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 16 */
#define ETH_DMABMR_RDP_4xPBL_32Beat            0x01100000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 32 */
#define ETH_DMABMR_RDP_4xPBL_64Beat            0x01200000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 64 */
#define ETH_DMABMR_RDP_4xPBL_128Beat           0x01400000U                     /* maximum number of beats to be transferred in one RxDMA transaction is 128 */  
#define ETH_DMABMR_FB_Pos                      (16U)                           
#define ETH_DMABMR_FB_Msk                      (0x1UL << ETH_DMABMR_FB_Pos)     /*!< 0x00010000 */
#define ETH_DMABMR_FB                          ETH_DMABMR_FB_Msk               /* Fixed Burst */
#define ETH_DMABMR_RTPR_Pos                    (14U)                           
#define ETH_DMABMR_RTPR_Msk                    (0x3UL << ETH_DMABMR_RTPR_Pos)   /*!< 0x0000C000 */
#define ETH_DMABMR_RTPR                        ETH_DMABMR_RTPR_Msk             /* Rx Tx priority ratio */
#define ETH_DMABMR_RTPR_1_1                    0x00000000U                     /* Rx Tx priority ratio */
#define ETH_DMABMR_RTPR_2_1                    0x00004000U                     /* Rx Tx priority ratio */
#define ETH_DMABMR_RTPR_3_1                    0x00008000U                     /* Rx Tx priority ratio */
#define ETH_DMABMR_RTPR_4_1                    0x0000C000U                     /* Rx Tx priority ratio */  
#define ETH_DMABMR_PBL_Pos                     (8U)                            
#define ETH_DMABMR_PBL_Msk                     (0x3FUL << ETH_DMABMR_PBL_Pos)   /*!< 0x00003F00 */
#define ETH_DMABMR_PBL                         ETH_DMABMR_PBL_Msk              /* Programmable burst length */
#define ETH_DMABMR_PBL_1Beat                   0x00000100U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 1 */
#define ETH_DMABMR_PBL_2Beat                   0x00000200U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 2 */
#define ETH_DMABMR_PBL_4Beat                   0x00000400U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 4 */
#define ETH_DMABMR_PBL_8Beat                   0x00000800U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 8 */
#define ETH_DMABMR_PBL_16Beat                  0x00001000U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 16 */
#define ETH_DMABMR_PBL_32Beat                  0x00002000U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 32 */                
#define ETH_DMABMR_PBL_4xPBL_4Beat             0x01000100U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 4 */
#define ETH_DMABMR_PBL_4xPBL_8Beat             0x01000200U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 8 */
#define ETH_DMABMR_PBL_4xPBL_16Beat            0x01000400U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 16 */
#define ETH_DMABMR_PBL_4xPBL_32Beat            0x01000800U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 32 */
#define ETH_DMABMR_PBL_4xPBL_64Beat            0x01001000U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 64 */
#define ETH_DMABMR_PBL_4xPBL_128Beat           0x01002000U                     /* maximum number of beats to be transferred in one TxDMA (or both) transaction is 128 */
#define ETH_DMABMR_DSL_Pos                     (2U)                            
#define ETH_DMABMR_DSL_Msk                     (0x1FUL << ETH_DMABMR_DSL_Pos)   /*!< 0x0000007C */
#define ETH_DMABMR_DSL                         ETH_DMABMR_DSL_Msk              /* Descriptor Skip Length */
#define ETH_DMABMR_DA_Pos                      (1U)                            
#define ETH_DMABMR_DA_Msk                      (0x1UL << ETH_DMABMR_DA_Pos)     /*!< 0x00000002 */
#define ETH_DMABMR_DA                          ETH_DMABMR_DA_Msk               /* DMA arbitration scheme */
#define ETH_DMABMR_SR_Pos                      (0U)                            
#define ETH_DMABMR_SR_Msk                      (0x1UL << ETH_DMABMR_SR_Pos)     /*!< 0x00000001 */
#define ETH_DMABMR_SR                          ETH_DMABMR_SR_Msk               /* Software reset */

/* Bit definition for Ethernet DMA Transmit Poll Demand Register */
#define ETH_DMATPDR_TPD_Pos                    (0U)                            
#define ETH_DMATPDR_TPD_Msk                    (0xFFFFFFFFUL << ETH_DMATPDR_TPD_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMATPDR_TPD                        ETH_DMATPDR_TPD_Msk             /* Transmit poll demand */

/* Bit definition for Ethernet DMA Receive Poll Demand Register */
#define ETH_DMARPDR_RPD_Pos                    (0U)                            
#define ETH_DMARPDR_RPD_Msk                    (0xFFFFFFFFUL << ETH_DMARPDR_RPD_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMARPDR_RPD                        ETH_DMARPDR_RPD_Msk             /* Receive poll demand  */

/* Bit definition for Ethernet DMA Receive Descriptor List Address Register */
#define ETH_DMARDLAR_SRL_Pos                   (0U)                            
#define ETH_DMARDLAR_SRL_Msk                   (0xFFFFFFFFUL << ETH_DMARDLAR_SRL_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMARDLAR_SRL                       ETH_DMARDLAR_SRL_Msk            /* Start of receive list */

/* Bit definition for Ethernet DMA Transmit Descriptor List Address Register */
#define ETH_DMATDLAR_STL_Pos                   (0U)                            
#define ETH_DMATDLAR_STL_Msk                   (0xFFFFFFFFUL << ETH_DMATDLAR_STL_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMATDLAR_STL                       ETH_DMATDLAR_STL_Msk            /* Start of transmit list */

/* Bit definition for Ethernet DMA Status Register */
#define ETH_DMASR_TSTS_Pos                     (29U)                           
#define ETH_DMASR_TSTS_Msk                     (0x1UL << ETH_DMASR_TSTS_Pos)    /*!< 0x20000000 */
#define ETH_DMASR_TSTS                         ETH_DMASR_TSTS_Msk              /* Time-stamp trigger status */
#define ETH_DMASR_PMTS_Pos                     (28U)                           
#define ETH_DMASR_PMTS_Msk                     (0x1UL << ETH_DMASR_PMTS_Pos)    /*!< 0x10000000 */
#define ETH_DMASR_PMTS                         ETH_DMASR_PMTS_Msk              /* PMT status */
#define ETH_DMASR_MMCS_Pos                     (27U)                           
#define ETH_DMASR_MMCS_Msk                     (0x1UL << ETH_DMASR_MMCS_Pos)    /*!< 0x08000000 */
#define ETH_DMASR_MMCS                         ETH_DMASR_MMCS_Msk              /* MMC status */
#define ETH_DMASR_EBS_Pos                      (23U)                           
#define ETH_DMASR_EBS_Msk                      (0x7UL << ETH_DMASR_EBS_Pos)     /*!< 0x03800000 */
#define ETH_DMASR_EBS                          ETH_DMASR_EBS_Msk               /* Error bits status */
  /* combination with EBS[2:0] for GetFlagStatus function */
#define ETH_DMASR_EBS_DescAccess_Pos           (25U)                           
#define ETH_DMASR_EBS_DescAccess_Msk           (0x1UL << ETH_DMASR_EBS_DescAccess_Pos) /*!< 0x02000000 */
#define ETH_DMASR_EBS_DescAccess               ETH_DMASR_EBS_DescAccess_Msk    /* Error bits 0-data buffer, 1-desc. access */
#define ETH_DMASR_EBS_ReadTransf_Pos           (24U)                           
#define ETH_DMASR_EBS_ReadTransf_Msk           (0x1UL << ETH_DMASR_EBS_ReadTransf_Pos) /*!< 0x01000000 */
#define ETH_DMASR_EBS_ReadTransf               ETH_DMASR_EBS_ReadTransf_Msk    /* Error bits 0-write trnsf, 1-read transfr */
#define ETH_DMASR_EBS_DataTransfTx_Pos         (23U)                           
#define ETH_DMASR_EBS_DataTransfTx_Msk         (0x1UL << ETH_DMASR_EBS_DataTransfTx_Pos) /*!< 0x00800000 */
#define ETH_DMASR_EBS_DataTransfTx             ETH_DMASR_EBS_DataTransfTx_Msk  /* Error bits 0-Rx DMA, 1-Tx DMA */
#define ETH_DMASR_TPS_Pos                      (20U)                           
#define ETH_DMASR_TPS_Msk                      (0x7UL << ETH_DMASR_TPS_Pos)     /*!< 0x00700000 */
#define ETH_DMASR_TPS                          ETH_DMASR_TPS_Msk               /* Transmit process state */
#define ETH_DMASR_TPS_Stopped                  0x00000000U                     /* Stopped - Reset or Stop Tx Command issued  */
#define ETH_DMASR_TPS_Fetching_Pos             (20U)                           
#define ETH_DMASR_TPS_Fetching_Msk             (0x1UL << ETH_DMASR_TPS_Fetching_Pos) /*!< 0x00100000 */
#define ETH_DMASR_TPS_Fetching                 ETH_DMASR_TPS_Fetching_Msk      /* Running - fetching the Tx descriptor */
#define ETH_DMASR_TPS_Waiting_Pos              (21U)                           
#define ETH_DMASR_TPS_Waiting_Msk              (0x1UL << ETH_DMASR_TPS_Waiting_Pos) /*!< 0x00200000 */
#define ETH_DMASR_TPS_Waiting                  ETH_DMASR_TPS_Waiting_Msk       /* Running - waiting for status */
#define ETH_DMASR_TPS_Reading_Pos              (20U)                           
#define ETH_DMASR_TPS_Reading_Msk              (0x3UL << ETH_DMASR_TPS_Reading_Pos) /*!< 0x00300000 */
#define ETH_DMASR_TPS_Reading                  ETH_DMASR_TPS_Reading_Msk       /* Running - reading the data from host memory */
#define ETH_DMASR_TPS_Suspended_Pos            (21U)                           
#define ETH_DMASR_TPS_Suspended_Msk            (0x3UL << ETH_DMASR_TPS_Suspended_Pos) /*!< 0x00600000 */
#define ETH_DMASR_TPS_Suspended                ETH_DMASR_TPS_Suspended_Msk     /* Suspended - Tx Descriptor unavailabe */
#define ETH_DMASR_TPS_Closing_Pos              (20U)                           
#define ETH_DMASR_TPS_Closing_Msk              (0x7UL << ETH_DMASR_TPS_Closing_Pos) /*!< 0x00700000 */
#define ETH_DMASR_TPS_Closing                  ETH_DMASR_TPS_Closing_Msk       /* Running - closing Rx descriptor */
#define ETH_DMASR_RPS_Pos                      (17U)                           
#define ETH_DMASR_RPS_Msk                      (0x7UL << ETH_DMASR_RPS_Pos)     /*!< 0x000E0000 */
#define ETH_DMASR_RPS                          ETH_DMASR_RPS_Msk               /* Receive process state */
#define ETH_DMASR_RPS_Stopped                  0x00000000U                     /* Stopped - Reset or Stop Rx Command issued */
#define ETH_DMASR_RPS_Fetching_Pos             (17U)                           
#define ETH_DMASR_RPS_Fetching_Msk             (0x1UL << ETH_DMASR_RPS_Fetching_Pos) /*!< 0x00020000 */
#define ETH_DMASR_RPS_Fetching                 ETH_DMASR_RPS_Fetching_Msk      /* Running - fetching the Rx descriptor */
#define ETH_DMASR_RPS_Waiting_Pos              (17U)                           
#define ETH_DMASR_RPS_Waiting_Msk              (0x3UL << ETH_DMASR_RPS_Waiting_Pos) /*!< 0x00060000 */
#define ETH_DMASR_RPS_Waiting                  ETH_DMASR_RPS_Waiting_Msk       /* Running - waiting for packet */
#define ETH_DMASR_RPS_Suspended_Pos            (19U)                           
#define ETH_DMASR_RPS_Suspended_Msk            (0x1UL << ETH_DMASR_RPS_Suspended_Pos) /*!< 0x00080000 */
#define ETH_DMASR_RPS_Suspended                ETH_DMASR_RPS_Suspended_Msk     /* Suspended - Rx Descriptor unavailable */
#define ETH_DMASR_RPS_Closing_Pos              (17U)                           
#define ETH_DMASR_RPS_Closing_Msk              (0x5UL << ETH_DMASR_RPS_Closing_Pos) /*!< 0x000A0000 */
#define ETH_DMASR_RPS_Closing                  ETH_DMASR_RPS_Closing_Msk       /* Running - closing descriptor */
#define ETH_DMASR_RPS_Queuing_Pos              (17U)                           
#define ETH_DMASR_RPS_Queuing_Msk              (0x7UL << ETH_DMASR_RPS_Queuing_Pos) /*!< 0x000E0000 */
#define ETH_DMASR_RPS_Queuing                  ETH_DMASR_RPS_Queuing_Msk       /* Running - queuing the recieve frame into host memory */
#define ETH_DMASR_NIS_Pos                      (16U)                           
#define ETH_DMASR_NIS_Msk                      (0x1UL << ETH_DMASR_NIS_Pos)     /*!< 0x00010000 */
#define ETH_DMASR_NIS                          ETH_DMASR_NIS_Msk               /* Normal interrupt summary */
#define ETH_DMASR_AIS_Pos                      (15U)                           
#define ETH_DMASR_AIS_Msk                      (0x1UL << ETH_DMASR_AIS_Pos)     /*!< 0x00008000 */
#define ETH_DMASR_AIS                          ETH_DMASR_AIS_Msk               /* Abnormal interrupt summary */
#define ETH_DMASR_ERS_Pos                      (14U)                           
#define ETH_DMASR_ERS_Msk                      (0x1UL << ETH_DMASR_ERS_Pos)     /*!< 0x00004000 */
#define ETH_DMASR_ERS                          ETH_DMASR_ERS_Msk               /* Early receive status */
#define ETH_DMASR_FBES_Pos                     (13U)                           
#define ETH_DMASR_FBES_Msk                     (0x1UL << ETH_DMASR_FBES_Pos)    /*!< 0x00002000 */
#define ETH_DMASR_FBES                         ETH_DMASR_FBES_Msk              /* Fatal bus error status */
#define ETH_DMASR_ETS_Pos                      (10U)                           
#define ETH_DMASR_ETS_Msk                      (0x1UL << ETH_DMASR_ETS_Pos)     /*!< 0x00000400 */
#define ETH_DMASR_ETS                          ETH_DMASR_ETS_Msk               /* Early transmit status */
#define ETH_DMASR_RWTS_Pos                     (9U)                            
#define ETH_DMASR_RWTS_Msk                     (0x1UL << ETH_DMASR_RWTS_Pos)    /*!< 0x00000200 */
#define ETH_DMASR_RWTS                         ETH_DMASR_RWTS_Msk              /* Receive watchdog timeout status */
#define ETH_DMASR_RPSS_Pos                     (8U)                            
#define ETH_DMASR_RPSS_Msk                     (0x1UL << ETH_DMASR_RPSS_Pos)    /*!< 0x00000100 */
#define ETH_DMASR_RPSS                         ETH_DMASR_RPSS_Msk              /* Receive process stopped status */
#define ETH_DMASR_RBUS_Pos                     (7U)                            
#define ETH_DMASR_RBUS_Msk                     (0x1UL << ETH_DMASR_RBUS_Pos)    /*!< 0x00000080 */
#define ETH_DMASR_RBUS                         ETH_DMASR_RBUS_Msk              /* Receive buffer unavailable status */
#define ETH_DMASR_RS_Pos                       (6U)                            
#define ETH_DMASR_RS_Msk                       (0x1UL << ETH_DMASR_RS_Pos)      /*!< 0x00000040 */
#define ETH_DMASR_RS                           ETH_DMASR_RS_Msk                /* Receive status */
#define ETH_DMASR_TUS_Pos                      (5U)                            
#define ETH_DMASR_TUS_Msk                      (0x1UL << ETH_DMASR_TUS_Pos)     /*!< 0x00000020 */
#define ETH_DMASR_TUS                          ETH_DMASR_TUS_Msk               /* Transmit underflow status */
#define ETH_DMASR_ROS_Pos                      (4U)                            
#define ETH_DMASR_ROS_Msk                      (0x1UL << ETH_DMASR_ROS_Pos)     /*!< 0x00000010 */
#define ETH_DMASR_ROS                          ETH_DMASR_ROS_Msk               /* Receive overflow status */
#define ETH_DMASR_TJTS_Pos                     (3U)                            
#define ETH_DMASR_TJTS_Msk                     (0x1UL << ETH_DMASR_TJTS_Pos)    /*!< 0x00000008 */
#define ETH_DMASR_TJTS                         ETH_DMASR_TJTS_Msk              /* Transmit jabber timeout status */
#define ETH_DMASR_TBUS_Pos                     (2U)                            
#define ETH_DMASR_TBUS_Msk                     (0x1UL << ETH_DMASR_TBUS_Pos)    /*!< 0x00000004 */
#define ETH_DMASR_TBUS                         ETH_DMASR_TBUS_Msk              /* Transmit buffer unavailable status */
#define ETH_DMASR_TPSS_Pos                     (1U)                            
#define ETH_DMASR_TPSS_Msk                     (0x1UL << ETH_DMASR_TPSS_Pos)    /*!< 0x00000002 */
#define ETH_DMASR_TPSS                         ETH_DMASR_TPSS_Msk              /* Transmit process stopped status */
#define ETH_DMASR_TS_Pos                       (0U)                            
#define ETH_DMASR_TS_Msk                       (0x1UL << ETH_DMASR_TS_Pos)      /*!< 0x00000001 */
#define ETH_DMASR_TS                           ETH_DMASR_TS_Msk                /* Transmit status */

/* Bit definition for Ethernet DMA Operation Mode Register */
#define ETH_DMAOMR_DTCEFD_Pos                  (26U)                           
#define ETH_DMAOMR_DTCEFD_Msk                  (0x1UL << ETH_DMAOMR_DTCEFD_Pos) /*!< 0x04000000 */
#define ETH_DMAOMR_DTCEFD                      ETH_DMAOMR_DTCEFD_Msk           /* Disable Dropping of TCP/IP checksum error frames */
#define ETH_DMAOMR_RSF_Pos                     (25U)                           
#define ETH_DMAOMR_RSF_Msk                     (0x1UL << ETH_DMAOMR_RSF_Pos)    /*!< 0x02000000 */
#define ETH_DMAOMR_RSF                         ETH_DMAOMR_RSF_Msk              /* Receive store and forward */
#define ETH_DMAOMR_DFRF_Pos                    (24U)                           
#define ETH_DMAOMR_DFRF_Msk                    (0x1UL << ETH_DMAOMR_DFRF_Pos)   /*!< 0x01000000 */
#define ETH_DMAOMR_DFRF                        ETH_DMAOMR_DFRF_Msk             /* Disable flushing of received frames */
#define ETH_DMAOMR_TSF_Pos                     (21U)                           
#define ETH_DMAOMR_TSF_Msk                     (0x1UL << ETH_DMAOMR_TSF_Pos)    /*!< 0x00200000 */
#define ETH_DMAOMR_TSF                         ETH_DMAOMR_TSF_Msk              /* Transmit store and forward */
#define ETH_DMAOMR_FTF_Pos                     (20U)                           
#define ETH_DMAOMR_FTF_Msk                     (0x1UL << ETH_DMAOMR_FTF_Pos)    /*!< 0x00100000 */
#define ETH_DMAOMR_FTF                         ETH_DMAOMR_FTF_Msk              /* Flush transmit FIFO */
#define ETH_DMAOMR_TTC_Pos                     (14U)                           
#define ETH_DMAOMR_TTC_Msk                     (0x7UL << ETH_DMAOMR_TTC_Pos)    /*!< 0x0001C000 */
#define ETH_DMAOMR_TTC                         ETH_DMAOMR_TTC_Msk              /* Transmit threshold control */
#define ETH_DMAOMR_TTC_64Bytes                 0x00000000U                     /* threshold level of the MTL Transmit FIFO is 64 Bytes */
#define ETH_DMAOMR_TTC_128Bytes                0x00004000U                     /* threshold level of the MTL Transmit FIFO is 128 Bytes */
#define ETH_DMAOMR_TTC_192Bytes                0x00008000U                     /* threshold level of the MTL Transmit FIFO is 192 Bytes */
#define ETH_DMAOMR_TTC_256Bytes                0x0000C000U                     /* threshold level of the MTL Transmit FIFO is 256 Bytes */
#define ETH_DMAOMR_TTC_40Bytes                 0x00010000U                     /* threshold level of the MTL Transmit FIFO is 40 Bytes */
#define ETH_DMAOMR_TTC_32Bytes                 0x00014000U                     /* threshold level of the MTL Transmit FIFO is 32 Bytes */
#define ETH_DMAOMR_TTC_24Bytes                 0x00018000U                     /* threshold level of the MTL Transmit FIFO is 24 Bytes */
#define ETH_DMAOMR_TTC_16Bytes                 0x0001C000U                     /* threshold level of the MTL Transmit FIFO is 16 Bytes */
#define ETH_DMAOMR_ST_Pos                      (13U)                           
#define ETH_DMAOMR_ST_Msk                      (0x1UL << ETH_DMAOMR_ST_Pos)     /*!< 0x00002000 */
#define ETH_DMAOMR_ST                          ETH_DMAOMR_ST_Msk               /* Start/stop transmission command */
#define ETH_DMAOMR_FEF_Pos                     (7U)                            
#define ETH_DMAOMR_FEF_Msk                     (0x1UL << ETH_DMAOMR_FEF_Pos)    /*!< 0x00000080 */
#define ETH_DMAOMR_FEF                         ETH_DMAOMR_FEF_Msk              /* Forward error frames */
#define ETH_DMAOMR_FUGF_Pos                    (6U)                            
#define ETH_DMAOMR_FUGF_Msk                    (0x1UL << ETH_DMAOMR_FUGF_Pos)   /*!< 0x00000040 */
#define ETH_DMAOMR_FUGF                        ETH_DMAOMR_FUGF_Msk             /* Forward undersized good frames */
#define ETH_DMAOMR_RTC_Pos                     (3U)                            
#define ETH_DMAOMR_RTC_Msk                     (0x3UL << ETH_DMAOMR_RTC_Pos)    /*!< 0x00000018 */
#define ETH_DMAOMR_RTC                         ETH_DMAOMR_RTC_Msk              /* receive threshold control */
#define ETH_DMAOMR_RTC_64Bytes                 0x00000000U                     /* threshold level of the MTL Receive FIFO is 64 Bytes */
#define ETH_DMAOMR_RTC_32Bytes                 0x00000008U                     /* threshold level of the MTL Receive FIFO is 32 Bytes */
#define ETH_DMAOMR_RTC_96Bytes                 0x00000010U                     /* threshold level of the MTL Receive FIFO is 96 Bytes */
#define ETH_DMAOMR_RTC_128Bytes                0x00000018U                     /* threshold level of the MTL Receive FIFO is 128 Bytes */
#define ETH_DMAOMR_OSF_Pos                     (2U)                            
#define ETH_DMAOMR_OSF_Msk                     (0x1UL << ETH_DMAOMR_OSF_Pos)    /*!< 0x00000004 */
#define ETH_DMAOMR_OSF                         ETH_DMAOMR_OSF_Msk              /* operate on second frame */
#define ETH_DMAOMR_SR_Pos                      (1U)                            
#define ETH_DMAOMR_SR_Msk                      (0x1UL << ETH_DMAOMR_SR_Pos)     /*!< 0x00000002 */
#define ETH_DMAOMR_SR                          ETH_DMAOMR_SR_Msk               /* Start/stop receive */

/* Bit definition for Ethernet DMA Interrupt Enable Register */
#define ETH_DMAIER_NISE_Pos                    (16U)                           
#define ETH_DMAIER_NISE_Msk                    (0x1UL << ETH_DMAIER_NISE_Pos)   /*!< 0x00010000 */
#define ETH_DMAIER_NISE                        ETH_DMAIER_NISE_Msk             /* Normal interrupt summary enable */
#define ETH_DMAIER_AISE_Pos                    (15U)                           
#define ETH_DMAIER_AISE_Msk                    (0x1UL << ETH_DMAIER_AISE_Pos)   /*!< 0x00008000 */
#define ETH_DMAIER_AISE                        ETH_DMAIER_AISE_Msk             /* Abnormal interrupt summary enable */
#define ETH_DMAIER_ERIE_Pos                    (14U)                           
#define ETH_DMAIER_ERIE_Msk                    (0x1UL << ETH_DMAIER_ERIE_Pos)   /*!< 0x00004000 */
#define ETH_DMAIER_ERIE                        ETH_DMAIER_ERIE_Msk             /* Early receive interrupt enable */
#define ETH_DMAIER_FBEIE_Pos                   (13U)                           
#define ETH_DMAIER_FBEIE_Msk                   (0x1UL << ETH_DMAIER_FBEIE_Pos)  /*!< 0x00002000 */
#define ETH_DMAIER_FBEIE                       ETH_DMAIER_FBEIE_Msk            /* Fatal bus error interrupt enable */
#define ETH_DMAIER_ETIE_Pos                    (10U)                           
#define ETH_DMAIER_ETIE_Msk                    (0x1UL << ETH_DMAIER_ETIE_Pos)   /*!< 0x00000400 */
#define ETH_DMAIER_ETIE                        ETH_DMAIER_ETIE_Msk             /* Early transmit interrupt enable */
#define ETH_DMAIER_RWTIE_Pos                   (9U)                            
#define ETH_DMAIER_RWTIE_Msk                   (0x1UL << ETH_DMAIER_RWTIE_Pos)  /*!< 0x00000200 */
#define ETH_DMAIER_RWTIE                       ETH_DMAIER_RWTIE_Msk            /* Receive watchdog timeout interrupt enable */
#define ETH_DMAIER_RPSIE_Pos                   (8U)                            
#define ETH_DMAIER_RPSIE_Msk                   (0x1UL << ETH_DMAIER_RPSIE_Pos)  /*!< 0x00000100 */
#define ETH_DMAIER_RPSIE                       ETH_DMAIER_RPSIE_Msk            /* Receive process stopped interrupt enable */
#define ETH_DMAIER_RBUIE_Pos                   (7U)                            
#define ETH_DMAIER_RBUIE_Msk                   (0x1UL << ETH_DMAIER_RBUIE_Pos)  /*!< 0x00000080 */
#define ETH_DMAIER_RBUIE                       ETH_DMAIER_RBUIE_Msk            /* Receive buffer unavailable interrupt enable */
#define ETH_DMAIER_RIE_Pos                     (6U)                            
#define ETH_DMAIER_RIE_Msk                     (0x1UL << ETH_DMAIER_RIE_Pos)    /*!< 0x00000040 */
#define ETH_DMAIER_RIE                         ETH_DMAIER_RIE_Msk              /* Receive interrupt enable */
#define ETH_DMAIER_TUIE_Pos                    (5U)                            
#define ETH_DMAIER_TUIE_Msk                    (0x1UL << ETH_DMAIER_TUIE_Pos)   /*!< 0x00000020 */
#define ETH_DMAIER_TUIE                        ETH_DMAIER_TUIE_Msk             /* Transmit Underflow interrupt enable */
#define ETH_DMAIER_ROIE_Pos                    (4U)                            
#define ETH_DMAIER_ROIE_Msk                    (0x1UL << ETH_DMAIER_ROIE_Pos)   /*!< 0x00000010 */
#define ETH_DMAIER_ROIE                        ETH_DMAIER_ROIE_Msk             /* Receive Overflow interrupt enable */
#define ETH_DMAIER_TJTIE_Pos                   (3U)                            
#define ETH_DMAIER_TJTIE_Msk                   (0x1UL << ETH_DMAIER_TJTIE_Pos)  /*!< 0x00000008 */
#define ETH_DMAIER_TJTIE                       ETH_DMAIER_TJTIE_Msk            /* Transmit jabber timeout interrupt enable */
#define ETH_DMAIER_TBUIE_Pos                   (2U)                            
#define ETH_DMAIER_TBUIE_Msk                   (0x1UL << ETH_DMAIER_TBUIE_Pos)  /*!< 0x00000004 */
#define ETH_DMAIER_TBUIE                       ETH_DMAIER_TBUIE_Msk            /* Transmit buffer unavailable interrupt enable */
#define ETH_DMAIER_TPSIE_Pos                   (1U)                            
#define ETH_DMAIER_TPSIE_Msk                   (0x1UL << ETH_DMAIER_TPSIE_Pos)  /*!< 0x00000002 */
#define ETH_DMAIER_TPSIE                       ETH_DMAIER_TPSIE_Msk            /* Transmit process stopped interrupt enable */
#define ETH_DMAIER_TIE_Pos                     (0U)                            
#define ETH_DMAIER_TIE_Msk                     (0x1UL << ETH_DMAIER_TIE_Pos)    /*!< 0x00000001 */
#define ETH_DMAIER_TIE                         ETH_DMAIER_TIE_Msk              /* Transmit interrupt enable */

/* Bit definition for Ethernet DMA Missed Frame and Buffer Overflow Counter Register */
#define ETH_DMAMFBOCR_OFOC_Pos                 (28U)                           
#define ETH_DMAMFBOCR_OFOC_Msk                 (0x1UL << ETH_DMAMFBOCR_OFOC_Pos) /*!< 0x10000000 */
#define ETH_DMAMFBOCR_OFOC                     ETH_DMAMFBOCR_OFOC_Msk          /* Overflow bit for FIFO overflow counter */
#define ETH_DMAMFBOCR_MFA_Pos                  (17U)                           
#define ETH_DMAMFBOCR_MFA_Msk                  (0x7FFUL << ETH_DMAMFBOCR_MFA_Pos) /*!< 0x0FFE0000 */
#define ETH_DMAMFBOCR_MFA                      ETH_DMAMFBOCR_MFA_Msk           /* Number of frames missed by the application */
#define ETH_DMAMFBOCR_OMFC_Pos                 (16U)                           
#define ETH_DMAMFBOCR_OMFC_Msk                 (0x1UL << ETH_DMAMFBOCR_OMFC_Pos) /*!< 0x00010000 */
#define ETH_DMAMFBOCR_OMFC                     ETH_DMAMFBOCR_OMFC_Msk          /* Overflow bit for missed frame counter */
#define ETH_DMAMFBOCR_MFC_Pos                  (0U)                            
#define ETH_DMAMFBOCR_MFC_Msk                  (0xFFFFUL << ETH_DMAMFBOCR_MFC_Pos) /*!< 0x0000FFFF */
#define ETH_DMAMFBOCR_MFC                      ETH_DMAMFBOCR_MFC_Msk           /* Number of frames missed by the controller */

/********************************* Bit definition for ETH_DMARIWT register ******************************************/
#define ETH_DMARIWT_RIWT_Pos                   (0U)
#define ETH_DMARIWT_RIWT_Msk                   (0xFFUL<<ETH_DMARIWT_RIWT_Pos)                    /*!< 0x000000FF */
#define ETH_DMARIWT_RIWT                       ETH_DMARIWT_RIWT_Msk

/* Bit definition for Ethernet DMA Current Host Transmit Descriptor Register */
#define ETH_DMACHTDR_HTDAP_Pos                 (0U)                            
#define ETH_DMACHTDR_HTDAP_Msk                 (0xFFFFFFFFUL << ETH_DMACHTDR_HTDAP_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMACHTDR_HTDAP                     ETH_DMACHTDR_HTDAP_Msk          /* Host transmit descriptor address pointer */

/* Bit definition for Ethernet DMA Current Host Receive Descriptor Register */
#define ETH_DMACHRDR_HRDAP_Pos                 (0U)                            
#define ETH_DMACHRDR_HRDAP_Msk                 (0xFFFFFFFFUL << ETH_DMACHRDR_HRDAP_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMACHRDR_HRDAP                     ETH_DMACHRDR_HRDAP_Msk          /* Host receive descriptor address pointer */

/* Bit definition for Ethernet DMA Current Host Transmit Buffer Address Register */
#define ETH_DMACHTBAR_HTBAP_Pos                (0U)                            
#define ETH_DMACHTBAR_HTBAP_Msk                (0xFFFFFFFFUL << ETH_DMACHTBAR_HTBAP_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMACHTBAR_HTBAP                    ETH_DMACHTBAR_HTBAP_Msk         /* Host transmit buffer address pointer */

/* Bit definition for Ethernet DMA Current Host Receive Buffer Address Register */
#define ETH_DMACHRBAR_HRBAP_Pos                (0U)                            
#define ETH_DMACHRBAR_HRBAP_Msk                (0xFFFFFFFFUL << ETH_DMACHRBAR_HRBAP_Pos) /*!< 0xFFFFFFFF */
#define ETH_DMACHRBAR_HRBAP                    ETH_DMACHRBAR_HRBAP_Msk         /* Host receive buffer address pointer */


/********************************************************************************************************************/
/********************************* EXTI *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for EXTI_IMR1 register ********************************************/
#define EXTI_IMR1_IMR0_Pos                        (0U)
#define EXTI_IMR1_IMR0_Msk                        (0x1UL<<EXTI_IMR1_IMR0_Pos)                       /*!< 0x00000001 */
#define EXTI_IMR1_IMR0                            EXTI_IMR1_IMR0_Msk                                
#define EXTI_IMR1_IMR1_Pos                        (1U)
#define EXTI_IMR1_IMR1_Msk                        (0x1UL<<EXTI_IMR1_IMR1_Pos)                       /*!< 0x00000002 */
#define EXTI_IMR1_IMR1                            EXTI_IMR1_IMR1_Msk                                
#define EXTI_IMR1_IMR2_Pos                        (2U)
#define EXTI_IMR1_IMR2_Msk                        (0x1UL<<EXTI_IMR1_IMR2_Pos)                       /*!< 0x00000004 */
#define EXTI_IMR1_IMR2                            EXTI_IMR1_IMR2_Msk                                
#define EXTI_IMR1_IMR3_Pos                        (3U)
#define EXTI_IMR1_IMR3_Msk                        (0x1UL<<EXTI_IMR1_IMR3_Pos)                       /*!< 0x00000008 */
#define EXTI_IMR1_IMR3                            EXTI_IMR1_IMR3_Msk                                
#define EXTI_IMR1_IMR4_Pos                        (4U)
#define EXTI_IMR1_IMR4_Msk                        (0x1UL<<EXTI_IMR1_IMR4_Pos)                       /*!< 0x00000010 */
#define EXTI_IMR1_IMR4                            EXTI_IMR1_IMR4_Msk                                
#define EXTI_IMR1_IMR5_Pos                        (5U)
#define EXTI_IMR1_IMR5_Msk                        (0x1UL<<EXTI_IMR1_IMR5_Pos)                       /*!< 0x00000020 */
#define EXTI_IMR1_IMR5                            EXTI_IMR1_IMR5_Msk                                
#define EXTI_IMR1_IMR6_Pos                        (6U)
#define EXTI_IMR1_IMR6_Msk                        (0x1UL<<EXTI_IMR1_IMR6_Pos)                       /*!< 0x00000040 */
#define EXTI_IMR1_IMR6                            EXTI_IMR1_IMR6_Msk                                
#define EXTI_IMR1_IMR7_Pos                        (7U)
#define EXTI_IMR1_IMR7_Msk                        (0x1UL<<EXTI_IMR1_IMR7_Pos)                       /*!< 0x00000080 */
#define EXTI_IMR1_IMR7                            EXTI_IMR1_IMR7_Msk                                
#define EXTI_IMR1_IMR8_Pos                        (8U)
#define EXTI_IMR1_IMR8_Msk                        (0x1UL<<EXTI_IMR1_IMR8_Pos)                       /*!< 0x00000100 */
#define EXTI_IMR1_IMR8                            EXTI_IMR1_IMR8_Msk                                
#define EXTI_IMR1_IMR9_Pos                        (9U)
#define EXTI_IMR1_IMR9_Msk                        (0x1UL<<EXTI_IMR1_IMR9_Pos)                       /*!< 0x00000200 */
#define EXTI_IMR1_IMR9                            EXTI_IMR1_IMR9_Msk                                
#define EXTI_IMR1_IMR10_Pos                       (10U)
#define EXTI_IMR1_IMR10_Msk                       (0x1UL<<EXTI_IMR1_IMR10_Pos)                      /*!< 0x00000400 */
#define EXTI_IMR1_IMR10                           EXTI_IMR1_IMR10_Msk                               
#define EXTI_IMR1_IMR11_Pos                       (11U)
#define EXTI_IMR1_IMR11_Msk                       (0x1UL<<EXTI_IMR1_IMR11_Pos)                      /*!< 0x00000800 */
#define EXTI_IMR1_IMR11                           EXTI_IMR1_IMR11_Msk                               
#define EXTI_IMR1_IMR12_Pos                       (12U)
#define EXTI_IMR1_IMR12_Msk                       (0x1UL<<EXTI_IMR1_IMR12_Pos)                      /*!< 0x00001000 */
#define EXTI_IMR1_IMR12                           EXTI_IMR1_IMR12_Msk                               
#define EXTI_IMR1_IMR13_Pos                       (13U)
#define EXTI_IMR1_IMR13_Msk                       (0x1UL<<EXTI_IMR1_IMR13_Pos)                      /*!< 0x00002000 */
#define EXTI_IMR1_IMR13                           EXTI_IMR1_IMR13_Msk                               
#define EXTI_IMR1_IMR14_Pos                       (14U)
#define EXTI_IMR1_IMR14_Msk                       (0x1UL<<EXTI_IMR1_IMR14_Pos)                      /*!< 0x00004000 */
#define EXTI_IMR1_IMR14                           EXTI_IMR1_IMR14_Msk                               
#define EXTI_IMR1_IMR15_Pos                       (15U)
#define EXTI_IMR1_IMR15_Msk                       (0x1UL<<EXTI_IMR1_IMR15_Pos)                      /*!< 0x00008000 */
#define EXTI_IMR1_IMR15                           EXTI_IMR1_IMR15_Msk                               
#define EXTI_IMR1_IMR16_Pos                       (16U)
#define EXTI_IMR1_IMR16_Msk                       (0x1UL<<EXTI_IMR1_IMR16_Pos)                      /*!< 0x00010000 */
#define EXTI_IMR1_IMR16                           EXTI_IMR1_IMR16_Msk                               
#define EXTI_IMR1_IMR17_Pos                       (17U)
#define EXTI_IMR1_IMR17_Msk                       (0x1UL<<EXTI_IMR1_IMR17_Pos)                      /*!< 0x00020000 */
#define EXTI_IMR1_IMR17                           EXTI_IMR1_IMR17_Msk                               
#define EXTI_IMR1_IMR18_Pos                       (18U)
#define EXTI_IMR1_IMR18_Msk                       (0x1UL<<EXTI_IMR1_IMR18_Pos)                      /*!< 0x00040000 */
#define EXTI_IMR1_IMR18                           EXTI_IMR1_IMR18_Msk                               
#define EXTI_IMR1_IMR19_Pos                       (19U)
#define EXTI_IMR1_IMR19_Msk                       (0x1UL<<EXTI_IMR1_IMR19_Pos)                      /*!< 0x00080000 */
#define EXTI_IMR1_IMR19                           EXTI_IMR1_IMR19_Msk                               
#define EXTI_IMR1_IMR20_Pos                       (20U)
#define EXTI_IMR1_IMR20_Msk                       (0x1UL<<EXTI_IMR1_IMR20_Pos)                      /*!< 0x00100000 */
#define EXTI_IMR1_IMR20                           EXTI_IMR1_IMR20_Msk                               
#define EXTI_IMR1_IMR21_Pos                       (21U)
#define EXTI_IMR1_IMR21_Msk                       (0x1UL<<EXTI_IMR1_IMR21_Pos)                      /*!< 0x00200000 */
#define EXTI_IMR1_IMR21                           EXTI_IMR1_IMR21_Msk                               
#define EXTI_IMR1_IMR22_Pos                       (22U)
#define EXTI_IMR1_IMR22_Msk                       (0x1UL<<EXTI_IMR1_IMR22_Pos)                      /*!< 0x00400000 */
#define EXTI_IMR1_IMR22                           EXTI_IMR1_IMR22_Msk                               
#define EXTI_IMR1_IMR23_Pos                       (23U)
#define EXTI_IMR1_IMR23_Msk                       (0x1UL<<EXTI_IMR1_IMR23_Pos)                      /*!< 0x00800000 */
#define EXTI_IMR1_IMR23                           EXTI_IMR1_IMR23_Msk                               
#define EXTI_IMR1_IMR24_Pos                       (24U)
#define EXTI_IMR1_IMR24_Msk                       (0x1UL<<EXTI_IMR1_IMR24_Pos)                      /*!< 0x01000000 */
#define EXTI_IMR1_IMR24                           EXTI_IMR1_IMR24_Msk                               
#define EXTI_IMR1_IMR25_Pos                       (25U)
#define EXTI_IMR1_IMR25_Msk                       (0x1UL<<EXTI_IMR1_IMR25_Pos)                      /*!< 0x02000000 */
#define EXTI_IMR1_IMR25                           EXTI_IMR1_IMR25_Msk                               
#define EXTI_IMR1_IMR26_Pos                       (26U)
#define EXTI_IMR1_IMR26_Msk                       (0x1UL<<EXTI_IMR1_IMR26_Pos)                      /*!< 0x04000000 */
#define EXTI_IMR1_IMR26                           EXTI_IMR1_IMR26_Msk                               
#define EXTI_IMR1_IMR27_Pos                       (27U)
#define EXTI_IMR1_IMR27_Msk                       (0x1UL<<EXTI_IMR1_IMR27_Pos)                      /*!< 0x08000000 */
#define EXTI_IMR1_IMR27                           EXTI_IMR1_IMR27_Msk                               
#define EXTI_IMR1_IMR28_Pos                       (28U)
#define EXTI_IMR1_IMR28_Msk                       (0x1UL<<EXTI_IMR1_IMR28_Pos)                      /*!< 0x10000000 */
#define EXTI_IMR1_IMR28                           EXTI_IMR1_IMR28_Msk                               
#define EXTI_IMR1_IMR29_Pos                       (29U)
#define EXTI_IMR1_IMR29_Msk                       (0x1UL<<EXTI_IMR1_IMR29_Pos)                      /*!< 0x20000000 */
#define EXTI_IMR1_IMR29                           EXTI_IMR1_IMR29_Msk                               
#define EXTI_IMR1_IMR30_Pos                       (30U)
#define EXTI_IMR1_IMR30_Msk                       (0x1UL<<EXTI_IMR1_IMR30_Pos)                      /*!< 0x40000000 */
#define EXTI_IMR1_IMR30                           EXTI_IMR1_IMR30_Msk                               
#define EXTI_IMR1_IMR31_Pos                       (31U)
#define EXTI_IMR1_IMR31_Msk                       (0x1UL<<EXTI_IMR1_IMR31_Pos)                      /*!< 0x80000000 */
#define EXTI_IMR1_IMR31                           EXTI_IMR1_IMR31_Msk                               
#define EXTI_IMR1_IMR_Pos                         (0U)
#define EXTI_IMR1_IMR_Msk                         (0xFFFFFFFFUL<<EXTI_IMR1_IMR_Pos)                 /*!< 0xFFFFFFFF */
#define EXTI_IMR1_IMR                              EXTI_IMR1_IMR_Msk                               

/********************************* Bit definition for EXTI_EMR1 register ********************************************/
#define EXTI_EMR1_EMR0_Pos                        (0U)
#define EXTI_EMR1_EMR0_Msk                        (0x1UL<<EXTI_EMR1_EMR0_Pos)                       /*!< 0x00000001 */
#define EXTI_EMR1_EMR0                            EXTI_EMR1_EMR0_Msk                                
#define EXTI_EMR1_EMR1_Pos                        (1U)
#define EXTI_EMR1_EMR1_Msk                        (0x1UL<<EXTI_EMR1_EMR1_Pos)                       /*!< 0x00000002 */
#define EXTI_EMR1_EMR1                            EXTI_EMR1_EMR1_Msk                                
#define EXTI_EMR1_EMR2_Pos                        (2U)
#define EXTI_EMR1_EMR2_Msk                        (0x1UL<<EXTI_EMR1_EMR2_Pos)                       /*!< 0x00000004 */
#define EXTI_EMR1_EMR2                            EXTI_EMR1_EMR2_Msk                                
#define EXTI_EMR1_EMR3_Pos                        (3U)
#define EXTI_EMR1_EMR3_Msk                        (0x1UL<<EXTI_EMR1_EMR3_Pos)                       /*!< 0x00000008 */
#define EXTI_EMR1_EMR3                            EXTI_EMR1_EMR3_Msk                                
#define EXTI_EMR1_EMR4_Pos                        (4U)
#define EXTI_EMR1_EMR4_Msk                        (0x1UL<<EXTI_EMR1_EMR4_Pos)                       /*!< 0x00000010 */
#define EXTI_EMR1_EMR4                            EXTI_EMR1_EMR4_Msk                                
#define EXTI_EMR1_EMR5_Pos                        (5U)
#define EXTI_EMR1_EMR5_Msk                        (0x1UL<<EXTI_EMR1_EMR5_Pos)                       /*!< 0x00000020 */
#define EXTI_EMR1_EMR5                            EXTI_EMR1_EMR5_Msk                                
#define EXTI_EMR1_EMR6_Pos                        (6U)
#define EXTI_EMR1_EMR6_Msk                        (0x1UL<<EXTI_EMR1_EMR6_Pos)                       /*!< 0x00000040 */
#define EXTI_EMR1_EMR6                            EXTI_EMR1_EMR6_Msk                                
#define EXTI_EMR1_EMR7_Pos                        (7U)
#define EXTI_EMR1_EMR7_Msk                        (0x1UL<<EXTI_EMR1_EMR7_Pos)                       /*!< 0x00000080 */
#define EXTI_EMR1_EMR7                            EXTI_EMR1_EMR7_Msk                                
#define EXTI_EMR1_EMR8_Pos                        (8U)
#define EXTI_EMR1_EMR8_Msk                        (0x1UL<<EXTI_EMR1_EMR8_Pos)                       /*!< 0x00000100 */
#define EXTI_EMR1_EMR8                            EXTI_EMR1_EMR8_Msk                                
#define EXTI_EMR1_EMR9_Pos                        (9U)
#define EXTI_EMR1_EMR9_Msk                        (0x1UL<<EXTI_EMR1_EMR9_Pos)                       /*!< 0x00000200 */
#define EXTI_EMR1_EMR9                            EXTI_EMR1_EMR9_Msk                                
#define EXTI_EMR1_EMR10_Pos                       (10U)
#define EXTI_EMR1_EMR10_Msk                       (0x1UL<<EXTI_EMR1_EMR10_Pos)                      /*!< 0x00000400 */
#define EXTI_EMR1_EMR10                           EXTI_EMR1_EMR10_Msk                               
#define EXTI_EMR1_EMR11_Pos                       (11U)
#define EXTI_EMR1_EMR11_Msk                       (0x1UL<<EXTI_EMR1_EMR11_Pos)                      /*!< 0x00000800 */
#define EXTI_EMR1_EMR11                           EXTI_EMR1_EMR11_Msk                               
#define EXTI_EMR1_EMR12_Pos                       (12U)
#define EXTI_EMR1_EMR12_Msk                       (0x1UL<<EXTI_EMR1_EMR12_Pos)                      /*!< 0x00001000 */
#define EXTI_EMR1_EMR12                           EXTI_EMR1_EMR12_Msk                               
#define EXTI_EMR1_EMR13_Pos                       (13U)
#define EXTI_EMR1_EMR13_Msk                       (0x1UL<<EXTI_EMR1_EMR13_Pos)                      /*!< 0x00002000 */
#define EXTI_EMR1_EMR13                           EXTI_EMR1_EMR13_Msk                               
#define EXTI_EMR1_EMR14_Pos                       (14U)
#define EXTI_EMR1_EMR14_Msk                       (0x1UL<<EXTI_EMR1_EMR14_Pos)                      /*!< 0x00004000 */
#define EXTI_EMR1_EMR14                           EXTI_EMR1_EMR14_Msk                               
#define EXTI_EMR1_EMR15_Pos                       (15U)
#define EXTI_EMR1_EMR15_Msk                       (0x1UL<<EXTI_EMR1_EMR15_Pos)                      /*!< 0x00008000 */
#define EXTI_EMR1_EMR15                           EXTI_EMR1_EMR15_Msk                               
#define EXTI_EMR1_EMR16_Pos                       (16U)
#define EXTI_EMR1_EMR16_Msk                       (0x1UL<<EXTI_EMR1_EMR16_Pos)                      /*!< 0x00010000 */
#define EXTI_EMR1_EMR16                           EXTI_EMR1_EMR16_Msk                               
#define EXTI_EMR1_EMR17_Pos                       (17U)
#define EXTI_EMR1_EMR17_Msk                       (0x1UL<<EXTI_EMR1_EMR17_Pos)                      /*!< 0x00020000 */
#define EXTI_EMR1_EMR17                           EXTI_EMR1_EMR17_Msk                               
#define EXTI_EMR1_EMR18_Pos                       (18U)
#define EXTI_EMR1_EMR18_Msk                       (0x1UL<<EXTI_EMR1_EMR18_Pos)                      /*!< 0x00040000 */
#define EXTI_EMR1_EMR18                           EXTI_EMR1_EMR18_Msk                               
#define EXTI_EMR1_EMR19_Pos                       (19U)
#define EXTI_EMR1_EMR19_Msk                       (0x1UL<<EXTI_EMR1_EMR19_Pos)                      /*!< 0x00080000 */
#define EXTI_EMR1_EMR19                           EXTI_EMR1_EMR19_Msk                               
#define EXTI_EMR1_EMR20_Pos                       (20U)
#define EXTI_EMR1_EMR20_Msk                       (0x1UL<<EXTI_EMR1_EMR20_Pos)                      /*!< 0x00100000 */
#define EXTI_EMR1_EMR20                           EXTI_EMR1_EMR20_Msk                               
#define EXTI_EMR1_EMR21_Pos                       (21U)
#define EXTI_EMR1_EMR21_Msk                       (0x1UL<<EXTI_EMR1_EMR21_Pos)                      /*!< 0x00200000 */
#define EXTI_EMR1_EMR21                           EXTI_EMR1_EMR21_Msk                               
#define EXTI_EMR1_EMR22_Pos                       (22U)
#define EXTI_EMR1_EMR22_Msk                       (0x1UL<<EXTI_EMR1_EMR22_Pos)                      /*!< 0x00400000 */
#define EXTI_EMR1_EMR22                           EXTI_EMR1_EMR22_Msk                               
#define EXTI_EMR1_EMR23_Pos                       (23U)
#define EXTI_EMR1_EMR23_Msk                       (0x1UL<<EXTI_EMR1_EMR23_Pos)                      /*!< 0x00800000 */
#define EXTI_EMR1_EMR23                           EXTI_EMR1_EMR23_Msk                               
#define EXTI_EMR1_EMR24_Pos                       (24U)
#define EXTI_EMR1_EMR24_Msk                       (0x1UL<<EXTI_EMR1_EMR24_Pos)                      /*!< 0x01000000 */
#define EXTI_EMR1_EMR24                           EXTI_EMR1_EMR24_Msk                               
#define EXTI_EMR1_EMR25_Pos                       (25U)
#define EXTI_EMR1_EMR25_Msk                       (0x1UL<<EXTI_EMR1_EMR25_Pos)                      /*!< 0x02000000 */
#define EXTI_EMR1_EMR25                           EXTI_EMR1_EMR25_Msk                               
#define EXTI_EMR1_EMR26_Pos                       (26U)
#define EXTI_EMR1_EMR26_Msk                       (0x1UL<<EXTI_EMR1_EMR26_Pos)                      /*!< 0x04000000 */
#define EXTI_EMR1_EMR26                           EXTI_EMR1_EMR26_Msk                               
#define EXTI_EMR1_EMR27_Pos                       (27U)
#define EXTI_EMR1_EMR27_Msk                       (0x1UL<<EXTI_EMR1_EMR27_Pos)                      /*!< 0x08000000 */
#define EXTI_EMR1_EMR27                           EXTI_EMR1_EMR27_Msk                               
#define EXTI_EMR1_EMR28_Pos                       (28U)
#define EXTI_EMR1_EMR28_Msk                       (0x1UL<<EXTI_EMR1_EMR28_Pos)                      /*!< 0x10000000 */
#define EXTI_EMR1_EMR28                           EXTI_EMR1_EMR28_Msk                               
#define EXTI_EMR1_EMR29_Pos                       (29U)
#define EXTI_EMR1_EMR29_Msk                       (0x1UL<<EXTI_EMR1_EMR29_Pos)                      /*!< 0x20000000 */
#define EXTI_EMR1_EMR29                           EXTI_EMR1_EMR29_Msk                               
#define EXTI_EMR1_EMR30_Pos                       (30U)
#define EXTI_EMR1_EMR30_Msk                       (0x1UL<<EXTI_EMR1_EMR30_Pos)                      /*!< 0x40000000 */
#define EXTI_EMR1_EMR30                           EXTI_EMR1_EMR30_Msk                               
#define EXTI_EMR1_EMR31_Pos                       (31U)
#define EXTI_EMR1_EMR31_Msk                       (0x1UL<<EXTI_EMR1_EMR31_Pos)                      /*!< 0x80000000 */
#define EXTI_EMR1_EMR31                           EXTI_EMR1_EMR31_Msk                               

/********************************* Bit definition for EXTI_RTSR register ********************************************/
#define EXTI_RTSR_RTSR0_Pos                       (0U)
#define EXTI_RTSR_RTSR0_Msk                       (0x1UL<<EXTI_RTSR_RTSR0_Pos)                      /*!< 0x00000001 */
#define EXTI_RTSR_RTSR0                           EXTI_RTSR_RTSR0_Msk                               
#define EXTI_RTSR_RTSR1_Pos                       (1U)
#define EXTI_RTSR_RTSR1_Msk                       (0x1UL<<EXTI_RTSR_RTSR1_Pos)                      /*!< 0x00000002 */
#define EXTI_RTSR_RTSR1                           EXTI_RTSR_RTSR1_Msk                               
#define EXTI_RTSR_RTSR2_Pos                       (2U)
#define EXTI_RTSR_RTSR2_Msk                       (0x1UL<<EXTI_RTSR_RTSR2_Pos)                      /*!< 0x00000004 */
#define EXTI_RTSR_RTSR2                           EXTI_RTSR_RTSR2_Msk                               
#define EXTI_RTSR_RTSR3_Pos                       (3U)
#define EXTI_RTSR_RTSR3_Msk                       (0x1UL<<EXTI_RTSR_RTSR3_Pos)                      /*!< 0x00000008 */
#define EXTI_RTSR_RTSR3                           EXTI_RTSR_RTSR3_Msk                               
#define EXTI_RTSR_RTSR4_Pos                       (4U)
#define EXTI_RTSR_RTSR4_Msk                       (0x1UL<<EXTI_RTSR_RTSR4_Pos)                      /*!< 0x00000010 */
#define EXTI_RTSR_RTSR4                           EXTI_RTSR_RTSR4_Msk                               
#define EXTI_RTSR_RTSR5_Pos                       (5U)
#define EXTI_RTSR_RTSR5_Msk                       (0x1UL<<EXTI_RTSR_RTSR5_Pos)                      /*!< 0x00000020 */
#define EXTI_RTSR_RTSR5                           EXTI_RTSR_RTSR5_Msk                               
#define EXTI_RTSR_RTSR6_Pos                       (6U)
#define EXTI_RTSR_RTSR6_Msk                       (0x1UL<<EXTI_RTSR_RTSR6_Pos)                      /*!< 0x00000040 */
#define EXTI_RTSR_RTSR6                           EXTI_RTSR_RTSR6_Msk                               
#define EXTI_RTSR_RTSR7_Pos                       (7U)
#define EXTI_RTSR_RTSR7_Msk                       (0x1UL<<EXTI_RTSR_RTSR7_Pos)                      /*!< 0x00000080 */
#define EXTI_RTSR_RTSR7                           EXTI_RTSR_RTSR7_Msk                               
#define EXTI_RTSR_RTSR8_Pos                       (8U)
#define EXTI_RTSR_RTSR8_Msk                       (0x1UL<<EXTI_RTSR_RTSR8_Pos)                      /*!< 0x00000100 */
#define EXTI_RTSR_RTSR8                           EXTI_RTSR_RTSR8_Msk                               
#define EXTI_RTSR_RTSR9_Pos                       (9U)
#define EXTI_RTSR_RTSR9_Msk                       (0x1UL<<EXTI_RTSR_RTSR9_Pos)                      /*!< 0x00000200 */
#define EXTI_RTSR_RTSR9                           EXTI_RTSR_RTSR9_Msk                               
#define EXTI_RTSR_RTSR10_Pos                      (10U)
#define EXTI_RTSR_RTSR10_Msk                      (0x1UL<<EXTI_RTSR_RTSR10_Pos)                     /*!< 0x00000400 */
#define EXTI_RTSR_RTSR10                          EXTI_RTSR_RTSR10_Msk                              
#define EXTI_RTSR_RTSR11_Pos                      (11U)
#define EXTI_RTSR_RTSR11_Msk                      (0x1UL<<EXTI_RTSR_RTSR11_Pos)                     /*!< 0x00000800 */
#define EXTI_RTSR_RTSR11                          EXTI_RTSR_RTSR11_Msk                              
#define EXTI_RTSR_RTSR12_Pos                      (12U)
#define EXTI_RTSR_RTSR12_Msk                      (0x1UL<<EXTI_RTSR_RTSR12_Pos)                     /*!< 0x00001000 */
#define EXTI_RTSR_RTSR12                          EXTI_RTSR_RTSR12_Msk                              
#define EXTI_RTSR_RTSR13_Pos                      (13U)
#define EXTI_RTSR_RTSR13_Msk                      (0x1UL<<EXTI_RTSR_RTSR13_Pos)                     /*!< 0x00002000 */
#define EXTI_RTSR_RTSR13                          EXTI_RTSR_RTSR13_Msk                              
#define EXTI_RTSR_RTSR14_Pos                      (14U)
#define EXTI_RTSR_RTSR14_Msk                      (0x1UL<<EXTI_RTSR_RTSR14_Pos)                     /*!< 0x00004000 */
#define EXTI_RTSR_RTSR14                          EXTI_RTSR_RTSR14_Msk                              
#define EXTI_RTSR_RTSR15_Pos                      (15U)
#define EXTI_RTSR_RTSR15_Msk                      (0x1UL<<EXTI_RTSR_RTSR15_Pos)                     /*!< 0x00008000 */
#define EXTI_RTSR_RTSR15                          EXTI_RTSR_RTSR15_Msk                              
#define EXTI_RTSR_RTSR16_Pos                      (16U)
#define EXTI_RTSR_RTSR16_Msk                      (0x1UL<<EXTI_RTSR_RTSR16_Pos)                     /*!< 0x00010000 */
#define EXTI_RTSR_RTSR16                          EXTI_RTSR_RTSR16_Msk                              
#define EXTI_RTSR_RTSR17_Pos                      (17U)
#define EXTI_RTSR_RTSR17_Msk                      (0x1UL<<EXTI_RTSR_RTSR17_Pos)                     /*!< 0x00020000 */
#define EXTI_RTSR_RTSR17                          EXTI_RTSR_RTSR17_Msk                              
#define EXTI_RTSR_RTSR18_Pos                      (18U)
#define EXTI_RTSR_RTSR18_Msk                      (0x1UL<<EXTI_RTSR_RTSR18_Pos)                     /*!< 0x00040000 */
#define EXTI_RTSR_RTSR18                          EXTI_RTSR_RTSR18_Msk                              
#define EXTI_RTSR_RTSR19_Pos                      (19U)
#define EXTI_RTSR_RTSR19_Msk                      (0x1UL<<EXTI_RTSR_RTSR19_Pos)                     /*!< 0x00080000 */
#define EXTI_RTSR_RTSR19                          EXTI_RTSR_RTSR19_Msk                              
#define EXTI_RTSR_RTSR20_Pos                      (20U)
#define EXTI_RTSR_RTSR20_Msk                      (0x1UL<<EXTI_RTSR_RTSR20_Pos)                     /*!< 0x00100000 */
#define EXTI_RTSR_RTSR20                          EXTI_RTSR_RTSR20_Msk                              
#define EXTI_RTSR_RTSR21_Pos                      (21U)
#define EXTI_RTSR_RTSR21_Msk                      (0x1UL<<EXTI_RTSR_RTSR21_Pos)                     /*!< 0x00200000 */
#define EXTI_RTSR_RTSR21                          EXTI_RTSR_RTSR21_Msk                              

/********************************* Bit definition for EXTI_FTSR register ********************************************/
#define EXTI_FTSR_FTSR0_Pos                       (0U)
#define EXTI_FTSR_FTSR0_Msk                       (0x1UL<<EXTI_FTSR_FTSR0_Pos)                      /*!< 0x00000001 */
#define EXTI_FTSR_FTSR0                           EXTI_FTSR_FTSR0_Msk                               
#define EXTI_FTSR_FTSR1_Pos                       (1U)
#define EXTI_FTSR_FTSR1_Msk                       (0x1UL<<EXTI_FTSR_FTSR1_Pos)                      /*!< 0x00000002 */
#define EXTI_FTSR_FTSR1                           EXTI_FTSR_FTSR1_Msk                               
#define EXTI_FTSR_FTSR2_Pos                       (2U)
#define EXTI_FTSR_FTSR2_Msk                       (0x1UL<<EXTI_FTSR_FTSR2_Pos)                      /*!< 0x00000004 */
#define EXTI_FTSR_FTSR2                           EXTI_FTSR_FTSR2_Msk                               
#define EXTI_FTSR_FTSR3_Pos                       (3U)
#define EXTI_FTSR_FTSR3_Msk                       (0x1UL<<EXTI_FTSR_FTSR3_Pos)                      /*!< 0x00000008 */
#define EXTI_FTSR_FTSR3                           EXTI_FTSR_FTSR3_Msk                               
#define EXTI_FTSR_FTSR4_Pos                       (4U)
#define EXTI_FTSR_FTSR4_Msk                       (0x1UL<<EXTI_FTSR_FTSR4_Pos)                      /*!< 0x00000010 */
#define EXTI_FTSR_FTSR4                           EXTI_FTSR_FTSR4_Msk                               
#define EXTI_FTSR_FTSR5_Pos                       (5U)
#define EXTI_FTSR_FTSR5_Msk                       (0x1UL<<EXTI_FTSR_FTSR5_Pos)                      /*!< 0x00000020 */
#define EXTI_FTSR_FTSR5                           EXTI_FTSR_FTSR5_Msk                               
#define EXTI_FTSR_FTSR6_Pos                       (6U)
#define EXTI_FTSR_FTSR6_Msk                       (0x1UL<<EXTI_FTSR_FTSR6_Pos)                      /*!< 0x00000040 */
#define EXTI_FTSR_FTSR6                           EXTI_FTSR_FTSR6_Msk                               
#define EXTI_FTSR_FTSR7_Pos                       (7U)
#define EXTI_FTSR_FTSR7_Msk                       (0x1UL<<EXTI_FTSR_FTSR7_Pos)                      /*!< 0x00000080 */
#define EXTI_FTSR_FTSR7                           EXTI_FTSR_FTSR7_Msk                               
#define EXTI_FTSR_FTSR8_Pos                       (8U)
#define EXTI_FTSR_FTSR8_Msk                       (0x1UL<<EXTI_FTSR_FTSR8_Pos)                      /*!< 0x00000100 */
#define EXTI_FTSR_FTSR8                           EXTI_FTSR_FTSR8_Msk                               
#define EXTI_FTSR_FTSR9_Pos                       (9U)
#define EXTI_FTSR_FTSR9_Msk                       (0x1UL<<EXTI_FTSR_FTSR9_Pos)                      /*!< 0x00000200 */
#define EXTI_FTSR_FTSR9                           EXTI_FTSR_FTSR9_Msk                               
#define EXTI_FTSR_FTSR10_Pos                      (10U)
#define EXTI_FTSR_FTSR10_Msk                      (0x1UL<<EXTI_FTSR_FTSR10_Pos)                     /*!< 0x00000400 */
#define EXTI_FTSR_FTSR10                          EXTI_FTSR_FTSR10_Msk                              
#define EXTI_FTSR_FTSR11_Pos                      (11U)
#define EXTI_FTSR_FTSR11_Msk                      (0x1UL<<EXTI_FTSR_FTSR11_Pos)                     /*!< 0x00000800 */
#define EXTI_FTSR_FTSR11                          EXTI_FTSR_FTSR11_Msk                              
#define EXTI_FTSR_FTSR12_Pos                      (12U)
#define EXTI_FTSR_FTSR12_Msk                      (0x1UL<<EXTI_FTSR_FTSR12_Pos)                     /*!< 0x00001000 */
#define EXTI_FTSR_FTSR12                          EXTI_FTSR_FTSR12_Msk                              
#define EXTI_FTSR_FTSR13_Pos                      (13U)
#define EXTI_FTSR_FTSR13_Msk                      (0x1UL<<EXTI_FTSR_FTSR13_Pos)                     /*!< 0x00002000 */
#define EXTI_FTSR_FTSR13                          EXTI_FTSR_FTSR13_Msk                              
#define EXTI_FTSR_FTSR14_Pos                      (14U)
#define EXTI_FTSR_FTSR14_Msk                      (0x1UL<<EXTI_FTSR_FTSR14_Pos)                     /*!< 0x00004000 */
#define EXTI_FTSR_FTSR14                          EXTI_FTSR_FTSR14_Msk                              
#define EXTI_FTSR_FTSR15_Pos                      (15U)
#define EXTI_FTSR_FTSR15_Msk                      (0x1UL<<EXTI_FTSR_FTSR15_Pos)                     /*!< 0x00008000 */
#define EXTI_FTSR_FTSR15                          EXTI_FTSR_FTSR15_Msk                              
#define EXTI_FTSR_FTSR16_Pos                      (16U)
#define EXTI_FTSR_FTSR16_Msk                      (0x1UL<<EXTI_FTSR_FTSR16_Pos)                     /*!< 0x00010000 */
#define EXTI_FTSR_FTSR16                          EXTI_FTSR_FTSR16_Msk                              
#define EXTI_FTSR_FTSR17_Pos                      (17U)
#define EXTI_FTSR_FTSR17_Msk                      (0x1UL<<EXTI_FTSR_FTSR17_Pos)                     /*!< 0x00020000 */
#define EXTI_FTSR_FTSR17                          EXTI_FTSR_FTSR17_Msk                              
#define EXTI_FTSR_FTSR18_Pos                      (18U)
#define EXTI_FTSR_FTSR18_Msk                      (0x1UL<<EXTI_FTSR_FTSR18_Pos)                     /*!< 0x00040000 */
#define EXTI_FTSR_FTSR18                          EXTI_FTSR_FTSR18_Msk                              
#define EXTI_FTSR_FTSR19_Pos                      (19U)
#define EXTI_FTSR_FTSR19_Msk                      (0x1UL<<EXTI_FTSR_FTSR19_Pos)                     /*!< 0x00080000 */
#define EXTI_FTSR_FTSR19                          EXTI_FTSR_FTSR19_Msk                              
#define EXTI_FTSR_FTSR20_Pos                      (20U)
#define EXTI_FTSR_FTSR20_Msk                      (0x1UL<<EXTI_FTSR_FTSR20_Pos)                     /*!< 0x00100000 */
#define EXTI_FTSR_FTSR20                          EXTI_FTSR_FTSR20_Msk                              
#define EXTI_FTSR_FTSR21_Pos                      (21U)
#define EXTI_FTSR_FTSR21_Msk                      (0x1UL<<EXTI_FTSR_FTSR21_Pos)                     /*!< 0x00200000 */
#define EXTI_FTSR_FTSR21                          EXTI_FTSR_FTSR21_Msk                              

/********************************* Bit definition for EXTI_SWIER register *******************************************/
#define EXTI_SWIER_SWIER0_Pos                     (0U)
#define EXTI_SWIER_SWIER0_Msk                     (0x1UL<<EXTI_SWIER_SWIER0_Pos)                    /*!< 0x00000001 */
#define EXTI_SWIER_SWIER0                         EXTI_SWIER_SWIER0_Msk                             
#define EXTI_SWIER_SWIER1_Pos                     (1U)
#define EXTI_SWIER_SWIER1_Msk                     (0x1UL<<EXTI_SWIER_SWIER1_Pos)                    /*!< 0x00000002 */
#define EXTI_SWIER_SWIER1                         EXTI_SWIER_SWIER1_Msk                             
#define EXTI_SWIER_SWIER2_Pos                     (2U)
#define EXTI_SWIER_SWIER2_Msk                     (0x1UL<<EXTI_SWIER_SWIER2_Pos)                    /*!< 0x00000004 */
#define EXTI_SWIER_SWIER2                         EXTI_SWIER_SWIER2_Msk                             
#define EXTI_SWIER_SWIER3_Pos                     (3U)
#define EXTI_SWIER_SWIER3_Msk                     (0x1UL<<EXTI_SWIER_SWIER3_Pos)                    /*!< 0x00000008 */
#define EXTI_SWIER_SWIER3                         EXTI_SWIER_SWIER3_Msk                             
#define EXTI_SWIER_SWIER4_Pos                     (4U)
#define EXTI_SWIER_SWIER4_Msk                     (0x1UL<<EXTI_SWIER_SWIER4_Pos)                    /*!< 0x00000010 */
#define EXTI_SWIER_SWIER4                         EXTI_SWIER_SWIER4_Msk                             
#define EXTI_SWIER_SWIER5_Pos                     (5U)
#define EXTI_SWIER_SWIER5_Msk                     (0x1UL<<EXTI_SWIER_SWIER5_Pos)                    /*!< 0x00000020 */
#define EXTI_SWIER_SWIER5                         EXTI_SWIER_SWIER5_Msk                             
#define EXTI_SWIER_SWIER6_Pos                     (6U)
#define EXTI_SWIER_SWIER6_Msk                     (0x1UL<<EXTI_SWIER_SWIER6_Pos)                    /*!< 0x00000040 */
#define EXTI_SWIER_SWIER6                         EXTI_SWIER_SWIER6_Msk                             
#define EXTI_SWIER_SWIER7_Pos                     (7U)
#define EXTI_SWIER_SWIER7_Msk                     (0x1UL<<EXTI_SWIER_SWIER7_Pos)                    /*!< 0x00000080 */
#define EXTI_SWIER_SWIER7                         EXTI_SWIER_SWIER7_Msk                             
#define EXTI_SWIER_SWIER8_Pos                     (8U)
#define EXTI_SWIER_SWIER8_Msk                     (0x1UL<<EXTI_SWIER_SWIER8_Pos)                    /*!< 0x00000100 */
#define EXTI_SWIER_SWIER8                         EXTI_SWIER_SWIER8_Msk                             
#define EXTI_SWIER_SWIER9_Pos                     (9U)
#define EXTI_SWIER_SWIER9_Msk                     (0x1UL<<EXTI_SWIER_SWIER9_Pos)                    /*!< 0x00000200 */
#define EXTI_SWIER_SWIER9                         EXTI_SWIER_SWIER9_Msk                             
#define EXTI_SWIER_SWIER10_Pos                    (10U)
#define EXTI_SWIER_SWIER10_Msk                    (0x1UL<<EXTI_SWIER_SWIER10_Pos)                   /*!< 0x00000400 */
#define EXTI_SWIER_SWIER10                        EXTI_SWIER_SWIER10_Msk                            
#define EXTI_SWIER_SWIER11_Pos                    (11U)
#define EXTI_SWIER_SWIER11_Msk                    (0x1UL<<EXTI_SWIER_SWIER11_Pos)                   /*!< 0x00000800 */
#define EXTI_SWIER_SWIER11                        EXTI_SWIER_SWIER11_Msk                            
#define EXTI_SWIER_SWIER12_Pos                    (12U)
#define EXTI_SWIER_SWIER12_Msk                    (0x1UL<<EXTI_SWIER_SWIER12_Pos)                   /*!< 0x00001000 */
#define EXTI_SWIER_SWIER12                        EXTI_SWIER_SWIER12_Msk                            
#define EXTI_SWIER_SWIER13_Pos                    (13U)
#define EXTI_SWIER_SWIER13_Msk                    (0x1UL<<EXTI_SWIER_SWIER13_Pos)                   /*!< 0x00002000 */
#define EXTI_SWIER_SWIER13                        EXTI_SWIER_SWIER13_Msk                            
#define EXTI_SWIER_SWIER14_Pos                    (14U)
#define EXTI_SWIER_SWIER14_Msk                    (0x1UL<<EXTI_SWIER_SWIER14_Pos)                   /*!< 0x00004000 */
#define EXTI_SWIER_SWIER14                        EXTI_SWIER_SWIER14_Msk                            
#define EXTI_SWIER_SWIER15_Pos                    (15U)
#define EXTI_SWIER_SWIER15_Msk                    (0x1UL<<EXTI_SWIER_SWIER15_Pos)                   /*!< 0x00008000 */
#define EXTI_SWIER_SWIER15                        EXTI_SWIER_SWIER15_Msk                            
#define EXTI_SWIER_SWIER16_Pos                    (16U)
#define EXTI_SWIER_SWIER16_Msk                    (0x1UL<<EXTI_SWIER_SWIER16_Pos)                   /*!< 0x00010000 */
#define EXTI_SWIER_SWIER16                        EXTI_SWIER_SWIER16_Msk                            
#define EXTI_SWIER_SWIER17_Pos                    (17U)
#define EXTI_SWIER_SWIER17_Msk                    (0x1UL<<EXTI_SWIER_SWIER17_Pos)                   /*!< 0x00020000 */
#define EXTI_SWIER_SWIER17                        EXTI_SWIER_SWIER17_Msk                            
#define EXTI_SWIER_SWIER18_Pos                    (18U)
#define EXTI_SWIER_SWIER18_Msk                    (0x1UL<<EXTI_SWIER_SWIER18_Pos)                   /*!< 0x00040000 */
#define EXTI_SWIER_SWIER18                        EXTI_SWIER_SWIER18_Msk                            
#define EXTI_SWIER_SWIER19_Pos                    (19U)
#define EXTI_SWIER_SWIER19_Msk                    (0x1UL<<EXTI_SWIER_SWIER19_Pos)                   /*!< 0x00080000 */
#define EXTI_SWIER_SWIER19                        EXTI_SWIER_SWIER19_Msk                            
#define EXTI_SWIER_SWIER20_Pos                    (20U)
#define EXTI_SWIER_SWIER20_Msk                    (0x1UL<<EXTI_SWIER_SWIER20_Pos)                   /*!< 0x00100000 */
#define EXTI_SWIER_SWIER20                        EXTI_SWIER_SWIER20_Msk                            
#define EXTI_SWIER_SWIER21_Pos                    (21U)
#define EXTI_SWIER_SWIER21_Msk                    (0x1UL<<EXTI_SWIER_SWIER21_Pos)                   /*!< 0x00200000 */
#define EXTI_SWIER_SWIER21                        EXTI_SWIER_SWIER21_Msk                            

/********************************* Bit definition for EXTI_PR register **********************************************/
#define EXTI_PR_PR0_Pos                           (0U)
#define EXTI_PR_PR0_Msk                           (0x1UL<<EXTI_PR_PR0_Pos)                          /*!< 0x00000001 */
#define EXTI_PR_PR0                               EXTI_PR_PR0_Msk                                   
#define EXTI_PR_PR1_Pos                           (1U)
#define EXTI_PR_PR1_Msk                           (0x1UL<<EXTI_PR_PR1_Pos)                          /*!< 0x00000002 */
#define EXTI_PR_PR1                               EXTI_PR_PR1_Msk                                   
#define EXTI_PR_PR2_Pos                           (2U)
#define EXTI_PR_PR2_Msk                           (0x1UL<<EXTI_PR_PR2_Pos)                          /*!< 0x00000004 */
#define EXTI_PR_PR2                               EXTI_PR_PR2_Msk                                   
#define EXTI_PR_PR3_Pos                           (3U)
#define EXTI_PR_PR3_Msk                           (0x1UL<<EXTI_PR_PR3_Pos)                          /*!< 0x00000008 */
#define EXTI_PR_PR3                               EXTI_PR_PR3_Msk                                   
#define EXTI_PR_PR4_Pos                           (4U)
#define EXTI_PR_PR4_Msk                           (0x1UL<<EXTI_PR_PR4_Pos)                          /*!< 0x00000010 */
#define EXTI_PR_PR4                               EXTI_PR_PR4_Msk                                   
#define EXTI_PR_PR5_Pos                           (5U)
#define EXTI_PR_PR5_Msk                           (0x1UL<<EXTI_PR_PR5_Pos)                          /*!< 0x00000020 */
#define EXTI_PR_PR5                               EXTI_PR_PR5_Msk                                   
#define EXTI_PR_PR6_Pos                           (6U)
#define EXTI_PR_PR6_Msk                           (0x1UL<<EXTI_PR_PR6_Pos)                          /*!< 0x00000040 */
#define EXTI_PR_PR6                               EXTI_PR_PR6_Msk                                   
#define EXTI_PR_PR7_Pos                           (7U)
#define EXTI_PR_PR7_Msk                           (0x1UL<<EXTI_PR_PR7_Pos)                          /*!< 0x00000080 */
#define EXTI_PR_PR7                               EXTI_PR_PR7_Msk                                   
#define EXTI_PR_PR8_Pos                           (8U)
#define EXTI_PR_PR8_Msk                           (0x1UL<<EXTI_PR_PR8_Pos)                          /*!< 0x00000100 */
#define EXTI_PR_PR8                               EXTI_PR_PR8_Msk                                   
#define EXTI_PR_PR9_Pos                           (9U)
#define EXTI_PR_PR9_Msk                           (0x1UL<<EXTI_PR_PR9_Pos)                          /*!< 0x00000200 */
#define EXTI_PR_PR9                               EXTI_PR_PR9_Msk                                   
#define EXTI_PR_PR10_Pos                          (10U)
#define EXTI_PR_PR10_Msk                          (0x1UL<<EXTI_PR_PR10_Pos)                         /*!< 0x00000400 */
#define EXTI_PR_PR10                              EXTI_PR_PR10_Msk                                  
#define EXTI_PR_PR11_Pos                          (11U)
#define EXTI_PR_PR11_Msk                          (0x1UL<<EXTI_PR_PR11_Pos)                         /*!< 0x00000800 */
#define EXTI_PR_PR11                              EXTI_PR_PR11_Msk                                  
#define EXTI_PR_PR12_Pos                          (12U)
#define EXTI_PR_PR12_Msk                          (0x1UL<<EXTI_PR_PR12_Pos)                         /*!< 0x00001000 */
#define EXTI_PR_PR12                              EXTI_PR_PR12_Msk                                  
#define EXTI_PR_PR13_Pos                          (13U)
#define EXTI_PR_PR13_Msk                          (0x1UL<<EXTI_PR_PR13_Pos)                         /*!< 0x00002000 */
#define EXTI_PR_PR13                              EXTI_PR_PR13_Msk                                  
#define EXTI_PR_PR14_Pos                          (14U)
#define EXTI_PR_PR14_Msk                          (0x1UL<<EXTI_PR_PR14_Pos)                         /*!< 0x00004000 */
#define EXTI_PR_PR14                              EXTI_PR_PR14_Msk                                  
#define EXTI_PR_PR15_Pos                          (15U)
#define EXTI_PR_PR15_Msk                          (0x1UL<<EXTI_PR_PR15_Pos)                         /*!< 0x00008000 */
#define EXTI_PR_PR15                              EXTI_PR_PR15_Msk                                  
#define EXTI_PR_PR16_Pos                          (16U)
#define EXTI_PR_PR16_Msk                          (0x1UL<<EXTI_PR_PR16_Pos)                         /*!< 0x00010000 */
#define EXTI_PR_PR16                              EXTI_PR_PR16_Msk                                  
#define EXTI_PR_PR17_Pos                          (17U)
#define EXTI_PR_PR17_Msk                          (0x1UL<<EXTI_PR_PR17_Pos)                         /*!< 0x00020000 */
#define EXTI_PR_PR17                              EXTI_PR_PR17_Msk                                  
#define EXTI_PR_PR18_Pos                          (18U)
#define EXTI_PR_PR18_Msk                          (0x1UL<<EXTI_PR_PR18_Pos)                         /*!< 0x00040000 */
#define EXTI_PR_PR18                              EXTI_PR_PR18_Msk                                  
#define EXTI_PR_PR19_Pos                          (19U)
#define EXTI_PR_PR19_Msk                          (0x1UL<<EXTI_PR_PR19_Pos)                         /*!< 0x00080000 */
#define EXTI_PR_PR19                              EXTI_PR_PR19_Msk                                  
#define EXTI_PR_PR20_Pos                          (20U)
#define EXTI_PR_PR20_Msk                          (0x1UL<<EXTI_PR_PR20_Pos)                         /*!< 0x00100000 */
#define EXTI_PR_PR20                              EXTI_PR_PR20_Msk                                  
#define EXTI_PR_PR21_Pos                          (21U)
#define EXTI_PR_PR21_Msk                          (0x1UL<<EXTI_PR_PR21_Pos)                         /*!< 0x00200000 */
#define EXTI_PR_PR21                              EXTI_PR_PR21_Msk                                  

/********************************* Bit definition for EXTI_IMR2 register ********************************************/
#define EXTI_IMR2_IMR32_Pos                       (0U)
#define EXTI_IMR2_IMR32_Msk                       (0x1UL<<EXTI_IMR2_IMR32_Pos)                      /*!< 0x00000001 */
#define EXTI_IMR2_IMR32                           EXTI_IMR2_IMR32_Msk                               
#define EXTI_IMR2_IMR33_Pos                       (1U)
#define EXTI_IMR2_IMR33_Msk                       (0x1UL<<EXTI_IMR2_IMR33_Pos)                      /*!< 0x00000002 */
#define EXTI_IMR2_IMR33                           EXTI_IMR2_IMR33_Msk                               
#define EXTI_IMR2_IMR34_Pos                       (2U)
#define EXTI_IMR2_IMR34_Msk                       (0x1UL<<EXTI_IMR2_IMR34_Pos)                      /*!< 0x00000004 */
#define EXTI_IMR2_IMR34                           EXTI_IMR2_IMR34_Msk                               
#define EXTI_IMR2_IMR_Pos                         (0U)
#define EXTI_IMR2_IMR_Msk                         (0x7UL<<EXTI_IMR2_IMR_Pos)                        /*!< 0x00000007 */
#define EXTI_IMR2_IMR                             EXTI_IMR2_IMR_Msk                               

/********************************* Bit definition for EXTI_EMR2 register ********************************************/
#define EXTI_EMR2_EMR32_Pos                       (0U)
#define EXTI_EMR2_EMR32_Msk                       (0x1UL<<EXTI_EMR2_EMR32_Pos)                      /*!< 0x00000001 */
#define EXTI_EMR2_EMR32                           EXTI_EMR2_EMR32_Msk                               
#define EXTI_EMR2_EMR33_Pos                       (1U)
#define EXTI_EMR2_EMR33_Msk                       (0x1UL<<EXTI_EMR2_EMR33_Pos)                      /*!< 0x00000002 */
#define EXTI_EMR2_EMR33                           EXTI_EMR2_EMR33_Msk                               
#define EXTI_EMR2_EMR34_Pos                       (2U)
#define EXTI_EMR2_EMR34_Msk                       (0x1UL<<EXTI_EMR2_EMR34_Pos)                      /*!< 0x00000004 */
#define EXTI_EMR2_EMR34                           EXTI_EMR2_EMR34_Msk                               

/********************************************************************************************************************/
/********************************* FDCAN ****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for FDCAN_ENDN register *******************************************/
#define FDCAN_ENDN_ETV_Pos                        (0U)
#define FDCAN_ENDN_ETV_Msk                        (0xFFFFFFFFUL<<FDCAN_ENDN_ETV_Pos)                /*!< 0xFFFFFFFF */
#define FDCAN_ENDN_ETV                            FDCAN_ENDN_ETV_Msk                                /*!< Endian Register */

/********************************* Bit definition for FDCAN_DBTP register *******************************************/
#define FDCAN_DBTP_DSJW_Pos                       (0U)
#define FDCAN_DBTP_DSJW_Msk                       (0xFUL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x0000000F */
#define FDCAN_DBTP_DSJW                           FDCAN_DBTP_DSJW_Msk                               /*!< Data (Re)Synchronization Jump Width */
#define FDCAN_DBTP_DSJW_0                         (0x1UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000001 */
#define FDCAN_DBTP_DSJW_1                         (0x2UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000002 */
#define FDCAN_DBTP_DSJW_2                         (0x4UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000004 */
#define FDCAN_DBTP_DSJW_3                         (0x8UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000008 */
#define FDCAN_DBTP_DTSEG2_Pos                     (4U)
#define FDCAN_DBTP_DTSEG2_Msk                     (0xFUL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x000000F0 */
#define FDCAN_DBTP_DTSEG2                         FDCAN_DBTP_DTSEG2_Msk                             /*!< Data time segment after sample point */
#define FDCAN_DBTP_DTSEG2_0                       (0x1UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000010 */
#define FDCAN_DBTP_DTSEG2_1                       (0x2UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000020 */
#define FDCAN_DBTP_DTSEG2_2                       (0x4UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000040 */
#define FDCAN_DBTP_DTSEG2_3                       (0x8UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000080 */
#define FDCAN_DBTP_DTSEG1_Pos                     (8U)
#define FDCAN_DBTP_DTSEG1_Msk                     (0x1FUL<<FDCAN_DBTP_DTSEG1_Pos)                   /*!< 0x00001F00 */
#define FDCAN_DBTP_DTSEG1                         FDCAN_DBTP_DTSEG1_Msk                             /*!< Data time segment before sample point */
#define FDCAN_DBTP_DTSEG1_0                       (0x1UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000100 */
#define FDCAN_DBTP_DTSEG1_1                       (0x2UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000200 */
#define FDCAN_DBTP_DTSEG1_2                       (0x4UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000400 */
#define FDCAN_DBTP_DTSEG1_3                       (0x8UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000800 */
#define FDCAN_DBTP_DTSEG1_4                       (0x10UL<<FDCAN_DBTP_DTSEG1_Pos)                   /*!< 0x00001000 */
#define FDCAN_DBTP_DBRP_Pos                       (16U)
#define FDCAN_DBTP_DBRP_Msk                       (0x1FUL<<FDCAN_DBTP_DBRP_Pos)                     /*!< 0x001F0000 */
#define FDCAN_DBTP_DBRP                           FDCAN_DBTP_DBRP_Msk                               /*!< Data Bit Rate Prescaler */
#define FDCAN_DBTP_DBRP_0                         (0x1UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00010000 */
#define FDCAN_DBTP_DBRP_1                         (0x2UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00020000 */
#define FDCAN_DBTP_DBRP_2                         (0x4UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00040000 */
#define FDCAN_DBTP_DBRP_3                         (0x8UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00080000 */
#define FDCAN_DBTP_DBRP_4                         (0x10UL<<FDCAN_DBTP_DBRP_Pos)                     /*!< 0x00100000 */
#define FDCAN_DBTP_TDC_Pos                        (23U)
#define FDCAN_DBTP_TDC_Msk                        (0x1UL<<FDCAN_DBTP_TDC_Pos)                       /*!< 0x00800000 */
#define FDCAN_DBTP_TDC                            FDCAN_DBTP_TDC_Msk                                /*!< Transmitter Delay Compensation */

/********************************* Bit definition for FDCAN_TEST register *******************************************/
#define FDCAN_TEST_LBCK_Pos                       (4U)
#define FDCAN_TEST_LBCK_Msk                       (0x1UL<<FDCAN_TEST_LBCK_Pos)                      /*!< 0x00000010 */
#define FDCAN_TEST_LBCK                           FDCAN_TEST_LBCK_Msk                               /*!< Loop Back Mode */
#define FDCAN_TEST_TX_Pos                         (5U)
#define FDCAN_TEST_TX_Msk                         (0x3UL<<FDCAN_TEST_TX_Pos)                        /*!< 0x00000060 */
#define FDCAN_TEST_TX                             FDCAN_TEST_TX_Msk                                 /*!< Control of Transmit Pin */
#define FDCAN_TEST_TX_0                           (0x1UL<<FDCAN_TEST_TX_Pos)                        /*!< 0x00000020 */
#define FDCAN_TEST_TX_1                           (0x2UL<<FDCAN_TEST_TX_Pos)                        /*!< 0x00000040 */
#define FDCAN_TEST_RX_Pos                         (7U)
#define FDCAN_TEST_RX_Msk                         (0x1UL<<FDCAN_TEST_RX_Pos)                        /*!< 0x00000080 */
#define FDCAN_TEST_RX                             FDCAN_TEST_RX_Msk                                 /*!< Receive Pin */
#define FDCAN_TEST_TXBNP_Pos                      (8U)
#define FDCAN_TEST_TXBNP_Msk                      (0x1FUL<<FDCAN_TEST_TXBNP_Pos)                    /*!< 0x00001F00 */
#define FDCAN_TEST_TXBNP                          FDCAN_TEST_TXBNP_Msk                              /*!< Tx Buffer Number Prepared */
#define FDCAN_TEST_TXBNP_0                        (0x1UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000100 */
#define FDCAN_TEST_TXBNP_1                        (0x2UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000200 */
#define FDCAN_TEST_TXBNP_2                        (0x4UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000400 */
#define FDCAN_TEST_TXBNP_3                        (0x8UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000800 */
#define FDCAN_TEST_TXBNP_4                        (0x10UL<<FDCAN_TEST_TXBNP_Pos)                    /*!< 0x00001000 */
#define FDCAN_TEST_PVAL_Pos                       (13U)
#define FDCAN_TEST_PVAL_Msk                       (0x1UL<<FDCAN_TEST_PVAL_Pos)                      /*!< 0x00002000 */
#define FDCAN_TEST_PVAL                           FDCAN_TEST_PVAL_Msk                               /*!< Prepared Valid */
#define FDCAN_TEST_TXBNS_Pos                      (16U)
#define FDCAN_TEST_TXBNS_Msk                      (0x1FUL<<FDCAN_TEST_TXBNS_Pos)                    /*!< 0x001F0000 */
#define FDCAN_TEST_TXBNS                          FDCAN_TEST_TXBNS_Msk                              /*!< Tx Buffer Number Started */
#define FDCAN_TEST_TXBNS_0                        (0x1UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00010000 */
#define FDCAN_TEST_TXBNS_1                        (0x2UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00020000 */
#define FDCAN_TEST_TXBNS_2                        (0x4UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00040000 */
#define FDCAN_TEST_TXBNS_3                        (0x8UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00080000 */
#define FDCAN_TEST_TXBNS_4                        (0x10UL<<FDCAN_TEST_TXBNS_Pos)                    /*!< 0x00100000 */
#define FDCAN_TEST_SVAL_Pos                       (21U)
#define FDCAN_TEST_SVAL_Msk                       (0x1UL<<FDCAN_TEST_SVAL_Pos)                      /*!< 0x00200000 */
#define FDCAN_TEST_SVAL                           FDCAN_TEST_SVAL_Msk                               /*!< Started Valid */

/********************************* Bit definition for FDCAN_RWD register ********************************************/
#define FDCAN_RWD_WDC_Pos                         (0U)
#define FDCAN_RWD_WDC_Msk                         (0xFFUL<<FDCAN_RWD_WDC_Pos)                       /*!< 0x000000FF */
#define FDCAN_RWD_WDC                             FDCAN_RWD_WDC_Msk                                 /*!< Watchdog Configuration */
#define FDCAN_RWD_WDV_Pos                         (8U)
#define FDCAN_RWD_WDV_Msk                         (0xFFUL<<FDCAN_RWD_WDV_Pos)                       /*!< 0x0000FF00 */
#define FDCAN_RWD_WDV                             FDCAN_RWD_WDV_Msk                                 /*!< Watchdog Value */

/********************************* Bit definition for FDCAN_CCCR register *******************************************/
#define FDCAN_CCCR_INIT_Pos                       (0U)
#define FDCAN_CCCR_INIT_Msk                       (0x1UL<<FDCAN_CCCR_INIT_Pos)                      /*!< 0x00000001 */
#define FDCAN_CCCR_INIT                           FDCAN_CCCR_INIT_Msk                               /*!< Initialization */
#define FDCAN_CCCR_CCE_Pos                        (1U)
#define FDCAN_CCCR_CCE_Msk                        (0x1UL<<FDCAN_CCCR_CCE_Pos)                       /*!< 0x00000002 */
#define FDCAN_CCCR_CCE                            FDCAN_CCCR_CCE_Msk                                /*!< Configuration Change Enable */
#define FDCAN_CCCR_ASM_Pos                        (2U)
#define FDCAN_CCCR_ASM_Msk                        (0x1UL<<FDCAN_CCCR_ASM_Pos)                       /*!< 0x00000004 */
#define FDCAN_CCCR_ASM                            FDCAN_CCCR_ASM_Msk                                /*!< Restricted Operation Mode */
#define FDCAN_CCCR_CSA_Pos                        (3U)
#define FDCAN_CCCR_CSA_Msk                        (0x1UL<<FDCAN_CCCR_CSA_Pos)                       /*!< 0x00000008 */
#define FDCAN_CCCR_CSA                            FDCAN_CCCR_CSA_Msk                                /*!< Clock Stop Acknowledge */
#define FDCAN_CCCR_CSR_Pos                        (4U)
#define FDCAN_CCCR_CSR_Msk                        (0x1UL<<FDCAN_CCCR_CSR_Pos)                       /*!< 0x00000010 */
#define FDCAN_CCCR_CSR                            FDCAN_CCCR_CSR_Msk                                /*!< Clock Stop Request */
#define FDCAN_CCCR_MON_Pos                        (5U)
#define FDCAN_CCCR_MON_Msk                        (0x1UL<<FDCAN_CCCR_MON_Pos)                       /*!< 0x00000020 */
#define FDCAN_CCCR_MON                            FDCAN_CCCR_MON_Msk                                /*!< Bus Monitoring Mode */
#define FDCAN_CCCR_DAR_Pos                        (6U)
#define FDCAN_CCCR_DAR_Msk                        (0x1UL<<FDCAN_CCCR_DAR_Pos)                       /*!< 0x00000040 */
#define FDCAN_CCCR_DAR                            FDCAN_CCCR_DAR_Msk                                /*!< Disable Automatic Retransmission */
#define FDCAN_CCCR_TEST_Pos                       (7U)
#define FDCAN_CCCR_TEST_Msk                       (0x1UL<<FDCAN_CCCR_TEST_Pos)                      /*!< 0x00000080 */
#define FDCAN_CCCR_TEST                           FDCAN_CCCR_TEST_Msk                               /*!< Test Mode Enable */
#define FDCAN_CCCR_FDOE_Pos                       (8U)
#define FDCAN_CCCR_FDOE_Msk                       (0x1UL<<FDCAN_CCCR_FDOE_Pos)                      /*!< 0x00000100 */
#define FDCAN_CCCR_FDOE                           FDCAN_CCCR_FDOE_Msk                               /*!< FD Operation Enable */
#define FDCAN_CCCR_BRSE_Pos                       (9U)
#define FDCAN_CCCR_BRSE_Msk                       (0x1UL<<FDCAN_CCCR_BRSE_Pos)                      /*!< 0x00000200 */
#define FDCAN_CCCR_BRSE                           FDCAN_CCCR_BRSE_Msk                               /*!< Bit Rate Switch Enable */
#define FDCAN_CCCR_WMM_Pos                        (11U)
#define FDCAN_CCCR_WMM_Msk                        (0x1UL<<FDCAN_CCCR_WMM_Pos)                       /*!< 0x00000800 */
#define FDCAN_CCCR_WMM                            FDCAN_CCCR_WMM_Msk                                /*!< Wide Message Marker */
#define FDCAN_CCCR_PXHD_Pos                       (12U)
#define FDCAN_CCCR_PXHD_Msk                       (0x1UL<<FDCAN_CCCR_PXHD_Pos)                      /*!< 0x00001000 */
#define FDCAN_CCCR_PXHD                           FDCAN_CCCR_PXHD_Msk                               /*!< Protocol Exception Handling Disable */
#define FDCAN_CCCR_EFBI_Pos                       (13U)
#define FDCAN_CCCR_EFBI_Msk                       (0x1UL<<FDCAN_CCCR_EFBI_Pos)                      /*!< 0x00002000 */
#define FDCAN_CCCR_EFBI                           FDCAN_CCCR_EFBI_Msk                               /*!< Edge Filtering during Bus Integration */
#define FDCAN_CCCR_TXP_Pos                        (14U)
#define FDCAN_CCCR_TXP_Msk                        (0x1UL<<FDCAN_CCCR_TXP_Pos)                       /*!< 0x00004000 */
#define FDCAN_CCCR_TXP                            FDCAN_CCCR_TXP_Msk                                /*!< Transmit Pause */
#define FDCAN_CCCR_NISO_Pos                       (15U)
#define FDCAN_CCCR_NISO_Msk                       (0x1UL<<FDCAN_CCCR_NISO_Pos)                      /*!< 0x00008000 */
#define FDCAN_CCCR_NISO                           FDCAN_CCCR_NISO_Msk                               /*!< Non ISO Operation */

/********************************* Bit definition for FDCAN_NBTP register *******************************************/
#define FDCAN_NBTP_NTSEG2_Pos                     (0U)
#define FDCAN_NBTP_NTSEG2_Msk                     (0x7FUL<<FDCAN_NBTP_NTSEG2_Pos)                   /*!< 0x0000007F */
#define FDCAN_NBTP_NTSEG2                         FDCAN_NBTP_NTSEG2_Msk                             /*!< Nominal Time segment after sample point */
#define FDCAN_NBTP_NTSEG1_Pos                     (8U)
#define FDCAN_NBTP_NTSEG1_Msk                     (0xFFUL<<FDCAN_NBTP_NTSEG1_Pos)                   /*!< 0x0000FF00 */
#define FDCAN_NBTP_NTSEG1                         FDCAN_NBTP_NTSEG1_Msk                             /*!< Nominal Time segment before sample point */
#define FDCAN_NBTP_NBRP_Pos                       (16U)
#define FDCAN_NBTP_NBRP_Msk                       (0x1FFUL<<FDCAN_NBTP_NBRP_Pos)                    /*!< 0x01FF0000 */
#define FDCAN_NBTP_NBRP                           FDCAN_NBTP_NBRP_Msk                               /*!< Nominal Bit Rate Prescaler */
#define FDCAN_NBTP_NSJW_Pos                       (25U)
#define FDCAN_NBTP_NSJW_Msk                       (0x7FUL<<FDCAN_NBTP_NSJW_Pos)                     /*!< 0xFE000000 */
#define FDCAN_NBTP_NSJW                           FDCAN_NBTP_NSJW_Msk                               /*!< Nominal (Re)Synchronization Jump Width */

/********************************* Bit definition for FDCAN_TSCC register *******************************************/
#define FDCAN_TSCC_TSS_Pos                        (0U)
#define FDCAN_TSCC_TSS_Msk                        (0x3UL<<FDCAN_TSCC_TSS_Pos)                       /*!< 0x00000003 */
#define FDCAN_TSCC_TSS                            FDCAN_TSCC_TSS_Msk                                /*!< Timestamp Select */
#define FDCAN_TSCC_TSS_0                          (0x1UL<<FDCAN_TSCC_TSS_Pos)                       /*!< 0x00000001 */
#define FDCAN_TSCC_TSS_1                          (0x2UL<<FDCAN_TSCC_TSS_Pos)                       /*!< 0x00000002 */
#define FDCAN_TSCC_TCP_Pos                        (16U)
#define FDCAN_TSCC_TCP_Msk                        (0xFUL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x000F0000 */
#define FDCAN_TSCC_TCP                            FDCAN_TSCC_TCP_Msk                                /*!< Timestamp Counter Prescaler */
#define FDCAN_TSCC_TCP_0                          (0x1UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00010000 */
#define FDCAN_TSCC_TCP_1                          (0x2UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00020000 */
#define FDCAN_TSCC_TCP_2                          (0x4UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00040000 */
#define FDCAN_TSCC_TCP_3                          (0x8UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00080000 */

/********************************* Bit definition for FDCAN_TSCV register *******************************************/
#define FDCAN_TSCV_TSC_Pos                        (0U)
#define FDCAN_TSCV_TSC_Msk                        (0xFFFFUL<<FDCAN_TSCV_TSC_Pos)                    /*!< 0x0000FFFF */
#define FDCAN_TSCV_TSC                            FDCAN_TSCV_TSC_Msk                                /*!< Timestamp Counter */

/********************************* Bit definition for FDCAN_TOCC register *******************************************/
#define FDCAN_TOCC_ETOC_Pos                       (0U)
#define FDCAN_TOCC_ETOC_Msk                       (0x1UL<<FDCAN_TOCC_ETOC_Pos)                      /*!< 0x00000001 */
#define FDCAN_TOCC_ETOC                           FDCAN_TOCC_ETOC_Msk                               /*!< Enable Timeout Counter */
#define FDCAN_TOCC_TOS_Pos                        (1U)
#define FDCAN_TOCC_TOS_Msk                        (0x3UL<<FDCAN_TOCC_TOS_Pos)                       /*!< 0x00000006 */
#define FDCAN_TOCC_TOS                            FDCAN_TOCC_TOS_Msk                                /*!< Timeout Select */
#define FDCAN_TOCC_TOS_0                          (0x1UL<<FDCAN_TOCC_TOS_Pos)                       /*!< 0x00000002 */
#define FDCAN_TOCC_TOS_1                          (0x2UL<<FDCAN_TOCC_TOS_Pos)                       /*!< 0x00000004 */
#define FDCAN_TOCC_TOP_Pos                        (16U)
#define FDCAN_TOCC_TOP_Msk                        (0xFFFFUL<<FDCAN_TOCC_TOP_Pos)                    /*!< 0xFFFF0000 */
#define FDCAN_TOCC_TOP                            FDCAN_TOCC_TOP_Msk                                /*!< Timeout Period */

/********************************* Bit definition for FDCAN_TOCV register *******************************************/
#define FDCAN_TOCV_TOC_Pos                        (0U)
#define FDCAN_TOCV_TOC_Msk                        (0xFFFFUL<<FDCAN_TOCV_TOC_Pos)                    /*!< 0x0000FFFF */
#define FDCAN_TOCV_TOC                            FDCAN_TOCV_TOC_Msk                                /*!< Timeout Counter */

/********************************* Bit definition for FDCAN_ECR register ********************************************/
#define FDCAN_ECR_TEC_Pos                         (0U)
#define FDCAN_ECR_TEC_Msk                         (0xFFUL<<FDCAN_ECR_TEC_Pos)                       /*!< 0x000000FF */
#define FDCAN_ECR_TEC                             FDCAN_ECR_TEC_Msk                                 /*!< Transmit Error Counter */
#define FDCAN_ECR_REC_Pos                         (8U)
#define FDCAN_ECR_REC_Msk                         (0x7FUL<<FDCAN_ECR_REC_Pos)                       /*!< 0x00007F00 */
#define FDCAN_ECR_REC                             FDCAN_ECR_REC_Msk                                 /*!< Receive Error Counter */
#define FDCAN_ECR_RP_Pos                          (15U)
#define FDCAN_ECR_RP_Msk                          (0x1UL<<FDCAN_ECR_RP_Pos)                         /*!< 0x00008000 */
#define FDCAN_ECR_RP                              FDCAN_ECR_RP_Msk                                  /*!< Receive Error Passive */
#define FDCAN_ECR_CEL_Pos                         (16U)
#define FDCAN_ECR_CEL_Msk                         (0xFFUL<<FDCAN_ECR_CEL_Pos)                       /*!< 0x00FF0000 */
#define FDCAN_ECR_CEL                             FDCAN_ECR_CEL_Msk                                 /*!< CAN Error Logging */

/********************************* Bit definition for FDCAN_PSR register ********************************************/
#define FDCAN_PSR_LEC_Pos                         (0U)
#define FDCAN_PSR_LEC_Msk                         (0x7UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000007 */
#define FDCAN_PSR_LEC                             FDCAN_PSR_LEC_Msk                                 /*!< Last Error Code */
#define FDCAN_PSR_LEC_0                           (0x1UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000001 */
#define FDCAN_PSR_LEC_1                           (0x2UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000002 */
#define FDCAN_PSR_LEC_2                           (0x4UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000004 */
#define FDCAN_PSR_ACT_Pos                         (3U)
#define FDCAN_PSR_ACT_Msk                         (0x3UL<<FDCAN_PSR_ACT_Pos)                        /*!< 0x00000018 */
#define FDCAN_PSR_ACT                             FDCAN_PSR_ACT_Msk                                 /*!< Activity */
#define FDCAN_PSR_ACT_0                           (0x1UL<<FDCAN_PSR_ACT_Pos)                        /*!< 0x00000008 */
#define FDCAN_PSR_ACT_1                           (0x2UL<<FDCAN_PSR_ACT_Pos)                        /*!< 0x00000010 */
#define FDCAN_PSR_EP_Pos                          (5U)
#define FDCAN_PSR_EP_Msk                          (0x1UL<<FDCAN_PSR_EP_Pos)                         /*!< 0x00000020 */
#define FDCAN_PSR_EP                              FDCAN_PSR_EP_Msk                                  /*!< Error Passive */
#define FDCAN_PSR_EW_Pos                          (6U)
#define FDCAN_PSR_EW_Msk                          (0x1UL<<FDCAN_PSR_EW_Pos)                         /*!< 0x00000040 */
#define FDCAN_PSR_EW                              FDCAN_PSR_EW_Msk                                  /*!< Warning Status */
#define FDCAN_PSR_BO_Pos                          (7U)
#define FDCAN_PSR_BO_Msk                          (0x1UL<<FDCAN_PSR_BO_Pos)                         /*!< 0x00000080 */
#define FDCAN_PSR_BO                              FDCAN_PSR_BO_Msk                                  /*!< Bus_Off Status */
#define FDCAN_PSR_DLEC_Pos                        (8U)
#define FDCAN_PSR_DLEC_Msk                        (0x7UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000700 */
#define FDCAN_PSR_DLEC                            FDCAN_PSR_DLEC_Msk                                /*!< Data Phase Last Error Code */
#define FDCAN_PSR_DLEC_0                          (0x1UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000100 */
#define FDCAN_PSR_DLEC_1                          (0x2UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000200 */
#define FDCAN_PSR_DLEC_2                          (0x4UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000400 */
#define FDCAN_PSR_RESI_Pos                        (11U)
#define FDCAN_PSR_RESI_Msk                        (0x1UL<<FDCAN_PSR_RESI_Pos)                       /*!< 0x00000800 */
#define FDCAN_PSR_RESI                            FDCAN_PSR_RESI_Msk                                /*!< ESI flag of last received CAN FD Message */
#define FDCAN_PSR_RBRS_Pos                        (12U)
#define FDCAN_PSR_RBRS_Msk                        (0x1UL<<FDCAN_PSR_RBRS_Pos)                       /*!< 0x00001000 */
#define FDCAN_PSR_RBRS                            FDCAN_PSR_RBRS_Msk                                /*!< BRS flag of last received CAN FD Message */
#define FDCAN_PSR_RFDF_Pos                        (13U)
#define FDCAN_PSR_RFDF_Msk                        (0x1UL<<FDCAN_PSR_RFDF_Pos)                       /*!< 0x00002000 */
#define FDCAN_PSR_RFDF                            FDCAN_PSR_RFDF_Msk                                /*!< Received a CAN FD Message */
#define FDCAN_PSR_PXE_Pos                         (14U)
#define FDCAN_PSR_PXE_Msk                         (0x1UL<<FDCAN_PSR_PXE_Pos)                        /*!< 0x00004000 */
#define FDCAN_PSR_PXE                             FDCAN_PSR_PXE_Msk                                 /*!< Protocol Exception Event */
#define FDCAN_PSR_TDCV_Pos                        (16U)
#define FDCAN_PSR_TDCV_Msk                        (0x7FUL<<FDCAN_PSR_TDCV_Pos)                      /*!< 0x007F0000 */
#define FDCAN_PSR_TDCV                            FDCAN_PSR_TDCV_Msk                                /*!< Transmitter Delay Compensation Value */

/********************************* Bit definition for FDCAN_TDCR register *******************************************/
#define FDCAN_TDCR_TDCF_Pos                       (0U)
#define FDCAN_TDCR_TDCF_Msk                       (0x7FUL<<FDCAN_TDCR_TDCF_Pos)                     /*!< 0x0000007F */
#define FDCAN_TDCR_TDCF                           FDCAN_TDCR_TDCF_Msk                               /*!< Transmitter Delay Compensation Filter Window Length */
#define FDCAN_TDCR_TDCO_Pos                       (8U)
#define FDCAN_TDCR_TDCO_Msk                       (0x7FUL<<FDCAN_TDCR_TDCO_Pos)                     /*!< 0x00007F00 */
#define FDCAN_TDCR_TDCO                           FDCAN_TDCR_TDCO_Msk                               /*!< Transmitter Delay Compensation SSP Offset */

/********************************* Bit definition for FDCAN_IR register *********************************************/
#define FDCAN_IR_RF0N_Pos                         (0U)
#define FDCAN_IR_RF0N_Msk                         (0x1UL<<FDCAN_IR_RF0N_Pos)                        /*!< 0x00000001 */
#define FDCAN_IR_RF0N                             FDCAN_IR_RF0N_Msk                                 /*!< Rx FIFO 0 New Message */
#define FDCAN_IR_RF0W_Pos                         (1U)
#define FDCAN_IR_RF0W_Msk                         (0x1UL<<FDCAN_IR_RF0W_Pos)                        /*!< 0x00000002 */
#define FDCAN_IR_RF0W                             FDCAN_IR_RF0W_Msk                                 /*!< Rx FIFO 0 Watermark Reached */
#define FDCAN_IR_RF0F_Pos                         (2U)
#define FDCAN_IR_RF0F_Msk                         (0x1UL<<FDCAN_IR_RF0F_Pos)                        /*!< 0x00000004 */
#define FDCAN_IR_RF0F                             FDCAN_IR_RF0F_Msk                                 /*!< Rx FIFO 0 Full */
#define FDCAN_IR_RF0L_Pos                         (3U)
#define FDCAN_IR_RF0L_Msk                         (0x1UL<<FDCAN_IR_RF0L_Pos)                        /*!< 0x00000008 */
#define FDCAN_IR_RF0L                             FDCAN_IR_RF0L_Msk                                 /*!< Rx FIFO 0 Message Lost */
#define FDCAN_IR_RF1N_Pos                         (4U)
#define FDCAN_IR_RF1N_Msk                         (0x1UL<<FDCAN_IR_RF1N_Pos)                        /*!< 0x00000010 */
#define FDCAN_IR_RF1N                             FDCAN_IR_RF1N_Msk                                 /*!< Rx FIFO 1 New Message */
#define FDCAN_IR_RF1W_Pos                         (5U)
#define FDCAN_IR_RF1W_Msk                         (0x1UL<<FDCAN_IR_RF1W_Pos)                        /*!< 0x00000020 */
#define FDCAN_IR_RF1W                             FDCAN_IR_RF1W_Msk                                 /*!< Rx FIFO 1 Watermark Reached */
#define FDCAN_IR_RF1F_Pos                         (6U)
#define FDCAN_IR_RF1F_Msk                         (0x1UL<<FDCAN_IR_RF1F_Pos)                        /*!< 0x00000040 */
#define FDCAN_IR_RF1F                             FDCAN_IR_RF1F_Msk                                 /*!< Rx FIFO 1 Full */
#define FDCAN_IR_RF1L_Pos                         (7U)
#define FDCAN_IR_RF1L_Msk                         (0x1UL<<FDCAN_IR_RF1L_Pos)                        /*!< 0x00000080 */
#define FDCAN_IR_RF1L                             FDCAN_IR_RF1L_Msk                                 /*!< Rx FIFO 1 Message Lost */
#define FDCAN_IR_HPM_Pos                          (8U)
#define FDCAN_IR_HPM_Msk                          (0x1UL<<FDCAN_IR_HPM_Pos)                         /*!< 0x00000100 */
#define FDCAN_IR_HPM                              FDCAN_IR_HPM_Msk                                  /*!< High Priority Message */
#define FDCAN_IR_TC_Pos                           (9U)
#define FDCAN_IR_TC_Msk                           (0x1UL<<FDCAN_IR_TC_Pos)                          /*!< 0x00000200 */
#define FDCAN_IR_TC                               FDCAN_IR_TC_Msk                                   /*!< Transmission Completed */
#define FDCAN_IR_TCF_Pos                          (10U)
#define FDCAN_IR_TCF_Msk                          (0x1UL<<FDCAN_IR_TCF_Pos)                         /*!< 0x00000400 */
#define FDCAN_IR_TCF                              FDCAN_IR_TCF_Msk                                  /*!< Transmission Cancellation Finished */
#define FDCAN_IR_TFE_Pos                          (11U)
#define FDCAN_IR_TFE_Msk                          (0x1UL<<FDCAN_IR_TFE_Pos)                         /*!< 0x00000800 */
#define FDCAN_IR_TFE                              FDCAN_IR_TFE_Msk                                  /*!< Tx FIFO Empty */
#define FDCAN_IR_TEFN_Pos                         (12U)
#define FDCAN_IR_TEFN_Msk                         (0x1UL<<FDCAN_IR_TEFN_Pos)                        /*!< 0x00001000 */
#define FDCAN_IR_TEFN                             FDCAN_IR_TEFN_Msk                                 /*!< Tx Event FIFO New Entry */
#define FDCAN_IR_TEFW_Pos                         (13U)
#define FDCAN_IR_TEFW_Msk                         (0x1UL<<FDCAN_IR_TEFW_Pos)                        /*!< 0x00002000 */
#define FDCAN_IR_TEFW                             FDCAN_IR_TEFW_Msk                                 /*!< Tx Event FIFO Watermark Reached */
#define FDCAN_IR_TEFF_Pos                         (14U)
#define FDCAN_IR_TEFF_Msk                         (0x1UL<<FDCAN_IR_TEFF_Pos)                        /*!< 0x00004000 */
#define FDCAN_IR_TEFF                             FDCAN_IR_TEFF_Msk                                 /*!< Tx Event FIFO Full */
#define FDCAN_IR_TEFL_Pos                         (15U)
#define FDCAN_IR_TEFL_Msk                         (0x1UL<<FDCAN_IR_TEFL_Pos)                        /*!< 0x00008000 */
#define FDCAN_IR_TEFL                             FDCAN_IR_TEFL_Msk                                 /*!< Tx Event FIFO Element Lost */
#define FDCAN_IR_TSW_Pos                          (16U)
#define FDCAN_IR_TSW_Msk                          (0x1UL<<FDCAN_IR_TSW_Pos)                         /*!< 0x00010000 */
#define FDCAN_IR_TSW                              FDCAN_IR_TSW_Msk                                  /*!< Timestamp Wraparound */
#define FDCAN_IR_MRAF_Pos                         (17U)
#define FDCAN_IR_MRAF_Msk                         (0x1UL<<FDCAN_IR_MRAF_Pos)                        /*!< 0x00020000 */
#define FDCAN_IR_MRAF                             FDCAN_IR_MRAF_Msk                                 /*!< Message RAM Access Failure */
#define FDCAN_IR_TOO_Pos                          (18U)
#define FDCAN_IR_TOO_Msk                          (0x1UL<<FDCAN_IR_TOO_Pos)                         /*!< 0x00040000 */
#define FDCAN_IR_TOO                              FDCAN_IR_TOO_Msk                                  /*!< Timeout Occurred */
#define FDCAN_IR_DRX_Pos                          (19U)
#define FDCAN_IR_DRX_Msk                          (0x1UL<<FDCAN_IR_DRX_Pos)                         /*!< 0x00080000 */
#define FDCAN_IR_DRX                              FDCAN_IR_DRX_Msk                                  /*!< Message stored to Dedicated Rx Buffer */
#define FDCAN_IR_ELO_Pos                          (22U)
#define FDCAN_IR_ELO_Msk                          (0x1UL<<FDCAN_IR_ELO_Pos)                         /*!< 0x00400000 */
#define FDCAN_IR_ELO                              FDCAN_IR_ELO_Msk                                  /*!< Error Logging Overflow */
#define FDCAN_IR_EP_Pos                           (23U)
#define FDCAN_IR_EP_Msk                           (0x1UL<<FDCAN_IR_EP_Pos)                          /*!< 0x00800000 */
#define FDCAN_IR_EP                               FDCAN_IR_EP_Msk                                   /*!< Error Passive */
#define FDCAN_IR_EW_Pos                           (24U)
#define FDCAN_IR_EW_Msk                           (0x1UL<<FDCAN_IR_EW_Pos)                          /*!< 0x01000000 */
#define FDCAN_IR_EW                               FDCAN_IR_EW_Msk                                   /*!< Warning Status */
#define FDCAN_IR_BO_Pos                           (25U)
#define FDCAN_IR_BO_Msk                           (0x1UL<<FDCAN_IR_BO_Pos)                          /*!< 0x02000000 */
#define FDCAN_IR_BO                               FDCAN_IR_BO_Msk                                   /*!< Bus_Off Status */
#define FDCAN_IR_WDI_Pos                          (26U)
#define FDCAN_IR_WDI_Msk                          (0x1UL<<FDCAN_IR_WDI_Pos)                         /*!< 0x04000000 */
#define FDCAN_IR_WDI                              FDCAN_IR_WDI_Msk                                  /*!< Watchdog Interrupt */
#define FDCAN_IR_PEA_Pos                          (27U)
#define FDCAN_IR_PEA_Msk                          (0x1UL<<FDCAN_IR_PEA_Pos)                         /*!< 0x08000000 */
#define FDCAN_IR_PEA                              FDCAN_IR_PEA_Msk                                  /*!< Protocol Error in Arbitration Phase  */
#define FDCAN_IR_PED_Pos                          (28U)
#define FDCAN_IR_PED_Msk                          (0x1UL<<FDCAN_IR_PED_Pos)                         /*!< 0x10000000 */
#define FDCAN_IR_PED                              FDCAN_IR_PED_Msk                                  /*!< Protocol Error in Data Phase (Data Bit Time is used) */
#define FDCAN_IR_ARA_Pos                          (29U)
#define FDCAN_IR_ARA_Msk                          (0x1UL<<FDCAN_IR_ARA_Pos)                         /*!< 0x20000000 */
#define FDCAN_IR_ARA                              FDCAN_IR_ARA_Msk                                  /*!< Access to Reserved Address */

/********************************* Bit definition for FDCAN_IE register *********************************************/
#define FDCAN_IE_RF0NE_Pos                        (0U)
#define FDCAN_IE_RF0NE_Msk                        (0x1UL<<FDCAN_IE_RF0NE_Pos)                       /*!< 0x00000001 */
#define FDCAN_IE_RF0NE                            FDCAN_IE_RF0NE_Msk                                /*!< Rx FIFO 1 New Message Interrupt Enable */
#define FDCAN_IE_RF0WE_Pos                        (1U)
#define FDCAN_IE_RF0WE_Msk                        (0x1UL<<FDCAN_IE_RF0WE_Pos)                       /*!< 0x00000002 */
#define FDCAN_IE_RF0WE                            FDCAN_IE_RF0WE_Msk                                /*!< Rx FIFO 1 Watermark Reached Interrupt Enable */
#define FDCAN_IE_RF0FE_Pos                        (2U)
#define FDCAN_IE_RF0FE_Msk                        (0x1UL<<FDCAN_IE_RF0FE_Pos)                       /*!< 0x00000004 */
#define FDCAN_IE_RF0FE                            FDCAN_IE_RF0FE_Msk                                /*!< Rx FIFO 1 Full Interrupt Enable */
#define FDCAN_IE_RF0LE_Pos                        (3U)
#define FDCAN_IE_RF0LE_Msk                        (0x1UL<<FDCAN_IE_RF0LE_Pos)                       /*!< 0x00000008 */
#define FDCAN_IE_RF0LE                            FDCAN_IE_RF0LE_Msk                                /*!< Rx FIFO 1 Message Lost Interrupt Enable */
#define FDCAN_IE_RF1NE_Pos                        (4U)
#define FDCAN_IE_RF1NE_Msk                        (0x1UL<<FDCAN_IE_RF1NE_Pos)                       /*!< 0x00000010 */
#define FDCAN_IE_RF1NE                            FDCAN_IE_RF1NE_Msk                                /*!< Rx FIFO 2 New Message Interrupt Enable */
#define FDCAN_IE_RF1WE_Pos                        (5U)
#define FDCAN_IE_RF1WE_Msk                        (0x1UL<<FDCAN_IE_RF1WE_Pos)                       /*!< 0x00000020 */
#define FDCAN_IE_RF1WE                            FDCAN_IE_RF1WE_Msk                                /*!< Rx FIFO 2 Watermark Reached Interrupt Enable */
#define FDCAN_IE_RF1FE_Pos                        (6U)
#define FDCAN_IE_RF1FE_Msk                        (0x1UL<<FDCAN_IE_RF1FE_Pos)                       /*!< 0x00000040 */
#define FDCAN_IE_RF1FE                            FDCAN_IE_RF1FE_Msk                                /*!< Rx FIFO 2 Full Interrupt Enable */
#define FDCAN_IE_RF1LE_Pos                        (7U)
#define FDCAN_IE_RF1LE_Msk                        (0x1UL<<FDCAN_IE_RF1LE_Pos)                       /*!< 0x00000080 */
#define FDCAN_IE_RF1LE                            FDCAN_IE_RF1LE_Msk                                /*!< Rx FIFO 2 Message Lost Interrupt Enable */
#define FDCAN_IE_HPME_Pos                         (8U)
#define FDCAN_IE_HPME_Msk                         (0x1UL<<FDCAN_IE_HPME_Pos)                        /*!< 0x00000100 */
#define FDCAN_IE_HPME                             FDCAN_IE_HPME_Msk                                 /*!< High Priority Message Interrupt Enable */
#define FDCAN_IE_TCE_Pos                          (9U)
#define FDCAN_IE_TCE_Msk                          (0x1UL<<FDCAN_IE_TCE_Pos)                         /*!< 0x00000200 */
#define FDCAN_IE_TCE                              FDCAN_IE_TCE_Msk                                  /*!< Transmission Completed Interrupt Enable */
#define FDCAN_IE_TCFE_Pos                         (10U)
#define FDCAN_IE_TCFE_Msk                         (0x1UL<<FDCAN_IE_TCFE_Pos)                        /*!< 0x00000400 */
#define FDCAN_IE_TCFE                             FDCAN_IE_TCFE_Msk                                 /*!< Transmission Cancellation Finished Interrupt Enable */
#define FDCAN_IE_TFEE_Pos                         (11U)
#define FDCAN_IE_TFEE_Msk                         (0x1UL<<FDCAN_IE_TFEE_Pos)                        /*!< 0x00000800 */
#define FDCAN_IE_TFEE                             FDCAN_IE_TFEE_Msk                                 /*!< Tx FIFO Empty Interrupt Enable */
#define FDCAN_IE_TEFNE_Pos                        (12U)
#define FDCAN_IE_TEFNE_Msk                        (0x1UL<<FDCAN_IE_TEFNE_Pos)                       /*!< 0x00001000 */
#define FDCAN_IE_TEFNE                            FDCAN_IE_TEFNE_Msk                                /*!< Tx Event FIFO New Entry Interrupt Enable */
#define FDCAN_IE_TEFWE_Pos                        (13U)
#define FDCAN_IE_TEFWE_Msk                        (0x1UL<<FDCAN_IE_TEFWE_Pos)                       /*!< 0x00002000 */
#define FDCAN_IE_TEFWE                            FDCAN_IE_TEFWE_Msk                                /*!< Tx Event FIFO Watermark Reached Interrupt Enable */
#define FDCAN_IE_TEFFE_Pos                        (14U)
#define FDCAN_IE_TEFFE_Msk                        (0x1UL<<FDCAN_IE_TEFFE_Pos)                       /*!< 0x00004000 */
#define FDCAN_IE_TEFFE                            FDCAN_IE_TEFFE_Msk                                /*!< Tx Event FIFO Full Interrupt Enable */
#define FDCAN_IE_TEFLE_Pos                        (15U)
#define FDCAN_IE_TEFLE_Msk                        (0x1UL<<FDCAN_IE_TEFLE_Pos)                       /*!< 0x00008000 */
#define FDCAN_IE_TEFLE                            FDCAN_IE_TEFLE_Msk                                /*!< Tx Event FIFO Event Lost Interrupt Enable */
#define FDCAN_IE_TSWE_Pos                         (16U)
#define FDCAN_IE_TSWE_Msk                         (0x1UL<<FDCAN_IE_TSWE_Pos)                        /*!< 0x00010000 */
#define FDCAN_IE_TSWE                             FDCAN_IE_TSWE_Msk                                 /*!< Timestamp Wraparound Interrupt Enable */
#define FDCAN_IE_MRAFE_Pos                        (17U)
#define FDCAN_IE_MRAFE_Msk                        (0x1UL<<FDCAN_IE_MRAFE_Pos)                       /*!< 0x00020000 */
#define FDCAN_IE_MRAFE                            FDCAN_IE_MRAFE_Msk                                /*!< Message RAM Access Failure Interrupt Enable */
#define FDCAN_IE_TOOE_Pos                         (18U)
#define FDCAN_IE_TOOE_Msk                         (0x1UL<<FDCAN_IE_TOOE_Pos)                        /*!< 0x00040000 */
#define FDCAN_IE_TOOE                             FDCAN_IE_TOOE_Msk                                 /*!< Timeout Occurred Interrupt Enable */
#define FDCAN_IE_DRXE_Pos                         (19U)
#define FDCAN_IE_DRXE_Msk                         (0x1UL<<FDCAN_IE_DRXE_Pos)                        /*!< 0x00080000 */
#define FDCAN_IE_DRXE                             FDCAN_IE_DRXE_Msk                                 /*!< Message stored to Dedicated Rx Buffer Interrupt Enable */
#define FDCAN_IE_ELOE_Pos                         (22U)
#define FDCAN_IE_ELOE_Msk                         (0x1UL<<FDCAN_IE_ELOE_Pos)                        /*!< 0x00400000 */
#define FDCAN_IE_ELOE                             FDCAN_IE_ELOE_Msk                                 /*!< Error Logging Overflow Interrupt Enable */
#define FDCAN_IE_EPE_Pos                          (23U)
#define FDCAN_IE_EPE_Msk                          (0x1UL<<FDCAN_IE_EPE_Pos)                         /*!< 0x00800000 */
#define FDCAN_IE_EPE                              FDCAN_IE_EPE_Msk                                  /*!< Error Passive Interrupt Enable */
#define FDCAN_IE_EWE_Pos                          (24U)
#define FDCAN_IE_EWE_Msk                          (0x1UL<<FDCAN_IE_EWE_Pos)                         /*!< 0x01000000 */
#define FDCAN_IE_EWE                              FDCAN_IE_EWE_Msk                                  /*!< Warning Status Interrupt Enable */
#define FDCAN_IE_BOE_Pos                          (25U)
#define FDCAN_IE_BOE_Msk                          (0x1UL<<FDCAN_IE_BOE_Pos)                         /*!< 0x02000000 */
#define FDCAN_IE_BOE                              FDCAN_IE_BOE_Msk                                  /*!< Bus_Off Status Interrupt Enable */
#define FDCAN_IE_WDIE_Pos                         (26U)
#define FDCAN_IE_WDIE_Msk                         (0x1UL<<FDCAN_IE_WDIE_Pos)                        /*!< 0x04000000 */
#define FDCAN_IE_WDIE                             FDCAN_IE_WDIE_Msk                                 /*!< Watchdog Interrupt Enable */
#define FDCAN_IE_PEAE_Pos                         (27U)
#define FDCAN_IE_PEAE_Msk                         (0x1UL<<FDCAN_IE_PEAE_Pos)                        /*!< 0x08000000 */
#define FDCAN_IE_PEAE                             FDCAN_IE_PEAE_Msk                                 /*!< Protocol Error in Arbitration Phase Enable */
#define FDCAN_IE_PEDE_Pos                         (28U)
#define FDCAN_IE_PEDE_Msk                         (0x1UL<<FDCAN_IE_PEDE_Pos)                        /*!< 0x10000000 */
#define FDCAN_IE_PEDE                             FDCAN_IE_PEDE_Msk                                 /*!< Protocol Error in Data Phase Enable */
#define FDCAN_IE_ARAE_Pos                         (29U)
#define FDCAN_IE_ARAE_Msk                         (0x1UL<<FDCAN_IE_ARAE_Pos)                        /*!< 0x20000000 */
#define FDCAN_IE_ARAE                             FDCAN_IE_ARAE_Msk                                 /*!< Access to Reserved Address Enable */

/********************************* Bit definition for FDCAN_ILS register ********************************************/
#define FDCAN_ILS_RF0NL_Pos                       (0U)
#define FDCAN_ILS_RF0NL_Msk                       (0x1UL<<FDCAN_ILS_RF0NL_Pos)                      /*!< 0x00000001 */
#define FDCAN_ILS_RF0NL                           FDCAN_ILS_RF0NL_Msk                               /*!< Rx FIFO 1 New Message Interrupt Line */
#define FDCAN_ILS_RF0WL_Pos                       (1U)
#define FDCAN_ILS_RF0WL_Msk                       (0x1UL<<FDCAN_ILS_RF0WL_Pos)                      /*!< 0x00000002 */
#define FDCAN_ILS_RF0WL                           FDCAN_ILS_RF0WL_Msk                               /*!< Rx FIFO 1 Watermark Reached Interrupt Line */
#define FDCAN_ILS_RF0FL_Pos                       (2U)
#define FDCAN_ILS_RF0FL_Msk                       (0x1UL<<FDCAN_ILS_RF0FL_Pos)                      /*!< 0x00000004 */
#define FDCAN_ILS_RF0FL                           FDCAN_ILS_RF0FL_Msk                               /*!< Rx FIFO 1 Full Interrupt Line */
#define FDCAN_ILS_RF0LL_Pos                       (3U)
#define FDCAN_ILS_RF0LL_Msk                       (0x1UL<<FDCAN_ILS_RF0LL_Pos)                      /*!< 0x00000008 */
#define FDCAN_ILS_RF0LL                           FDCAN_ILS_RF0LL_Msk                               /*!< Rx FIFO 1 Message Lost Interrupt Line */
#define FDCAN_ILS_RF1NL_Pos                       (4U)
#define FDCAN_ILS_RF1NL_Msk                       (0x1UL<<FDCAN_ILS_RF1NL_Pos)                      /*!< 0x00000010 */
#define FDCAN_ILS_RF1NL                           FDCAN_ILS_RF1NL_Msk                               /*!< Rx FIFO 2 New Message Interrupt Line */
#define FDCAN_ILS_RF1WL_Pos                       (5U)
#define FDCAN_ILS_RF1WL_Msk                       (0x1UL<<FDCAN_ILS_RF1WL_Pos)                      /*!< 0x00000020 */
#define FDCAN_ILS_RF1WL                           FDCAN_ILS_RF1WL_Msk                               /*!< Rx FIFO 2 Watermark Reached Interrupt Line */
#define FDCAN_ILS_RF1FL_Pos                       (6U)
#define FDCAN_ILS_RF1FL_Msk                       (0x1UL<<FDCAN_ILS_RF1FL_Pos)                      /*!< 0x00000040 */
#define FDCAN_ILS_RF1FL                           FDCAN_ILS_RF1FL_Msk                               /*!< Rx FIFO 2 Full Interrupt Line */
#define FDCAN_ILS_RF1LL_Pos                       (7U)
#define FDCAN_ILS_RF1LL_Msk                       (0x1UL<<FDCAN_ILS_RF1LL_Pos)                      /*!< 0x00000080 */
#define FDCAN_ILS_RF1LL                           FDCAN_ILS_RF1LL_Msk                               /*!< Rx FIFO 2 Message Lost Interrupt Line */
#define FDCAN_ILS_HPML_Pos                        (8U)
#define FDCAN_ILS_HPML_Msk                        (0x1UL<<FDCAN_ILS_HPML_Pos)                       /*!< 0x00000100 */
#define FDCAN_ILS_HPML                            FDCAN_ILS_HPML_Msk                                /*!< High Priority Message Interrupt Line */
#define FDCAN_ILS_TCL_Pos                         (9U)
#define FDCAN_ILS_TCL_Msk                         (0x1UL<<FDCAN_ILS_TCL_Pos)                        /*!< 0x00000200 */
#define FDCAN_ILS_TCL                             FDCAN_ILS_TCL_Msk                                 /*!< Transmission Completed Interrupt Line */
#define FDCAN_ILS_TCFL_Pos                        (10U)
#define FDCAN_ILS_TCFL_Msk                        (0x1UL<<FDCAN_ILS_TCFL_Pos)                       /*!< 0x00000400 */
#define FDCAN_ILS_TCFL                            FDCAN_ILS_TCFL_Msk                                /*!< Transmission Cancellation Finished Interrupt Line */
#define FDCAN_ILS_TFEL_Pos                        (11U)
#define FDCAN_ILS_TFEL_Msk                        (0x1UL<<FDCAN_ILS_TFEL_Pos)                       /*!< 0x00000800 */
#define FDCAN_ILS_TFEL                            FDCAN_ILS_TFEL_Msk                                /*!< Tx FIFO Empty Interrupt Line */
#define FDCAN_ILS_TEFNL_Pos                       (12U)
#define FDCAN_ILS_TEFNL_Msk                       (0x1UL<<FDCAN_ILS_TEFNL_Pos)                      /*!< 0x00001000 */
#define FDCAN_ILS_TEFNL                           FDCAN_ILS_TEFNL_Msk                               /*!< Tx Event FIFO New Entry Interrupt Line */
#define FDCAN_ILS_TEFWL_Pos                       (13U)
#define FDCAN_ILS_TEFWL_Msk                       (0x1UL<<FDCAN_ILS_TEFWL_Pos)                      /*!< 0x00002000 */
#define FDCAN_ILS_TEFWL                           FDCAN_ILS_TEFWL_Msk                               /*!< Tx Event FIFO Watermark Reached Interrupt Line */
#define FDCAN_ILS_TEFFL_Pos                       (14U)
#define FDCAN_ILS_TEFFL_Msk                       (0x1UL<<FDCAN_ILS_TEFFL_Pos)                      /*!< 0x00004000 */
#define FDCAN_ILS_TEFFL                           FDCAN_ILS_TEFFL_Msk                               /*!< Tx Event FIFO Full Interrupt Line */
#define FDCAN_ILS_TEFLL_Pos                       (15U)
#define FDCAN_ILS_TEFLL_Msk                       (0x1UL<<FDCAN_ILS_TEFLL_Pos)                      /*!< 0x00008000 */
#define FDCAN_ILS_TEFLL                           FDCAN_ILS_TEFLL_Msk                               /*!< Tx Event FIFO Event Lost Interrupt Line */
#define FDCAN_ILS_TSWL_Pos                        (16U)
#define FDCAN_ILS_TSWL_Msk                        (0x1UL<<FDCAN_ILS_TSWL_Pos)                       /*!< 0x00010000 */
#define FDCAN_ILS_TSWL                            FDCAN_ILS_TSWL_Msk                                /*!< Timestamp Wraparound Interrupt Line */
#define FDCAN_ILS_MRAFL_Pos                       (17U)
#define FDCAN_ILS_MRAFL_Msk                       (0x1UL<<FDCAN_ILS_MRAFL_Pos)                      /*!< 0x00020000 */
#define FDCAN_ILS_MRAFL                           FDCAN_ILS_MRAFL_Msk                               /*!< Message RAM Access Failure Interrupt Line */
#define FDCAN_ILS_TOOL_Pos                        (18U)
#define FDCAN_ILS_TOOL_Msk                        (0x1UL<<FDCAN_ILS_TOOL_Pos)                       /*!< 0x00040000 */
#define FDCAN_ILS_TOOL                            FDCAN_ILS_TOOL_Msk                                /*!< Timeout Occurred Interrupt Line */
#define FDCAN_ILS_DRXL_Pos                        (19U)
#define FDCAN_ILS_DRXL_Msk                        (0x1UL<<FDCAN_ILS_DRXL_Pos)                       /*!< 0x00080000 */
#define FDCAN_ILS_DRXL                            FDCAN_ILS_DRXL_Msk                                /*!< Message stored to Dedicated Rx Buffer Interrupt Line */
#define FDCAN_ILS_ELOL_Pos                        (22U)
#define FDCAN_ILS_ELOL_Msk                        (0x1UL<<FDCAN_ILS_ELOL_Pos)                       /*!< 0x00400000 */
#define FDCAN_ILS_ELOL                            FDCAN_ILS_ELOL_Msk                                /*!< Error Logging Overflow Interrupt Line */
#define FDCAN_ILS_EPL_Pos                         (23U)
#define FDCAN_ILS_EPL_Msk                         (0x1UL<<FDCAN_ILS_EPL_Pos)                        /*!< 0x00800000 */
#define FDCAN_ILS_EPL                             FDCAN_ILS_EPL_Msk                                 /*!< Error Passive Interrupt Line */
#define FDCAN_ILS_EWL_Pos                         (24U)
#define FDCAN_ILS_EWL_Msk                         (0x1UL<<FDCAN_ILS_EWL_Pos)                        /*!< 0x01000000 */
#define FDCAN_ILS_EWL                             FDCAN_ILS_EWL_Msk                                 /*!< Warning Status Interrupt Line */
#define FDCAN_ILS_BOL_Pos                         (25U)
#define FDCAN_ILS_BOL_Msk                         (0x1UL<<FDCAN_ILS_BOL_Pos)                        /*!< 0x02000000 */
#define FDCAN_ILS_BOL                             FDCAN_ILS_BOL_Msk                                 /*!< Bus_Off Status Interrupt Line */
#define FDCAN_ILS_WDIL_Pos                        (26U)
#define FDCAN_ILS_WDIL_Msk                        (0x1UL<<FDCAN_ILS_WDIL_Pos)                       /*!< 0x04000000 */
#define FDCAN_ILS_WDIL                            FDCAN_ILS_WDIL_Msk                                /*!< Watchdog Interrupt Line */
#define FDCAN_ILS_PEAL_Pos                        (27U)
#define FDCAN_ILS_PEAL_Msk                        (0x1UL<<FDCAN_ILS_PEAL_Pos)                       /*!< 0x08000000 */
#define FDCAN_ILS_PEAL                            FDCAN_ILS_PEAL_Msk                                /*!< Protocol Error in Arbitration Phase Line */
#define FDCAN_ILS_PEDL_Pos                        (28U)
#define FDCAN_ILS_PEDL_Msk                        (0x1UL<<FDCAN_ILS_PEDL_Pos)                       /*!< 0x10000000 */
#define FDCAN_ILS_PEDL                            FDCAN_ILS_PEDL_Msk                                /*!< Protocol Error in Data Phase Line */
#define FDCAN_ILS_ARAL_Pos                        (29U)
#define FDCAN_ILS_ARAL_Msk                        (0x1UL<<FDCAN_ILS_ARAL_Pos)                       /*!< 0x20000000 */
#define FDCAN_ILS_ARAL                            FDCAN_ILS_ARAL_Msk                                /*!< Access to Reserved Address Line */

/********************************* Bit definition for FDCAN_ILE register ********************************************/
#define FDCAN_ILE_EINT0_Pos                       (0U)
#define FDCAN_ILE_EINT0_Msk                       (0x1UL<<FDCAN_ILE_EINT0_Pos)                      /*!< 0x00000001 */
#define FDCAN_ILE_EINT0                           FDCAN_ILE_EINT0_Msk                               /*!< Enable Interrupt Line 0 */
#define FDCAN_ILE_EINT1_Pos                       (1U)
#define FDCAN_ILE_EINT1_Msk                       (0x1UL<<FDCAN_ILE_EINT1_Pos)                      /*!< 0x00000002 */
#define FDCAN_ILE_EINT1                           FDCAN_ILE_EINT1_Msk                               /*!< Enable Interrupt Line 1 */

/********************************* Bit definition for FDCAN_GFC register ********************************************/
#define FDCAN_GFC_RRFE_Pos                        (0U)
#define FDCAN_GFC_RRFE_Msk                        (0x1UL<<FDCAN_GFC_RRFE_Pos)                       /*!< 0x00000001 */
#define FDCAN_GFC_RRFE                            FDCAN_GFC_RRFE_Msk                                /*!< Reject Remote Frames Extended */
#define FDCAN_GFC_RRFS_Pos                        (1U)
#define FDCAN_GFC_RRFS_Msk                        (0x1UL<<FDCAN_GFC_RRFS_Pos)                       /*!< 0x00000002 */
#define FDCAN_GFC_RRFS                            FDCAN_GFC_RRFS_Msk                                /*!< Reject Remote Frames Standard */
#define FDCAN_GFC_ANFE_Pos                        (2U)
#define FDCAN_GFC_ANFE_Msk                        (0x3UL<<FDCAN_GFC_ANFE_Pos)                       /*!< 0x0000000C */
#define FDCAN_GFC_ANFE                            FDCAN_GFC_ANFE_Msk                                /*!< Accept Non-matching Frames Extended */
#define FDCAN_GFC_ANFE_0                          (0x1UL<<FDCAN_GFC_ANFE_Pos)                       /*!< 0x00000004 */
#define FDCAN_GFC_ANFE_1                          (0x2UL<<FDCAN_GFC_ANFE_Pos)                       /*!< 0x00000008 */
#define FDCAN_GFC_ANFS_Pos                        (4U)
#define FDCAN_GFC_ANFS_Msk                        (0x3UL<<FDCAN_GFC_ANFS_Pos)                       /*!< 0x00000030 */
#define FDCAN_GFC_ANFS                            FDCAN_GFC_ANFS_Msk                                /*!< Accept Non-matching Frames Standard */
#define FDCAN_GFC_ANFS_0                          (0x1UL<<FDCAN_GFC_ANFS_Pos)                       /*!< 0x00000010 */
#define FDCAN_GFC_ANFS_1                          (0x2UL<<FDCAN_GFC_ANFS_Pos)                       /*!< 0x00000020 */

/********************************* Bit definition for FDCAN_SIDFC register ******************************************/
#define FDCAN_SIDFC_FLSSA_Pos                     (2U)
#define FDCAN_SIDFC_FLSSA_Msk                     (0x3FFFUL<<FDCAN_SIDFC_FLSSA_Pos)                 /*!< 0x0000FFFC */
#define FDCAN_SIDFC_FLSSA                         FDCAN_SIDFC_FLSSA_Msk                             /*!< Filter List Standard Start Address */
#define FDCAN_SIDFC_LSS_Pos                       (16U)
#define FDCAN_SIDFC_LSS_Msk                       (0xFFUL<<FDCAN_SIDFC_LSS_Pos)                     /*!< 0x00FF0000 */
#define FDCAN_SIDFC_LSS                           FDCAN_SIDFC_LSS_Msk                               /*!< List Size Standard */

/********************************* Bit definition for FDCAN_XIDFC register ******************************************/
#define FDCAN_XIDFC_FLESA_Pos                     (2U)
#define FDCAN_XIDFC_FLESA_Msk                     (0x3FFFUL<<FDCAN_XIDFC_FLESA_Pos)                 /*!< 0x0000FFFC */
#define FDCAN_XIDFC_FLESA                         FDCAN_XIDFC_FLESA_Msk                             /*!< Filter List Extended Start Address */
#define FDCAN_XIDFC_LSE_Pos                       (16U)
#define FDCAN_XIDFC_LSE_Msk                       (0x7FUL<<FDCAN_XIDFC_LSE_Pos)                     /*!< 0x007F0000 */
#define FDCAN_XIDFC_LSE                           FDCAN_XIDFC_LSE_Msk                               /*!<  List Size Extended */

/********************************* Bit definition for FDCAN_XIDAM register ******************************************/
#define FDCAN_XIDAM_EIDM_Pos                      (0U)
#define FDCAN_XIDAM_EIDM_Msk                      (0x1FFFFFFFUL<<FDCAN_XIDAM_EIDM_Pos)              /*!< 0x1FFFFFFF */
#define FDCAN_XIDAM_EIDM                          FDCAN_XIDAM_EIDM_Msk                              /*!< Extended ID Mask */

/********************************* Bit definition for FDCAN_HPMS register *******************************************/
#define FDCAN_HPMS_BIDX_Pos                       (0U)
#define FDCAN_HPMS_BIDX_Msk                       (0x3FUL<<FDCAN_HPMS_BIDX_Pos)                     /*!< 0x0000003F */
#define FDCAN_HPMS_BIDX                           FDCAN_HPMS_BIDX_Msk                               /*!< Buffer Index */
#define FDCAN_HPMS_BIDX_0                         (0x1UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000001 */
#define FDCAN_HPMS_BIDX_1                         (0x2UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000002 */
#define FDCAN_HPMS_BIDX_2                         (0x4UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000004 */
#define FDCAN_HPMS_BIDX_3                         (0x8UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000008 */
#define FDCAN_HPMS_BIDX_4                         (0x10UL<<FDCAN_HPMS_BIDX_Pos)                     /*!< 0x00000010 */
#define FDCAN_HPMS_BIDX_5                         (0x20UL<<FDCAN_HPMS_BIDX_Pos)                     /*!< 0x00000020 */
#define FDCAN_HPMS_MSI_Pos                        (6U)
#define FDCAN_HPMS_MSI_Msk                        (0x3UL<<FDCAN_HPMS_MSI_Pos)                       /*!< 0x000000C0 */
#define FDCAN_HPMS_MSI                            FDCAN_HPMS_MSI_Msk                                /*!<  Message Storage Indicator */
#define FDCAN_HPMS_MSI_0                          (0x1UL<<FDCAN_HPMS_MSI_Pos)                       /*!< 0x00000040 */
#define FDCAN_HPMS_MSI_1                          (0x2UL<<FDCAN_HPMS_MSI_Pos)                       /*!< 0x00000080 */
#define FDCAN_HPMS_FIDX_Pos                       (8U)
#define FDCAN_HPMS_FIDX_Msk                       (0x7FUL<<FDCAN_HPMS_FIDX_Pos)                     /*!< 0x00007F00 */
#define FDCAN_HPMS_FIDX                           FDCAN_HPMS_FIDX_Msk                               /*!< Filter Index */
#define FDCAN_HPMS_FLST_Pos                       (15U)
#define FDCAN_HPMS_FLST_Msk                       (0x1UL<<FDCAN_HPMS_FLST_Pos)                      /*!< 0x00008000 */
#define FDCAN_HPMS_FLST                           FDCAN_HPMS_FLST_Msk                               /*!< Filter List */

/********************************* Bit definition for FDCAN_NDAT1 register ******************************************/
#define FDCAN_NDAT1_ND0_Pos                       (0U)
#define FDCAN_NDAT1_ND0_Msk                       (0x1UL<<FDCAN_NDAT1_ND0_Pos)                      /*!< 0x00000001 */
#define FDCAN_NDAT1_ND0                           FDCAN_NDAT1_ND0_Msk                               /*!< New Data0 */
#define FDCAN_NDAT1_ND1_Pos                       (1U)
#define FDCAN_NDAT1_ND1_Msk                       (0x1UL<<FDCAN_NDAT1_ND1_Pos)                      /*!< 0x00000002 */
#define FDCAN_NDAT1_ND1                           FDCAN_NDAT1_ND1_Msk                               /*!< New Data1 */
#define FDCAN_NDAT1_ND2_Pos                       (2U)
#define FDCAN_NDAT1_ND2_Msk                       (0x1UL<<FDCAN_NDAT1_ND2_Pos)                      /*!< 0x00000004 */
#define FDCAN_NDAT1_ND2                           FDCAN_NDAT1_ND2_Msk                               /*!< New Data2 */
#define FDCAN_NDAT1_ND3_Pos                       (3U)
#define FDCAN_NDAT1_ND3_Msk                       (0x1UL<<FDCAN_NDAT1_ND3_Pos)                      /*!< 0x00000008 */
#define FDCAN_NDAT1_ND3                           FDCAN_NDAT1_ND3_Msk                               /*!< New Data3 */
#define FDCAN_NDAT1_ND4_Pos                       (4U)
#define FDCAN_NDAT1_ND4_Msk                       (0x1UL<<FDCAN_NDAT1_ND4_Pos)                      /*!< 0x00000010 */
#define FDCAN_NDAT1_ND4                           FDCAN_NDAT1_ND4_Msk                               /*!< New Data4 */
#define FDCAN_NDAT1_ND5_Pos                       (5U)
#define FDCAN_NDAT1_ND5_Msk                       (0x1UL<<FDCAN_NDAT1_ND5_Pos)                      /*!< 0x00000020 */
#define FDCAN_NDAT1_ND5                           FDCAN_NDAT1_ND5_Msk                               /*!< New Data5 */
#define FDCAN_NDAT1_ND6_Pos                       (6U)
#define FDCAN_NDAT1_ND6_Msk                       (0x1UL<<FDCAN_NDAT1_ND6_Pos)                      /*!< 0x00000040 */
#define FDCAN_NDAT1_ND6                           FDCAN_NDAT1_ND6_Msk                               /*!< New Data6 */
#define FDCAN_NDAT1_ND7_Pos                       (7U)
#define FDCAN_NDAT1_ND7_Msk                       (0x1UL<<FDCAN_NDAT1_ND7_Pos)                      /*!< 0x00000080 */
#define FDCAN_NDAT1_ND7                           FDCAN_NDAT1_ND7_Msk                               /*!< New Data7 */
#define FDCAN_NDAT1_ND8_Pos                       (8U)
#define FDCAN_NDAT1_ND8_Msk                       (0x1UL<<FDCAN_NDAT1_ND8_Pos)                      /*!< 0x00000100 */
#define FDCAN_NDAT1_ND8                           FDCAN_NDAT1_ND8_Msk                               /*!< New Data8 */
#define FDCAN_NDAT1_ND9_Pos                       (9U)
#define FDCAN_NDAT1_ND9_Msk                       (0x1UL<<FDCAN_NDAT1_ND9_Pos)                      /*!< 0x00000200 */
#define FDCAN_NDAT1_ND9                           FDCAN_NDAT1_ND9_Msk                               /*!< New Data9 */
#define FDCAN_NDAT1_ND10_Pos                      (10U)
#define FDCAN_NDAT1_ND10_Msk                      (0x1UL<<FDCAN_NDAT1_ND10_Pos)                     /*!< 0x00000400 */
#define FDCAN_NDAT1_ND10                          FDCAN_NDAT1_ND10_Msk                              /*!< New Data10 */
#define FDCAN_NDAT1_ND11_Pos                      (11U)
#define FDCAN_NDAT1_ND11_Msk                      (0x1UL<<FDCAN_NDAT1_ND11_Pos)                     /*!< 0x00000800 */
#define FDCAN_NDAT1_ND11                          FDCAN_NDAT1_ND11_Msk                              /*!< New Data11 */
#define FDCAN_NDAT1_ND12_Pos                      (12U)
#define FDCAN_NDAT1_ND12_Msk                      (0x1UL<<FDCAN_NDAT1_ND12_Pos)                     /*!< 0x00001000 */
#define FDCAN_NDAT1_ND12                          FDCAN_NDAT1_ND12_Msk                              /*!< New Data12 */
#define FDCAN_NDAT1_ND13_Pos                      (13U)
#define FDCAN_NDAT1_ND13_Msk                      (0x1UL<<FDCAN_NDAT1_ND13_Pos)                     /*!< 0x00002000 */
#define FDCAN_NDAT1_ND13                          FDCAN_NDAT1_ND13_Msk                              /*!< New Data13 */
#define FDCAN_NDAT1_ND14_Pos                      (14U)
#define FDCAN_NDAT1_ND14_Msk                      (0x1UL<<FDCAN_NDAT1_ND14_Pos)                     /*!< 0x00004000 */
#define FDCAN_NDAT1_ND14                          FDCAN_NDAT1_ND14_Msk                              /*!< New Data14 */
#define FDCAN_NDAT1_ND15_Pos                      (15U)
#define FDCAN_NDAT1_ND15_Msk                      (0x1UL<<FDCAN_NDAT1_ND15_Pos)                     /*!< 0x00008000 */
#define FDCAN_NDAT1_ND15                          FDCAN_NDAT1_ND15_Msk                              /*!< New Data15 */
#define FDCAN_NDAT1_ND16_Pos                      (16U)
#define FDCAN_NDAT1_ND16_Msk                      (0x1UL<<FDCAN_NDAT1_ND16_Pos)                     /*!< 0x00010000 */
#define FDCAN_NDAT1_ND16                          FDCAN_NDAT1_ND16_Msk                              /*!< New Data16 */
#define FDCAN_NDAT1_ND17_Pos                      (17U)
#define FDCAN_NDAT1_ND17_Msk                      (0x1UL<<FDCAN_NDAT1_ND17_Pos)                     /*!< 0x00020000 */
#define FDCAN_NDAT1_ND17                          FDCAN_NDAT1_ND17_Msk                              /*!< New Data17 */
#define FDCAN_NDAT1_ND18_Pos                      (18U)
#define FDCAN_NDAT1_ND18_Msk                      (0x1UL<<FDCAN_NDAT1_ND18_Pos)                     /*!< 0x00040000 */
#define FDCAN_NDAT1_ND18                          FDCAN_NDAT1_ND18_Msk                              /*!< New Data18 */
#define FDCAN_NDAT1_ND19_Pos                      (19U)
#define FDCAN_NDAT1_ND19_Msk                      (0x1UL<<FDCAN_NDAT1_ND19_Pos)                     /*!< 0x00080000 */
#define FDCAN_NDAT1_ND19                          FDCAN_NDAT1_ND19_Msk                              /*!< New Data19 */
#define FDCAN_NDAT1_ND20_Pos                      (20U)
#define FDCAN_NDAT1_ND20_Msk                      (0x1UL<<FDCAN_NDAT1_ND20_Pos)                     /*!< 0x00100000 */
#define FDCAN_NDAT1_ND20                          FDCAN_NDAT1_ND20_Msk                              /*!< New Data20 */
#define FDCAN_NDAT1_ND21_Pos                      (21U)
#define FDCAN_NDAT1_ND21_Msk                      (0x1UL<<FDCAN_NDAT1_ND21_Pos)                     /*!< 0x00200000 */
#define FDCAN_NDAT1_ND21                          FDCAN_NDAT1_ND21_Msk                              /*!< New Data21 */
#define FDCAN_NDAT1_ND22_Pos                      (22U)
#define FDCAN_NDAT1_ND22_Msk                      (0x1UL<<FDCAN_NDAT1_ND22_Pos)                     /*!< 0x00400000 */
#define FDCAN_NDAT1_ND22                          FDCAN_NDAT1_ND22_Msk                              /*!< New Data22 */
#define FDCAN_NDAT1_ND23_Pos                      (23U)
#define FDCAN_NDAT1_ND23_Msk                      (0x1UL<<FDCAN_NDAT1_ND23_Pos)                     /*!< 0x00800000 */
#define FDCAN_NDAT1_ND23                          FDCAN_NDAT1_ND23_Msk                              /*!< New Data23 */
#define FDCAN_NDAT1_ND24_Pos                      (24U)
#define FDCAN_NDAT1_ND24_Msk                      (0x1UL<<FDCAN_NDAT1_ND24_Pos)                     /*!< 0x01000000 */
#define FDCAN_NDAT1_ND24                          FDCAN_NDAT1_ND24_Msk                              /*!< New Data24 */
#define FDCAN_NDAT1_ND25_Pos                      (25U)
#define FDCAN_NDAT1_ND25_Msk                      (0x1UL<<FDCAN_NDAT1_ND25_Pos)                     /*!< 0x02000000 */
#define FDCAN_NDAT1_ND25                          FDCAN_NDAT1_ND25_Msk                              /*!< New Data25 */
#define FDCAN_NDAT1_ND26_Pos                      (26U)
#define FDCAN_NDAT1_ND26_Msk                      (0x1UL<<FDCAN_NDAT1_ND26_Pos)                     /*!< 0x04000000 */
#define FDCAN_NDAT1_ND26                          FDCAN_NDAT1_ND26_Msk                              /*!< New Data26 */
#define FDCAN_NDAT1_ND27_Pos                      (27U)
#define FDCAN_NDAT1_ND27_Msk                      (0x1UL<<FDCAN_NDAT1_ND27_Pos)                     /*!< 0x08000000 */
#define FDCAN_NDAT1_ND27                          FDCAN_NDAT1_ND27_Msk                              /*!< New Data27 */
#define FDCAN_NDAT1_ND28_Pos                      (28U)
#define FDCAN_NDAT1_ND28_Msk                      (0x1UL<<FDCAN_NDAT1_ND28_Pos)                     /*!< 0x10000000 */
#define FDCAN_NDAT1_ND28                          FDCAN_NDAT1_ND28_Msk                              /*!< New Data28 */
#define FDCAN_NDAT1_ND29_Pos                      (29U)
#define FDCAN_NDAT1_ND29_Msk                      (0x1UL<<FDCAN_NDAT1_ND29_Pos)                     /*!< 0x20000000 */
#define FDCAN_NDAT1_ND29                          FDCAN_NDAT1_ND29_Msk                              /*!< New Data29 */
#define FDCAN_NDAT1_ND30_Pos                      (30U)
#define FDCAN_NDAT1_ND30_Msk                      (0x1UL<<FDCAN_NDAT1_ND30_Pos)                     /*!< 0x40000000 */
#define FDCAN_NDAT1_ND30                          FDCAN_NDAT1_ND30_Msk                              /*!< New Data30 */
#define FDCAN_NDAT1_ND31_Pos                      (31U)
#define FDCAN_NDAT1_ND31_Msk                      (0x1UL<<FDCAN_NDAT1_ND31_Pos)                     /*!< 0x80000000 */
#define FDCAN_NDAT1_ND31                          FDCAN_NDAT1_ND31_Msk                              /*!< New Data31 */

/********************************* Bit definition for FDCAN_NDAT2 register ******************************************/
#define FDCAN_NDAT2_ND32_Pos                      (0U)
#define FDCAN_NDAT2_ND32_Msk                      (0x1UL<<FDCAN_NDAT2_ND32_Pos)                     /*!< 0x00000001 */
#define FDCAN_NDAT2_ND32                          FDCAN_NDAT2_ND32_Msk                              /*!< New Data32 */
#define FDCAN_NDAT2_ND33_Pos                      (1U)
#define FDCAN_NDAT2_ND33_Msk                      (0x1UL<<FDCAN_NDAT2_ND33_Pos)                     /*!< 0x00000002 */
#define FDCAN_NDAT2_ND33                          FDCAN_NDAT2_ND33_Msk                              /*!< New Data33 */
#define FDCAN_NDAT2_ND34_Pos                      (2U)
#define FDCAN_NDAT2_ND34_Msk                      (0x1UL<<FDCAN_NDAT2_ND34_Pos)                     /*!< 0x00000004 */
#define FDCAN_NDAT2_ND34                          FDCAN_NDAT2_ND34_Msk                              /*!< New Data34 */
#define FDCAN_NDAT2_ND35_Pos                      (3U)
#define FDCAN_NDAT2_ND35_Msk                      (0x1UL<<FDCAN_NDAT2_ND35_Pos)                     /*!< 0x00000008 */
#define FDCAN_NDAT2_ND35                          FDCAN_NDAT2_ND35_Msk                              /*!< New Data35 */
#define FDCAN_NDAT2_ND36_Pos                      (4U)
#define FDCAN_NDAT2_ND36_Msk                      (0x1UL<<FDCAN_NDAT2_ND36_Pos)                     /*!< 0x00000010 */
#define FDCAN_NDAT2_ND36                          FDCAN_NDAT2_ND36_Msk                              /*!< New Data36 */
#define FDCAN_NDAT2_ND37_Pos                      (5U)
#define FDCAN_NDAT2_ND37_Msk                      (0x1UL<<FDCAN_NDAT2_ND37_Pos)                     /*!< 0x00000020 */
#define FDCAN_NDAT2_ND37                          FDCAN_NDAT2_ND37_Msk                              /*!< New Data37 */
#define FDCAN_NDAT2_ND38_Pos                      (6U)
#define FDCAN_NDAT2_ND38_Msk                      (0x1UL<<FDCAN_NDAT2_ND38_Pos)                     /*!< 0x00000040 */
#define FDCAN_NDAT2_ND38                          FDCAN_NDAT2_ND38_Msk                              /*!< New Data38 */
#define FDCAN_NDAT2_ND39_Pos                      (7U)
#define FDCAN_NDAT2_ND39_Msk                      (0x1UL<<FDCAN_NDAT2_ND39_Pos)                     /*!< 0x00000080 */
#define FDCAN_NDAT2_ND39                          FDCAN_NDAT2_ND39_Msk                              /*!< New Data39 */
#define FDCAN_NDAT2_ND40_Pos                      (8U)
#define FDCAN_NDAT2_ND40_Msk                      (0x1UL<<FDCAN_NDAT2_ND40_Pos)                     /*!< 0x00000100 */
#define FDCAN_NDAT2_ND40                          FDCAN_NDAT2_ND40_Msk                              /*!< New Data40 */
#define FDCAN_NDAT2_ND41_Pos                      (9U)
#define FDCAN_NDAT2_ND41_Msk                      (0x1UL<<FDCAN_NDAT2_ND41_Pos)                     /*!< 0x00000200 */
#define FDCAN_NDAT2_ND41                          FDCAN_NDAT2_ND41_Msk                              /*!< New Data41 */
#define FDCAN_NDAT2_ND42_Pos                      (10U)
#define FDCAN_NDAT2_ND42_Msk                      (0x1UL<<FDCAN_NDAT2_ND42_Pos)                     /*!< 0x00000400 */
#define FDCAN_NDAT2_ND42                          FDCAN_NDAT2_ND42_Msk                              /*!< New Data42 */
#define FDCAN_NDAT2_ND43_Pos                      (11U)
#define FDCAN_NDAT2_ND43_Msk                      (0x1UL<<FDCAN_NDAT2_ND43_Pos)                     /*!< 0x00000800 */
#define FDCAN_NDAT2_ND43                          FDCAN_NDAT2_ND43_Msk                              /*!< New Data43 */
#define FDCAN_NDAT2_ND44_Pos                      (12U)
#define FDCAN_NDAT2_ND44_Msk                      (0x1UL<<FDCAN_NDAT2_ND44_Pos)                     /*!< 0x00001000 */
#define FDCAN_NDAT2_ND44                          FDCAN_NDAT2_ND44_Msk                              /*!< New Data44 */
#define FDCAN_NDAT2_ND45_Pos                      (13U)
#define FDCAN_NDAT2_ND45_Msk                      (0x1UL<<FDCAN_NDAT2_ND45_Pos)                     /*!< 0x00002000 */
#define FDCAN_NDAT2_ND45                          FDCAN_NDAT2_ND45_Msk                              /*!< New Data45 */
#define FDCAN_NDAT2_ND46_Pos                      (14U)
#define FDCAN_NDAT2_ND46_Msk                      (0x1UL<<FDCAN_NDAT2_ND46_Pos)                     /*!< 0x00004000 */
#define FDCAN_NDAT2_ND46                          FDCAN_NDAT2_ND46_Msk                              /*!< New Data46 */
#define FDCAN_NDAT2_ND47_Pos                      (15U)
#define FDCAN_NDAT2_ND47_Msk                      (0x1UL<<FDCAN_NDAT2_ND47_Pos)                     /*!< 0x00008000 */
#define FDCAN_NDAT2_ND47                          FDCAN_NDAT2_ND47_Msk                              /*!< New Data47 */
#define FDCAN_NDAT2_ND48_Pos                      (16U)
#define FDCAN_NDAT2_ND48_Msk                      (0x1UL<<FDCAN_NDAT2_ND48_Pos)                     /*!< 0x00010000 */
#define FDCAN_NDAT2_ND48                          FDCAN_NDAT2_ND48_Msk                              /*!< New Data48 */
#define FDCAN_NDAT2_ND49_Pos                      (17U)
#define FDCAN_NDAT2_ND49_Msk                      (0x1UL<<FDCAN_NDAT2_ND49_Pos)                     /*!< 0x00020000 */
#define FDCAN_NDAT2_ND49                          FDCAN_NDAT2_ND49_Msk                              /*!< New Data49 */
#define FDCAN_NDAT2_ND50_Pos                      (18U)
#define FDCAN_NDAT2_ND50_Msk                      (0x1UL<<FDCAN_NDAT2_ND50_Pos)                     /*!< 0x00040000 */
#define FDCAN_NDAT2_ND50                          FDCAN_NDAT2_ND50_Msk                              /*!< New Data50 */
#define FDCAN_NDAT2_ND51_Pos                      (19U)
#define FDCAN_NDAT2_ND51_Msk                      (0x1UL<<FDCAN_NDAT2_ND51_Pos)                     /*!< 0x00080000 */
#define FDCAN_NDAT2_ND51                          FDCAN_NDAT2_ND51_Msk                              /*!< New Data51 */
#define FDCAN_NDAT2_ND52_Pos                      (20U)
#define FDCAN_NDAT2_ND52_Msk                      (0x1UL<<FDCAN_NDAT2_ND52_Pos)                     /*!< 0x00100000 */
#define FDCAN_NDAT2_ND52                          FDCAN_NDAT2_ND52_Msk                              /*!< New Data52 */
#define FDCAN_NDAT2_ND53_Pos                      (21U)
#define FDCAN_NDAT2_ND53_Msk                      (0x1UL<<FDCAN_NDAT2_ND53_Pos)                     /*!< 0x00200000 */
#define FDCAN_NDAT2_ND53                          FDCAN_NDAT2_ND53_Msk                              /*!< New Data53 */
#define FDCAN_NDAT2_ND54_Pos                      (22U)
#define FDCAN_NDAT2_ND54_Msk                      (0x1UL<<FDCAN_NDAT2_ND54_Pos)                     /*!< 0x00400000 */
#define FDCAN_NDAT2_ND54                          FDCAN_NDAT2_ND54_Msk                              /*!< New Data54 */
#define FDCAN_NDAT2_ND55_Pos                      (23U)
#define FDCAN_NDAT2_ND55_Msk                      (0x1UL<<FDCAN_NDAT2_ND55_Pos)                     /*!< 0x00800000 */
#define FDCAN_NDAT2_ND55                          FDCAN_NDAT2_ND55_Msk                              /*!< New Data55 */
#define FDCAN_NDAT2_ND56_Pos                      (24U)
#define FDCAN_NDAT2_ND56_Msk                      (0x1UL<<FDCAN_NDAT2_ND56_Pos)                     /*!< 0x01000000 */
#define FDCAN_NDAT2_ND56                          FDCAN_NDAT2_ND56_Msk                              /*!< New Data56 */
#define FDCAN_NDAT2_ND57_Pos                      (25U)
#define FDCAN_NDAT2_ND57_Msk                      (0x1UL<<FDCAN_NDAT2_ND57_Pos)                     /*!< 0x02000000 */
#define FDCAN_NDAT2_ND57                          FDCAN_NDAT2_ND57_Msk                              /*!< New Data57 */
#define FDCAN_NDAT2_ND58_Pos                      (26U)
#define FDCAN_NDAT2_ND58_Msk                      (0x1UL<<FDCAN_NDAT2_ND58_Pos)                     /*!< 0x04000000 */
#define FDCAN_NDAT2_ND58                          FDCAN_NDAT2_ND58_Msk                              /*!< New Data58 */
#define FDCAN_NDAT2_ND59_Pos                      (27U)
#define FDCAN_NDAT2_ND59_Msk                      (0x1UL<<FDCAN_NDAT2_ND59_Pos)                     /*!< 0x08000000 */
#define FDCAN_NDAT2_ND59                          FDCAN_NDAT2_ND59_Msk                              /*!< New Data59 */
#define FDCAN_NDAT2_ND60_Pos                      (28U)
#define FDCAN_NDAT2_ND60_Msk                      (0x1UL<<FDCAN_NDAT2_ND60_Pos)                     /*!< 0x10000000 */
#define FDCAN_NDAT2_ND60                          FDCAN_NDAT2_ND60_Msk                              /*!< New Data60 */
#define FDCAN_NDAT2_ND61_Pos                      (29U)
#define FDCAN_NDAT2_ND61_Msk                      (0x1UL<<FDCAN_NDAT2_ND61_Pos)                     /*!< 0x20000000 */
#define FDCAN_NDAT2_ND61                          FDCAN_NDAT2_ND61_Msk                              /*!< New Data61 */
#define FDCAN_NDAT2_ND62_Pos                      (30U)
#define FDCAN_NDAT2_ND62_Msk                      (0x1UL<<FDCAN_NDAT2_ND62_Pos)                     /*!< 0x40000000 */
#define FDCAN_NDAT2_ND62                          FDCAN_NDAT2_ND62_Msk                              /*!< New Data62 */
#define FDCAN_NDAT2_ND63_Pos                      (31U)
#define FDCAN_NDAT2_ND63_Msk                      (0x1UL<<FDCAN_NDAT2_ND63_Pos)                     /*!< 0x80000000 */
#define FDCAN_NDAT2_ND63                          FDCAN_NDAT2_ND63_Msk                              /*!< New Data63 */

/********************************* Bit definition for FDCAN_RXF0C register ******************************************/
#define FDCAN_RXF0C_F0SA_Pos                      (2U)
#define FDCAN_RXF0C_F0SA_Msk                      (0x3FFFUL<<FDCAN_RXF0C_F0SA_Pos)                  /*!< 0x0000FFFC */
#define FDCAN_RXF0C_F0SA                          FDCAN_RXF0C_F0SA_Msk                              /*!< Rx FIFO 0 Start Address */
#define FDCAN_RXF0C_F0S_Pos                       (16U)
#define FDCAN_RXF0C_F0S_Msk                       (0x7FUL<<FDCAN_RXF0C_F0S_Pos)                     /*!< 0x007F0000 */
#define FDCAN_RXF0C_F0S                           FDCAN_RXF0C_F0S_Msk                               /*!< Rx FIFO 0 Size */
#define FDCAN_RXF0C_F0WM_Pos                      (24U)
#define FDCAN_RXF0C_F0WM_Msk                      (0x7FUL<<FDCAN_RXF0C_F0WM_Pos)                    /*!< 0x7F000000 */
#define FDCAN_RXF0C_F0WM                          FDCAN_RXF0C_F0WM_Msk                              /*!< Rx FIFO 0 Watermark */
#define FDCAN_RXF0C_F0OM_Pos                      (31U)
#define FDCAN_RXF0C_F0OM_Msk                      (0x1UL<<FDCAN_RXF0C_F0OM_Pos)                     /*!< 0x80000000 */
#define FDCAN_RXF0C_F0OM                          FDCAN_RXF0C_F0OM_Msk                              /*!< FIFO 0 Operation Mode */

/********************************* Bit definition for FDCAN_RXF0S register ******************************************/
#define FDCAN_RXF0S_F0FL_Pos                      (0U)
#define FDCAN_RXF0S_F0FL_Msk                      (0x7FUL<<FDCAN_RXF0S_F0FL_Pos)                    /*!< 0x0000007F */
#define FDCAN_RXF0S_F0FL                          FDCAN_RXF0S_F0FL_Msk                              /*!< Rx FIFO 0 Fill Level */
#define FDCAN_RXF0S_F0GI_Pos                      (8U)
#define FDCAN_RXF0S_F0GI_Msk                      (0x3FUL<<FDCAN_RXF0S_F0GI_Pos)                    /*!< 0x00003F00 */
#define FDCAN_RXF0S_F0GI                          FDCAN_RXF0S_F0GI_Msk                              /*!< Rx FIFO 0 Get Index */
#define FDCAN_RXF0S_F0GI_0                        (0x1UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000100 */
#define FDCAN_RXF0S_F0GI_1                        (0x2UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000200 */
#define FDCAN_RXF0S_F0GI_2                        (0x4UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000400 */
#define FDCAN_RXF0S_F0GI_3                        (0x8UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000800 */
#define FDCAN_RXF0S_F0GI_4                        (0x10UL<<FDCAN_RXF0S_F0GI_Pos)                    /*!< 0x00001000 */
#define FDCAN_RXF0S_F0GI_5                        (0x20UL<<FDCAN_RXF0S_F0GI_Pos)                    /*!< 0x00002000 */
#define FDCAN_RXF0S_F0PI_Pos                      (16U)
#define FDCAN_RXF0S_F0PI_Msk                      (0x3FUL<<FDCAN_RXF0S_F0PI_Pos)                    /*!< 0x003F0000 */
#define FDCAN_RXF0S_F0PI                          FDCAN_RXF0S_F0PI_Msk                              /*!< Rx FIFO 0 Put Index */
#define FDCAN_RXF0S_F0PI_0                        (0x1UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00010000 */
#define FDCAN_RXF0S_F0PI_1                        (0x2UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00020000 */
#define FDCAN_RXF0S_F0PI_2                        (0x4UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00040000 */
#define FDCAN_RXF0S_F0PI_3                        (0x8UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00080000 */
#define FDCAN_RXF0S_F0PI_4                        (0x10UL<<FDCAN_RXF0S_F0PI_Pos)                    /*!< 0x00100000 */
#define FDCAN_RXF0S_F0PI_5                        (0x20UL<<FDCAN_RXF0S_F0PI_Pos)                    /*!< 0x00200000 */
#define FDCAN_RXF0S_F0F_Pos                       (24U)
#define FDCAN_RXF0S_F0F_Msk                       (0x1UL<<FDCAN_RXF0S_F0F_Pos)                      /*!< 0x01000000 */
#define FDCAN_RXF0S_F0F                           FDCAN_RXF0S_F0F_Msk                               /*!< Rx FIFO 0 Full */
#define FDCAN_RXF0S_RF0L_Pos                      (25U)
#define FDCAN_RXF0S_RF0L_Msk                      (0x1UL<<FDCAN_RXF0S_RF0L_Pos)                     /*!< 0x02000000 */
#define FDCAN_RXF0S_RF0L                          FDCAN_RXF0S_RF0L_Msk                              /*!< Rx FIFO 0 Message Lost */

/********************************* Bit definition for FDCAN_RXF0A register ******************************************/
#define FDCAN_RXF0A_F0AI_Pos                      (0U)
#define FDCAN_RXF0A_F0AI_Msk                      (0x3FUL<<FDCAN_RXF0A_F0AI_Pos)                    /*!< 0x0000003F */
#define FDCAN_RXF0A_F0AI                          FDCAN_RXF0A_F0AI_Msk                              /*!< Rx FIFO 0 Acknowledge Index */
#define FDCAN_RXF0A_F0AI_0                        (0x1UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000001 */
#define FDCAN_RXF0A_F0AI_1                        (0x2UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000002 */
#define FDCAN_RXF0A_F0AI_2                        (0x4UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXF0A_F0AI_3                        (0x8UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000008 */
#define FDCAN_RXF0A_F0AI_4                        (0x10UL<<FDCAN_RXF0A_F0AI_Pos)                    /*!< 0x00000010 */
#define FDCAN_RXF0A_F0AI_5                        (0x20UL<<FDCAN_RXF0A_F0AI_Pos)                    /*!< 0x00000020 */

/********************************* Bit definition for FDCAN_RXBC register *******************************************/
#define FDCAN_RXBC_RBSA_Pos                       (2U)
#define FDCAN_RXBC_RBSA_Msk                       (0x3FFFUL<<FDCAN_RXBC_RBSA_Pos)                   /*!< 0x0000FFFC */
#define FDCAN_RXBC_RBSA                           FDCAN_RXBC_RBSA_Msk                               /*!< Rx Buffer Start Address */

/********************************* Bit definition for FDCAN_RXF1C register ******************************************/
#define FDCAN_RXF1C_F1SA_Pos                      (2U)
#define FDCAN_RXF1C_F1SA_Msk                      (0xFUL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x0000003C */
#define FDCAN_RXF1C_F1SA                          FDCAN_RXF1C_F1SA_Msk                              /*!< Rx FIFO 1 Start Address */
#define FDCAN_RXF1C_F1SA_0                        (0x1UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXF1C_F1SA_1                        (0x2UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000008 */
#define FDCAN_RXF1C_F1SA_2                        (0x4UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000010 */
#define FDCAN_RXF1C_F1SA_3                        (0x8UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000020 */
#define FDCAN_RXF1C_F1S_Pos                       (16U)
#define FDCAN_RXF1C_F1S_Msk                       (0x7FUL<<FDCAN_RXF1C_F1S_Pos)                     /*!< 0x007F0000 */
#define FDCAN_RXF1C_F1S                           FDCAN_RXF1C_F1S_Msk                               /*!< Rx FIFO 1 Size */
#define FDCAN_RXF1C_F1WM_Pos                      (24U)
#define FDCAN_RXF1C_F1WM_Msk                      (0x7FUL<<FDCAN_RXF1C_F1WM_Pos)                    /*!< 0x7F000000 */
#define FDCAN_RXF1C_F1WM                          FDCAN_RXF1C_F1WM_Msk                              /*!< Rx FIFO 1 Watermark */
#define FDCAN_RXF1C_F1OM_Pos                      (31U)
#define FDCAN_RXF1C_F1OM_Msk                      (0x1UL<<FDCAN_RXF1C_F1OM_Pos)                     /*!< 0x80000000 */
#define FDCAN_RXF1C_F1OM                          FDCAN_RXF1C_F1OM_Msk                              /*!< FIFO 1 Operation Mode */

/********************************* Bit definition for FDCAN_RXF1S register ******************************************/
#define FDCAN_RXF1S_F1FL_Pos                      (0U)
#define FDCAN_RXF1S_F1FL_Msk                      (0x7FUL<<FDCAN_RXF1S_F1FL_Pos)                    /*!< 0x0000007F */
#define FDCAN_RXF1S_F1FL                          FDCAN_RXF1S_F1FL_Msk                              /*!< Rx FIFO 1 Fill Level */
#define FDCAN_RXF1S_F1GI_Pos                      (8U)
#define FDCAN_RXF1S_F1GI_Msk                      (0x3FUL<<FDCAN_RXF1S_F1GI_Pos)                    /*!< 0x00003F00 */
#define FDCAN_RXF1S_F1GI                          FDCAN_RXF1S_F1GI_Msk                              /*!< Rx FIFO 1 Get Index */
#define FDCAN_RXF1S_F1GI_0                        (0x1UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000100 */
#define FDCAN_RXF1S_F1GI_1                        (0x2UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000200 */
#define FDCAN_RXF1S_F1GI_2                        (0x4UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000400 */
#define FDCAN_RXF1S_F1GI_3                        (0x8UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000800 */
#define FDCAN_RXF1S_F1GI_4                        (0x10UL<<FDCAN_RXF1S_F1GI_Pos)                    /*!< 0x00001000 */
#define FDCAN_RXF1S_F1GI_5                        (0x20UL<<FDCAN_RXF1S_F1GI_Pos)                    /*!< 0x00002000 */
#define FDCAN_RXF1S_F1PI_Pos                      (16U)
#define FDCAN_RXF1S_F1PI_Msk                      (0x3FUL<<FDCAN_RXF1S_F1PI_Pos)                    /*!< 0x003F0000 */
#define FDCAN_RXF1S_F1PI                          FDCAN_RXF1S_F1PI_Msk                              /*!< Rx FIFO 1 Put Index */
#define FDCAN_RXF1S_F1PI_0                        (0x1UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00010000 */
#define FDCAN_RXF1S_F1PI_1                        (0x2UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00020000 */
#define FDCAN_RXF1S_F1PI_2                        (0x4UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00040000 */
#define FDCAN_RXF1S_F1PI_3                        (0x8UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00080000 */
#define FDCAN_RXF1S_F1PI_4                        (0x10UL<<FDCAN_RXF1S_F1PI_Pos)                    /*!< 0x00100000 */
#define FDCAN_RXF1S_F1PI_5                        (0x20UL<<FDCAN_RXF1S_F1PI_Pos)                    /*!< 0x00200000 */
#define FDCAN_RXF1S_F1F_Pos                       (24U)
#define FDCAN_RXF1S_F1F_Msk                       (0x1UL<<FDCAN_RXF1S_F1F_Pos)                      /*!< 0x01000000 */
#define FDCAN_RXF1S_F1F                           FDCAN_RXF1S_F1F_Msk                               /*!< Rx FIFO 1 Full */
#define FDCAN_RXF1S_RF1L_Pos                      (25U)
#define FDCAN_RXF1S_RF1L_Msk                      (0x1UL<<FDCAN_RXF1S_RF1L_Pos)                     /*!< 0x02000000 */
#define FDCAN_RXF1S_RF1L                          FDCAN_RXF1S_RF1L_Msk                              /*!< Rx FIFO 1 Message Lost */
#define FDCAN_RXF1S_DMS_Pos                       (30U)
#define FDCAN_RXF1S_DMS_Msk                       (0x3UL<<FDCAN_RXF1S_DMS_Pos)                      /*!< 0xC0000000 */
#define FDCAN_RXF1S_DMS                           FDCAN_RXF1S_DMS_Msk                               /*!< Debug Message Status */
#define FDCAN_RXF1S_DMS_0                         (0x1UL<<FDCAN_RXF1S_DMS_Pos)                      /*!< 0x40000000 */
#define FDCAN_RXF1S_DMS_1                         (0x2UL<<FDCAN_RXF1S_DMS_Pos)                      /*!< 0x80000000 */

/********************************* Bit definition for FDCAN_RXF1A register ******************************************/
#define FDCAN_RXF1A_F1AI_Pos                      (0U)
#define FDCAN_RXF1A_F1AI_Msk                      (0x3FUL<<FDCAN_RXF1A_F1AI_Pos)                    /*!< 0x0000003F */
#define FDCAN_RXF1A_F1AI                          FDCAN_RXF1A_F1AI_Msk                              /*!< Rx FIFO 1 Acknowledge Index */
#define FDCAN_RXF1A_F1AI_0                        (0x1UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000001 */
#define FDCAN_RXF1A_F1AI_1                        (0x2UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000002 */
#define FDCAN_RXF1A_F1AI_2                        (0x4UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXF1A_F1AI_3                        (0x8UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000008 */
#define FDCAN_RXF1A_F1AI_4                        (0x10UL<<FDCAN_RXF1A_F1AI_Pos)                    /*!< 0x00000010 */
#define FDCAN_RXF1A_F1AI_5                        (0x20UL<<FDCAN_RXF1A_F1AI_Pos)                    /*!< 0x00000020 */

/********************************* Bit definition for FDCAN_RXESC register ******************************************/
#define FDCAN_RXESC_F0DS_Pos                      (0U)
#define FDCAN_RXESC_F0DS_Msk                      (0x7UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000007 */
#define FDCAN_RXESC_F0DS                          FDCAN_RXESC_F0DS_Msk                              /*!< Rx FIFO 0 Data Field Size */
#define FDCAN_RXESC_F0DS_0                        (0x1UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000001 */
#define FDCAN_RXESC_F0DS_1                        (0x2UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000002 */
#define FDCAN_RXESC_F0DS_2                        (0x4UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXESC_F1DS_Pos                      (4U)
#define FDCAN_RXESC_F1DS_Msk                      (0x7UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000070 */
#define FDCAN_RXESC_F1DS                          FDCAN_RXESC_F1DS_Msk                              /*!< Rx FIFO 1 Data Field Size */
#define FDCAN_RXESC_F1DS_0                        (0x1UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000010 */
#define FDCAN_RXESC_F1DS_1                        (0x2UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000020 */
#define FDCAN_RXESC_F1DS_2                        (0x4UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000040 */
#define FDCAN_RXESC_RBDS_Pos                      (8U)
#define FDCAN_RXESC_RBDS_Msk                      (0x7UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000700 */
#define FDCAN_RXESC_RBDS                          FDCAN_RXESC_RBDS_Msk                              /*!< Rx Buffer Data Field Size */
#define FDCAN_RXESC_RBDS_0                        (0x1UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000100 */
#define FDCAN_RXESC_RBDS_1                        (0x2UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000200 */
#define FDCAN_RXESC_RBDS_2                        (0x4UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000400 */

/********************************* Bit definition for FDCAN_TXBC register *******************************************/
#define FDCAN_TXBC_TBSA_Pos                       (2U)
#define FDCAN_TXBC_TBSA_Msk                       (0x3FFFUL<<FDCAN_TXBC_TBSA_Pos)                   /*!< 0x0000FFFC */
#define FDCAN_TXBC_TBSA                           FDCAN_TXBC_TBSA_Msk                               /*!< Tx Buffers Start Address */
#define FDCAN_TXBC_NDTB_Pos                       (16U)
#define FDCAN_TXBC_NDTB_Msk                       (0x3FUL<<FDCAN_TXBC_NDTB_Pos)                     /*!< 0x003F0000 */
#define FDCAN_TXBC_NDTB                           FDCAN_TXBC_NDTB_Msk                               /*!< Number of Dedicated Transmit Buffers */
#define FDCAN_TXBC_NDTB_0                         (0x1UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00010000 */
#define FDCAN_TXBC_NDTB_1                         (0x2UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00020000 */
#define FDCAN_TXBC_NDTB_2                         (0x4UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00040000 */
#define FDCAN_TXBC_NDTB_3                         (0x8UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00080000 */
#define FDCAN_TXBC_NDTB_4                         (0x10UL<<FDCAN_TXBC_NDTB_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBC_NDTB_5                         (0x20UL<<FDCAN_TXBC_NDTB_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBC_TFQS_Pos                       (24U)
#define FDCAN_TXBC_TFQS_Msk                       (0x3FUL<<FDCAN_TXBC_TFQS_Pos)                     /*!< 0x3F000000 */
#define FDCAN_TXBC_TFQS                           FDCAN_TXBC_TFQS_Msk                               /*!< Transmit FIFO/Queue Size */
#define FDCAN_TXBC_TFQS_0                         (0x1UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x01000000 */
#define FDCAN_TXBC_TFQS_1                         (0x2UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x02000000 */
#define FDCAN_TXBC_TFQS_2                         (0x4UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x04000000 */
#define FDCAN_TXBC_TFQS_3                         (0x8UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x08000000 */
#define FDCAN_TXBC_TFQS_4                         (0x10UL<<FDCAN_TXBC_TFQS_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBC_TFQS_5                         (0x20UL<<FDCAN_TXBC_TFQS_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBC_TFQM_Pos                       (30U)
#define FDCAN_TXBC_TFQM_Msk                       (0x1UL<<FDCAN_TXBC_TFQM_Pos)                      /*!< 0x40000000 */
#define FDCAN_TXBC_TFQM                           FDCAN_TXBC_TFQM_Msk                               /*!< Tx FIFO/Queue Mode */

/********************************* Bit definition for FDCAN_TXFQS register ******************************************/
#define FDCAN_TXFQS_TFFL_Pos                      (0U)
#define FDCAN_TXFQS_TFFL_Msk                      (0x3FUL<<FDCAN_TXFQS_TFFL_Pos)                    /*!< 0x0000003F */
#define FDCAN_TXFQS_TFFL                          FDCAN_TXFQS_TFFL_Msk                              /*!< Tx FIFO Free Level */
#define FDCAN_TXFQS_TFFL_0                        (0x1UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXFQS_TFFL_1                        (0x2UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXFQS_TFFL_2                        (0x4UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXFQS_TFFL_3                        (0x8UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXFQS_TFFL_4                        (0x10UL<<FDCAN_TXFQS_TFFL_Pos)                    /*!< 0x00000010 */
#define FDCAN_TXFQS_TFFL_5                        (0x20UL<<FDCAN_TXFQS_TFFL_Pos)                    /*!< 0x00000020 */
#define FDCAN_TXFQS_TFGI_Pos                      (8U)
#define FDCAN_TXFQS_TFGI_Msk                      (0x1FUL<<FDCAN_TXFQS_TFGI_Pos)                    /*!< 0x00001F00 */
#define FDCAN_TXFQS_TFGI                          FDCAN_TXFQS_TFGI_Msk                              /*!< Tx FIFO Get Index */
#define FDCAN_TXFQS_TFGI_0                        (0x1UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000100 */
#define FDCAN_TXFQS_TFGI_1                        (0x2UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000200 */
#define FDCAN_TXFQS_TFGI_2                        (0x4UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXFQS_TFGI_3                        (0x8UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXFQS_TFGI_4                        (0x10UL<<FDCAN_TXFQS_TFGI_Pos)                    /*!< 0x00001000 */
#define FDCAN_TXFQS_TFQPI_Pos                     (16U)
#define FDCAN_TXFQS_TFQPI_Msk                     (0x1FUL<<FDCAN_TXFQS_TFQPI_Pos)                   /*!< 0x001F0000 */
#define FDCAN_TXFQS_TFQPI                         FDCAN_TXFQS_TFQPI_Msk                             /*!< Tx FIFO/Queue Put Index */
#define FDCAN_TXFQS_TFQPI_0                       (0x1UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00010000 */
#define FDCAN_TXFQS_TFQPI_1                       (0x2UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00020000 */
#define FDCAN_TXFQS_TFQPI_2                       (0x4UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00040000 */
#define FDCAN_TXFQS_TFQPI_3                       (0x8UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00080000 */
#define FDCAN_TXFQS_TFQPI_4                       (0x10UL<<FDCAN_TXFQS_TFQPI_Pos)                   /*!< 0x00100000 */
#define FDCAN_TXFQS_TFQF_Pos                      (21U)
#define FDCAN_TXFQS_TFQF_Msk                      (0x1UL<<FDCAN_TXFQS_TFQF_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXFQS_TFQF                          FDCAN_TXFQS_TFQF_Msk                              /*!< Tx FIFO/Queue Full */

/********************************* Bit definition for FDCAN_TXESC register ******************************************/
#define FDCAN_TXESC_TBDS_Pos                      (0U)
#define FDCAN_TXESC_TBDS_Msk                      (0x7UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000007 */
#define FDCAN_TXESC_TBDS                          FDCAN_TXESC_TBDS_Msk                              /*!< Tx Buffer Data Field Size */
#define FDCAN_TXESC_TBDS_0                        (0x1UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXESC_TBDS_1                        (0x2UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXESC_TBDS_2                        (0x4UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000004 */

/********************************* Bit definition for FDCAN_TXBRP register ******************************************/
#define FDCAN_TXBRP_TRP0_Pos                      (0U)
#define FDCAN_TXBRP_TRP0_Msk                      (0x1UL<<FDCAN_TXBRP_TRP0_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXBRP_TRP0                          FDCAN_TXBRP_TRP0_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP1_Pos                      (1U)
#define FDCAN_TXBRP_TRP1_Msk                      (0x1UL<<FDCAN_TXBRP_TRP1_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXBRP_TRP1                          FDCAN_TXBRP_TRP1_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP2_Pos                      (2U)
#define FDCAN_TXBRP_TRP2_Msk                      (0x1UL<<FDCAN_TXBRP_TRP2_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXBRP_TRP2                          FDCAN_TXBRP_TRP2_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP3_Pos                      (3U)
#define FDCAN_TXBRP_TRP3_Msk                      (0x1UL<<FDCAN_TXBRP_TRP3_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXBRP_TRP3                          FDCAN_TXBRP_TRP3_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP4_Pos                      (4U)
#define FDCAN_TXBRP_TRP4_Msk                      (0x1UL<<FDCAN_TXBRP_TRP4_Pos)                     /*!< 0x00000010 */
#define FDCAN_TXBRP_TRP4                          FDCAN_TXBRP_TRP4_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP5_Pos                      (5U)
#define FDCAN_TXBRP_TRP5_Msk                      (0x1UL<<FDCAN_TXBRP_TRP5_Pos)                     /*!< 0x00000020 */
#define FDCAN_TXBRP_TRP5                          FDCAN_TXBRP_TRP5_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP6_Pos                      (6U)
#define FDCAN_TXBRP_TRP6_Msk                      (0x1UL<<FDCAN_TXBRP_TRP6_Pos)                     /*!< 0x00000040 */
#define FDCAN_TXBRP_TRP6                          FDCAN_TXBRP_TRP6_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP7_Pos                      (7U)
#define FDCAN_TXBRP_TRP7_Msk                      (0x1UL<<FDCAN_TXBRP_TRP7_Pos)                     /*!< 0x00000080 */
#define FDCAN_TXBRP_TRP7                          FDCAN_TXBRP_TRP7_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP8_Pos                      (8U)
#define FDCAN_TXBRP_TRP8_Msk                      (0x1UL<<FDCAN_TXBRP_TRP8_Pos)                     /*!< 0x00000100 */
#define FDCAN_TXBRP_TRP8                          FDCAN_TXBRP_TRP8_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP9_Pos                      (9U)
#define FDCAN_TXBRP_TRP9_Msk                      (0x1UL<<FDCAN_TXBRP_TRP9_Pos)                     /*!< 0x00000200 */
#define FDCAN_TXBRP_TRP9                          FDCAN_TXBRP_TRP9_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP10_Pos                     (10U)
#define FDCAN_TXBRP_TRP10_Msk                     (0x1UL<<FDCAN_TXBRP_TRP10_Pos)                    /*!< 0x00000400 */
#define FDCAN_TXBRP_TRP10                         FDCAN_TXBRP_TRP10_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP11_Pos                     (11U)
#define FDCAN_TXBRP_TRP11_Msk                     (0x1UL<<FDCAN_TXBRP_TRP11_Pos)                    /*!< 0x00000800 */
#define FDCAN_TXBRP_TRP11                         FDCAN_TXBRP_TRP11_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP12_Pos                     (12U)
#define FDCAN_TXBRP_TRP12_Msk                     (0x1UL<<FDCAN_TXBRP_TRP12_Pos)                    /*!< 0x00001000 */
#define FDCAN_TXBRP_TRP12                         FDCAN_TXBRP_TRP12_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP13_Pos                     (13U)
#define FDCAN_TXBRP_TRP13_Msk                     (0x1UL<<FDCAN_TXBRP_TRP13_Pos)                    /*!< 0x00002000 */
#define FDCAN_TXBRP_TRP13                         FDCAN_TXBRP_TRP13_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP14_Pos                     (14U)
#define FDCAN_TXBRP_TRP14_Msk                     (0x1UL<<FDCAN_TXBRP_TRP14_Pos)                    /*!< 0x00004000 */
#define FDCAN_TXBRP_TRP14                         FDCAN_TXBRP_TRP14_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP15_Pos                     (15U)
#define FDCAN_TXBRP_TRP15_Msk                     (0x1UL<<FDCAN_TXBRP_TRP15_Pos)                    /*!< 0x00008000 */
#define FDCAN_TXBRP_TRP15                         FDCAN_TXBRP_TRP15_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP16_Pos                     (16U)
#define FDCAN_TXBRP_TRP16_Msk                     (0x1UL<<FDCAN_TXBRP_TRP16_Pos)                    /*!< 0x00010000 */
#define FDCAN_TXBRP_TRP16                         FDCAN_TXBRP_TRP16_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP17_Pos                     (17U)
#define FDCAN_TXBRP_TRP17_Msk                     (0x1UL<<FDCAN_TXBRP_TRP17_Pos)                    /*!< 0x00020000 */
#define FDCAN_TXBRP_TRP17                         FDCAN_TXBRP_TRP17_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP18_Pos                     (18U)
#define FDCAN_TXBRP_TRP18_Msk                     (0x1UL<<FDCAN_TXBRP_TRP18_Pos)                    /*!< 0x00040000 */
#define FDCAN_TXBRP_TRP18                         FDCAN_TXBRP_TRP18_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP19_Pos                     (19U)
#define FDCAN_TXBRP_TRP19_Msk                     (0x1UL<<FDCAN_TXBRP_TRP19_Pos)                    /*!< 0x00080000 */
#define FDCAN_TXBRP_TRP19                         FDCAN_TXBRP_TRP19_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP20_Pos                     (20U)
#define FDCAN_TXBRP_TRP20_Msk                     (0x1UL<<FDCAN_TXBRP_TRP20_Pos)                    /*!< 0x00100000 */
#define FDCAN_TXBRP_TRP20                         FDCAN_TXBRP_TRP20_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP21_Pos                     (21U)
#define FDCAN_TXBRP_TRP21_Msk                     (0x1UL<<FDCAN_TXBRP_TRP21_Pos)                    /*!< 0x00200000 */
#define FDCAN_TXBRP_TRP21                         FDCAN_TXBRP_TRP21_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP22_Pos                     (22U)
#define FDCAN_TXBRP_TRP22_Msk                     (0x1UL<<FDCAN_TXBRP_TRP22_Pos)                    /*!< 0x00400000 */
#define FDCAN_TXBRP_TRP22                         FDCAN_TXBRP_TRP22_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP23_Pos                     (23U)
#define FDCAN_TXBRP_TRP23_Msk                     (0x1UL<<FDCAN_TXBRP_TRP23_Pos)                    /*!< 0x00800000 */
#define FDCAN_TXBRP_TRP23                         FDCAN_TXBRP_TRP23_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP24_Pos                     (24U)
#define FDCAN_TXBRP_TRP24_Msk                     (0x1UL<<FDCAN_TXBRP_TRP24_Pos)                    /*!< 0x01000000 */
#define FDCAN_TXBRP_TRP24                         FDCAN_TXBRP_TRP24_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP25_Pos                     (25U)
#define FDCAN_TXBRP_TRP25_Msk                     (0x1UL<<FDCAN_TXBRP_TRP25_Pos)                    /*!< 0x02000000 */
#define FDCAN_TXBRP_TRP25                         FDCAN_TXBRP_TRP25_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP26_Pos                     (26U)
#define FDCAN_TXBRP_TRP26_Msk                     (0x1UL<<FDCAN_TXBRP_TRP26_Pos)                    /*!< 0x04000000 */
#define FDCAN_TXBRP_TRP26                         FDCAN_TXBRP_TRP26_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP27_Pos                     (27U)
#define FDCAN_TXBRP_TRP27_Msk                     (0x1UL<<FDCAN_TXBRP_TRP27_Pos)                    /*!< 0x08000000 */
#define FDCAN_TXBRP_TRP27                         FDCAN_TXBRP_TRP27_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP28_Pos                     (28U)
#define FDCAN_TXBRP_TRP28_Msk                     (0x1UL<<FDCAN_TXBRP_TRP28_Pos)                    /*!< 0x10000000 */
#define FDCAN_TXBRP_TRP28                         FDCAN_TXBRP_TRP28_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP29_Pos                     (29U)
#define FDCAN_TXBRP_TRP29_Msk                     (0x1UL<<FDCAN_TXBRP_TRP29_Pos)                    /*!< 0x20000000 */
#define FDCAN_TXBRP_TRP29                         FDCAN_TXBRP_TRP29_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP30_Pos                     (30U)
#define FDCAN_TXBRP_TRP30_Msk                     (0x1UL<<FDCAN_TXBRP_TRP30_Pos)                    /*!< 0x40000000 */
#define FDCAN_TXBRP_TRP30                         FDCAN_TXBRP_TRP30_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP31_Pos                     (31U)
#define FDCAN_TXBRP_TRP31_Msk                     (0x1UL<<FDCAN_TXBRP_TRP31_Pos)                    /*!< 0x80000000 */
#define FDCAN_TXBRP_TRP31                         FDCAN_TXBRP_TRP31_Msk                             /*!< Transmission Request Pending */

/********************************* Bit definition for FDCAN_TXBAR register ******************************************/
#define FDCAN_TXBAR_AR0_Pos                       (0U)
#define FDCAN_TXBAR_AR0_Msk                       (0x1UL<<FDCAN_TXBAR_AR0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBAR_AR0                           FDCAN_TXBAR_AR0_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR1_Pos                       (1U)
#define FDCAN_TXBAR_AR1_Msk                       (0x1UL<<FDCAN_TXBAR_AR1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBAR_AR1                           FDCAN_TXBAR_AR1_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR2_Pos                       (2U)
#define FDCAN_TXBAR_AR2_Msk                       (0x1UL<<FDCAN_TXBAR_AR2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBAR_AR2                           FDCAN_TXBAR_AR2_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR3_Pos                       (3U)
#define FDCAN_TXBAR_AR3_Msk                       (0x1UL<<FDCAN_TXBAR_AR3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBAR_AR3                           FDCAN_TXBAR_AR3_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR4_Pos                       (4U)
#define FDCAN_TXBAR_AR4_Msk                       (0x1UL<<FDCAN_TXBAR_AR4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBAR_AR4                           FDCAN_TXBAR_AR4_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR5_Pos                       (5U)
#define FDCAN_TXBAR_AR5_Msk                       (0x1UL<<FDCAN_TXBAR_AR5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBAR_AR5                           FDCAN_TXBAR_AR5_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR6_Pos                       (6U)
#define FDCAN_TXBAR_AR6_Msk                       (0x1UL<<FDCAN_TXBAR_AR6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBAR_AR6                           FDCAN_TXBAR_AR6_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR7_Pos                       (7U)
#define FDCAN_TXBAR_AR7_Msk                       (0x1UL<<FDCAN_TXBAR_AR7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBAR_AR7                           FDCAN_TXBAR_AR7_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR8_Pos                       (8U)
#define FDCAN_TXBAR_AR8_Msk                       (0x1UL<<FDCAN_TXBAR_AR8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBAR_AR8                           FDCAN_TXBAR_AR8_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR9_Pos                       (9U)
#define FDCAN_TXBAR_AR9_Msk                       (0x1UL<<FDCAN_TXBAR_AR9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBAR_AR9                           FDCAN_TXBAR_AR9_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR10_Pos                      (10U)
#define FDCAN_TXBAR_AR10_Msk                      (0x1UL<<FDCAN_TXBAR_AR10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBAR_AR10                          FDCAN_TXBAR_AR10_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR11_Pos                      (11U)
#define FDCAN_TXBAR_AR11_Msk                      (0x1UL<<FDCAN_TXBAR_AR11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBAR_AR11                          FDCAN_TXBAR_AR11_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR12_Pos                      (12U)
#define FDCAN_TXBAR_AR12_Msk                      (0x1UL<<FDCAN_TXBAR_AR12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBAR_AR12                          FDCAN_TXBAR_AR12_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR13_Pos                      (13U)
#define FDCAN_TXBAR_AR13_Msk                      (0x1UL<<FDCAN_TXBAR_AR13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBAR_AR13                          FDCAN_TXBAR_AR13_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR14_Pos                      (14U)
#define FDCAN_TXBAR_AR14_Msk                      (0x1UL<<FDCAN_TXBAR_AR14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBAR_AR14                          FDCAN_TXBAR_AR14_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR15_Pos                      (15U)
#define FDCAN_TXBAR_AR15_Msk                      (0x1UL<<FDCAN_TXBAR_AR15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBAR_AR15                          FDCAN_TXBAR_AR15_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR16_Pos                      (16U)
#define FDCAN_TXBAR_AR16_Msk                      (0x1UL<<FDCAN_TXBAR_AR16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBAR_AR16                          FDCAN_TXBAR_AR16_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR17_Pos                      (17U)
#define FDCAN_TXBAR_AR17_Msk                      (0x1UL<<FDCAN_TXBAR_AR17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBAR_AR17                          FDCAN_TXBAR_AR17_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR18_Pos                      (18U)
#define FDCAN_TXBAR_AR18_Msk                      (0x1UL<<FDCAN_TXBAR_AR18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBAR_AR18                          FDCAN_TXBAR_AR18_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR19_Pos                      (19U)
#define FDCAN_TXBAR_AR19_Msk                      (0x1UL<<FDCAN_TXBAR_AR19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBAR_AR19                          FDCAN_TXBAR_AR19_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR20_Pos                      (20U)
#define FDCAN_TXBAR_AR20_Msk                      (0x1UL<<FDCAN_TXBAR_AR20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBAR_AR20                          FDCAN_TXBAR_AR20_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR21_Pos                      (21U)
#define FDCAN_TXBAR_AR21_Msk                      (0x1UL<<FDCAN_TXBAR_AR21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBAR_AR21                          FDCAN_TXBAR_AR21_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR22_Pos                      (22U)
#define FDCAN_TXBAR_AR22_Msk                      (0x1UL<<FDCAN_TXBAR_AR22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBAR_AR22                          FDCAN_TXBAR_AR22_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR23_Pos                      (23U)
#define FDCAN_TXBAR_AR23_Msk                      (0x1UL<<FDCAN_TXBAR_AR23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBAR_AR23                          FDCAN_TXBAR_AR23_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR24_Pos                      (24U)
#define FDCAN_TXBAR_AR24_Msk                      (0x1UL<<FDCAN_TXBAR_AR24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBAR_AR24                          FDCAN_TXBAR_AR24_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR25_Pos                      (25U)
#define FDCAN_TXBAR_AR25_Msk                      (0x1UL<<FDCAN_TXBAR_AR25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBAR_AR25                          FDCAN_TXBAR_AR25_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR26_Pos                      (26U)
#define FDCAN_TXBAR_AR26_Msk                      (0x1UL<<FDCAN_TXBAR_AR26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBAR_AR26                          FDCAN_TXBAR_AR26_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR27_Pos                      (27U)
#define FDCAN_TXBAR_AR27_Msk                      (0x1UL<<FDCAN_TXBAR_AR27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBAR_AR27                          FDCAN_TXBAR_AR27_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR28_Pos                      (28U)
#define FDCAN_TXBAR_AR28_Msk                      (0x1UL<<FDCAN_TXBAR_AR28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBAR_AR28                          FDCAN_TXBAR_AR28_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR29_Pos                      (29U)
#define FDCAN_TXBAR_AR29_Msk                      (0x1UL<<FDCAN_TXBAR_AR29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBAR_AR29                          FDCAN_TXBAR_AR29_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR30_Pos                      (30U)
#define FDCAN_TXBAR_AR30_Msk                      (0x1UL<<FDCAN_TXBAR_AR30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBAR_AR30                          FDCAN_TXBAR_AR30_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR31_Pos                      (31U)
#define FDCAN_TXBAR_AR31_Msk                      (0x1UL<<FDCAN_TXBAR_AR31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBAR_AR31                          FDCAN_TXBAR_AR31_Msk                              /*!< Add Request */

/********************************* Bit definition for FDCAN_TXBCR register ******************************************/
#define FDCAN_TXBCR_CR0_Pos                       (0U)
#define FDCAN_TXBCR_CR0_Msk                       (0x1UL<<FDCAN_TXBCR_CR0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBCR_CR0                           FDCAN_TXBCR_CR0_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR1_Pos                       (1U)
#define FDCAN_TXBCR_CR1_Msk                       (0x1UL<<FDCAN_TXBCR_CR1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBCR_CR1                           FDCAN_TXBCR_CR1_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR2_Pos                       (2U)
#define FDCAN_TXBCR_CR2_Msk                       (0x1UL<<FDCAN_TXBCR_CR2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBCR_CR2                           FDCAN_TXBCR_CR2_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR3_Pos                       (3U)
#define FDCAN_TXBCR_CR3_Msk                       (0x1UL<<FDCAN_TXBCR_CR3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBCR_CR3                           FDCAN_TXBCR_CR3_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR4_Pos                       (4U)
#define FDCAN_TXBCR_CR4_Msk                       (0x1UL<<FDCAN_TXBCR_CR4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBCR_CR4                           FDCAN_TXBCR_CR4_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR5_Pos                       (5U)
#define FDCAN_TXBCR_CR5_Msk                       (0x1UL<<FDCAN_TXBCR_CR5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBCR_CR5                           FDCAN_TXBCR_CR5_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR6_Pos                       (6U)
#define FDCAN_TXBCR_CR6_Msk                       (0x1UL<<FDCAN_TXBCR_CR6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBCR_CR6                           FDCAN_TXBCR_CR6_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR7_Pos                       (7U)
#define FDCAN_TXBCR_CR7_Msk                       (0x1UL<<FDCAN_TXBCR_CR7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBCR_CR7                           FDCAN_TXBCR_CR7_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR8_Pos                       (8U)
#define FDCAN_TXBCR_CR8_Msk                       (0x1UL<<FDCAN_TXBCR_CR8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBCR_CR8                           FDCAN_TXBCR_CR8_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR9_Pos                       (9U)
#define FDCAN_TXBCR_CR9_Msk                       (0x1UL<<FDCAN_TXBCR_CR9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBCR_CR9                           FDCAN_TXBCR_CR9_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR10_Pos                      (10U)
#define FDCAN_TXBCR_CR10_Msk                      (0x1UL<<FDCAN_TXBCR_CR10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBCR_CR10                          FDCAN_TXBCR_CR10_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR11_Pos                      (11U)
#define FDCAN_TXBCR_CR11_Msk                      (0x1UL<<FDCAN_TXBCR_CR11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBCR_CR11                          FDCAN_TXBCR_CR11_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR12_Pos                      (12U)
#define FDCAN_TXBCR_CR12_Msk                      (0x1UL<<FDCAN_TXBCR_CR12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBCR_CR12                          FDCAN_TXBCR_CR12_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR13_Pos                      (13U)
#define FDCAN_TXBCR_CR13_Msk                      (0x1UL<<FDCAN_TXBCR_CR13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBCR_CR13                          FDCAN_TXBCR_CR13_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR14_Pos                      (14U)
#define FDCAN_TXBCR_CR14_Msk                      (0x1UL<<FDCAN_TXBCR_CR14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBCR_CR14                          FDCAN_TXBCR_CR14_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR15_Pos                      (15U)
#define FDCAN_TXBCR_CR15_Msk                      (0x1UL<<FDCAN_TXBCR_CR15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBCR_CR15                          FDCAN_TXBCR_CR15_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR16_Pos                      (16U)
#define FDCAN_TXBCR_CR16_Msk                      (0x1UL<<FDCAN_TXBCR_CR16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBCR_CR16                          FDCAN_TXBCR_CR16_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR17_Pos                      (17U)
#define FDCAN_TXBCR_CR17_Msk                      (0x1UL<<FDCAN_TXBCR_CR17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBCR_CR17                          FDCAN_TXBCR_CR17_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR18_Pos                      (18U)
#define FDCAN_TXBCR_CR18_Msk                      (0x1UL<<FDCAN_TXBCR_CR18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBCR_CR18                          FDCAN_TXBCR_CR18_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR19_Pos                      (19U)
#define FDCAN_TXBCR_CR19_Msk                      (0x1UL<<FDCAN_TXBCR_CR19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBCR_CR19                          FDCAN_TXBCR_CR19_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR20_Pos                      (20U)
#define FDCAN_TXBCR_CR20_Msk                      (0x1UL<<FDCAN_TXBCR_CR20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBCR_CR20                          FDCAN_TXBCR_CR20_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR21_Pos                      (21U)
#define FDCAN_TXBCR_CR21_Msk                      (0x1UL<<FDCAN_TXBCR_CR21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBCR_CR21                          FDCAN_TXBCR_CR21_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR22_Pos                      (22U)
#define FDCAN_TXBCR_CR22_Msk                      (0x1UL<<FDCAN_TXBCR_CR22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBCR_CR22                          FDCAN_TXBCR_CR22_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR23_Pos                      (23U)
#define FDCAN_TXBCR_CR23_Msk                      (0x1UL<<FDCAN_TXBCR_CR23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBCR_CR23                          FDCAN_TXBCR_CR23_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR24_Pos                      (24U)
#define FDCAN_TXBCR_CR24_Msk                      (0x1UL<<FDCAN_TXBCR_CR24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBCR_CR24                          FDCAN_TXBCR_CR24_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR25_Pos                      (25U)
#define FDCAN_TXBCR_CR25_Msk                      (0x1UL<<FDCAN_TXBCR_CR25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBCR_CR25                          FDCAN_TXBCR_CR25_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR26_Pos                      (26U)
#define FDCAN_TXBCR_CR26_Msk                      (0x1UL<<FDCAN_TXBCR_CR26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBCR_CR26                          FDCAN_TXBCR_CR26_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR27_Pos                      (27U)
#define FDCAN_TXBCR_CR27_Msk                      (0x1UL<<FDCAN_TXBCR_CR27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBCR_CR27                          FDCAN_TXBCR_CR27_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR28_Pos                      (28U)
#define FDCAN_TXBCR_CR28_Msk                      (0x1UL<<FDCAN_TXBCR_CR28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBCR_CR28                          FDCAN_TXBCR_CR28_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR29_Pos                      (29U)
#define FDCAN_TXBCR_CR29_Msk                      (0x1UL<<FDCAN_TXBCR_CR29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBCR_CR29                          FDCAN_TXBCR_CR29_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR30_Pos                      (30U)
#define FDCAN_TXBCR_CR30_Msk                      (0x1UL<<FDCAN_TXBCR_CR30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBCR_CR30                          FDCAN_TXBCR_CR30_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR31_Pos                      (31U)
#define FDCAN_TXBCR_CR31_Msk                      (0x1UL<<FDCAN_TXBCR_CR31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBCR_CR31                          FDCAN_TXBCR_CR31_Msk                              /*!< Cancellation Request */

/********************************* Bit definition for FDCAN_TXBTO register ******************************************/
#define FDCAN_TXBTO_TO0_Pos                       (0U)
#define FDCAN_TXBTO_TO0_Msk                       (0x1UL<<FDCAN_TXBTO_TO0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBTO_TO0                           FDCAN_TXBTO_TO0_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO1_Pos                       (1U)
#define FDCAN_TXBTO_TO1_Msk                       (0x1UL<<FDCAN_TXBTO_TO1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBTO_TO1                           FDCAN_TXBTO_TO1_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO2_Pos                       (2U)
#define FDCAN_TXBTO_TO2_Msk                       (0x1UL<<FDCAN_TXBTO_TO2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBTO_TO2                           FDCAN_TXBTO_TO2_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO3_Pos                       (3U)
#define FDCAN_TXBTO_TO3_Msk                       (0x1UL<<FDCAN_TXBTO_TO3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBTO_TO3                           FDCAN_TXBTO_TO3_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO4_Pos                       (4U)
#define FDCAN_TXBTO_TO4_Msk                       (0x1UL<<FDCAN_TXBTO_TO4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBTO_TO4                           FDCAN_TXBTO_TO4_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO5_Pos                       (5U)
#define FDCAN_TXBTO_TO5_Msk                       (0x1UL<<FDCAN_TXBTO_TO5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBTO_TO5                           FDCAN_TXBTO_TO5_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO6_Pos                       (6U)
#define FDCAN_TXBTO_TO6_Msk                       (0x1UL<<FDCAN_TXBTO_TO6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBTO_TO6                           FDCAN_TXBTO_TO6_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO7_Pos                       (7U)
#define FDCAN_TXBTO_TO7_Msk                       (0x1UL<<FDCAN_TXBTO_TO7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBTO_TO7                           FDCAN_TXBTO_TO7_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO8_Pos                       (8U)
#define FDCAN_TXBTO_TO8_Msk                       (0x1UL<<FDCAN_TXBTO_TO8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBTO_TO8                           FDCAN_TXBTO_TO8_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO9_Pos                       (9U)
#define FDCAN_TXBTO_TO9_Msk                       (0x1UL<<FDCAN_TXBTO_TO9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBTO_TO9                           FDCAN_TXBTO_TO9_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO10_Pos                      (10U)
#define FDCAN_TXBTO_TO10_Msk                      (0x1UL<<FDCAN_TXBTO_TO10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBTO_TO10                          FDCAN_TXBTO_TO10_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO11_Pos                      (11U)
#define FDCAN_TXBTO_TO11_Msk                      (0x1UL<<FDCAN_TXBTO_TO11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBTO_TO11                          FDCAN_TXBTO_TO11_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO12_Pos                      (12U)
#define FDCAN_TXBTO_TO12_Msk                      (0x1UL<<FDCAN_TXBTO_TO12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBTO_TO12                          FDCAN_TXBTO_TO12_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO13_Pos                      (13U)
#define FDCAN_TXBTO_TO13_Msk                      (0x1UL<<FDCAN_TXBTO_TO13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBTO_TO13                          FDCAN_TXBTO_TO13_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO14_Pos                      (14U)
#define FDCAN_TXBTO_TO14_Msk                      (0x1UL<<FDCAN_TXBTO_TO14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBTO_TO14                          FDCAN_TXBTO_TO14_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO15_Pos                      (15U)
#define FDCAN_TXBTO_TO15_Msk                      (0x1UL<<FDCAN_TXBTO_TO15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBTO_TO15                          FDCAN_TXBTO_TO15_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO16_Pos                      (16U)
#define FDCAN_TXBTO_TO16_Msk                      (0x1UL<<FDCAN_TXBTO_TO16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBTO_TO16                          FDCAN_TXBTO_TO16_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO17_Pos                      (17U)
#define FDCAN_TXBTO_TO17_Msk                      (0x1UL<<FDCAN_TXBTO_TO17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBTO_TO17                          FDCAN_TXBTO_TO17_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO18_Pos                      (18U)
#define FDCAN_TXBTO_TO18_Msk                      (0x1UL<<FDCAN_TXBTO_TO18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBTO_TO18                          FDCAN_TXBTO_TO18_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO19_Pos                      (19U)
#define FDCAN_TXBTO_TO19_Msk                      (0x1UL<<FDCAN_TXBTO_TO19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBTO_TO19                          FDCAN_TXBTO_TO19_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO20_Pos                      (20U)
#define FDCAN_TXBTO_TO20_Msk                      (0x1UL<<FDCAN_TXBTO_TO20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBTO_TO20                          FDCAN_TXBTO_TO20_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO21_Pos                      (21U)
#define FDCAN_TXBTO_TO21_Msk                      (0x1UL<<FDCAN_TXBTO_TO21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBTO_TO21                          FDCAN_TXBTO_TO21_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO22_Pos                      (22U)
#define FDCAN_TXBTO_TO22_Msk                      (0x1UL<<FDCAN_TXBTO_TO22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBTO_TO22                          FDCAN_TXBTO_TO22_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO23_Pos                      (23U)
#define FDCAN_TXBTO_TO23_Msk                      (0x1UL<<FDCAN_TXBTO_TO23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBTO_TO23                          FDCAN_TXBTO_TO23_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO24_Pos                      (24U)
#define FDCAN_TXBTO_TO24_Msk                      (0x1UL<<FDCAN_TXBTO_TO24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBTO_TO24                          FDCAN_TXBTO_TO24_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO25_Pos                      (25U)
#define FDCAN_TXBTO_TO25_Msk                      (0x1UL<<FDCAN_TXBTO_TO25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBTO_TO25                          FDCAN_TXBTO_TO25_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO26_Pos                      (26U)
#define FDCAN_TXBTO_TO26_Msk                      (0x1UL<<FDCAN_TXBTO_TO26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBTO_TO26                          FDCAN_TXBTO_TO26_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO27_Pos                      (27U)
#define FDCAN_TXBTO_TO27_Msk                      (0x1UL<<FDCAN_TXBTO_TO27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBTO_TO27                          FDCAN_TXBTO_TO27_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO28_Pos                      (28U)
#define FDCAN_TXBTO_TO28_Msk                      (0x1UL<<FDCAN_TXBTO_TO28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBTO_TO28                          FDCAN_TXBTO_TO28_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO29_Pos                      (29U)
#define FDCAN_TXBTO_TO29_Msk                      (0x1UL<<FDCAN_TXBTO_TO29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBTO_TO29                          FDCAN_TXBTO_TO29_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO30_Pos                      (30U)
#define FDCAN_TXBTO_TO30_Msk                      (0x1UL<<FDCAN_TXBTO_TO30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBTO_TO30                          FDCAN_TXBTO_TO30_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO31_Pos                      (31U)
#define FDCAN_TXBTO_TO31_Msk                      (0x1UL<<FDCAN_TXBTO_TO31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBTO_TO31                          FDCAN_TXBTO_TO31_Msk                              /*!< Transmission Occurred */

/********************************* Bit definition for FDCAN_TXBCF register ******************************************/
#define FDCAN_TXBCF_CF0_Pos                       (0U)
#define FDCAN_TXBCF_CF0_Msk                       (0x1UL<<FDCAN_TXBCF_CF0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBCF_CF0                           FDCAN_TXBCF_CF0_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF1_Pos                       (1U)
#define FDCAN_TXBCF_CF1_Msk                       (0x1UL<<FDCAN_TXBCF_CF1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBCF_CF1                           FDCAN_TXBCF_CF1_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF2_Pos                       (2U)
#define FDCAN_TXBCF_CF2_Msk                       (0x1UL<<FDCAN_TXBCF_CF2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBCF_CF2                           FDCAN_TXBCF_CF2_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF3_Pos                       (3U)
#define FDCAN_TXBCF_CF3_Msk                       (0x1UL<<FDCAN_TXBCF_CF3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBCF_CF3                           FDCAN_TXBCF_CF3_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF4_Pos                       (4U)
#define FDCAN_TXBCF_CF4_Msk                       (0x1UL<<FDCAN_TXBCF_CF4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBCF_CF4                           FDCAN_TXBCF_CF4_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF5_Pos                       (5U)
#define FDCAN_TXBCF_CF5_Msk                       (0x1UL<<FDCAN_TXBCF_CF5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBCF_CF5                           FDCAN_TXBCF_CF5_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF6_Pos                       (6U)
#define FDCAN_TXBCF_CF6_Msk                       (0x1UL<<FDCAN_TXBCF_CF6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBCF_CF6                           FDCAN_TXBCF_CF6_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF7_Pos                       (7U)
#define FDCAN_TXBCF_CF7_Msk                       (0x1UL<<FDCAN_TXBCF_CF7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBCF_CF7                           FDCAN_TXBCF_CF7_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF8_Pos                       (8U)
#define FDCAN_TXBCF_CF8_Msk                       (0x1UL<<FDCAN_TXBCF_CF8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBCF_CF8                           FDCAN_TXBCF_CF8_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF9_Pos                       (9U)
#define FDCAN_TXBCF_CF9_Msk                       (0x1UL<<FDCAN_TXBCF_CF9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBCF_CF9                           FDCAN_TXBCF_CF9_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF10_Pos                      (10U)
#define FDCAN_TXBCF_CF10_Msk                      (0x1UL<<FDCAN_TXBCF_CF10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBCF_CF10                          FDCAN_TXBCF_CF10_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF11_Pos                      (11U)
#define FDCAN_TXBCF_CF11_Msk                      (0x1UL<<FDCAN_TXBCF_CF11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBCF_CF11                          FDCAN_TXBCF_CF11_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF12_Pos                      (12U)
#define FDCAN_TXBCF_CF12_Msk                      (0x1UL<<FDCAN_TXBCF_CF12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBCF_CF12                          FDCAN_TXBCF_CF12_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF13_Pos                      (13U)
#define FDCAN_TXBCF_CF13_Msk                      (0x1UL<<FDCAN_TXBCF_CF13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBCF_CF13                          FDCAN_TXBCF_CF13_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF14_Pos                      (14U)
#define FDCAN_TXBCF_CF14_Msk                      (0x1UL<<FDCAN_TXBCF_CF14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBCF_CF14                          FDCAN_TXBCF_CF14_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF15_Pos                      (15U)
#define FDCAN_TXBCF_CF15_Msk                      (0x1UL<<FDCAN_TXBCF_CF15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBCF_CF15                          FDCAN_TXBCF_CF15_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF16_Pos                      (16U)
#define FDCAN_TXBCF_CF16_Msk                      (0x1UL<<FDCAN_TXBCF_CF16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBCF_CF16                          FDCAN_TXBCF_CF16_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF17_Pos                      (17U)
#define FDCAN_TXBCF_CF17_Msk                      (0x1UL<<FDCAN_TXBCF_CF17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBCF_CF17                          FDCAN_TXBCF_CF17_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF18_Pos                      (18U)
#define FDCAN_TXBCF_CF18_Msk                      (0x1UL<<FDCAN_TXBCF_CF18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBCF_CF18                          FDCAN_TXBCF_CF18_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF19_Pos                      (19U)
#define FDCAN_TXBCF_CF19_Msk                      (0x1UL<<FDCAN_TXBCF_CF19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBCF_CF19                          FDCAN_TXBCF_CF19_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF20_Pos                      (20U)
#define FDCAN_TXBCF_CF20_Msk                      (0x1UL<<FDCAN_TXBCF_CF20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBCF_CF20                          FDCAN_TXBCF_CF20_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF21_Pos                      (21U)
#define FDCAN_TXBCF_CF21_Msk                      (0x1UL<<FDCAN_TXBCF_CF21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBCF_CF21                          FDCAN_TXBCF_CF21_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF22_Pos                      (22U)
#define FDCAN_TXBCF_CF22_Msk                      (0x1UL<<FDCAN_TXBCF_CF22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBCF_CF22                          FDCAN_TXBCF_CF22_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF23_Pos                      (23U)
#define FDCAN_TXBCF_CF23_Msk                      (0x1UL<<FDCAN_TXBCF_CF23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBCF_CF23                          FDCAN_TXBCF_CF23_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF24_Pos                      (24U)
#define FDCAN_TXBCF_CF24_Msk                      (0x1UL<<FDCAN_TXBCF_CF24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBCF_CF24                          FDCAN_TXBCF_CF24_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF25_Pos                      (25U)
#define FDCAN_TXBCF_CF25_Msk                      (0x1UL<<FDCAN_TXBCF_CF25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBCF_CF25                          FDCAN_TXBCF_CF25_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF26_Pos                      (26U)
#define FDCAN_TXBCF_CF26_Msk                      (0x1UL<<FDCAN_TXBCF_CF26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBCF_CF26                          FDCAN_TXBCF_CF26_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF27_Pos                      (27U)
#define FDCAN_TXBCF_CF27_Msk                      (0x1UL<<FDCAN_TXBCF_CF27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBCF_CF27                          FDCAN_TXBCF_CF27_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF28_Pos                      (28U)
#define FDCAN_TXBCF_CF28_Msk                      (0x1UL<<FDCAN_TXBCF_CF28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBCF_CF28                          FDCAN_TXBCF_CF28_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF29_Pos                      (29U)
#define FDCAN_TXBCF_CF29_Msk                      (0x1UL<<FDCAN_TXBCF_CF29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBCF_CF29                          FDCAN_TXBCF_CF29_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF30_Pos                      (30U)
#define FDCAN_TXBCF_CF30_Msk                      (0x1UL<<FDCAN_TXBCF_CF30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBCF_CF30                          FDCAN_TXBCF_CF30_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF31_Pos                      (31U)
#define FDCAN_TXBCF_CF31_Msk                      (0x1UL<<FDCAN_TXBCF_CF31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBCF_CF31                          FDCAN_TXBCF_CF31_Msk                              /*!< Cancellation Finished */

/********************************* Bit definition for FDCAN_TXBTIE register *****************************************/
#define FDCAN_TXBTIE_TIE0_Pos                     (0U)
#define FDCAN_TXBTIE_TIE0_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE0_Pos)                    /*!< 0x00000001 */
#define FDCAN_TXBTIE_TIE0                         FDCAN_TXBTIE_TIE0_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE1_Pos                     (1U)
#define FDCAN_TXBTIE_TIE1_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE1_Pos)                    /*!< 0x00000002 */
#define FDCAN_TXBTIE_TIE1                         FDCAN_TXBTIE_TIE1_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE2_Pos                     (2U)
#define FDCAN_TXBTIE_TIE2_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE2_Pos)                    /*!< 0x00000004 */
#define FDCAN_TXBTIE_TIE2                         FDCAN_TXBTIE_TIE2_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE3_Pos                     (3U)
#define FDCAN_TXBTIE_TIE3_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE3_Pos)                    /*!< 0x00000008 */
#define FDCAN_TXBTIE_TIE3                         FDCAN_TXBTIE_TIE3_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE4_Pos                     (4U)
#define FDCAN_TXBTIE_TIE4_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE4_Pos)                    /*!< 0x00000010 */
#define FDCAN_TXBTIE_TIE4                         FDCAN_TXBTIE_TIE4_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE5_Pos                     (5U)
#define FDCAN_TXBTIE_TIE5_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE5_Pos)                    /*!< 0x00000020 */
#define FDCAN_TXBTIE_TIE5                         FDCAN_TXBTIE_TIE5_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE6_Pos                     (6U)
#define FDCAN_TXBTIE_TIE6_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE6_Pos)                    /*!< 0x00000040 */
#define FDCAN_TXBTIE_TIE6                         FDCAN_TXBTIE_TIE6_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE7_Pos                     (7U)
#define FDCAN_TXBTIE_TIE7_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE7_Pos)                    /*!< 0x00000080 */
#define FDCAN_TXBTIE_TIE7                         FDCAN_TXBTIE_TIE7_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE8_Pos                     (8U)
#define FDCAN_TXBTIE_TIE8_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE8_Pos)                    /*!< 0x00000100 */
#define FDCAN_TXBTIE_TIE8                         FDCAN_TXBTIE_TIE8_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE9_Pos                     (9U)
#define FDCAN_TXBTIE_TIE9_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE9_Pos)                    /*!< 0x00000200 */
#define FDCAN_TXBTIE_TIE9                         FDCAN_TXBTIE_TIE9_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE10_Pos                    (10U)
#define FDCAN_TXBTIE_TIE10_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE10_Pos)                   /*!< 0x00000400 */
#define FDCAN_TXBTIE_TIE10                        FDCAN_TXBTIE_TIE10_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE11_Pos                    (11U)
#define FDCAN_TXBTIE_TIE11_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE11_Pos)                   /*!< 0x00000800 */
#define FDCAN_TXBTIE_TIE11                        FDCAN_TXBTIE_TIE11_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE12_Pos                    (12U)
#define FDCAN_TXBTIE_TIE12_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE12_Pos)                   /*!< 0x00001000 */
#define FDCAN_TXBTIE_TIE12                        FDCAN_TXBTIE_TIE12_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE13_Pos                    (13U)
#define FDCAN_TXBTIE_TIE13_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE13_Pos)                   /*!< 0x00002000 */
#define FDCAN_TXBTIE_TIE13                        FDCAN_TXBTIE_TIE13_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE14_Pos                    (14U)
#define FDCAN_TXBTIE_TIE14_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE14_Pos)                   /*!< 0x00004000 */
#define FDCAN_TXBTIE_TIE14                        FDCAN_TXBTIE_TIE14_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE15_Pos                    (15U)
#define FDCAN_TXBTIE_TIE15_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE15_Pos)                   /*!< 0x00008000 */
#define FDCAN_TXBTIE_TIE15                        FDCAN_TXBTIE_TIE15_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE16_Pos                    (16U)
#define FDCAN_TXBTIE_TIE16_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE16_Pos)                   /*!< 0x00010000 */
#define FDCAN_TXBTIE_TIE16                        FDCAN_TXBTIE_TIE16_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE17_Pos                    (17U)
#define FDCAN_TXBTIE_TIE17_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE17_Pos)                   /*!< 0x00020000 */
#define FDCAN_TXBTIE_TIE17                        FDCAN_TXBTIE_TIE17_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE18_Pos                    (18U)
#define FDCAN_TXBTIE_TIE18_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE18_Pos)                   /*!< 0x00040000 */
#define FDCAN_TXBTIE_TIE18                        FDCAN_TXBTIE_TIE18_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE19_Pos                    (19U)
#define FDCAN_TXBTIE_TIE19_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE19_Pos)                   /*!< 0x00080000 */
#define FDCAN_TXBTIE_TIE19                        FDCAN_TXBTIE_TIE19_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE20_Pos                    (20U)
#define FDCAN_TXBTIE_TIE20_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE20_Pos)                   /*!< 0x00100000 */
#define FDCAN_TXBTIE_TIE20                        FDCAN_TXBTIE_TIE20_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE21_Pos                    (21U)
#define FDCAN_TXBTIE_TIE21_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE21_Pos)                   /*!< 0x00200000 */
#define FDCAN_TXBTIE_TIE21                        FDCAN_TXBTIE_TIE21_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE22_Pos                    (22U)
#define FDCAN_TXBTIE_TIE22_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE22_Pos)                   /*!< 0x00400000 */
#define FDCAN_TXBTIE_TIE22                        FDCAN_TXBTIE_TIE22_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE23_Pos                    (23U)
#define FDCAN_TXBTIE_TIE23_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE23_Pos)                   /*!< 0x00800000 */
#define FDCAN_TXBTIE_TIE23                        FDCAN_TXBTIE_TIE23_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE24_Pos                    (24U)
#define FDCAN_TXBTIE_TIE24_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE24_Pos)                   /*!< 0x01000000 */
#define FDCAN_TXBTIE_TIE24                        FDCAN_TXBTIE_TIE24_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE25_Pos                    (25U)
#define FDCAN_TXBTIE_TIE25_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE25_Pos)                   /*!< 0x02000000 */
#define FDCAN_TXBTIE_TIE25                        FDCAN_TXBTIE_TIE25_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE26_Pos                    (26U)
#define FDCAN_TXBTIE_TIE26_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE26_Pos)                   /*!< 0x04000000 */
#define FDCAN_TXBTIE_TIE26                        FDCAN_TXBTIE_TIE26_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE27_Pos                    (27U)
#define FDCAN_TXBTIE_TIE27_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE27_Pos)                   /*!< 0x08000000 */
#define FDCAN_TXBTIE_TIE27                        FDCAN_TXBTIE_TIE27_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE28_Pos                    (28U)
#define FDCAN_TXBTIE_TIE28_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE28_Pos)                   /*!< 0x10000000 */
#define FDCAN_TXBTIE_TIE28                        FDCAN_TXBTIE_TIE28_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE29_Pos                    (29U)
#define FDCAN_TXBTIE_TIE29_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE29_Pos)                   /*!< 0x20000000 */
#define FDCAN_TXBTIE_TIE29                        FDCAN_TXBTIE_TIE29_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE30_Pos                    (30U)
#define FDCAN_TXBTIE_TIE30_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE30_Pos)                   /*!< 0x40000000 */
#define FDCAN_TXBTIE_TIE30                        FDCAN_TXBTIE_TIE30_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE31_Pos                    (31U)
#define FDCAN_TXBTIE_TIE31_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE31_Pos)                   /*!< 0x80000000 */
#define FDCAN_TXBTIE_TIE31                        FDCAN_TXBTIE_TIE31_Msk                            /*!< Transmission Interrupt Enable */

/********************************* Bit definition for FDCAN_TXBCIE register *****************************************/
#define FDCAN_TXBCIE_CFIE0_Pos                    (0U)
#define FDCAN_TXBCIE_CFIE0_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE0_Pos)                   /*!< 0x00000001 */
#define FDCAN_TXBCIE_CFIE0                        FDCAN_TXBCIE_CFIE0_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE1_Pos                    (1U)
#define FDCAN_TXBCIE_CFIE1_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE1_Pos)                   /*!< 0x00000002 */
#define FDCAN_TXBCIE_CFIE1                        FDCAN_TXBCIE_CFIE1_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE2_Pos                    (2U)
#define FDCAN_TXBCIE_CFIE2_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE2_Pos)                   /*!< 0x00000004 */
#define FDCAN_TXBCIE_CFIE2                        FDCAN_TXBCIE_CFIE2_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE3_Pos                    (3U)
#define FDCAN_TXBCIE_CFIE3_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE3_Pos)                   /*!< 0x00000008 */
#define FDCAN_TXBCIE_CFIE3                        FDCAN_TXBCIE_CFIE3_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE4_Pos                    (4U)
#define FDCAN_TXBCIE_CFIE4_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE4_Pos)                   /*!< 0x00000010 */
#define FDCAN_TXBCIE_CFIE4                        FDCAN_TXBCIE_CFIE4_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE5_Pos                    (5U)
#define FDCAN_TXBCIE_CFIE5_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE5_Pos)                   /*!< 0x00000020 */
#define FDCAN_TXBCIE_CFIE5                        FDCAN_TXBCIE_CFIE5_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE6_Pos                    (6U)
#define FDCAN_TXBCIE_CFIE6_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE6_Pos)                   /*!< 0x00000040 */
#define FDCAN_TXBCIE_CFIE6                        FDCAN_TXBCIE_CFIE6_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE7_Pos                    (7U)
#define FDCAN_TXBCIE_CFIE7_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE7_Pos)                   /*!< 0x00000080 */
#define FDCAN_TXBCIE_CFIE7                        FDCAN_TXBCIE_CFIE7_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE8_Pos                    (8U)
#define FDCAN_TXBCIE_CFIE8_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE8_Pos)                   /*!< 0x00000100 */
#define FDCAN_TXBCIE_CFIE8                        FDCAN_TXBCIE_CFIE8_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE9_Pos                    (9U)
#define FDCAN_TXBCIE_CFIE9_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE9_Pos)                   /*!< 0x00000200 */
#define FDCAN_TXBCIE_CFIE9                        FDCAN_TXBCIE_CFIE9_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE10_Pos                   (10U)
#define FDCAN_TXBCIE_CFIE10_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE10_Pos)                  /*!< 0x00000400 */
#define FDCAN_TXBCIE_CFIE10                       FDCAN_TXBCIE_CFIE10_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE11_Pos                   (11U)
#define FDCAN_TXBCIE_CFIE11_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE11_Pos)                  /*!< 0x00000800 */
#define FDCAN_TXBCIE_CFIE11                       FDCAN_TXBCIE_CFIE11_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE12_Pos                   (12U)
#define FDCAN_TXBCIE_CFIE12_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE12_Pos)                  /*!< 0x00001000 */
#define FDCAN_TXBCIE_CFIE12                       FDCAN_TXBCIE_CFIE12_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE13_Pos                   (13U)
#define FDCAN_TXBCIE_CFIE13_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE13_Pos)                  /*!< 0x00002000 */
#define FDCAN_TXBCIE_CFIE13                       FDCAN_TXBCIE_CFIE13_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE14_Pos                   (14U)
#define FDCAN_TXBCIE_CFIE14_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE14_Pos)                  /*!< 0x00004000 */
#define FDCAN_TXBCIE_CFIE14                       FDCAN_TXBCIE_CFIE14_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE15_Pos                   (15U)
#define FDCAN_TXBCIE_CFIE15_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE15_Pos)                  /*!< 0x00008000 */
#define FDCAN_TXBCIE_CFIE15                       FDCAN_TXBCIE_CFIE15_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE16_Pos                   (16U)
#define FDCAN_TXBCIE_CFIE16_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE16_Pos)                  /*!< 0x00010000 */
#define FDCAN_TXBCIE_CFIE16                       FDCAN_TXBCIE_CFIE16_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE17_Pos                   (17U)
#define FDCAN_TXBCIE_CFIE17_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE17_Pos)                  /*!< 0x00020000 */
#define FDCAN_TXBCIE_CFIE17                       FDCAN_TXBCIE_CFIE17_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE18_Pos                   (18U)
#define FDCAN_TXBCIE_CFIE18_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE18_Pos)                  /*!< 0x00040000 */
#define FDCAN_TXBCIE_CFIE18                       FDCAN_TXBCIE_CFIE18_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE19_Pos                   (19U)
#define FDCAN_TXBCIE_CFIE19_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE19_Pos)                  /*!< 0x00080000 */
#define FDCAN_TXBCIE_CFIE19                       FDCAN_TXBCIE_CFIE19_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE20_Pos                   (20U)
#define FDCAN_TXBCIE_CFIE20_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE20_Pos)                  /*!< 0x00100000 */
#define FDCAN_TXBCIE_CFIE20                       FDCAN_TXBCIE_CFIE20_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE21_Pos                   (21U)
#define FDCAN_TXBCIE_CFIE21_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE21_Pos)                  /*!< 0x00200000 */
#define FDCAN_TXBCIE_CFIE21                       FDCAN_TXBCIE_CFIE21_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE22_Pos                   (22U)
#define FDCAN_TXBCIE_CFIE22_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE22_Pos)                  /*!< 0x00400000 */
#define FDCAN_TXBCIE_CFIE22                       FDCAN_TXBCIE_CFIE22_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE23_Pos                   (23U)
#define FDCAN_TXBCIE_CFIE23_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE23_Pos)                  /*!< 0x00800000 */
#define FDCAN_TXBCIE_CFIE23                       FDCAN_TXBCIE_CFIE23_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE24_Pos                   (24U)
#define FDCAN_TXBCIE_CFIE24_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE24_Pos)                  /*!< 0x01000000 */
#define FDCAN_TXBCIE_CFIE24                       FDCAN_TXBCIE_CFIE24_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE25_Pos                   (25U)
#define FDCAN_TXBCIE_CFIE25_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE25_Pos)                  /*!< 0x02000000 */
#define FDCAN_TXBCIE_CFIE25                       FDCAN_TXBCIE_CFIE25_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE26_Pos                   (26U)
#define FDCAN_TXBCIE_CFIE26_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE26_Pos)                  /*!< 0x04000000 */
#define FDCAN_TXBCIE_CFIE26                       FDCAN_TXBCIE_CFIE26_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE27_Pos                   (27U)
#define FDCAN_TXBCIE_CFIE27_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE27_Pos)                  /*!< 0x08000000 */
#define FDCAN_TXBCIE_CFIE27                       FDCAN_TXBCIE_CFIE27_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE28_Pos                   (28U)
#define FDCAN_TXBCIE_CFIE28_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE28_Pos)                  /*!< 0x10000000 */
#define FDCAN_TXBCIE_CFIE28                       FDCAN_TXBCIE_CFIE28_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE29_Pos                   (29U)
#define FDCAN_TXBCIE_CFIE29_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE29_Pos)                  /*!< 0x20000000 */
#define FDCAN_TXBCIE_CFIE29                       FDCAN_TXBCIE_CFIE29_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE30_Pos                   (30U)
#define FDCAN_TXBCIE_CFIE30_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE30_Pos)                  /*!< 0x40000000 */
#define FDCAN_TXBCIE_CFIE30                       FDCAN_TXBCIE_CFIE30_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE31_Pos                   (31U)
#define FDCAN_TXBCIE_CFIE31_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE31_Pos)                  /*!< 0x80000000 */
#define FDCAN_TXBCIE_CFIE31                       FDCAN_TXBCIE_CFIE31_Msk                           /*!< Cancellation Finished Interrupt Enable */

/********************************* Bit definition for FDCAN_TXEFC register ******************************************/
#define FDCAN_TXEFC_EFSA_Pos                      (2U)
#define FDCAN_TXEFC_EFSA_Msk                      (0x3FFFUL<<FDCAN_TXEFC_EFSA_Pos)                  /*!< 0x0000FFFC */
#define FDCAN_TXEFC_EFSA                          FDCAN_TXEFC_EFSA_Msk                              /*!< Event FIFO Start Address */
#define FDCAN_TXEFC_EFS_Pos                       (16U)
#define FDCAN_TXEFC_EFS_Msk                       (0x3FUL<<FDCAN_TXEFC_EFS_Pos)                     /*!< 0x003F0000 */
#define FDCAN_TXEFC_EFS                           FDCAN_TXEFC_EFS_Msk                               /*!< Event FIFO Size */
#define FDCAN_TXEFC_EFS_0                         (0x1UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00010000 */
#define FDCAN_TXEFC_EFS_1                         (0x2UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00020000 */
#define FDCAN_TXEFC_EFS_2                         (0x4UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00040000 */
#define FDCAN_TXEFC_EFS_3                         (0x8UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00080000 */
#define FDCAN_TXEFC_EFS_4                         (0x10UL<<FDCAN_TXEFC_EFS_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXEFC_EFS_5                         (0x20UL<<FDCAN_TXEFC_EFS_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXEFC_EFWM_Pos                      (24U)
#define FDCAN_TXEFC_EFWM_Msk                      (0x3FUL<<FDCAN_TXEFC_EFWM_Pos)                    /*!< 0x3F000000 */
#define FDCAN_TXEFC_EFWM                          FDCAN_TXEFC_EFWM_Msk                              /*!< Event FIFO Watermark */
#define FDCAN_TXEFC_EFWM_0                        (0x1UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXEFC_EFWM_1                        (0x2UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXEFC_EFWM_2                        (0x4UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXEFC_EFWM_3                        (0x8UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXEFC_EFWM_4                        (0x10UL<<FDCAN_TXEFC_EFWM_Pos)                    /*!< 0x10000000 */
#define FDCAN_TXEFC_EFWM_5                        (0x20UL<<FDCAN_TXEFC_EFWM_Pos)                    /*!< 0x20000000 */

/********************************* Bit definition for FDCAN_TXEFS register ******************************************/
#define FDCAN_TXEFS_EFFL_Pos                      (0U)
#define FDCAN_TXEFS_EFFL_Msk                      (0x3FUL<<FDCAN_TXEFS_EFFL_Pos)                    /*!< 0x0000003F */
#define FDCAN_TXEFS_EFFL                          FDCAN_TXEFS_EFFL_Msk                              /*!< Event FIFO Fill Level */
#define FDCAN_TXEFS_EFFL_0                        (0x1UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXEFS_EFFL_1                        (0x2UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXEFS_EFFL_2                        (0x4UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXEFS_EFFL_3                        (0x8UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXEFS_EFFL_4                        (0x10UL<<FDCAN_TXEFS_EFFL_Pos)                    /*!< 0x00000010 */
#define FDCAN_TXEFS_EFFL_5                        (0x20UL<<FDCAN_TXEFS_EFFL_Pos)                    /*!< 0x00000020 */
#define FDCAN_TXEFS_EFGI_Pos                      (8U)
#define FDCAN_TXEFS_EFGI_Msk                      (0x1FUL<<FDCAN_TXEFS_EFGI_Pos)                    /*!< 0x00001F00 */
#define FDCAN_TXEFS_EFGI                          FDCAN_TXEFS_EFGI_Msk                              /*!< Event FIFO Get Index */
#define FDCAN_TXEFS_EFGI_0                        (0x1UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000100 */
#define FDCAN_TXEFS_EFGI_1                        (0x2UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000200 */
#define FDCAN_TXEFS_EFGI_2                        (0x4UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXEFS_EFGI_3                        (0x8UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXEFS_EFGI_4                        (0x10UL<<FDCAN_TXEFS_EFGI_Pos)                    /*!< 0x00001000 */
#define FDCAN_TXEFS_EFPI_Pos                      (16U)
#define FDCAN_TXEFS_EFPI_Msk                      (0x1FUL<<FDCAN_TXEFS_EFPI_Pos)                    /*!< 0x001F0000 */
#define FDCAN_TXEFS_EFPI                          FDCAN_TXEFS_EFPI_Msk                              /*!< Event FIFO Put Index */
#define FDCAN_TXEFS_EFPI_0                        (0x1UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXEFS_EFPI_1                        (0x2UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXEFS_EFPI_2                        (0x4UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXEFS_EFPI_3                        (0x8UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXEFS_EFPI_4                        (0x10UL<<FDCAN_TXEFS_EFPI_Pos)                    /*!< 0x00100000 */
#define FDCAN_TXEFS_EFF_Pos                       (24U)
#define FDCAN_TXEFS_EFF_Msk                       (0x1UL<<FDCAN_TXEFS_EFF_Pos)                      /*!< 0x01000000 */
#define FDCAN_TXEFS_EFF                           FDCAN_TXEFS_EFF_Msk                               /*!< Event FIFO Full */
#define FDCAN_TXEFS_TEFL_Pos                      (25U)
#define FDCAN_TXEFS_TEFL_Msk                      (0x1UL<<FDCAN_TXEFS_TEFL_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXEFS_TEFL                          FDCAN_TXEFS_TEFL_Msk                              /*!< Tx Event FIFO Element Lost */

/********************************* Bit definition for FDCAN_TXEFA register ******************************************/
#define FDCAN_TXEFA_EFAI_Pos                      (0U)
#define FDCAN_TXEFA_EFAI_Msk                      (0x1FUL<<FDCAN_TXEFA_EFAI_Pos)                    /*!< 0x0000001F */
#define FDCAN_TXEFA_EFAI                          FDCAN_TXEFA_EFAI_Msk                              /*!< Event FIFO Acknowledge Index */
#define FDCAN_TXEFA_EFAI_0                        (0x1UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXEFA_EFAI_1                        (0x2UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXEFA_EFAI_2                        (0x4UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXEFA_EFAI_3                        (0x8UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXEFA_EFAI_4                        (0x10UL<<FDCAN_TXEFA_EFAI_Pos)                    /*!< 0x00000010 */

/********************************************************************************************************************/
/********************************* FMC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for FLASH_ACR register ********************************************/
#define FLASH_ACR_LATENCY_Pos                     (0U)
#define FLASH_ACR_LATENCY_Msk                     (0xFUL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x0000000F */
#define FLASH_ACR_LATENCY                         FLASH_ACR_LATENCY_Msk                             /*!< Latency */
#define FLASH_ACR_LATENCY_0                       (0x1UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000001 */
#define FLASH_ACR_LATENCY_1                       (0x2UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000002 */
#define FLASH_ACR_LATENCY_2                       (0x4UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000004 */
#define FLASH_ACR_LATENCY_3                       (0x8UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000008 */

/********************************* Bit definition for FLASH_KEYR register *******************************************/
#define FLASH_KEYR_KEY_Pos                        (0U)
#define FLASH_KEYR_KEY_Msk                        (0xFFFFFFFFUL<<FLASH_KEYR_KEY_Pos)                /*!< 0xFFFFFFFF */
#define FLASH_KEYR_KEY                            FLASH_KEYR_KEY_Msk                                /*!< Flash CR key */

/********************************* Bit definition for FLASH_OPTKEYR register ****************************************/
#define FLASH_OPTKEYR_OPTKEY_Pos                  (0U)
#define FLASH_OPTKEYR_OPTKEY_Msk                  (0xFFFFFFFFUL<<FLASH_OPTKEYR_OPTKEY_Pos)          /*!< 0xFFFFFFFF */
#define FLASH_OPTKEYR_OPTKEY                      FLASH_OPTKEYR_OPTKEY_Msk                          /*!< Flash Option key */

/********************************* Bit definition for FLASH_SR register *********************************************/
#define FLASH_SR_EOP_Pos                          (0U)
#define FLASH_SR_EOP_Msk                          (0x1UL<<FLASH_SR_EOP_Pos)                         /*!< 0x00000001 */
#define FLASH_SR_EOP                              FLASH_SR_EOP_Msk                                  /*!< End of operation */
#define FLASH_SR_WRPERR_Pos                       (4U)
#define FLASH_SR_WRPERR_Msk                       (0x1UL<<FLASH_SR_WRPERR_Pos)                      /*!< 0x00000010 */
#define FLASH_SR_WRPERR                           FLASH_SR_WRPERR_Msk                               /*!< Write protection error */
#define FLASH_SR_RDERR_Pos                        (14U)
#define FLASH_SR_RDERR_Msk                        (0x1UL<<FLASH_SR_RDERR_Pos)                       /*!< 0x00004000 */
#define FLASH_SR_RDERR                            FLASH_SR_RDERR_Msk                                
#define FLASH_SR_OPTVERR_Pos                      (15U)
#define FLASH_SR_OPTVERR_Msk                      (0x1UL<<FLASH_SR_OPTVERR_Pos)                     /*!< 0x00008000 */
#define FLASH_SR_OPTVERR                          FLASH_SR_OPTVERR_Msk                              /*!< Option and trimming bits loading validity error */
#define FLASH_SR_BSY0_Pos                         (16U)
#define FLASH_SR_BSY0_Msk                         (0x1UL<<FLASH_SR_BSY0_Pos)                        /*!< 0x00010000 */
#define FLASH_SR_BSY0                             FLASH_SR_BSY0_Msk                                 /*!< Bank0 Busy */
#define FLASH_SR_BSY1_Pos                         (17U)
#define FLASH_SR_BSY1_Msk                         (0x1UL<<FLASH_SR_BSY1_Pos)                        /*!< 0x00020000 */
#define FLASH_SR_BSY1                             FLASH_SR_BSY1_Msk                                 /*!< Bank1 Busy */

/********************************* Bit definition for FLASH_CR register *********************************************/
#define FLASH_CR_PG_Pos                           (0U)
#define FLASH_CR_PG_Msk                           (0x1UL<<FLASH_CR_PG_Pos)                          /*!< 0x00000001 */
#define FLASH_CR_PG                               FLASH_CR_PG_Msk                                   /*!< Page Program */
#define FLASH_CR_PER_Pos                          (1U)
#define FLASH_CR_PER_Msk                          (0x1UL<<FLASH_CR_PER_Pos)                         /*!< 0x00000002 */
#define FLASH_CR_PER                              FLASH_CR_PER_Msk                                  /*!< Page Erase */
#define FLASH_CR_MER0_Pos                         (2U)
#define FLASH_CR_MER0_Msk                         (0x1UL<<FLASH_CR_MER0_Pos)                        /*!< 0x00000004 */
#define FLASH_CR_MER0                             FLASH_CR_MER0_Msk                                 /*!< Bank 0 Mass Erase */
#define FLASH_CR_MER1_Pos                         (3U)
#define FLASH_CR_MER1_Msk                         (0x1UL<<FLASH_CR_MER1_Pos)                        /*!< 0x00000008 */
#define FLASH_CR_MER1                             FLASH_CR_MER1_Msk                                 /*!< Bank 1 Mass Erase */
#define FLASH_CR_SER_Pos                          (11U)
#define FLASH_CR_SER_Msk                          (0x1UL<<FLASH_CR_SER_Pos)                         /*!< 0x00000800 */
#define FLASH_CR_SER                              FLASH_CR_SER_Msk                                  /*!< Sector Erase */
#define FLASH_CR_BER_Pos                          (12U)
#define FLASH_CR_BER_Msk                          (0x1UL<<FLASH_CR_BER_Pos)                         /*!< 0x00001000 */
#define FLASH_CR_BER                              FLASH_CR_BER_Msk                                  /*!< Block Erase */
#define FLASH_CR_OPTSTRT_Pos                      (17U)
#define FLASH_CR_OPTSTRT_Msk                      (0x1UL<<FLASH_CR_OPTSTRT_Pos)                     /*!< 0x00020000 */
#define FLASH_CR_OPTSTRT                          FLASH_CR_OPTSTRT_Msk                              /*!< Option bytes Programming Start */
#define FLASH_CR_PGSTRT_Pos                       (19U)
#define FLASH_CR_PGSTRT_Msk                       (0x1UL<<FLASH_CR_PGSTRT_Pos)                      /*!< 0x00080000 */
#define FLASH_CR_PGSTRT                           FLASH_CR_PGSTRT_Msk                               /*!< Programming Start */
#define FLASH_CR_EOPIE_Pos                        (24U)
#define FLASH_CR_EOPIE_Msk                        (0x1UL<<FLASH_CR_EOPIE_Pos)                       /*!< 0x01000000 */
#define FLASH_CR_EOPIE                            FLASH_CR_EOPIE_Msk                                /*!< End of operation interrupt enable */
#define FLASH_CR_ERRIE_Pos                        (25U)
#define FLASH_CR_ERRIE_Msk                        (0x1UL<<FLASH_CR_ERRIE_Pos)                       /*!< 0x02000000 */
#define FLASH_CR_ERRIE                            FLASH_CR_ERRIE_Msk                                /*!< Error interrupt enable */
#define FLASH_CR_RDERRIE_Pos                      (26U)
#define FLASH_CR_RDERRIE_Msk                      (0x1UL<<FLASH_CR_RDERRIE_Pos)                     /*!< 0x04000000 */
#define FLASH_CR_RDERRIE                          FLASH_CR_RDERRIE_Msk                              
#define FLASH_CR_OBL_LAUNCH_Pos                   (27U)
#define FLASH_CR_OBL_LAUNCH_Msk                   (0x1UL<<FLASH_CR_OBL_LAUNCH_Pos)                  /*!< 0x08000000 */
#define FLASH_CR_OBL_LAUNCH                       FLASH_CR_OBL_LAUNCH_Msk                           /*!< Force the option bytes loading */
#define FLASH_CR_OPTLOCK_Pos                      (30U)
#define FLASH_CR_OPTLOCK_Msk                      (0x1UL<<FLASH_CR_OPTLOCK_Pos)                     /*!< 0x40000000 */
#define FLASH_CR_OPTLOCK                          FLASH_CR_OPTLOCK_Msk                              /*!< Option Lock */
#define FLASH_CR_LOCK_Pos                         (31U)
#define FLASH_CR_LOCK_Msk                         (0x1UL<<FLASH_CR_LOCK_Pos)                        /*!< 0x80000000 */
#define FLASH_CR_LOCK                             FLASH_CR_LOCK_Msk                                 /*!< Lock */

/********************************* Bit definition for FLASH_ECCR register *******************************************/
#define FLASH_ECCR_ADDR_ECC_Pos                   (0U)
#define FLASH_ECCR_ADDR_ECC_Msk                   (0x3FFFFUL<<FLASH_ECCR_ADDR_ECC_Pos)              /*!< 0x0003FFFF */
#define FLASH_ECCR_ADDR_ECC                       FLASH_ECCR_ADDR_ECC_Msk
#define FLASH_ECCR_BK_ECC_Pos                     (21U)
#define FLASH_ECCR_BK_ECC_Msk                     (0x1UL<<FLASH_ECCR_BK_ECC_Pos)                    /*!< 0x00200000 */
#define FLASH_ECCR_BK_ECC                         FLASH_ECCR_BK_ECC_Msk                             
#define FLASH_ECCR_SYSF_ECC_Pos                   (22U)
#define FLASH_ECCR_SYSF_ECC_Msk                   (0x1UL<<FLASH_ECCR_SYSF_ECC_Pos)                  /*!< 0x00400000 */
#define FLASH_ECCR_SYSF_ECC                       FLASH_ECCR_SYSF_ECC_Msk                           
#define FLASH_ECCR_ECCCIE_Pos                     (24U)
#define FLASH_ECCR_ECCCIE_Msk                     (0x1UL<<FLASH_ECCR_ECCCIE_Pos)                    /*!< 0x01000000 */
#define FLASH_ECCR_ECCCIE                         FLASH_ECCR_ECCCIE_Msk                             
#define FLASH_ECCR_ECCC_Pos                       (30U)
#define FLASH_ECCR_ECCC_Msk                       (0x1UL<<FLASH_ECCR_ECCC_Pos)                      /*!< 0x40000000 */
#define FLASH_ECCR_ECCC                           FLASH_ECCR_ECCC_Msk                               
#define FLASH_ECCR_ECCD_Pos                       (31U)
#define FLASH_ECCR_ECCD_Msk                       (0x1UL<<FLASH_ECCR_ECCD_Pos)                      /*!< 0x80000000 */
#define FLASH_ECCR_ECCD                           FLASH_ECCR_ECCD_Msk                               

/********************************* Bit definition for FLASH_OPTR1 register ******************************************/
#define FLASH_OPTR1_RDP_Pos                       (0U)
#define FLASH_OPTR1_RDP_Msk                       (0xFFUL<<FLASH_OPTR1_RDP_Pos)                     /*!< 0x000000FF */
#define FLASH_OPTR1_RDP                           FLASH_OPTR1_RDP_Msk                               /*!< Read Protection */
#define FLASH_OPTR1_IWDG_STOP_Pos                 (9U)
#define FLASH_OPTR1_IWDG_STOP_Msk                 (0x1UL<<FLASH_OPTR1_IWDG_STOP_Pos)                /*!< 0x00000200 */
#define FLASH_OPTR1_IWDG_STOP                     FLASH_OPTR1_IWDG_STOP_Msk                         
#define FLASH_OPTR1_IWDG_STDBY_Pos                (10U)
#define FLASH_OPTR1_IWDG_STDBY_Msk                (0x1UL<<FLASH_OPTR1_IWDG_STDBY_Pos)               /*!< 0x00000400 */
#define FLASH_OPTR1_IWDG_STDBY                    FLASH_OPTR1_IWDG_STDBY_Msk                        
#define FLASH_OPTR1_BFB_Pos                       (11U)
#define FLASH_OPTR1_BFB_Msk                       (0x1UL<<FLASH_OPTR1_BFB_Pos)                      /*!< 0x00000800 */
#define FLASH_OPTR1_BFB                           FLASH_OPTR1_BFB_Msk                               
#define FLASH_OPTR1_IWDG_SW_Pos                   (12U)
#define FLASH_OPTR1_IWDG_SW_Msk                   (0x1UL<<FLASH_OPTR1_IWDG_SW_Pos)                  /*!< 0x00001000 */
#define FLASH_OPTR1_IWDG_SW                       FLASH_OPTR1_IWDG_SW_Msk                           /*!< IWDG Software Enable */
#define FLASH_OPTR1_NRST_STOP_Pos                 (13U)
#define FLASH_OPTR1_NRST_STOP_Msk                 (0x1UL<<FLASH_OPTR1_NRST_STOP_Pos)                /*!< 0x00002000 */
#define FLASH_OPTR1_NRST_STOP                     FLASH_OPTR1_NRST_STOP_Msk                         
#define FLASH_OPTR1_NRST_STDBY_Pos                (14U)
#define FLASH_OPTR1_NRST_STDBY_Msk                (0x1UL<<FLASH_OPTR1_NRST_STDBY_Pos)               /*!< 0x00004000 */
#define FLASH_OPTR1_NRST_STDBY                    FLASH_OPTR1_NRST_STDBY_Msk                        
#define FLASH_OPTR1_WWDG_SW_Pos                   (15U)
#define FLASH_OPTR1_WWDG_SW_Msk                   (0x1UL<<FLASH_OPTR1_WWDG_SW_Pos)                  /*!< 0x00008000 */
#define FLASH_OPTR1_WWDG_SW                       FLASH_OPTR1_WWDG_SW_Msk                           

/********************************* Bit definition for FLASH_OPTR2 register ******************************************/
#define FLASH_OPTR2_NBOOT1_Pos                    (8U)
#define FLASH_OPTR2_NBOOT1_Msk                    (0x1UL<<FLASH_OPTR2_NBOOT1_Pos)                   /*!< 0x00000100 */
#define FLASH_OPTR2_NBOOT1                        FLASH_OPTR2_NBOOT1_Msk                            
#define FLASH_OPTR2_NRST_MODE_Pos                 (9U)
#define FLASH_OPTR2_NRST_MODE_Msk                 (0x3UL<<FLASH_OPTR2_NRST_MODE_Pos)                /*!< 0x00000600 */
#define FLASH_OPTR2_NRST_MODE                     FLASH_OPTR2_NRST_MODE_Msk
#define FLASH_OPTR2_NRST_MODE_0                   (0x1UL<<FLASH_OPTR2_NRST_MODE_Pos)                /*!< 0x00000200 */
#define FLASH_OPTR2_NRST_MODE_1                   (0x2UL<<FLASH_OPTR2_NRST_MODE_Pos)                /*!< 0x00000400 */
#define FLASH_OPTR2_SRAM_PE_Pos                   (11U)
#define FLASH_OPTR2_SRAM_PE_Msk                   (0x1UL<<FLASH_OPTR2_SRAM_PE_Pos)                  /*!< 0x00000800 */
#define FLASH_OPTR2_SRAM_PE                       FLASH_OPTR2_SRAM_PE_Msk                           
#define FLASH_OPTR2_CCMSRAM_RST_Pos               (12U)
#define FLASH_OPTR2_CCMSRAM_RST_Msk               (0x1UL<<FLASH_OPTR2_CCMSRAM_RST_Pos)              /*!< 0x00001000 */
#define FLASH_OPTR2_CCMSRAM_RST                   FLASH_OPTR2_CCMSRAM_RST_Msk                       
#define FLASH_OPTR2_NSWBOOT0_Pos                  (13U)
#define FLASH_OPTR2_NSWBOOT0_Msk                  (0x1UL<<FLASH_OPTR2_NSWBOOT0_Pos)                 /*!< 0x00002000 */
#define FLASH_OPTR2_NSWBOOT0                      FLASH_OPTR2_NSWBOOT0_Msk                          
#define FLASH_OPTR2_NBOOT0_Pos                    (14U)
#define FLASH_OPTR2_NBOOT0_Msk                    (0x1UL<<FLASH_OPTR2_NBOOT0_Pos)                   /*!< 0x00004000 */
#define FLASH_OPTR2_NBOOT0                        FLASH_OPTR2_NBOOT0_Msk                            
#define FLASH_OPTR2_BOOT_LOCK_Pos                 (15U)
#define FLASH_OPTR2_BOOT_LOCK_Msk                 (0x1UL<<FLASH_OPTR2_BOOT_LOCK_Pos)                /*!< 0x00008000 */
#define FLASH_OPTR2_BOOT_LOCK                     FLASH_OPTR2_BOOT_LOCK_Msk                         

/********************************* Bit definition for FLASH_WRPR register *******************************************/
#define FLASH_WRPR_WRP_Pos                        (0U)
#define FLASH_WRPR_WRP_Msk                        (0xFFUL<<FLASH_WRPR_WRP_Pos)                      /*!< 0x000000FF */
#define FLASH_WRPR_WRP                            FLASH_WRPR_WRP_Msk                                /*!< Write Protection Disable */

/********************************* Bit definition for FLASH_PCROP0SR register ***************************************/
#define FLASH_PCROP0SR_PCROP0SR_Pos               (0U)
#define FLASH_PCROP0SR_PCROP0SR_Msk               (0x1FFUL<<FLASH_PCROP0SR_PCROP0SR_Pos)            /*!< 0x000001FF */
#define FLASH_PCROP0SR_PCROP0SR                   FLASH_PCROP0SR_PCROP0SR_Msk

/********************************* Bit definition for FLASH_PCROP0ER register ***************************************/
#define FLASH_PCROP0ER_PCROP0ER_Pos               (0U)
#define FLASH_PCROP0ER_PCROP0ER_Msk               (0x1FFUL<<FLASH_PCROP0ER_PCROP0ER_Pos)            /*!< 0x000001FF */
#define FLASH_PCROP0ER_PCROP0ER                   FLASH_PCROP0ER_PCROP0ER_Msk

/********************************* Bit definition for FLASH_PCROP1SR register ***************************************/
#define FLASH_PCROP1SR_PCROP1SR_Pos               (0U)
#define FLASH_PCROP1SR_PCROP1SR_Msk               (0x1FFUL<<FLASH_PCROP1SR_PCROP1SR_Pos)            /*!< 0x000001FF */
#define FLASH_PCROP1SR_PCROP1SR                   FLASH_PCROP1SR_PCROP1SR_Msk

/********************************* Bit definition for FLASH_PCROP1ER register ***************************************/
#define FLASH_PCROP1ER_PCROP1ER_Pos               (0U)
#define FLASH_PCROP1ER_PCROP1ER_Msk               (0x1FFUL<<FLASH_PCROP1ER_PCROP1ER_Pos)            /*!< 0x000001FF */
#define FLASH_PCROP1ER_PCROP1ER                   FLASH_PCROP1ER_PCROP1ER_Msk
/********************************* Bit definition for FLASH_KEY register ********************************************/
#define FLASH_KEY1_Pos                            (0U)
#define FLASH_KEY1_Msk                            (0x45670123UL << FLASH_KEY1_Pos)                  /*!< 0x45670123 */
#define FLASH_KEY1                                FLASH_KEY1_Msk                                    /*!< Flash program erase key1 */
#define FLASH_KEY2_Pos                            (0U)
#define FLASH_KEY2_Msk                            (0xCDEF89ABUL << FLASH_KEY2_Pos)                  /*!< 0xCDEF89AB */
#define FLASH_KEY2                                FLASH_KEY2_Msk                                    /*!< Flash program erase key2: used with FLASH_PEKEY1 to unlock the write access to the FPEC. */

#define FLASH_OPTKEY1_Pos                         (0U)
#define FLASH_OPTKEY1_Msk                         (0x08192A3BUL << FLASH_OPTKEY1_Pos)               /*!< 0x08192A3B */
#define FLASH_OPTKEY1                             FLASH_OPTKEY1_Msk                                 /*!< Flash option key1 */
#define FLASH_OPTKEY2_Pos                         (0U)
#define FLASH_OPTKEY2_Msk                         (0x4C5D6E7FUL << FLASH_OPTKEY2_Pos)               /*!< 0x4C5D6E7F */
#define FLASH_OPTKEY2                             FLASH_OPTKEY2_Msk                                 /*!< Flash option key2: used with FLASH_OPTKEY1 to unlock the write access to the option byte block */

/*********************  Bits Define For Peripheral GPIO  *********************/
/*!< GPIO_MODER */
#define GPIO_MODER_MODE0_Pos                      (0U)
#define GPIO_MODER_MODE0_Msk                      (0x3UL << GPIO_MODER_MODE0_Pos)                   /*!< 0x00000003 */
#define GPIO_MODER_MODE0                          GPIO_MODER_MODE0_Msk                              /*!< MODE0[1:0] bits (desc MODE0) */
#define GPIO_MODER_MODE0_0                        (0x1UL << GPIO_MODER_MODE0_Pos)                   /*!< 0x00000001 */
#define GPIO_MODER_MODE0_1                        (0x2UL << GPIO_MODER_MODE0_Pos)                   /*!< 0x00000002 */

#define GPIO_MODER_MODE1_Pos                      (2U)
#define GPIO_MODER_MODE1_Msk                      (0x3UL << GPIO_MODER_MODE1_Pos)                   /*!< 0x0000000C */
#define GPIO_MODER_MODE1                          GPIO_MODER_MODE1_Msk                              /*!< MODE1[3:2] bits (desc MODE1) */
#define GPIO_MODER_MODE1_0                        (0x1UL << GPIO_MODER_MODE1_Pos)                   /*!< 0x00000004 */
#define GPIO_MODER_MODE1_1                        (0x2UL << GPIO_MODER_MODE1_Pos)                   /*!< 0x00000008 */

#define GPIO_MODER_MODE2_Pos                      (4U)
#define GPIO_MODER_MODE2_Msk                      (0x3UL << GPIO_MODER_MODE2_Pos)                   /*!< 0x00000030 */
#define GPIO_MODER_MODE2                          GPIO_MODER_MODE2_Msk                              /*!< MODE2[5:4] bits (desc MODE2) */
#define GPIO_MODER_MODE2_0                        (0x1UL << GPIO_MODER_MODE2_Pos)                   /*!< 0x00000010 */
#define GPIO_MODER_MODE2_1                        (0x2UL << GPIO_MODER_MODE2_Pos)                   /*!< 0x00000020 */

#define GPIO_MODER_MODE3_Pos                      (6U)
#define GPIO_MODER_MODE3_Msk                      (0x3UL << GPIO_MODER_MODE3_Pos)                   /*!< 0x000000C0 */
#define GPIO_MODER_MODE3                          GPIO_MODER_MODE3_Msk                              /*!< MODE3[7:6] bits (desc MODE3) */
#define GPIO_MODER_MODE3_0                        (0x1UL << GPIO_MODER_MODE3_Pos)                   /*!< 0x00000040 */
#define GPIO_MODER_MODE3_1                        (0x2UL << GPIO_MODER_MODE3_Pos)                   /*!< 0x00000080 */

#define GPIO_MODER_MODE4_Pos                      (8U)
#define GPIO_MODER_MODE4_Msk                      (0x3UL << GPIO_MODER_MODE4_Pos)                   /*!< 0x00000300 */
#define GPIO_MODER_MODE4                          GPIO_MODER_MODE4_Msk                              /*!< MODE4[9:8] bits (desc MODE4) */
#define GPIO_MODER_MODE4_0                        (0x1UL << GPIO_MODER_MODE4_Pos)                   /*!< 0x00000100 */
#define GPIO_MODER_MODE4_1                        (0x2UL << GPIO_MODER_MODE4_Pos)                   /*!< 0x00000200 */

#define GPIO_MODER_MODE5_Pos                      (10U)
#define GPIO_MODER_MODE5_Msk                      (0x3UL << GPIO_MODER_MODE5_Pos)                   /*!< 0x00000C00 */
#define GPIO_MODER_MODE5                          GPIO_MODER_MODE5_Msk                              /*!< MODE5[11:10] bits (desc MODE5) */
#define GPIO_MODER_MODE5_0                        (0x1UL << GPIO_MODER_MODE5_Pos)                   /*!< 0x00000400 */
#define GPIO_MODER_MODE5_1                        (0x2UL << GPIO_MODER_MODE5_Pos)                   /*!< 0x00000800 */

#define GPIO_MODER_MODE6_Pos                      (12U)
#define GPIO_MODER_MODE6_Msk                      (0x3UL << GPIO_MODER_MODE6_Pos)                   /*!< 0x00003000 */
#define GPIO_MODER_MODE6                          GPIO_MODER_MODE6_Msk                              /*!< MODE6[13:12] bits (desc MODE6) */
#define GPIO_MODER_MODE6_0                        (0x1UL << GPIO_MODER_MODE6_Pos)                   /*!< 0x00001000 */
#define GPIO_MODER_MODE6_1                        (0x2UL << GPIO_MODER_MODE6_Pos)                   /*!< 0x00002000 */

#define GPIO_MODER_MODE7_Pos                      (14U)
#define GPIO_MODER_MODE7_Msk                      (0x3UL << GPIO_MODER_MODE7_Pos)                   /*!< 0x0000C000 */
#define GPIO_MODER_MODE7                          GPIO_MODER_MODE7_Msk                              /*!< MODE7[15:14] bits (desc MODE7) */
#define GPIO_MODER_MODE7_0                        (0x1UL << GPIO_MODER_MODE7_Pos)                   /*!< 0x00004000 */
#define GPIO_MODER_MODE7_1                        (0x2UL << GPIO_MODER_MODE7_Pos)                   /*!< 0x00008000 */

#define GPIO_MODER_MODE8_Pos                      (16U)
#define GPIO_MODER_MODE8_Msk                      (0x3UL << GPIO_MODER_MODE8_Pos)                   /*!< 0x00030000 */
#define GPIO_MODER_MODE8                          GPIO_MODER_MODE8_Msk                              /*!< MODE8[17:16] bits (desc MODE8) */
#define GPIO_MODER_MODE8_0                        (0x1UL << GPIO_MODER_MODE8_Pos)                   /*!< 0x00010000 */
#define GPIO_MODER_MODE8_1                        (0x2UL << GPIO_MODER_MODE8_Pos)                   /*!< 0x00020000 */

#define GPIO_MODER_MODE9_Pos                      (18U)
#define GPIO_MODER_MODE9_Msk                      (0x3UL << GPIO_MODER_MODE9_Pos)                   /*!< 0x000C0000 */
#define GPIO_MODER_MODE9                          GPIO_MODER_MODE9_Msk                              /*!< MODE9[19:18] bits (desc MODE9) */
#define GPIO_MODER_MODE9_0                        (0x1UL << GPIO_MODER_MODE9_Pos)                   /*!< 0x00040000 */
#define GPIO_MODER_MODE9_1                        (0x2UL << GPIO_MODER_MODE9_Pos)                   /*!< 0x00080000 */

#define GPIO_MODER_MODE10_Pos                     (20U)
#define GPIO_MODER_MODE10_Msk                     (0x3UL << GPIO_MODER_MODE10_Pos)                  /*!< 0x00300000 */
#define GPIO_MODER_MODE10                         GPIO_MODER_MODE10_Msk                             /*!< MODE10[21:20] bits (desc MODE10) */
#define GPIO_MODER_MODE10_0                       (0x1UL << GPIO_MODER_MODE10_Pos)                  /*!< 0x00100000 */
#define GPIO_MODER_MODE10_1                       (0x2UL << GPIO_MODER_MODE10_Pos)                  /*!< 0x00200000 */

#define GPIO_MODER_MODE11_Pos                     (22U)
#define GPIO_MODER_MODE11_Msk                     (0x3UL << GPIO_MODER_MODE11_Pos)                  /*!< 0x00C00000 */
#define GPIO_MODER_MODE11                         GPIO_MODER_MODE11_Msk                             /*!< MODE11[23:22] bits (desc MODE11) */
#define GPIO_MODER_MODE11_0                       (0x1UL << GPIO_MODER_MODE11_Pos)                  /*!< 0x00400000 */
#define GPIO_MODER_MODE11_1                       (0x2UL << GPIO_MODER_MODE11_Pos)                  /*!< 0x00800000 */

#define GPIO_MODER_MODE12_Pos                     (24U)
#define GPIO_MODER_MODE12_Msk                     (0x3UL << GPIO_MODER_MODE12_Pos)                  /*!< 0x03000000 */
#define GPIO_MODER_MODE12                         GPIO_MODER_MODE12_Msk                             /*!< MODE12[25:24] bits (desc MODE12) */
#define GPIO_MODER_MODE12_0                       (0x1UL << GPIO_MODER_MODE12_Pos)                  /*!< 0x01000000 */
#define GPIO_MODER_MODE12_1                       (0x2UL << GPIO_MODER_MODE12_Pos)                  /*!< 0x02000000 */

#define GPIO_MODER_MODE13_Pos                     (26U)
#define GPIO_MODER_MODE13_Msk                     (0x3UL << GPIO_MODER_MODE13_Pos)                  /*!< 0x0C000000 */
#define GPIO_MODER_MODE13                         GPIO_MODER_MODE13_Msk                             /*!< MODE13[27:26] bits (desc MODE13) */
#define GPIO_MODER_MODE13_0                       (0x1UL << GPIO_MODER_MODE13_Pos)                  /*!< 0x04000000 */
#define GPIO_MODER_MODE13_1                       (0x2UL << GPIO_MODER_MODE13_Pos)                  /*!< 0x08000000 */

#define GPIO_MODER_MODE14_Pos                     (28U)
#define GPIO_MODER_MODE14_Msk                     (0x3UL << GPIO_MODER_MODE14_Pos)                  /*!< 0x30000000 */
#define GPIO_MODER_MODE14                         GPIO_MODER_MODE14_Msk                             /*!< MODE14[29:28] bits (desc MODE14) */
#define GPIO_MODER_MODE14_0                       (0x1UL << GPIO_MODER_MODE14_Pos)                  /*!< 0x10000000 */
#define GPIO_MODER_MODE14_1                       (0x2UL << GPIO_MODER_MODE14_Pos)                  /*!< 0x20000000 */

#define GPIO_MODER_MODE15_Pos                     (30U)
#define GPIO_MODER_MODE15_Msk                     (0x3UL << GPIO_MODER_MODE15_Pos)                  /*!< 0xC0000000 */
#define GPIO_MODER_MODE15                         GPIO_MODER_MODE15_Msk                             /*!< MODE15[31:30] bits (desc MODE15) */
#define GPIO_MODER_MODE15_0                       (0x1UL << GPIO_MODER_MODE15_Pos)                  /*!< 0x40000000 */
#define GPIO_MODER_MODE15_1                       (0x2UL << GPIO_MODER_MODE15_Pos)                  /*!< 0x80000000 */


/*!< GPIO_OTYPER */
#define GPIO_OTYPER_OT0_Pos                       (0U)
#define GPIO_OTYPER_OT0_Msk                       (0x1UL << GPIO_OTYPER_OT0_Pos)                    /*!< 0x00000001 */
#define GPIO_OTYPER_OT0                           GPIO_OTYPER_OT0_Msk                               /*!< desc OT0 */
#define GPIO_OTYPER_OT1_Pos                       (1U)
#define GPIO_OTYPER_OT1_Msk                       (0x1UL << GPIO_OTYPER_OT1_Pos)                    /*!< 0x00000002 */
#define GPIO_OTYPER_OT1                           GPIO_OTYPER_OT1_Msk                               /*!< desc OT1 */
#define GPIO_OTYPER_OT2_Pos                       (2U)
#define GPIO_OTYPER_OT2_Msk                       (0x1UL << GPIO_OTYPER_OT2_Pos)                    /*!< 0x00000004 */
#define GPIO_OTYPER_OT2                           GPIO_OTYPER_OT2_Msk                               /*!< desc OT2 */
#define GPIO_OTYPER_OT3_Pos                       (3U)
#define GPIO_OTYPER_OT3_Msk                       (0x1UL << GPIO_OTYPER_OT3_Pos)                    /*!< 0x00000008 */
#define GPIO_OTYPER_OT3                           GPIO_OTYPER_OT3_Msk                               /*!< desc OT3 */
#define GPIO_OTYPER_OT4_Pos                       (4U)
#define GPIO_OTYPER_OT4_Msk                       (0x1UL << GPIO_OTYPER_OT4_Pos)                    /*!< 0x00000010 */
#define GPIO_OTYPER_OT4                           GPIO_OTYPER_OT4_Msk                               /*!< desc OT4 */
#define GPIO_OTYPER_OT5_Pos                       (5U)
#define GPIO_OTYPER_OT5_Msk                       (0x1UL << GPIO_OTYPER_OT5_Pos)                    /*!< 0x00000020 */
#define GPIO_OTYPER_OT5                           GPIO_OTYPER_OT5_Msk                               /*!< desc OT5 */
#define GPIO_OTYPER_OT6_Pos                       (6U)
#define GPIO_OTYPER_OT6_Msk                       (0x1UL << GPIO_OTYPER_OT6_Pos)                    /*!< 0x00000040 */
#define GPIO_OTYPER_OT6                           GPIO_OTYPER_OT6_Msk                               /*!< desc OT6 */
#define GPIO_OTYPER_OT7_Pos                       (7U)
#define GPIO_OTYPER_OT7_Msk                       (0x1UL << GPIO_OTYPER_OT7_Pos)                    /*!< 0x00000080 */
#define GPIO_OTYPER_OT7                           GPIO_OTYPER_OT7_Msk                               /*!< desc OT7 */
#define GPIO_OTYPER_OT8_Pos                       (8U)
#define GPIO_OTYPER_OT8_Msk                       (0x1UL << GPIO_OTYPER_OT8_Pos)                    /*!< 0x00000100 */
#define GPIO_OTYPER_OT8                           GPIO_OTYPER_OT8_Msk                               /*!< desc OT8 */
#define GPIO_OTYPER_OT9_Pos                       (9U)
#define GPIO_OTYPER_OT9_Msk                       (0x1UL << GPIO_OTYPER_OT9_Pos)                    /*!< 0x00000200 */
#define GPIO_OTYPER_OT9                           GPIO_OTYPER_OT9_Msk                               /*!< desc OT9 */
#define GPIO_OTYPER_OT10_Pos                      (10U)
#define GPIO_OTYPER_OT10_Msk                      (0x1UL << GPIO_OTYPER_OT10_Pos)                   /*!< 0x00000400 */
#define GPIO_OTYPER_OT10                          GPIO_OTYPER_OT10_Msk                              /*!< desc OT10 */
#define GPIO_OTYPER_OT11_Pos                      (11U)
#define GPIO_OTYPER_OT11_Msk                      (0x1UL << GPIO_OTYPER_OT11_Pos)                   /*!< 0x00000800 */
#define GPIO_OTYPER_OT11                          GPIO_OTYPER_OT11_Msk                              /*!< desc OT11 */
#define GPIO_OTYPER_OT12_Pos                      (12U)
#define GPIO_OTYPER_OT12_Msk                      (0x1UL << GPIO_OTYPER_OT12_Pos)                   /*!< 0x00001000 */
#define GPIO_OTYPER_OT12                          GPIO_OTYPER_OT12_Msk                              /*!< desc OT12 */
#define GPIO_OTYPER_OT13_Pos                      (13U)
#define GPIO_OTYPER_OT13_Msk                      (0x1UL << GPIO_OTYPER_OT13_Pos)                   /*!< 0x00002000 */
#define GPIO_OTYPER_OT13                          GPIO_OTYPER_OT13_Msk                              /*!< desc OT13 */
#define GPIO_OTYPER_OT14_Pos                      (14U)
#define GPIO_OTYPER_OT14_Msk                      (0x1UL << GPIO_OTYPER_OT14_Pos)                   /*!< 0x00004000 */
#define GPIO_OTYPER_OT14                          GPIO_OTYPER_OT14_Msk                              /*!< desc OT14 */
#define GPIO_OTYPER_OT15_Pos                      (15U)
#define GPIO_OTYPER_OT15_Msk                      (0x1UL << GPIO_OTYPER_OT15_Pos)                   /*!< 0x00008000 */
#define GPIO_OTYPER_OT15                          GPIO_OTYPER_OT15_Msk                              /*!< desc OT15 */

/*!< GPIO_OSPEEDR */
#define GPIO_OSPEEDR_OSPEED0_Pos                  (0U)
#define GPIO_OSPEEDR_OSPEED0_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED0_Pos)               /*!< 0x00000003 */
#define GPIO_OSPEEDR_OSPEED0                      GPIO_OSPEEDR_OSPEED0_Msk                          /*!< OSPEED0[1:0] bits (desc OSPEED0) */
#define GPIO_OSPEEDR_OSPEED0_0                    (0x1UL << GPIO_OSPEEDR_OSPEED0_Pos)               /*!< 0x00000001 */
#define GPIO_OSPEEDR_OSPEED0_1                    (0x2UL << GPIO_OSPEEDR_OSPEED0_Pos)               /*!< 0x00000002 */

#define GPIO_OSPEEDR_OSPEED1_Pos                  (2U)
#define GPIO_OSPEEDR_OSPEED1_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED1_Pos)               /*!< 0x0000000C */
#define GPIO_OSPEEDR_OSPEED1                      GPIO_OSPEEDR_OSPEED1_Msk                          /*!< OSPEED1[3:2] bits (desc OSPEED1) */
#define GPIO_OSPEEDR_OSPEED1_0                    (0x1UL << GPIO_OSPEEDR_OSPEED1_Pos)               /*!< 0x00000004 */
#define GPIO_OSPEEDR_OSPEED1_1                    (0x2UL << GPIO_OSPEEDR_OSPEED1_Pos)               /*!< 0x00000008 */

#define GPIO_OSPEEDR_OSPEED2_Pos                  (4U)
#define GPIO_OSPEEDR_OSPEED2_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED2_Pos)               /*!< 0x00000030 */
#define GPIO_OSPEEDR_OSPEED2                      GPIO_OSPEEDR_OSPEED2_Msk                          /*!< OSPEED2[5:4] bits (desc OSPEED2) */
#define GPIO_OSPEEDR_OSPEED2_0                    (0x1UL << GPIO_OSPEEDR_OSPEED2_Pos)               /*!< 0x00000010 */
#define GPIO_OSPEEDR_OSPEED2_1                    (0x2UL << GPIO_OSPEEDR_OSPEED2_Pos)               /*!< 0x00000020 */

#define GPIO_OSPEEDR_OSPEED3_Pos                  (6U)
#define GPIO_OSPEEDR_OSPEED3_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED3_Pos)               /*!< 0x000000C0 */
#define GPIO_OSPEEDR_OSPEED3                      GPIO_OSPEEDR_OSPEED3_Msk                          /*!< OSPEED3[7:6] bits (desc OSPEED3) */
#define GPIO_OSPEEDR_OSPEED3_0                    (0x1UL << GPIO_OSPEEDR_OSPEED3_Pos)               /*!< 0x00000040 */
#define GPIO_OSPEEDR_OSPEED3_1                    (0x2UL << GPIO_OSPEEDR_OSPEED3_Pos)               /*!< 0x00000080 */

#define GPIO_OSPEEDR_OSPEED4_Pos                  (8U)
#define GPIO_OSPEEDR_OSPEED4_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED4_Pos)               /*!< 0x00000300 */
#define GPIO_OSPEEDR_OSPEED4                      GPIO_OSPEEDR_OSPEED4_Msk                          /*!< OSPEED4[9:8] bits (desc OSPEED4) */
#define GPIO_OSPEEDR_OSPEED4_0                    (0x1UL << GPIO_OSPEEDR_OSPEED4_Pos)               /*!< 0x00000100 */
#define GPIO_OSPEEDR_OSPEED4_1                    (0x2UL << GPIO_OSPEEDR_OSPEED4_Pos)               /*!< 0x00000200 */

#define GPIO_OSPEEDR_OSPEED5_Pos                  (10U)
#define GPIO_OSPEEDR_OSPEED5_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED5_Pos)               /*!< 0x00000C00 */
#define GPIO_OSPEEDR_OSPEED5                      GPIO_OSPEEDR_OSPEED5_Msk                          /*!< OSPEED5[11:10] bits (desc OSPEED5) */
#define GPIO_OSPEEDR_OSPEED5_0                    (0x1UL << GPIO_OSPEEDR_OSPEED5_Pos)               /*!< 0x00000400 */
#define GPIO_OSPEEDR_OSPEED5_1                    (0x2UL << GPIO_OSPEEDR_OSPEED5_Pos)               /*!< 0x00000800 */

#define GPIO_OSPEEDR_OSPEED6_Pos                  (12U)
#define GPIO_OSPEEDR_OSPEED6_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED6_Pos)               /*!< 0x00003000 */
#define GPIO_OSPEEDR_OSPEED6                      GPIO_OSPEEDR_OSPEED6_Msk                          /*!< OSPEED6[13:12] bits (desc OSPEED6) */
#define GPIO_OSPEEDR_OSPEED6_0                    (0x1UL << GPIO_OSPEEDR_OSPEED6_Pos)               /*!< 0x00001000 */
#define GPIO_OSPEEDR_OSPEED6_1                    (0x2UL << GPIO_OSPEEDR_OSPEED6_Pos)               /*!< 0x00002000 */

#define GPIO_OSPEEDR_OSPEED7_Pos                  (14U)
#define GPIO_OSPEEDR_OSPEED7_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED7_Pos)               /*!< 0x0000C000 */
#define GPIO_OSPEEDR_OSPEED7                      GPIO_OSPEEDR_OSPEED7_Msk                          /*!< OSPEED7[15:14] bits (desc OSPEED7) */
#define GPIO_OSPEEDR_OSPEED7_0                    (0x1UL << GPIO_OSPEEDR_OSPEED7_Pos)               /*!< 0x00004000 */
#define GPIO_OSPEEDR_OSPEED7_1                    (0x2UL << GPIO_OSPEEDR_OSPEED7_Pos)               /*!< 0x00008000 */

#define GPIO_OSPEEDR_OSPEED8_Pos                  (16U)
#define GPIO_OSPEEDR_OSPEED8_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED8_Pos)               /*!< 0x00030000 */
#define GPIO_OSPEEDR_OSPEED8                      GPIO_OSPEEDR_OSPEED8_Msk                          /*!< OSPEED8[17:16] bits (desc OSPEED8) */
#define GPIO_OSPEEDR_OSPEED8_0                    (0x1UL << GPIO_OSPEEDR_OSPEED8_Pos)               /*!< 0x00010000 */
#define GPIO_OSPEEDR_OSPEED8_1                    (0x2UL << GPIO_OSPEEDR_OSPEED8_Pos)               /*!< 0x00020000 */

#define GPIO_OSPEEDR_OSPEED9_Pos                  (18U)
#define GPIO_OSPEEDR_OSPEED9_Msk                  (0x3UL << GPIO_OSPEEDR_OSPEED9_Pos)               /*!< 0x000C0000 */
#define GPIO_OSPEEDR_OSPEED9                      GPIO_OSPEEDR_OSPEED9_Msk                          /*!< OSPEED9[19:18] bits (desc OSPEED9) */
#define GPIO_OSPEEDR_OSPEED9_0                    (0x1UL << GPIO_OSPEEDR_OSPEED9_Pos)               /*!< 0x00040000 */
#define GPIO_OSPEEDR_OSPEED9_1                    (0x2UL << GPIO_OSPEEDR_OSPEED9_Pos)               /*!< 0x00080000 */

#define GPIO_OSPEEDR_OSPEED10_Pos                 (20U)
#define GPIO_OSPEEDR_OSPEED10_Msk                 (0x3UL << GPIO_OSPEEDR_OSPEED10_Pos)              /*!< 0x00300000 */
#define GPIO_OSPEEDR_OSPEED10                     GPIO_OSPEEDR_OSPEED10_Msk                         /*!< OSPEED10[21:20] bits (desc OSPEED10) */
#define GPIO_OSPEEDR_OSPEED10_0                   (0x1UL << GPIO_OSPEEDR_OSPEED10_Pos)              /*!< 0x00100000 */
#define GPIO_OSPEEDR_OSPEED10_1                   (0x2UL << GPIO_OSPEEDR_OSPEED10_Pos)              /*!< 0x00200000 */

#define GPIO_OSPEEDR_OSPEED11_Pos                 (22U)
#define GPIO_OSPEEDR_OSPEED11_Msk                 (0x3UL << GPIO_OSPEEDR_OSPEED11_Pos)              /*!< 0x00C00000 */
#define GPIO_OSPEEDR_OSPEED11                     GPIO_OSPEEDR_OSPEED11_Msk                         /*!< OSPEED11[23:22] bits (desc OSPEED11) */
#define GPIO_OSPEEDR_OSPEED11_0                   (0x1UL << GPIO_OSPEEDR_OSPEED11_Pos)              /*!< 0x00400000 */
#define GPIO_OSPEEDR_OSPEED11_1                   (0x2UL << GPIO_OSPEEDR_OSPEED11_Pos)              /*!< 0x00800000 */

#define GPIO_OSPEEDR_OSPEED12_Pos                 (24U)
#define GPIO_OSPEEDR_OSPEED12_Msk                 (0x3UL << GPIO_OSPEEDR_OSPEED12_Pos)              /*!< 0x03000000 */
#define GPIO_OSPEEDR_OSPEED12                     GPIO_OSPEEDR_OSPEED12_Msk                         /*!< OSPEED12[25:24] bits (desc OSPEED12) */
#define GPIO_OSPEEDR_OSPEED12_0                   (0x1UL << GPIO_OSPEEDR_OSPEED12_Pos)              /*!< 0x01000000 */
#define GPIO_OSPEEDR_OSPEED12_1                   (0x2UL << GPIO_OSPEEDR_OSPEED12_Pos)              /*!< 0x02000000 */

#define GPIO_OSPEEDR_OSPEED13_Pos                 (26U)
#define GPIO_OSPEEDR_OSPEED13_Msk                 (0x3UL << GPIO_OSPEEDR_OSPEED13_Pos)              /*!< 0x0C000000 */
#define GPIO_OSPEEDR_OSPEED13                     GPIO_OSPEEDR_OSPEED13_Msk                         /*!< OSPEED13[27:26] bits (desc OSPEED13) */
#define GPIO_OSPEEDR_OSPEED13_0                   (0x1UL << GPIO_OSPEEDR_OSPEED13_Pos)              /*!< 0x04000000 */
#define GPIO_OSPEEDR_OSPEED13_1                   (0x2UL << GPIO_OSPEEDR_OSPEED13_Pos)              /*!< 0x08000000 */

#define GPIO_OSPEEDR_OSPEED14_Pos                 (28U)
#define GPIO_OSPEEDR_OSPEED14_Msk                 (0x3UL << GPIO_OSPEEDR_OSPEED14_Pos)              /*!< 0x30000000 */
#define GPIO_OSPEEDR_OSPEED14                     GPIO_OSPEEDR_OSPEED14_Msk                         /*!< OSPEED14[29:28] bits (desc OSPEED14) */
#define GPIO_OSPEEDR_OSPEED14_0                   (0x1UL << GPIO_OSPEEDR_OSPEED14_Pos)              /*!< 0x10000000 */
#define GPIO_OSPEEDR_OSPEED14_1                   (0x2UL << GPIO_OSPEEDR_OSPEED14_Pos)              /*!< 0x20000000 */

#define GPIO_OSPEEDR_OSPEED15_Pos                 (30U)
#define GPIO_OSPEEDR_OSPEED15_Msk                 (0x3UL << GPIO_OSPEEDR_OSPEED15_Pos)              /*!< 0xC0000000 */
#define GPIO_OSPEEDR_OSPEED15                     GPIO_OSPEEDR_OSPEED15_Msk                         /*!< OSPEED15[31:30] bits (desc OSPEED15) */
#define GPIO_OSPEEDR_OSPEED15_0                   (0x1UL << GPIO_OSPEEDR_OSPEED15_Pos)              /*!< 0x40000000 */
#define GPIO_OSPEEDR_OSPEED15_1                   (0x2UL << GPIO_OSPEEDR_OSPEED15_Pos)              /*!< 0x80000000 */


/*!< GPIO_PUPDR */
#define GPIO_PUPDR_PUPD0_Pos                      (0U)
#define GPIO_PUPDR_PUPD0_Msk                      (0x3UL << GPIO_PUPDR_PUPD0_Pos)                   /*!< 0x00000003 */
#define GPIO_PUPDR_PUPD0                          GPIO_PUPDR_PUPD0_Msk                              /*!< PUPD0[1:0] bits (desc PUPD0) */
#define GPIO_PUPDR_PUPD0_0                        (0x1UL << GPIO_PUPDR_PUPD0_Pos)                   /*!< 0x00000001 */
#define GPIO_PUPDR_PUPD0_1                        (0x2UL << GPIO_PUPDR_PUPD0_Pos)                   /*!< 0x00000002 */

#define GPIO_PUPDR_PUPD1_Pos                      (2U)
#define GPIO_PUPDR_PUPD1_Msk                      (0x3UL << GPIO_PUPDR_PUPD1_Pos)                   /*!< 0x0000000C */
#define GPIO_PUPDR_PUPD1                          GPIO_PUPDR_PUPD1_Msk                              /*!< PUPD1[3:2] bits (desc PUPD1) */
#define GPIO_PUPDR_PUPD1_0                        (0x1UL << GPIO_PUPDR_PUPD1_Pos)                   /*!< 0x00000004 */
#define GPIO_PUPDR_PUPD1_1                        (0x2UL << GPIO_PUPDR_PUPD1_Pos)                   /*!< 0x00000008 */

#define GPIO_PUPDR_PUPD2_Pos                      (4U)
#define GPIO_PUPDR_PUPD2_Msk                      (0x3UL << GPIO_PUPDR_PUPD2_Pos)                   /*!< 0x00000030 */
#define GPIO_PUPDR_PUPD2                          GPIO_PUPDR_PUPD2_Msk                              /*!< PUPD2[5:4] bits (desc PUPD2) */
#define GPIO_PUPDR_PUPD2_0                        (0x1UL << GPIO_PUPDR_PUPD2_Pos)                   /*!< 0x00000010 */
#define GPIO_PUPDR_PUPD2_1                        (0x2UL << GPIO_PUPDR_PUPD2_Pos)                   /*!< 0x00000020 */

#define GPIO_PUPDR_PUPD3_Pos                      (6U)
#define GPIO_PUPDR_PUPD3_Msk                      (0x3UL << GPIO_PUPDR_PUPD3_Pos)                   /*!< 0x000000C0 */
#define GPIO_PUPDR_PUPD3                          GPIO_PUPDR_PUPD3_Msk                              /*!< PUPD3[7:6] bits (desc PUPD3) */
#define GPIO_PUPDR_PUPD3_0                        (0x1UL << GPIO_PUPDR_PUPD3_Pos)                   /*!< 0x00000040 */
#define GPIO_PUPDR_PUPD3_1                        (0x2UL << GPIO_PUPDR_PUPD3_Pos)                   /*!< 0x00000080 */

#define GPIO_PUPDR_PUPD4_Pos                      (8U)
#define GPIO_PUPDR_PUPD4_Msk                      (0x3UL << GPIO_PUPDR_PUPD4_Pos)                   /*!< 0x00000300 */
#define GPIO_PUPDR_PUPD4                          GPIO_PUPDR_PUPD4_Msk                              /*!< PUPD4[9:8] bits (desc PUPD4) */
#define GPIO_PUPDR_PUPD4_0                        (0x1UL << GPIO_PUPDR_PUPD4_Pos)                   /*!< 0x00000100 */
#define GPIO_PUPDR_PUPD4_1                        (0x2UL << GPIO_PUPDR_PUPD4_Pos)                   /*!< 0x00000200 */

#define GPIO_PUPDR_PUPD5_Pos                      (10U)
#define GPIO_PUPDR_PUPD5_Msk                      (0x3UL << GPIO_PUPDR_PUPD5_Pos)                   /*!< 0x00000C00 */
#define GPIO_PUPDR_PUPD5                          GPIO_PUPDR_PUPD5_Msk                              /*!< PUPD5[11:10] bits (desc PUPD5) */
#define GPIO_PUPDR_PUPD5_0                        (0x1UL << GPIO_PUPDR_PUPD5_Pos)                   /*!< 0x00000400 */
#define GPIO_PUPDR_PUPD5_1                        (0x2UL << GPIO_PUPDR_PUPD5_Pos)                   /*!< 0x00000800 */

#define GPIO_PUPDR_PUPD6_Pos                      (12U)
#define GPIO_PUPDR_PUPD6_Msk                      (0x3UL << GPIO_PUPDR_PUPD6_Pos)                   /*!< 0x00003000 */
#define GPIO_PUPDR_PUPD6                          GPIO_PUPDR_PUPD6_Msk                              /*!< PUPD6[13:12] bits (desc PUPD6) */
#define GPIO_PUPDR_PUPD6_0                        (0x1UL << GPIO_PUPDR_PUPD6_Pos)                   /*!< 0x00001000 */
#define GPIO_PUPDR_PUPD6_1                        (0x2UL << GPIO_PUPDR_PUPD6_Pos)                   /*!< 0x00002000 */

#define GPIO_PUPDR_PUPD7_Pos                      (14U)
#define GPIO_PUPDR_PUPD7_Msk                      (0x3UL << GPIO_PUPDR_PUPD7_Pos)                   /*!< 0x0000C000 */
#define GPIO_PUPDR_PUPD7                          GPIO_PUPDR_PUPD7_Msk                              /*!< PUPD7[15:14] bits (desc PUPD7) */
#define GPIO_PUPDR_PUPD7_0                        (0x1UL << GPIO_PUPDR_PUPD7_Pos)                   /*!< 0x00004000 */
#define GPIO_PUPDR_PUPD7_1                        (0x2UL << GPIO_PUPDR_PUPD7_Pos)                   /*!< 0x00008000 */

#define GPIO_PUPDR_PUPD8_Pos                      (16U)
#define GPIO_PUPDR_PUPD8_Msk                      (0x3UL << GPIO_PUPDR_PUPD8_Pos)                   /*!< 0x00030000 */
#define GPIO_PUPDR_PUPD8                          GPIO_PUPDR_PUPD8_Msk                              /*!< PUPD8[17:16] bits (desc PUPD8) */
#define GPIO_PUPDR_PUPD8_0                        (0x1UL << GPIO_PUPDR_PUPD8_Pos)                   /*!< 0x00010000 */
#define GPIO_PUPDR_PUPD8_1                        (0x2UL << GPIO_PUPDR_PUPD8_Pos)                   /*!< 0x00020000 */

#define GPIO_PUPDR_PUPD9_Pos                      (18U)
#define GPIO_PUPDR_PUPD9_Msk                      (0x3UL << GPIO_PUPDR_PUPD9_Pos)                   /*!< 0x000C0000 */
#define GPIO_PUPDR_PUPD9                          GPIO_PUPDR_PUPD9_Msk                              /*!< PUPD9[19:18] bits (desc PUPD9) */
#define GPIO_PUPDR_PUPD9_0                        (0x1UL << GPIO_PUPDR_PUPD9_Pos)                   /*!< 0x00040000 */
#define GPIO_PUPDR_PUPD9_1                        (0x2UL << GPIO_PUPDR_PUPD9_Pos)                   /*!< 0x00080000 */

#define GPIO_PUPDR_PUPD10_Pos                     (20U)
#define GPIO_PUPDR_PUPD10_Msk                     (0x3UL << GPIO_PUPDR_PUPD10_Pos)                  /*!< 0x00300000 */
#define GPIO_PUPDR_PUPD10                         GPIO_PUPDR_PUPD10_Msk                             /*!< PUPD10[21:20] bits (desc PUPD10) */
#define GPIO_PUPDR_PUPD10_0                       (0x1UL << GPIO_PUPDR_PUPD10_Pos)                  /*!< 0x00100000 */
#define GPIO_PUPDR_PUPD10_1                       (0x2UL << GPIO_PUPDR_PUPD10_Pos)                  /*!< 0x00200000 */

#define GPIO_PUPDR_PUPD11_Pos                     (22U)
#define GPIO_PUPDR_PUPD11_Msk                     (0x3UL << GPIO_PUPDR_PUPD11_Pos)                  /*!< 0x00C00000 */
#define GPIO_PUPDR_PUPD11                         GPIO_PUPDR_PUPD11_Msk                             /*!< PUPD11[23:22] bits (desc PUPD11) */
#define GPIO_PUPDR_PUPD11_0                       (0x1UL << GPIO_PUPDR_PUPD11_Pos)                  /*!< 0x00400000 */
#define GPIO_PUPDR_PUPD11_1                       (0x2UL << GPIO_PUPDR_PUPD11_Pos)                  /*!< 0x00800000 */

#define GPIO_PUPDR_PUPD12_Pos                     (24U)
#define GPIO_PUPDR_PUPD12_Msk                     (0x3UL << GPIO_PUPDR_PUPD12_Pos)                  /*!< 0x03000000 */
#define GPIO_PUPDR_PUPD12                         GPIO_PUPDR_PUPD12_Msk                             /*!< PUPD12[25:24] bits (desc PUPD12) */
#define GPIO_PUPDR_PUPD12_0                       (0x1UL << GPIO_PUPDR_PUPD12_Pos)                  /*!< 0x01000000 */
#define GPIO_PUPDR_PUPD12_1                       (0x2UL << GPIO_PUPDR_PUPD12_Pos)                  /*!< 0x02000000 */

#define GPIO_PUPDR_PUPD13_Pos                     (26U)
#define GPIO_PUPDR_PUPD13_Msk                     (0x3UL << GPIO_PUPDR_PUPD13_Pos)                  /*!< 0x0C000000 */
#define GPIO_PUPDR_PUPD13                         GPIO_PUPDR_PUPD13_Msk                             /*!< PUPD13[27:26] bits (desc PUPD13) */
#define GPIO_PUPDR_PUPD13_0                       (0x1UL << GPIO_PUPDR_PUPD13_Pos)                  /*!< 0x04000000 */
#define GPIO_PUPDR_PUPD13_1                       (0x2UL << GPIO_PUPDR_PUPD13_Pos)                  /*!< 0x08000000 */

#define GPIO_PUPDR_PUPD14_Pos                     (28U)
#define GPIO_PUPDR_PUPD14_Msk                     (0x3UL << GPIO_PUPDR_PUPD14_Pos)                  /*!< 0x30000000 */
#define GPIO_PUPDR_PUPD14                         GPIO_PUPDR_PUPD14_Msk                             /*!< PUPD14[29:28] bits (desc PUPD14) */
#define GPIO_PUPDR_PUPD14_0                       (0x1UL << GPIO_PUPDR_PUPD14_Pos)                  /*!< 0x10000000 */
#define GPIO_PUPDR_PUPD14_1                       (0x2UL << GPIO_PUPDR_PUPD14_Pos)                  /*!< 0x20000000 */

#define GPIO_PUPDR_PUPD15_Pos                     (30U)
#define GPIO_PUPDR_PUPD15_Msk                     (0x3UL << GPIO_PUPDR_PUPD15_Pos)                  /*!< 0xC0000000 */
#define GPIO_PUPDR_PUPD15                         GPIO_PUPDR_PUPD15_Msk                             /*!< PUPD15[31:30] bits (desc PUPD15) */
#define GPIO_PUPDR_PUPD15_0                       (0x1UL << GPIO_PUPDR_PUPD15_Pos)                  /*!< 0x40000000 */
#define GPIO_PUPDR_PUPD15_1                       (0x2UL << GPIO_PUPDR_PUPD15_Pos)                  /*!< 0x80000000 */


/*!< GPIO_IDR */
#define GPIO_IDR_ID0_Pos                          (0U)
#define GPIO_IDR_ID0_Msk                          (0x1UL << GPIO_IDR_ID0_Pos)                       /*!< 0x00000001 */
#define GPIO_IDR_ID0                              GPIO_IDR_ID0_Msk                                  /*!< desc ID0 */
#define GPIO_IDR_ID1_Pos                          (1U)
#define GPIO_IDR_ID1_Msk                          (0x1UL << GPIO_IDR_ID1_Pos)                       /*!< 0x00000002 */
#define GPIO_IDR_ID1                              GPIO_IDR_ID1_Msk                                  /*!< desc ID1 */
#define GPIO_IDR_ID2_Pos                          (2U)
#define GPIO_IDR_ID2_Msk                          (0x1UL << GPIO_IDR_ID2_Pos)                       /*!< 0x00000004 */
#define GPIO_IDR_ID2                              GPIO_IDR_ID2_Msk                                  /*!< desc ID2 */
#define GPIO_IDR_ID3_Pos                          (3U)
#define GPIO_IDR_ID3_Msk                          (0x1UL << GPIO_IDR_ID3_Pos)                       /*!< 0x00000008 */
#define GPIO_IDR_ID3                              GPIO_IDR_ID3_Msk                                  /*!< desc ID3 */
#define GPIO_IDR_ID4_Pos                          (4U)
#define GPIO_IDR_ID4_Msk                          (0x1UL << GPIO_IDR_ID4_Pos)                       /*!< 0x00000010 */
#define GPIO_IDR_ID4                              GPIO_IDR_ID4_Msk                                  /*!< desc ID4 */
#define GPIO_IDR_ID5_Pos                          (5U)
#define GPIO_IDR_ID5_Msk                          (0x1UL << GPIO_IDR_ID5_Pos)                       /*!< 0x00000020 */
#define GPIO_IDR_ID5                              GPIO_IDR_ID5_Msk                                  /*!< desc ID5 */
#define GPIO_IDR_ID6_Pos                          (6U)
#define GPIO_IDR_ID6_Msk                          (0x1UL << GPIO_IDR_ID6_Pos)                       /*!< 0x00000040 */
#define GPIO_IDR_ID6                              GPIO_IDR_ID6_Msk                                  /*!< desc ID6 */
#define GPIO_IDR_ID7_Pos                          (7U)
#define GPIO_IDR_ID7_Msk                          (0x1UL << GPIO_IDR_ID7_Pos)                       /*!< 0x00000080 */
#define GPIO_IDR_ID7                              GPIO_IDR_ID7_Msk                                  /*!< desc ID7 */
#define GPIO_IDR_ID8_Pos                          (8U)
#define GPIO_IDR_ID8_Msk                          (0x1UL << GPIO_IDR_ID8_Pos)                       /*!< 0x00000100 */
#define GPIO_IDR_ID8                              GPIO_IDR_ID8_Msk                                  /*!< desc ID8 */
#define GPIO_IDR_ID9_Pos                          (9U)
#define GPIO_IDR_ID9_Msk                          (0x1UL << GPIO_IDR_ID9_Pos)                       /*!< 0x00000200 */
#define GPIO_IDR_ID9                              GPIO_IDR_ID9_Msk                                  /*!< desc ID9 */
#define GPIO_IDR_ID10_Pos                         (10U)
#define GPIO_IDR_ID10_Msk                         (0x1UL << GPIO_IDR_ID10_Pos)                      /*!< 0x00000400 */
#define GPIO_IDR_ID10                             GPIO_IDR_ID10_Msk                                 /*!< desc ID10 */
#define GPIO_IDR_ID11_Pos                         (11U)
#define GPIO_IDR_ID11_Msk                         (0x1UL << GPIO_IDR_ID11_Pos)                      /*!< 0x00000800 */
#define GPIO_IDR_ID11                             GPIO_IDR_ID11_Msk                                 /*!< desc ID11 */
#define GPIO_IDR_ID12_Pos                         (12U)
#define GPIO_IDR_ID12_Msk                         (0x1UL << GPIO_IDR_ID12_Pos)                      /*!< 0x00001000 */
#define GPIO_IDR_ID12                             GPIO_IDR_ID12_Msk                                 /*!< desc ID12 */
#define GPIO_IDR_ID13_Pos                         (13U)
#define GPIO_IDR_ID13_Msk                         (0x1UL << GPIO_IDR_ID13_Pos)                      /*!< 0x00002000 */
#define GPIO_IDR_ID13                             GPIO_IDR_ID13_Msk                                 /*!< desc ID13 */
#define GPIO_IDR_ID14_Pos                         (14U)
#define GPIO_IDR_ID14_Msk                         (0x1UL << GPIO_IDR_ID14_Pos)                      /*!< 0x00004000 */
#define GPIO_IDR_ID14                             GPIO_IDR_ID14_Msk                                 /*!< desc ID14 */
#define GPIO_IDR_ID15_Pos                         (15U)
#define GPIO_IDR_ID15_Msk                         (0x1UL << GPIO_IDR_ID15_Pos)                      /*!< 0x00008000 */
#define GPIO_IDR_ID15                             GPIO_IDR_ID15_Msk                                 /*!< desc ID15 */

/*!< GPIO_ODR */
#define GPIO_ODR_OD0_Pos                          (0U)
#define GPIO_ODR_OD0_Msk                          (0x1UL << GPIO_ODR_OD0_Pos)                       /*!< 0x00000001 */
#define GPIO_ODR_OD0                              GPIO_ODR_OD0_Msk                                  /*!< desc OD0 */
#define GPIO_ODR_OD1_Pos                          (1U)
#define GPIO_ODR_OD1_Msk                          (0x1UL << GPIO_ODR_OD1_Pos)                       /*!< 0x00000002 */
#define GPIO_ODR_OD1                              GPIO_ODR_OD1_Msk                                  /*!< desc OD1 */
#define GPIO_ODR_OD2_Pos                          (2U)
#define GPIO_ODR_OD2_Msk                          (0x1UL << GPIO_ODR_OD2_Pos)                       /*!< 0x00000004 */
#define GPIO_ODR_OD2                              GPIO_ODR_OD2_Msk                                  /*!< desc OD2 */
#define GPIO_ODR_OD3_Pos                          (3U)
#define GPIO_ODR_OD3_Msk                          (0x1UL << GPIO_ODR_OD3_Pos)                       /*!< 0x00000008 */
#define GPIO_ODR_OD3                              GPIO_ODR_OD3_Msk                                  /*!< desc OD3 */
#define GPIO_ODR_OD4_Pos                          (4U)
#define GPIO_ODR_OD4_Msk                          (0x1UL << GPIO_ODR_OD4_Pos)                       /*!< 0x00000010 */
#define GPIO_ODR_OD4                              GPIO_ODR_OD4_Msk                                  /*!< desc OD4 */
#define GPIO_ODR_OD5_Pos                          (5U)
#define GPIO_ODR_OD5_Msk                          (0x1UL << GPIO_ODR_OD5_Pos)                       /*!< 0x00000020 */
#define GPIO_ODR_OD5                              GPIO_ODR_OD5_Msk                                  /*!< desc OD5 */
#define GPIO_ODR_OD6_Pos                          (6U)
#define GPIO_ODR_OD6_Msk                          (0x1UL << GPIO_ODR_OD6_Pos)                       /*!< 0x00000040 */
#define GPIO_ODR_OD6                              GPIO_ODR_OD6_Msk                                  /*!< desc OD6 */
#define GPIO_ODR_OD7_Pos                          (7U)
#define GPIO_ODR_OD7_Msk                          (0x1UL << GPIO_ODR_OD7_Pos)                       /*!< 0x00000080 */
#define GPIO_ODR_OD7                              GPIO_ODR_OD7_Msk                                  /*!< desc OD7 */
#define GPIO_ODR_OD8_Pos                          (8U)
#define GPIO_ODR_OD8_Msk                          (0x1UL << GPIO_ODR_OD8_Pos)                       /*!< 0x00000100 */
#define GPIO_ODR_OD8                              GPIO_ODR_OD8_Msk                                  /*!< desc OD8 */
#define GPIO_ODR_OD9_Pos                          (9U)
#define GPIO_ODR_OD9_Msk                          (0x1UL << GPIO_ODR_OD9_Pos)                       /*!< 0x00000200 */
#define GPIO_ODR_OD9                              GPIO_ODR_OD9_Msk                                  /*!< desc OD9 */
#define GPIO_ODR_OD10_Pos                         (10U)
#define GPIO_ODR_OD10_Msk                         (0x1UL << GPIO_ODR_OD10_Pos)                      /*!< 0x00000400 */
#define GPIO_ODR_OD10                             GPIO_ODR_OD10_Msk                                 /*!< desc OD10 */
#define GPIO_ODR_OD11_Pos                         (11U)
#define GPIO_ODR_OD11_Msk                         (0x1UL << GPIO_ODR_OD11_Pos)                      /*!< 0x00000800 */
#define GPIO_ODR_OD11                             GPIO_ODR_OD11_Msk                                 /*!< desc OD11 */
#define GPIO_ODR_OD12_Pos                         (12U)
#define GPIO_ODR_OD12_Msk                         (0x1UL << GPIO_ODR_OD12_Pos)                      /*!< 0x00001000 */
#define GPIO_ODR_OD12                             GPIO_ODR_OD12_Msk                                 /*!< desc OD12 */
#define GPIO_ODR_OD13_Pos                         (13U)
#define GPIO_ODR_OD13_Msk                         (0x1UL << GPIO_ODR_OD13_Pos)                      /*!< 0x00002000 */
#define GPIO_ODR_OD13                             GPIO_ODR_OD13_Msk                                 /*!< desc OD13 */
#define GPIO_ODR_OD14_Pos                         (14U)
#define GPIO_ODR_OD14_Msk                         (0x1UL << GPIO_ODR_OD14_Pos)                      /*!< 0x00004000 */
#define GPIO_ODR_OD14                             GPIO_ODR_OD14_Msk                                 /*!< desc OD14 */
#define GPIO_ODR_OD15_Pos                         (15U)
#define GPIO_ODR_OD15_Msk                         (0x1UL << GPIO_ODR_OD15_Pos)                      /*!< 0x00008000 */
#define GPIO_ODR_OD15                             GPIO_ODR_OD15_Msk                                 /*!< desc OD15 */

/*!< GPIO_BSRR */
#define GPIO_BSRR_BS0_Pos                         (0U)
#define GPIO_BSRR_BS0_Msk                         (0x1UL << GPIO_BSRR_BS0_Pos)                      /*!< 0x00000001 */
#define GPIO_BSRR_BS0                             GPIO_BSRR_BS0_Msk                                 /*!< desc BS0 */
#define GPIO_BSRR_BS1_Pos                         (1U)
#define GPIO_BSRR_BS1_Msk                         (0x1UL << GPIO_BSRR_BS1_Pos)                      /*!< 0x00000002 */
#define GPIO_BSRR_BS1                             GPIO_BSRR_BS1_Msk                                 /*!< desc BS1 */
#define GPIO_BSRR_BS2_Pos                         (2U)
#define GPIO_BSRR_BS2_Msk                         (0x1UL << GPIO_BSRR_BS2_Pos)                      /*!< 0x00000004 */
#define GPIO_BSRR_BS2                             GPIO_BSRR_BS2_Msk                                 /*!< desc BS2 */
#define GPIO_BSRR_BS3_Pos                         (3U)
#define GPIO_BSRR_BS3_Msk                         (0x1UL << GPIO_BSRR_BS3_Pos)                      /*!< 0x00000008 */
#define GPIO_BSRR_BS3                             GPIO_BSRR_BS3_Msk                                 /*!< desc BS3 */
#define GPIO_BSRR_BS4_Pos                         (4U)
#define GPIO_BSRR_BS4_Msk                         (0x1UL << GPIO_BSRR_BS4_Pos)                      /*!< 0x00000010 */
#define GPIO_BSRR_BS4                             GPIO_BSRR_BS4_Msk                                 /*!< desc BS4 */
#define GPIO_BSRR_BS5_Pos                         (5U)
#define GPIO_BSRR_BS5_Msk                         (0x1UL << GPIO_BSRR_BS5_Pos)                      /*!< 0x00000020 */
#define GPIO_BSRR_BS5                             GPIO_BSRR_BS5_Msk                                 /*!< desc BS5 */
#define GPIO_BSRR_BS6_Pos                         (6U)
#define GPIO_BSRR_BS6_Msk                         (0x1UL << GPIO_BSRR_BS6_Pos)                      /*!< 0x00000040 */
#define GPIO_BSRR_BS6                             GPIO_BSRR_BS6_Msk                                 /*!< desc BS6 */
#define GPIO_BSRR_BS7_Pos                         (7U)
#define GPIO_BSRR_BS7_Msk                         (0x1UL << GPIO_BSRR_BS7_Pos)                      /*!< 0x00000080 */
#define GPIO_BSRR_BS7                             GPIO_BSRR_BS7_Msk                                 /*!< desc BS7 */
#define GPIO_BSRR_BS8_Pos                         (8U)
#define GPIO_BSRR_BS8_Msk                         (0x1UL << GPIO_BSRR_BS8_Pos)                      /*!< 0x00000100 */
#define GPIO_BSRR_BS8                             GPIO_BSRR_BS8_Msk                                 /*!< desc BS8 */
#define GPIO_BSRR_BS9_Pos                         (9U)
#define GPIO_BSRR_BS9_Msk                         (0x1UL << GPIO_BSRR_BS9_Pos)                      /*!< 0x00000200 */
#define GPIO_BSRR_BS9                             GPIO_BSRR_BS9_Msk                                 /*!< desc BS9 */
#define GPIO_BSRR_BS10_Pos                        (10U)
#define GPIO_BSRR_BS10_Msk                        (0x1UL << GPIO_BSRR_BS10_Pos)                     /*!< 0x00000400 */
#define GPIO_BSRR_BS10                            GPIO_BSRR_BS10_Msk                                /*!< desc BS10 */
#define GPIO_BSRR_BS11_Pos                        (11U)
#define GPIO_BSRR_BS11_Msk                        (0x1UL << GPIO_BSRR_BS11_Pos)                     /*!< 0x00000800 */
#define GPIO_BSRR_BS11                            GPIO_BSRR_BS11_Msk                                /*!< desc BS11 */
#define GPIO_BSRR_BS12_Pos                        (12U)
#define GPIO_BSRR_BS12_Msk                        (0x1UL << GPIO_BSRR_BS12_Pos)                     /*!< 0x00001000 */
#define GPIO_BSRR_BS12                            GPIO_BSRR_BS12_Msk                                /*!< desc BS12 */
#define GPIO_BSRR_BS13_Pos                        (13U)
#define GPIO_BSRR_BS13_Msk                        (0x1UL << GPIO_BSRR_BS13_Pos)                     /*!< 0x00002000 */
#define GPIO_BSRR_BS13                            GPIO_BSRR_BS13_Msk                                /*!< desc BS13 */
#define GPIO_BSRR_BS14_Pos                        (14U)
#define GPIO_BSRR_BS14_Msk                        (0x1UL << GPIO_BSRR_BS14_Pos)                     /*!< 0x00004000 */
#define GPIO_BSRR_BS14                            GPIO_BSRR_BS14_Msk                                /*!< desc BS14 */
#define GPIO_BSRR_BS15_Pos                        (15U)
#define GPIO_BSRR_BS15_Msk                        (0x1UL << GPIO_BSRR_BS15_Pos)                     /*!< 0x00008000 */
#define GPIO_BSRR_BS15                            GPIO_BSRR_BS15_Msk                                /*!< desc BS15 */
#define GPIO_BSRR_BR0_Pos                         (16U)
#define GPIO_BSRR_BR0_Msk                         (0x1UL << GPIO_BSRR_BR0_Pos)                      /*!< 0x00010000 */
#define GPIO_BSRR_BR0                             GPIO_BSRR_BR0_Msk                                 /*!< desc BR0 */
#define GPIO_BSRR_BR1_Pos                         (17U)
#define GPIO_BSRR_BR1_Msk                         (0x1UL << GPIO_BSRR_BR1_Pos)                      /*!< 0x00020000 */
#define GPIO_BSRR_BR1                             GPIO_BSRR_BR1_Msk                                 /*!< desc BR1 */
#define GPIO_BSRR_BR2_Pos                         (18U)
#define GPIO_BSRR_BR2_Msk                         (0x1UL << GPIO_BSRR_BR2_Pos)                      /*!< 0x00040000 */
#define GPIO_BSRR_BR2                             GPIO_BSRR_BR2_Msk                                 /*!< desc BR2 */
#define GPIO_BSRR_BR3_Pos                         (19U)
#define GPIO_BSRR_BR3_Msk                         (0x1UL << GPIO_BSRR_BR3_Pos)                      /*!< 0x00080000 */
#define GPIO_BSRR_BR3                             GPIO_BSRR_BR3_Msk                                 /*!< desc BR3 */
#define GPIO_BSRR_BR4_Pos                         (20U)
#define GPIO_BSRR_BR4_Msk                         (0x1UL << GPIO_BSRR_BR4_Pos)                      /*!< 0x00100000 */
#define GPIO_BSRR_BR4                             GPIO_BSRR_BR4_Msk                                 /*!< desc BR4 */
#define GPIO_BSRR_BR5_Pos                         (21U)
#define GPIO_BSRR_BR5_Msk                         (0x1UL << GPIO_BSRR_BR5_Pos)                      /*!< 0x00200000 */
#define GPIO_BSRR_BR5                             GPIO_BSRR_BR5_Msk                                 /*!< desc BR5 */
#define GPIO_BSRR_BR6_Pos                         (22U)
#define GPIO_BSRR_BR6_Msk                         (0x1UL << GPIO_BSRR_BR6_Pos)                      /*!< 0x00400000 */
#define GPIO_BSRR_BR6                             GPIO_BSRR_BR6_Msk                                 /*!< desc BR6 */
#define GPIO_BSRR_BR7_Pos                         (23U)
#define GPIO_BSRR_BR7_Msk                         (0x1UL << GPIO_BSRR_BR7_Pos)                      /*!< 0x00800000 */
#define GPIO_BSRR_BR7                             GPIO_BSRR_BR7_Msk                                 /*!< desc BR7 */
#define GPIO_BSRR_BR8_Pos                         (24U)
#define GPIO_BSRR_BR8_Msk                         (0x1UL << GPIO_BSRR_BR8_Pos)                      /*!< 0x01000000 */
#define GPIO_BSRR_BR8                             GPIO_BSRR_BR8_Msk                                 /*!< desc BR8 */
#define GPIO_BSRR_BR9_Pos                         (25U)
#define GPIO_BSRR_BR9_Msk                         (0x1UL << GPIO_BSRR_BR9_Pos)                      /*!< 0x02000000 */
#define GPIO_BSRR_BR9                             GPIO_BSRR_BR9_Msk                                 /*!< desc BR9 */
#define GPIO_BSRR_BR10_Pos                        (26U)
#define GPIO_BSRR_BR10_Msk                        (0x1UL << GPIO_BSRR_BR10_Pos)                     /*!< 0x04000000 */
#define GPIO_BSRR_BR10                            GPIO_BSRR_BR10_Msk                                /*!< desc BR10 */
#define GPIO_BSRR_BR11_Pos                        (27U)
#define GPIO_BSRR_BR11_Msk                        (0x1UL << GPIO_BSRR_BR11_Pos)                     /*!< 0x08000000 */
#define GPIO_BSRR_BR11                            GPIO_BSRR_BR11_Msk                                /*!< desc BR11 */
#define GPIO_BSRR_BR12_Pos                        (28U)
#define GPIO_BSRR_BR12_Msk                        (0x1UL << GPIO_BSRR_BR12_Pos)                     /*!< 0x10000000 */
#define GPIO_BSRR_BR12                            GPIO_BSRR_BR12_Msk                                /*!< desc BR12 */
#define GPIO_BSRR_BR13_Pos                        (29U)
#define GPIO_BSRR_BR13_Msk                        (0x1UL << GPIO_BSRR_BR13_Pos)                     /*!< 0x20000000 */
#define GPIO_BSRR_BR13                            GPIO_BSRR_BR13_Msk                                /*!< desc BR13 */
#define GPIO_BSRR_BR14_Pos                        (30U)
#define GPIO_BSRR_BR14_Msk                        (0x1UL << GPIO_BSRR_BR14_Pos)                     /*!< 0x40000000 */
#define GPIO_BSRR_BR14                            GPIO_BSRR_BR14_Msk                                /*!< desc BR14 */
#define GPIO_BSRR_BR15_Pos                        (31U)
#define GPIO_BSRR_BR15_Msk                        (0x1UL << GPIO_BSRR_BR15_Pos)                     /*!< 0x80000000 */
#define GPIO_BSRR_BR15                            GPIO_BSRR_BR15_Msk                                /*!< desc BR15 */

/*!< GPIO_LCKR */
#define GPIO_LCKR_LCK0_Pos                        (0U)
#define GPIO_LCKR_LCK0_Msk                        (0x1UL << GPIO_LCKR_LCK0_Pos)                     /*!< 0x00000001 */
#define GPIO_LCKR_LCK0                            GPIO_LCKR_LCK0_Msk                                /*!< desc LCK0 */
#define GPIO_LCKR_LCK1_Pos                        (1U)
#define GPIO_LCKR_LCK1_Msk                        (0x1UL << GPIO_LCKR_LCK1_Pos)                     /*!< 0x00000002 */
#define GPIO_LCKR_LCK1                            GPIO_LCKR_LCK1_Msk                                /*!< desc LCK1 */
#define GPIO_LCKR_LCK2_Pos                        (2U)
#define GPIO_LCKR_LCK2_Msk                        (0x1UL << GPIO_LCKR_LCK2_Pos)                     /*!< 0x00000004 */
#define GPIO_LCKR_LCK2                            GPIO_LCKR_LCK2_Msk                                /*!< desc LCK2 */
#define GPIO_LCKR_LCK3_Pos                        (3U)
#define GPIO_LCKR_LCK3_Msk                        (0x1UL << GPIO_LCKR_LCK3_Pos)                     /*!< 0x00000008 */
#define GPIO_LCKR_LCK3                            GPIO_LCKR_LCK3_Msk                                /*!< desc LCK3 */
#define GPIO_LCKR_LCK4_Pos                        (4U)
#define GPIO_LCKR_LCK4_Msk                        (0x1UL << GPIO_LCKR_LCK4_Pos)                     /*!< 0x00000010 */
#define GPIO_LCKR_LCK4                            GPIO_LCKR_LCK4_Msk                                /*!< desc LCK4 */
#define GPIO_LCKR_LCK5_Pos                        (5U)
#define GPIO_LCKR_LCK5_Msk                        (0x1UL << GPIO_LCKR_LCK5_Pos)                     /*!< 0x00000020 */
#define GPIO_LCKR_LCK5                            GPIO_LCKR_LCK5_Msk                                /*!< desc LCK5 */
#define GPIO_LCKR_LCK6_Pos                        (6U)
#define GPIO_LCKR_LCK6_Msk                        (0x1UL << GPIO_LCKR_LCK6_Pos)                     /*!< 0x00000040 */
#define GPIO_LCKR_LCK6                            GPIO_LCKR_LCK6_Msk                                /*!< desc LCK6 */
#define GPIO_LCKR_LCK7_Pos                        (7U)
#define GPIO_LCKR_LCK7_Msk                        (0x1UL << GPIO_LCKR_LCK7_Pos)                     /*!< 0x00000080 */
#define GPIO_LCKR_LCK7                            GPIO_LCKR_LCK7_Msk                                /*!< desc LCK7 */
#define GPIO_LCKR_LCK8_Pos                        (8U)
#define GPIO_LCKR_LCK8_Msk                        (0x1UL << GPIO_LCKR_LCK8_Pos)                     /*!< 0x00000100 */
#define GPIO_LCKR_LCK8                            GPIO_LCKR_LCK8_Msk                                /*!< desc LCK8 */
#define GPIO_LCKR_LCK9_Pos                        (9U)
#define GPIO_LCKR_LCK9_Msk                        (0x1UL << GPIO_LCKR_LCK9_Pos)                     /*!< 0x00000200 */
#define GPIO_LCKR_LCK9                            GPIO_LCKR_LCK9_Msk                                /*!< desc LCK9 */
#define GPIO_LCKR_LCK10_Pos                       (10U)
#define GPIO_LCKR_LCK10_Msk                       (0x1UL << GPIO_LCKR_LCK10_Pos)                    /*!< 0x00000400 */
#define GPIO_LCKR_LCK10                           GPIO_LCKR_LCK10_Msk                               /*!< desc LCK10 */
#define GPIO_LCKR_LCK11_Pos                       (11U)
#define GPIO_LCKR_LCK11_Msk                       (0x1UL << GPIO_LCKR_LCK11_Pos)                    /*!< 0x00000800 */
#define GPIO_LCKR_LCK11                           GPIO_LCKR_LCK11_Msk                               /*!< desc LCK11 */
#define GPIO_LCKR_LCK12_Pos                       (12U)
#define GPIO_LCKR_LCK12_Msk                       (0x1UL << GPIO_LCKR_LCK12_Pos)                    /*!< 0x00001000 */
#define GPIO_LCKR_LCK12                           GPIO_LCKR_LCK12_Msk                               /*!< desc LCK12 */
#define GPIO_LCKR_LCK13_Pos                       (13U)
#define GPIO_LCKR_LCK13_Msk                       (0x1UL << GPIO_LCKR_LCK13_Pos)                    /*!< 0x00002000 */
#define GPIO_LCKR_LCK13                           GPIO_LCKR_LCK13_Msk                               /*!< desc LCK13 */
#define GPIO_LCKR_LCK14_Pos                       (14U)
#define GPIO_LCKR_LCK14_Msk                       (0x1UL << GPIO_LCKR_LCK14_Pos)                    /*!< 0x00004000 */
#define GPIO_LCKR_LCK14                           GPIO_LCKR_LCK14_Msk                               /*!< desc LCK14 */
#define GPIO_LCKR_LCK15_Pos                       (15U)
#define GPIO_LCKR_LCK15_Msk                       (0x1UL << GPIO_LCKR_LCK15_Pos)                    /*!< 0x00008000 */
#define GPIO_LCKR_LCK15                           GPIO_LCKR_LCK15_Msk                               /*!< desc LCK15 */
#define GPIO_LCKR_LCKK_Pos                        (16U)
#define GPIO_LCKR_LCKK_Msk                        (0x1UL << GPIO_LCKR_LCKK_Pos)                     /*!< 0x00010000 */
#define GPIO_LCKR_LCKK                            GPIO_LCKR_LCKK_Msk                                /*!< desc LCKK */

/*!< GPIO_AFRL */
#define GPIO_AFRL_AFSEL0_Pos                      (0U)
#define GPIO_AFRL_AFSEL0_Msk                      (0xFUL << GPIO_AFRL_AFSEL0_Pos)                   /*!< 0x0000000F */
#define GPIO_AFRL_AFSEL0                          GPIO_AFRL_AFSEL0_Msk                              /*!< AFSEL0[3:0] bits (desc AFSEL0) */
#define GPIO_AFRL_AFSEL0_0                        (0x1UL << GPIO_AFRL_AFSEL0_Pos)                   /*!< 0x00000001 */
#define GPIO_AFRL_AFSEL0_1                        (0x2UL << GPIO_AFRL_AFSEL0_Pos)                   /*!< 0x00000002 */
#define GPIO_AFRL_AFSEL0_2                        (0x4UL << GPIO_AFRL_AFSEL0_Pos)                   /*!< 0x00000004 */
#define GPIO_AFRL_AFSEL0_3                        (0x8UL << GPIO_AFRL_AFSEL0_Pos)                   /*!< 0x00000008 */

#define GPIO_AFRL_AFSEL1_Pos                      (4U)
#define GPIO_AFRL_AFSEL1_Msk                      (0xFUL << GPIO_AFRL_AFSEL1_Pos)                   /*!< 0x000000F0 */
#define GPIO_AFRL_AFSEL1                          GPIO_AFRL_AFSEL1_Msk                              /*!< AFSEL1[7:4] bits (desc AFSEL1) */
#define GPIO_AFRL_AFSEL1_0                        (0x1UL << GPIO_AFRL_AFSEL1_Pos)                   /*!< 0x00000010 */
#define GPIO_AFRL_AFSEL1_1                        (0x2UL << GPIO_AFRL_AFSEL1_Pos)                   /*!< 0x00000020 */
#define GPIO_AFRL_AFSEL1_2                        (0x4UL << GPIO_AFRL_AFSEL1_Pos)                   /*!< 0x00000040 */
#define GPIO_AFRL_AFSEL1_3                        (0x8UL << GPIO_AFRL_AFSEL1_Pos)                   /*!< 0x00000080 */

#define GPIO_AFRL_AFSEL2_Pos                      (8U)
#define GPIO_AFRL_AFSEL2_Msk                      (0xFUL << GPIO_AFRL_AFSEL2_Pos)                   /*!< 0x00000F00 */
#define GPIO_AFRL_AFSEL2                          GPIO_AFRL_AFSEL2_Msk                              /*!< AFSEL2[11:8] bits (desc AFSEL2) */
#define GPIO_AFRL_AFSEL2_0                        (0x1UL << GPIO_AFRL_AFSEL2_Pos)                   /*!< 0x00000100 */
#define GPIO_AFRL_AFSEL2_1                        (0x2UL << GPIO_AFRL_AFSEL2_Pos)                   /*!< 0x00000200 */
#define GPIO_AFRL_AFSEL2_2                        (0x4UL << GPIO_AFRL_AFSEL2_Pos)                   /*!< 0x00000400 */
#define GPIO_AFRL_AFSEL2_3                        (0x8UL << GPIO_AFRL_AFSEL2_Pos)                   /*!< 0x00000800 */

#define GPIO_AFRL_AFSEL3_Pos                      (12U)
#define GPIO_AFRL_AFSEL3_Msk                      (0xFUL << GPIO_AFRL_AFSEL3_Pos)                   /*!< 0x0000F000 */
#define GPIO_AFRL_AFSEL3                          GPIO_AFRL_AFSEL3_Msk                              /*!< AFSEL3[15:12] bits (desc AFSEL3) */
#define GPIO_AFRL_AFSEL3_0                        (0x1UL << GPIO_AFRL_AFSEL3_Pos)                   /*!< 0x00001000 */
#define GPIO_AFRL_AFSEL3_1                        (0x2UL << GPIO_AFRL_AFSEL3_Pos)                   /*!< 0x00002000 */
#define GPIO_AFRL_AFSEL3_2                        (0x4UL << GPIO_AFRL_AFSEL3_Pos)                   /*!< 0x00004000 */
#define GPIO_AFRL_AFSEL3_3                        (0x8UL << GPIO_AFRL_AFSEL3_Pos)                   /*!< 0x00008000 */

#define GPIO_AFRL_AFSEL4_Pos                      (16U)
#define GPIO_AFRL_AFSEL4_Msk                      (0xFUL << GPIO_AFRL_AFSEL4_Pos)                   /*!< 0x000F0000 */
#define GPIO_AFRL_AFSEL4                          GPIO_AFRL_AFSEL4_Msk                              /*!< AFSEL4[19:16] bits (desc AFSEL4) */
#define GPIO_AFRL_AFSEL4_0                        (0x1UL << GPIO_AFRL_AFSEL4_Pos)                   /*!< 0x00010000 */
#define GPIO_AFRL_AFSEL4_1                        (0x2UL << GPIO_AFRL_AFSEL4_Pos)                   /*!< 0x00020000 */
#define GPIO_AFRL_AFSEL4_2                        (0x4UL << GPIO_AFRL_AFSEL4_Pos)                   /*!< 0x00040000 */
#define GPIO_AFRL_AFSEL4_3                        (0x8UL << GPIO_AFRL_AFSEL4_Pos)                   /*!< 0x00080000 */

#define GPIO_AFRL_AFSEL5_Pos                      (20U)
#define GPIO_AFRL_AFSEL5_Msk                      (0xFUL << GPIO_AFRL_AFSEL5_Pos)                   /*!< 0x00F00000 */
#define GPIO_AFRL_AFSEL5                          GPIO_AFRL_AFSEL5_Msk                              /*!< AFSEL5[23:20] bits (desc AFSEL5) */
#define GPIO_AFRL_AFSEL5_0                        (0x1UL << GPIO_AFRL_AFSEL5_Pos)                   /*!< 0x00100000 */
#define GPIO_AFRL_AFSEL5_1                        (0x2UL << GPIO_AFRL_AFSEL5_Pos)                   /*!< 0x00200000 */
#define GPIO_AFRL_AFSEL5_2                        (0x4UL << GPIO_AFRL_AFSEL5_Pos)                   /*!< 0x00400000 */
#define GPIO_AFRL_AFSEL5_3                        (0x8UL << GPIO_AFRL_AFSEL5_Pos)                   /*!< 0x00800000 */

#define GPIO_AFRL_AFSEL6_Pos                      (24U)
#define GPIO_AFRL_AFSEL6_Msk                      (0xFUL << GPIO_AFRL_AFSEL6_Pos)                   /*!< 0x0F000000 */
#define GPIO_AFRL_AFSEL6                          GPIO_AFRL_AFSEL6_Msk                              /*!< AFSEL6[27:24] bits (desc AFSEL6) */
#define GPIO_AFRL_AFSEL6_0                        (0x1UL << GPIO_AFRL_AFSEL6_Pos)                   /*!< 0x01000000 */
#define GPIO_AFRL_AFSEL6_1                        (0x2UL << GPIO_AFRL_AFSEL6_Pos)                   /*!< 0x02000000 */
#define GPIO_AFRL_AFSEL6_2                        (0x4UL << GPIO_AFRL_AFSEL6_Pos)                   /*!< 0x04000000 */
#define GPIO_AFRL_AFSEL6_3                        (0x8UL << GPIO_AFRL_AFSEL6_Pos)                   /*!< 0x08000000 */

#define GPIO_AFRL_AFSEL7_Pos                      (28U)
#define GPIO_AFRL_AFSEL7_Msk                      (0xFUL << GPIO_AFRL_AFSEL7_Pos)                   /*!< 0xF0000000 */
#define GPIO_AFRL_AFSEL7                          GPIO_AFRL_AFSEL7_Msk                              /*!< AFSEL7[31:28] bits (desc AFSEL7) */
#define GPIO_AFRL_AFSEL7_0                        (0x1UL << GPIO_AFRL_AFSEL7_Pos)                   /*!< 0x10000000 */
#define GPIO_AFRL_AFSEL7_1                        (0x2UL << GPIO_AFRL_AFSEL7_Pos)                   /*!< 0x20000000 */
#define GPIO_AFRL_AFSEL7_2                        (0x4UL << GPIO_AFRL_AFSEL7_Pos)                   /*!< 0x40000000 */
#define GPIO_AFRL_AFSEL7_3                        (0x8UL << GPIO_AFRL_AFSEL7_Pos)                   /*!< 0x80000000 */


/*!< GPIO_AFRH */
#define GPIO_AFRH_AFSEL8_Pos                      (0U)
#define GPIO_AFRH_AFSEL8_Msk                      (0xFUL << GPIO_AFRH_AFSEL8_Pos)                   /*!< 0x0000000F */
#define GPIO_AFRH_AFSEL8                          GPIO_AFRH_AFSEL8_Msk                              /*!< AFSEL8[3:0] bits (desc AFSEL8) */
#define GPIO_AFRH_AFSEL8_0                        (0x1UL << GPIO_AFRH_AFSEL8_Pos)                   /*!< 0x00000001 */
#define GPIO_AFRH_AFSEL8_1                        (0x2UL << GPIO_AFRH_AFSEL8_Pos)                   /*!< 0x00000002 */
#define GPIO_AFRH_AFSEL8_2                        (0x4UL << GPIO_AFRH_AFSEL8_Pos)                   /*!< 0x00000004 */
#define GPIO_AFRH_AFSEL8_3                        (0x8UL << GPIO_AFRH_AFSEL8_Pos)                   /*!< 0x00000008 */

#define GPIO_AFRH_AFSEL9_Pos                      (4U)
#define GPIO_AFRH_AFSEL9_Msk                      (0xFUL << GPIO_AFRH_AFSEL9_Pos)                   /*!< 0x000000F0 */
#define GPIO_AFRH_AFSEL9                          GPIO_AFRH_AFSEL9_Msk                              /*!< AFSEL9[7:4] bits (desc AFSEL9) */
#define GPIO_AFRH_AFSEL9_0                        (0x1UL << GPIO_AFRH_AFSEL9_Pos)                   /*!< 0x00000010 */
#define GPIO_AFRH_AFSEL9_1                        (0x2UL << GPIO_AFRH_AFSEL9_Pos)                   /*!< 0x00000020 */
#define GPIO_AFRH_AFSEL9_2                        (0x4UL << GPIO_AFRH_AFSEL9_Pos)                   /*!< 0x00000040 */
#define GPIO_AFRH_AFSEL9_3                        (0x8UL << GPIO_AFRH_AFSEL9_Pos)                   /*!< 0x00000080 */

#define GPIO_AFRH_AFSEL10_Pos                     (8U)
#define GPIO_AFRH_AFSEL10_Msk                     (0xFUL << GPIO_AFRH_AFSEL10_Pos)                  /*!< 0x00000F00 */
#define GPIO_AFRH_AFSEL10                         GPIO_AFRH_AFSEL10_Msk                             /*!< AFSEL10[11:8] bits (desc AFSEL10) */
#define GPIO_AFRH_AFSEL10_0                       (0x1UL << GPIO_AFRH_AFSEL10_Pos)                  /*!< 0x00000100 */
#define GPIO_AFRH_AFSEL10_1                       (0x2UL << GPIO_AFRH_AFSEL10_Pos)                  /*!< 0x00000200 */
#define GPIO_AFRH_AFSEL10_2                       (0x4UL << GPIO_AFRH_AFSEL10_Pos)                  /*!< 0x00000400 */
#define GPIO_AFRH_AFSEL10_3                       (0x8UL << GPIO_AFRH_AFSEL10_Pos)                  /*!< 0x00000800 */

#define GPIO_AFRH_AFSEL11_Pos                     (12U)
#define GPIO_AFRH_AFSEL11_Msk                     (0xFUL << GPIO_AFRH_AFSEL11_Pos)                  /*!< 0x0000F000 */
#define GPIO_AFRH_AFSEL11                         GPIO_AFRH_AFSEL11_Msk                             /*!< AFSEL11[15:12] bits (desc AFSEL11) */
#define GPIO_AFRH_AFSEL11_0                       (0x1UL << GPIO_AFRH_AFSEL11_Pos)                  /*!< 0x00001000 */
#define GPIO_AFRH_AFSEL11_1                       (0x2UL << GPIO_AFRH_AFSEL11_Pos)                  /*!< 0x00002000 */
#define GPIO_AFRH_AFSEL11_2                       (0x4UL << GPIO_AFRH_AFSEL11_Pos)                  /*!< 0x00004000 */
#define GPIO_AFRH_AFSEL11_3                       (0x8UL << GPIO_AFRH_AFSEL11_Pos)                  /*!< 0x00008000 */

#define GPIO_AFRH_AFSEL12_Pos                     (16U)
#define GPIO_AFRH_AFSEL12_Msk                     (0xFUL << GPIO_AFRH_AFSEL12_Pos)                  /*!< 0x000F0000 */
#define GPIO_AFRH_AFSEL12                         GPIO_AFRH_AFSEL12_Msk                             /*!< AFSEL12[19:16] bits (desc AFSEL12) */
#define GPIO_AFRH_AFSEL12_0                       (0x1UL << GPIO_AFRH_AFSEL12_Pos)                  /*!< 0x00010000 */
#define GPIO_AFRH_AFSEL12_1                       (0x2UL << GPIO_AFRH_AFSEL12_Pos)                  /*!< 0x00020000 */
#define GPIO_AFRH_AFSEL12_2                       (0x4UL << GPIO_AFRH_AFSEL12_Pos)                  /*!< 0x00040000 */
#define GPIO_AFRH_AFSEL12_3                       (0x8UL << GPIO_AFRH_AFSEL12_Pos)                  /*!< 0x00080000 */

#define GPIO_AFRH_AFSEL13_Pos                     (20U)
#define GPIO_AFRH_AFSEL13_Msk                     (0xFUL << GPIO_AFRH_AFSEL13_Pos)                  /*!< 0x00F00000 */
#define GPIO_AFRH_AFSEL13                         GPIO_AFRH_AFSEL13_Msk                             /*!< AFSEL13[23:20] bits (desc AFSEL13) */
#define GPIO_AFRH_AFSEL13_0                       (0x1UL << GPIO_AFRH_AFSEL13_Pos)                  /*!< 0x00100000 */
#define GPIO_AFRH_AFSEL13_1                       (0x2UL << GPIO_AFRH_AFSEL13_Pos)                  /*!< 0x00200000 */
#define GPIO_AFRH_AFSEL13_2                       (0x4UL << GPIO_AFRH_AFSEL13_Pos)                  /*!< 0x00400000 */
#define GPIO_AFRH_AFSEL13_3                       (0x8UL << GPIO_AFRH_AFSEL13_Pos)                  /*!< 0x00800000 */

#define GPIO_AFRH_AFSEL14_Pos                     (24U)
#define GPIO_AFRH_AFSEL14_Msk                     (0xFUL << GPIO_AFRH_AFSEL14_Pos)                  /*!< 0x0F000000 */
#define GPIO_AFRH_AFSEL14                         GPIO_AFRH_AFSEL14_Msk                             /*!< AFSEL14[27:24] bits (desc AFSEL14) */
#define GPIO_AFRH_AFSEL14_0                       (0x1UL << GPIO_AFRH_AFSEL14_Pos)                  /*!< 0x01000000 */
#define GPIO_AFRH_AFSEL14_1                       (0x2UL << GPIO_AFRH_AFSEL14_Pos)                  /*!< 0x02000000 */
#define GPIO_AFRH_AFSEL14_2                       (0x4UL << GPIO_AFRH_AFSEL14_Pos)                  /*!< 0x04000000 */
#define GPIO_AFRH_AFSEL14_3                       (0x8UL << GPIO_AFRH_AFSEL14_Pos)                  /*!< 0x08000000 */

#define GPIO_AFRH_AFSEL15_Pos                     (28U)
#define GPIO_AFRH_AFSEL15_Msk                     (0xFUL << GPIO_AFRH_AFSEL15_Pos)                  /*!< 0xF0000000 */
#define GPIO_AFRH_AFSEL15                         GPIO_AFRH_AFSEL15_Msk                             /*!< AFSEL15[31:28] bits (desc AFSEL15) */
#define GPIO_AFRH_AFSEL15_0                       (0x1UL << GPIO_AFRH_AFSEL15_Pos)                  /*!< 0x10000000 */
#define GPIO_AFRH_AFSEL15_1                       (0x2UL << GPIO_AFRH_AFSEL15_Pos)                  /*!< 0x20000000 */
#define GPIO_AFRH_AFSEL15_2                       (0x4UL << GPIO_AFRH_AFSEL15_Pos)                  /*!< 0x40000000 */
#define GPIO_AFRH_AFSEL15_3                       (0x8UL << GPIO_AFRH_AFSEL15_Pos)                  /*!< 0x80000000 */


/*!< GPIO_BRR */
#define GPIO_BRR_BR0_Pos                          (0U)
#define GPIO_BRR_BR0_Msk                          (0x1UL << GPIO_BRR_BR0_Pos)                       /*!< 0x00000001 */
#define GPIO_BRR_BR0                              GPIO_BRR_BR0_Msk                                  /*!< desc BR0 */
#define GPIO_BRR_BR1_Pos                          (1U)
#define GPIO_BRR_BR1_Msk                          (0x1UL << GPIO_BRR_BR1_Pos)                       /*!< 0x00000002 */
#define GPIO_BRR_BR1                              GPIO_BRR_BR1_Msk                                  /*!< desc BR1 */
#define GPIO_BRR_BR2_Pos                          (2U)
#define GPIO_BRR_BR2_Msk                          (0x1UL << GPIO_BRR_BR2_Pos)                       /*!< 0x00000004 */
#define GPIO_BRR_BR2                              GPIO_BRR_BR2_Msk                                  /*!< desc BR2 */
#define GPIO_BRR_BR3_Pos                          (3U)
#define GPIO_BRR_BR3_Msk                          (0x1UL << GPIO_BRR_BR3_Pos)                       /*!< 0x00000008 */
#define GPIO_BRR_BR3                              GPIO_BRR_BR3_Msk                                  /*!< desc BR3 */
#define GPIO_BRR_BR4_Pos                          (4U)
#define GPIO_BRR_BR4_Msk                          (0x1UL << GPIO_BRR_BR4_Pos)                       /*!< 0x00000010 */
#define GPIO_BRR_BR4                              GPIO_BRR_BR4_Msk                                  /*!< desc BR4 */
#define GPIO_BRR_BR5_Pos                          (5U)
#define GPIO_BRR_BR5_Msk                          (0x1UL << GPIO_BRR_BR5_Pos)                       /*!< 0x00000020 */
#define GPIO_BRR_BR5                              GPIO_BRR_BR5_Msk                                  /*!< desc BR5 */
#define GPIO_BRR_BR6_Pos                          (6U)
#define GPIO_BRR_BR6_Msk                          (0x1UL << GPIO_BRR_BR6_Pos)                       /*!< 0x00000040 */
#define GPIO_BRR_BR6                              GPIO_BRR_BR6_Msk                                  /*!< desc BR6 */
#define GPIO_BRR_BR7_Pos                          (7U)
#define GPIO_BRR_BR7_Msk                          (0x1UL << GPIO_BRR_BR7_Pos)                       /*!< 0x00000080 */
#define GPIO_BRR_BR7                              GPIO_BRR_BR7_Msk                                  /*!< desc BR7 */
#define GPIO_BRR_BR8_Pos                          (8U)
#define GPIO_BRR_BR8_Msk                          (0x1UL << GPIO_BRR_BR8_Pos)                       /*!< 0x00000100 */
#define GPIO_BRR_BR8                              GPIO_BRR_BR8_Msk                                  /*!< desc BR8 */
#define GPIO_BRR_BR9_Pos                          (9U)
#define GPIO_BRR_BR9_Msk                          (0x1UL << GPIO_BRR_BR9_Pos)                       /*!< 0x00000200 */
#define GPIO_BRR_BR9                              GPIO_BRR_BR9_Msk                                  /*!< desc BR9 */
#define GPIO_BRR_BR10_Pos                         (10U)
#define GPIO_BRR_BR10_Msk                         (0x1UL << GPIO_BRR_BR10_Pos)                      /*!< 0x00000400 */
#define GPIO_BRR_BR10                             GPIO_BRR_BR10_Msk                                 /*!< desc BR10 */
#define GPIO_BRR_BR11_Pos                         (11U)
#define GPIO_BRR_BR11_Msk                         (0x1UL << GPIO_BRR_BR11_Pos)                      /*!< 0x00000800 */
#define GPIO_BRR_BR11                             GPIO_BRR_BR11_Msk                                 /*!< desc BR11 */
#define GPIO_BRR_BR12_Pos                         (12U)
#define GPIO_BRR_BR12_Msk                         (0x1UL << GPIO_BRR_BR12_Pos)                      /*!< 0x00001000 */
#define GPIO_BRR_BR12                             GPIO_BRR_BR12_Msk                                 /*!< desc BR12 */
#define GPIO_BRR_BR13_Pos                         (13U)
#define GPIO_BRR_BR13_Msk                         (0x1UL << GPIO_BRR_BR13_Pos)                      /*!< 0x00002000 */
#define GPIO_BRR_BR13                             GPIO_BRR_BR13_Msk                                 /*!< desc BR13 */
#define GPIO_BRR_BR14_Pos                         (14U)
#define GPIO_BRR_BR14_Msk                         (0x1UL << GPIO_BRR_BR14_Pos)                      /*!< 0x00004000 */
#define GPIO_BRR_BR14                             GPIO_BRR_BR14_Msk                                 /*!< desc BR14 */
#define GPIO_BRR_BR15_Pos                         (15U)
#define GPIO_BRR_BR15_Msk                         (0x1UL << GPIO_BRR_BR15_Pos)                      /*!< 0x00008000 */
#define GPIO_BRR_BR15                             GPIO_BRR_BR15_Msk                                 /*!< desc BR15 */

/********************************************************************************************************************/
/********************************* OPA ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for OPA_CSR register *******************************************/
#define OPA_CSR_OPAEN_Pos                      (0U)
#define OPA_CSR_OPAEN_Msk                      (0x1UL<<OPA_CSR_OPAEN_Pos)                     /*!< 0x00000001 */
#define OPA_CSR_OPAEN                          OPA_CSR_OPAEN_Msk                              /*!< OPA Enable */
#define OPA_CSR_VP_SEL_Pos                     (2U)
#define OPA_CSR_VP_SEL_Msk                     (0x3UL<<OPA_CSR_VP_SEL_Pos)                    /*!< 0x0000000C */
#define OPA_CSR_VP_SEL                         OPA_CSR_VP_SEL_Msk                             /*!< Non reverse input selection */
#define OPA_CSR_VP_SEL_0                       (0x1UL<<OPA_CSR_VP_SEL_Pos)                    /*!< 0x00000004 */
#define OPA_CSR_VP_SEL_1                       (0x2UL<<OPA_CSR_VP_SEL_Pos)                    /*!< 0x00000008 */
#define OPA_CSR_VM_SEL_Pos                     (5U)
#define OPA_CSR_VM_SEL_Msk                     (0x3UL<<OPA_CSR_VM_SEL_Pos)                    /*!< 0x00000060 */
#define OPA_CSR_VM_SEL                         OPA_CSR_VM_SEL_Msk                             /*!< Reverse input selection */
#define OPA_CSR_VM_SEL_0                       (0x1UL<<OPA_CSR_VM_SEL_Pos)                    /*!< 0x00000020 */
#define OPA_CSR_VM_SEL_1                       (0x2UL<<OPA_CSR_VM_SEL_Pos)                    /*!< 0x00000040 */
#define OPA_CSR_OPAHSM_Pos                     (7U)
#define OPA_CSR_OPAHSM_Msk                     (0x1UL<<OPA_CSR_OPAHSM_Pos)                    /*!< 0x00000080 */
#define OPA_CSR_OPAHSM                         OPA_CSR_OPAHSM_Msk                             /*!< OPA high-speed mode */
#define OPA_CSR_OPAINTOEN_Pos                  (8U)
#define OPA_CSR_OPAINTOEN_Msk                  (0x1UL<<OPA_CSR_OPAINTOEN_Pos)                 /*!< 0x00000100 */
#define OPA_CSR_OPAINTOEN                      OPA_CSR_OPAINTOEN_Msk                          /*!< OPA internal output enable */
#define OPA_CSR_OPA_VBIAS_Pos                  (9U)
#define OPA_CSR_OPA_VBIAS_Msk                  (0x3UL<<OPA_CSR_OPA_VBIAS_Pos)                 /*!< 0x00000600 */
#define OPA_CSR_OPA_VBIAS                      OPA_CSR_OPA_VBIAS_Msk                          /*!< OPA Bias voltage selection */
#define OPA_CSR_OPA_VBIAS_0                    (0x1UL<<OPA_CSR_OPA_VBIAS_Pos)                 /*!< 0x00000200 */
#define OPA_CSR_OPA_VBIAS_1                    (0x2UL<<OPA_CSR_OPA_VBIAS_Pos)                 /*!< 0x00000400 */
#define OPA_CSR_PGA_GAIN_Pos                   (14U)
#define OPA_CSR_PGA_GAIN_Msk                   (0x1FUL<<OPA_CSR_PGA_GAIN_Pos)                 /*!< 0x0007C000 */
#define OPA_CSR_PGA_GAIN                       OPA_CSR_PGA_GAIN_Msk                           /*!< OPA configurable amplifier gain value */
#define OPA_CSR_PGA_GAIN_0                     (0x1UL<<OPA_CSR_PGA_GAIN_Pos)                  /*!< 0x00004000 */
#define OPA_CSR_PGA_GAIN_1                     (0x2UL<<OPA_CSR_PGA_GAIN_Pos)                  /*!< 0x00008000 */
#define OPA_CSR_PGA_GAIN_2                     (0x4UL<<OPA_CSR_PGA_GAIN_Pos)                  /*!< 0x00010000 */
#define OPA_CSR_PGA_GAIN_3                     (0x8UL<<OPA_CSR_PGA_GAIN_Pos)                  /*!< 0x00020000 */
#define OPA_CSR_PGA_GAIN_4                     (0x10UL<<OPA_CSR_PGA_GAIN_Pos)                 /*!< 0x00040000 */
#define OPA_CSR_OPA_VBSEL_Pos                  (29U)
#define OPA_CSR_OPA_VBSEL_Msk                  (0x1UL<<OPA_CSR_OPA_VBSEL_Pos)                 /*!< 0x20000000 */
#define OPA_CSR_OPA_VBSEL                      OPA_CSR_OPA_VBSEL_Msk                          /*!< OPA1 Bias selection */
#define OPA_CSR_LOCK_Pos                       (31U)
#define OPA_CSR_LOCK_Msk                       (0x1UL<<OPA_CSR_LOCK_Pos)                      /*!< 0x80000000 */
#define OPA_CSR_LOCK                           OPA_CSR_LOCK_Msk                               /*!< OPA_CSR register Lock */

/********************************* Bit definition for OPA_TCMR register ******************************************/
#define OPA_TCMR_VMS_SEL_Pos                   (0U)
#define OPA_TCMR_VMS_SEL_Msk                   (0x1UL<<OPA_TCMR_VMS_SEL_Pos)                  /*!< 0x00000001 */
#define OPA_TCMR_VMS_SEL                       OPA_TCMR_VMS_SEL_Msk                           /*!< OPAx reverse phase input secondary selection */
#define OPA_TCMR_VPS_SEL_Pos                   (1U)
#define OPA_TCMR_VPS_SEL_Msk                   (0x3UL<<OPA_TCMR_VPS_SEL_Pos)                  /*!< 0x00000006 */
#define OPA_TCMR_VPS_SEL                       OPA_TCMR_VPS_SEL_Msk                           /*!< OPAx non inverting input secondary selection */
#define OPA_TCMR_VPS_SEL_0                     (0x1UL<<OPA_TCMR_VPS_SEL_Pos)                  /*!< 0x00000002 */
#define OPA_TCMR_VPS_SEL_1                     (0x2UL<<OPA_TCMR_VPS_SEL_Pos)                  /*!< 0x00000004 */
#define OPA_TCMR_T1CM_EN_Pos                   (3U)
#define OPA_TCMR_T1CM_EN_Msk                   (0x1UL<<OPA_TCMR_T1CM_EN_Pos)                  /*!< 0x00000008 */
#define OPA_TCMR_T1CM_EN                       OPA_TCMR_T1CM_EN_Msk                           /*!< TIM1 Control Multiplexing Mode Enable */
#define OPA_TCMR_T8CM_EN_Pos                   (4U)
#define OPA_TCMR_T8CM_EN_Msk                   (0x1UL<<OPA_TCMR_T8CM_EN_Pos)                  /*!< 0x00000010 */
#define OPA_TCMR_T8CM_EN                       OPA_TCMR_T8CM_EN_Msk                           /*!< TIM8 Control Multiplexing Mode Enable */
#define OPA_TCMR_LOCK_Pos                      (31U)
#define OPA_TCMR_LOCK_Msk                      (0x1UL<<OPA_TCMR_LOCK_Pos)                     /*!< 0x80000000 */
#define OPA_TCMR_LOCK                          OPA_TCMR_LOCK_Msk                              /*!< OPA_TCMR register Lock */

/********************************************************************************************************************/
/********************************* I2C ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for I2C_CR1 register **********************************************/
#define I2C_CR1_PE_Pos                            (0U)
#define I2C_CR1_PE_Msk                            (0x1UL<<I2C_CR1_PE_Pos)                           /*!< 0x00000001 */
#define I2C_CR1_PE                                I2C_CR1_PE_Msk                                    /*!< Peripheral enable */
#define I2C_CR1_SMBUS_Pos                         (1U)
#define I2C_CR1_SMBUS_Msk                         (0x1UL<<I2C_CR1_SMBUS_Pos)                        /*!< 0x00000002 */
#define I2C_CR1_SMBUS                             I2C_CR1_SMBUS_Msk                                 /*!< SMBus mode */
#define I2C_CR1_WUPEN_Pos                         (2U)
#define I2C_CR1_WUPEN_Msk                         (0x1UL<<I2C_CR1_WUPEN_Pos)                        /*!< 0x00000004 */
#define I2C_CR1_WUPEN                             I2C_CR1_WUPEN_Msk                                 /*!< Wakeup from Stop mode enable */
#define I2C_CR1_SMBTYPE_Pos                       (3U)
#define I2C_CR1_SMBTYPE_Msk                       (0x1UL<<I2C_CR1_SMBTYPE_Pos)                      /*!< 0x00000008 */
#define I2C_CR1_SMBTYPE                           I2C_CR1_SMBTYPE_Msk                               /*!< SMBus type */
#define I2C_CR1_ENARP_Pos                         (4U)
#define I2C_CR1_ENARP_Msk                         (0x1UL<<I2C_CR1_ENARP_Pos)                        /*!< 0x00000010 */
#define I2C_CR1_ENARP                             I2C_CR1_ENARP_Msk                                 /*!< ARP enable */
#define I2C_CR1_ENPEC_Pos                         (5U)
#define I2C_CR1_ENPEC_Msk                         (0x1UL<<I2C_CR1_ENPEC_Pos)                        /*!< 0x00000020 */
#define I2C_CR1_ENPEC                             I2C_CR1_ENPEC_Msk                                 /*!< PEC enable */
#define I2C_CR1_ENGC_Pos                          (6U)
#define I2C_CR1_ENGC_Msk                          (0x1UL<<I2C_CR1_ENGC_Pos)                         /*!< 0x00000040 */
#define I2C_CR1_ENGC                              I2C_CR1_ENGC_Msk                                  /*!< General call enable */
#define I2C_CR1_NOSTRETCH_Pos                     (7U)
#define I2C_CR1_NOSTRETCH_Msk                     (0x1UL<<I2C_CR1_NOSTRETCH_Pos)                    /*!< 0x00000080 */
#define I2C_CR1_NOSTRETCH                         I2C_CR1_NOSTRETCH_Msk                             /*!< Clock stretching disable (Slave mode) */
#define I2C_CR1_START_Pos                         (8U)
#define I2C_CR1_START_Msk                         (0x1UL<<I2C_CR1_START_Pos)                        /*!< 0x00000100 */
#define I2C_CR1_START                             I2C_CR1_START_Msk                                 /*!< Start generation */
#define I2C_CR1_STOP_Pos                          (9U)
#define I2C_CR1_STOP_Msk                          (0x1UL<<I2C_CR1_STOP_Pos)                         /*!< 0x00000200 */
#define I2C_CR1_STOP                              I2C_CR1_STOP_Msk                                  /*!< Stop generation */
#define I2C_CR1_ACK_Pos                           (10U)
#define I2C_CR1_ACK_Msk                           (0x1UL<<I2C_CR1_ACK_Pos)                          /*!< 0x00000400 */
#define I2C_CR1_ACK                               I2C_CR1_ACK_Msk                                   /*!< Acknowledge enable */
#define I2C_CR1_POS_Pos                           (11U)
#define I2C_CR1_POS_Msk                           (0x1UL<<I2C_CR1_POS_Pos)                          /*!< 0x00000800 */
#define I2C_CR1_POS                               I2C_CR1_POS_Msk                                   /*!< Acknowledge(for data reception) */
#define I2C_CR1_PEC_Pos                           (12U)
#define I2C_CR1_PEC_Msk                           (0x1UL<<I2C_CR1_PEC_Pos)                          /*!< 0x00001000 */
#define I2C_CR1_PEC                               I2C_CR1_PEC_Msk                                   /*!< Packet error checking */
#define I2C_CR1_ALERT_Pos                         (13U)
#define I2C_CR1_ALERT_Msk                         (0x1UL<<I2C_CR1_ALERT_Pos)                        /*!< 0x00002000 */
#define I2C_CR1_ALERT                             I2C_CR1_ALERT_Msk                                 /*!< SMBus alert */
#define I2C_CR1_SWRST_Pos                         (15U)
#define I2C_CR1_SWRST_Msk                         (0x1UL<<I2C_CR1_SWRST_Pos)                        /*!< 0x00008000 */
#define I2C_CR1_SWRST                             I2C_CR1_SWRST_Msk                                 /*!< Software reset */
#define I2C_CR1_WKUP_DIV_Pos                      (16U)
#define I2C_CR1_WKUP_DIV_Msk                      (0x3UL<<I2C_CR1_WKUP_DIV_Pos)                     /*!< 0x00030000 */
#define I2C_CR1_WKUP_DIV                          I2C_CR1_WKUP_DIV_Msk                              /*!< PCLK division */
#define I2C_CR1_WKUP_DIV_0                        (0x1UL<<I2C_CR1_WKUP_DIV_Pos)                     /*!< 0x00010000 */
#define I2C_CR1_WKUP_DIV_1                        (0x2UL<<I2C_CR1_WKUP_DIV_Pos)                     /*!< 0x00020000 */
#define I2C_CR1_WKUP_CNT_Pos                      (18U)
#define I2C_CR1_WKUP_CNT_Msk                      (0x3UL<<I2C_CR1_WKUP_CNT_Pos)                     /*!< 0x000C0000 */
#define I2C_CR1_WKUP_CNT                          I2C_CR1_WKUP_CNT_Msk                              /*!< Timeout count number */
#define I2C_CR1_WKUP_CNT_0                        (0x1UL<<I2C_CR1_WKUP_CNT_Pos)                     /*!< 0x00040000 */
#define I2C_CR1_WKUP_CNT_1                        (0x2UL<<I2C_CR1_WKUP_CNT_Pos)                     /*!< 0x00080000 */

/********************************* Bit definition for I2C_CR2 register **********************************************/
#define I2C_CR2_FREQ_Pos                          (0U)
#define I2C_CR2_FREQ_Msk                          (0x7FUL<<I2C_CR2_FREQ_Pos)                        /*!< 0x0000007F */
#define I2C_CR2_FREQ                              I2C_CR2_FREQ_Msk                                  /*!< Peripheral clock frequency */
#define I2C_CR2_ITERREN_Pos                       (8U)
#define I2C_CR2_ITERREN_Msk                       (0x1UL<<I2C_CR2_ITERREN_Pos)                      /*!< 0x00000100 */
#define I2C_CR2_ITERREN                           I2C_CR2_ITERREN_Msk                               /*!< Error interrupt enable */
#define I2C_CR2_ITEVTEN_Pos                       (9U)
#define I2C_CR2_ITEVTEN_Msk                       (0x1UL<<I2C_CR2_ITEVTEN_Pos)                      /*!< 0x00000200 */
#define I2C_CR2_ITEVTEN                           I2C_CR2_ITEVTEN_Msk                               /*!< Event interrupt enable */
#define I2C_CR2_ITBUFEN_Pos                       (10U)
#define I2C_CR2_ITBUFEN_Msk                       (0x1UL<<I2C_CR2_ITBUFEN_Pos)                      /*!< 0x00000400 */
#define I2C_CR2_ITBUFEN                           I2C_CR2_ITBUFEN_Msk                               /*!< Buffer interrupt enable */
#define I2C_CR2_DMAEN_Pos                         (11U)
#define I2C_CR2_DMAEN_Msk                         (0x1UL<<I2C_CR2_DMAEN_Pos)                        /*!< 0x00000800 */
#define I2C_CR2_DMAEN                             I2C_CR2_DMAEN_Msk                                 /*!< DMA requests enable */
#define I2C_CR2_LAST_Pos                          (12U)
#define I2C_CR2_LAST_Msk                          (0x1UL<<I2C_CR2_LAST_Pos)                         /*!< 0x00001000 */
#define I2C_CR2_LAST                              I2C_CR2_LAST_Msk                                  /*!< DMA last transfer */

/********************************* Bit definition for I2C_OAR1 register *********************************************/
/*!< I2C_OAR1 */
#define I2C_OAR1_ADD0_Pos                         (0U)
#define I2C_OAR1_ADD0_Msk                         (0x1UL << I2C_OAR1_ADD0_Pos)                      /*!< 0x00000001 */
#define I2C_OAR1_ADD0                             I2C_OAR1_ADD0_Msk                                 /*!< desc ADD0 */

#define I2C_OAR1_ADD1_Pos                         (1U)
#define I2C_OAR1_ADD1_Msk                         (0x1UL << I2C_OAR1_ADD1_Pos)                      /*!< 0x00000002 */
#define I2C_OAR1_ADD1                             I2C_OAR1_ADD1_Msk                                 /*!< Bit 1 */
#define I2C_OAR1_ADD2_Pos                         (2U)
#define I2C_OAR1_ADD2_Msk                         (0x1UL << I2C_OAR1_ADD2_Pos)                      /*!< 0x00000004 */
#define I2C_OAR1_ADD2                             I2C_OAR1_ADD2_Msk                                 /*!< Bit 2 */
#define I2C_OAR1_ADD3_Pos                         (3U)
#define I2C_OAR1_ADD3_Msk                         (0x1UL << I2C_OAR1_ADD3_Pos)                      /*!< 0x00000008 */
#define I2C_OAR1_ADD3                             I2C_OAR1_ADD3_Msk                                 /*!< Bit 3 */
#define I2C_OAR1_ADD4_Pos                         (4U)
#define I2C_OAR1_ADD4_Msk                         (0x1UL << I2C_OAR1_ADD4_Pos)                      /*!< 0x00000010 */
#define I2C_OAR1_ADD4                             I2C_OAR1_ADD4_Msk                                 /*!< Bit 4 */
#define I2C_OAR1_ADD5_Pos                         (5U)
#define I2C_OAR1_ADD5_Msk                         (0x1UL << I2C_OAR1_ADD5_Pos)                      /*!< 0x00000020 */
#define I2C_OAR1_ADD5                             I2C_OAR1_ADD5_Msk                                 /*!< Bit 5 */
#define I2C_OAR1_ADD6_Pos                         (6U)
#define I2C_OAR1_ADD6_Msk                         (0x1UL << I2C_OAR1_ADD6_Pos)                      /*!< 0x00000040 */
#define I2C_OAR1_ADD6                             I2C_OAR1_ADD6_Msk                                 /*!< Bit 6 */
#define I2C_OAR1_ADD7_Pos                         (7U)
#define I2C_OAR1_ADD7_Msk                         (0x1UL << I2C_OAR1_ADD7_Pos)                      /*!< 0x00000080 */
#define I2C_OAR1_ADD7                             I2C_OAR1_ADD7_Msk                                 /*!< Bit 7 */
#define I2C_OAR1_ADD8_Pos                         (8U)
#define I2C_OAR1_ADD8_Msk                         (0x1UL << I2C_OAR1_ADD8_Pos)                      /*!< 0x00000100 */
#define I2C_OAR1_ADD8                             I2C_OAR1_ADD8_Msk                                 /*!< Bit 8 */
#define I2C_OAR1_ADD9_Pos                         (9U)
#define I2C_OAR1_ADD9_Msk                         (0x1UL << I2C_OAR1_ADD9_Pos)                      /*!< 0x00000200 */
#define I2C_OAR1_ADD9                             I2C_OAR1_ADD9_Msk                                 /*!< Bit 9 */

#define I2C_OAR1_ADD1_7                           0x000000FEU                                       /*!< Interface Address */
#define I2C_OAR1_ADD8_9                           0x00000300U                                       /*!< Interface Address */

#define I2C_OAR1_ADDMODE_Pos                      (15U)
#define I2C_OAR1_ADDMODE_Msk                      (0x1UL<<I2C_OAR1_ADDMODE_Pos)                     /*!< 0x00008000 */
#define I2C_OAR1_ADDMODE                          I2C_OAR1_ADDMODE_Msk                              /*!< Addressing mode (slave mode) */

/********************************* Bit definition for I2C_OAR2 register *********************************************/
#define I2C_OAR2_ENDUAL_Pos                       (0U)
#define I2C_OAR2_ENDUAL_Msk                       (0x1UL<<I2C_OAR2_ENDUAL_Pos)                      /*!< 0x00000001 */
#define I2C_OAR2_ENDUAL                           I2C_OAR2_ENDUAL_Msk                               /*!< Dual addressing mode enable */
#define I2C_OAR2_ADD2_Pos                         (1U)
#define I2C_OAR2_ADD2_Msk                         (0x7FUL<<I2C_OAR2_ADD2_Pos)                       /*!< 0x000000FE */
#define I2C_OAR2_ADD2                             I2C_OAR2_ADD2_Msk                                 /*!< Interface address */
#define I2C_OAR2_OA2MSK_Pos                       (8U)
#define I2C_OAR2_OA2MSK_Msk                       (0x7UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000700 */
#define I2C_OAR2_OA2MSK                           I2C_OAR2_OA2MSK_Msk                               /*!< Own Address 2 masks */
#define I2C_OAR2_OA2MSK_0                         (0x1UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000100 */
#define I2C_OAR2_OA2MSK_1                         (0x2UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000200 */
#define I2C_OAR2_OA2MSK_2                         (0x4UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000400 */

/********************************* Bit definition for I2C_DR register ***********************************************/
#define I2C_DR_DR_Pos                             (0U)
#define I2C_DR_DR_Msk                             (0xFFUL<<I2C_DR_DR_Pos)                           /*!< 0x000000FF */
#define I2C_DR_DR                                 I2C_DR_DR_Msk                                     /*!< 8-bit data register */

/********************************* Bit definition for I2C_SR1 register **********************************************/
#define I2C_SR1_SB_Pos                            (0U)
#define I2C_SR1_SB_Msk                            (0x1UL<<I2C_SR1_SB_Pos)                           /*!< 0x00000001 */
#define I2C_SR1_SB                                I2C_SR1_SB_Msk                                    /*!< Start bit (Master mode) */
#define I2C_SR1_ADDR_Pos                          (1U)
#define I2C_SR1_ADDR_Msk                          (0x1UL<<I2C_SR1_ADDR_Pos)                         /*!< 0x00000002 */
#define I2C_SR1_ADDR                              I2C_SR1_ADDR_Msk                                  /*!< Address sent (master mode)/matched (slave mode) */
#define I2C_SR1_BTF_Pos                           (2U)
#define I2C_SR1_BTF_Msk                           (0x1UL<<I2C_SR1_BTF_Pos)                          /*!< 0x00000004 */
#define I2C_SR1_BTF                               I2C_SR1_BTF_Msk                                   /*!< Byte transfer finished */
#define I2C_SR1_ADD10_Pos                         (3U)
#define I2C_SR1_ADD10_Msk                         (0x1UL<<I2C_SR1_ADD10_Pos)                        /*!< 0x00000008 */
#define I2C_SR1_ADD10                             I2C_SR1_ADD10_Msk                                 /*!< 10-bit header sent (Master mode) */
#define I2C_SR1_STOPF_Pos                         (4U)
#define I2C_SR1_STOPF_Msk                         (0x1UL<<I2C_SR1_STOPF_Pos)                        /*!< 0x00000010 */
#define I2C_SR1_STOPF                             I2C_SR1_STOPF_Msk                                 /*!< Stop detection (slave mode) */
#define I2C_SR1_RXNE_Pos                          (6U)
#define I2C_SR1_RXNE_Msk                          (0x1UL<<I2C_SR1_RXNE_Pos)                         /*!< 0x00000040 */
#define I2C_SR1_RXNE                              I2C_SR1_RXNE_Msk                                  /*!< Data register not empty (receivers) */
#define I2C_SR1_TXE_Pos                           (7U)
#define I2C_SR1_TXE_Msk                           (0x1UL<<I2C_SR1_TXE_Pos)                          /*!< 0x00000080 */
#define I2C_SR1_TXE                               I2C_SR1_TXE_Msk                                   /*!< Data register empty (transmitters) */
#define I2C_SR1_BERR_Pos                          (8U)
#define I2C_SR1_BERR_Msk                          (0x1UL<<I2C_SR1_BERR_Pos)                         /*!< 0x00000100 */
#define I2C_SR1_BERR                              I2C_SR1_BERR_Msk                                  /*!< Bus error */
#define I2C_SR1_ARLO_Pos                          (9U)
#define I2C_SR1_ARLO_Msk                          (0x1UL<<I2C_SR1_ARLO_Pos)                         /*!< 0x00000200 */
#define I2C_SR1_ARLO                              I2C_SR1_ARLO_Msk                                  /*!< Arbitration lost (master mode) */
#define I2C_SR1_AF_Pos                            (10U)
#define I2C_SR1_AF_Msk                            (0x1UL<<I2C_SR1_AF_Pos)                           /*!< 0x00000400 */
#define I2C_SR1_AF                                I2C_SR1_AF_Msk                                    /*!< Acknowledge failure */
#define I2C_SR1_OVR_Pos                           (11U)
#define I2C_SR1_OVR_Msk                           (0x1UL<<I2C_SR1_OVR_Pos)                          /*!< 0x00000800 */
#define I2C_SR1_OVR                               I2C_SR1_OVR_Msk                                   /*!< Overrun/Underrun */
#define I2C_SR1_PECERR_Pos                        (12U)
#define I2C_SR1_PECERR_Msk                        (0x1UL<<I2C_SR1_PECERR_Pos)                       /*!< 0x00001000 */
#define I2C_SR1_PECERR                            I2C_SR1_PECERR_Msk                                /*!< PEC Error in reception */
#define I2C_SR1_TIMEOUT_Pos                       (14U)
#define I2C_SR1_TIMEOUT_Msk                       (0x1UL<<I2C_SR1_TIMEOUT_Pos)                      /*!< 0x00004000 */
#define I2C_SR1_TIMEOUT                           I2C_SR1_TIMEOUT_Msk                               /*!< Timeout or Tlow error */
#define I2C_SR1_SMBALERT_Pos                      (15U)
#define I2C_SR1_SMBALERT_Msk                      (0x1UL<<I2C_SR1_SMBALERT_Pos)                     /*!< 0x00008000 */
#define I2C_SR1_SMBALERT                          I2C_SR1_SMBALERT_Msk                              /*!< SMBus alert */

/********************************* Bit definition for I2C_SR2 register **********************************************/
#define I2C_SR2_MSL_Pos                           (0U)
#define I2C_SR2_MSL_Msk                           (0x1UL<<I2C_SR2_MSL_Pos)                          /*!< 0x00000001 */
#define I2C_SR2_MSL                               I2C_SR2_MSL_Msk                                   /*!< Master/slave */
#define I2C_SR2_BUSY_Pos                          (1U)
#define I2C_SR2_BUSY_Msk                          (0x1UL<<I2C_SR2_BUSY_Pos)                         /*!< 0x00000002 */
#define I2C_SR2_BUSY                              I2C_SR2_BUSY_Msk                                  /*!< Bus busy */
#define I2C_SR2_TRA_Pos                           (2U)
#define I2C_SR2_TRA_Msk                           (0x1UL<<I2C_SR2_TRA_Pos)                          /*!< 0x00000004 */
#define I2C_SR2_TRA                               I2C_SR2_TRA_Msk                                   /*!< Transmitter/receiver */
#define I2C_SR2_GENCALL_Pos                       (4U)
#define I2C_SR2_GENCALL_Msk                       (0x1UL<<I2C_SR2_GENCALL_Pos)                      /*!< 0x00000010 */
#define I2C_SR2_GENCALL                           I2C_SR2_GENCALL_Msk                               /*!< General call address (Slave mode) */
#define I2C_SR2_SMBDEFAULT_Pos                    (5U)
#define I2C_SR2_SMBDEFAULT_Msk                    (0x1UL<<I2C_SR2_SMBDEFAULT_Pos)                   /*!< 0x00000020 */
#define I2C_SR2_SMBDEFAULT                        I2C_SR2_SMBDEFAULT_Msk                            /*!< SMBus device default address (Slave mode) */
#define I2C_SR2_SMBHOST_Pos                       (6U)
#define I2C_SR2_SMBHOST_Msk                       (0x1UL<<I2C_SR2_SMBHOST_Pos)                      /*!< 0x00000040 */
#define I2C_SR2_SMBHOST                           I2C_SR2_SMBHOST_Msk                               /*!< SMBus host header (Slave mode) */
#define I2C_SR2_DUALF_Pos                         (7U)
#define I2C_SR2_DUALF_Msk                         (0x1UL<<I2C_SR2_DUALF_Pos)                        /*!< 0x00000080 */
#define I2C_SR2_DUALF                             I2C_SR2_DUALF_Msk                                 /*!< Dual flag (Slave mode) */
#define I2C_SR2_PEC_Pos                           (8U)
#define I2C_SR2_PEC_Msk                           (0xFFUL<<I2C_SR2_PEC_Pos)                         /*!< 0x0000FF00 */
#define I2C_SR2_PEC                               I2C_SR2_PEC_Msk                                   /*!< Packet error checking register */

/********************************* Bit definition for I2C_CCR register **********************************************/
#define I2C_CCR_CCR_Pos                           (0U)
#define I2C_CCR_CCR_Msk                           (0xFFFUL<<I2C_CCR_CCR_Pos)                        /*!< 0x00000FFF */
#define I2C_CCR_CCR                               I2C_CCR_CCR_Msk                                   /*!< Clock control register in Fm/Sm mode (Master mode) */
#define I2C_CCR_FP_Pos                            (13U)
#define I2C_CCR_FP_Msk                            (0x1UL<<I2C_CCR_FP_Pos)                           /*!< 0x00002000 */
#define I2C_CCR_FP                                I2C_CCR_FP_Msk                                    /*!< I2C master mode selection */
#define I2C_CCR_DUTY_Pos                          (14U)
#define I2C_CCR_DUTY_Msk                          (0x1UL<<I2C_CCR_DUTY_Pos)                         /*!< 0x00004000 */
#define I2C_CCR_DUTY                              I2C_CCR_DUTY_Msk                                  /*!< Fm mode duty cycle */
#define I2C_CCR_FS_Pos                            (15U)
#define I2C_CCR_FS_Msk                            (0x1UL<<I2C_CCR_FS_Pos)                           /*!< 0x00008000 */
#define I2C_CCR_FS                                I2C_CCR_FS_Msk                                    /*!< I2C master mode selection */

/********************************* Bit definition for I2C_TRISE register ********************************************/
#define I2C_TRISE_TRISE_Pos                       (0U)
#define I2C_TRISE_TRISE_Msk                       (0x7FUL<<I2C_TRISE_TRISE_Pos)                     /*!< 0x0000007F */
#define I2C_TRISE_TRISE                           I2C_TRISE_TRISE_Msk                               /*!< Maximum rise time in Fm+/Fm/Sm mode (Master mode) */
#define I2C_TRISE_THOLDDATA_Pos                   (7U)
#define I2C_TRISE_THOLDDATA_Msk                   (0x1FUL<<I2C_TRISE_THOLDDATA_Pos)                 /*!< 0x00000F80 */
#define I2C_TRISE_THOLDDATA                       I2C_TRISE_THOLDDATA_Msk                           /*!< Data hold time */
#define I2C_TRISE_THOLDDATA_0                     (0x1UL<<I2C_TRISE_THOLDDATA_Pos)                  /*!< 0x00000080 */
#define I2C_TRISE_THOLDDATA_1                     (0x2UL<<I2C_TRISE_THOLDDATA_Pos)                  /*!< 0x00000100 */
#define I2C_TRISE_THOLDDATA_2                     (0x4UL<<I2C_TRISE_THOLDDATA_Pos)                  /*!< 0x00000200 */
#define I2C_TRISE_THOLDDATA_3                     (0x8UL<<I2C_TRISE_THOLDDATA_Pos)                  /*!< 0x00000400 */
#define I2C_TRISE_THOLDDATA_4                     (0x10UL<<I2C_TRISE_THOLDDATA_Pos)                 /*!< 0x00000800 */
#define I2C_TRISE_THOLDDATA_SEL_Pos               (12U)
#define I2C_TRISE_THOLDDATA_SEL_Msk               (0x1UL<<I2C_TRISE_THOLDDATA_SEL_Pos)              /*!< 0x00001000 */
#define I2C_TRISE_THOLDDATA_SEL                   I2C_TRISE_THOLDDATA_SEL_Msk                       /*!< Data hold time select */

/********************************* Bit definition for I2C_TIMEOUTR register *****************************************/
#define I2C_TIMEOUTR_TIMEOUTA_Pos                 (0U)
#define I2C_TIMEOUTR_TIMEOUTA_Msk                 (0xFFFUL<<I2C_TIMEOUTR_TIMEOUTA_Pos)              /*!< 0x00000FFF */
#define I2C_TIMEOUTR_TIMEOUTA                     I2C_TIMEOUTR_TIMEOUTA_Msk                         /*!< Bus Timeout A */
#define I2C_TIMEOUTR_TIDLE_Pos                    (12U)
#define I2C_TIMEOUTR_TIDLE_Msk                    (0x1UL<<I2C_TIMEOUTR_TIDLE_Pos)                   /*!< 0x00001000 */
#define I2C_TIMEOUTR_TIDLE                        I2C_TIMEOUTR_TIDLE_Msk                            /*!< Idle clock timeout detection */
#define I2C_TIMEOUTR_TIMEOUTEN_Pos                (15U)
#define I2C_TIMEOUTR_TIMEOUTEN_Msk                (0x1UL<<I2C_TIMEOUTR_TIMEOUTEN_Pos)               /*!< 0x00008000 */
#define I2C_TIMEOUTR_TIMEOUTEN                    I2C_TIMEOUTR_TIMEOUTEN_Msk                        /*!< Clock timeout enable */
#define I2C_TIMEOUTR_TIMEOUTB_Pos                 (16U)
#define I2C_TIMEOUTR_TIMEOUTB_Msk                 (0xFFFUL<<I2C_TIMEOUTR_TIMEOUTB_Pos)              /*!< 0x0FFF0000 */
#define I2C_TIMEOUTR_TIMEOUTB                     I2C_TIMEOUTR_TIMEOUTB_Msk                         /*!< Bus timeout B */
#define I2C_TIMEOUTR_TEXTEN_Pos                   (31U)
#define I2C_TIMEOUTR_TEXTEN_Msk                   (0x1UL<<I2C_TIMEOUTR_TEXTEN_Pos)                  /*!< 0x80000000 */
#define I2C_TIMEOUTR_TEXTEN                       I2C_TIMEOUTR_TEXTEN_Msk                           /*!< Extended clock timeout enable */

/********************************************************************************************************************/
/********************************* IWDG *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for IWDG_KR register **********************************************/
#define IWDG_KR_KEY_Pos                           (0U)
#define IWDG_KR_KEY_Msk                           (0xFFFFUL<<IWDG_KR_KEY_Pos)                       /*!< 0x0000FFFF */
#define IWDG_KR_KEY                               IWDG_KR_KEY_Msk                                   /*!< IWDG KEY */

/********************************* Bit definition for IWDG_PR register **********************************************/
#define IWDG_PR_PR_Pos                            (0U)
#define IWDG_PR_PR_Msk                            (0x7UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000007 */
#define IWDG_PR_PR                                IWDG_PR_PR_Msk                                    /*!< IWDG preload */
#define IWDG_PR_PR_0                              (0x1UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000001 */
#define IWDG_PR_PR_1                              (0x2UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000002 */
#define IWDG_PR_PR_2                              (0x4UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000004 */

/********************************* Bit definition for IWDG_RLR register *********************************************/
#define IWDG_RLR_RL_Pos                           (0U)
#define IWDG_RLR_RL_Msk                           (0xFFFUL<<IWDG_RLR_RL_Pos)                        /*!< 0x00000FFF */
#define IWDG_RLR_RL                               IWDG_RLR_RL_Msk                                   /*!< IWDG reload */

/********************************* Bit definition for IWDG_SR register **********************************************/
#define IWDG_SR_PVU_Pos                           (0U)
#define IWDG_SR_PVU_Msk                           (0x1UL<<IWDG_SR_PVU_Pos)                          /*!< 0x00000001 */
#define IWDG_SR_PVU                               IWDG_SR_PVU_Msk                                   /*!< Watchdog prescaler value update */
#define IWDG_SR_RVU_Pos                           (1U)
#define IWDG_SR_RVU_Msk                           (0x1UL<<IWDG_SR_RVU_Pos)                          /*!< 0x00000002 */
#define IWDG_SR_RVU                               IWDG_SR_RVU_Msk                                   /*!< Watchdog counter reload value update */

/********************************************************************************************************************/
/********************************* LCDC *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for LCDC_CSR register *********************************************/
#define LCDC_CSR_EN0_Pos                          (0U)
#define LCDC_CSR_EN0_Msk                          (0x1UL<<LCDC_CSR_EN0_Pos)                         /*!< 0x00000001 */
#define LCDC_CSR_EN0                              LCDC_CSR_EN0_Msk                                  /*!< LCD0 enable */
#define LCDC_CSR_EN1_Pos                          (1U)
#define LCDC_CSR_EN1_Msk                          (0x1UL<<LCDC_CSR_EN1_Pos)                         /*!< 0x00000002 */
#define LCDC_CSR_EN1                              LCDC_CSR_EN1_Msk                                  /*!< LCD1 enable */
#define LCDC_CSR_EN2_Pos                          (2U)
#define LCDC_CSR_EN2_Msk                          (0x1UL<<LCDC_CSR_EN2_Pos)                         /*!< 0x00000004 */
#define LCDC_CSR_EN2                              LCDC_CSR_EN2_Msk                                  /*!< LCD2 enable */
#define LCDC_CSR_EN3_Pos                          (3U)
#define LCDC_CSR_EN3_Msk                          (0x1UL<<LCDC_CSR_EN3_Pos)                         /*!< 0x00000008 */
#define LCDC_CSR_EN3                              LCDC_CSR_EN3_Msk                                  /*!< LCD3 enable */
#define LCDC_CSR_MODE0_Pos                        (4U)
#define LCDC_CSR_MODE0_Msk                        (0x1UL<<LCDC_CSR_MODE0_Pos)                       /*!< 0x00000010 */
#define LCDC_CSR_MODE0                            LCDC_CSR_MODE0_Msk                                /*!< LCD0 mode */
#define LCDC_CSR_MODE1_Pos                        (5U)
#define LCDC_CSR_MODE1_Msk                        (0x1UL<<LCDC_CSR_MODE1_Pos)                       /*!< 0x00000020 */
#define LCDC_CSR_MODE1                            LCDC_CSR_MODE1_Msk                                /*!< LCD1 mode */
#define LCDC_CSR_MODE2_Pos                        (6U)
#define LCDC_CSR_MODE2_Msk                        (0x1UL<<LCDC_CSR_MODE2_Pos)                       /*!< 0x00000040 */
#define LCDC_CSR_MODE2                            LCDC_CSR_MODE2_Msk                                /*!< LCD2 mode */
#define LCDC_CSR_MODE3_Pos                        (7U)
#define LCDC_CSR_MODE3_Msk                        (0x1UL<<LCDC_CSR_MODE3_Pos)                       /*!< 0x00000080 */
#define LCDC_CSR_MODE3                            LCDC_CSR_MODE3_Msk                                /*!< LCD3 mode */
#define LCDC_CSR_DMAEN_Pos                        (16U)
#define LCDC_CSR_DMAEN_Msk                        (0x1UL<<LCDC_CSR_DMAEN_Pos)                       /*!< 0x00010000 */
#define LCDC_CSR_DMAEN                            LCDC_CSR_DMAEN_Msk                                /*!< LCDC DMA enable */
#define LCDC_CSR_BUSY_Pos                         (17U)
#define LCDC_CSR_BUSY_Msk                         (0x1UL<<LCDC_CSR_BUSY_Pos)                        /*!< 0x00020000 */
#define LCDC_CSR_BUSY                             LCDC_CSR_BUSY_Msk                                 /*!< LCDC busy */

/********************************* Bit definition for LCDC_DR register ***********************************************/
#define LCDC_DR_D_Pos                           (0U)
#define LCDC_DR_D_Msk                           (0x3FFFFUL << LCDC_DR_D_Pos)                        /*!< 0x0003FFFF */
#define LCDC_DR_D                               LCDC_DR_D_Msk    

/********************************* Bit definition for LCDC_CMDR register *********************************************/
#define LCDC_CMDR_CMD_Pos                           (0U)
#define LCDC_CMDR_CMD_Msk                           (0x3FFFFUL << LCDC_CMDR_CMD_Pos)                        
#define LCDC_CMDR_CMD                               LCDC_CMDR_CMD_Msk    

/********************************* Bit definition for LCDC_WCFGR register ********************************************/
#define LCDC_WCFGR_ADDSET_Pos                           (0U)
#define LCDC_WCFGR_ADDSET_Msk                           (0xFUL << LCDC_WCFGR_ADDSET_Pos)                        
#define LCDC_WCFGR_ADDSET                               LCDC_WCFGR_ADDSET_Msk 
#define LCDC_WCFGR_ADDSET_0                       (0x1UL<<LCDC_WCFGR_ADDSET_Pos)                    /*!< 0x00000001 */
#define LCDC_WCFGR_ADDSET_1                       (0x2UL<<LCDC_WCFGR_ADDSET_Pos)                    /*!< 0x00000002 */
#define LCDC_WCFGR_ADDSET_2                       (0x4UL<<LCDC_WCFGR_ADDSET_Pos)                    /*!< 0x00000004 */
#define LCDC_WCFGR_ADDSET_3                       (0x8UL<<LCDC_WCFGR_ADDSET_Pos)                    /*!< 0x00000008 */
#define LCDC_WCFGR_DATAST_Pos                     (8U)
#define LCDC_WCFGR_DATAST_Msk                     (0xFFUL<<LCDC_WCFGR_DATAST_Pos)                   /*!< 0x0000FF00 */
#define LCDC_WCFGR_DATAST                         LCDC_WCFGR_DATAST_Msk                             /*!< LCDC write DATAST */
#define LCDC_WCFGR_DATAHLD_Pos                    (30U)
#define LCDC_WCFGR_DATAHLD_Msk                    (0x3UL<<LCDC_WCFGR_DATAHLD_Pos)                   /*!< 0xC0000000 */
#define LCDC_WCFGR_DATAHLD                        LCDC_WCFGR_DATAHLD_Msk                            /*!< LCDC write DATAHLD */
#define LCDC_WCFGR_DATAHLD_0                      (0x1UL<<LCDC_WCFGR_DATAHLD_Pos)                   /*!< 0x40000000 */
#define LCDC_WCFGR_DATAHLD_1                      (0x2UL<<LCDC_WCFGR_DATAHLD_Pos)                   /*!< 0x80000000 */  

/********************************* Bit definition for LCDC_RCFGR register ********************************************/
#define LCDC_RCFGR_ADDSET_Pos                     (0U)
#define LCDC_RCFGR_ADDSET_Msk                     (0xFUL<<LCDC_RCFGR_ADDSET_Pos)                    /*!< 0x0000000F */
#define LCDC_RCFGR_ADDSET                         LCDC_RCFGR_ADDSET_Msk                             /*!< LCDC read ADDSET */
#define LCDC_RCFGR_ADDSET_0                       (0x1UL<<LCDC_RCFGR_ADDSET_Pos)                    /*!< 0x00000001 */
#define LCDC_RCFGR_ADDSET_1                       (0x2UL<<LCDC_RCFGR_ADDSET_Pos)                    /*!< 0x00000002 */
#define LCDC_RCFGR_ADDSET_2                       (0x4UL<<LCDC_RCFGR_ADDSET_Pos)                    /*!< 0x00000004 */
#define LCDC_RCFGR_ADDSET_3                       (0x8UL<<LCDC_RCFGR_ADDSET_Pos)                    /*!< 0x00000008 */
#define LCDC_RCFGR_DATAST_Pos                     (8U)
#define LCDC_RCFGR_DATAST_Msk                     (0xFFUL<<LCDC_RCFGR_DATAST_Pos)                   /*!< 0x0000FF00 */
#define LCDC_RCFGR_DATAST                         LCDC_RCFGR_DATAST_Msk                             /*!< LCDC read DATAST */
#define LCDC_RCFGR_DATAHLD_Pos                    (30U)
#define LCDC_RCFGR_DATAHLD_Msk                    (0x3UL<<LCDC_RCFGR_DATAHLD_Pos)                   /*!< 0xC0000000 */
#define LCDC_RCFGR_DATAHLD                        LCDC_RCFGR_DATAHLD_Msk                            /*!< LCDC read DATAHLD */
#define LCDC_RCFGR_DATAHLD_0                      (0x1UL<<LCDC_RCFGR_DATAHLD_Pos)                   /*!< 0x40000000 */
#define LCDC_RCFGR_DATAHLD_1                      (0x2UL<<LCDC_RCFGR_DATAHLD_Pos)                   /*!< 0x80000000 */

/********************************************************************************************************************/
/********************************* LPTIM ****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for LPTIM_ISR register ********************************************/
#define LPTIM_ISR_CMPM_Pos                        (0U)
#define LPTIM_ISR_CMPM_Msk                        (0x1UL<<LPTIM_ISR_CMPM_Pos)                       /*!< 0x00000001 */
#define LPTIM_ISR_CMPM                            LPTIM_ISR_CMPM_Msk                                /*!< Compare match */
#define LPTIM_ISR_ARRM_Pos                        (1U)
#define LPTIM_ISR_ARRM_Msk                        (0x1UL<<LPTIM_ISR_ARRM_Pos)                       /*!< 0x00000002 */
#define LPTIM_ISR_ARRM                            LPTIM_ISR_ARRM_Msk                                /*!< Autoreload match */
#define LPTIM_ISR_EXTTRIG_Pos                     (2U)
#define LPTIM_ISR_EXTTRIG_Msk                     (0x1UL<<LPTIM_ISR_EXTTRIG_Pos)                    /*!< 0x00000004 */
#define LPTIM_ISR_EXTTRIG                         LPTIM_ISR_EXTTRIG_Msk                             /*!< External trigger edge event */
#define LPTIM_ISR_CMPOK_Pos                       (3U)
#define LPTIM_ISR_CMPOK_Msk                       (0x1UL<<LPTIM_ISR_CMPOK_Pos)                      /*!< 0x00000008 */
#define LPTIM_ISR_CMPOK                           LPTIM_ISR_CMPOK_Msk                               /*!< Compare register update OK */
#define LPTIM_ISR_ARROK_Pos                       (4U)
#define LPTIM_ISR_ARROK_Msk                       (0x1UL<<LPTIM_ISR_ARROK_Pos)                      /*!< 0x00000010 */
#define LPTIM_ISR_ARROK                           LPTIM_ISR_ARROK_Msk                               /*!< Autoreload register update OK */
#define LPTIM_ISR_UP_Pos                          (5U)
#define LPTIM_ISR_UP_Msk                          (0x1UL<<LPTIM_ISR_UP_Pos)                         /*!< 0x00000020 */
#define LPTIM_ISR_UP                              LPTIM_ISR_UP_Msk                                  /*!< Counter direction change down to up */
#define LPTIM_ISR_DOWN_Pos                        (6U)
#define LPTIM_ISR_DOWN_Msk                        (0x1UL<<LPTIM_ISR_DOWN_Pos)                       /*!< 0x00000040 */
#define LPTIM_ISR_DOWN                            LPTIM_ISR_DOWN_Msk                                /*!< Counter direction change up to down */

/********************************* Bit definition for LPTIM_ICR register ********************************************/
#define LPTIM_ICR_CMPMCF_Pos                      (0U)
#define LPTIM_ICR_CMPMCF_Msk                      (0x1UL<<LPTIM_ICR_CMPMCF_Pos)                     /*!< 0x00000001 */
#define LPTIM_ICR_CMPMCF                          LPTIM_ICR_CMPMCF_Msk                              /*!< Compare match clear flag */
#define LPTIM_ICR_ARRMCF_Pos                      (1U)
#define LPTIM_ICR_ARRMCF_Msk                      (0x1UL<<LPTIM_ICR_ARRMCF_Pos)                     /*!< 0x00000002 */
#define LPTIM_ICR_ARRMCF                          LPTIM_ICR_ARRMCF_Msk                              /*!< Autoreload match clear flag */
#define LPTIM_ICR_EXTTRIGCF_Pos                   (2U)
#define LPTIM_ICR_EXTTRIGCF_Msk                   (0x1UL<<LPTIM_ICR_EXTTRIGCF_Pos)                  /*!< 0x00000004 */
#define LPTIM_ICR_EXTTRIGCF                       LPTIM_ICR_EXTTRIGCF_Msk                           /*!< External trigger valid edge clear flag */
#define LPTIM_ICR_CMPOKCF_Pos                     (3U)
#define LPTIM_ICR_CMPOKCF_Msk                     (0x1UL<<LPTIM_ICR_CMPOKCF_Pos)                    /*!< 0x00000008 */
#define LPTIM_ICR_CMPOKCF                         LPTIM_ICR_CMPOKCF_Msk                             /*!< Compare register update OK clear flag */
#define LPTIM_ICR_ARROKCF_Pos                     (4U)
#define LPTIM_ICR_ARROKCF_Msk                     (0x1UL<<LPTIM_ICR_ARROKCF_Pos)                    /*!< 0x00000010 */
#define LPTIM_ICR_ARROKCF                         LPTIM_ICR_ARROKCF_Msk                             /*!< Autoreload register update OK clear flag */
#define LPTIM_ICR_UPCF_Pos                        (5U)
#define LPTIM_ICR_UPCF_Msk                        (0x1UL<<LPTIM_ICR_UPCF_Pos)                       /*!< 0x00000020 */
#define LPTIM_ICR_UPCF                            LPTIM_ICR_UPCF_Msk                                /*!< Direction change to UP clear flag */
#define LPTIM_ICR_DOWNCF_Pos                      (6U)
#define LPTIM_ICR_DOWNCF_Msk                      (0x1UL<<LPTIM_ICR_DOWNCF_Pos)                     /*!< 0x00000040 */
#define LPTIM_ICR_DOWNCF                          LPTIM_ICR_DOWNCF_Msk                              /*!< Direction change to down clear flag */

/********************************* Bit definition for LPTIM_IER register ********************************************/
#define LPTIM_IER_CMPMIE_Pos                      (0U)
#define LPTIM_IER_CMPMIE_Msk                      (0x1UL<<LPTIM_IER_CMPMIE_Pos)                     /*!< 0x00000001 */
#define LPTIM_IER_CMPMIE                          LPTIM_IER_CMPMIE_Msk                              /*!< Compare match Interrupt Enable */
#define LPTIM_IER_ARRMIE_Pos                      (1U)
#define LPTIM_IER_ARRMIE_Msk                      (0x1UL<<LPTIM_IER_ARRMIE_Pos)                     /*!< 0x00000002 */
#define LPTIM_IER_ARRMIE                          LPTIM_IER_ARRMIE_Msk                              /*!< Autoreload match Interrupt Enable */
#define LPTIM_IER_EXTTRIGIE_Pos                   (2U)
#define LPTIM_IER_EXTTRIGIE_Msk                   (0x1UL<<LPTIM_IER_EXTTRIGIE_Pos)                  /*!< 0x00000004 */
#define LPTIM_IER_EXTTRIGIE                       LPTIM_IER_EXTTRIGIE_Msk                           /*!< External trigger valid edge Interrupt Enable */
#define LPTIM_IER_CMPOKIE_Pos                     (3U)
#define LPTIM_IER_CMPOKIE_Msk                     (0x1UL<<LPTIM_IER_CMPOKIE_Pos)                    /*!< 0x00000008 */
#define LPTIM_IER_CMPOKIE                         LPTIM_IER_CMPOKIE_Msk                             /*!< Compare register update OK Interrupt Enable */
#define LPTIM_IER_ARROKIE_Pos                     (4U)
#define LPTIM_IER_ARROKIE_Msk                     (0x1UL<<LPTIM_IER_ARROKIE_Pos)                    /*!< 0x00000010 */
#define LPTIM_IER_ARROKIE                         LPTIM_IER_ARROKIE_Msk                             /*!< Autoreload register update OK Interrupt Enable */
#define LPTIM_IER_UPIE_Pos                        (5U)
#define LPTIM_IER_UPIE_Msk                        (0x1UL<<LPTIM_IER_UPIE_Pos)                       /*!< 0x00000020 */
#define LPTIM_IER_UPIE                            LPTIM_IER_UPIE_Msk                                /*!< Direction change to UP Interrupt Enable */
#define LPTIM_IER_DOWNIE_Pos                      (6U)
#define LPTIM_IER_DOWNIE_Msk                      (0x1UL<<LPTIM_IER_DOWNIE_Pos)                     /*!< 0x00000040 */
#define LPTIM_IER_DOWNIE                          LPTIM_IER_DOWNIE_Msk                              /*!< Direction change to down Interrupt Enable */

/********************************* Bit definition for LPTIM_CFGR register *******************************************/
#define LPTIM_CFGR_CKSEL_Pos                      (0U)
#define LPTIM_CFGR_CKSEL_Msk                      (0x1UL<<LPTIM_CFGR_CKSEL_Pos)                     /*!< 0x00000001 */
#define LPTIM_CFGR_CKSEL                          LPTIM_CFGR_CKSEL_Msk                              /*!< Clock selector */
#define LPTIM_CFGR_CKPOL_Pos                      (1U)
#define LPTIM_CFGR_CKPOL_Msk                      (0x3UL<<LPTIM_CFGR_CKPOL_Pos)                     /*!< 0x00000006 */
#define LPTIM_CFGR_CKPOL                          LPTIM_CFGR_CKPOL_Msk                              /*!< Clock Polarity */
#define LPTIM_CFGR_CKPOL_0                        (0x1UL<<LPTIM_CFGR_CKPOL_Pos)                     /*!< 0x00000002 */
#define LPTIM_CFGR_CKPOL_1                        (0x2UL<<LPTIM_CFGR_CKPOL_Pos)                     /*!< 0x00000004 */
#define LPTIM_CFGR_CKFLT_Pos                      (3U)
#define LPTIM_CFGR_CKFLT_Msk                      (0x3UL<<LPTIM_CFGR_CKFLT_Pos)                     /*!< 0x00000018 */
#define LPTIM_CFGR_CKFLT                          LPTIM_CFGR_CKFLT_Msk                              /*!< Configurable digital filter for external clock */
#define LPTIM_CFGR_CKFLT_0                        (0x1UL<<LPTIM_CFGR_CKFLT_Pos)                     /*!< 0x00000008 */
#define LPTIM_CFGR_CKFLT_1                        (0x2UL<<LPTIM_CFGR_CKFLT_Pos)                     /*!< 0x00000010 */
#define LPTIM_CFGR_TRGFLT_Pos                     (6U)
#define LPTIM_CFGR_TRGFLT_Msk                     (0x3UL<<LPTIM_CFGR_TRGFLT_Pos)                    /*!< 0x000000C0 */
#define LPTIM_CFGR_TRGFLT                         LPTIM_CFGR_TRGFLT_Msk                             /*!< Configurable digital filter for trigger */
#define LPTIM_CFGR_TRGFLT_0                       (0x1UL<<LPTIM_CFGR_TRGFLT_Pos)                    /*!< 0x00000040 */
#define LPTIM_CFGR_TRGFLT_1                       (0x2UL<<LPTIM_CFGR_TRGFLT_Pos)                    /*!< 0x00000080 */
#define LPTIM_CFGR_PRESC_Pos                      (9U)
#define LPTIM_CFGR_PRESC_Msk                      (0x7UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000E00 */
#define LPTIM_CFGR_PRESC                          LPTIM_CFGR_PRESC_Msk                              /*!< Clock prescaler */
#define LPTIM_CFGR_PRESC_0                        (0x1UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000200 */
#define LPTIM_CFGR_PRESC_1                        (0x2UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000400 */
#define LPTIM_CFGR_PRESC_2                        (0x4UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000800 */
#define LPTIM_CFGR_TRIGSEL_Pos                    (13U)
#define LPTIM_CFGR_TRIGSEL_Msk                    (0x7UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x0000E000 */
#define LPTIM_CFGR_TRIGSEL                        LPTIM_CFGR_TRIGSEL_Msk                            /*!< Trigger selector */
#define LPTIM_CFGR_TRIGSEL_0                      (0x1UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x00002000 */
#define LPTIM_CFGR_TRIGSEL_1                      (0x2UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x00004000 */
#define LPTIM_CFGR_TRIGSEL_2                      (0x4UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x00008000 */
#define LPTIM_CFGR_TRIGEN_Pos                     (17U)
#define LPTIM_CFGR_TRIGEN_Msk                     (0x3UL<<LPTIM_CFGR_TRIGEN_Pos)                    /*!< 0x00060000 */
#define LPTIM_CFGR_TRIGEN                         LPTIM_CFGR_TRIGEN_Msk                             /*!< Trigger enable and polarity */
#define LPTIM_CFGR_TRIGEN_0                       (0x1UL<<LPTIM_CFGR_TRIGEN_Pos)                    /*!< 0x00020000 */
#define LPTIM_CFGR_TRIGEN_1                       (0x2UL<<LPTIM_CFGR_TRIGEN_Pos)                    /*!< 0x00040000 */
//#define LPTIM_CFGR_TIMOUT_Pos                     (19U)
//#define LPTIM_CFGR_TIMOUT_Msk                     (0x1UL<<LPTIM_CFGR_TIMOUT_Pos)                    /*!< 0x00080000 */
//#define LPTIM_CFGR_TIMOUT                         LPTIM_CFGR_TIMOUT_Msk                             /*!<  Timeout enable */
#define LPTIM_CFGR_WAVE_Pos                       (20U)
#define LPTIM_CFGR_WAVE_Msk                       (0x1UL<<LPTIM_CFGR_WAVE_Pos)                      /*!< 0x00100000 */
#define LPTIM_CFGR_WAVE                           LPTIM_CFGR_WAVE_Msk                               /*!< Waveform shape */
#define LPTIM_CFGR_WAVPOL_Pos                     (21U)
#define LPTIM_CFGR_WAVPOL_Msk                     (0x1UL<<LPTIM_CFGR_WAVPOL_Pos)                    /*!< 0x00200000 */
#define LPTIM_CFGR_WAVPOL                         LPTIM_CFGR_WAVPOL_Msk                             /*!< Waveform shape polarity */
#define LPTIM_CFGR_PRELOAD_Pos                    (22U)
#define LPTIM_CFGR_PRELOAD_Msk                    (0x1UL<<LPTIM_CFGR_PRELOAD_Pos)                   /*!< 0x00400000 */
#define LPTIM_CFGR_PRELOAD                        LPTIM_CFGR_PRELOAD_Msk                            /*!< Registers update mode */
//#define LPTIM_CFGR_COUNTMODE_Pos                  (23U)
//#define LPTIM_CFGR_COUNTMODE_Msk                  (0x1UL<<LPTIM_CFGR_COUNTMODE_Pos)                 /*!< 0x00800000 */
//#define LPTIM_CFGR_COUNTMODE                      LPTIM_CFGR_COUNTMODE_Msk                          /*!< counter mode enabled */
#define LPTIM_CFGR_ENC_Pos                        (24U)
#define LPTIM_CFGR_ENC_Msk                        (0x1UL<<LPTIM_CFGR_ENC_Pos)                       /*!< 0x01000000 */
#define LPTIM_CFGR_ENC                            LPTIM_CFGR_ENC_Msk                                /*!< Encoder mode enable */

/********************************* Bit definition for LPTIM_CR register *********************************************/
#define LPTIM_CR_ENABLE_Pos                       (0U)
#define LPTIM_CR_ENABLE_Msk                       (0x1UL<<LPTIM_CR_ENABLE_Pos)                      /*!< 0x00000001 */
#define LPTIM_CR_ENABLE                           LPTIM_CR_ENABLE_Msk                               /*!< LPTIM enable */
#define LPTIM_CR_SNGSTRT_Pos                      (1U)
#define LPTIM_CR_SNGSTRT_Msk                      (0x1UL<<LPTIM_CR_SNGSTRT_Pos)                     /*!< 0x00000002 */
#define LPTIM_CR_SNGSTRT                          LPTIM_CR_SNGSTRT_Msk                              /*!< LPTIM start in Single mode */
#define LPTIM_CR_CNTSTRT_Pos                      (2U)
#define LPTIM_CR_CNTSTRT_Msk                      (0x1UL<<LPTIM_CR_CNTSTRT_Pos)                     /*!< 0x00000004 */
#define LPTIM_CR_CNTSTRT                          LPTIM_CR_CNTSTRT_Msk                              /*!< Timer start in Continuous mode */
//#define LPTIM_CR_COUNTRST_Pos                     (3U)
//#define LPTIM_CR_COUNTRST_Msk                     (0x1UL<<LPTIM_CR_COUNTRST_Pos)                    /*!< 0x00000008 */
//#define LPTIM_CR_COUNTRST                         LPTIM_CR_COUNTRST_Msk                             /*!< Counter reset */
//#define LPTIM_CR_RSTARE_Pos                       (4U)
//#define LPTIM_CR_RSTARE_Msk                       (0x1UL<<LPTIM_CR_RSTARE_Pos)                      /*!< 0x00000010 */
//#define LPTIM_CR_RSTARE                           LPTIM_CR_RSTARE_Msk                               /*!< Reset after read enable */

/********************************* Bit definition for LPTIM_CMP register ********************************************/
#define LPTIM_CMP_CMP_Pos                         (0U)
#define LPTIM_CMP_CMP_Msk                         (0xFFFFUL<<LPTIM_CMP_CMP_Pos)                     /*!< 0x0000FFFF */
#define LPTIM_CMP_CMP                             LPTIM_CMP_CMP_Msk                                 /*!< Compare value */

/********************************* Bit definition for LPTIM_ARR register ********************************************/
#define LPTIM_ARR_ARR_Pos                         (0U)
#define LPTIM_ARR_ARR_Msk                         (0xFFFFUL<<LPTIM_ARR_ARR_Pos)                     /*!< 0x0000FFFF */
#define LPTIM_ARR_ARR                             LPTIM_ARR_ARR_Msk                                 /*!< Auto reload value */

/********************************* Bit definition for LPTIM_CNT register ********************************************/
#define LPTIM_CNT_CNT_Pos                         (0U)
#define LPTIM_CNT_CNT_Msk                         (0xFFFFUL<<LPTIM_CNT_CNT_Pos)                     /*!< 0x0000FFFF */
#define LPTIM_CNT_CNT                             LPTIM_CNT_CNT_Msk                                 /*!< Counter value */

/********************************* Bit definition for LPTIM_OR register *********************************************/
#define LPTIM_OR_IN1_Pos                          (0U)
#define LPTIM_OR_IN1_Msk                          (0xDUL<<LPTIM_OR_IN1_Pos)                         /*!< 0x0000000D */
#define LPTIM_OR_IN1                              LPTIM_OR_IN1_Msk                                  /*!< LPTimer input 1 remap extension */
#define LPTIM_OR_IN1_0                            (0x1UL<<LPTIM_OR_IN1_Pos)                         /*!< 0x00000001 */
#define LPTIM_OR_IN1_1                            (0x4UL<<LPTIM_OR_IN1_Pos)                         /*!< 0x00000004 */
#define LPTIM_OR_IN1_2                            (0x8UL<<LPTIM_OR_IN1_Pos)                         /*!< 0x00000008 */
#define LPTIM_OR_IN2_Pos                          (1U)
#define LPTIM_OR_IN2_Msk                          (0x19UL<<LPTIM_OR_IN2_Pos)                        /*!< 0x00000032 */
#define LPTIM_OR_IN2                              LPTIM_OR_IN2_Msk                                  /*!< LPTimer input 2 remap extension */
#define LPTIM_OR_IN2_0                            (0x1UL<<LPTIM_OR_IN2_Pos)                         /*!< 0x00000002 */
#define LPTIM_OR_IN2_1                            (0x8UL<<LPTIM_OR_IN2_Pos)                         /*!< 0x00000010 */
#define LPTIM_OR_IN2_2                            (0x10UL<<LPTIM_OR_IN2_Pos)                        /*!< 0x00000020 */

/********************************************************************************************************************/
/********************************* LPUART ***************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for LPUART_CR1 register *******************************************/
#define LPUART_CR1_UE_Pos                         (0U)
#define LPUART_CR1_UE_Msk                         (0x1UL<<LPUART_CR1_UE_Pos)                        /*!< 0x00000001 */
#define LPUART_CR1_UE                             LPUART_CR1_UE_Msk                                 /*!< LPUART enabled */
#define LPUART_CR1_UESM_Pos                       (1U)
#define LPUART_CR1_UESM_Msk                       (0x1UL<<LPUART_CR1_UESM_Pos)                      /*!< 0x00000002 */
#define LPUART_CR1_UESM                           LPUART_CR1_UESM_Msk                               /*!< LPUART enabled in Stop mode */
#define LPUART_CR1_RE_Pos                         (2U)
#define LPUART_CR1_RE_Msk                         (0x1UL<<LPUART_CR1_RE_Pos)                        /*!< 0x00000004 */
#define LPUART_CR1_RE                             LPUART_CR1_RE_Msk                                 /*!< Receive enable */
#define LPUART_CR1_TE_Pos                         (3U)
#define LPUART_CR1_TE_Msk                         (0x1UL<<LPUART_CR1_TE_Pos)                        /*!< 0x00000008 */
#define LPUART_CR1_TE                             LPUART_CR1_TE_Msk                                 /*!< Transfer enable */
#define LPUART_CR1_IDLEIE_Pos                     (4U)
#define LPUART_CR1_IDLEIE_Msk                     (0x1UL<<LPUART_CR1_IDLEIE_Pos)                    /*!< 0x00000010 */
#define LPUART_CR1_IDLEIE                         LPUART_CR1_IDLEIE_Msk                             /*!< IDLE interrupt enable */
#define LPUART_CR1_RXNEIE_Pos                     (5U)
#define LPUART_CR1_RXNEIE_Msk                     (0x1UL<<LPUART_CR1_RXNEIE_Pos)                    /*!< 0x00000020 */
#define LPUART_CR1_RXNEIE                         LPUART_CR1_RXNEIE_Msk                             /*!< RX non-null interrupt enable */
#define LPUART_CR1_TCIE_Pos                       (6U)
#define LPUART_CR1_TCIE_Msk                       (0x1UL<<LPUART_CR1_TCIE_Pos)                      /*!< 0x00000040 */
#define LPUART_CR1_TCIE                           LPUART_CR1_TCIE_Msk                               /*!< End of Transfer interrupt enabled */
#define LPUART_CR1_TXEIE_Pos                      (7U)
#define LPUART_CR1_TXEIE_Msk                      (0x1UL<<LPUART_CR1_TXEIE_Pos)                     /*!< 0x00000080 */
#define LPUART_CR1_TXEIE                          LPUART_CR1_TXEIE_Msk                              /*!< TX is not full interrupt enabled */
#define LPUART_CR1_PEIE_Pos                       (8U)
#define LPUART_CR1_PEIE_Msk                       (0x1UL<<LPUART_CR1_PEIE_Pos)                      /*!< 0x00000100 */
#define LPUART_CR1_PEIE                           LPUART_CR1_PEIE_Msk                               /*!< PE interrupt enable */
#define LPUART_CR1_PS_Pos                         (9U)
#define LPUART_CR1_PS_Msk                         (0x1UL<<LPUART_CR1_PS_Pos)                        /*!< 0x00000200 */
#define LPUART_CR1_PS                             LPUART_CR1_PS_Msk                                 /*!< Parity selection */
#define LPUART_CR1_PCE_Pos                        (10U)
#define LPUART_CR1_PCE_Msk                        (0x1UL<<LPUART_CR1_PCE_Pos)                       /*!< 0x00000400 */
#define LPUART_CR1_PCE                            LPUART_CR1_PCE_Msk                                /*!< Parity control */
#define LPUART_CR1_WAKE_Pos                       (11U)
#define LPUART_CR1_WAKE_Msk                       (0x1UL<<LPUART_CR1_WAKE_Pos)                      /*!< 0x00000800 */
#define LPUART_CR1_WAKE                           LPUART_CR1_WAKE_Msk                               /*!< Receive wake-up mode */
#define LPUART_CR1_M0_Pos                         (12U)
#define LPUART_CR1_M0_Msk                         (0x1UL<<LPUART_CR1_M0_Pos)                        /*!< 0x00001000 */
#define LPUART_CR1_M0                             LPUART_CR1_M0_Msk                                 /*!< Configure the word length in combination with M1 */
#define LPUART_CR1_MME_Pos                        (13U)
#define LPUART_CR1_MME_Msk                        (0x1UL<<LPUART_CR1_MME_Pos)                       /*!< 0x00002000 */
#define LPUART_CR1_MME                            LPUART_CR1_MME_Msk                                /*!< Silent mode enabled */
#define LPUART_CR1_CMIE_Pos                       (14U)
#define LPUART_CR1_CMIE_Msk                       (0x1UL<<LPUART_CR1_CMIE_Pos)                      /*!< 0x00004000 */
#define LPUART_CR1_CMIE                           LPUART_CR1_CMIE_Msk                               /*!< Character match interrupt enabled */
#define LPUART_CR1_DEDT_Pos                       (16U)
#define LPUART_CR1_DEDT_Msk                       (0x1FUL<<LPUART_CR1_DEDT_Pos)                     /*!< 0x001F0000 */
#define LPUART_CR1_DEDT                           LPUART_CR1_DEDT_Msk                               /*!< The time between the transmit frame stop bit and the invalid DE (driver enable) signal */
#define LPUART_CR1_DEDT_0                         (0x1UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00010000 */
#define LPUART_CR1_DEDT_1                         (0x2UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00020000 */
#define LPUART_CR1_DEDT_2                         (0x4UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00040000 */
#define LPUART_CR1_DEDT_3                         (0x8UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00080000 */
#define LPUART_CR1_DEDT_4                         (0x10UL<<LPUART_CR1_DEDT_Pos)                     /*!< 0x00100000 */
#define LPUART_CR1_DEAT_Pos                       (21U)
#define LPUART_CR1_DEAT_Msk                       (0x1FUL<<LPUART_CR1_DEAT_Pos)                     /*!< 0x03E00000 */
#define LPUART_CR1_DEAT                           LPUART_CR1_DEAT_Msk                               /*!< The time between the DE (driver enable) signal validity and the start bit */
#define LPUART_CR1_DEAT_0                         (0x1UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x00200000 */
#define LPUART_CR1_DEAT_1                         (0x2UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x00400000 */
#define LPUART_CR1_DEAT_2                         (0x4UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x00800000 */
#define LPUART_CR1_DEAT_3                         (0x8UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x01000000 */
#define LPUART_CR1_DEAT_4                         (0x10UL<<LPUART_CR1_DEAT_Pos)                     /*!< 0x02000000 */
#define LPUART_CR1_M1_Pos                         (28U)
#define LPUART_CR1_M1_Msk                         (0x1UL<<LPUART_CR1_M1_Pos)                        /*!< 0x10000000 */
#define LPUART_CR1_M1                             LPUART_CR1_M1_Msk                                 /*!< The value of {M1,M0} the configurate length */
#define LPUART_CR1_FIFOEN_Pos                     (29U)
#define LPUART_CR1_FIFOEN_Msk                     (0x1UL<<LPUART_CR1_FIFOEN_Pos)                    /*!< 0x20000000 */
#define LPUART_CR1_FIFOEN                         LPUART_CR1_FIFOEN_Msk                             /*!< FIFO mode enabled */
#define LPUART_CR1_TXFEIE_Pos                     (30U)
#define LPUART_CR1_TXFEIE_Msk                     (0x1UL<<LPUART_CR1_TXFEIE_Pos)                    /*!< 0x40000000 */
#define LPUART_CR1_TXFEIE                         LPUART_CR1_TXFEIE_Msk                             /*!< TXFIFO null interrupt enabled */
#define LPUART_CR1_RXFFIE_Pos                     (31U)
#define LPUART_CR1_RXFFIE_Msk                     (0x1UL<<LPUART_CR1_RXFFIE_Pos)                    /*!< 0x80000000 */
#define LPUART_CR1_RXFFIE                         LPUART_CR1_RXFFIE_Msk                             /*!< RXFIFO full interrupt enabled */

/********************************* Bit definition for LPUART_CR2 register *******************************************/
#define LPUART_CR2_ADDM7_Pos                      (4U)
#define LPUART_CR2_ADDM7_Msk                      (0x1UL<<LPUART_CR2_ADDM7_Pos)                     /*!< 0x00000010 */
#define LPUART_CR2_ADDM7                          LPUART_CR2_ADDM7_Msk                              /*!< 7bit/4bit address detection configuration */
#define LPUART_CR2_STOP_Pos                       (12U)
#define LPUART_CR2_STOP_Msk                       (0x3UL<<LPUART_CR2_STOP_Pos)                      /*!< 0x00003000 */
#define LPUART_CR2_STOP                           LPUART_CR2_STOP_Msk                               /*!< Stop bit configuration */
#define LPUART_CR2_STOP_0                         (0x1UL<<LPUART_CR2_STOP_Pos)                      /*!< 0x00001000 */
#define LPUART_CR2_STOP_1                         (0x2UL<<LPUART_CR2_STOP_Pos)                      /*!< 0x00002000 */
#define LPUART_CR2_SWAP_Pos                       (15U)
#define LPUART_CR2_SWAP_Msk                       (0x1UL<<LPUART_CR2_SWAP_Pos)                      /*!< 0x00008000 */
#define LPUART_CR2_SWAP                           LPUART_CR2_SWAP_Msk                               /*!< TX/RX pin interchangeable */
#define LPUART_CR2_RXINV_Pos                      (16U)
#define LPUART_CR2_RXINV_Msk                      (0x1UL<<LPUART_CR2_RXINV_Pos)                     /*!< 0x00010000 */
#define LPUART_CR2_RXINV                          LPUART_CR2_RXINV_Msk                              /*!< The RX pin is active level inverting */
#define LPUART_CR2_TXINV_Pos                      (17U)
#define LPUART_CR2_TXINV_Msk                      (0x1UL<<LPUART_CR2_TXINV_Pos)                     /*!< 0x00020000 */
#define LPUART_CR2_TXINV                          LPUART_CR2_TXINV_Msk                              /*!< TX pin active level inverted */
#define LPUART_CR2_DATAINV_Pos                    (18U)
#define LPUART_CR2_DATAINV_Msk                    (0x1UL<<LPUART_CR2_DATAINV_Pos)                   /*!< 0x00040000 */
#define LPUART_CR2_DATAINV                        LPUART_CR2_DATAINV_Msk                            /*!< Inverse binary data processing */
#define LPUART_CR2_MSBFIRST_Pos                   (19U)
#define LPUART_CR2_MSBFIRST_Msk                   (0x1UL<<LPUART_CR2_MSBFIRST_Pos)                  /*!< 0x00080000 */
#define LPUART_CR2_MSBFIRST                       LPUART_CR2_MSBFIRST_Msk                           /*!< MSB first */
#define LPUART_CR2_TXOE_ALWAYS_ON_Pos             (20U)
#define LPUART_CR2_TXOE_ALWAYS_ON_Msk             (0x1UL<<LPUART_CR2_TXOE_ALWAYS_ON_Pos)            /*!< 0x00100000 */
#define LPUART_CR2_TXOE_ALWAYS_ON                 LPUART_CR2_TXOE_ALWAYS_ON_Msk                     /*!< LPUART output control */
#define LPUART_CR2_ADD_Pos                        (24U)
#define LPUART_CR2_ADD_Msk                        (0xFFUL<<LPUART_CR2_ADD_Pos)                      /*!< 0xFF000000 */
#define LPUART_CR2_ADD                            LPUART_CR2_ADD_Msk                                /*!< LPUART address */

/********************************* Bit definition for LPUART_CR3 register *******************************************/
#define LPUART_CR3_EIE_Pos                        (0U)
#define LPUART_CR3_EIE_Msk                        (0x1UL<<LPUART_CR3_EIE_Pos)                       /*!< 0x00000001 */
#define LPUART_CR3_EIE                            LPUART_CR3_EIE_Msk                                /*!< Error interrupts are enabled */
#define LPUART_CR3_HDSEL_Pos                      (3U)
#define LPUART_CR3_HDSEL_Msk                      (0x1UL<<LPUART_CR3_HDSEL_Pos)                     /*!< 0x00000008 */
#define LPUART_CR3_HDSEL                          LPUART_CR3_HDSEL_Msk                              /*!< Half-duplex selection */
#define LPUART_CR3_DMAR_Pos                       (6U)
#define LPUART_CR3_DMAR_Msk                       (0x1UL<<LPUART_CR3_DMAR_Pos)                      /*!< 0x00000040 */
#define LPUART_CR3_DMAR                           LPUART_CR3_DMAR_Msk                               /*!< Enable DMA on reception */
#define LPUART_CR3_DMAT_Pos                       (7U)
#define LPUART_CR3_DMAT_Msk                       (0x1UL<<LPUART_CR3_DMAT_Pos)                      /*!< 0x00000080 */
#define LPUART_CR3_DMAT                           LPUART_CR3_DMAT_Msk                               /*!< Enable DMA while transmitting */
#define LPUART_CR3_RTSE_Pos                       (8U)
#define LPUART_CR3_RTSE_Msk                       (0x1UL<<LPUART_CR3_RTSE_Pos)                      /*!< 0x00000100 */
#define LPUART_CR3_RTSE                           LPUART_CR3_RTSE_Msk                               /*!< RTS enabled */
#define LPUART_CR3_CTSE_Pos                       (9U)
#define LPUART_CR3_CTSE_Msk                       (0x1UL<<LPUART_CR3_CTSE_Pos)                      /*!< 0x00000200 */
#define LPUART_CR3_CTSE                           LPUART_CR3_CTSE_Msk                               /*!< CTS enabled */
#define LPUART_CR3_CTSIE_Pos                      (10U)
#define LPUART_CR3_CTSIE_Msk                      (0x1UL<<LPUART_CR3_CTSIE_Pos)                     /*!< 0x00000400 */
#define LPUART_CR3_CTSIE                          LPUART_CR3_CTSIE_Msk                              /*!< CTS interrupt enable */
#define LPUART_CR3_OVRDIS_Pos                     (12U)
#define LPUART_CR3_OVRDIS_Msk                     (0x1UL<<LPUART_CR3_OVRDIS_Pos)                    /*!< 0x00001000 */
#define LPUART_CR3_OVRDIS                         LPUART_CR3_OVRDIS_Msk                             /*!< Overrun is prohibited */
#define LPUART_CR3_DDRE_Pos                       (13U)
#define LPUART_CR3_DDRE_Msk                       (0x1UL<<LPUART_CR3_DDRE_Pos)                      /*!< 0x00002000 */
#define LPUART_CR3_DDRE                           LPUART_CR3_DDRE_Msk                               /*!< Whether DMA is disabled when receiving an error */
#define LPUART_CR3_DEM_Pos                        (14U)
#define LPUART_CR3_DEM_Msk                        (0x1UL<<LPUART_CR3_DEM_Pos)                       /*!< 0x00004000 */
#define LPUART_CR3_DEM                            LPUART_CR3_DEM_Msk                                /*!< DE mode enabled */
#define LPUART_CR3_DEP_Pos                        (15U)
#define LPUART_CR3_DEP_Msk                        (0x1UL<<LPUART_CR3_DEP_Pos)                       /*!< 0x00008000 */
#define LPUART_CR3_DEP                            LPUART_CR3_DEP_Msk                                /*!< DE polarity selection */
#define LPUART_CR3_WUS_Pos                        (20U)
#define LPUART_CR3_WUS_Msk                        (0x3UL<<LPUART_CR3_WUS_Pos)                       /*!< 0x00300000 */
#define LPUART_CR3_WUS                            LPUART_CR3_WUS_Msk                                /*!< Low-power wake-up options */
#define LPUART_CR3_WUS_0                          (0x1UL<<LPUART_CR3_WUS_Pos)                       /*!< 0x00100000 */
#define LPUART_CR3_WUS_1                          (0x2UL<<LPUART_CR3_WUS_Pos)                       /*!< 0x00200000 */
#define LPUART_CR3_WUFIE_Pos                      (22U)
#define LPUART_CR3_WUFIE_Msk                      (0x1UL<<LPUART_CR3_WUFIE_Pos)                     /*!< 0x00400000 */
#define LPUART_CR3_WUFIE                          LPUART_CR3_WUFIE_Msk                              /*!< Low-power wake-up interrupt enable */
#define LPUART_CR3_TXFTIE_Pos                     (23U)
#define LPUART_CR3_TXFTIE_Msk                     (0x1UL<<LPUART_CR3_TXFTIE_Pos)                    /*!< 0x00800000 */
#define LPUART_CR3_TXFTIE                         LPUART_CR3_TXFTIE_Msk                             /*!< TXFIFO threshold interrupt enabled */
#define LPUART_CR3_RXFTCFG_Pos                    (25U)
#define LPUART_CR3_RXFTCFG_Msk                    (0x7UL<<LPUART_CR3_RXFTCFG_Pos)                   /*!< 0x0E000000 */
#define LPUART_CR3_RXFTCFG                        LPUART_CR3_RXFTCFG_Msk                            /*!< RXFIFO full threshold configuration */
#define LPUART_CR3_RXFTCFG_0                      (0x1UL<<LPUART_CR3_RXFTCFG_Pos)                   /*!< 0x02000000 */
#define LPUART_CR3_RXFTCFG_1                      (0x2UL<<LPUART_CR3_RXFTCFG_Pos)                   /*!< 0x04000000 */
#define LPUART_CR3_RXFTCFG_2                      (0x4UL<<LPUART_CR3_RXFTCFG_Pos)                   /*!< 0x08000000 */
#define LPUART_CR3_RXFTIE_Pos                     (28U)
#define LPUART_CR3_RXFTIE_Msk                     (0x1UL<<LPUART_CR3_RXFTIE_Pos)                    /*!< 0x10000000 */
#define LPUART_CR3_RXFTIE                         LPUART_CR3_RXFTIE_Msk                             /*!< RXFIFO threshold interrupt enabled */
#define LPUART_CR3_TXFTCFG_Pos                    (29U)
#define LPUART_CR3_TXFTCFG_Msk                    (0x7UL<<LPUART_CR3_TXFTCFG_Pos)                   /*!< 0xE0000000 */
#define LPUART_CR3_TXFTCFG                        LPUART_CR3_TXFTCFG_Msk                            /*!< TXFIFO threshold configuration */
#define LPUART_CR3_TXFTCFG_0                      (0x1UL<<LPUART_CR3_TXFTCFG_Pos)                   /*!< 0x20000000 */
#define LPUART_CR3_TXFTCFG_1                      (0x2UL<<LPUART_CR3_TXFTCFG_Pos)                   /*!< 0x40000000 */
#define LPUART_CR3_TXFTCFG_2                      (0x4UL<<LPUART_CR3_TXFTCFG_Pos)                   /*!< 0x80000000 */

/********************************* Bit definition for LPUART_BRR register *******************************************/
#define LPUART_BRR_BBR_Pos                        (0U)
#define LPUART_BRR_BBR_Msk                        (0xFFFFFUL<<LPUART_BRR_BBR_Pos)                   /*!< 0x000FFFFF */
#define LPUART_BRR_BBR                            LPUART_BRR_BBR_Msk                                /*!< LPUART baud rate */

/********************************* Bit definition for LPUART_RQR register *******************************************/
#define LPUART_RQR_SBKRQ_Pos                      (1U)
#define LPUART_RQR_SBKRQ_Msk                      (0x1UL<<LPUART_RQR_SBKRQ_Pos)                     /*!< 0x00000002 */
#define LPUART_RQR_SBKRQ                          LPUART_RQR_SBKRQ_Msk                              /*!< Send an abort request */
#define LPUART_RQR_MMRQ_Pos                       (2U)
#define LPUART_RQR_MMRQ_Msk                       (0x1UL<<LPUART_RQR_MMRQ_Pos)                      /*!< 0x00000004 */
#define LPUART_RQR_MMRQ                           LPUART_RQR_MMRQ_Msk                               /*!< Silent mode request */
#define LPUART_RQR_RXFRQ_Pos                      (3U)
#define LPUART_RQR_RXFRQ_Msk                      (0x1UL<<LPUART_RQR_RXFRQ_Pos)                     /*!< 0x00000008 */
#define LPUART_RQR_RXFRQ                          LPUART_RQR_RXFRQ_Msk                              /*!< Receive a data refresh request */
#define LPUART_RQR_TXFRQ_Pos                      (4U)
#define LPUART_RQR_TXFRQ_Msk                      (0x1UL<<LPUART_RQR_TXFRQ_Pos)                     /*!< 0x00000010 */
#define LPUART_RQR_TXFRQ                          LPUART_RQR_TXFRQ_Msk                              /*!< Route a data refresh request */

/********************************* Bit definition for LPUART_ISR register *******************************************/
#define LPUART_ISR_PE_Pos                         (0U)
#define LPUART_ISR_PE_Msk                         (0x1UL<<LPUART_ISR_PE_Pos)                        /*!< 0x00000001 */
#define LPUART_ISR_PE                             LPUART_ISR_PE_Msk                                 /*!< Parity error */
#define LPUART_ISR_FE_Pos                         (1U)
#define LPUART_ISR_FE_Msk                         (0x1UL<<LPUART_ISR_FE_Pos)                        /*!< 0x00000002 */
#define LPUART_ISR_FE                             LPUART_ISR_FE_Msk                                 /*!< Frame error flag */
#define LPUART_ISR_NE_Pos                         (2U)
#define LPUART_ISR_NE_Msk                         (0x1UL<<LPUART_ISR_NE_Pos)                        /*!< 0x00000004 */
#define LPUART_ISR_NE                             LPUART_ISR_NE_Msk                                 /*!< START bit noise flag */
#define LPUART_ISR_ORE_Pos                        (3U)
#define LPUART_ISR_ORE_Msk                        (0x1UL<<LPUART_ISR_ORE_Pos)                       /*!< 0x00000008 */
#define LPUART_ISR_ORE                            LPUART_ISR_ORE_Msk                                /*!< Overrun error flag */
#define LPUART_ISR_IDLE_Pos                       (4U)
#define LPUART_ISR_IDLE_Msk                       (0x1UL<<LPUART_ISR_IDLE_Pos)                      /*!< 0x00000010 */
#define LPUART_ISR_IDLE                           LPUART_ISR_IDLE_Msk                               /*!< idle flag */
#define LPUART_ISR_RXNE_Pos                       (5U)
#define LPUART_ISR_RXNE_Msk                       (0x1UL<<LPUART_ISR_RXNE_Pos)                      /*!< 0x00000020 */
#define LPUART_ISR_RXNE                           LPUART_ISR_RXNE_Msk                               /*!< RX register not empty */
#define LPUART_ISR_TC_Pos                         (6U)
#define LPUART_ISR_TC_Msk                         (0x1UL<<LPUART_ISR_TC_Pos)                        /*!< 0x00000040 */
#define LPUART_ISR_TC                             LPUART_ISR_TC_Msk                                 /*!< Transmission completion flag */
#define LPUART_ISR_TXE_Pos                        (7U)
#define LPUART_ISR_TXE_Msk                        (0x1UL<<LPUART_ISR_TXE_Pos)                       /*!< 0x00000080 */
#define LPUART_ISR_TXE                            LPUART_ISR_TXE_Msk                                /*!< TX register not full */
#define LPUART_ISR_CTSIF_Pos                      (9U)
#define LPUART_ISR_CTSIF_Msk                      (0x1UL<<LPUART_ISR_CTSIF_Pos)                     /*!< 0x00000200 */
#define LPUART_ISR_CTSIF                          LPUART_ISR_CTSIF_Msk                              /*!< CTS interrupt flag */
#define LPUART_ISR_CTS_Pos                        (10U)
#define LPUART_ISR_CTS_Msk                        (0x1UL<<LPUART_ISR_CTS_Pos)                       /*!< 0x00000400 */
#define LPUART_ISR_CTS                            LPUART_ISR_CTS_Msk                                /*!< CTS flag */
#define LPUART_ISR_BUSY_Pos                       (16U)
#define LPUART_ISR_BUSY_Msk                       (0x1UL<<LPUART_ISR_BUSY_Pos)                      /*!< 0x00010000 */
#define LPUART_ISR_BUSY                           LPUART_ISR_BUSY_Msk                               /*!< Busy flag */
#define LPUART_ISR_CMF_Pos                        (17U)
#define LPUART_ISR_CMF_Msk                        (0x1UL<<LPUART_ISR_CMF_Pos)                       /*!< 0x00020000 */
#define LPUART_ISR_CMF                            LPUART_ISR_CMF_Msk                                /*!< Address match flag */
#define LPUART_ISR_SBKF_Pos                       (18U)
#define LPUART_ISR_SBKF_Msk                       (0x1UL<<LPUART_ISR_SBKF_Pos)                      /*!< 0x00040000 */
#define LPUART_ISR_SBKF                           LPUART_ISR_SBKF_Msk                               /*!< Send abort flag */
#define LPUART_ISR_RWU_Pos                        (19U)
#define LPUART_ISR_RWU_Msk                        (0x1UL<<LPUART_ISR_RWU_Pos)                       /*!< 0x00080000 */
#define LPUART_ISR_RWU                            LPUART_ISR_RWU_Msk                                /*!< Receive Mute mode wake-up flag */
#define LPUART_ISR_WUF_Pos                        (20U)
#define LPUART_ISR_WUF_Msk                        (0x1UL<<LPUART_ISR_WUF_Pos)                       /*!< 0x00100000 */
#define LPUART_ISR_WUF                            LPUART_ISR_WUF_Msk                                /*!< Wake flag from low-power mode */
#define LPUART_ISR_TEACK_Pos                      (21U)
#define LPUART_ISR_TEACK_Msk                      (0x1UL<<LPUART_ISR_TEACK_Pos)                     /*!< 0x00200000 */
#define LPUART_ISR_TEACK                          LPUART_ISR_TEACK_Msk                              /*!< Transport enable acknowledgement flag */
#define LPUART_ISR_REACK_Pos                      (22U)
#define LPUART_ISR_REACK_Msk                      (0x1UL<<LPUART_ISR_REACK_Pos)                     /*!< 0x00400000 */
#define LPUART_ISR_REACK                          LPUART_ISR_REACK_Msk                              /*!< Receive the enable acknowledgment flag */
#define LPUART_ISR_TXFE_Pos                       (23U)
#define LPUART_ISR_TXFE_Msk                       (0x1UL<<LPUART_ISR_TXFE_Pos)                      /*!< 0x00800000 */
#define LPUART_ISR_TXFE                           LPUART_ISR_TXFE_Msk                               /*!< TXFIFO empty */
#define LPUART_ISR_RXFF_Pos                       (24U)
#define LPUART_ISR_RXFF_Msk                       (0x1UL<<LPUART_ISR_RXFF_Pos)                      /*!< 0x01000000 */
#define LPUART_ISR_RXFF                           LPUART_ISR_RXFF_Msk                               /*!< RXFIFO full logo */
#define LPUART_ISR_RXFT_Pos                       (26U)
#define LPUART_ISR_RXFT_Msk                       (0x1UL<<LPUART_ISR_RXFT_Pos)                      /*!< 0x04000000 */
#define LPUART_ISR_RXFT                           LPUART_ISR_RXFT_Msk                               /*!< RCFIFO threshold flag */
#define LPUART_ISR_TXFT_Pos                       (27U)
#define LPUART_ISR_TXFT_Msk                       (0x1UL<<LPUART_ISR_TXFT_Pos)                      /*!< 0x08000000 */
#define LPUART_ISR_TXFT                           LPUART_ISR_TXFT_Msk                               /*!< TXFIFO threshold flag */

/********************************* Bit definition for LPUART_ICR register *******************************************/
#define LPUART_ICR_PECF_Pos                       (0U)
#define LPUART_ICR_PECF_Msk                       (0x1UL<<LPUART_ICR_PECF_Pos)                      /*!< 0x00000001 */
#define LPUART_ICR_PECF                           LPUART_ICR_PECF_Msk                               /*!< Check value error flag reset */
#define LPUART_ICR_FECF_Pos                       (1U)
#define LPUART_ICR_FECF_Msk                       (0x1UL<<LPUART_ICR_FECF_Pos)                      /*!< 0x00000002 */
#define LPUART_ICR_FECF                           LPUART_ICR_FECF_Msk                               /*!< Frame error flag reset */
#define LPUART_ICR_NECF_Pos                       (2U)
#define LPUART_ICR_NECF_Msk                       (0x1UL<<LPUART_ICR_NECF_Pos)                      /*!< 0x00000004 */
#define LPUART_ICR_NECF                           LPUART_ICR_NECF_Msk                               /*!< Clear the START bit noise flag */
#define LPUART_ICR_ORECF_Pos                      (3U)
#define LPUART_ICR_ORECF_Msk                      (0x1UL<<LPUART_ICR_ORECF_Pos)                     /*!< 0x00000008 */
#define LPUART_ICR_ORECF                          LPUART_ICR_ORECF_Msk                              /*!< Overrun error flag reset */
#define LPUART_ICR_IDLECF_Pos                     (4U)
#define LPUART_ICR_IDLECF_Msk                     (0x1UL<<LPUART_ICR_IDLECF_Pos)                    /*!< 0x00000010 */
#define LPUART_ICR_IDLECF                         LPUART_ICR_IDLECF_Msk                             /*!< Clear idle flag  */
#define LPUART_ICR_TCCF_Pos                       (6U)
#define LPUART_ICR_TCCF_Msk                       (0x1UL<<LPUART_ICR_TCCF_Pos)                      /*!< 0x00000040 */
#define LPUART_ICR_TCCF                           LPUART_ICR_TCCF_Msk                               /*!< Transfer completion flag reset */
#define LPUART_ICR_CTSCF_Pos                      (9U)
#define LPUART_ICR_CTSCF_Msk                      (0x1UL<<LPUART_ICR_CTSCF_Pos)                     /*!< 0x00000200 */
#define LPUART_ICR_CTSCF                          LPUART_ICR_CTSCF_Msk                              /*!< CTS flag reset */
#define LPUART_ICR_CMCF_Pos                       (17U)
#define LPUART_ICR_CMCF_Msk                       (0x1UL<<LPUART_ICR_CMCF_Pos)                      /*!< 0x00020000 */
#define LPUART_ICR_CMCF                           LPUART_ICR_CMCF_Msk                               /*!< Address matching flag reset */
#define LPUART_ICR_WUCF_Pos                       (20U)
#define LPUART_ICR_WUCF_Msk                       (0x1UL<<LPUART_ICR_WUCF_Pos)                      /*!< 0x00100000 */
#define LPUART_ICR_WUCF                           LPUART_ICR_WUCF_Msk                               /*!< Low power wake-up flag reset */

/********************************* Bit definition for LPUART_RDR register *******************************************/
#define LPUART_RDR_RDR_Pos                        (0U)
#define LPUART_RDR_RDR_Msk                        (0x1FFUL<<LPUART_RDR_RDR_Pos)                     /*!< 0x000001FF */
#define LPUART_RDR_RDR                            LPUART_RDR_RDR_Msk                                /*!< Receive Data Register */

/********************************* Bit definition for LPUART_TDR register *******************************************/
#define LPUART_TDR_TDR_Pos                        (0U)
#define LPUART_TDR_TDR_Msk                        (0x1FFUL<<LPUART_TDR_TDR_Pos)                     /*!< 0x000001FF */
#define LPUART_TDR_TDR                            LPUART_TDR_TDR_Msk                                /*!< Transmit Data Register */

/********************************* Bit definition for LPUART_PRESC register *****************************************/
#define LPUART_PRESC_PRESCALER_Pos                (0U)
#define LPUART_PRESC_PRESCALER_Msk                (0xFUL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x0000000F */
#define LPUART_PRESC_PRESCALER                    LPUART_PRESC_PRESCALER_Msk                        /*!< Input clock prescaler register */
#define LPUART_PRESC_PRESCALER_0                  (0x1UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000001 */
#define LPUART_PRESC_PRESCALER_1                  (0x2UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000002 */
#define LPUART_PRESC_PRESCALER_2                  (0x4UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000004 */
#define LPUART_PRESC_PRESCALER_3                  (0x8UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000008 */

/********************************************************************************************************************/
/********************************* PWR ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for PWR_CR1 register **********************************************/
#define PWR_CR1_LPMS_Pos                          (0U)
#define PWR_CR1_LPMS_Msk                          (0x3UL<<PWR_CR1_LPMS_Pos)                         /*!< 0x00000003 */
#define PWR_CR1_LPMS                              PWR_CR1_LPMS_Msk
#define PWR_CR1_LPMS_0                            (0x1UL<<PWR_CR1_LPMS_Pos)                         /*!< 0x00000001 */
#define PWR_CR1_LPMS_1                            (0x2UL<<PWR_CR1_LPMS_Pos)                         /*!< 0x00000002 */
#define PWR_CR1_LPMS_STOP0                        (0x00000000U)                                     /*!< Stop 0 mode */
#define PWR_CR1_LPMS_STOP1_Pos                    (0U)
#define PWR_CR1_LPMS_STOP1_Msk                    (0x1UL << PWR_CR1_LPMS_STOP1_Pos)                 /*!< 0x00000001 */
#define PWR_CR1_LPMS_STOP1                        PWR_CR1_LPMS_STOP1_Msk                            /*!< Stop 1 mode */
#define PWR_CR1_LPMS_STANDBY_Pos                  (0U)
#define PWR_CR1_LPMS_STANDBY_Msk                  (0x3UL << PWR_CR1_LPMS_STANDBY_Pos)               /*!< 0x00000003 */
#define PWR_CR1_LPMS_STANDBY                      PWR_CR1_LPMS_STANDBY_Msk                          /*!< Stand-by mode */
#define PWR_CR1_DBP_Pos                           (8U)
#define PWR_CR1_DBP_Msk                           (0x1UL<<PWR_CR1_DBP_Pos)                          /*!< 0x00000100 */
#define PWR_CR1_DBP                               PWR_CR1_DBP_Msk                                   
#define PWR_CR1_VOS_Pos                           (9U)
#define PWR_CR1_VOS_Msk                           (0x3UL<<PWR_CR1_VOS_Pos)                          /*!< 0x00000600 */
#define PWR_CR1_VOS                               PWR_CR1_VOS_Msk
#define PWR_CR1_VOS_0                             (0x1UL<<PWR_CR1_VOS_Pos)                          /*!< 0x00000200 */
#define PWR_CR1_VOS_1                             (0x2UL<<PWR_CR1_VOS_Pos)                          /*!< 0x00000400 */
#define PWR_CR1_SRAM_RETV_Pos                     (11U)
#define PWR_CR1_SRAM_RETV_Msk                     (0x3UL<<PWR_CR1_SRAM_RETV_Pos)                    /*!< 0x00001800 */
#define PWR_CR1_SRAM_RETV                         PWR_CR1_SRAM_RETV_Msk
#define PWR_CR1_SRAM_RETV_0                       (0x1UL<<PWR_CR1_SRAM_RETV_Pos)                    /*!< 0x00000800 */
#define PWR_CR1_SRAM_RETV_1                       (0x2UL<<PWR_CR1_SRAM_RETV_Pos)                    /*!< 0x00001000 */
#define PWR_CR1_LPR_Pos                           (14U)
#define PWR_CR1_LPR_Msk                           (0x1UL<<PWR_CR1_LPR_Pos)                          /*!< 0x00004000 */
#define PWR_CR1_LPR                               PWR_CR1_LPR_Msk                                   
#define PWR_CR1_HSION_CTRL_Pos                    (16U)
#define PWR_CR1_HSION_CTRL_Msk                    (0x1UL<<PWR_CR1_HSION_CTRL_Pos)                   /*!< 0x00010000 */
#define PWR_CR1_HSION_CTRL                        PWR_CR1_HSION_CTRL_Msk                            
#define PWR_CR1_FLS_WUPT_Pos                      (17U)
#define PWR_CR1_FLS_WUPT_Msk                      (0x3UL<<PWR_CR1_FLS_WUPT_Pos)                     /*!< 0x00060000 */
#define PWR_CR1_FLS_WUPT                          PWR_CR1_FLS_WUPT_Msk
#define PWR_CR1_FLS_WUPT_0                        (0x1UL<<PWR_CR1_FLS_WUPT_Pos)                     /*!< 0x00020000 */
#define PWR_CR1_FLS_WUPT_1                        (0x2UL<<PWR_CR1_FLS_WUPT_Pos)                     /*!< 0x00040000 */
#define PWR_CR1_STDBY_MRRDY_WAIT_Pos              (19U)
#define PWR_CR1_STDBY_MRRDY_WAIT_Msk              (0x3UL<<PWR_CR1_STDBY_MRRDY_WAIT_Pos)             /*!< 0x00180000 */
#define PWR_CR1_STDBY_MRRDY_WAIT                  PWR_CR1_STDBY_MRRDY_WAIT_Msk
#define PWR_CR1_STDBY_MRRDY_WAIT_0                (0x1UL<<PWR_CR1_STDBY_MRRDY_WAIT_Pos)             /*!< 0x00080000 */
#define PWR_CR1_STDBY_MRRDY_WAIT_1                (0x2UL<<PWR_CR1_STDBY_MRRDY_WAIT_Pos)             /*!< 0x00100000 */

/********************************* Bit definition for PWR_CR2 register **********************************************/
#define PWR_CR2_PVDE_Pos                          (0U)
#define PWR_CR2_PVDE_Msk                          (0x1UL<<PWR_CR2_PVDE_Pos)                         /*!< 0x00000001 */
#define PWR_CR2_PVDE                              PWR_CR2_PVDE_Msk                                  
#define PWR_CR2_PLS_Pos                           (1U)
#define PWR_CR2_PLS_Msk                           (0x7UL<<PWR_CR2_PLS_Pos)                          /*!< 0x0000000E */
#define PWR_CR2_PLS                               PWR_CR2_PLS_Msk
#define PWR_CR2_PLS_0                             (0x1UL<<PWR_CR2_PLS_Pos)                          /*!< 0x00000002 */
#define PWR_CR2_PLS_1                             (0x2UL<<PWR_CR2_PLS_Pos)                          /*!< 0x00000004 */
#define PWR_CR2_PLS_2                             (0x4UL<<PWR_CR2_PLS_Pos)                          /*!< 0x00000008 */

/*!< PVD level configuration */
/* #define PWR_CR2_PLS_LEV0                          0x00000000U */                                      /*!< PVD level 1.8V */
/* #define PWR_CR2_PLS_LEV1                          0x00000002U */                                      /*!< PVD level 2.0V */
/* #define PWR_CR2_PLS_LEV2                          0x00000004U */                                      /*!< PVD level 2.2V */
#define PWR_CR2_PLS_LEV3                          0x00000006U                                       /*!< PVD level 2.4V */
#define PWR_CR2_PLS_LEV4                          0x00000008U                                       /*!< PVD level 2.6V */
#define PWR_CR2_PLS_LEV5                          0x0000000AU                                       /*!< PVD level 2.8V */
#define PWR_CR2_PLS_LEV6                          0x0000000CU                                       /*!< PVD level 3.0V */
#define PWR_CR2_PLS_LEV7                          0x0000000EU

/* Legacy defines */
/* #define PWR_CR2_PLS_1V8                           PWR_CR2_PLS_LEV0 */
/* #define PWR_CR2_PLS_2V0                           PWR_CR2_PLS_LEV1 */
/* #define PWR_CR2_PLS_2V2                           PWR_CR2_PLS_LEV2 */
#define PWR_CR2_PLS_2V4                           PWR_CR2_PLS_LEV3
#define PWR_CR2_PLS_2V6                           PWR_CR2_PLS_LEV4
#define PWR_CR2_PLS_2V8                           PWR_CR2_PLS_LEV5
#define PWR_CR2_PLS_3V0                           PWR_CR2_PLS_LEV6
#define PWR_CR2_PLS_IO_IN                         PWR_CR2_PLS_LEV7

#define PWR_CR2_FLTEN_Pos                         (8U)
#define PWR_CR2_FLTEN_Msk                         (0x1UL<<PWR_CR2_FLTEN_Pos)                        /*!< 0x00000100 */
#define PWR_CR2_FLTEN                             PWR_CR2_FLTEN_Msk                                 
#define PWR_CR2_FLT_CTRL_Pos                      (9U)
#define PWR_CR2_FLT_CTRL_Msk                      (0x7UL<<PWR_CR2_FLT_CTRL_Pos)                     /*!< 0x00000E00 */
#define PWR_CR2_FLT_CTRL                          PWR_CR2_FLT_CTRL_Msk
#define PWR_CR2_FLT_CTRL_0                        (0x1UL<<PWR_CR2_FLT_CTRL_Pos)                     /*!< 0x00000200 */
#define PWR_CR2_FLT_CTRL_1                        (0x2UL<<PWR_CR2_FLT_CTRL_Pos)                     /*!< 0x00000400 */
#define PWR_CR2_FLT_CTRL_2                        (0x4UL<<PWR_CR2_FLT_CTRL_Pos)                     /*!< 0x00000800 */

/********************************* Bit definition for PWR_CR3 register **********************************************/
#define PWR_CR3_EWUP_Pos                          (0U)
#define PWR_CR3_EWUP_Msk                          (0x1FUL<<PWR_CR3_EWUP_Pos)                        /*!< 0x0000001F */
#define PWR_CR3_EWUP                              PWR_CR3_EWUP_Msk                                 
#define PWR_CR3_EWUP1_Pos                         (0U)
#define PWR_CR3_EWUP1_Msk                         (0x1UL<<PWR_CR3_EWUP1_Pos)                        /*!< 0x00000001 */
#define PWR_CR3_EWUP1                             PWR_CR3_EWUP1_Msk                                 
#define PWR_CR3_EWUP2_Pos                         (1U)
#define PWR_CR3_EWUP2_Msk                         (0x1UL<<PWR_CR3_EWUP2_Pos)                        /*!< 0x00000002 */
#define PWR_CR3_EWUP2                             PWR_CR3_EWUP2_Msk                                 
#define PWR_CR3_EWUP3_Pos                         (2U)
#define PWR_CR3_EWUP3_Msk                         (0x1UL<<PWR_CR3_EWUP3_Pos)                        /*!< 0x00000004 */
#define PWR_CR3_EWUP3                             PWR_CR3_EWUP3_Msk                                 
#define PWR_CR3_EWUP4_Pos                         (3U)
#define PWR_CR3_EWUP4_Msk                         (0x1UL<<PWR_CR3_EWUP4_Pos)                        /*!< 0x00000008 */
#define PWR_CR3_EWUP4                             PWR_CR3_EWUP4_Msk                                 
#define PWR_CR3_EWUP5_Pos                         (4U)
#define PWR_CR3_EWUP5_Msk                         (0x1UL<<PWR_CR3_EWUP5_Pos)                        /*!< 0x00000010 */
#define PWR_CR3_EWUP5                             PWR_CR3_EWUP5_Msk                                 
#define PWR_CR3_RRS_Pos                           (8U)
#define PWR_CR3_RRS_Msk                           (0x1UL<<PWR_CR3_RRS_Pos)                          /*!< 0x00000100 */
#define PWR_CR3_RRS                               PWR_CR3_RRS_Msk                                   

/********************************* Bit definition for PWR_CR4 register **********************************************/
#define PWR_CR4_WP1_Pos                           (0U)
#define PWR_CR4_WP1_Msk                           (0x1UL<<PWR_CR4_WP1_Pos)                          /*!< 0x00000001 */
#define PWR_CR4_WP1                               PWR_CR4_WP1_Msk                                   
#define PWR_CR4_WP2_Pos                           (1U)
#define PWR_CR4_WP2_Msk                           (0x1UL<<PWR_CR4_WP2_Pos)                          /*!< 0x00000002 */
#define PWR_CR4_WP2                               PWR_CR4_WP2_Msk                                   
#define PWR_CR4_WP3_Pos                           (2U)
#define PWR_CR4_WP3_Msk                           (0x1UL<<PWR_CR4_WP3_Pos)                          /*!< 0x00000004 */
#define PWR_CR4_WP3                               PWR_CR4_WP3_Msk                                   
#define PWR_CR4_WP4_Pos                           (3U)
#define PWR_CR4_WP4_Msk                           (0x1UL<<PWR_CR4_WP4_Pos)                          /*!< 0x00000008 */
#define PWR_CR4_WP4                               PWR_CR4_WP4_Msk                                   
#define PWR_CR4_WP5_Pos                           (4U)
#define PWR_CR4_WP5_Msk                           (0x1UL<<PWR_CR4_WP5_Pos)                          /*!< 0x00000010 */
#define PWR_CR4_WP5                               PWR_CR4_WP5_Msk                                   

/********************************* Bit definition for PWR_SR register ***********************************************/
#define PWR_SR_WUF1_Pos                           (0U)
#define PWR_SR_WUF1_Msk                           (0x1UL<<PWR_SR_WUF1_Pos)                          /*!< 0x00000001 */
#define PWR_SR_WUF1                               PWR_SR_WUF1_Msk                                   
#define PWR_SR_WUF2_Pos                           (1U)
#define PWR_SR_WUF2_Msk                           (0x1UL<<PWR_SR_WUF2_Pos)                          /*!< 0x00000002 */
#define PWR_SR_WUF2                               PWR_SR_WUF2_Msk                                   
#define PWR_SR_WUF3_Pos                           (2U)
#define PWR_SR_WUF3_Msk                           (0x1UL<<PWR_SR_WUF3_Pos)                          /*!< 0x00000004 */
#define PWR_SR_WUF3                               PWR_SR_WUF3_Msk                                   
#define PWR_SR_WUF4_Pos                           (3U)
#define PWR_SR_WUF4_Msk                           (0x1UL<<PWR_SR_WUF4_Pos)                          /*!< 0x00000008 */
#define PWR_SR_WUF4                               PWR_SR_WUF4_Msk                                   
#define PWR_SR_WUF5_Pos                           (4U)
#define PWR_SR_WUF5_Msk                           (0x1UL<<PWR_SR_WUF5_Pos)                          /*!< 0x00000010 */
#define PWR_SR_WUF5                               PWR_SR_WUF5_Msk                                   
#define PWR_SR_WUF_Pos                            (0U)
#define PWR_SR_WUF_Msk                            (0x1FUL<<PWR_SR_WUF_Pos)                          /*!< 0x0000001F */
#define PWR_SR_WUF                                PWR_SR_WUF_Msk                                   
#define PWR_SR_SBF_Pos                            (8U)
#define PWR_SR_SBF_Msk                            (0x1UL<<PWR_SR_SBF_Pos)                           /*!< 0x00000100 */
#define PWR_SR_SBF                                PWR_SR_SBF_Msk                                    
#define PWR_SR_MR_RDY_Pos                         (9U)
#define PWR_SR_MR_RDY_Msk                         (0x1UL<<PWR_SR_MR_RDY_Pos)                        /*!< 0x00000200 */
#define PWR_SR_MR_RDY                             PWR_SR_MR_RDY_Msk                                 
#define PWR_SR_PVDO_Pos                           (16U)
#define PWR_SR_PVDO_Msk                           (0x1UL<<PWR_SR_PVDO_Pos)                          /*!< 0x00010000 */
#define PWR_SR_PVDO                               PWR_SR_PVDO_Msk                                   

/********************************* Bit definition for PWR_SCR register **********************************************/
#define PWR_SCR_CWUF_Pos                          (0U)
#define PWR_SCR_CWUF_Msk                          (0x1FUL<<PWR_SCR_CWUF_Pos)                        /*!< 0x0000001F */
#define PWR_SCR_CWUF                              PWR_SCR_CWUF_Msk                                 
#define PWR_SCR_CWUF1_Pos                         (0U)
#define PWR_SCR_CWUF1_Msk                         (0x1UL<<PWR_SCR_CWUF1_Pos)                        /*!< 0x00000001 */
#define PWR_SCR_CWUF1                             PWR_SCR_CWUF1_Msk                                 
#define PWR_SCR_CWUF2_Pos                         (1U)
#define PWR_SCR_CWUF2_Msk                         (0x1UL<<PWR_SCR_CWUF2_Pos)                        /*!< 0x00000002 */
#define PWR_SCR_CWUF2                             PWR_SCR_CWUF2_Msk                                 
#define PWR_SCR_CWUF3_Pos                         (2U)
#define PWR_SCR_CWUF3_Msk                         (0x1UL<<PWR_SCR_CWUF3_Pos)                        /*!< 0x00000004 */
#define PWR_SCR_CWUF3                             PWR_SCR_CWUF3_Msk                                 
#define PWR_SCR_CWUF4_Pos                         (3U)
#define PWR_SCR_CWUF4_Msk                         (0x1UL<<PWR_SCR_CWUF4_Pos)                        /*!< 0x00000008 */
#define PWR_SCR_CWUF4                             PWR_SCR_CWUF4_Msk                                 
#define PWR_SCR_CWUF5_Pos                         (4U)
#define PWR_SCR_CWUF5_Msk                         (0x1UL<<PWR_SCR_CWUF5_Pos)                        /*!< 0x00000010 */
#define PWR_SCR_CWUF5                             PWR_SCR_CWUF5_Msk                                 
#define PWR_SCR_CSBF_Pos                          (8U)
#define PWR_SCR_CSBF_Msk                          (0x1UL<<PWR_SCR_CSBF_Pos)                         /*!< 0x00000100 */
#define PWR_SCR_CSBF                              PWR_SCR_CSBF_Msk                                  

/********************************************************************************************************************/
/********************************* ESMC *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for ESMC_CR register **********************************************/
#define ESMC_CR_SOFTRST_Pos                       (0U)
#define ESMC_CR_SOFTRST_Msk                       (0x1UL<<ESMC_CR_SOFTRST_Pos)                      /*!< 0x00000001 */
#define ESMC_CR_SOFTRST                           ESMC_CR_SOFTRST_Msk                               /*!< Software Reset */
#define ESMC_CR_2QSPI_Pos                         (2U)
#define ESMC_CR_2QSPI_Msk                         (0x1UL<<ESMC_CR_2QSPI_Pos)                        /*!< 0x00000004 */
#define ESMC_CR_2QSPI                             ESMC_CR_2QSPI_Msk                                 /*!< Dual Flash Mode */
#define ESMC_CR_DMAEN_Pos                         (3U)
#define ESMC_CR_DMAEN_Msk                         (0x1UL<<ESMC_CR_DMAEN_Pos)                        /*!< 0x00000008 */
#define ESMC_CR_DMAEN                             ESMC_CR_DMAEN_Msk                                 /*!< dma en */
#define ESMC_CR_GIE_Pos                           (4U)
#define ESMC_CR_GIE_Msk                           (0x1UL<<ESMC_CR_GIE_Pos)                          /*!< 0x00000010 */
#define ESMC_CR_GIE                               ESMC_CR_GIE_Msk                                   /*!< globle interrupt enable  */
#define ESMC_CR_XIPEN_Pos                         (6U)
#define ESMC_CR_XIPEN_Msk                         (0x1UL<<ESMC_CR_XIPEN_Pos)                        /*!< 0x00000040 */
#define ESMC_CR_XIPEN                             ESMC_CR_XIPEN_Msk                                 /*!< xip mode en */
#define ESMC_CR_SPIEN_Pos                         (7U)
#define ESMC_CR_SPIEN_Msk                         (0x1UL<<ESMC_CR_SPIEN_Pos)                        /*!< 0x00000080 */
#define ESMC_CR_SPIEN                             ESMC_CR_SPIEN_Msk                                 /*!< spi en */

/********************************* Bit definition for ESMC_CR2 register *********************************************/
#define ESMC_CR2_CPHA_Pos                         (0U)
#define ESMC_CR2_CPHA_Msk                         (0x1UL<<ESMC_CR2_CPHA_Pos)                        /*!< 0x00000001 */
#define ESMC_CR2_CPHA                             ESMC_CR2_CPHA_Msk                                 /*!< clock phase */
#define ESMC_CR2_CPOL_Pos                         (1U)
#define ESMC_CR2_CPOL_Msk                         (0x1UL<<ESMC_CR2_CPOL_Pos)                        /*!< 0x00000002 */
#define ESMC_CR2_CPOL                             ESMC_CR2_CPOL_Msk                                 /*!< clock polarity  */
#define ESMC_CR2_TXFIFO_CLR_Pos                   (3U)
#define ESMC_CR2_TXFIFO_CLR_Msk                   (0x1UL<<ESMC_CR2_TXFIFO_CLR_Pos)                  /*!< 0x00000008 */
#define ESMC_CR2_TXFIFO_CLR                       ESMC_CR2_TXFIFO_CLR_Msk                           /*!< txfifo clear */
#define ESMC_CR2_RXFIFO_CLR_Pos                   (4U)
#define ESMC_CR2_RXFIFO_CLR_Msk                   (0x1UL<<ESMC_CR2_RXFIFO_CLR_Pos)                  /*!< 0x00000010 */
#define ESMC_CR2_RXFIFO_CLR                       ESMC_CR2_RXFIFO_CLR_Msk                           /*!< rxfifo clear */
#define ESMC_CR2_XSPI_DUAL_LAT_OFF_Pos            (7U)
#define ESMC_CR2_XSPI_DUAL_LAT_OFF_Msk            (0x1UL<<ESMC_CR2_XSPI_DUAL_LAT_OFF_Pos)           /*!< 0x00000080 */
#define ESMC_CR2_XSPI_DUAL_LAT_OFF                ESMC_CR2_XSPI_DUAL_LAT_OFF_Msk                    /*!< xspi dual latency off enable */

/********************************* Bit definition for ESMC_TCR register *********************************************/
#define ESMC_TCR_HOLD_Pos                         (0U)
#define ESMC_TCR_HOLD_Msk                         (0xFUL<<ESMC_TCR_HOLD_Pos)                        /*!< 0x0000000F */
#define ESMC_TCR_HOLD                             ESMC_TCR_HOLD_Msk                                 /*!< setup/hold time */
#define ESMC_TCR_HOLD_0                           (0x1UL<<ESMC_TCR_HOLD_Pos)                        /*!< 0x00000001 */
#define ESMC_TCR_HOLD_1                           (0x2UL<<ESMC_TCR_HOLD_Pos)                        /*!< 0x00000002 */
#define ESMC_TCR_HOLD_2                           (0x4UL<<ESMC_TCR_HOLD_Pos)                        /*!< 0x00000004 */
#define ESMC_TCR_HOLD_3                           (0x8UL<<ESMC_TCR_HOLD_Pos)                        /*!< 0x00000008 */
#define ESMC_TCR_HIGH_Pos                         (4U)
#define ESMC_TCR_HIGH_Msk                         (0xFUL<<ESMC_TCR_HIGH_Pos)                        /*!< 0x000000F0 */
#define ESMC_TCR_HIGH                             ESMC_TCR_HIGH_Msk                                 /*!< high time */
#define ESMC_TCR_HIGH_0                           (0x1UL<<ESMC_TCR_HIGH_Pos)                        /*!< 0x00000010 */
#define ESMC_TCR_HIGH_1                           (0x2UL<<ESMC_TCR_HIGH_Pos)                        /*!< 0x00000020 */
#define ESMC_TCR_HIGH_2                           (0x4UL<<ESMC_TCR_HIGH_Pos)                        /*!< 0x00000040 */
#define ESMC_TCR_HIGH_3                           (0x8UL<<ESMC_TCR_HIGH_Pos)                        /*!< 0x00000080 */

/********************************* Bit definition for ESMC_BAUD register ********************************************/
#define ESMC_BAUD_BAUD_Pos                        (0U)
#define ESMC_BAUD_BAUD_Msk                        (0xFFUL<<ESMC_BAUD_BAUD_Pos)                      /*!< 0x000000FF */
#define ESMC_BAUD_BAUD                            ESMC_BAUD_BAUD_Msk                                /*!< baud set */

/********************************* Bit definition for ESMC_SFCR register ********************************************/
#define ESMC_SFCR_SFCR0_Pos                       (0U)
#define ESMC_SFCR_SFCR0_Msk                       (0xFFUL<<ESMC_SFCR_SFCR0_Pos)                     /*!< 0x000000FF */
#define ESMC_SFCR_SFCR0                           ESMC_SFCR_SFCR0_Msk                               /*!< Command Only */

/********************************* Bit definition for ESMC_SOCR register ********************************************/
#define ESMC_SOCR_SPIMODE_Pos                     (0U)
#define ESMC_SOCR_SPIMODE_Msk                     (0x3UL<<ESMC_SOCR_SPIMODE_Pos)                    /*!< 0x00000003 */
#define ESMC_SOCR_SPIMODE                         ESMC_SOCR_SPIMODE_Msk                             /*!< spi mode */
#define ESMC_SOCR_SPIMODE_0                       (0x1UL<<ESMC_SOCR_SPIMODE_Pos)                    /*!< 0x00000001 */
#define ESMC_SOCR_SPIMODE_1                       (0x2UL<<ESMC_SOCR_SPIMODE_Pos)                    /*!< 0x00000002 */
#define ESMC_SOCR_SENM_Pos                        (2U)
#define ESMC_SOCR_SENM_Msk                        (0x1UL<<ESMC_SOCR_SENM_Pos)                       /*!< 0x00000004 */
#define ESMC_SOCR_SENM                            ESMC_SOCR_SENM_Msk                                /*!< send mode enable */
#define ESMC_SOCR_MULTICMD_Pos                    (3U)
#define ESMC_SOCR_MULTICMD_Msk                    (0x1UL<<ESMC_SOCR_MULTICMD_Pos)                   /*!< 0x00000008 */
#define ESMC_SOCR_MULTICMD                        ESMC_SOCR_MULTICMD_Msk                            /*!< muti command  */
#define ESMC_SOCR_MULTIADDR_Pos                   (4U)
#define ESMC_SOCR_MULTIADDR_Msk                   (0x1UL<<ESMC_SOCR_MULTIADDR_Pos)                  /*!< 0x00000010 */
#define ESMC_SOCR_MULTIADDR                       ESMC_SOCR_MULTIADDR_Msk                           /*!< muti address */
#define ESMC_SOCR_DDRCMD_Pos                      (5U)
#define ESMC_SOCR_DDRCMD_Msk                      (0x1UL<<ESMC_SOCR_DDRCMD_Pos)                     /*!< 0x00000020 */
#define ESMC_SOCR_DDRCMD                          ESMC_SOCR_DDRCMD_Msk                              /*!< ddr command */
#define ESMC_SOCR_DDRADDR_Pos                     (6U)
#define ESMC_SOCR_DDRADDR_Msk                     (0x1UL<<ESMC_SOCR_DDRADDR_Pos)                    /*!< 0x00000040 */
#define ESMC_SOCR_DDRADDR                         ESMC_SOCR_DDRADDR_Msk                             /*!< ddr address */
#define ESMC_SOCR_DDRDATA_Pos                     (7U)
#define ESMC_SOCR_DDRDATA_Msk                     (0x1UL<<ESMC_SOCR_DDRDATA_Pos)                    /*!< 0x00000080 */
#define ESMC_SOCR_DDRDATA                         ESMC_SOCR_DDRDATA_Msk                             /*!< ddr data */

/********************************* Bit definition for ESMC_DCR register *********************************************/
#define ESMC_DCR_DUMMY_Pos                        (0U)
#define ESMC_DCR_DUMMY_Msk                        (0x3FUL<<ESMC_DCR_DUMMY_Pos)                      /*!< 0x0000003F */
#define ESMC_DCR_DUMMY                            ESMC_DCR_DUMMY_Msk                                /*!< dummy cycyle set */
#define ESMC_DCR_DUMMY_0                          (0x1UL<<ESMC_DCR_DUMMY_Pos)                       /*!< 0x00000001 */
#define ESMC_DCR_DUMMY_1                          (0x2UL<<ESMC_DCR_DUMMY_Pos)                       /*!< 0x00000002 */
#define ESMC_DCR_DUMMY_2                          (0x4UL<<ESMC_DCR_DUMMY_Pos)                       /*!< 0x00000004 */
#define ESMC_DCR_DUMMY_3                          (0x8UL<<ESMC_DCR_DUMMY_Pos)                       /*!< 0x00000008 */
#define ESMC_DCR_DUMMY_4                          (0x10UL<<ESMC_DCR_DUMMY_Pos)                      /*!< 0x00000010 */
#define ESMC_DCR_DUMMY_5                          (0x20UL<<ESMC_DCR_DUMMY_Pos)                      /*!< 0x00000020 */
#define ESMC_DCR_NO_CMD_Pos                       (6U)
#define ESMC_DCR_NO_CMD_Msk                       (0x1UL<<ESMC_DCR_NO_CMD_Pos)                      /*!< 0x00000040 */
#define ESMC_DCR_NO_CMD                           ESMC_DCR_NO_CMD_Msk                               /*!< no command  */
#define ESMC_DCR_REC_Pos                          (7U)
#define ESMC_DCR_REC_Msk                          (0x1UL<<ESMC_DCR_REC_Pos)                         /*!< 0x00000080 */
#define ESMC_DCR_REC                              ESMC_DCR_REC_Msk                                  /*!< receive enable  */

/********************************* Bit definition for ESMC_CR3 register *********************************************/
#define ESMC_CR3_ADDR8BIT_Pos                     (0U)
#define ESMC_CR3_ADDR8BIT_Msk                     (0x1UL<<ESMC_CR3_ADDR8BIT_Pos)                    /*!< 0x00000001 */
#define ESMC_CR3_ADDR8BIT                         ESMC_CR3_ADDR8BIT_Msk                             /*!< 8bit address */
#define ESMC_CR3_ADDR16BIT_Pos                    (1U)
#define ESMC_CR3_ADDR16BIT_Msk                    (0x1UL<<ESMC_CR3_ADDR16BIT_Pos)                   /*!< 0x00000002 */
#define ESMC_CR3_ADDR16BIT                        ESMC_CR3_ADDR16BIT_Msk                            /*!< 16bit address */
#define ESMC_CR3_ADDR32BIT_Pos                    (2U)
#define ESMC_CR3_ADDR32BIT_Msk                    (0x1UL<<ESMC_CR3_ADDR32BIT_Pos)                   /*!< 0x00000004 */
#define ESMC_CR3_ADDR32BIT                        ESMC_CR3_ADDR32BIT_Msk                            /*!< 32bit address */
#define ESMC_CR3_NO_ADDR_Pos                      (3U)
#define ESMC_CR3_NO_ADDR_Msk                      (0x1UL<<ESMC_CR3_NO_ADDR_Pos)                     /*!< 0x00000008 */
#define ESMC_CR3_NO_ADDR                          ESMC_CR3_NO_ADDR_Msk                              /*!< no address enable */
#define ESMC_CR3_XSPI_RWDS_Pos                    (4U)
#define ESMC_CR3_XSPI_RWDS_Msk                    (0x1UL<<ESMC_CR3_XSPI_RWDS_Pos)                   /*!< 0x00000010 */
#define ESMC_CR3_XSPI_RWDS                        ESMC_CR3_XSPI_RWDS_Msk                            /*!< xSPI RWDS enable */
#define ESMC_CR3_CMD_EXT_Pos                      (5U)
#define ESMC_CR3_CMD_EXT_Msk                      (0x1UL<<ESMC_CR3_CMD_EXT_Pos)                     /*!< 0x00000020 */
#define ESMC_CR3_CMD_EXT                          ESMC_CR3_CMD_EXT_Msk                              /*!< command extend enable */
#define ESMC_CR3_NO_DATA_Pos                      (7U)
#define ESMC_CR3_NO_DATA_Msk                      (0x1UL<<ESMC_CR3_NO_DATA_Pos)                     /*!< 0x00000080 */
#define ESMC_CR3_NO_DATA                          ESMC_CR3_NO_DATA_Msk                              /*!< no data enable */

/********************************* Bit definition for ESMC_ADDR24 register ******************************************/
#define ESMC_ADDR24_INS1_Pos                      (0U)
#define ESMC_ADDR24_INS1_Msk                      (0xFFUL<<ESMC_ADDR24_INS1_Pos)                    /*!< 0x000000FF */
#define ESMC_ADDR24_INS1                          ESMC_ADDR24_INS1_Msk                              /*!< command1 */
#define ESMC_ADDR24_ADDR0_Pos                     (8U)
#define ESMC_ADDR24_ADDR0_Msk                     (0xFFUL<<ESMC_ADDR24_ADDR0_Pos)                   /*!< 0x0000FF00 */
#define ESMC_ADDR24_ADDR0                         ESMC_ADDR24_ADDR0_Msk                             /*!< address0 */
#define ESMC_ADDR24_ADDR1_Pos                     (16U)
#define ESMC_ADDR24_ADDR1_Msk                     (0xFFUL<<ESMC_ADDR24_ADDR1_Pos)                   /*!< 0x00FF0000 */
#define ESMC_ADDR24_ADDR1                         ESMC_ADDR24_ADDR1_Msk                             /*!< address1 */
#define ESMC_ADDR24_ADDR2_Pos                     (24U)
#define ESMC_ADDR24_ADDR2_Msk                     (0xFFUL<<ESMC_ADDR24_ADDR2_Pos)                   /*!< 0xFF000000 */
#define ESMC_ADDR24_ADDR2                         ESMC_ADDR24_ADDR2_Msk                             /*!< address2 */

/********************************* Bit definition for ESMC_ADDR32 register ******************************************/
#define ESMC_ADDR32_ADDR3_Pos                     (0U)
#define ESMC_ADDR32_ADDR3_Msk                     (0xFFUL<<ESMC_ADDR32_ADDR3_Pos)                   /*!< 0x000000FF */
#define ESMC_ADDR32_ADDR3                         ESMC_ADDR32_ADDR3_Msk                             /*!< address3 */
#define ESMC_ADDR32_MREG_Pos                      (8U)
#define ESMC_ADDR32_MREG_Msk                      (0xFFUL<<ESMC_ADDR32_MREG_Pos)                    /*!< 0x0000FF00 */
#define ESMC_ADDR32_MREG                          ESMC_ADDR32_MREG_Msk                              /*!< Mode register */
#define ESMC_ADDR32_SSXO_Pos                      (16U)
#define ESMC_ADDR32_SSXO_Msk                      (0x3FUL<<ESMC_ADDR32_SSXO_Pos)                    /*!< 0x003F0000 */
#define ESMC_ADDR32_SSXO                          ESMC_ADDR32_SSXO_Msk                              /*!< Slave select enable */
#define ESMC_ADDR32_SSXO_0                        (0x1UL<<ESMC_ADDR32_SSXO_Pos)                     /*!< 0x00010000 */
#define ESMC_ADDR32_SSXO_1                        (0x2UL<<ESMC_ADDR32_SSXO_Pos)                     /*!< 0x00020000 */
#define ESMC_ADDR32_SSXO_2                        (0x4UL<<ESMC_ADDR32_SSXO_Pos)                     /*!< 0x00040000 */
#define ESMC_ADDR32_SSXO_3                        (0x8UL<<ESMC_ADDR32_SSXO_Pos)                     /*!< 0x00080000 */
#define ESMC_ADDR32_SSXO_4                        (0x10UL<<ESMC_ADDR32_SSXO_Pos)                    /*!< 0x00100000 */
#define ESMC_ADDR32_SSXO_5                        (0x20UL<<ESMC_ADDR32_SSXO_Pos)                    /*!< 0x00200000 */
#define ESMC_ADDR32_SS_CLR_Pos                    (22U)
#define ESMC_ADDR32_SS_CLR_Msk                    (0x1UL<<ESMC_ADDR32_SS_CLR_Pos)                   /*!< 0x00400000 */
#define ESMC_ADDR32_SS_CLR                        ESMC_ADDR32_SS_CLR_Msk                            /*!< Slave select clear */
#define ESMC_ADDR32_SS_SET_Pos                    (23U)
#define ESMC_ADDR32_SS_SET_Msk                    (0x1UL<<ESMC_ADDR32_SS_SET_Pos)                   /*!< 0x00800000 */
#define ESMC_ADDR32_SS_SET                        ESMC_ADDR32_SS_SET_Msk                            /*!< Slave select set */

/********************************* Bit definition for ESMC_DATA register ********************************************/
#define ESMC_DATA_DATA0_Pos                       (0U)
#define ESMC_DATA_DATA0_Msk                       (0xFFUL<<ESMC_DATA_DATA0_Pos)                     /*!< 0x000000FF */
#define ESMC_DATA_DATA0                           ESMC_DATA_DATA0_Msk                               /*!< data0 */
#define ESMC_DATA_DATA1_Pos                       (8U)
#define ESMC_DATA_DATA1_Msk                       (0xFFUL<<ESMC_DATA_DATA1_Pos)                     /*!< 0x0000FF00 */
#define ESMC_DATA_DATA1                           ESMC_DATA_DATA1_Msk                               /*!< data1 */
#define ESMC_DATA_DATA2_Pos                       (16U)
#define ESMC_DATA_DATA2_Msk                       (0xFFUL<<ESMC_DATA_DATA2_Pos)                     /*!< 0x00FF0000 */
#define ESMC_DATA_DATA2                           ESMC_DATA_DATA2_Msk                               /*!< data2 */
#define ESMC_DATA_DATA3_Pos                       (24U)
#define ESMC_DATA_DATA3_Msk                       (0xFFUL<<ESMC_DATA_DATA3_Pos)                     /*!< 0xFF000000 */
#define ESMC_DATA_DATA3                           ESMC_DATA_DATA3_Msk                               /*!< data3 */

/********************************* Bit definition for ESMC_SR register **********************************************/
#define ESMC_SR_SSACTIVE_Pos                      (0U)
#define ESMC_SR_SSACTIVE_Msk                      (0x1UL<<ESMC_SR_SSACTIVE_Pos)                     /*!< 0x00000001 */
#define ESMC_SR_SSACTIVE                          ESMC_SR_SSACTIVE_Msk                              /*!< slave select active */
#define ESMC_SR_RXBUSY_Pos                        (1U)
#define ESMC_SR_RXBUSY_Msk                        (0x1UL<<ESMC_SR_RXBUSY_Pos)                       /*!< 0x00000002 */
#define ESMC_SR_RXBUSY                            ESMC_SR_RXBUSY_Msk                                /*!< rx busy state */
#define ESMC_SR_TXBUSY_Pos                        (2U)
#define ESMC_SR_TXBUSY_Msk                        (0x1UL<<ESMC_SR_TXBUSY_Pos)                       /*!< 0x00000004 */
#define ESMC_SR_TXBUSY                            ESMC_SR_TXBUSY_Msk                                /*!< tx busy state */
#define ESMC_SR_XIPBUSY_Pos                       (3U)
#define ESMC_SR_XIPBUSY_Msk                       (0x1UL<<ESMC_SR_XIPBUSY_Pos)                      /*!< 0x00000008 */
#define ESMC_SR_XIPBUSY                           ESMC_SR_XIPBUSY_Msk                               /*!< xip busy state */
#define ESMC_SR_IDLE_STATE_Pos                    (4U)
#define ESMC_SR_IDLE_STATE_Msk                    (0x1UL<<ESMC_SR_IDLE_STATE_Pos)                   /*!< 0x00000010 */
#define ESMC_SR_IDLE_STATE                        ESMC_SR_IDLE_STATE_Msk                            /*!< idle state */
#define ESMC_SR_TXFIFOOV_Pos                      (5U)
#define ESMC_SR_TXFIFOOV_Msk                      (0x1UL<<ESMC_SR_TXFIFOOV_Pos)                     /*!< 0x00000020 */
#define ESMC_SR_TXFIFOOV                          ESMC_SR_TXFIFOOV_Msk                              /*!< txfifo overflow state */
#define ESMC_SR_DATAWAIT_Pos                      (6U)
#define ESMC_SR_DATAWAIT_Msk                      (0x1UL<<ESMC_SR_DATAWAIT_Pos)                     /*!< 0x00000040 */
#define ESMC_SR_DATAWAIT                          ESMC_SR_DATAWAIT_Msk                              /*!< data wait */
#define ESMC_SR_SPIF_Pos                          (7U)
#define ESMC_SR_SPIF_Msk                          (0x1UL<<ESMC_SR_SPIF_Pos)                         /*!< 0x00000080 */
#define ESMC_SR_SPIF                              ESMC_SR_SPIF_Msk                                  /*!< spi interrupt flag */

/********************************* Bit definition for ESMC_SR1 register *********************************************/
#define ESMC_SR1_RWDSI_ND_Pos                     (0U)
#define ESMC_SR1_RWDSI_ND_Msk                     (0x1UL<<ESMC_SR1_RWDSI_ND_Pos)                    /*!< 0x00000001 */
#define ESMC_SR1_RWDSI_ND                         ESMC_SR1_RWDSI_ND_Msk                             /*!< rwdsi not detect */
#define ESMC_SR1_XSPI_DUAL_LATENCY_Pos            (1U)
#define ESMC_SR1_XSPI_DUAL_LATENCY_Msk            (0x1UL<<ESMC_SR1_XSPI_DUAL_LATENCY_Pos)           /*!< 0x00000002 */
#define ESMC_SR1_XSPI_DUAL_LATENCY                ESMC_SR1_XSPI_DUAL_LATENCY_Msk                    /*!< xspi dual latency flag */

/********************************* Bit definition for ESMC_RXSTAT register ******************************************/
#define ESMC_RXSTAT_RXSTAT_Pos                    (0U)
#define ESMC_RXSTAT_RXSTAT_Msk                    (0xFFUL<<ESMC_RXSTAT_RXSTAT_Pos)                  /*!< 0x000000FF */
#define ESMC_RXSTAT_RXSTAT                        ESMC_RXSTAT_RXSTAT_Msk                            /*!< rx state */

/********************************* Bit definition for ESMC_IFR1 register ********************************************/
#define ESMC_IFR1_CMDIF_Pos                       (0U)
#define ESMC_IFR1_CMDIF_Msk                       (0x1UL<<ESMC_IFR1_CMDIF_Pos)                      /*!< 0x00000001 */
#define ESMC_IFR1_CMDIF                           ESMC_IFR1_CMDIF_Msk                               /*!< command done  interrupt flag */
#define ESMC_IFR1_RXFIFONEIF_Pos                  (1U)
#define ESMC_IFR1_RXFIFONEIF_Msk                  (0x1UL<<ESMC_IFR1_RXFIFONEIF_Pos)                 /*!< 0x00000002 */
#define ESMC_IFR1_RXFIFONEIF                      ESMC_IFR1_RXFIFONEIF_Msk                          /*!< rxfifo not empty  interrupt flag */
#define ESMC_IFR1_RXFIFOHFIF_Pos                  (2U)
#define ESMC_IFR1_RXFIFOHFIF_Msk                  (0x1UL<<ESMC_IFR1_RXFIFOHFIF_Pos)                 /*!< 0x00000004 */
#define ESMC_IFR1_RXFIFOHFIF                      ESMC_IFR1_RXFIFOHFIF_Msk                          /*!< rxfifo half full  interrupt flag */
#define ESMC_IFR1_RXFIFOFIF_Pos                   (3U)
#define ESMC_IFR1_RXFIFOFIF_Msk                   (0x1UL<<ESMC_IFR1_RXFIFOFIF_Pos)                  /*!< 0x00000008 */
#define ESMC_IFR1_RXFIFOFIF                       ESMC_IFR1_RXFIFOFIF_Msk                           /*!< rxfifo full  interrupt flag */
#define ESMC_IFR1_IDLEDONEIF_Pos                  (4U)
#define ESMC_IFR1_IDLEDONEIF_Msk                  (0x1UL<<ESMC_IFR1_IDLEDONEIF_Pos)                 /*!< 0x00000010 */
#define ESMC_IFR1_IDLEDONEIF                      ESMC_IFR1_IDLEDONEIF_Msk                          /*!< idle  interrupt flag */
#define ESMC_IFR1_DATAWAITIF_Pos                  (5U)
#define ESMC_IFR1_DATAWAITIF_Msk                  (0x1UL<<ESMC_IFR1_DATAWAITIF_Pos)                 /*!< 0x00000020 */
#define ESMC_IFR1_DATAWAITIF                      ESMC_IFR1_DATAWAITIF_Msk                          /*!< data wait  interrupt flag */
#define ESMC_IFR1_RXFIFOWIF_Pos                   (6U)
#define ESMC_IFR1_RXFIFOWIF_Msk                   (0x1UL<<ESMC_IFR1_RXFIFOWIF_Pos)                  /*!< 0x00000040 */
#define ESMC_IFR1_RXFIFOWIF                       ESMC_IFR1_RXFIFOWIF_Msk                           /*!< rx fifo word  interrupt flag */
#define ESMC_IFR1_ADDRDONEIF_Pos                  (7U)
#define ESMC_IFR1_ADDRDONEIF_Msk                  (0x1UL<<ESMC_IFR1_ADDRDONEIF_Pos)                 /*!< 0x00000080 */
#define ESMC_IFR1_ADDRDONEIF                      ESMC_IFR1_ADDRDONEIF_Msk                          /*!< address done interrupt flag */

/********************************* Bit definition for ESMC_IER1 register ********************************************/
#define ESMC_IER1_CMDIE_Pos                       (0U)
#define ESMC_IER1_CMDIE_Msk                       (0x1UL<<ESMC_IER1_CMDIE_Pos)                      /*!< 0x00000001 */
#define ESMC_IER1_CMDIE                           ESMC_IER1_CMDIE_Msk                               /*!< command done interrupt enable */
#define ESMC_IER1_RXFIFONEIE_Pos                  (1U)
#define ESMC_IER1_RXFIFONEIE_Msk                  (0x1UL<<ESMC_IER1_RXFIFONEIE_Pos)                 /*!< 0x00000002 */
#define ESMC_IER1_RXFIFONEIE                      ESMC_IER1_RXFIFONEIE_Msk                          /*!< rx fifo not empty interrupt enable */
#define ESMC_IER1_RXFIFOHFIE_Pos                  (2U)
#define ESMC_IER1_RXFIFOHFIE_Msk                  (0x1UL<<ESMC_IER1_RXFIFOHFIE_Pos)                 /*!< 0x00000004 */
#define ESMC_IER1_RXFIFOHFIE                      ESMC_IER1_RXFIFOHFIE_Msk                          /*!< rx fifo half interrupt enable */
#define ESMC_IER1_RXFIFOFIE_Pos                   (3U)
#define ESMC_IER1_RXFIFOFIE_Msk                   (0x1UL<<ESMC_IER1_RXFIFOFIE_Pos)                  /*!< 0x00000008 */
#define ESMC_IER1_RXFIFOFIE                       ESMC_IER1_RXFIFOFIE_Msk                           /*!< rx fifo full interrupt enable */
#define ESMC_IER1_IDLEDONEIE_Pos                  (4U)
#define ESMC_IER1_IDLEDONEIE_Msk                  (0x1UL<<ESMC_IER1_IDLEDONEIE_Pos)                 /*!< 0x00000010 */
#define ESMC_IER1_IDLEDONEIE                      ESMC_IER1_IDLEDONEIE_Msk                          /*!< idle interrupt enable */
#define ESMC_IER1_DATAWAITIE_Pos                  (5U)
#define ESMC_IER1_DATAWAITIE_Msk                  (0x1UL<<ESMC_IER1_DATAWAITIE_Pos)                 /*!< 0x00000020 */
#define ESMC_IER1_DATAWAITIE                      ESMC_IER1_DATAWAITIE_Msk                          /*!< busy interrupt enable */
#define ESMC_IER1_RXFIFOWIE_Pos                   (6U)
#define ESMC_IER1_RXFIFOWIE_Msk                   (0x1UL<<ESMC_IER1_RXFIFOWIE_Pos)                  /*!< 0x00000040 */
#define ESMC_IER1_RXFIFOWIE                       ESMC_IER1_RXFIFOWIE_Msk                           /*!< rx fifo word  interrupt enable */
#define ESMC_IER1_ADDRDONEIE_Pos                  (7U)
#define ESMC_IER1_ADDRDONEIE_Msk                  (0x1UL<<ESMC_IER1_ADDRDONEIE_Pos)                 /*!< 0x00000080 */
#define ESMC_IER1_ADDRDONEIE                      ESMC_IER1_ADDRDONEIE_Msk                          /*!< address done  interrupt enable */

/********************************* Bit definition for ESMC_IFR2 register ********************************************/
#define ESMC_IFR2_XIPRDINSTRSETIF_Pos             (0U)
#define ESMC_IFR2_XIPRDINSTRSETIF_Msk             (0x1UL<<ESMC_IFR2_XIPRDINSTRSETIF_Pos)            /*!< 0x00000001 */
#define ESMC_IFR2_XIPRDINSTRSETIF                 ESMC_IFR2_XIPRDINSTRSETIF_Msk                     /*!< xip read  instr set interrupt flag */
#define ESMC_IFR2_TXFIFOEIF_Pos                   (1U)
#define ESMC_IFR2_TXFIFOEIF_Msk                   (0x1UL<<ESMC_IFR2_TXFIFOEIF_Pos)                  /*!< 0x00000002 */
#define ESMC_IFR2_TXFIFOEIF                       ESMC_IFR2_TXFIFOEIF_Msk                           /*!< tx fifo empty  interrupt flag */
#define ESMC_IFR2_TXFIFOHEIF_Pos                  (2U)
#define ESMC_IFR2_TXFIFOHEIF_Msk                  (0x1UL<<ESMC_IFR2_TXFIFOHEIF_Pos)                 /*!< 0x00000004 */
#define ESMC_IFR2_TXFIFOHEIF                      ESMC_IFR2_TXFIFOHEIF_Msk                          /*!< tx fifo half empty interrupt flag */
#define ESMC_IFR2_TXFIFOFIF_Pos                   (3U)
#define ESMC_IFR2_TXFIFOFIF_Msk                   (0x1UL<<ESMC_IFR2_TXFIFOFIF_Pos)                  /*!< 0x00000008 */
#define ESMC_IFR2_TXFIFOFIF                       ESMC_IFR2_TXFIFOFIF_Msk                           /*!< tx fifo full interrupt flag */
#define ESMC_IFR2_TXFIFOOVIF_Pos                  (4U)
#define ESMC_IFR2_TXFIFOOVIF_Msk                  (0x1UL<<ESMC_IFR2_TXFIFOOVIF_Pos)                 /*!< 0x00000010 */
#define ESMC_IFR2_TXFIFOOVIF                      ESMC_IFR2_TXFIFOOVIF_Msk                          /*!< tx fifo overflow interrupt flag */
#define ESMC_IFR2_TXWORDOVIF_Pos                  (5U)
#define ESMC_IFR2_TXWORDOVIF_Msk                  (0x1UL<<ESMC_IFR2_TXWORDOVIF_Pos)                 /*!< 0x00000020 */
#define ESMC_IFR2_TXWORDOVIF                      ESMC_IFR2_TXWORDOVIF_Msk                          /*!< txfifo word interrupt flag */
#define ESMC_IFR2_XIPWEINSTRSETIF_Pos             (6U)
#define ESMC_IFR2_XIPWEINSTRSETIF_Msk             (0x1UL<<ESMC_IFR2_XIPWEINSTRSETIF_Pos)            /*!< 0x00000040 */
#define ESMC_IFR2_XIPWEINSTRSETIF                 ESMC_IFR2_XIPWEINSTRSETIF_Msk                     /*!< xip write instr set interrupt flag */
#define ESMC_IFR2_RXTOUTIF_Pos                    (7U)
#define ESMC_IFR2_RXTOUTIF_Msk                    (0x1UL<<ESMC_IFR2_RXTOUTIF_Pos)                   /*!< 0x00000080 */
#define ESMC_IFR2_RXTOUTIF                        ESMC_IFR2_RXTOUTIF_Msk                            /*!< rx timeout interrupt flag */

/********************************* Bit definition for ESMC_IER2 register ********************************************/
#define ESMC_IER2_TXFIFOEIE_Pos                   (1U)
#define ESMC_IER2_TXFIFOEIE_Msk                   (0x1UL<<ESMC_IER2_TXFIFOEIE_Pos)                  /*!< 0x00000002 */
#define ESMC_IER2_TXFIFOEIE                       ESMC_IER2_TXFIFOEIE_Msk                           /*!< tx fifo empty  interrupt enable */
#define ESMC_IER2_TXFIFOHEIE_Pos                  (2U)
#define ESMC_IER2_TXFIFOHEIE_Msk                  (0x1UL<<ESMC_IER2_TXFIFOHEIE_Pos)                 /*!< 0x00000004 */
#define ESMC_IER2_TXFIFOHEIE                      ESMC_IER2_TXFIFOHEIE_Msk                          /*!< tx fifo half full  interrupt enable */
#define ESMC_IER2_TXFIFOFIE_Pos                   (3U)
#define ESMC_IER2_TXFIFOFIE_Msk                   (0x1UL<<ESMC_IER2_TXFIFOFIE_Pos)                  /*!< 0x00000008 */
#define ESMC_IER2_TXFIFOFIE                       ESMC_IER2_TXFIFOFIE_Msk                           /*!< tx fifo full interrupt enable */
#define ESMC_IER2_TXFIFOOVIE_Pos                  (4U)
#define ESMC_IER2_TXFIFOOVIE_Msk                  (0x1UL<<ESMC_IER2_TXFIFOOVIE_Pos)                 /*!< 0x00000010 */
#define ESMC_IER2_TXFIFOOVIE                      ESMC_IER2_TXFIFOOVIE_Msk                          /*!< tx fifo overflow interrupt enable */
#define ESMC_IER2_TXWORDOVIE_Pos                  (5U)
#define ESMC_IER2_TXWORDOVIE_Msk                  (0x1UL<<ESMC_IER2_TXWORDOVIE_Pos)                 /*!< 0x00000020 */
#define ESMC_IER2_TXWORDOVIE                      ESMC_IER2_TXWORDOVIE_Msk                          /*!< txfifo word interrupt enable */

/********************************* Bit definition for ESMC_STRR register ********************************************/
#define ESMC_STRR_SMP_Pos                         (0U)
#define ESMC_STRR_SMP_Msk                         (0x3FUL<<ESMC_STRR_SMP_Pos)                       /*!< 0x0000003F */
#define ESMC_STRR_SMP                             ESMC_STRR_SMP_Msk                                 /*!< sample  */
#define ESMC_STRR_SMP_0                           (0x1UL<<ESMC_STRR_SMP_Pos)                        /*!< 0x00000001 */
#define ESMC_STRR_SMP_1                           (0x2UL<<ESMC_STRR_SMP_Pos)                        /*!< 0x00000002 */
#define ESMC_STRR_SMP_2                           (0x4UL<<ESMC_STRR_SMP_Pos)                        /*!< 0x00000004 */
#define ESMC_STRR_SMP_3                           (0x8UL<<ESMC_STRR_SMP_Pos)                        /*!< 0x00000008 */
#define ESMC_STRR_SMP_4                           (0x10UL<<ESMC_STRR_SMP_Pos)                       /*!< 0x00000010 */
#define ESMC_STRR_SMP_5                           (0x20UL<<ESMC_STRR_SMP_Pos)                       /*!< 0x00000020 */

/********************************* Bit definition for ESMC_XSFCR register *******************************************/
#define ESMC_XSFCR_XINS_Pos                       (0U)
#define ESMC_XSFCR_XINS_Msk                       (0xFFUL<<ESMC_XSFCR_XINS_Pos)                     /*!< 0x000000FF */
#define ESMC_XSFCR_XINS                           ESMC_XSFCR_XINS_Msk                               /*!< XIP read command */

/********************************* Bit definition for ESMC_XSOCR register *******************************************/
#define ESMC_XSOCR_SPI_MODE_Pos                   (0U)
#define ESMC_XSOCR_SPI_MODE_Msk                   (0x3UL<<ESMC_XSOCR_SPI_MODE_Pos)                  /*!< 0x00000003 */
#define ESMC_XSOCR_SPI_MODE                       ESMC_XSOCR_SPI_MODE_Msk                           /*!< xip read spi mode */
#define ESMC_XSOCR_SPI_MODE_0                     (0x1UL<<ESMC_XSOCR_SPI_MODE_Pos)                  /*!< 0x00000001 */
#define ESMC_XSOCR_SPI_MODE_1                     (0x2UL<<ESMC_XSOCR_SPI_MODE_Pos)                  /*!< 0x00000002 */
#define ESMC_XSOCR_SEND_M_Pos                     (2U)
#define ESMC_XSOCR_SEND_M_Msk                     (0x1UL<<ESMC_XSOCR_SEND_M_Pos)                    /*!< 0x00000004 */
#define ESMC_XSOCR_SEND_M                         ESMC_XSOCR_SEND_M_Msk                             /*!< xip read send mode enable */
#define ESMC_XSOCR_MULTICMD_Pos                   (3U)
#define ESMC_XSOCR_MULTICMD_Msk                   (0x1UL<<ESMC_XSOCR_MULTICMD_Pos)                  /*!< 0x00000008 */
#define ESMC_XSOCR_MULTICMD                       ESMC_XSOCR_MULTICMD_Msk                           /*!< xip read muti command  */
#define ESMC_XSOCR_MULTIADDR_Pos                  (4U)
#define ESMC_XSOCR_MULTIADDR_Msk                  (0x1UL<<ESMC_XSOCR_MULTIADDR_Pos)                 /*!< 0x00000010 */
#define ESMC_XSOCR_MULTIADDR                      ESMC_XSOCR_MULTIADDR_Msk                          /*!< xip read muti address */
#define ESMC_XSOCR_DDRCMD_Pos                     (5U)
#define ESMC_XSOCR_DDRCMD_Msk                     (0x1UL<<ESMC_XSOCR_DDRCMD_Pos)                    /*!< 0x00000020 */
#define ESMC_XSOCR_DDRCMD                         ESMC_XSOCR_DDRCMD_Msk                             /*!< xip read ddr command */
#define ESMC_XSOCR_DDRADDR_Pos                    (6U)
#define ESMC_XSOCR_DDRADDR_Msk                    (0x1UL<<ESMC_XSOCR_DDRADDR_Pos)                   /*!< 0x00000040 */
#define ESMC_XSOCR_DDRADDR                        ESMC_XSOCR_DDRADDR_Msk                            /*!< xip read ddr address */
#define ESMC_XSOCR_DDRDATA_Pos                    (7U)
#define ESMC_XSOCR_DDRDATA_Msk                    (0x1UL<<ESMC_XSOCR_DDRDATA_Pos)                   /*!< 0x00000080 */
#define ESMC_XSOCR_DDRDATA                        ESMC_XSOCR_DDRDATA_Msk                            /*!< xip read ddr data */

/********************************* Bit definition for ESMC_XDCR register ********************************************/
#define ESMC_XDCR_XDUMMY_Pos                      (0U)
#define ESMC_XDCR_XDUMMY_Msk                      (0x3FUL<<ESMC_XDCR_XDUMMY_Pos)                    /*!< 0x0000003F */
#define ESMC_XDCR_XDUMMY                          ESMC_XDCR_XDUMMY_Msk                              /*!< xip read dummy cycle set */
#define ESMC_XDCR_XDUMMY_0                        (0x1UL<<ESMC_XDCR_XDUMMY_Pos)                     /*!< 0x00000001 */
#define ESMC_XDCR_XDUMMY_1                        (0x2UL<<ESMC_XDCR_XDUMMY_Pos)                     /*!< 0x00000002 */
#define ESMC_XDCR_XDUMMY_2                        (0x4UL<<ESMC_XDCR_XDUMMY_Pos)                     /*!< 0x00000004 */
#define ESMC_XDCR_XDUMMY_3                        (0x8UL<<ESMC_XDCR_XDUMMY_Pos)                     /*!< 0x00000008 */
#define ESMC_XDCR_XDUMMY_4                        (0x10UL<<ESMC_XDCR_XDUMMY_Pos)                    /*!< 0x00000010 */
#define ESMC_XDCR_XDUMMY_5                        (0x20UL<<ESMC_XDCR_XDUMMY_Pos)                    /*!< 0x00000020 */
#define ESMC_XDCR_NO_CMD_Pos                      (6U)
#define ESMC_XDCR_NO_CMD_Msk                      (0x1UL<<ESMC_XDCR_NO_CMD_Pos)                     /*!< 0x00000040 */
#define ESMC_XDCR_NO_CMD                          ESMC_XDCR_NO_CMD_Msk                              /*!< xip read no command  */

/********************************* Bit definition for ESMC_XCR3 register ********************************************/
#define ESMC_XCR3_ADDR8BIT_Pos                    (0U)
#define ESMC_XCR3_ADDR8BIT_Msk                    (0x1UL<<ESMC_XCR3_ADDR8BIT_Pos)                   /*!< 0x00000001 */
#define ESMC_XCR3_ADDR8BIT                        ESMC_XCR3_ADDR8BIT_Msk                            /*!< xip read 8bit address */
#define ESMC_XCR3_ADDR16BIT_Pos                   (1U)
#define ESMC_XCR3_ADDR16BIT_Msk                   (0x1UL<<ESMC_XCR3_ADDR16BIT_Pos)                  /*!< 0x00000002 */
#define ESMC_XCR3_ADDR16BIT                       ESMC_XCR3_ADDR16BIT_Msk                           /*!< xip read 16bit address */
#define ESMC_XCR3_ADDR32BIT_Pos                   (2U)
#define ESMC_XCR3_ADDR32BIT_Msk                   (0x1UL<<ESMC_XCR3_ADDR32BIT_Pos)                  /*!< 0x00000004 */
#define ESMC_XCR3_ADDR32BIT                       ESMC_XCR3_ADDR32BIT_Msk                           /*!< xip read 32bit address */
#define ESMC_XCR3_NO_ADDR_Pos                     (3U)
#define ESMC_XCR3_NO_ADDR_Msk                     (0x1UL<<ESMC_XCR3_NO_ADDR_Pos)                    /*!< 0x00000008 */
#define ESMC_XCR3_NO_ADDR                         ESMC_XCR3_NO_ADDR_Msk                             /*!< xip read no address enable */
#define ESMC_XCR3_XSPI_RWDS_Pos                   (4U)
#define ESMC_XCR3_XSPI_RWDS_Msk                   (0x1UL<<ESMC_XCR3_XSPI_RWDS_Pos)                  /*!< 0x00000010 */
#define ESMC_XCR3_XSPI_RWDS                       ESMC_XCR3_XSPI_RWDS_Msk                           /*!< xip read xSPI RWDS enable */
#define ESMC_XCR3_CMD_EXT_Pos                     (5U)
#define ESMC_XCR3_CMD_EXT_Msk                     (0x1UL<<ESMC_XCR3_CMD_EXT_Pos)                    /*!< 0x00000020 */
#define ESMC_XCR3_CMD_EXT                         ESMC_XCR3_CMD_EXT_Msk                             /*!< xip read command extend enable */

/********************************* Bit definition for ESMC_XMODE register *******************************************/
#define ESMC_XMODE_XMODE_Pos                      (0U)
#define ESMC_XMODE_XMODE_Msk                      (0xFFUL<<ESMC_XMODE_XMODE_Pos)                    /*!< 0x000000FF */
#define ESMC_XMODE_XMODE                          ESMC_XMODE_XMODE_Msk                              /*!< xip send modereg */

/********************************* Bit definition for ESMC_XMODE_WE register ****************************************/
#define ESMC_XMODE_WE_XMODE_WE_Pos                (0U)
#define ESMC_XMODE_WE_XMODE_WE_Msk                (0xFFUL<<ESMC_XMODE_WE_XMODE_WE_Pos)              /*!< 0x000000FF */
#define ESMC_XMODE_WE_XMODE_WE                    ESMC_XMODE_WE_XMODE_WE_Msk                        /*!< xip send modereg_we */

/********************************* Bit definition for ESMC_XSSOCR register ******************************************/
#define ESMC_XSSOCR_SSXO_Pos                      (0U)
#define ESMC_XSSOCR_SSXO_Msk                      (0x3FUL<<ESMC_XSSOCR_SSXO_Pos)                    /*!< 0x0000003F */
#define ESMC_XSSOCR_SSXO                          ESMC_XSSOCR_SSXO_Msk                              /*!< xip read Slave select enable */
#define ESMC_XSSOCR_SSXO_0                        (0x1UL<<ESMC_XSSOCR_SSXO_Pos)                     /*!< 0x00000001 */
#define ESMC_XSSOCR_SSXO_1                        (0x2UL<<ESMC_XSSOCR_SSXO_Pos)                     /*!< 0x00000002 */
#define ESMC_XSSOCR_SSXO_2                        (0x4UL<<ESMC_XSSOCR_SSXO_Pos)                     /*!< 0x00000004 */
#define ESMC_XSSOCR_SSXO_3                        (0x8UL<<ESMC_XSSOCR_SSXO_Pos)                     /*!< 0x00000008 */
#define ESMC_XSSOCR_SSXO_4                        (0x10UL<<ESMC_XSSOCR_SSXO_Pos)                    /*!< 0x00000010 */
#define ESMC_XSSOCR_SSXO_5                        (0x20UL<<ESMC_XSSOCR_SSXO_Pos)                    /*!< 0x00000020 */
#define ESMC_XSSOCR_SS_CLR_RQ_Pos                 (6U)
#define ESMC_XSSOCR_SS_CLR_RQ_Msk                 (0x1UL<<ESMC_XSSOCR_SS_CLR_RQ_Pos)                /*!< 0x00000040 */
#define ESMC_XSSOCR_SS_CLR_RQ                     ESMC_XSSOCR_SS_CLR_RQ_Msk                         /*!< xip read Slave select clear */

/********************************* Bit definition for ESMC_XSSOCR_WE register ***************************************/
#define ESMC_XSSOCR_WE_SSXO_Pos                   (0U)
#define ESMC_XSSOCR_WE_SSXO_Msk                   (0x3FUL<<ESMC_XSSOCR_WE_SSXO_Pos)                 /*!< 0x0000003F */
#define ESMC_XSSOCR_WE_SSXO                       ESMC_XSSOCR_WE_SSXO_Msk                           /*!< xip write Slave select enable */
#define ESMC_XSSOCR_WE_SSXO_0                     (0x1UL<<ESMC_XSSOCR_WE_SSXO_Pos)                  /*!< 0x00000001 */
#define ESMC_XSSOCR_WE_SSXO_1                     (0x2UL<<ESMC_XSSOCR_WE_SSXO_Pos)                  /*!< 0x00000002 */
#define ESMC_XSSOCR_WE_SSXO_2                     (0x4UL<<ESMC_XSSOCR_WE_SSXO_Pos)                  /*!< 0x00000004 */
#define ESMC_XSSOCR_WE_SSXO_3                     (0x8UL<<ESMC_XSSOCR_WE_SSXO_Pos)                  /*!< 0x00000008 */
#define ESMC_XSSOCR_WE_SSXO_4                     (0x10UL<<ESMC_XSSOCR_WE_SSXO_Pos)                 /*!< 0x00000010 */
#define ESMC_XSSOCR_WE_SSXO_5                     (0x20UL<<ESMC_XSSOCR_WE_SSXO_Pos)                 /*!< 0x00000020 */
#define ESMC_XSSOCR_WE_SS_CLR_RQ_Pos              (6U)
#define ESMC_XSSOCR_WE_SS_CLR_RQ_Msk              (0x1UL<<ESMC_XSSOCR_WE_SS_CLR_RQ_Pos)             /*!< 0x00000040 */
#define ESMC_XSSOCR_WE_SS_CLR_RQ                  ESMC_XSSOCR_WE_SS_CLR_RQ_Msk                      /*!< xip write Slave select clear */

/********************************* Bit definition for ESMC_XSFCR_WE register ****************************************/
#define ESMC_XSFCR_WE_XINS_WE_Pos                 (0U)
#define ESMC_XSFCR_WE_XINS_WE_Msk                 (0xFFUL<<ESMC_XSFCR_WE_XINS_WE_Pos)               /*!< 0x000000FF */
#define ESMC_XSFCR_WE_XINS_WE                     ESMC_XSFCR_WE_XINS_WE_Msk                         /*!< XIP write command */

/********************************* Bit definition for ESMC_XSOCR_WE register ****************************************/
#define ESMC_XSOCR_WE_SPI_MODE_Pos                (0U)
#define ESMC_XSOCR_WE_SPI_MODE_Msk                (0x3UL<<ESMC_XSOCR_WE_SPI_MODE_Pos)               /*!< 0x00000003 */
#define ESMC_XSOCR_WE_SPI_MODE                    ESMC_XSOCR_WE_SPI_MODE_Msk                        /*!< xip write spi mode */
#define ESMC_XSOCR_WE_SPI_MODE_0                  (0x1UL<<ESMC_XSOCR_WE_SPI_MODE_Pos)               /*!< 0x00000001 */
#define ESMC_XSOCR_WE_SPI_MODE_1                  (0x2UL<<ESMC_XSOCR_WE_SPI_MODE_Pos)               /*!< 0x00000002 */
#define ESMC_XSOCR_WE_SEND_M_Pos                  (2U)
#define ESMC_XSOCR_WE_SEND_M_Msk                  (0x1UL<<ESMC_XSOCR_WE_SEND_M_Pos)                 /*!< 0x00000004 */
#define ESMC_XSOCR_WE_SEND_M                      ESMC_XSOCR_WE_SEND_M_Msk                          /*!< xip write send mode enable */
#define ESMC_XSOCR_WE_MULTICMD_Pos                (3U)
#define ESMC_XSOCR_WE_MULTICMD_Msk                (0x1UL<<ESMC_XSOCR_WE_MULTICMD_Pos)               /*!< 0x00000008 */
#define ESMC_XSOCR_WE_MULTICMD                    ESMC_XSOCR_WE_MULTICMD_Msk                        /*!< xip write muti command  */
#define ESMC_XSOCR_WE_MULTIADDR_Pos               (4U)
#define ESMC_XSOCR_WE_MULTIADDR_Msk               (0x1UL<<ESMC_XSOCR_WE_MULTIADDR_Pos)              /*!< 0x00000010 */
#define ESMC_XSOCR_WE_MULTIADDR                   ESMC_XSOCR_WE_MULTIADDR_Msk                       /*!< xip write muti address */
#define ESMC_XSOCR_WE_DDRCMD_Pos                  (5U)
#define ESMC_XSOCR_WE_DDRCMD_Msk                  (0x1UL<<ESMC_XSOCR_WE_DDRCMD_Pos)                 /*!< 0x00000020 */
#define ESMC_XSOCR_WE_DDRCMD                      ESMC_XSOCR_WE_DDRCMD_Msk                          /*!< xip write ddr command */
#define ESMC_XSOCR_WE_DDRADDR_Pos                 (6U)
#define ESMC_XSOCR_WE_DDRADDR_Msk                 (0x1UL<<ESMC_XSOCR_WE_DDRADDR_Pos)                /*!< 0x00000040 */
#define ESMC_XSOCR_WE_DDRADDR                     ESMC_XSOCR_WE_DDRADDR_Msk                         /*!< xip write ddr address */
#define ESMC_XSOCR_WE_DDRDATA_Pos                 (7U)
#define ESMC_XSOCR_WE_DDRDATA_Msk                 (0x1UL<<ESMC_XSOCR_WE_DDRDATA_Pos)                /*!< 0x00000080 */
#define ESMC_XSOCR_WE_DDRDATA                     ESMC_XSOCR_WE_DDRDATA_Msk                         /*!< xip write ddr data */

/********************************* Bit definition for ESMC_XDCR_WE register *****************************************/
#define ESMC_XDCR_WE_XDUMMY_Pos                   (0U)
#define ESMC_XDCR_WE_XDUMMY_Msk                   (0x3FUL<<ESMC_XDCR_WE_XDUMMY_Pos)                 /*!< 0x0000003F */
#define ESMC_XDCR_WE_XDUMMY                       ESMC_XDCR_WE_XDUMMY_Msk                           /*!< xip write dummy cycle set */
#define ESMC_XDCR_WE_XDUMMY_0                     (0x1UL<<ESMC_XDCR_WE_XDUMMY_Pos)                  /*!< 0x00000001 */
#define ESMC_XDCR_WE_XDUMMY_1                     (0x2UL<<ESMC_XDCR_WE_XDUMMY_Pos)                  /*!< 0x00000002 */
#define ESMC_XDCR_WE_XDUMMY_2                     (0x4UL<<ESMC_XDCR_WE_XDUMMY_Pos)                  /*!< 0x00000004 */
#define ESMC_XDCR_WE_XDUMMY_3                     (0x8UL<<ESMC_XDCR_WE_XDUMMY_Pos)                  /*!< 0x00000008 */
#define ESMC_XDCR_WE_XDUMMY_4                     (0x10UL<<ESMC_XDCR_WE_XDUMMY_Pos)                 /*!< 0x00000010 */
#define ESMC_XDCR_WE_XDUMMY_5                     (0x20UL<<ESMC_XDCR_WE_XDUMMY_Pos)                 /*!< 0x00000020 */
#define ESMC_XDCR_WE_NO_CMD_Pos                   (6U)
#define ESMC_XDCR_WE_NO_CMD_Msk                   (0x1UL<<ESMC_XDCR_WE_NO_CMD_Pos)                  /*!< 0x00000040 */
#define ESMC_XDCR_WE_NO_CMD                       ESMC_XDCR_WE_NO_CMD_Msk                           /*!< xip write no command  */

/********************************* Bit definition for ESMC_XCR3_WE register *****************************************/
#define ESMC_XCR3_WE_ADDR8BIT_Pos                 (0U)
#define ESMC_XCR3_WE_ADDR8BIT_Msk                 (0x1UL<<ESMC_XCR3_WE_ADDR8BIT_Pos)                /*!< 0x00000001 */
#define ESMC_XCR3_WE_ADDR8BIT                     ESMC_XCR3_WE_ADDR8BIT_Msk                         /*!< xip write 8bit address */
#define ESMC_XCR3_WE_ADDR16BIT_Pos                (1U)
#define ESMC_XCR3_WE_ADDR16BIT_Msk                (0x1UL<<ESMC_XCR3_WE_ADDR16BIT_Pos)               /*!< 0x00000002 */
#define ESMC_XCR3_WE_ADDR16BIT                    ESMC_XCR3_WE_ADDR16BIT_Msk                        /*!< xip write 16bit address */
#define ESMC_XCR3_WE_ADDR32BIT_Pos                (2U)
#define ESMC_XCR3_WE_ADDR32BIT_Msk                (0x1UL<<ESMC_XCR3_WE_ADDR32BIT_Pos)               /*!< 0x00000004 */
#define ESMC_XCR3_WE_ADDR32BIT                    ESMC_XCR3_WE_ADDR32BIT_Msk                        /*!< xip write 32bit address */
#define ESMC_XCR3_WE_NO_ADDR_Pos                  (3U)
#define ESMC_XCR3_WE_NO_ADDR_Msk                  (0x1UL<<ESMC_XCR3_WE_NO_ADDR_Pos)                 /*!< 0x00000008 */
#define ESMC_XCR3_WE_NO_ADDR                      ESMC_XCR3_WE_NO_ADDR_Msk                          /*!< xip write no address enable */
#define ESMC_XCR3_WE_XSPI_RWDS_Pos                (4U)
#define ESMC_XCR3_WE_XSPI_RWDS_Msk                (0x1UL<<ESMC_XCR3_WE_XSPI_RWDS_Pos)               /*!< 0x00000010 */
#define ESMC_XCR3_WE_XSPI_RWDS                    ESMC_XCR3_WE_XSPI_RWDS_Msk                        /*!< xip write xSPI RWDS enable */
#define ESMC_XCR3_WE_CMD_EXT_Pos                  (5U)
#define ESMC_XCR3_WE_CMD_EXT_Msk                  (0x1UL<<ESMC_XCR3_WE_CMD_EXT_Pos)                 /*!< 0x00000020 */
#define ESMC_XCR3_WE_CMD_EXT                      ESMC_XCR3_WE_CMD_EXT_Msk                          /*!< xip write command extend enable */

/********************************* Bit definition for ESMC_BCR register *********************************************/
#define ESMC_BCR_BCR_Pos                          (0U)
#define ESMC_BCR_BCR_Msk                          (0xFFFFFFFFUL<<ESMC_BCR_BCR_Pos)                  /*!< 0xFFFFFFFF */
#define ESMC_BCR_BCR                              ESMC_BCR_BCR_Msk                                  /*!< bit control register */

/********************************* Bit definition for ESMC_RXTOUT register ******************************************/
#define ESMC_RXTOUT_RXTOUT_Pos                    (0U)
#define ESMC_RXTOUT_RXTOUT_Msk                    (0xFFUL<<ESMC_RXTOUT_RXTOUT_Pos)                  /*!< 0x000000FF */
#define ESMC_RXTOUT_RXTOUT                        ESMC_RXTOUT_RXTOUT_Msk                            /*!< rx timeout */

/********************************* Bit definition for ESMC_XSTRR register *******************************************/
#define ESMC_XSTRR_XSMP_Pos                       (0U)
#define ESMC_XSTRR_XSMP_Msk                       (0x3FUL<<ESMC_XSTRR_XSMP_Pos)                     /*!< 0x0000003F */
#define ESMC_XSTRR_XSMP                           ESMC_XSTRR_XSMP_Msk                               /*!< xip sample sample */
#define ESMC_XSTRR_XSMP_0                         (0x1UL<<ESMC_XSTRR_XSMP_Pos)                      /*!< 0x00000001 */
#define ESMC_XSTRR_XSMP_1                         (0x2UL<<ESMC_XSTRR_XSMP_Pos)                      /*!< 0x00000002 */
#define ESMC_XSTRR_XSMP_2                         (0x4UL<<ESMC_XSTRR_XSMP_Pos)                      /*!< 0x00000004 */
#define ESMC_XSTRR_XSMP_3                         (0x8UL<<ESMC_XSTRR_XSMP_Pos)                      /*!< 0x00000008 */
#define ESMC_XSTRR_XSMP_4                         (0x10UL<<ESMC_XSTRR_XSMP_Pos)                     /*!< 0x00000010 */
#define ESMC_XSTRR_XSMP_5                         (0x20UL<<ESMC_XSTRR_XSMP_Pos)                     /*!< 0x00000020 */

/********************************************************************************************************************/
/********************************* RCC ******************************************************************************/
/********************************************************************************************************************/
#define RCC_PLL_SUPPORT
#define RCC_LSE_SUPPORT

/********************************* Bit definition for RCC_CR register ***********************************************/
#define RCC_CR_HSION_Pos                          (0U)
#define RCC_CR_HSION_Msk                          (0x1UL<<RCC_CR_HSION_Pos)                         /*!< 0x00000001 */
#define RCC_CR_HSION                              RCC_CR_HSION_Msk                                  
#define RCC_CR_HSIRDY_Pos                         (1U)
#define RCC_CR_HSIRDY_Msk                         (0x1UL<<RCC_CR_HSIRDY_Pos)                        /*!< 0x00000002 */
#define RCC_CR_HSIRDY                             RCC_CR_HSIRDY_Msk                                 
#define RCC_CR_HSIKERON_Pos                       (2U)
#define RCC_CR_HSIKERON_Msk                       (0x1UL<<RCC_CR_HSIKERON_Pos)                      /*!< 0x00000004 */
#define RCC_CR_HSIKERON                           RCC_CR_HSIKERON_Msk                                 
#define RCC_CR_HSITRIM_Pos                        (3U)
#define RCC_CR_HSITRIM_Msk                        (0x1FUL<<RCC_CR_HSITRIM_Pos)                      /*!< 0x000000F8 */
#define RCC_CR_HSITRIM                            RCC_CR_HSITRIM_Msk
#define RCC_CR_HSITRIM_0                          (0x1UL<<RCC_CR_HSITRIM_Pos)                       /*!< 0x00000008 */
#define RCC_CR_HSITRIM_1                          (0x2UL<<RCC_CR_HSITRIM_Pos)                       /*!< 0x00000010 */
#define RCC_CR_HSITRIM_2                          (0x4UL<<RCC_CR_HSITRIM_Pos)                       /*!< 0x00000020 */
#define RCC_CR_HSITRIM_3                          (0x8UL<<RCC_CR_HSITRIM_Pos)                       /*!< 0x00000040 */
#define RCC_CR_HSITRIM_4                          (0x10UL<<RCC_CR_HSITRIM_Pos)                      /*!< 0x00000080 */
#define RCC_CR_HSICAL_Pos                         (8U)
#define RCC_CR_HSICAL_Msk                         (0xE0FFUL<<RCC_CR_HSICAL_Pos)                     /*!< 0x00E0FF00 */
#define RCC_CR_HSICAL                             RCC_CR_HSICAL_Msk
#define RCC_CR_HSEON_Pos                          (16U)
#define RCC_CR_HSEON_Msk                          (0x1UL<<RCC_CR_HSEON_Pos)                         /*!< 0x00010000 */
#define RCC_CR_HSEON                              RCC_CR_HSEON_Msk                                  
#define RCC_CR_HSERDY_Pos                         (17U)
#define RCC_CR_HSERDY_Msk                         (0x1UL<<RCC_CR_HSERDY_Pos)                        /*!< 0x00020000 */
#define RCC_CR_HSERDY                             RCC_CR_HSERDY_Msk                                 
#define RCC_CR_HSEBYP_Pos                         (18U)
#define RCC_CR_HSEBYP_Msk                         (0x1UL<<RCC_CR_HSEBYP_Pos)                        /*!< 0x00040000 */
#define RCC_CR_HSEBYP                             RCC_CR_HSEBYP_Msk                                 
#define RCC_CR_CSSON_Pos                          (19U)
#define RCC_CR_CSSON_Msk                          (0x1UL<<RCC_CR_CSSON_Pos)                         /*!< 0x00080000 */
#define RCC_CR_CSSON                              RCC_CR_CSSON_Msk                                  
#define RCC_CR_PLLON_Pos                          (24U)
#define RCC_CR_PLLON_Msk                          (0x1UL<<RCC_CR_PLLON_Pos)                         /*!< 0x01000000 */
#define RCC_CR_PLLON                              RCC_CR_PLLON_Msk                                  
#define RCC_CR_PLLRDY_Pos                         (25U)
#define RCC_CR_PLLRDY_Msk                         (0x1UL<<RCC_CR_PLLRDY_Pos)                        /*!< 0x02000000 */
#define RCC_CR_PLLRDY                             RCC_CR_PLLRDY_Msk                                 

/********************************* Bit definition for RCC_CFGR register *********************************************/
#define RCC_CFGR_SW_Pos                           (0U)
#define RCC_CFGR_SW_Msk                           (0x3UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000003 */
#define RCC_CFGR_SW                               RCC_CFGR_SW_Msk
#define RCC_CFGR_SW_0                             (0x1UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000001 */
#define RCC_CFGR_SW_1                             (0x2UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000002 */
#define RCC_CFGR_SW_HSI                           (0x00000000U)                                      /*!< HSI selected as system clock */
#define RCC_CFGR_SW_HSE                           (0x00000001U)                                      /*!< HSE selected as system clock */
#define RCC_CFGR_SW_PLL                           (0x00000002U)

#define RCC_CFGR_SWS_Pos                          (2U)
#define RCC_CFGR_SWS_Msk                          (0x3UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x0000000C */
#define RCC_CFGR_SWS                              RCC_CFGR_SWS_Msk
#define RCC_CFGR_SWS_0                            (0x1UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x00000004 */
#define RCC_CFGR_SWS_1                            (0x2UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x00000008 */
#define RCC_CFGR_SWS_HSI                          (0x00000000U)                                     /*!< HSI oscillator used as system clock */
#define RCC_CFGR_SWS_HSE                          (0x00000004U)                                     /*!< HSE oscillator used as system clock */
#define RCC_CFGR_SWS_PLL                          (0x00000008U)    

#define RCC_CFGR_HPRE_Pos                         (4U)
#define RCC_CFGR_HPRE_Msk                         (0xFUL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x000000F0 */
#define RCC_CFGR_HPRE                             RCC_CFGR_HPRE_Msk
#define RCC_CFGR_HPRE_0                           (0x1UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000010 */
#define RCC_CFGR_HPRE_1                           (0x2UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000020 */
#define RCC_CFGR_HPRE_2                           (0x4UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000040 */
#define RCC_CFGR_HPRE_3                           (0x8UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000080 */
#define RCC_CFGR_HPRE_DIV1                        (0x00000000U)                                     /*!< SYSCLK not divided */
#define RCC_CFGR_HPRE_DIV2                        (0x00000080U)                                     /*!< SYSCLK divided by 2 */
#define RCC_CFGR_HPRE_DIV4                        (0x00000090U)                                     /*!< SYSCLK divided by 4 */
#define RCC_CFGR_HPRE_DIV8                        (0x000000A0U)                                     /*!< SYSCLK divided by 8 */
#define RCC_CFGR_HPRE_DIV16                       (0x000000B0U)                                     /*!< SYSCLK divided by 16 */
#define RCC_CFGR_HPRE_DIV64                       (0x000000C0U)                                     /*!< SYSCLK divided by 64 */
#define RCC_CFGR_HPRE_DIV128                      (0x000000D0U)                                     /*!< SYSCLK divided by 128 */
#define RCC_CFGR_HPRE_DIV256                      (0x000000E0U)                                     /*!< SYSCLK divided by 256 */
#define RCC_CFGR_HPRE_DIV512                      (0x000000F0U) 

#define RCC_CFGR_PPRE1_Pos                        (8U)
#define RCC_CFGR_PPRE1_Msk                        (0x7UL<<RCC_CFGR_PPRE1_Pos)                       /*!< 0x00000700 */
#define RCC_CFGR_PPRE1                            RCC_CFGR_PPRE1_Msk
#define RCC_CFGR_PPRE1_0                          (0x1UL<<RCC_CFGR_PPRE1_Pos)                       /*!< 0x00000100 */
#define RCC_CFGR_PPRE1_1                          (0x2UL<<RCC_CFGR_PPRE1_Pos)                       /*!< 0x00000200 */
#define RCC_CFGR_PPRE1_2                          (0x4UL<<RCC_CFGR_PPRE1_Pos)                       /*!< 0x00000400 */
#define RCC_CFGR_PPRE1_DIV1                       (0x00000000U)                                     /*!< HCLK not divided */
#define RCC_CFGR_PPRE1_DIV2                       (0x00000400U)                                     /*!< HCLK divided by 2 */
#define RCC_CFGR_PPRE1_DIV4                       (0x00000500U)                                     /*!< HCLK divided by 4 */
#define RCC_CFGR_PPRE1_DIV8                       (0x00000600U)                                     /*!< HCLK divided by 8 */
#define RCC_CFGR_PPRE1_DIV16                      (0x00000700U)   

#define RCC_CFGR_PPRE2_Pos                        (11U)
#define RCC_CFGR_PPRE2_Msk                        (0x7UL<<RCC_CFGR_PPRE2_Pos)                       /*!< 0x00003800 */
#define RCC_CFGR_PPRE2                            RCC_CFGR_PPRE2_Msk
#define RCC_CFGR_PPRE2_0                          (0x1UL<<RCC_CFGR_PPRE2_Pos)                       /*!< 0x00000800 */
#define RCC_CFGR_PPRE2_1                          (0x2UL<<RCC_CFGR_PPRE2_Pos)                       /*!< 0x00001000 */
#define RCC_CFGR_PPRE2_2                          (0x4UL<<RCC_CFGR_PPRE2_Pos)                       /*!< 0x00002000 */
#define RCC_CFGR_PPRE2_DIV1                       (0x00000000U)                                     /*!< HCLK not divided */
#define RCC_CFGR_PPRE2_DIV2                       (0x00002000U)                                     /*!< HCLK divided by 2 */
#define RCC_CFGR_PPRE2_DIV4                       (0x00002800U)                                     /*!< HCLK divided by 4 */
#define RCC_CFGR_PPRE2_DIV8                       (0x00003000U)                                     /*!< HCLK divided by 8 */
#define RCC_CFGR_PPRE2_DIV16                      (0x00003800U)                                     /*!< HCLK divided by 16 */
#define RCC_CFGR_ADCPRE_Pos                       (16U)
#define RCC_CFGR_ADCPRE_Msk                       (0x7UL<<RCC_CFGR_ADCPRE_Pos)                      /*!< 0x00070000 */
#define RCC_CFGR_ADCPRE                           RCC_CFGR_ADCPRE_Msk
#define RCC_CFGR_ADCPRE_0                         (0x1UL<<RCC_CFGR_ADCPRE_Pos)                      /*!< 0x00010000 */
#define RCC_CFGR_ADCPRE_1                         (0x2UL<<RCC_CFGR_ADCPRE_Pos)                      /*!< 0x00020000 */
#define RCC_CFGR_ADCPRE_2                         (0x4UL<<RCC_CFGR_ADCPRE_Pos)                      /*!< 0x00040000 */
#define RCC_CFGR_ADCPRE_DIV2                      (0x00000000U)                                     /*!< PCLK2 divided by 2 */
#define RCC_CFGR_ADCPRE_DIV4                      (0x00010000U)                                     /*!< PCLK2 divided by 4 */
#define RCC_CFGR_ADCPRE_DIV6                      (0x00020000U)                                     /*!< PCLK2 divided by 6 */
#define RCC_CFGR_ADCPRE_DIV8                      (0x00030000U)                                     /*!< PCLK2 divided by 8 */
#define RCC_CFGR_ADCPRE_DIV12                     (0x00050000U)                                     /*!< PCLK2 divided by 12 */
#define RCC_CFGR_ADCPRE_DIV16                     (0x00070000U)                                     /*!< PCLK2 divided by 16 */

#define RCC_CFGR_USBPRE_Pos                       (20U)
#define RCC_CFGR_USBPRE_Msk                       (0x7UL<<RCC_CFGR_USBPRE_Pos)                      /*!< 0x00700000 */
#define RCC_CFGR_USBPRE                           RCC_CFGR_USBPRE_Msk
#define RCC_CFGR_USBPRE_0                         (0x1UL<<RCC_CFGR_USBPRE_Pos)                      /*!< 0x00100000 */
#define RCC_CFGR_USBPRE_1                         (0x2UL<<RCC_CFGR_USBPRE_Pos)                      /*!< 0x00200000 */
#define RCC_CFGR_USBPRE_2                         (0x4UL<<RCC_CFGR_USBPRE_Pos)                      /*!< 0x00400000 */
#define RCC_CFGR_USBPRE_DIV1P5                    (0x00000000U)                                     /*!< USBCLK divided by 1.5 */
#define RCC_CFGR_USBPRE_DIV1                      (0x00100000U)                                     /*!< USBCLK divided by 1 */
#define RCC_CFGR_USBPRE_DIV2P5                    (0x00200000U)                                     /*!< USBCLK divided by 2.5 */
#define RCC_CFGR_USBPRE_DIV2                      (0x00300000U)                                     /*!< USBCLK divided by 2 */
#define RCC_CFGR_USBPRE_DIV3                      (0x00400000U)                                     /*!< USBCLK divided by 3 */
#define RCC_CFGR_USBPRE_DIV3P5                    (0x00500000U)                                     /*!< USBCLK divided by 3.5 */
#define RCC_CFGR_USBPRE_DIV4                      (0x00600000U)                                     /*!< USBCLK divided by 4 */
#define RCC_CFGR_SRAM2PRE_Pos                     (23U)
#define RCC_CFGR_SRAM2PRE_Msk                     (0x1UL<<RCC_CFGR_SRAM2PRE_Pos)                    /*!< 0x00800000 */
#define RCC_CFGR_SRAM2PRE                         RCC_CFGR_SRAM2PRE_Msk
#define RCC_CFGR_MCO_Pos                          (24U)
#define RCC_CFGR_MCO_Msk                          (0xFUL<<RCC_CFGR_MCO_Pos)                         /*!< 0x0F000000 */
#define RCC_CFGR_MCO                              RCC_CFGR_MCO_Msk
#define RCC_CFGR_MCO_0                            (0x1UL<<RCC_CFGR_MCO_Pos)                         /*!< 0x01000000 */
#define RCC_CFGR_MCO_1                            (0x2UL<<RCC_CFGR_MCO_Pos)                         /*!< 0x02000000 */
#define RCC_CFGR_MCO_2                            (0x4UL<<RCC_CFGR_MCO_Pos)                         /*!< 0x04000000 */
#define RCC_CFGR_MCO_3                            (0x8UL<<RCC_CFGR_MCO_Pos)                         /*!< 0x08000000 */

#define RCC_CFGR_MCO_NOCLOCK                      0x00000000U                                       /*!< No clock */
#define RCC_CFGR_MCO_LSE                          0x01000000U                                       /*!< LSE selected as MCO source */
#define RCC_CFGR_MCO_LSI                          0x02000000U                                       /*!< LSI selected as MCO source */
#define RCC_CFGR_MCO_HSI48M                       0x03000000U                                       /*!< HSI48 selected as MCO source */
#define RCC_CFGR_MCO_SYSCLK                       0x04000000U                                       /*!< SYSTEM CLOCK selected as MCO source */
#define RCC_CFGR_MCO_HSI16M                       0x05000000U                                       /*!< HSI16M selected as MCO source*/
#define RCC_CFGR_MCO_HSE                          0x06000000U                                       /*!< HSE selected as MCO source*/
#define RCC_CFGR_MCO_PLL                          0x07000000U                                       /*!< PLL selected as MCO source */
#define RCC_CFGR_MCO_HCLK                         0x08000000U                                       /*!< HCLK selected as MCO source */
#define RCC_CFGR_MCO_PCLK1                         0x09000000U                                      /*!< APB1CLK selected as MCO source */
#define RCC_CFGR_MCO_PCLK2                         0x0A000000U                                      /*!< APB2CLK selected as MCO source */
#define RCC_CFGR_MCO_HSI10M                       0x0B000000U                                       /*!< HSI10M selected as MCO source */

/********************************* Bit definition for RCC_CIR register **********************************************/
#define RCC_CIR_LSIRDYF_Pos                       (0U)
#define RCC_CIR_LSIRDYF_Msk                       (0x1UL<<RCC_CIR_LSIRDYF_Pos)                      /*!< 0x00000001 */
#define RCC_CIR_LSIRDYF                           RCC_CIR_LSIRDYF_Msk                               
#define RCC_CIR_LSERDYF_Pos                       (1U)
#define RCC_CIR_LSERDYF_Msk                       (0x1UL<<RCC_CIR_LSERDYF_Pos)                      /*!< 0x00000002 */
#define RCC_CIR_LSERDYF                           RCC_CIR_LSERDYF_Msk                               
#define RCC_CIR_HSIRDYF_Pos                       (2U)
#define RCC_CIR_HSIRDYF_Msk                       (0x1UL<<RCC_CIR_HSIRDYF_Pos)                      /*!< 0x00000004 */
#define RCC_CIR_HSIRDYF                           RCC_CIR_HSIRDYF_Msk                               
#define RCC_CIR_HSERDYF_Pos                       (3U)
#define RCC_CIR_HSERDYF_Msk                       (0x1UL<<RCC_CIR_HSERDYF_Pos)                      /*!< 0x00000008 */
#define RCC_CIR_HSERDYF                           RCC_CIR_HSERDYF_Msk                               
#define RCC_CIR_PLLRDYF_Pos                       (4U)
#define RCC_CIR_PLLRDYF_Msk                       (0x1UL<<RCC_CIR_PLLRDYF_Pos)                      /*!< 0x00000010 */
#define RCC_CIR_PLLRDYF                           RCC_CIR_PLLRDYF_Msk                               
#define RCC_CIR_HSI48RDYF_Pos                     (5U)
#define RCC_CIR_HSI48RDYF_Msk                     (0x1UL<<RCC_CIR_HSI48RDYF_Pos)                    /*!< 0x00000020 */
#define RCC_CIR_HSI48RDYF                         RCC_CIR_HSI48RDYF_Msk                             
#define RCC_CIR_CSSF_Pos                          (7U)
#define RCC_CIR_CSSF_Msk                          (0x1UL<<RCC_CIR_CSSF_Pos)                         /*!< 0x00000080 */
#define RCC_CIR_CSSF                              RCC_CIR_CSSF_Msk                                  
#define RCC_CIR_LSIRDYIE_Pos                      (8U)
#define RCC_CIR_LSIRDYIE_Msk                      (0x1UL<<RCC_CIR_LSIRDYIE_Pos)                     /*!< 0x00000100 */
#define RCC_CIR_LSIRDYIE                          RCC_CIR_LSIRDYIE_Msk                              
#define RCC_CIR_LSERDYIE_Pos                      (9U)
#define RCC_CIR_LSERDYIE_Msk                      (0x1UL<<RCC_CIR_LSERDYIE_Pos)                     /*!< 0x00000200 */
#define RCC_CIR_LSERDYIE                          RCC_CIR_LSERDYIE_Msk                              
#define RCC_CIR_HSIRDYIE_Pos                      (10U)
#define RCC_CIR_HSIRDYIE_Msk                      (0x1UL<<RCC_CIR_HSIRDYIE_Pos)                     /*!< 0x00000400 */
#define RCC_CIR_HSIRDYIE                          RCC_CIR_HSIRDYIE_Msk                              
#define RCC_CIR_HSERDYIE_Pos                      (11U)
#define RCC_CIR_HSERDYIE_Msk                      (0x1UL<<RCC_CIR_HSERDYIE_Pos)                     /*!< 0x00000800 */
#define RCC_CIR_HSERDYIE                          RCC_CIR_HSERDYIE_Msk                              
#define RCC_CIR_PLLRDYIE_Pos                      (12U)
#define RCC_CIR_PLLRDYIE_Msk                      (0x1UL<<RCC_CIR_PLLRDYIE_Pos)                     /*!< 0x00001000 */
#define RCC_CIR_PLLRDYIE                          RCC_CIR_PLLRDYIE_Msk                              
#define RCC_CIR_HSI48RDYIE_Pos                    (13U)
#define RCC_CIR_HSI48RDYIE_Msk                    (0x1UL<<RCC_CIR_HSI48RDYIE_Pos)                   /*!< 0x00002000 */
#define RCC_CIR_HSI48RDYIE                        RCC_CIR_HSI48RDYIE_Msk                            
#define RCC_CIR_LSIRDYC_Pos                       (16U)
#define RCC_CIR_LSIRDYC_Msk                       (0x1UL<<RCC_CIR_LSIRDYC_Pos)                      /*!< 0x00010000 */
#define RCC_CIR_LSIRDYC                           RCC_CIR_LSIRDYC_Msk                               
#define RCC_CIR_LSERDYC_Pos                       (17U)
#define RCC_CIR_LSERDYC_Msk                       (0x1UL<<RCC_CIR_LSERDYC_Pos)                      /*!< 0x00020000 */
#define RCC_CIR_LSERDYC                           RCC_CIR_LSERDYC_Msk                               
#define RCC_CIR_HSIRDYC_Pos                       (18U)
#define RCC_CIR_HSIRDYC_Msk                       (0x1UL<<RCC_CIR_HSIRDYC_Pos)                      /*!< 0x00040000 */
#define RCC_CIR_HSIRDYC                           RCC_CIR_HSIRDYC_Msk                               
#define RCC_CIR_HSERDYC_Pos                       (19U)
#define RCC_CIR_HSERDYC_Msk                       (0x1UL<<RCC_CIR_HSERDYC_Pos)                      /*!< 0x00080000 */
#define RCC_CIR_HSERDYC                           RCC_CIR_HSERDYC_Msk                               
#define RCC_CIR_PLLRDYC_Pos                       (20U)
#define RCC_CIR_PLLRDYC_Msk                       (0x1UL<<RCC_CIR_PLLRDYC_Pos)                      /*!< 0x00100000 */
#define RCC_CIR_PLLRDYC                           RCC_CIR_PLLRDYC_Msk                               
#define RCC_CIR_HSI48RDYC_Pos                     (21U)
#define RCC_CIR_HSI48RDYC_Msk                     (0x1UL<<RCC_CIR_HSI48RDYC_Pos)                    /*!< 0x00200000 */
#define RCC_CIR_HSI48RDYC                         RCC_CIR_HSI48RDYC_Msk                             
#define RCC_CIR_CSSC_Pos                          (23U)
#define RCC_CIR_CSSC_Msk                          (0x1UL<<RCC_CIR_CSSC_Pos)                         /*!< 0x00800000 */
#define RCC_CIR_CSSC                              RCC_CIR_CSSC_Msk                                  

/********************************* Bit definition for RCC_AHB1RSTR register *****************************************/
#define RCC_AHB1RSTR_DMA1RST_Pos                  (0U)
#define RCC_AHB1RSTR_DMA1RST_Msk                  (0x1UL<<RCC_AHB1RSTR_DMA1RST_Pos)                 /*!< 0x00000001 */
#define RCC_AHB1RSTR_DMA1RST                      RCC_AHB1RSTR_DMA1RST_Msk                          
#define RCC_AHB1RSTR_DMA2RST_Pos                  (1U)
#define RCC_AHB1RSTR_DMA2RST_Msk                  (0x1UL<<RCC_AHB1RSTR_DMA2RST_Pos)                 /*!< 0x00000002 */
#define RCC_AHB1RSTR_DMA2RST                      RCC_AHB1RSTR_DMA2RST_Msk                          
#define RCC_AHB1RSTR_CORDICRST_Pos                (5U)
#define RCC_AHB1RSTR_CORDICRST_Msk                (0x1UL<<RCC_AHB1RSTR_CORDICRST_Pos)               /*!< 0x00000020 */
#define RCC_AHB1RSTR_CORDICRST                    RCC_AHB1RSTR_CORDICRST_Msk                        
#define RCC_AHB1RSTR_CRCRST_Pos                   (6U)
#define RCC_AHB1RSTR_CRCRST_Msk                   (0x1UL<<RCC_AHB1RSTR_CRCRST_Pos)                  /*!< 0x00000040 */
#define RCC_AHB1RSTR_CRCRST                       RCC_AHB1RSTR_CRCRST_Msk                           
#define RCC_AHB1RSTR_ESMCRST_Pos                  (12U)
#define RCC_AHB1RSTR_ESMCRST_Msk                  (0x1UL<<RCC_AHB1RSTR_ESMCRST_Pos)                 /*!< 0x00001000 */
#define RCC_AHB1RSTR_ESMCRST                      RCC_AHB1RSTR_ESMCRST_Msk                          

/********************************* Bit definition for RCC_AHB2RSTR register *****************************************/
#define RCC_AHB2RSTR_IOPARST_Pos                  (2U)
#define RCC_AHB2RSTR_IOPARST_Msk                  (0x1UL<<RCC_AHB2RSTR_IOPARST_Pos)                 /*!< 0x00000004 */
#define RCC_AHB2RSTR_IOPARST                      RCC_AHB2RSTR_IOPARST_Msk                          
#define RCC_AHB2RSTR_IOPBRST_Pos                  (3U)
#define RCC_AHB2RSTR_IOPBRST_Msk                  (0x1UL<<RCC_AHB2RSTR_IOPBRST_Pos)                 /*!< 0x00000008 */
#define RCC_AHB2RSTR_IOPBRST                      RCC_AHB2RSTR_IOPBRST_Msk                          
#define RCC_AHB2RSTR_IOPCRST_Pos                  (4U)
#define RCC_AHB2RSTR_IOPCRST_Msk                  (0x1UL<<RCC_AHB2RSTR_IOPCRST_Pos)                 /*!< 0x00000010 */
#define RCC_AHB2RSTR_IOPCRST                      RCC_AHB2RSTR_IOPCRST_Msk                          
#define RCC_AHB2RSTR_IOPDRST_Pos                  (5U)
#define RCC_AHB2RSTR_IOPDRST_Msk                  (0x1UL<<RCC_AHB2RSTR_IOPDRST_Pos)                 /*!< 0x00000020 */
#define RCC_AHB2RSTR_IOPDRST                      RCC_AHB2RSTR_IOPDRST_Msk                          
#define RCC_AHB2RSTR_IOPERST_Pos                  (6U)
#define RCC_AHB2RSTR_IOPERST_Msk                  (0x1UL<<RCC_AHB2RSTR_IOPERST_Pos)                 /*!< 0x00000040 */
#define RCC_AHB2RSTR_IOPERST                      RCC_AHB2RSTR_IOPERST_Msk                          
#define RCC_AHB2RSTR_IOPFRST_Pos                  (7U)
#define RCC_AHB2RSTR_IOPFRST_Msk                  (0x1UL<<RCC_AHB2RSTR_IOPFRST_Pos)                 /*!< 0x00000080 */
#define RCC_AHB2RSTR_IOPFRST                      RCC_AHB2RSTR_IOPFRST_Msk                          
#define RCC_AHB2RSTR_SDIORST_Pos                  (9U)
#define RCC_AHB2RSTR_SDIORST_Msk                  (0x1UL<<RCC_AHB2RSTR_SDIORST_Pos)                 /*!< 0x00000200 */
#define RCC_AHB2RSTR_SDIORST                      RCC_AHB2RSTR_SDIORST_Msk                          
#define RCC_AHB2RSTR_AESRST_Pos                   (10U)
#define RCC_AHB2RSTR_AESRST_Msk                   (0x1UL<<RCC_AHB2RSTR_AESRST_Pos)                  /*!< 0x00000400 */
#define RCC_AHB2RSTR_AESRST                       RCC_AHB2RSTR_AESRST_Msk                           
#define RCC_AHB2RSTR_USB1RST_Pos                  (13U)
#define RCC_AHB2RSTR_USB1RST_Msk                  (0x1UL<<RCC_AHB2RSTR_USB1RST_Pos)                 /*!< 0x00002000 */
#define RCC_AHB2RSTR_USB1RST                      RCC_AHB2RSTR_USB1RST_Msk                          
#define RCC_AHB2RSTR_USB2RST_Pos                  (14U)
#define RCC_AHB2RSTR_USB2RST_Msk                  (0x1UL<<RCC_AHB2RSTR_USB2RST_Pos)                 /*!< 0x00004000 */
#define RCC_AHB2RSTR_USB2RST                      RCC_AHB2RSTR_USB2RST_Msk                          
#define RCC_AHB2RSTR_ETHRST_Pos                   (15U)
#define RCC_AHB2RSTR_ETHRST_Msk                   (0x1UL<<RCC_AHB2RSTR_ETHRST_Pos)                  /*!< 0x00008000 */
#define RCC_AHB2RSTR_ETHRST                       RCC_AHB2RSTR_ETHRST_Msk                           

/********************************* Bit definition for RCC_APB1RSTR1 register ****************************************/
#define RCC_APB1RSTR1_TIM2RST_Pos                 (0U)
#define RCC_APB1RSTR1_TIM2RST_Msk                 (0x1UL<<RCC_APB1RSTR1_TIM2RST_Pos)                /*!< 0x00000001 */
#define RCC_APB1RSTR1_TIM2RST                     RCC_APB1RSTR1_TIM2RST_Msk                         
#define RCC_APB1RSTR1_TIM3RST_Pos                 (1U)
#define RCC_APB1RSTR1_TIM3RST_Msk                 (0x1UL<<RCC_APB1RSTR1_TIM3RST_Pos)                /*!< 0x00000002 */
#define RCC_APB1RSTR1_TIM3RST                     RCC_APB1RSTR1_TIM3RST_Msk                         
#define RCC_APB1RSTR1_TIM4RST_Pos                 (2U)
#define RCC_APB1RSTR1_TIM4RST_Msk                 (0x1UL<<RCC_APB1RSTR1_TIM4RST_Pos)                /*!< 0x00000004 */
#define RCC_APB1RSTR1_TIM4RST                     RCC_APB1RSTR1_TIM4RST_Msk                         
#define RCC_APB1RSTR1_TIM5RST_Pos                 (3U)
#define RCC_APB1RSTR1_TIM5RST_Msk                 (0x1UL<<RCC_APB1RSTR1_TIM5RST_Pos)                /*!< 0x00000008 */
#define RCC_APB1RSTR1_TIM5RST                     RCC_APB1RSTR1_TIM5RST_Msk                         
#define RCC_APB1RSTR1_TIM6RST_Pos                 (4U)
#define RCC_APB1RSTR1_TIM6RST_Msk                 (0x1UL<<RCC_APB1RSTR1_TIM6RST_Pos)                /*!< 0x00000010 */
#define RCC_APB1RSTR1_TIM6RST                     RCC_APB1RSTR1_TIM6RST_Msk                         
#define RCC_APB1RSTR1_TIM7RST_Pos                 (5U)
#define RCC_APB1RSTR1_TIM7RST_Msk                 (0x1UL<<RCC_APB1RSTR1_TIM7RST_Pos)                /*!< 0x00000020 */
#define RCC_APB1RSTR1_TIM7RST                     RCC_APB1RSTR1_TIM7RST_Msk                         
#define RCC_APB1RSTR1_TIM12RST_Pos                (6U)
#define RCC_APB1RSTR1_TIM12RST_Msk                (0x1UL<<RCC_APB1RSTR1_TIM12RST_Pos)               /*!< 0x00000040 */
#define RCC_APB1RSTR1_TIM12RST                    RCC_APB1RSTR1_TIM12RST_Msk                        
#define RCC_APB1RSTR1_TIM13RST_Pos                (7U)
#define RCC_APB1RSTR1_TIM13RST_Msk                (0x1UL<<RCC_APB1RSTR1_TIM13RST_Pos)               /*!< 0x00000080 */
#define RCC_APB1RSTR1_TIM13RST                    RCC_APB1RSTR1_TIM13RST_Msk                        
#define RCC_APB1RSTR1_TIM14RST_Pos                (8U)
#define RCC_APB1RSTR1_TIM14RST_Msk                (0x1UL<<RCC_APB1RSTR1_TIM14RST_Pos)               /*!< 0x00000100 */
#define RCC_APB1RSTR1_TIM14RST                    RCC_APB1RSTR1_TIM14RST_Msk                        
#define RCC_APB1RSTR1_TIM18RST_Pos                (9U)
#define RCC_APB1RSTR1_TIM18RST_Msk                (0x1UL<<RCC_APB1RSTR1_TIM18RST_Pos)               /*!< 0x00000200 */
#define RCC_APB1RSTR1_TIM18RST                    RCC_APB1RSTR1_TIM18RST_Msk                        
#define RCC_APB1RSTR1_WWDGRST_Pos                 (11U)
#define RCC_APB1RSTR1_WWDGRST_Msk                 (0x1UL<<RCC_APB1RSTR1_WWDGRST_Pos)                /*!< 0x00000800 */
#define RCC_APB1RSTR1_WWDGRST                     RCC_APB1RSTR1_WWDGRST_Msk                         
#define RCC_APB1RSTR1_SPI2RST_Pos                 (14U)
#define RCC_APB1RSTR1_SPI2RST_Msk                 (0x1UL<<RCC_APB1RSTR1_SPI2RST_Pos)                /*!< 0x00004000 */
#define RCC_APB1RSTR1_SPI2RST                     RCC_APB1RSTR1_SPI2RST_Msk                         
#define RCC_APB1RSTR1_SPI3RST_Pos                 (15U)
#define RCC_APB1RSTR1_SPI3RST_Msk                 (0x1UL<<RCC_APB1RSTR1_SPI3RST_Pos)                /*!< 0x00008000 */
#define RCC_APB1RSTR1_SPI3RST                     RCC_APB1RSTR1_SPI3RST_Msk                         
#define RCC_APB1RSTR1_USART2RST_Pos               (17U)
#define RCC_APB1RSTR1_USART2RST_Msk               (0x1UL<<RCC_APB1RSTR1_USART2RST_Pos)              /*!< 0x00020000 */
#define RCC_APB1RSTR1_USART2RST                   RCC_APB1RSTR1_USART2RST_Msk                       
#define RCC_APB1RSTR1_USART3RST_Pos               (18U)
#define RCC_APB1RSTR1_USART3RST_Msk               (0x1UL<<RCC_APB1RSTR1_USART3RST_Pos)              /*!< 0x00040000 */
#define RCC_APB1RSTR1_USART3RST                   RCC_APB1RSTR1_USART3RST_Msk                       
#define RCC_APB1RSTR1_UART1RST_Pos                (19U)
#define RCC_APB1RSTR1_UART1RST_Msk                (0x1UL<<RCC_APB1RSTR1_UART1RST_Pos)               /*!< 0x00080000 */
#define RCC_APB1RSTR1_UART1RST                    RCC_APB1RSTR1_UART1RST_Msk                        
#define RCC_APB1RSTR1_UART2RST_Pos                (20U)
#define RCC_APB1RSTR1_UART2RST_Msk                (0x1UL<<RCC_APB1RSTR1_UART2RST_Pos)               /*!< 0x00100000 */
#define RCC_APB1RSTR1_UART2RST                    RCC_APB1RSTR1_UART2RST_Msk                        
#define RCC_APB1RSTR1_I2C1RST_Pos                 (21U)
#define RCC_APB1RSTR1_I2C1RST_Msk                 (0x1UL<<RCC_APB1RSTR1_I2C1RST_Pos)                /*!< 0x00200000 */
#define RCC_APB1RSTR1_I2C1RST                     RCC_APB1RSTR1_I2C1RST_Msk                         
#define RCC_APB1RSTR1_I2C2RST_Pos                 (22U)
#define RCC_APB1RSTR1_I2C2RST_Msk                 (0x1UL<<RCC_APB1RSTR1_I2C2RST_Pos)                /*!< 0x00400000 */
#define RCC_APB1RSTR1_I2C2RST                     RCC_APB1RSTR1_I2C2RST_Msk                         
#define RCC_APB1RSTR1_UART3RST_Pos                (23U)
#define RCC_APB1RSTR1_UART3RST_Msk                (0x1UL<<RCC_APB1RSTR1_UART3RST_Pos)               /*!< 0x00800000 */
#define RCC_APB1RSTR1_UART3RST                    RCC_APB1RSTR1_UART3RST_Msk                        
#define RCC_APB1RSTR1_CAN1RST_Pos                 (25U)
#define RCC_APB1RSTR1_CAN1RST_Msk                 (0x1UL<<RCC_APB1RSTR1_CAN1RST_Pos)                /*!< 0x02000000 */
#define RCC_APB1RSTR1_CAN1RST                     RCC_APB1RSTR1_CAN1RST_Msk                         
#define RCC_APB1RSTR1_CAN2RST_Pos                 (26U)
#define RCC_APB1RSTR1_CAN2RST_Msk                 (0x1UL<<RCC_APB1RSTR1_CAN2RST_Pos)                /*!< 0x04000000 */
#define RCC_APB1RSTR1_CAN2RST                     RCC_APB1RSTR1_CAN2RST_Msk                         
#define RCC_APB1RSTR1_PWRRST_Pos                  (28U)
#define RCC_APB1RSTR1_PWRRST_Msk                  (0x1UL<<RCC_APB1RSTR1_PWRRST_Pos)                 /*!< 0x10000000 */
#define RCC_APB1RSTR1_PWRRST                      RCC_APB1RSTR1_PWRRST_Msk                          
#define RCC_APB1RSTR1_DACRST_Pos                  (29U)
#define RCC_APB1RSTR1_DACRST_Msk                  (0x1UL<<RCC_APB1RSTR1_DACRST_Pos)                 /*!< 0x20000000 */
#define RCC_APB1RSTR1_DACRST                      RCC_APB1RSTR1_DACRST_Msk                          
#define RCC_APB1RSTR1_I2C3RST_Pos                 (30U)
#define RCC_APB1RSTR1_I2C3RST_Msk                 (0x1UL<<RCC_APB1RSTR1_I2C3RST_Pos)                /*!< 0x40000000 */
#define RCC_APB1RSTR1_I2C3RST                     RCC_APB1RSTR1_I2C3RST_Msk                         
#define RCC_APB1RSTR1_CTCRST_Pos                  (31U)
#define RCC_APB1RSTR1_CTCRST_Msk                  (0x1UL<<RCC_APB1RSTR1_CTCRST_Pos)                 /*!< 0x80000000 */
#define RCC_APB1RSTR1_CTCRST                      RCC_APB1RSTR1_CTCRST_Msk                          

/********************************* Bit definition for RCC_APB1RSTR2 register ****************************************/
#define RCC_APB1RSTR2_LPTIM1RST_Pos               (0U)
#define RCC_APB1RSTR2_LPTIM1RST_Msk               (0x1UL<<RCC_APB1RSTR2_LPTIM1RST_Pos)              /*!< 0x00000001 */
#define RCC_APB1RSTR2_LPTIM1RST                   RCC_APB1RSTR2_LPTIM1RST_Msk                       
#define RCC_APB1RSTR2_LPUART1RST_Pos              (1U)
#define RCC_APB1RSTR2_LPUART1RST_Msk              (0x1UL<<RCC_APB1RSTR2_LPUART1RST_Pos)             /*!< 0x00000002 */
#define RCC_APB1RSTR2_LPUART1RST                  RCC_APB1RSTR2_LPUART1RST_Msk                      
#define RCC_APB1RSTR2_I2C4RST_Pos                 (2U)
#define RCC_APB1RSTR2_I2C4RST_Msk                 (0x1UL<<RCC_APB1RSTR2_I2C4RST_Pos)                /*!< 0x00000004 */
#define RCC_APB1RSTR2_I2C4RST                     RCC_APB1RSTR2_I2C4RST_Msk                         

/********************************* Bit definition for RCC_APB2RSTR register *****************************************/
#define RCC_APB2RSTR_SYSCFGRST_Pos                (0U)
#define RCC_APB2RSTR_SYSCFGRST_Msk                (0x1UL<<RCC_APB2RSTR_SYSCFGRST_Pos)               /*!< 0x00000001 */
#define RCC_APB2RSTR_SYSCFGRST                    RCC_APB2RSTR_SYSCFGRST_Msk                        
#define RCC_APB2RSTR_ADC1RST_Pos                  (9U)
#define RCC_APB2RSTR_ADC1RST_Msk                  (0x1UL<<RCC_APB2RSTR_ADC1RST_Pos)                 /*!< 0x00000200 */
#define RCC_APB2RSTR_ADC1RST                      RCC_APB2RSTR_ADC1RST_Msk                          
#define RCC_APB2RSTR_ADC2RST_Pos                  (10U)
#define RCC_APB2RSTR_ADC2RST_Msk                  (0x1UL<<RCC_APB2RSTR_ADC2RST_Pos)                 /*!< 0x00000400 */
#define RCC_APB2RSTR_ADC2RST                      RCC_APB2RSTR_ADC2RST_Msk                          
#define RCC_APB2RSTR_TIM1RST_Pos                  (11U)
#define RCC_APB2RSTR_TIM1RST_Msk                  (0x1UL<<RCC_APB2RSTR_TIM1RST_Pos)                 /*!< 0x00000800 */
#define RCC_APB2RSTR_TIM1RST                      RCC_APB2RSTR_TIM1RST_Msk                          
#define RCC_APB2RSTR_SPI1RST_Pos                  (12U)
#define RCC_APB2RSTR_SPI1RST_Msk                  (0x1UL<<RCC_APB2RSTR_SPI1RST_Pos)                 /*!< 0x00001000 */
#define RCC_APB2RSTR_SPI1RST                      RCC_APB2RSTR_SPI1RST_Msk                          
#define RCC_APB2RSTR_TIM8RST_Pos                  (13U)
#define RCC_APB2RSTR_TIM8RST_Msk                  (0x1UL<<RCC_APB2RSTR_TIM8RST_Pos)                 /*!< 0x00002000 */
#define RCC_APB2RSTR_TIM8RST                      RCC_APB2RSTR_TIM8RST_Msk                          
#define RCC_APB2RSTR_USART1RST_Pos                (14U)
#define RCC_APB2RSTR_USART1RST_Msk                (0x1UL<<RCC_APB2RSTR_USART1RST_Pos)               /*!< 0x00004000 */
#define RCC_APB2RSTR_USART1RST                    RCC_APB2RSTR_USART1RST_Msk                        
#define RCC_APB2RSTR_ADC3RST_Pos                  (15U)
#define RCC_APB2RSTR_ADC3RST_Msk                  (0x1UL<<RCC_APB2RSTR_ADC3RST_Pos)                 /*!< 0x00008000 */
#define RCC_APB2RSTR_ADC3RST                      RCC_APB2RSTR_ADC3RST_Msk                          
#define RCC_APB2RSTR_TIM9RST_Pos                  (19U)
#define RCC_APB2RSTR_TIM9RST_Msk                  (0x1UL<<RCC_APB2RSTR_TIM9RST_Pos)                 /*!< 0x00080000 */
#define RCC_APB2RSTR_TIM9RST                      RCC_APB2RSTR_TIM9RST_Msk                          
#define RCC_APB2RSTR_TIM10RST_Pos                 (20U)
#define RCC_APB2RSTR_TIM10RST_Msk                 (0x1UL<<RCC_APB2RSTR_TIM10RST_Pos)                /*!< 0x00100000 */
#define RCC_APB2RSTR_TIM10RST                     RCC_APB2RSTR_TIM10RST_Msk                         
#define RCC_APB2RSTR_TIM11RST_Pos                 (21U)
#define RCC_APB2RSTR_TIM11RST_Msk                 (0x1UL<<RCC_APB2RSTR_TIM11RST_Pos)                /*!< 0x00200000 */
#define RCC_APB2RSTR_TIM11RST                     RCC_APB2RSTR_TIM11RST_Msk                         
#define RCC_APB2RSTR_TIM15RST_Pos                 (22U)
#define RCC_APB2RSTR_TIM15RST_Msk                 (0x1UL<<RCC_APB2RSTR_TIM15RST_Pos)                /*!< 0x00400000 */
#define RCC_APB2RSTR_TIM15RST                     RCC_APB2RSTR_TIM15RST_Msk                         
#define RCC_APB2RSTR_TIM16RST_Pos                 (23U)
#define RCC_APB2RSTR_TIM16RST_Msk                 (0x1UL<<RCC_APB2RSTR_TIM16RST_Pos)                /*!< 0x00800000 */
#define RCC_APB2RSTR_TIM16RST                     RCC_APB2RSTR_TIM16RST_Msk                         
#define RCC_APB2RSTR_TIM17RST_Pos                 (24U)
#define RCC_APB2RSTR_TIM17RST_Msk                 (0x1UL<<RCC_APB2RSTR_TIM17RST_Pos)                /*!< 0x01000000 */
#define RCC_APB2RSTR_TIM17RST                     RCC_APB2RSTR_TIM17RST_Msk                         
#define RCC_APB2RSTR_TIM19RST_Pos                 (25U)
#define RCC_APB2RSTR_TIM19RST_Msk                 (0x1UL<<RCC_APB2RSTR_TIM19RST_Pos)                /*!< 0x02000000 */
#define RCC_APB2RSTR_TIM19RST                     RCC_APB2RSTR_TIM19RST_Msk                         
#define RCC_APB2RSTR_RNGRST_Pos                   (26U)
#define RCC_APB2RSTR_RNGRST_Msk                   (0x1UL<<RCC_APB2RSTR_RNGRST_Pos)                  /*!< 0x04000000 */
#define RCC_APB2RSTR_RNGRST                       RCC_APB2RSTR_RNGRST_Msk                           
#define RCC_APB2RSTR_COMPRST_Pos                  (27U)
#define RCC_APB2RSTR_COMPRST_Msk                  (0x1UL<<RCC_APB2RSTR_COMPRST_Pos)                 /*!< 0x08000000 */
#define RCC_APB2RSTR_COMPRST                      RCC_APB2RSTR_COMPRST_Msk                          
#define RCC_APB2RSTR_OPARST_Pos                   (28U)
#define RCC_APB2RSTR_OPARST_Msk                   (0x1UL<<RCC_APB2RSTR_OPARST_Pos)                /*!< 0x10000000 */
#define RCC_APB2RSTR_OPARST                       RCC_APB2RSTR_OPARST_Msk                         
#define RCC_APB2RSTR_LCDCRST_Pos                  (29U)
#define RCC_APB2RSTR_LCDCRST_Msk                  (0x1UL<<RCC_APB2RSTR_LCDCRST_Pos)                 /*!< 0x20000000 */
#define RCC_APB2RSTR_LCDCRST                      RCC_APB2RSTR_LCDCRST_Msk                          

/********************************* Bit definition for RCC_AHB1ENR register ******************************************/
#define RCC_AHB1ENR_DMA1EN_Pos                    (0U)
#define RCC_AHB1ENR_DMA1EN_Msk                    (0x1UL<<RCC_AHB1ENR_DMA1EN_Pos)                   /*!< 0x00000001 */
#define RCC_AHB1ENR_DMA1EN                        RCC_AHB1ENR_DMA1EN_Msk                            
#define RCC_AHB1ENR_DMA2EN_Pos                    (1U)
#define RCC_AHB1ENR_DMA2EN_Msk                    (0x1UL<<RCC_AHB1ENR_DMA2EN_Pos)                   /*!< 0x00000002 */
#define RCC_AHB1ENR_DMA2EN                        RCC_AHB1ENR_DMA2EN_Msk                            
#define RCC_AHB1ENR_SRAMEN_Pos                    (2U)
#define RCC_AHB1ENR_SRAMEN_Msk                    (0x1UL<<RCC_AHB1ENR_SRAMEN_Pos)                   /*!< 0x00000004 */
#define RCC_AHB1ENR_SRAMEN                        RCC_AHB1ENR_SRAMEN_Msk                            
#define RCC_AHB1ENR_FMCEN_Pos                     (4U)
#define RCC_AHB1ENR_FMCEN_Msk                     (0x1UL<<RCC_AHB1ENR_FMCEN_Pos)                    /*!< 0x00000010 */
#define RCC_AHB1ENR_FMCEN                         RCC_AHB1ENR_FMCEN_Msk                             
#define RCC_AHB1ENR_CORDICEN_Pos                  (5U)
#define RCC_AHB1ENR_CORDICEN_Msk                  (0x1UL<<RCC_AHB1ENR_CORDICEN_Pos)                 /*!< 0x00000020 */
#define RCC_AHB1ENR_CORDICEN                      RCC_AHB1ENR_CORDICEN_Msk                          
#define RCC_AHB1ENR_CRCEN_Pos                     (6U)
#define RCC_AHB1ENR_CRCEN_Msk                     (0x1UL<<RCC_AHB1ENR_CRCEN_Pos)                    /*!< 0x00000040 */
#define RCC_AHB1ENR_CRCEN                         RCC_AHB1ENR_CRCEN_Msk                             
#define RCC_AHB1ENR_ESMCEN_Pos                    (12U)
#define RCC_AHB1ENR_ESMCEN_Msk                    (0x1UL<<RCC_AHB1ENR_ESMCEN_Pos)                   /*!< 0x00001000 */
#define RCC_AHB1ENR_ESMCEN                        RCC_AHB1ENR_ESMCEN_Msk                            

/********************************* Bit definition for RCC_AHB2ENR register ******************************************/
#define RCC_AHB2ENR_IOPAEN_Pos                    (2U)
#define RCC_AHB2ENR_IOPAEN_Msk                    (0x1UL<<RCC_AHB2ENR_IOPAEN_Pos)                   /*!< 0x00000004 */
#define RCC_AHB2ENR_IOPAEN                        RCC_AHB2ENR_IOPAEN_Msk                            
#define RCC_AHB2ENR_IOPBEN_Pos                    (3U)
#define RCC_AHB2ENR_IOPBEN_Msk                    (0x1UL<<RCC_AHB2ENR_IOPBEN_Pos)                   /*!< 0x00000008 */
#define RCC_AHB2ENR_IOPBEN                        RCC_AHB2ENR_IOPBEN_Msk                            
#define RCC_AHB2ENR_IOPCEN_Pos                    (4U)
#define RCC_AHB2ENR_IOPCEN_Msk                    (0x1UL<<RCC_AHB2ENR_IOPCEN_Pos)                   /*!< 0x00000010 */
#define RCC_AHB2ENR_IOPCEN                        RCC_AHB2ENR_IOPCEN_Msk                            
#define RCC_AHB2ENR_IOPDEN_Pos                    (5U)
#define RCC_AHB2ENR_IOPDEN_Msk                    (0x1UL<<RCC_AHB2ENR_IOPDEN_Pos)                   /*!< 0x00000020 */
#define RCC_AHB2ENR_IOPDEN                        RCC_AHB2ENR_IOPDEN_Msk                            
#define RCC_AHB2ENR_IOPEEN_Pos                    (6U)
#define RCC_AHB2ENR_IOPEEN_Msk                    (0x1UL<<RCC_AHB2ENR_IOPEEN_Pos)                   /*!< 0x00000040 */
#define RCC_AHB2ENR_IOPEEN                        RCC_AHB2ENR_IOPEEN_Msk                            
#define RCC_AHB2ENR_IOPFEN_Pos                    (7U)
#define RCC_AHB2ENR_IOPFEN_Msk                    (0x1UL<<RCC_AHB2ENR_IOPFEN_Pos)                   /*!< 0x00000080 */
#define RCC_AHB2ENR_IOPFEN                        RCC_AHB2ENR_IOPFEN_Msk                            
#define RCC_AHB2ENR_SDIOEN_Pos                    (9U)
#define RCC_AHB2ENR_SDIOEN_Msk                    (0x1UL<<RCC_AHB2ENR_SDIOEN_Pos)                   /*!< 0x00000200 */
#define RCC_AHB2ENR_SDIOEN                        RCC_AHB2ENR_SDIOEN_Msk                            
#define RCC_AHB2ENR_AESEN_Pos                     (10U)
#define RCC_AHB2ENR_AESEN_Msk                     (0x1UL<<RCC_AHB2ENR_AESEN_Pos)                    /*!< 0x00000400 */
#define RCC_AHB2ENR_AESEN                         RCC_AHB2ENR_AESEN_Msk                             
#define RCC_AHB2ENR_USB1EN_Pos                    (13U)
#define RCC_AHB2ENR_USB1EN_Msk                    (0x1UL<<RCC_AHB2ENR_USB1EN_Pos)                   /*!< 0x00002000 */
#define RCC_AHB2ENR_USB1EN                        RCC_AHB2ENR_USB1EN_Msk                            
#define RCC_AHB2ENR_USB2EN_Pos                    (14U)
#define RCC_AHB2ENR_USB2EN_Msk                    (0x1UL<<RCC_AHB2ENR_USB2EN_Pos)                   /*!< 0x00004000 */
#define RCC_AHB2ENR_USB2EN                        RCC_AHB2ENR_USB2EN_Msk                            
#define RCC_AHB2ENR_ETHEN_Pos                     (15U)
#define RCC_AHB2ENR_ETHEN_Msk                     (0x1UL<<RCC_AHB2ENR_ETHEN_Pos)                    /*!< 0x00008000 */
#define RCC_AHB2ENR_ETHEN                         RCC_AHB2ENR_ETHEN_Msk                             
#define RCC_AHB2ENR_ETHTXEN_Pos                   (16U)
#define RCC_AHB2ENR_ETHTXEN_Msk                   (0x1UL<<RCC_AHB2ENR_ETHTXEN_Pos)                  /*!< 0x00010000 */
#define RCC_AHB2ENR_ETHTXEN                       RCC_AHB2ENR_ETHTXEN_Msk                           
#define RCC_AHB2ENR_ETHRXEN_Pos                   (17U)
#define RCC_AHB2ENR_ETHRXEN_Msk                   (0x1UL<<RCC_AHB2ENR_ETHRXEN_Pos)                  /*!< 0x00020000 */
#define RCC_AHB2ENR_ETHRXEN                       RCC_AHB2ENR_ETHRXEN_Msk                           

/********************************* Bit definition for RCC_APB1ENR1 register *****************************************/
#define RCC_APB1ENR1_TIM2EN_Pos                   (0U)
#define RCC_APB1ENR1_TIM2EN_Msk                   (0x1UL<<RCC_APB1ENR1_TIM2EN_Pos)                  /*!< 0x00000001 */
#define RCC_APB1ENR1_TIM2EN                       RCC_APB1ENR1_TIM2EN_Msk                           
#define RCC_APB1ENR1_TIM3EN_Pos                   (1U)
#define RCC_APB1ENR1_TIM3EN_Msk                   (0x1UL<<RCC_APB1ENR1_TIM3EN_Pos)                  /*!< 0x00000002 */
#define RCC_APB1ENR1_TIM3EN                       RCC_APB1ENR1_TIM3EN_Msk                           
#define RCC_APB1ENR1_TIM4EN_Pos                   (2U)
#define RCC_APB1ENR1_TIM4EN_Msk                   (0x1UL<<RCC_APB1ENR1_TIM4EN_Pos)                  /*!< 0x00000004 */
#define RCC_APB1ENR1_TIM4EN                       RCC_APB1ENR1_TIM4EN_Msk                           
#define RCC_APB1ENR1_TIM5EN_Pos                   (3U)
#define RCC_APB1ENR1_TIM5EN_Msk                   (0x1UL<<RCC_APB1ENR1_TIM5EN_Pos)                  /*!< 0x00000008 */
#define RCC_APB1ENR1_TIM5EN                       RCC_APB1ENR1_TIM5EN_Msk                           
#define RCC_APB1ENR1_TIM6EN_Pos                   (4U)
#define RCC_APB1ENR1_TIM6EN_Msk                   (0x1UL<<RCC_APB1ENR1_TIM6EN_Pos)                  /*!< 0x00000010 */
#define RCC_APB1ENR1_TIM6EN                       RCC_APB1ENR1_TIM6EN_Msk                           
#define RCC_APB1ENR1_TIM7EN_Pos                   (5U)
#define RCC_APB1ENR1_TIM7EN_Msk                   (0x1UL<<RCC_APB1ENR1_TIM7EN_Pos)                  /*!< 0x00000020 */
#define RCC_APB1ENR1_TIM7EN                       RCC_APB1ENR1_TIM7EN_Msk                           
#define RCC_APB1ENR1_TIM12EN_Pos                  (6U)
#define RCC_APB1ENR1_TIM12EN_Msk                  (0x1UL<<RCC_APB1ENR1_TIM12EN_Pos)                 /*!< 0x00000040 */
#define RCC_APB1ENR1_TIM12EN                      RCC_APB1ENR1_TIM12EN_Msk                          
#define RCC_APB1ENR1_TIM13EN_Pos                  (7U)
#define RCC_APB1ENR1_TIM13EN_Msk                  (0x1UL<<RCC_APB1ENR1_TIM13EN_Pos)                 /*!< 0x00000080 */
#define RCC_APB1ENR1_TIM13EN                      RCC_APB1ENR1_TIM13EN_Msk                          
#define RCC_APB1ENR1_TIM14EN_Pos                  (8U)
#define RCC_APB1ENR1_TIM14EN_Msk                  (0x1UL<<RCC_APB1ENR1_TIM14EN_Pos)                 /*!< 0x00000100 */
#define RCC_APB1ENR1_TIM14EN                      RCC_APB1ENR1_TIM14EN_Msk                          
#define RCC_APB1ENR1_TIM18EN_Pos                  (9U)
#define RCC_APB1ENR1_TIM18EN_Msk                  (0x1UL<<RCC_APB1ENR1_TIM18EN_Pos)                 /*!< 0x00000200 */
#define RCC_APB1ENR1_TIM18EN                      RCC_APB1ENR1_TIM18EN_Msk                          
#define RCC_APB1ENR1_WWDGEN_Pos                   (11U)
#define RCC_APB1ENR1_WWDGEN_Msk                   (0x1UL<<RCC_APB1ENR1_WWDGEN_Pos)                  /*!< 0x00000800 */
#define RCC_APB1ENR1_WWDGEN                       RCC_APB1ENR1_WWDGEN_Msk                           
#define RCC_APB1ENR1_SPI2EN_Pos                   (14U)
#define RCC_APB1ENR1_SPI2EN_Msk                   (0x1UL<<RCC_APB1ENR1_SPI2EN_Pos)                  /*!< 0x00004000 */
#define RCC_APB1ENR1_SPI2EN                       RCC_APB1ENR1_SPI2EN_Msk                           
#define RCC_APB1ENR1_SPI3EN_Pos                   (15U)
#define RCC_APB1ENR1_SPI3EN_Msk                   (0x1UL<<RCC_APB1ENR1_SPI3EN_Pos)                  /*!< 0x00008000 */
#define RCC_APB1ENR1_SPI3EN                       RCC_APB1ENR1_SPI3EN_Msk                           
#define RCC_APB1ENR1_USART2EN_Pos                 (17U)
#define RCC_APB1ENR1_USART2EN_Msk                 (0x1UL<<RCC_APB1ENR1_USART2EN_Pos)                /*!< 0x00020000 */
#define RCC_APB1ENR1_USART2EN                     RCC_APB1ENR1_USART2EN_Msk                         
#define RCC_APB1ENR1_USART3EN_Pos                 (18U)
#define RCC_APB1ENR1_USART3EN_Msk                 (0x1UL<<RCC_APB1ENR1_USART3EN_Pos)                /*!< 0x00040000 */
#define RCC_APB1ENR1_USART3EN                     RCC_APB1ENR1_USART3EN_Msk                         
#define RCC_APB1ENR1_UART1EN_Pos                  (19U)
#define RCC_APB1ENR1_UART1EN_Msk                  (0x1UL<<RCC_APB1ENR1_UART1EN_Pos)                 /*!< 0x00080000 */
#define RCC_APB1ENR1_UART1EN                      RCC_APB1ENR1_UART1EN_Msk                          
#define RCC_APB1ENR1_UART2EN_Pos                  (20U)
#define RCC_APB1ENR1_UART2EN_Msk                  (0x1UL<<RCC_APB1ENR1_UART2EN_Pos)                 /*!< 0x00100000 */
#define RCC_APB1ENR1_UART2EN                      RCC_APB1ENR1_UART2EN_Msk                          
#define RCC_APB1ENR1_I2C1EN_Pos                   (21U)
#define RCC_APB1ENR1_I2C1EN_Msk                   (0x1UL<<RCC_APB1ENR1_I2C1EN_Pos)                  /*!< 0x00200000 */
#define RCC_APB1ENR1_I2C1EN                       RCC_APB1ENR1_I2C1EN_Msk                           
#define RCC_APB1ENR1_I2C2EN_Pos                   (22U)
#define RCC_APB1ENR1_I2C2EN_Msk                   (0x1UL<<RCC_APB1ENR1_I2C2EN_Pos)                  /*!< 0x00400000 */
#define RCC_APB1ENR1_I2C2EN                       RCC_APB1ENR1_I2C2EN_Msk                           
#define RCC_APB1ENR1_UART3EN_Pos                  (23U)
#define RCC_APB1ENR1_UART3EN_Msk                  (0x1UL<<RCC_APB1ENR1_UART3EN_Pos)                 /*!< 0x00800000 */
#define RCC_APB1ENR1_UART3EN                      RCC_APB1ENR1_UART3EN_Msk                          
#define RCC_APB1ENR1_CAN1EN_Pos                   (25U)
#define RCC_APB1ENR1_CAN1EN_Msk                   (0x1UL<<RCC_APB1ENR1_CAN1EN_Pos)                  /*!< 0x02000000 */
#define RCC_APB1ENR1_CAN1EN                       RCC_APB1ENR1_CAN1EN_Msk                           
#define RCC_APB1ENR1_CAN2EN_Pos                   (26U)
#define RCC_APB1ENR1_CAN2EN_Msk                   (0x1UL<<RCC_APB1ENR1_CAN2EN_Pos)                  /*!< 0x04000000 */
#define RCC_APB1ENR1_CAN2EN                       RCC_APB1ENR1_CAN2EN_Msk                           
#define RCC_APB1ENR1_BKPEN_Pos                    (27U)
#define RCC_APB1ENR1_BKPEN_Msk                    (0x1UL<<RCC_APB1ENR1_BKPEN_Pos)                   /*!< 0x08000000 */
#define RCC_APB1ENR1_BKPEN                        RCC_APB1ENR1_BKPEN_Msk                            
#define RCC_APB1ENR1_PWREN_Pos                    (28U)
#define RCC_APB1ENR1_PWREN_Msk                    (0x1UL<<RCC_APB1ENR1_PWREN_Pos)                   /*!< 0x10000000 */
#define RCC_APB1ENR1_PWREN                        RCC_APB1ENR1_PWREN_Msk                            
#define RCC_APB1ENR1_DACEN_Pos                    (29U)
#define RCC_APB1ENR1_DACEN_Msk                    (0x1UL<<RCC_APB1ENR1_DACEN_Pos)                   /*!< 0x20000000 */
#define RCC_APB1ENR1_DACEN                        RCC_APB1ENR1_DACEN_Msk                            
#define RCC_APB1ENR1_I2C3EN_Pos                   (30U)
#define RCC_APB1ENR1_I2C3EN_Msk                   (0x1UL<<RCC_APB1ENR1_I2C3EN_Pos)                  /*!< 0x40000000 */
#define RCC_APB1ENR1_I2C3EN                       RCC_APB1ENR1_I2C3EN_Msk                           
#define RCC_APB1ENR1_CTCEN_Pos                    (31U)
#define RCC_APB1ENR1_CTCEN_Msk                    (0x1UL<<RCC_APB1ENR1_CTCEN_Pos)                   /*!< 0x80000000 */
#define RCC_APB1ENR1_CTCEN                        RCC_APB1ENR1_CTCEN_Msk                            

/********************************* Bit definition for RCC_APB1ENR2 register *****************************************/
#define RCC_APB1ENR2_LPTIM1EN_Pos                 (0U)
#define RCC_APB1ENR2_LPTIM1EN_Msk                 (0x1UL<<RCC_APB1ENR2_LPTIM1EN_Pos)                /*!< 0x00000001 */
#define RCC_APB1ENR2_LPTIM1EN                     RCC_APB1ENR2_LPTIM1EN_Msk                         
#define RCC_APB1ENR2_LPUART1EN_Pos                (1U)
#define RCC_APB1ENR2_LPUART1EN_Msk                (0x1UL<<RCC_APB1ENR2_LPUART1EN_Pos)               /*!< 0x00000002 */
#define RCC_APB1ENR2_LPUART1EN                    RCC_APB1ENR2_LPUART1EN_Msk                        
#define RCC_APB1ENR2_I2C4EN_Pos                   (2U)
#define RCC_APB1ENR2_I2C4EN_Msk                   (0x1UL<<RCC_APB1ENR2_I2C4EN_Pos)                  /*!< 0x00000004 */
#define RCC_APB1ENR2_I2C4EN                       RCC_APB1ENR2_I2C4EN_Msk                           

/********************************* Bit definition for RCC_APB2ENR register ******************************************/
#define RCC_APB2ENR_SYSCFGEN_Pos                  (0U)
#define RCC_APB2ENR_SYSCFGEN_Msk                  (0x1UL<<RCC_APB2ENR_SYSCFGEN_Pos)                 /*!< 0x00000001 */
#define RCC_APB2ENR_SYSCFGEN                      RCC_APB2ENR_SYSCFGEN_Msk                          
#define RCC_APB2ENR_ADC1EN_Pos                    (9U)
#define RCC_APB2ENR_ADC1EN_Msk                    (0x1UL<<RCC_APB2ENR_ADC1EN_Pos)                   /*!< 0x00000200 */
#define RCC_APB2ENR_ADC1EN                        RCC_APB2ENR_ADC1EN_Msk                            
#define RCC_APB2ENR_ADC2EN_Pos                    (10U)
#define RCC_APB2ENR_ADC2EN_Msk                    (0x1UL<<RCC_APB2ENR_ADC2EN_Pos)                   /*!< 0x00000400 */
#define RCC_APB2ENR_ADC2EN                        RCC_APB2ENR_ADC2EN_Msk                            
#define RCC_APB2ENR_TIM1EN_Pos                    (11U)
#define RCC_APB2ENR_TIM1EN_Msk                    (0x1UL<<RCC_APB2ENR_TIM1EN_Pos)                   /*!< 0x00000800 */
#define RCC_APB2ENR_TIM1EN                        RCC_APB2ENR_TIM1EN_Msk                            
#define RCC_APB2ENR_SPI1EN_Pos                    (12U)
#define RCC_APB2ENR_SPI1EN_Msk                    (0x1UL<<RCC_APB2ENR_SPI1EN_Pos)                   /*!< 0x00001000 */
#define RCC_APB2ENR_SPI1EN                        RCC_APB2ENR_SPI1EN_Msk                            
#define RCC_APB2ENR_TIM8EN_Pos                    (13U)
#define RCC_APB2ENR_TIM8EN_Msk                    (0x1UL<<RCC_APB2ENR_TIM8EN_Pos)                   /*!< 0x00002000 */
#define RCC_APB2ENR_TIM8EN                        RCC_APB2ENR_TIM8EN_Msk                            
#define RCC_APB2ENR_USART1EN_Pos                  (14U)
#define RCC_APB2ENR_USART1EN_Msk                  (0x1UL<<RCC_APB2ENR_USART1EN_Pos)                 /*!< 0x00004000 */
#define RCC_APB2ENR_USART1EN                      RCC_APB2ENR_USART1EN_Msk                          
#define RCC_APB2ENR_ADC3EN_Pos                    (15U)
#define RCC_APB2ENR_ADC3EN_Msk                    (0x1UL<<RCC_APB2ENR_ADC3EN_Pos)                   /*!< 0x00008000 */
#define RCC_APB2ENR_ADC3EN                        RCC_APB2ENR_ADC3EN_Msk                            
#define RCC_APB2ENR_TIM9EN_Pos                    (19U)
#define RCC_APB2ENR_TIM9EN_Msk                    (0x1UL<<RCC_APB2ENR_TIM9EN_Pos)                   /*!< 0x00080000 */
#define RCC_APB2ENR_TIM9EN                        RCC_APB2ENR_TIM9EN_Msk                            
#define RCC_APB2ENR_TIM10EN_Pos                   (20U)
#define RCC_APB2ENR_TIM10EN_Msk                   (0x1UL<<RCC_APB2ENR_TIM10EN_Pos)                  /*!< 0x00100000 */
#define RCC_APB2ENR_TIM10EN                       RCC_APB2ENR_TIM10EN_Msk                           
#define RCC_APB2ENR_TIM11EN_Pos                   (21U)
#define RCC_APB2ENR_TIM11EN_Msk                   (0x1UL<<RCC_APB2ENR_TIM11EN_Pos)                  /*!< 0x00200000 */
#define RCC_APB2ENR_TIM11EN                       RCC_APB2ENR_TIM11EN_Msk                           
#define RCC_APB2ENR_TIM15EN_Pos                   (22U)
#define RCC_APB2ENR_TIM15EN_Msk                   (0x1UL<<RCC_APB2ENR_TIM15EN_Pos)                  /*!< 0x00400000 */
#define RCC_APB2ENR_TIM15EN                       RCC_APB2ENR_TIM15EN_Msk                           
#define RCC_APB2ENR_TIM16EN_Pos                   (23U)
#define RCC_APB2ENR_TIM16EN_Msk                   (0x1UL<<RCC_APB2ENR_TIM16EN_Pos)                  /*!< 0x00800000 */
#define RCC_APB2ENR_TIM16EN                       RCC_APB2ENR_TIM16EN_Msk                           
#define RCC_APB2ENR_TIM17EN_Pos                   (24U)
#define RCC_APB2ENR_TIM17EN_Msk                   (0x1UL<<RCC_APB2ENR_TIM17EN_Pos)                  /*!< 0x01000000 */
#define RCC_APB2ENR_TIM17EN                       RCC_APB2ENR_TIM17EN_Msk                           
#define RCC_APB2ENR_TIM19EN_Pos                   (25U)
#define RCC_APB2ENR_TIM19EN_Msk                   (0x1UL<<RCC_APB2ENR_TIM19EN_Pos)                  /*!< 0x02000000 */
#define RCC_APB2ENR_TIM19EN                       RCC_APB2ENR_TIM19EN_Msk                           
#define RCC_APB2ENR_RNGEN_Pos                     (26U)
#define RCC_APB2ENR_RNGEN_Msk                     (0x1UL<<RCC_APB2ENR_RNGEN_Pos)                    /*!< 0x04000000 */
#define RCC_APB2ENR_RNGEN                         RCC_APB2ENR_RNGEN_Msk                             
#define RCC_APB2ENR_COMPEN_Pos                    (27U)
#define RCC_APB2ENR_COMPEN_Msk                    (0x1UL<<RCC_APB2ENR_COMPEN_Pos)                   /*!< 0x08000000 */
#define RCC_APB2ENR_COMPEN                        RCC_APB2ENR_COMPEN_Msk                            
#define RCC_APB2ENR_OPAEN_Pos                     (28U)
#define RCC_APB2ENR_OPAEN_Msk                     (0x1UL<<RCC_APB2ENR_OPAEN_Pos)                  /*!< 0x10000000 */
#define RCC_APB2ENR_OPAEN                         RCC_APB2ENR_OPAEN_Msk                           
#define RCC_APB2ENR_LCDCEN_Pos                    (29U)
#define RCC_APB2ENR_LCDCEN_Msk                    (0x1UL<<RCC_APB2ENR_LCDCEN_Pos)                   /*!< 0x20000000 */
#define RCC_APB2ENR_LCDCEN                        RCC_APB2ENR_LCDCEN_Msk                            

/********************************* Bit definition for RCC_CCIPR register ********************************************/
#define RCC_CCIPR_USART1SEL_Pos                   (0U)
#define RCC_CCIPR_USART1SEL_Msk                   (0x3UL<<RCC_CCIPR_USART1SEL_Pos)                  /*!< 0x00000003 */
#define RCC_CCIPR_USART1SEL                       RCC_CCIPR_USART1SEL_Msk
#define RCC_CCIPR_USART1SEL_0                     (0x1UL<<RCC_CCIPR_USART1SEL_Pos)                  /*!< 0x00000001 */
#define RCC_CCIPR_USART1SEL_1                     (0x2UL<<RCC_CCIPR_USART1SEL_Pos)                  /*!< 0x00000002 */
#define RCC_CCIPR_USART2SEL_Pos                   (2U)
#define RCC_CCIPR_USART2SEL_Msk                   (0x3UL<<RCC_CCIPR_USART2SEL_Pos)                  /*!< 0x0000000C */
#define RCC_CCIPR_USART2SEL                       RCC_CCIPR_USART2SEL_Msk
#define RCC_CCIPR_USART2SEL_0                     (0x1UL<<RCC_CCIPR_USART2SEL_Pos)                  /*!< 0x00000004 */
#define RCC_CCIPR_USART2SEL_1                     (0x2UL<<RCC_CCIPR_USART2SEL_Pos)                  /*!< 0x00000008 */
#define RCC_CCIPR_USART3SEL_Pos                   (4U)
#define RCC_CCIPR_USART3SEL_Msk                   (0x3UL<<RCC_CCIPR_USART3SEL_Pos)                  /*!< 0x00000030 */
#define RCC_CCIPR_USART3SEL                       RCC_CCIPR_USART3SEL_Msk
#define RCC_CCIPR_USART3SEL_0                     (0x1UL<<RCC_CCIPR_USART3SEL_Pos)                  /*!< 0x00000010 */
#define RCC_CCIPR_USART3SEL_1                     (0x2UL<<RCC_CCIPR_USART3SEL_Pos)                  /*!< 0x00000020 */
#define RCC_CCIPR_LPUART1SEL_Pos                  (12U)
#define RCC_CCIPR_LPUART1SEL_Msk                  (0x3UL<<RCC_CCIPR_LPUART1SEL_Pos)                 /*!< 0x00003000 */
#define RCC_CCIPR_LPUART1SEL                      RCC_CCIPR_LPUART1SEL_Msk
#define RCC_CCIPR_LPUART1SEL_0                    (0x1UL<<RCC_CCIPR_LPUART1SEL_Pos)                 /*!< 0x00001000 */
#define RCC_CCIPR_LPUART1SEL_1                    (0x2UL<<RCC_CCIPR_LPUART1SEL_Pos)                 /*!< 0x00002000 */
#define RCC_CCIPR_LPTIM1SEL_Pos                   (14U)
#define RCC_CCIPR_LPTIM1SEL_Msk                   (0x3UL<<RCC_CCIPR_LPTIM1SEL_Pos)                  /*!< 0x0000C000 */
#define RCC_CCIPR_LPTIM1SEL                       RCC_CCIPR_LPTIM1SEL_Msk
#define RCC_CCIPR_LPTIM1SEL_0                     (0x1UL<<RCC_CCIPR_LPTIM1SEL_Pos)                  /*!< 0x00004000 */
#define RCC_CCIPR_LPTIM1SEL_1                     (0x2UL<<RCC_CCIPR_LPTIM1SEL_Pos)                  /*!< 0x00008000 */
#define RCC_CCIPR_COMP1SEL_Pos                    (16U)
#define RCC_CCIPR_COMP1SEL_Msk                    (0x3UL<<RCC_CCIPR_COMP1SEL_Pos)                   /*!< 0x00030000 */
#define RCC_CCIPR_COMP1SEL                        RCC_CCIPR_COMP1SEL_Msk
#define RCC_CCIPR_COMP1SEL_0                      (0x1UL<<RCC_CCIPR_COMP1SEL_Pos)                   /*!< 0x00010000 */
#define RCC_CCIPR_COMP1SEL_1                      (0x2UL<<RCC_CCIPR_COMP1SEL_Pos)                   /*!< 0x00020000 */
#define RCC_CCIPR_COMP2SEL_Pos                    (18U)
#define RCC_CCIPR_COMP2SEL_Msk                    (0x3UL<<RCC_CCIPR_COMP2SEL_Pos)                   /*!< 0x000C0000 */
#define RCC_CCIPR_COMP2SEL                        RCC_CCIPR_COMP2SEL_Msk
#define RCC_CCIPR_COMP2SEL_0                      (0x1UL<<RCC_CCIPR_COMP2SEL_Pos)                   /*!< 0x00040000 */
#define RCC_CCIPR_COMP2SEL_1                      (0x2UL<<RCC_CCIPR_COMP2SEL_Pos)                   /*!< 0x00080000 */
#define RCC_CCIPR_COMP3SEL_Pos                    (20U)
#define RCC_CCIPR_COMP3SEL_Msk                    (0x3UL<<RCC_CCIPR_COMP3SEL_Pos)                   /*!< 0x00300000 */
#define RCC_CCIPR_COMP3SEL                        RCC_CCIPR_COMP3SEL_Msk
#define RCC_CCIPR_COMP3SEL_0                      (0x1UL<<RCC_CCIPR_COMP3SEL_Pos)                   /*!< 0x00100000 */
#define RCC_CCIPR_COMP3SEL_1                      (0x2UL<<RCC_CCIPR_COMP3SEL_Pos)                   /*!< 0x00200000 */
#define RCC_CCIPR_COMP4SEL_Pos                    (22U)
#define RCC_CCIPR_COMP4SEL_Msk                    (0x3UL<<RCC_CCIPR_COMP4SEL_Pos)                   /*!< 0x00C00000 */
#define RCC_CCIPR_COMP4SEL                        RCC_CCIPR_COMP4SEL_Msk
#define RCC_CCIPR_COMP4SEL_0                      (0x1UL<<RCC_CCIPR_COMP4SEL_Pos)                   /*!< 0x00400000 */
#define RCC_CCIPR_COMP4SEL_1                      (0x2UL<<RCC_CCIPR_COMP4SEL_Pos)                   /*!< 0x00800000 */

/********************************* Bit definition for RCC_BDCR register *********************************************/
#define RCC_BDCR_LSEON_Pos                        (0U)
#define RCC_BDCR_LSEON_Msk                        (0x1UL<<RCC_BDCR_LSEON_Pos)                       /*!< 0x00000001 */
#define RCC_BDCR_LSEON                            RCC_BDCR_LSEON_Msk                                
#define RCC_BDCR_LSERDY_Pos                       (1U)
#define RCC_BDCR_LSERDY_Msk                       (0x1UL<<RCC_BDCR_LSERDY_Pos)                      /*!< 0x00000002 */
#define RCC_BDCR_LSERDY                           RCC_BDCR_LSERDY_Msk                               
#define RCC_BDCR_LSEBYP_Pos                       (2U)
#define RCC_BDCR_LSEBYP_Msk                       (0x1UL<<RCC_BDCR_LSEBYP_Pos)                      /*!< 0x00000004 */
#define RCC_BDCR_LSEBYP                           RCC_BDCR_LSEBYP_Msk                               
#define RCC_BDCR_LSEDRV_Pos                       (3U)
#define RCC_BDCR_LSEDRV_Msk                       (0x3UL<<RCC_BDCR_LSEDRV_Pos)                      /*!< 0x00000018 */
#define RCC_BDCR_LSEDRV                           RCC_BDCR_LSEDRV_Msk
#define RCC_BDCR_LSEDRV_0                         (0x1UL<<RCC_BDCR_LSEDRV_Pos)                      /*!< 0x00000008 */
#define RCC_BDCR_LSEDRV_1                         (0x2UL<<RCC_BDCR_LSEDRV_Pos)                      /*!< 0x00000010 */
#define RCC_BDCR_RTCSEL_Pos                       (8U)
#define RCC_BDCR_RTCSEL_Msk                       (0x3UL<<RCC_BDCR_RTCSEL_Pos)                      /*!< 0x00000300 */
#define RCC_BDCR_RTCSEL                           RCC_BDCR_RTCSEL_Msk
#define RCC_BDCR_RTCSEL_0                         (0x1UL<<RCC_BDCR_RTCSEL_Pos)                      /*!< 0x00000100 */
#define RCC_BDCR_RTCSEL_1                         (0x2UL<<RCC_BDCR_RTCSEL_Pos)                      /*!< 0x00000200 */
#define RCC_BDCR_RTCSEL_NOCLOCK                   0x00000000U                                       /*!< No clock */
#define RCC_BDCR_RTCSEL_LSE                       0x00000100U                                       /*!< LSE oscillator clock used as RTC clock */
#define RCC_BDCR_RTCSEL_LSI                       0x00000200U                                       /*!< LSI oscillator clock used as RTC clock */
#define RCC_BDCR_RTCSEL_HSE_DIV128                0x00000300U 

#define RCC_BDCR_RTCEN_Pos                        (15U)
#define RCC_BDCR_RTCEN_Msk                        (0x1UL<<RCC_BDCR_RTCEN_Pos)                       /*!< 0x00008000 */
#define RCC_BDCR_RTCEN                            RCC_BDCR_RTCEN_Msk                                
#define RCC_BDCR_BDRST_Pos                        (16U)
#define RCC_BDCR_BDRST_Msk                        (0x1UL<<RCC_BDCR_BDRST_Pos)                       /*!< 0x00010000 */
#define RCC_BDCR_BDRST                            RCC_BDCR_BDRST_Msk                                

/********************************* Bit definition for RCC_CSR register **********************************************/
#define RCC_CSR_LSION_Pos                         (0U)
#define RCC_CSR_LSION_Msk                         (0x1UL<<RCC_CSR_LSION_Pos)                        /*!< 0x00000001 */
#define RCC_CSR_LSION                             RCC_CSR_LSION_Msk                                 
#define RCC_CSR_LSIRDY_Pos                        (1U)
#define RCC_CSR_LSIRDY_Msk                        (0x1UL<<RCC_CSR_LSIRDY_Pos)                       /*!< 0x00000002 */
#define RCC_CSR_LSIRDY                            RCC_CSR_LSIRDY_Msk                                
#define RCC_CSR_RMVF_Pos                          (24U)
#define RCC_CSR_RMVF_Msk                          (0x1UL<<RCC_CSR_RMVF_Pos)                         /*!< 0x01000000 */
#define RCC_CSR_RMVF                              RCC_CSR_RMVF_Msk                                  
#define RCC_CSR_OBLRSTF_Pos                       (25U)
#define RCC_CSR_OBLRSTF_Msk                       (0x1UL<<RCC_CSR_OBLRSTF_Pos)                      /*!< 0x02000000 */
#define RCC_CSR_OBLRSTF                           RCC_CSR_OBLRSTF_Msk                               
#define RCC_CSR_PINRSTF_Pos                       (26U)
#define RCC_CSR_PINRSTF_Msk                       (0x1UL<<RCC_CSR_PINRSTF_Pos)                      /*!< 0x04000000 */
#define RCC_CSR_PINRSTF                           RCC_CSR_PINRSTF_Msk                               
#define RCC_CSR_PWRRSTF_Pos                       (27U)
#define RCC_CSR_PWRRSTF_Msk                       (0x1UL<<RCC_CSR_PWRRSTF_Pos)                      /*!< 0x08000000 */
#define RCC_CSR_PWRRSTF                           RCC_CSR_PWRRSTF_Msk                               
#define RCC_CSR_SFTRSTF_Pos                       (28U)
#define RCC_CSR_SFTRSTF_Msk                       (0x1UL<<RCC_CSR_SFTRSTF_Pos)                      /*!< 0x10000000 */
#define RCC_CSR_SFTRSTF                           RCC_CSR_SFTRSTF_Msk                               
#define RCC_CSR_IWDGRSTF_Pos                      (29U)
#define RCC_CSR_IWDGRSTF_Msk                      (0x1UL<<RCC_CSR_IWDGRSTF_Pos)                     /*!< 0x20000000 */
#define RCC_CSR_IWDGRSTF                          RCC_CSR_IWDGRSTF_Msk                              
#define RCC_CSR_WWDGRSTF_Pos                      (30U)
#define RCC_CSR_WWDGRSTF_Msk                      (0x1UL<<RCC_CSR_WWDGRSTF_Pos)                     /*!< 0x40000000 */
#define RCC_CSR_WWDGRSTF                          RCC_CSR_WWDGRSTF_Msk                              
#define RCC_CSR_LPWRRSTF_Pos                      (31U)
#define RCC_CSR_LPWRRSTF_Msk                      (0x1UL<<RCC_CSR_LPWRRSTF_Pos)                     /*!< 0x80000000 */
#define RCC_CSR_LPWRRSTF                          RCC_CSR_LPWRRSTF_Msk                              

/********************************* Bit definition for RCC_CFGR1 register ********************************************/
#define RCC_CFGR1_MCOPRE_Pos                      (0U)
#define RCC_CFGR1_MCOPRE_Msk                      (0x7UL<<RCC_CFGR1_MCOPRE_Pos)                     /*!< 0x00000007 */
#define RCC_CFGR1_MCOPRE                          RCC_CFGR1_MCOPRE_Msk
#define RCC_CFGR1_MCOPRE_0                        (0x1UL<<RCC_CFGR1_MCOPRE_Pos)                     /*!< 0x00000001 */
#define RCC_CFGR1_MCOPRE_1                        (0x2UL<<RCC_CFGR1_MCOPRE_Pos)                     /*!< 0x00000002 */
#define RCC_CFGR1_MCOPRE_2                        (0x4UL<<RCC_CFGR1_MCOPRE_Pos)                     /*!< 0x00000004 */
#define RCC_CFGR1_HSI48ON_Pos                     (4U)
#define RCC_CFGR1_HSI48ON_Msk                     (0x1UL<<RCC_CFGR1_HSI48ON_Pos)                    /*!< 0x00000010 */
#define RCC_CFGR1_HSI48ON                         RCC_CFGR1_HSI48ON_Msk                             
#define RCC_CFGR1_HSI48RDY_Pos                    (5U)
#define RCC_CFGR1_HSI48RDY_Msk                    (0x1UL<<RCC_CFGR1_HSI48RDY_Pos)                   /*!< 0x00000020 */
#define RCC_CFGR1_HSI48RDY                        RCC_CFGR1_HSI48RDY_Msk                            
#define RCC_CFGR1_HSI48TRIM_Pos                   (9U)
#define RCC_CFGR1_HSI48TRIM_Msk                   (0x7FUL<<RCC_CFGR1_HSI48TRIM_Pos)                 /*!< 0x0000FE00 */
#define RCC_CFGR1_HSI48TRIM                       RCC_CFGR1_HSI48TRIM_Msk
#define RCC_CFGR1_HSI48CAL_Pos                    (16U)
#define RCC_CFGR1_HSI48CAL_Msk                    (0x1FFFUL<<RCC_CFGR1_HSI48CAL_Pos)                /*!< 0x1FFF0000 */
#define RCC_CFGR1_HSI48CAL                        RCC_CFGR1_HSI48CAL_Msk
#define RCC_CFGR1_PVDSEL_Pos                      (29U)
#define RCC_CFGR1_PVDSEL_Msk                      (0x3UL<<RCC_CFGR1_PVDSEL_Pos)                     /*!< 0x60000000 */
#define RCC_CFGR1_PVDSEL                          RCC_CFGR1_PVDSEL_Msk
#define RCC_CFGR1_PVDSEL_0                        (0x1UL<<RCC_CFGR1_PVDSEL_Pos)                     /*!< 0x20000000 */
#define RCC_CFGR1_PVDSEL_1                        (0x2UL<<RCC_CFGR1_PVDSEL_Pos)                     /*!< 0x40000000 */
#define RCC_CFGR1_USBSELHSI48_Pos                 (31U)
#define RCC_CFGR1_USBSELHSI48_Msk                 (0x1UL<<RCC_CFGR1_USBSELHSI48_Pos)                /*!< 0x80000000 */
#define RCC_CFGR1_USBSELHSI48                     RCC_CFGR1_USBSELHSI48_Msk                         

/********************************* Bit definition for RCC_CFGR2 register ********************************************/
#define RCC_CFGR2_CANCKSEL_Pos                    (0U)
#define RCC_CFGR2_CANCKSEL_Msk                    (0xFUL<<RCC_CFGR2_CANCKSEL_Pos)                   /*!< 0x0000000F */
#define RCC_CFGR2_CANCKSEL                        RCC_CFGR2_CANCKSEL_Msk
#define RCC_CFGR2_CANCKSEL_0                      (0x1UL<<RCC_CFGR2_CANCKSEL_Pos)                   /*!< 0x00000001 */
#define RCC_CFGR2_CANCKSEL_1                      (0x2UL<<RCC_CFGR2_CANCKSEL_Pos)                   /*!< 0x00000002 */
#define RCC_CFGR2_CANCKSEL_2                      (0x4UL<<RCC_CFGR2_CANCKSEL_Pos)                   /*!< 0x00000004 */
#define RCC_CFGR2_CANCKSEL_3                      (0x8UL<<RCC_CFGR2_CANCKSEL_Pos)                   /*!< 0x00000008 */
#define RCC_CFGR2_HSEDRV_Pos                      (8U)
#define RCC_CFGR2_HSEDRV_Msk                      (0x3UL<<RCC_CFGR2_HSEDRV_Pos)                     /*!< 0x00000300 */
#define RCC_CFGR2_HSEDRV                          RCC_CFGR2_HSEDRV_Msk
#define RCC_CFGR2_HSEDRV_0                        (0x1UL<<RCC_CFGR2_HSEDRV_Pos)                     /*!< 0x00000100 */
#define RCC_CFGR2_HSEDRV_1                        (0x2UL<<RCC_CFGR2_HSEDRV_Pos)                     /*!< 0x00000200 */

/********************************* Bit definition for RCC_CFGR3 register ********************************************/
#define RCC_CFGR3_PLLSRC_Pos                      (0U)
#define RCC_CFGR3_PLLSRC_Msk                      (0x1UL<<RCC_CFGR3_PLLSRC_Pos)                     /*!< 0x00000001 */
#define RCC_CFGR3_PLLSRC                          RCC_CFGR3_PLLSRC_Msk                              
#define RCC_CFGR3_PLLXTPRE_Pos                    (1U)
#define RCC_CFGR3_PLLXTPRE_Msk                    (0x1UL<<RCC_CFGR3_PLLXTPRE_Pos)                   /*!< 0x00000002 */
#define RCC_CFGR3_PLLXTPRE                        RCC_CFGR3_PLLXTPRE_Msk                            
#define RCC_CFGR3_PLLPREDIV_Pos                   (2U)
#define RCC_CFGR3_PLLPREDIV_Msk                   (0x3UL<<RCC_CFGR3_PLLPREDIV_Pos)                  /*!< 0x0000000C */
#define RCC_CFGR3_PLLPREDIV                       RCC_CFGR3_PLLPREDIV_Msk
#define RCC_CFGR3_PLLPREDIV_0                     (0x1UL<<RCC_CFGR3_PLLPREDIV_Pos)                  /*!< 0x00000004 */
#define RCC_CFGR3_PLLPREDIV_1                     (0x2UL<<RCC_CFGR3_PLLPREDIV_Pos)                  /*!< 0x00000008 */

#define RCC_CFGR3_PLLPREDIV1                      0x00000000U
#define RCC_CFGR3_PLLPREDIV2                      0x00000004U
#define RCC_CFGR3_PLLPREDIV4                      0x00000008U
#define RCC_CFGR3_PLLPREDIV8                      0x0000000CU

#define RCC_CFGR3_PLLPOSTDIV_Pos                  (4U)
#define RCC_CFGR3_PLLPOSTDIV_Msk                  (0x3UL<<RCC_CFGR3_PLLPOSTDIV_Pos)                 /*!< 0x00000030 */
#define RCC_CFGR3_PLLPOSTDIV                      RCC_CFGR3_PLLPOSTDIV_Msk
#define RCC_CFGR3_PLLPOSTDIV_0                    (0x1UL<<RCC_CFGR3_PLLPOSTDIV_Pos)                 /*!< 0x00000010 */
#define RCC_CFGR3_PLLPOSTDIV_1                    (0x2UL<<RCC_CFGR3_PLLPOSTDIV_Pos)                 /*!< 0x00000020 */

#define RCC_CFGR3_PLLPOSTDIV1                     0x00000000U
#define RCC_CFGR3_PLLPOSTDIV2                     0x00000010U
#define RCC_CFGR3_PLLPOSTDIV4                     0x00000020U
#define RCC_CFGR3_PLLPOSTDIV8                     0x00000030U

#define RCC_CFGR3_PLLFBDIV_Pos                    (8U)
#define RCC_CFGR3_PLLFBDIV_Msk                    (0xFFUL<<RCC_CFGR3_PLLFBDIV_Pos)                  /*!< 0x0000FF00 */
#define RCC_CFGR3_PLLFBDIV                        RCC_CFGR3_PLLFBDIV_Msk

/********************************************************************************************************************/
/********************************* RTC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for RTC_CRH register **********************************************/
#define RTC_CRH_SECIE_Pos                         (0U)
#define RTC_CRH_SECIE_Msk                         (0x1UL<<RTC_CRH_SECIE_Pos)                        /*!< 0x00000001 */
#define RTC_CRH_SECIE                             RTC_CRH_SECIE_Msk                                 /*!< Second interrupt enable */
#define RTC_CRH_ALRIE_Pos                         (1U)
#define RTC_CRH_ALRIE_Msk                         (0x1UL<<RTC_CRH_ALRIE_Pos)                        /*!< 0x00000002 */
#define RTC_CRH_ALRIE                             RTC_CRH_ALRIE_Msk                                 /*!< Alarm interrupt enable */
#define RTC_CRH_OWIE_Pos                          (2U)
#define RTC_CRH_OWIE_Msk                          (0x1UL<<RTC_CRH_OWIE_Pos)                         /*!< 0x00000004 */
#define RTC_CRH_OWIE                              RTC_CRH_OWIE_Msk                                  /*!< Overflow interrupt enable */

/********************************* Bit definition for RTC_CRL register **********************************************/
#define RTC_CRL_SECF_Pos                          (0U)
#define RTC_CRL_SECF_Msk                          (0x1UL<<RTC_CRL_SECF_Pos)                         /*!< 0x00000001 */
#define RTC_CRL_SECF                              RTC_CRL_SECF_Msk                                  /*!< Second flag */
#define RTC_CRL_ALRF_Pos                          (1U)
#define RTC_CRL_ALRF_Msk                          (0x1UL<<RTC_CRL_ALRF_Pos)                         /*!< 0x00000002 */
#define RTC_CRL_ALRF                              RTC_CRL_ALRF_Msk                                  /*!< Alarm flag */
#define RTC_CRL_OWF_Pos                           (2U)
#define RTC_CRL_OWF_Msk                           (0x1UL<<RTC_CRL_OWF_Pos)                          /*!< 0x00000004 */
#define RTC_CRL_OWF                               RTC_CRL_OWF_Msk                                   /*!< Overflow flag */
#define RTC_CRL_RSF_Pos                           (3U)
#define RTC_CRL_RSF_Msk                           (0x1UL<<RTC_CRL_RSF_Pos)                          /*!< 0x00000008 */
#define RTC_CRL_RSF                               RTC_CRL_RSF_Msk                                   /*!< Registers synchronized flag */
#define RTC_CRL_CNF_Pos                           (4U)
#define RTC_CRL_CNF_Msk                           (0x1UL<<RTC_CRL_CNF_Pos)                          /*!< 0x00000010 */
#define RTC_CRL_CNF                               RTC_CRL_CNF_Msk                                   /*!< Configuration flag */
#define RTC_CRL_RTOFF_Pos                         (5U)
#define RTC_CRL_RTOFF_Msk                         (0x1UL<<RTC_CRL_RTOFF_Pos)                        /*!< 0x00000020 */
#define RTC_CRL_RTOFF                             RTC_CRL_RTOFF_Msk                                 /*!< RTC operation OFF */

/********************************* Bit definition for RTC_PRLH register *********************************************/
#define RTC_PRLH_PRL_Pos                          (0U)
#define RTC_PRLH_PRL_Msk                          (0xFUL<<RTC_PRLH_PRL_Pos)                         /*!< 0x0000000F */
#define RTC_PRLH_PRL                              RTC_PRLH_PRL_Msk                                  /*!< RTC pre-scaler reload value high part */
#define RTC_PRLH_PRL_0                            (0x1UL<<RTC_PRLH_PRL_Pos)                         /*!< 0x00000001 */
#define RTC_PRLH_PRL_1                            (0x2UL<<RTC_PRLH_PRL_Pos)                         /*!< 0x00000002 */
#define RTC_PRLH_PRL_2                            (0x4UL<<RTC_PRLH_PRL_Pos)                         /*!< 0x00000004 */
#define RTC_PRLH_PRL_3                            (0x8UL<<RTC_PRLH_PRL_Pos)                         /*!< 0x00000008 */

/********************************* Bit definition for RTC_PRLL register *********************************************/
#define RTC_PRLL_PRL_Pos                          (0U)
#define RTC_PRLL_PRL_Msk                          (0xFFFFUL<<RTC_PRLL_PRL_Pos)                      /*!< 0x0000FFFF */
#define RTC_PRLL_PRL                              RTC_PRLL_PRL_Msk                                  /*!< RTC pre-scaler reload value low part */

/********************************* Bit definition for RTC_DIVH register *********************************************/
#define RTC_DIVH_RTC_DIV_Pos                      (0U)
#define RTC_DIVH_RTC_DIV_Msk                      (0xFUL<<RTC_DIVH_RTC_DIV_Pos)                     /*!< 0x0000000F */
#define RTC_DIVH_RTC_DIV                          RTC_DIVH_RTC_DIV_Msk                              /*!< RTC clock divider high part */
#define RTC_DIVH_RTC_DIV_0                        (0x1UL<<RTC_DIVH_RTC_DIV_Pos)                     /*!< 0x00000001 */
#define RTC_DIVH_RTC_DIV_1                        (0x2UL<<RTC_DIVH_RTC_DIV_Pos)                     /*!< 0x00000002 */
#define RTC_DIVH_RTC_DIV_2                        (0x4UL<<RTC_DIVH_RTC_DIV_Pos)                     /*!< 0x00000004 */
#define RTC_DIVH_RTC_DIV_3                        (0x8UL<<RTC_DIVH_RTC_DIV_Pos)                     /*!< 0x00000008 */

/********************************* Bit definition for RTC_DIVL register *********************************************/
#define RTC_DIVL_RTC_DIV_Pos                      (0U)
#define RTC_DIVL_RTC_DIV_Msk                      (0xFFFFUL<<RTC_DIVL_RTC_DIV_Pos)                  /*!< 0x0000FFFF */
#define RTC_DIVL_RTC_DIV                          RTC_DIVL_RTC_DIV_Msk                              /*!< RTC clock divider low part */

/********************************* Bit definition for RTC_CNTH register *********************************************/
#define RTC_CNTH_RTC_CNT_Pos                      (0U)
#define RTC_CNTH_RTC_CNT_Msk                      (0xFFFFUL<<RTC_CNTH_RTC_CNT_Pos)                  /*!< 0x0000FFFF */
#define RTC_CNTH_RTC_CNT                          RTC_CNTH_RTC_CNT_Msk                              /*!< RTC counter high part */

/********************************* Bit definition for RTC_CNTL register *********************************************/
#define RTC_CNTL_RTC_CNT_Pos                      (0U)
#define RTC_CNTL_RTC_CNT_Msk                      (0xFFFFUL<<RTC_CNTL_RTC_CNT_Pos)                  /*!< 0x0000FFFF */
#define RTC_CNTL_RTC_CNT                          RTC_CNTL_RTC_CNT_Msk                              /*!< RTC counter high part */

/********************************* Bit definition for RTC_ALRH register *********************************************/
#define RTC_ALRH_RTC_ALR_Pos                      (0U)
#define RTC_ALRH_RTC_ALR_Msk                      (0xFFFFUL<<RTC_ALRH_RTC_ALR_Pos)                  /*!< 0x0000FFFF */
#define RTC_ALRH_RTC_ALR                          RTC_ALRH_RTC_ALR_Msk                              /*!< RTC alarm high part */

/********************************* Bit definition for RTC_ALRL register *********************************************/
#define RTC_ALRL_RTC_ALR_Pos                      (0U)
#define RTC_ALRL_RTC_ALR_Msk                      (0xFFFFUL<<RTC_ALRL_RTC_ALR_Pos)                  /*!< 0x0000FFFF */
#define RTC_ALRL_RTC_ALR                          RTC_ALRL_RTC_ALR_Msk                              /*!< RTC alarm low part */

/********************************************************************************************************************/
/********************************* RNG ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for RNG_CR register ***********************************************/
#define RNG_CR_RBG0_EN_Pos                        (0U)
#define RNG_CR_RBG0_EN_Msk                        (0x1UL<<RNG_CR_RBG0_EN_Pos)                       /*!< 0x00000001 */
#define RNG_CR_RBG0_EN                            RNG_CR_RBG0_EN_Msk                                
#define RNG_CR_RBG1_EN_Pos                        (1U)
#define RNG_CR_RBG1_EN_Msk                        (0x1UL<<RNG_CR_RBG1_EN_Pos)                       /*!< 0x00000002 */
#define RNG_CR_RBG1_EN                            RNG_CR_RBG1_EN_Msk                                
#define RNG_CR_IE_Pos                             (2U)
#define RNG_CR_IE_Msk                             (0x1UL<<RNG_CR_IE_Pos)                            /*!< 0x00000004 */
#define RNG_CR_IE                                 RNG_CR_IE_Msk                                     

/********************************* Bit definition for RNG_LFSR register *********************************************/
#define RNG_LFSR_RNG_LFSR_Pos                     (0U)
#define RNG_LFSR_RNG_LFSR_Msk                     (0xFFFFFFFFUL<<RNG_LFSR_RNG_LFSR_Pos)             /*!< 0xFFFFFFFF */
#define RNG_LFSR_RNG_LFSR                         RNG_LFSR_RNG_LFSR_Msk

/********************************* Bit definition for RNG_SR register ***********************************************/
#define RNG_SR_DATA_RDY_Pos                       (0U)
#define RNG_SR_DATA_RDY_Msk                       (0x1UL<<RNG_SR_DATA_RDY_Pos)                      /*!< 0x00000001 */
#define RNG_SR_DATA_RDY                           RNG_SR_DATA_RDY_Msk                               

/********************************* Bit definition for RNG_DR register ***********************************************/
#define RNG_DR_RNG_DATA_Pos                       (0U)
#define RNG_DR_RNG_DATA_Msk                       (0xFFFFFFFFUL<<RNG_DR_RNG_DATA_Pos)               /*!< 0xFFFFFFFF */
#define RNG_DR_RNG_DATA                           RNG_DR_RNG_DATA_Msk

/*********************  Bits Define For Peripheral SDIO  *********************/
/*!< SDIO_POWER */
#define SDIO_POWER_PWRCTRL_Pos                    (0U)
#define SDIO_POWER_PWRCTRL_Msk                    (0x1UL << SDIO_POWER_PWRCTRL_Pos)                 /*!< 0x00000001 */
#define SDIO_POWER_PWRCTRL                        SDIO_POWER_PWRCTRL_Msk                            /*!< desc PWRCTRL */

/*!< SDIO_CLKCR */
#define SDIO_CLKCR_CLKDIV_Pos                     (0U)
#define SDIO_CLKCR_CLKDIV_Msk                     (0xFFUL << SDIO_CLKCR_CLKDIV_Pos)                 /*!< 0x000000FF */
#define SDIO_CLKCR_CLKDIV                         SDIO_CLKCR_CLKDIV_Msk                             /*!< CLKDIV[7:0] bits (desc CLKDIV) */
#define SDIO_CLKCR_CLKEN_Pos                      (8U)
#define SDIO_CLKCR_CLKEN_Msk                      (0x1UL << SDIO_CLKCR_CLKEN_Pos)                   /*!< 0x00000100 */
#define SDIO_CLKCR_CLKEN                          SDIO_CLKCR_CLKEN_Msk                              /*!< desc CLKEN */
#define SDIO_CLKCR_PWRSAV_Pos                     (9U)
#define SDIO_CLKCR_PWRSAV_Msk                     (0x1UL << SDIO_CLKCR_PWRSAV_Pos)                  /*!< 0x00000200 */
#define SDIO_CLKCR_PWRSAV                         SDIO_CLKCR_PWRSAV_Msk                             /*!< desc PWRSAV */
#define SDIO_CLKCR_WIDBUS_Pos                     (10U)
#define SDIO_CLKCR_WIDBUS_Msk                     (0x3UL << SDIO_CLKCR_WIDBUS_Pos)                  /*!< 0x00000C00 */
#define SDIO_CLKCR_WIDBUS                         SDIO_CLKCR_WIDBUS_Msk                             /*!< WIDBUS[11:10] bits (desc WIDBUS) */
#define SDIO_CLKCR_WIDBUS_0                       (0x1UL << SDIO_CLKCR_WIDBUS_Pos)                  /*!< 0x00000400 */
#define SDIO_CLKCR_WIDBUS_1                       (0x2UL << SDIO_CLKCR_WIDBUS_Pos)                  /*!< 0x00000800 */

#define SDIO_CLKCR_SMPCLKSEL_Pos                  (12U)
#define SDIO_CLKCR_SMPCLKSEL_Msk                  (0x1UL << SDIO_CLKCR_SMPCLKSEL_Pos)               /*!< 0x00001000 */
#define SDIO_CLKCR_SMPCLKSEL                      SDIO_CLKCR_SMPCLKSEL_Msk                          /*!< desc SMPCLKSEL */
#define SDIO_CLKCR_SMPEN_Pos                      (13U)
#define SDIO_CLKCR_SMPEN_Msk                      (0x1UL << SDIO_CLKCR_SMPEN_Pos)                   /*!< 0x00002000 */
#define SDIO_CLKCR_SMPEN                          SDIO_CLKCR_SMPEN_Msk                              /*!< desc SMPEN */
#define SDIO_CLKCR_CKSEL_Pos                      (14U)
#define SDIO_CLKCR_CKSEL_Msk                      (0x1UL << SDIO_CLKCR_CKSEL_Pos)                   /*!< 0x00004000 */
#define SDIO_CLKCR_CKSEL                          SDIO_CLKCR_CKSEL_Msk                              /*!< desc CKSEL */

/*!< SDIO_ARG */
#define SDIO_ARG_CMDARG_Pos                       (0U)
#define SDIO_ARG_CMDARG_Msk                       (0xFFFFFFFFUL << SDIO_ARG_CMDARG_Pos)             /*!< 0xFFFFFFFF */
#define SDIO_ARG_CMDARG                           SDIO_ARG_CMDARG_Msk                               /*!< CMDARG[31:0] bits (desc CMDARG) */

/*!< SDIO_CMD */
#define SDIO_CMD_CMDINDEX_Pos                     (0U)
#define SDIO_CMD_CMDINDEX_Msk                     (0x3FUL << SDIO_CMD_CMDINDEX_Pos)                 /*!< 0x0000003F */
#define SDIO_CMD_CMDINDEX                         SDIO_CMD_CMDINDEX_Msk                             /*!< CMDINDEX[5:0] bits (desc CMDINDEX) */
#define SDIO_CMD_CMDINDEX_0                       (0x1UL << SDIO_CMD_CMDINDEX_Pos)                  /*!< 0x00000001 */
#define SDIO_CMD_CMDINDEX_1                       (0x2UL << SDIO_CMD_CMDINDEX_Pos)                  /*!< 0x00000002 */
#define SDIO_CMD_CMDINDEX_2                       (0x4UL << SDIO_CMD_CMDINDEX_Pos)                  /*!< 0x00000004 */
#define SDIO_CMD_CMDINDEX_3                       (0x8UL << SDIO_CMD_CMDINDEX_Pos)                  /*!< 0x00000008 */
#define SDIO_CMD_CMDINDEX_4                       (0x10UL << SDIO_CMD_CMDINDEX_Pos)                 /*!< 0x00000010 */
#define SDIO_CMD_CMDINDEX_5                       (0x20UL << SDIO_CMD_CMDINDEX_Pos)                 /*!< 0x00000020 */

#define SDIO_CMD_WAITRESP_Pos                     (6U)
#define SDIO_CMD_WAITRESP_Msk                     (0x1UL << SDIO_CMD_WAITRESP_Pos)                  /*!< 0x00000040 */
#define SDIO_CMD_WAITRESP                         SDIO_CMD_WAITRESP_Msk                             /*!< desc WAITRESP */
#define SDIO_CMD_RESPLEN_Pos                      (7U)
#define SDIO_CMD_RESPLEN_Msk                      (0x1UL << SDIO_CMD_RESPLEN_Pos)                   /*!< 0x00000080 */
#define SDIO_CMD_RESPLEN                          SDIO_CMD_RESPLEN_Msk                              /*!< desc RESPLEN */
#define SDIO_CMD_CHECKRESPCRC_Pos                 (8U)
#define SDIO_CMD_CHECKRESPCRC_Msk                 (0x1UL << SDIO_CMD_CHECKRESPCRC_Pos)              /*!< 0x00000100 */
#define SDIO_CMD_CHECKRESPCRC                     SDIO_CMD_CHECKRESPCRC_Msk                         /*!< desc CHECKRESPCRC */
#define SDIO_CMD_DEXPECT_Pos                      (9U)
#define SDIO_CMD_DEXPECT_Msk                      (0x1UL << SDIO_CMD_DEXPECT_Pos)                   /*!< 0x00000200 */
#define SDIO_CMD_DEXPECT                          SDIO_CMD_DEXPECT_Msk                              /*!< desc DEXPECT */
#define SDIO_CMD_DIR_Pos                          (10U)
#define SDIO_CMD_DIR_Msk                          (0x1UL << SDIO_CMD_DIR_Pos)                       /*!< 0x00000400 */
#define SDIO_CMD_DIR                              SDIO_CMD_DIR_Msk                                  /*!< desc DIR */
#define SDIO_CMD_DTMODE_Pos                       (11U)
#define SDIO_CMD_DTMODE_Msk                       (0x1UL << SDIO_CMD_DTMODE_Pos)                    /*!< 0x00000800 */
#define SDIO_CMD_DTMODE                           SDIO_CMD_DTMODE_Msk                               /*!< desc DTMODE */
#define SDIO_CMD_AUTOSTOP_Pos                     (12U)
#define SDIO_CMD_AUTOSTOP_Msk                     (0x1UL << SDIO_CMD_AUTOSTOP_Pos)                  /*!< 0x00001000 */
#define SDIO_CMD_AUTOSTOP                         SDIO_CMD_AUTOSTOP_Msk                             /*!< desc AUTOSTOP */
#define SDIO_CMD_WAITPEND_Pos                     (13U)
#define SDIO_CMD_WAITPEND_Msk                     (0x1UL << SDIO_CMD_WAITPEND_Pos)                  /*!< 0x00002000 */
#define SDIO_CMD_WAITPEND                         SDIO_CMD_WAITPEND_Msk                             /*!< desc WAITPEND */
#define SDIO_CMD_ABORTCMD_Pos                     (14U)
#define SDIO_CMD_ABORTCMD_Msk                     (0x1UL << SDIO_CMD_ABORTCMD_Pos)                  /*!< 0x00004000 */
#define SDIO_CMD_ABORTCMD                         SDIO_CMD_ABORTCMD_Msk                             /*!< desc ABORTCMD */
#define SDIO_CMD_AUTOINIT_Pos                     (15U)
#define SDIO_CMD_AUTOINIT_Msk                     (0x1UL << SDIO_CMD_AUTOINIT_Pos)                  /*!< 0x00008000 */
#define SDIO_CMD_AUTOINIT                         SDIO_CMD_AUTOINIT_Msk                             /*!< desc AUTOINIT */
#define SDIO_CMD_REGSYNC_Pos                      (21U)
#define SDIO_CMD_REGSYNC_Msk                      (0x1UL << SDIO_CMD_REGSYNC_Pos)                   /*!< 0x00200000 */
#define SDIO_CMD_REGSYNC                          SDIO_CMD_REGSYNC_Msk                              /*!< desc REGSYNC */
#define SDIO_CMD_ATACMD_Pos                       (22U)
#define SDIO_CMD_ATACMD_Msk                       (0x1UL << SDIO_CMD_ATACMD_Pos)                    /*!< 0x00400000 */
#define SDIO_CMD_ATACMD                           SDIO_CMD_ATACMD_Msk                               /*!< desc ATACMD */
#define SDIO_CMD_IEN_Pos                          (23U)
#define SDIO_CMD_IEN_Msk                          (0x1UL << SDIO_CMD_IEN_Pos)                       /*!< 0x00800000 */
#define SDIO_CMD_IEN                              SDIO_CMD_IEN_Msk                                  /*!< desc IEN */
#define SDIO_CMD_BOOTEN_Pos                       (24U)
#define SDIO_CMD_BOOTEN_Msk                       (0x1UL << SDIO_CMD_BOOTEN_Pos)                    /*!< 0x01000000 */
#define SDIO_CMD_BOOTEN                           SDIO_CMD_BOOTEN_Msk                               /*!< desc BOOTEN */
#define SDIO_CMD_BOOTACK_Pos                      (25U)
#define SDIO_CMD_BOOTACK_Msk                      (0x1UL << SDIO_CMD_BOOTACK_Pos)                   /*!< 0x02000000 */
#define SDIO_CMD_BOOTACK                          SDIO_CMD_BOOTACK_Msk                              /*!< desc BOOTACK */
#define SDIO_CMD_BOOTDIS_Pos                      (26U)
#define SDIO_CMD_BOOTDIS_Msk                      (0x1UL << SDIO_CMD_BOOTDIS_Pos)                   /*!< 0x04000000 */
#define SDIO_CMD_BOOTDIS                          SDIO_CMD_BOOTDIS_Msk                              /*!< desc BOOTDIS */
#define SDIO_CMD_BOOTMODE_Pos                     (27U)
#define SDIO_CMD_BOOTMODE_Msk                     (0x1UL << SDIO_CMD_BOOTMODE_Pos)                  /*!< 0x08000000 */
#define SDIO_CMD_BOOTMODE                         SDIO_CMD_BOOTMODE_Msk                             /*!< desc BOOTMODE */
#define SDIO_CMD_STARTCMD_Pos                     (31U)
#define SDIO_CMD_STARTCMD_Msk                     (0x1UL << SDIO_CMD_STARTCMD_Pos)                  /*!< 0x80000000 */
#define SDIO_CMD_STARTCMD                         SDIO_CMD_STARTCMD_Msk                             /*!< desc STARTCMD */

/*!< SDIO_RESPCMD */
#define SDIO_RESPCMD_RESPCMD_Pos                  (0U)
#define SDIO_RESPCMD_RESPCMD_Msk                  (0x3FUL << SDIO_RESPCMD_RESPCMD_Pos)              /*!< 0x0000003F */
#define SDIO_RESPCMD_RESPCMD                      SDIO_RESPCMD_RESPCMD_Msk                          /*!< RESPCMD[5:0] bits (desc RESPCMD) */
#define SDIO_RESPCMD_RESPCMD_0                    (0x1UL << SDIO_RESPCMD_RESPCMD_Pos)               /*!< 0x00000001 */
#define SDIO_RESPCMD_RESPCMD_1                    (0x2UL << SDIO_RESPCMD_RESPCMD_Pos)               /*!< 0x00000002 */
#define SDIO_RESPCMD_RESPCMD_2                    (0x4UL << SDIO_RESPCMD_RESPCMD_Pos)               /*!< 0x00000004 */
#define SDIO_RESPCMD_RESPCMD_3                    (0x8UL << SDIO_RESPCMD_RESPCMD_Pos)               /*!< 0x00000008 */
#define SDIO_RESPCMD_RESPCMD_4                    (0x10UL << SDIO_RESPCMD_RESPCMD_Pos)              /*!< 0x00000010 */
#define SDIO_RESPCMD_RESPCMD_5                    (0x20UL << SDIO_RESPCMD_RESPCMD_Pos)              /*!< 0x00000020 */


/*!< SDIO_RESP0 */
#define SDIO_RESP0_CARDSTATUS1_Pos                (0U)
#define SDIO_RESP0_CARDSTATUS1_Msk                (0xFFFFFFFFUL << SDIO_RESP0_CARDSTATUS1_Pos)      /*!< 0xFFFFFFFF */
#define SDIO_RESP0_CARDSTATUS1                    SDIO_RESP0_CARDSTATUS1_Msk                        /*!< CARDSTATUS1[31:0] bits (desc CARDSTATUS1) */

/*!< SDIO_RESP1 */
#define SDIO_RESP1_CARDSTATUS2_Pos                (0U)
#define SDIO_RESP1_CARDSTATUS2_Msk                (0xFFFFFFFFUL << SDIO_RESP1_CARDSTATUS2_Pos)      /*!< 0xFFFFFFFF */
#define SDIO_RESP1_CARDSTATUS2                    SDIO_RESP1_CARDSTATUS2_Msk                        /*!< CARDSTATUS2[31:0] bits (desc CARDSTATUS2) */

/*!< SDIO_RESP2 */
#define SDIO_RESP2_CARDSTATUS3_Pos                (0U)
#define SDIO_RESP2_CARDSTATUS3_Msk                (0xFFFFFFFFUL << SDIO_RESP2_CARDSTATUS3_Pos)      /*!< 0xFFFFFFFF */
#define SDIO_RESP2_CARDSTATUS3                    SDIO_RESP2_CARDSTATUS3_Msk                        /*!< CARDSTATUS3[31:0] bits (desc CARDSTATUS3) */

/*!< SDIO_RESP3 */
#define SDIO_RESP3_CARDSTATUS4_Pos                (0U)
#define SDIO_RESP3_CARDSTATUS4_Msk                (0xFFFFFFFFUL << SDIO_RESP3_CARDSTATUS4_Pos)      /*!< 0xFFFFFFFF */
#define SDIO_RESP3_CARDSTATUS4                    SDIO_RESP3_CARDSTATUS4_Msk                        /*!< CARDSTATUS4[31:0] bits (desc CARDSTATUS4) */

/*!< SDIO_TMOUT */
#define SDIO_TMOUT_RESPTIME_Pos                   (0U)
#define SDIO_TMOUT_RESPTIME_Msk                   (0xFFUL << SDIO_TMOUT_RESPTIME_Pos)               /*!< 0x000000FF */
#define SDIO_TMOUT_RESPTIME                       SDIO_TMOUT_RESPTIME_Msk                           /*!< RESPTIME[7:0] bits (desc RESPTIME) */
#define SDIO_TMOUT_DATATIME_Pos                   (8U)
#define SDIO_TMOUT_DATATIME_Msk                   (0xFFFFFFUL << SDIO_TMOUT_DATATIME_Pos)           /*!< 0xFFFFFF00 */
#define SDIO_TMOUT_DATATIME                       SDIO_TMOUT_DATATIME_Msk                           /*!< DATATIME[31:8] bits (desc DATATIME) */

/*!< SDIO_BLKSIZ */
#define SDIO_BLKSIZ_DBLOCKSIZE_Pos                (0U)
#define SDIO_BLKSIZ_DBLOCKSIZE_Msk                (0xFFFFUL << SDIO_BLKSIZ_DBLOCKSIZE_Pos)          /*!< 0x0000FFFF */
#define SDIO_BLKSIZ_DBLOCKSIZE                    SDIO_BLKSIZ_DBLOCKSIZE_Msk                        /*!< DBLOCKSIZE[15:0] bits (desc DBLOCKSIZE) */

/*!< SDIO_DLEN */
#define SDIO_DLEN_DATALENGTH_Pos                  (0U)
#define SDIO_DLEN_DATALENGTH_Msk                  (0xFFFFFFFFUL << SDIO_DLEN_DATALENGTH_Pos)        /*!< 0xFFFFFFFF */
#define SDIO_DLEN_DATALENGTH                      SDIO_DLEN_DATALENGTH_Msk                          /*!< DATALENGTH[31:0] bits (desc DATALENGTH) */

/*!< SDIO_CTRL */
#define SDIO_CTRL_SDIORST_Pos                     (0U)
#define SDIO_CTRL_SDIORST_Msk                     (0x1UL << SDIO_CTRL_SDIORST_Pos)                  /*!< 0x00000001 */
#define SDIO_CTRL_SDIORST                         SDIO_CTRL_SDIORST_Msk                             /*!< desc SDIORST */
#define SDIO_CTRL_FIFORST_Pos                     (1U)
#define SDIO_CTRL_FIFORST_Msk                     (0x1UL << SDIO_CTRL_FIFORST_Pos)                  /*!< 0x00000002 */
#define SDIO_CTRL_FIFORST                         SDIO_CTRL_FIFORST_Msk                             /*!< desc FIFORST */
#define SDIO_CTRL_INTEN_Pos                       (4U)
#define SDIO_CTRL_INTEN_Msk                       (0x1UL << SDIO_CTRL_INTEN_Pos)                    /*!< 0x00000010 */
#define SDIO_CTRL_INTEN                           SDIO_CTRL_INTEN_Msk                               /*!< desc INTEN */
#define SDIO_CTRL_DMAEN_Pos                       (5U)
#define SDIO_CTRL_DMAEN_Msk                       (0x1UL << SDIO_CTRL_DMAEN_Pos)                    /*!< 0x00000020 */
#define SDIO_CTRL_DMAEN                           SDIO_CTRL_DMAEN_Msk                               /*!< desc DMAEN */
#define SDIO_CTRL_READWAIT_Pos                    (6U)
#define SDIO_CTRL_READWAIT_Msk                    (0x1UL << SDIO_CTRL_READWAIT_Pos)                 /*!< 0x00000040 */
#define SDIO_CTRL_READWAIT                        SDIO_CTRL_READWAIT_Msk                            /*!< desc READWAIT */
#define SDIO_CTRL_AUTOIRQRESP_Pos                 (7U)
#define SDIO_CTRL_AUTOIRQRESP_Msk                 (0x1UL << SDIO_CTRL_AUTOIRQRESP_Pos)              /*!< 0x00000080 */
#define SDIO_CTRL_AUTOIRQRESP                     SDIO_CTRL_AUTOIRQRESP_Msk                         /*!< desc AUTOIRQRESP */
#define SDIO_CTRL_ABORTRD_Pos                     (8U)
#define SDIO_CTRL_ABORTRD_Msk                     (0x1UL << SDIO_CTRL_ABORTRD_Pos)                  /*!< 0x00000100 */
#define SDIO_CTRL_ABORTRD                         SDIO_CTRL_ABORTRD_Msk                             /*!< desc ABORTRD */
#define SDIO_CTRL_CCSDEN_Pos                      (9U)
#define SDIO_CTRL_CCSDEN_Msk                      (0x1UL << SDIO_CTRL_CCSDEN_Pos)                   /*!< 0x00000200 */
#define SDIO_CTRL_CCSDEN                          SDIO_CTRL_CCSDEN_Msk                              /*!< desc CCSDEN */
#define SDIO_CTRL_AUTOSTOPCCSD_Pos                (10U)
#define SDIO_CTRL_AUTOSTOPCCSD_Msk                (0x1UL << SDIO_CTRL_AUTOSTOPCCSD_Pos)             /*!< 0x00000400 */
#define SDIO_CTRL_AUTOSTOPCCSD                    SDIO_CTRL_AUTOSTOPCCSD_Msk                        /*!< desc AUTOSTOPCCSD */
#define SDIO_CTRL_CEATAINTEN_Pos                  (11U)
#define SDIO_CTRL_CEATAINTEN_Msk                  (0x1UL << SDIO_CTRL_CEATAINTEN_Pos)               /*!< 0x00000800 */
#define SDIO_CTRL_CEATAINTEN                      SDIO_CTRL_CEATAINTEN_Msk                          /*!< desc CEATAINTEN */
#define SDIO_CTRL_ODPUEN_Pos                      (24U)
#define SDIO_CTRL_ODPUEN_Msk                      (0x1UL << SDIO_CTRL_ODPUEN_Pos)                   /*!< 0x01000000 */
#define SDIO_CTRL_ODPUEN                          SDIO_CTRL_ODPUEN_Msk                              /*!< desc ODPUEN */

/*!< SDIO_STATUS */
#define SDIO_STATUS_RXWMARK_Pos                   (0U)
#define SDIO_STATUS_RXWMARK_Msk                   (0x1UL << SDIO_STATUS_RXWMARK_Pos)                /*!< 0x00000001 */
#define SDIO_STATUS_RXWMARK                       SDIO_STATUS_RXWMARK_Msk                           /*!< desc RXWMARK */
#define SDIO_STATUS_TXWMARK_Pos                   (1U)
#define SDIO_STATUS_TXWMARK_Msk                   (0x1UL << SDIO_STATUS_TXWMARK_Pos)                /*!< 0x00000002 */
#define SDIO_STATUS_TXWMARK                       SDIO_STATUS_TXWMARK_Msk                           /*!< desc TXWMARK */
#define SDIO_STATUS_FIFOE_Pos                     (2U)
#define SDIO_STATUS_FIFOE_Msk                     (0x1UL << SDIO_STATUS_FIFOE_Pos)                  /*!< 0x00000004 */
#define SDIO_STATUS_FIFOE                         SDIO_STATUS_FIFOE_Msk                             /*!< desc FIFOE */
#define SDIO_STATUS_FIFOF_Pos                     (3U)
#define SDIO_STATUS_FIFOF_Msk                     (0x1UL << SDIO_STATUS_FIFOF_Pos)                  /*!< 0x00000008 */
#define SDIO_STATUS_FIFOF                         SDIO_STATUS_FIFOF_Msk                             /*!< desc FIFOF */
#define SDIO_STATUS_CMDFSM_Pos                    (4U)
#define SDIO_STATUS_CMDFSM_Msk                    (0xFUL << SDIO_STATUS_CMDFSM_Pos)                 /*!< 0x000000F0 */
#define SDIO_STATUS_CMDFSM                        SDIO_STATUS_CMDFSM_Msk                            /*!< CMDFSM[7:4] bits (desc CMDFSM) */
#define SDIO_STATUS_CMDFSM_0                      (0x1UL << SDIO_STATUS_CMDFSM_Pos)                 /*!< 0x00000010 */
#define SDIO_STATUS_CMDFSM_1                      (0x2UL << SDIO_STATUS_CMDFSM_Pos)                 /*!< 0x00000020 */
#define SDIO_STATUS_CMDFSM_2                      (0x4UL << SDIO_STATUS_CMDFSM_Pos)                 /*!< 0x00000040 */
#define SDIO_STATUS_CMDFSM_3                      (0x8UL << SDIO_STATUS_CMDFSM_Pos)                 /*!< 0x00000080 */

#define SDIO_STATUS_CARDPRESENT_Pos               (8U)
#define SDIO_STATUS_CARDPRESENT_Msk               (0x1UL << SDIO_STATUS_CARDPRESENT_Pos)            /*!< 0x00000100 */
#define SDIO_STATUS_CARDPRESENT                   SDIO_STATUS_CARDPRESENT_Msk                       /*!< desc CARDPRESENT */
#define SDIO_STATUS_CARDBSY_Pos                   (9U)
#define SDIO_STATUS_CARDBSY_Msk                   (0x1UL << SDIO_STATUS_CARDBSY_Pos)                /*!< 0x00000200 */
#define SDIO_STATUS_CARDBSY                       SDIO_STATUS_CARDBSY_Msk                           /*!< desc CARDBSY */
#define SDIO_STATUS_FIFOCNT_Pos                   (17U)
#define SDIO_STATUS_FIFOCNT_Msk                   (0x1FFFUL << SDIO_STATUS_FIFOCNT_Pos)             /*!< 0x3FFE0000 */
#define SDIO_STATUS_FIFOCNT                       SDIO_STATUS_FIFOCNT_Msk                           /*!< FIFOCNT[29:17] bits (desc FIFOCNT) */

/*!< SDIO_INTSTS */
#define SDIO_INTSTS_CAD_Pos                       (0U)
#define SDIO_INTSTS_CAD_Msk                       (0x1UL << SDIO_INTSTS_CAD_Pos)                    /*!< 0x00000001 */
#define SDIO_INTSTS_CAD                           SDIO_INTSTS_CAD_Msk                               /*!< desc CAD */
#define SDIO_INTSTS_RE_Pos                        (1U)
#define SDIO_INTSTS_RE_Msk                        (0x1UL << SDIO_INTSTS_RE_Pos)                     /*!< 0x00000002 */
#define SDIO_INTSTS_RE                            SDIO_INTSTS_RE_Msk                                /*!< desc RE */
#define SDIO_INTSTS_CD_Pos                        (2U)
#define SDIO_INTSTS_CD_Msk                        (0x1UL << SDIO_INTSTS_CD_Pos)                     /*!< 0x00000004 */
#define SDIO_INTSTS_CD                            SDIO_INTSTS_CD_Msk                                /*!< desc CD */
#define SDIO_INTSTS_DTO_Pos                       (3U)
#define SDIO_INTSTS_DTO_Msk                       (0x1UL << SDIO_INTSTS_DTO_Pos)                    /*!< 0x00000008 */
#define SDIO_INTSTS_DTO                           SDIO_INTSTS_DTO_Msk                               /*!< desc DTO */
#define SDIO_INTSTS_TXDR_Pos                      (4U)
#define SDIO_INTSTS_TXDR_Msk                      (0x1UL << SDIO_INTSTS_TXDR_Pos)                   /*!< 0x00000010 */
#define SDIO_INTSTS_TXDR                          SDIO_INTSTS_TXDR_Msk                              /*!< desc TXDR */
#define SDIO_INTSTS_RXDR_Pos                      (5U)
#define SDIO_INTSTS_RXDR_Msk                      (0x1UL << SDIO_INTSTS_RXDR_Pos)                   /*!< 0x00000020 */
#define SDIO_INTSTS_RXDR                          SDIO_INTSTS_RXDR_Msk                              /*!< desc RXDR */
#define SDIO_INTSTS_RCRC_Pos                      (6U)
#define SDIO_INTSTS_RCRC_Msk                      (0x1UL << SDIO_INTSTS_RCRC_Pos)                   /*!< 0x00000040 */
#define SDIO_INTSTS_RCRC                          SDIO_INTSTS_RCRC_Msk                              /*!< desc RCRC */
#define SDIO_INTSTS_DCRC_Pos                      (7U)
#define SDIO_INTSTS_DCRC_Msk                      (0x1UL << SDIO_INTSTS_DCRC_Pos)                   /*!< 0x00000080 */
#define SDIO_INTSTS_DCRC                          SDIO_INTSTS_DCRC_Msk                              /*!< desc DCRC */
#define SDIO_INTSTS_RTO_BAR_Pos                   (8U)
#define SDIO_INTSTS_RTO_BAR_Msk                   (0x1UL << SDIO_INTSTS_RTO_BAR_Pos)                /*!< 0x00000100 */
#define SDIO_INTSTS_RTO_BAR                       SDIO_INTSTS_RTO_BAR_Msk                           /*!< desc RTO_BAR */
#define SDIO_INTSTS_DRTO_BDS_Pos                  (9U)
#define SDIO_INTSTS_DRTO_BDS_Msk                  (0x1UL << SDIO_INTSTS_DRTO_BDS_Pos)               /*!< 0x00000200 */
#define SDIO_INTSTS_DRTO_BDS                      SDIO_INTSTS_DRTO_BDS_Msk                          /*!< desc DRTO_BDS */
#define SDIO_INTSTS_HTO_Pos                       (10U)
#define SDIO_INTSTS_HTO_Msk                       (0x1UL << SDIO_INTSTS_HTO_Pos)                    /*!< 0x00000400 */
#define SDIO_INTSTS_HTO                           SDIO_INTSTS_HTO_Msk                               /*!< desc HTO */
#define SDIO_INTSTS_FRUN_Pos                      (11U)
#define SDIO_INTSTS_FRUN_Msk                      (0x1UL << SDIO_INTSTS_FRUN_Pos)                   /*!< 0x00000800 */
#define SDIO_INTSTS_FRUN                          SDIO_INTSTS_FRUN_Msk                              /*!< desc FRUN */
#define SDIO_INTSTS_HLE_Pos                       (12U)
#define SDIO_INTSTS_HLE_Msk                       (0x1UL << SDIO_INTSTS_HLE_Pos)                    /*!< 0x00001000 */
#define SDIO_INTSTS_HLE                           SDIO_INTSTS_HLE_Msk                               /*!< desc HLE */
#define SDIO_INTSTS_SBE_Pos                       (13U)
#define SDIO_INTSTS_SBE_Msk                       (0x1UL << SDIO_INTSTS_SBE_Pos)                    /*!< 0x00002000 */
#define SDIO_INTSTS_SBE                           SDIO_INTSTS_SBE_Msk                               /*!< desc SBE */
#define SDIO_INTSTS_ACD_Pos                       (14U)
#define SDIO_INTSTS_ACD_Msk                       (0x1UL << SDIO_INTSTS_ACD_Pos)                    /*!< 0x00004000 */
#define SDIO_INTSTS_ACD                           SDIO_INTSTS_ACD_Msk                               /*!< desc ACD */
#define SDIO_INTSTS_EBE_Pos                       (15U)
#define SDIO_INTSTS_EBE_Msk                       (0x1UL << SDIO_INTSTS_EBE_Pos)                    /*!< 0x00008000 */
#define SDIO_INTSTS_EBE                           SDIO_INTSTS_EBE_Msk                               /*!< desc EBE */
#define SDIO_INTSTS_SDIOINT_Pos                   (16U)
#define SDIO_INTSTS_SDIOINT_Msk                   (0x1UL << SDIO_INTSTS_SDIOINT_Pos)                /*!< 0x00010000 */
#define SDIO_INTSTS_SDIOINT                       SDIO_INTSTS_SDIOINT_Msk                           /*!< desc SDIOINT */

/*!< SDIO_INTMASK */
#define SDIO_INTMASK_CADIE_Pos                    (0U)
#define SDIO_INTMASK_CADIE_Msk                    (0x1UL << SDIO_INTMASK_CADIE_Pos)                 /*!< 0x00000001 */
#define SDIO_INTMASK_CADIE                        SDIO_INTMASK_CADIE_Msk                            /*!< desc CADIE */
#define SDIO_INTMASK_REIE_Pos                     (1U)
#define SDIO_INTMASK_REIE_Msk                     (0x1UL << SDIO_INTMASK_REIE_Pos)                  /*!< 0x00000002 */
#define SDIO_INTMASK_REIE                         SDIO_INTMASK_REIE_Msk                             /*!< desc REIE */
#define SDIO_INTMASK_CDIE_Pos                     (2U)
#define SDIO_INTMASK_CDIE_Msk                     (0x1UL << SDIO_INTMASK_CDIE_Pos)                  /*!< 0x00000004 */
#define SDIO_INTMASK_CDIE                         SDIO_INTMASK_CDIE_Msk                             /*!< desc CDIE */
#define SDIO_INTMASK_DTOIE_Pos                    (3U)
#define SDIO_INTMASK_DTOIE_Msk                    (0x1UL << SDIO_INTMASK_DTOIE_Pos)                 /*!< 0x00000008 */
#define SDIO_INTMASK_DTOIE                        SDIO_INTMASK_DTOIE_Msk                            /*!< desc DTOIE */
#define SDIO_INTMASK_TXDRIE_Pos                   (4U)
#define SDIO_INTMASK_TXDRIE_Msk                   (0x1UL << SDIO_INTMASK_TXDRIE_Pos)                /*!< 0x00000010 */
#define SDIO_INTMASK_TXDRIE                       SDIO_INTMASK_TXDRIE_Msk                           /*!< desc TXDRIE */
#define SDIO_INTMASK_RXDRIE_Pos                   (5U)
#define SDIO_INTMASK_RXDRIE_Msk                   (0x1UL << SDIO_INTMASK_RXDRIE_Pos)                /*!< 0x00000020 */
#define SDIO_INTMASK_RXDRIE                       SDIO_INTMASK_RXDRIE_Msk                           /*!< desc RXDRIE */
#define SDIO_INTMASK_RCRCIE_Pos                   (6U)
#define SDIO_INTMASK_RCRCIE_Msk                   (0x1UL << SDIO_INTMASK_RCRCIE_Pos)                /*!< 0x00000040 */
#define SDIO_INTMASK_RCRCIE                       SDIO_INTMASK_RCRCIE_Msk                           /*!< desc RCRCIE */
#define SDIO_INTMASK_DCRCIE_Pos                   (7U)
#define SDIO_INTMASK_DCRCIE_Msk                   (0x1UL << SDIO_INTMASK_DCRCIE_Pos)                /*!< 0x00000080 */
#define SDIO_INTMASK_DCRCIE                       SDIO_INTMASK_DCRCIE_Msk                           /*!< desc DCRCIE */
#define SDIO_INTMASK_RTOIE_Pos                    (8U)
#define SDIO_INTMASK_RTOIE_Msk                    (0x1UL << SDIO_INTMASK_RTOIE_Pos)                 /*!< 0x00000100 */
#define SDIO_INTMASK_RTOIE                        SDIO_INTMASK_RTOIE_Msk                            /*!< desc RTOIE */
#define SDIO_INTMASK_DRTOIE_Pos                   (9U)
#define SDIO_INTMASK_DRTOIE_Msk                   (0x1UL << SDIO_INTMASK_DRTOIE_Pos)                /*!< 0x00000200 */
#define SDIO_INTMASK_DRTOIE                       SDIO_INTMASK_DRTOIE_Msk                           /*!< desc DRTOIE */
#define SDIO_INTMASK_HTOIE_Pos                    (10U)
#define SDIO_INTMASK_HTOIE_Msk                    (0x1UL << SDIO_INTMASK_HTOIE_Pos)                 /*!< 0x00000400 */
#define SDIO_INTMASK_HTOIE                        SDIO_INTMASK_HTOIE_Msk                            /*!< desc HTOIE */
#define SDIO_INTMASK_FRUNIE_Pos                   (11U)
#define SDIO_INTMASK_FRUNIE_Msk                   (0x1UL << SDIO_INTMASK_FRUNIE_Pos)                /*!< 0x00000800 */
#define SDIO_INTMASK_FRUNIE                       SDIO_INTMASK_FRUNIE_Msk                           /*!< desc FRUNIE */
#define SDIO_INTMASK_HLEIE_Pos                    (12U)
#define SDIO_INTMASK_HLEIE_Msk                    (0x1UL << SDIO_INTMASK_HLEIE_Pos)                 /*!< 0x00001000 */
#define SDIO_INTMASK_HLEIE                        SDIO_INTMASK_HLEIE_Msk                            /*!< desc HLEIE */
#define SDIO_INTMASK_SBEIE_Pos                    (13U)
#define SDIO_INTMASK_SBEIE_Msk                    (0x1UL << SDIO_INTMASK_SBEIE_Pos)                 /*!< 0x00002000 */
#define SDIO_INTMASK_SBEIE                        SDIO_INTMASK_SBEIE_Msk                            /*!< desc SBEIE */
#define SDIO_INTMASK_ACDIE_Pos                    (14U)
#define SDIO_INTMASK_ACDIE_Msk                    (0x1UL << SDIO_INTMASK_ACDIE_Pos)                 /*!< 0x00004000 */
#define SDIO_INTMASK_ACDIE                        SDIO_INTMASK_ACDIE_Msk                            /*!< desc ACDIE */
#define SDIO_INTMASK_EBEIE_Pos                    (15U)
#define SDIO_INTMASK_EBEIE_Msk                    (0x1UL << SDIO_INTMASK_EBEIE_Pos)                 /*!< 0x00008000 */
#define SDIO_INTMASK_EBEIE                        SDIO_INTMASK_EBEIE_Msk                            /*!< desc EBEIE */
#define SDIO_INTMASK_SDIOINTIE_Pos                (16U)
#define SDIO_INTMASK_SDIOINTIE_Msk                (0x1UL << SDIO_INTMASK_SDIOINTIE_Pos)             /*!< 0x00010000 */
#define SDIO_INTMASK_SDIOINTIE                    SDIO_INTMASK_SDIOINTIE_Msk                        /*!< desc SDIOINTIE */

/*!< SDIO_FIFOTH */
#define SDIO_FIFOTH_TXWMARK_Pos                   (0U)
#define SDIO_FIFOTH_TXWMARK_Msk                   (0xFFFUL << SDIO_FIFOTH_TXWMARK_Pos)              /*!< 0x00000FFF */
#define SDIO_FIFOTH_TXWMARK                       SDIO_FIFOTH_TXWMARK_Msk                           /*!< TXWMARK[11:0] bits (desc TXWMARK) */
#define SDIO_FIFOTH_RXWMARK_Pos                   (16U)
#define SDIO_FIFOTH_RXWMARK_Msk                   (0xFFFUL << SDIO_FIFOTH_RXWMARK_Pos)              /*!< 0x0FFF0000 */
#define SDIO_FIFOTH_RXWMARK                       SDIO_FIFOTH_RXWMARK_Msk                           /*!< RXWMARK[27:16] bits (desc RXWMARK) */

/*!< SDIO_TCBCNT */
#define SDIO_TCBCNT_TCBCNT_Pos                    (0U)
#define SDIO_TCBCNT_TCBCNT_Msk                    (0xFFFFFFFFUL << SDIO_TCBCNT_TCBCNT_Pos)          /*!< 0xFFFFFFFF */
#define SDIO_TCBCNT_TCBCNT                        SDIO_TCBCNT_TCBCNT_Msk                            /*!< TCBCNT[31:0] bits (desc TCBCNT) */

/*!< SDIO_TBBCNT */
#define SDIO_TBBCNT_TBBCNT_Pos                    (0U)
#define SDIO_TBBCNT_TBBCNT_Msk                    (0xFFFFFFFFUL << SDIO_TBBCNT_TBBCNT_Pos)          /*!< 0xFFFFFFFF */
#define SDIO_TBBCNT_TBBCNT                        SDIO_TBBCNT_TBBCNT_Msk                            /*!< TBBCNT[31:0] bits (desc TBBCNT) */

/*!< SDIO_FIFODATA */
#define SDIO_FIFODATA_FIFODATA_Pos                (0U)
#define SDIO_FIFODATA_FIFODATA_Msk                (0xFFFFFFFFUL << SDIO_FIFODATA_FIFODATA_Pos)      /*!< 0xFFFFFFFF */
#define SDIO_FIFODATA_FIFODATA                    SDIO_FIFODATA_FIFODATA_Msk                        /*!< FIFODATA[31:0] bits (desc FIFODATA) */

/********************************************************************************************************************/
/********************************* SPI1 *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for SPI_CR1 register **********************************************/
#define SPI_CR1_CPHA_Pos                          (0U)
#define SPI_CR1_CPHA_Msk                          (0x1UL<<SPI_CR1_CPHA_Pos)                         /*!< 0x00000001 */
#define SPI_CR1_CPHA                              SPI_CR1_CPHA_Msk                                  /*!< Clock Phase */
#define SPI_CR1_CPOL_Pos                          (1U)
#define SPI_CR1_CPOL_Msk                          (0x1UL<<SPI_CR1_CPOL_Pos)                         /*!< 0x00000002 */
#define SPI_CR1_CPOL                              SPI_CR1_CPOL_Msk                                  /*!< Clock Polarity  */
#define SPI_CR1_MSTR_Pos                          (2U)
#define SPI_CR1_MSTR_Msk                          (0x1UL<<SPI_CR1_MSTR_Pos)                         /*!< 0x00000004 */
#define SPI_CR1_MSTR                              SPI_CR1_MSTR_Msk                                  /*!< Master Selection */
#define SPI_CR1_BR_Pos                            (3U)
#define SPI_CR1_BR_Msk                            (0x7UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000038 */
#define SPI_CR1_BR                                SPI_CR1_BR_Msk                                    /*!< Baud Rate Control */
#define SPI_CR1_BR_0                              (0x1UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000008 */
#define SPI_CR1_BR_1                              (0x2UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000010 */
#define SPI_CR1_BR_2                              (0x4UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000020 */
#define SPI_CR1_SPE_Pos                           (6U)
#define SPI_CR1_SPE_Msk                           (0x1UL<<SPI_CR1_SPE_Pos)                          /*!< 0x00000040 */
#define SPI_CR1_SPE                               SPI_CR1_SPE_Msk                                   /*!< SPI Enable */
#define SPI_CR1_LSBFIRST_Pos                      (7U)
#define SPI_CR1_LSBFIRST_Msk                      (0x1UL<<SPI_CR1_LSBFIRST_Pos)                     /*!< 0x00000080 */
#define SPI_CR1_LSBFIRST                          SPI_CR1_LSBFIRST_Msk                              /*!<  Frame Format */
#define SPI_CR1_SSI_Pos                           (8U)
#define SPI_CR1_SSI_Msk                           (0x1UL<<SPI_CR1_SSI_Pos)                          /*!< 0x00000100 */
#define SPI_CR1_SSI                               SPI_CR1_SSI_Msk                                   /*!< Internal slave select */
#define SPI_CR1_SSM_Pos                           (9U)
#define SPI_CR1_SSM_Msk                           (0x1UL<<SPI_CR1_SSM_Pos)                          /*!< 0x00000200 */
#define SPI_CR1_SSM                               SPI_CR1_SSM_Msk                                   /*!< Software slave management  */
#define SPI_CR1_RXONLY_Pos                        (10U)
#define SPI_CR1_RXONLY_Msk                        (0x1UL<<SPI_CR1_RXONLY_Pos)                       /*!< 0x00000400 */
#define SPI_CR1_RXONLY                            SPI_CR1_RXONLY_Msk                                /*!< Receive only */
#define SPI_CR1_DFF_Pos                           (11U)
#define SPI_CR1_DFF_Msk                           (0x1UL<<SPI_CR1_DFF_Pos)                          /*!< 0x00000800 */
#define SPI_CR1_DFF                               SPI_CR1_DFF_Msk                                   /*!< Data frame format */
#define SPI_CR1_CRCNEXT_Pos                       (12U)
#define SPI_CR1_CRCNEXT_Msk                       (0x1UL << SPI_CR1_CRCNEXT_Pos)                    /*!< 0x00001000 */
#define SPI_CR1_CRCNEXT                           SPI_CR1_CRCNEXT_Msk                               /*!< desc CRCNEXT */
#define SPI_CR1_CRCEN_Pos                         (13U)
#define SPI_CR1_CRCEN_Msk                         (0x1UL << SPI_CR1_CRCEN_Pos)                      /*!< 0x00002000 */
#define SPI_CR1_CRCEN                             SPI_CR1_CRCEN_Msk                                 /*!< desc CRCEN */
#define SPI_CR1_BIDIOE_Pos                        (14U)
#define SPI_CR1_BIDIOE_Msk                        (0x1UL<<SPI_CR1_BIDIOE_Pos)                       /*!< 0x00004000 */
#define SPI_CR1_BIDIOE                            SPI_CR1_BIDIOE_Msk                                /*!< Output enable in bidirectional mode */
#define SPI_CR1_BIDIMODE_Pos                      (15U)
#define SPI_CR1_BIDIMODE_Msk                      (0x1UL<<SPI_CR1_BIDIMODE_Pos)                     /*!< 0x00008000 */
#define SPI_CR1_BIDIMODE                          SPI_CR1_BIDIMODE_Msk                              /*!< Bidirectional data mode enable */

/********************************* Bit definition for SPI_CR2 register **********************************************/
#define SPI_CR2_RXDMAEN_Pos                       (0U)
#define SPI_CR2_RXDMAEN_Msk                       (0x1UL<<SPI_CR2_RXDMAEN_Pos)                      /*!< 0x00000001 */
#define SPI_CR2_RXDMAEN                           SPI_CR2_RXDMAEN_Msk                               /*!< Rx buffer DMA enable */
#define SPI_CR2_TXDMAEN_Pos                       (1U)
#define SPI_CR2_TXDMAEN_Msk                       (0x1UL<<SPI_CR2_TXDMAEN_Pos)                      /*!< 0x00000002 */
#define SPI_CR2_TXDMAEN                           SPI_CR2_TXDMAEN_Msk                               /*!< Tx buffer DMA enable */
#define SPI_CR2_SSOE_Pos                          (2U)
#define SPI_CR2_SSOE_Msk                          (0x1UL<<SPI_CR2_SSOE_Pos)                         /*!< 0x00000004 */
#define SPI_CR2_SSOE                              SPI_CR2_SSOE_Msk                                  /*!< SS Output Enable  */
#define SPI_CR2_CLRTXFIFO_Pos                     (4U)
#define SPI_CR2_CLRTXFIFO_Msk                     (0x1UL<<SPI_CR2_CLRTXFIFO_Pos)                    /*!< 0x00000010 */
#define SPI_CR2_CLRTXFIFO                         SPI_CR2_CLRTXFIFO_Msk                             /*!< Clear FIFO */
#define SPI_CR2_ERRIE_Pos                         (5U)
#define SPI_CR2_ERRIE_Msk                         (0x1UL<<SPI_CR2_ERRIE_Pos)                        /*!< 0x00000020 */
#define SPI_CR2_ERRIE                             SPI_CR2_ERRIE_Msk                                 /*!< Error Interrupt Enable */
#define SPI_CR2_RXNEIE_Pos                        (6U)
#define SPI_CR2_RXNEIE_Msk                        (0x1UL<<SPI_CR2_RXNEIE_Pos)                       /*!< 0x00000040 */
#define SPI_CR2_RXNEIE                            SPI_CR2_RXNEIE_Msk                                /*!< RX buffer Not Empty Interrupt Enable */
#define SPI_CR2_TXEIE_Pos                         (7U)
#define SPI_CR2_TXEIE_Msk                         (0x1UL<<SPI_CR2_TXEIE_Pos)                        /*!< 0x00000080 */
#define SPI_CR2_TXEIE                             SPI_CR2_TXEIE_Msk                                 /*!< Tx buffer Empty Interrupt Enable */
#define SPI_CR2_FRXTH_Pos                         (12U)
#define SPI_CR2_FRXTH_Msk                         (0x1UL<<SPI_CR2_FRXTH_Pos)                        /*!< 0x00001000 */
#define SPI_CR2_FRXTH                             SPI_CR2_FRXTH_Msk                                 
#define SPI_CR2_FRF_Pos                           (15U)
#define SPI_CR2_FRF_Msk                           (0x1UL<<SPI_CR2_FRF_Pos)                          /*!< 0x00008000 */
#define SPI_CR2_FRF                               SPI_CR2_FRF_Msk                                   

/********************************* Bit definition for SPI_SR register ***********************************************/
#define SPI_SR_RXNE_Pos                           (0U)
#define SPI_SR_RXNE_Msk                           (0x1UL<<SPI_SR_RXNE_Pos)                          /*!< 0x00000001 */
#define SPI_SR_RXNE                               SPI_SR_RXNE_Msk                                   /*!< Receive buffer Not Empty  */
#define SPI_SR_TXE_Pos                            (1U)
#define SPI_SR_TXE_Msk                            (0x1UL<<SPI_SR_TXE_Pos)                           /*!< 0x00000002 */
#define SPI_SR_TXE                                SPI_SR_TXE_Msk                                    /*!< Transmit buffer Empty  */
#define SPI_SR_CHSIDE_Pos                         (2U)
#define SPI_SR_CHSIDE_Msk                         (0x1UL << SPI_SR_CHSIDE_Pos)                      /*!< 0x00000004 */
#define SPI_SR_CHSIDE                             SPI_SR_CHSIDE_Msk                                 /*!< desc CHSIDE */
#define SPI_SR_UDR_Pos                            (3U)
#define SPI_SR_UDR_Msk                            (0x1UL << SPI_SR_UDR_Pos)                         /*!< 0x00000008 */
#define SPI_SR_UDR                                SPI_SR_UDR_Msk                                    /*!< desc UDR */
#define SPI_SR_CRCERR_Pos                         (4U)
#define SPI_SR_CRCERR_Msk                         (0x1UL << SPI_SR_CRCERR_Pos)                      /*!< 0x00000010 */
#define SPI_SR_CRCERR                             SPI_SR_CRCERR_Msk                                 /*!< desc CRCERR */
#define SPI_SR_MODF_Pos                           (5U)
#define SPI_SR_MODF_Msk                           (0x1UL<<SPI_SR_MODF_Pos)                          /*!< 0x00000020 */
#define SPI_SR_MODF                               SPI_SR_MODF_Msk                                   /*!< Mode fault */
#define SPI_SR_OVR_Pos                            (6U)
#define SPI_SR_OVR_Msk                            (0x1UL<<SPI_SR_OVR_Pos)                           /*!< 0x00000040 */
#define SPI_SR_OVR                                SPI_SR_OVR_Msk                                    /*!<  Overrun flag */
#define SPI_SR_BSY_Pos                            (7U)
#define SPI_SR_BSY_Msk                            (0x1UL<<SPI_SR_BSY_Pos)                           /*!< 0x00000080 */
#define SPI_SR_BSY                                SPI_SR_BSY_Msk                                    /*!< Busy flag  */
#define SPI_SR_FRE_Pos                            (8U)
#define SPI_SR_FRE_Msk                            (0x1UL<<SPI_SR_FRE_Pos)                           /*!< 0x00000100 */
#define SPI_SR_FRE                                SPI_SR_FRE_Msk                                    
#define SPI_SR_FRLVL_Pos                          (9U)
#define SPI_SR_FRLVL_Msk                          (0x3UL<<SPI_SR_FRLVL_Pos)                         /*!< 0x00000600 */
#define SPI_SR_FRLVL                              SPI_SR_FRLVL_Msk                                  /*!< FIFO Reception Level */
#define SPI_SR_FRLVL_0                            (0x1UL<<SPI_SR_FRLVL_Pos)                         /*!< 0x00000200 */
#define SPI_SR_FRLVL_1                            (0x2UL<<SPI_SR_FRLVL_Pos)                         /*!< 0x00000400 */
#define SPI_SR_FTLVL_Pos                          (11U)
#define SPI_SR_FTLVL_Msk                          (0x3UL<<SPI_SR_FTLVL_Pos)                         /*!< 0x00001800 */
#define SPI_SR_FTLVL                              SPI_SR_FTLVL_Msk                                  /*!< FIFO Transmission Level */
#define SPI_SR_FTLVL_0                            (0x1UL<<SPI_SR_FTLVL_Pos)                         /*!< 0x00000800 */
#define SPI_SR_FTLVL_1                            (0x2UL<<SPI_SR_FTLVL_Pos)                         /*!< 0x00001000 */

/********************************* Bit definition for SPI_DR register ***********************************************/
#define SPI_DR_DR_Pos                             (0U)
#define SPI_DR_DR_Msk                             (0xFFFFUL<<SPI_DR_DR_Pos)                         /*!< 0x0000FFFF */
#define SPI_DR_DR                                 SPI_DR_DR_Msk                                     /*!< Data Register */

/********************************* Bit definition for SPI_CRCPR register ********************************************/
#define SPI_CRCPR_CRCPOLY_Pos                     (0U)
#define SPI_CRCPR_CRCPOLY_Msk                     (0xFFFFUL<<SPI_CRCPR_CRCPOLY_Pos)                 /*!< 0x0000FFFF */
#define SPI_CRCPR_CRCPOLY                         SPI_CRCPR_CRCPOLY_Msk                             /*!< CRC polynomial register */

/********************************* Bit definition for SPI_RXCRCR register *******************************************/
#define SPI_RXCRCR_RXCRC_Pos                      (0U)
#define SPI_RXCRCR_RXCRC_Msk                      (0xFFFFUL<<SPI_RXCRCR_RXCRC_Pos)                  /*!< 0x0000FFFF */
#define SPI_RXCRCR_RXCRC                          SPI_RXCRCR_RXCRC_Msk                              /*!< rcv crc reg */

/********************************* Bit definition for SPI_TXCRCR register *******************************************/
#define SPI_TXCRCR_TXCRC_Pos                      (0U)
#define SPI_TXCRCR_TXCRC_Msk                      (0xFFFFUL<<SPI_TXCRCR_TXCRC_Pos)                  /*!< 0x0000FFFF */
#define SPI_TXCRCR_TXCRC                          SPI_TXCRCR_TXCRC_Msk                              /*!< tx crc reg  */

/********************************* Bit definition for SPI_I2SCFGR register ******************************************/
#define SPI_I2SCFGR_CHLEN_Pos                     (0U)
#define SPI_I2SCFGR_CHLEN_Msk                     (0x1UL<<SPI_I2SCFGR_CHLEN_Pos)                    /*!< 0x00000001 */
#define SPI_I2SCFGR_CHLEN                         SPI_I2SCFGR_CHLEN_Msk                             /*!< Channel length (number of bits per audio channel */
#define SPI_I2SCFGR_DATLEN_Pos                    (1U)
#define SPI_I2SCFGR_DATLEN_Msk                    (0x3UL<<SPI_I2SCFGR_DATLEN_Pos)                   /*!< 0x00000006 */
#define SPI_I2SCFGR_DATLEN                        SPI_I2SCFGR_DATLEN_Msk                            /*!< Data length to be transferred */
#define SPI_I2SCFGR_DATLEN_0                      (0x1UL<<SPI_I2SCFGR_DATLEN_Pos)                   /*!< 0x00000002 */
#define SPI_I2SCFGR_DATLEN_1                      (0x2UL<<SPI_I2SCFGR_DATLEN_Pos)                   /*!< 0x00000004 */
#define SPI_I2SCFGR_CKPOL_Pos                     (3U)
#define SPI_I2SCFGR_CKPOL_Msk                     (0x1UL<<SPI_I2SCFGR_CKPOL_Pos)                    /*!< 0x00000008 */
#define SPI_I2SCFGR_CKPOL                         SPI_I2SCFGR_CKPOL_Msk                             /*!< Steady state clock polarity */
#define SPI_I2SCFGR_I2SSTD_Pos                    (4U)
#define SPI_I2SCFGR_I2SSTD_Msk                    (0x3UL<<SPI_I2SCFGR_I2SSTD_Pos)                   /*!< 0x00000030 */
#define SPI_I2SCFGR_I2SSTD                        SPI_I2SCFGR_I2SSTD_Msk                            /*!< I2S standard selection */
#define SPI_I2SCFGR_I2SSTD_0                      (0x1UL<<SPI_I2SCFGR_I2SSTD_Pos)                   /*!< 0x00000010 */
#define SPI_I2SCFGR_I2SSTD_1                      (0x2UL<<SPI_I2SCFGR_I2SSTD_Pos)                   /*!< 0x00000020 */
#define SPI_I2SCFGR_PCMSYNC_Pos                   (7U)
#define SPI_I2SCFGR_PCMSYNC_Msk                   (0x1UL<<SPI_I2SCFGR_PCMSYNC_Pos)                  /*!< 0x00000080 */
#define SPI_I2SCFGR_PCMSYNC                       SPI_I2SCFGR_PCMSYNC_Msk                           /*!< PCM frame synchronization */
#define SPI_I2SCFGR_I2SCFG_Pos                    (8U)
#define SPI_I2SCFGR_I2SCFG_Msk                    (0x3UL<<SPI_I2SCFGR_I2SCFG_Pos)                   /*!< 0x00000300 */
#define SPI_I2SCFGR_I2SCFG                        SPI_I2SCFGR_I2SCFG_Msk                            /*!< I2S configuration mode */
#define SPI_I2SCFGR_I2SCFG_0                      (0x1UL<<SPI_I2SCFGR_I2SCFG_Pos)                   /*!< 0x00000100 */
#define SPI_I2SCFGR_I2SCFG_1                      (0x2UL<<SPI_I2SCFGR_I2SCFG_Pos)                   /*!< 0x00000200 */
#define SPI_I2SCFGR_I2SE_Pos                      (10U)
#define SPI_I2SCFGR_I2SE_Msk                      (0x1UL<<SPI_I2SCFGR_I2SE_Pos)                     /*!< 0x00000400 */
#define SPI_I2SCFGR_I2SE                          SPI_I2SCFGR_I2SE_Msk                              /*!<  I2S enable */
#define SPI_I2SCFGR_I2SMOD_Pos                    (11U)
#define SPI_I2SCFGR_I2SMOD_Msk                    (0x1UL<<SPI_I2SCFGR_I2SMOD_Pos)                   /*!< 0x00000800 */
#define SPI_I2SCFGR_I2SMOD                        SPI_I2SCFGR_I2SMOD_Msk                            /*!< I2S mode selection */

/********************************* Bit definition for SPI_I2SPR register ********************************************/
#define SPI_I2SPR_I2SDIV_Pos                      (0U)
#define SPI_I2SPR_I2SDIV_Msk                      (0xFFUL<<SPI_I2SPR_I2SDIV_Pos)                    /*!< 0x000000FF */
#define SPI_I2SPR_I2SDIV                          SPI_I2SPR_I2SDIV_Msk                              /*!< I2S linear prescaler */
#define SPI_I2SPR_ODD_Pos                         (8U)
#define SPI_I2SPR_ODD_Msk                         (0x1UL<<SPI_I2SPR_ODD_Pos)                        /*!< 0x00000100 */
#define SPI_I2SPR_ODD                             SPI_I2SPR_ODD_Msk                                 /*!< Odd factor for the prescaler */
#define SPI_I2SPR_MCKOE_Pos                       (9U)
#define SPI_I2SPR_MCKOE_Msk                       (0x1UL<<SPI_I2SPR_MCKOE_Pos)                      /*!< 0x00000200 */
#define SPI_I2SPR_MCKOE                           SPI_I2SPR_MCKOE_Msk                               /*!<  Master clock output enable */

#define SPI_I2S_SUPPORT       /*!< I2S support */

/********************************************************************************************************************/
/********************************* SYSCFG ***************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for SYSCFG_CFGR1 register *****************************************/
#define SYSCFG_CFGR1_MEM_MODE_Pos                 (0U)
#define SYSCFG_CFGR1_MEM_MODE_Msk                 (0x3UL<<SYSCFG_CFGR1_MEM_MODE_Pos)                /*!< 0x00000003 */
#define SYSCFG_CFGR1_MEM_MODE                     SYSCFG_CFGR1_MEM_MODE_Msk                         /*!< memory mapping mode */
#define SYSCFG_CFGR1_MEM_MODE_0                   (0x1UL<<SYSCFG_CFGR1_MEM_MODE_Pos)                /*!< 0x00000001 */
#define SYSCFG_CFGR1_MEM_MODE_1                   (0x2UL<<SYSCFG_CFGR1_MEM_MODE_Pos)                /*!< 0x00000002 */
#define SYSCFG_CFGR1_ETH_PHYSEL_Pos               (2U)
#define SYSCFG_CFGR1_ETH_PHYSEL_Msk               (0x1UL<<SYSCFG_CFGR1_ETH_PHYSEL_Pos)              /*!< 0x00000004 */
#define SYSCFG_CFGR1_ETH_PHYSEL                   SYSCFG_CFGR1_ETH_PHYSEL_Msk                       /*!< Ethernet PHY Interface selection */
#define SYSCFG_CFGR1_TIM2ITR1_REMAP_Pos           (3U)
#define SYSCFG_CFGR1_TIM2ITR1_REMAP_Msk           (0x3UL<<SYSCFG_CFGR1_TIM2ITR1_REMAP_Pos)          /*!< 0x00000018 */
#define SYSCFG_CFGR1_TIM2ITR1_REMAP               SYSCFG_CFGR1_TIM2ITR1_REMAP_Msk                   /*!< timer2 inner trigger input 1 mapping */
#define SYSCFG_CFGR1_TIM2ITR1_REMAP_0             (0x1UL<<SYSCFG_CFGR1_TIM2ITR1_REMAP_Pos)          /*!< 0x00000008 */
#define SYSCFG_CFGR1_TIM2ITR1_REMAP_1             (0x2UL<<SYSCFG_CFGR1_TIM2ITR1_REMAP_Pos)          /*!< 0x00000010 */
#define SYSCFG_CFGR1_PTP_PPS_REMAP_Pos            (5U)
#define SYSCFG_CFGR1_PTP_PPS_REMAP_Msk            (0x1UL<<SYSCFG_CFGR1_PTP_PPS_REMAP_Pos)           /*!< 0x00000020 */
#define SYSCFG_CFGR1_PTP_PPS_REMAP                SYSCFG_CFGR1_PTP_PPS_REMAP_Msk                    /*!< ETH PTP PPS mapping */
#define SYSCFG_CFGR1_CTC_SOF_SEL_Pos              (6U)
#define SYSCFG_CFGR1_CTC_SOF_SEL_Msk              (0x1UL<<SYSCFG_CFGR1_CTC_SOF_SEL_Pos)             /*!< 0x00000040 */
#define SYSCFG_CFGR1_CTC_SOF_SEL                  SYSCFG_CFGR1_CTC_SOF_SEL_Msk                      /*!< selection of usb sof to ctc */
#define SYSCFG_CFGR1_I2C1_FMP_Pos                 (7U)
#define SYSCFG_CFGR1_I2C1_FMP_Msk                 (0x1UL<<SYSCFG_CFGR1_I2C1_FMP_Pos)                /*!< 0x00000080 */
#define SYSCFG_CFGR1_I2C1_FMP                     SYSCFG_CFGR1_I2C1_FMP_Msk                         /*!< I2C1 Fast-mode Plus driving capability activation */
#define SYSCFG_CFGR1_I2C2_FMP_Pos                 (8U)
#define SYSCFG_CFGR1_I2C2_FMP_Msk                 (0x1UL<<SYSCFG_CFGR1_I2C2_FMP_Pos)                /*!< 0x00000100 */
#define SYSCFG_CFGR1_I2C2_FMP                     SYSCFG_CFGR1_I2C2_FMP_Msk                         /*!< I2C2 Fast-mode Plus driving capability activation */
#define SYSCFG_CFGR1_I2C3_FMP_Pos                 (9U)
#define SYSCFG_CFGR1_I2C3_FMP_Msk                 (0x1UL<<SYSCFG_CFGR1_I2C3_FMP_Pos)                /*!< 0x00000200 */
#define SYSCFG_CFGR1_I2C3_FMP                     SYSCFG_CFGR1_I2C3_FMP_Msk                         /*!< I2C3 Fast-mode Plus driving capability activation */
#define SYSCFG_CFGR1_I2C4_FMP_Pos                 (10U)
#define SYSCFG_CFGR1_I2C4_FMP_Msk                 (0x1UL<<SYSCFG_CFGR1_I2C4_FMP_Pos)                /*!< 0x00000400 */
#define SYSCFG_CFGR1_I2C4_FMP                     SYSCFG_CFGR1_I2C4_FMP_Msk                         /*!< I2C4 Fast-mode Plus driving capability activation */
#define SYSCFG_CFGR1_I2C_EIIC_Pos                 (16U)
#define SYSCFG_CFGR1_I2C_EIIC_Msk                 (0xFFFFUL<<SYSCFG_CFGR1_I2C_EIIC_Pos)             /*!< 0xFFFF0000 */
#define SYSCFG_CFGR1_I2C_EIIC                     SYSCFG_CFGR1_I2C_EIIC_Msk                         /*!< I2C analog filter enable */
#define SYSCFG_CFGR1_I2C_EIIC_0                   (0x1UL << SYSCFG_CFGR1_I2C_EIIC_Pos)              /*!< 0x00010000 */
#define SYSCFG_CFGR1_I2C_EIIC_1                   (0x2UL << SYSCFG_CFGR1_I2C_EIIC_Pos)              /*!< 0x00020000 */
#define SYSCFG_CFGR1_I2C_EIIC_2                   (0x4UL << SYSCFG_CFGR1_I2C_EIIC_Pos)              /*!< 0x00040000 */
#define SYSCFG_CFGR1_I2C_EIIC_3                   (0x8UL << SYSCFG_CFGR1_I2C_EIIC_Pos)              /*!< 0x00080000 */
#define SYSCFG_CFGR1_I2C_EIIC_4                   (0x10UL << SYSCFG_CFGR1_I2C_EIIC_Pos)             /*!< 0x00100000 */
#define SYSCFG_CFGR1_I2C_EIIC_5                   (0x20UL << SYSCFG_CFGR1_I2C_EIIC_Pos)             /*!< 0x00200000 */
#define SYSCFG_CFGR1_I2C_EIIC_6                   (0x40UL << SYSCFG_CFGR1_I2C_EIIC_Pos)             /*!< 0x00400000 */
#define SYSCFG_CFGR1_I2C_EIIC_7                   (0x80UL << SYSCFG_CFGR1_I2C_EIIC_Pos)             /*!< 0x00800000 */
#define SYSCFG_CFGR1_I2C_EIIC_8                   (0x100UL << SYSCFG_CFGR1_I2C_EIIC_Pos)            /*!< 0x01000000 */
#define SYSCFG_CFGR1_I2C_EIIC_9                   (0x200UL << SYSCFG_CFGR1_I2C_EIIC_Pos)            /*!< 0x02000000 */
#define SYSCFG_CFGR1_I2C_EIIC_10                  (0x400UL << SYSCFG_CFGR1_I2C_EIIC_Pos)            /*!< 0x04000000 */
#define SYSCFG_CFGR1_I2C_EIIC_11                  (0x800UL << SYSCFG_CFGR1_I2C_EIIC_Pos)            /*!< 0x08000000 */
#define SYSCFG_CFGR1_I2C_EIIC_12                  (0x1000UL << SYSCFG_CFGR1_I2C_EIIC_Pos)           /*!< 0x10000000 */
#define SYSCFG_CFGR1_I2C_EIIC_13                  (0x2000UL << SYSCFG_CFGR1_I2C_EIIC_Pos)           /*!< 0x20000000 */
#define SYSCFG_CFGR1_I2C_EIIC_14                  (0x4000UL << SYSCFG_CFGR1_I2C_EIIC_Pos)           /*!< 0x40000000 */
#define SYSCFG_CFGR1_I2C_EIIC_15                  (0x8000UL << SYSCFG_CFGR1_I2C_EIIC_Pos)           /*!< 0x80000000 */
                                                                                                   
/********************************* Bit definition for SYSCFG_CFGR2 register *****************************************/
#define SYSCFG_CFGR2_CLL_Pos                      (0U)
#define SYSCFG_CFGR2_CLL_Msk                      (0x1UL<<SYSCFG_CFGR2_CLL_Pos)                     /*!< 0x00000001 */
#define SYSCFG_CFGR2_CLL                          SYSCFG_CFGR2_CLL_Msk                              /*!< core lockup enable */
#define SYSCFG_CFGR2_SPL_Pos                      (1U)
#define SYSCFG_CFGR2_SPL_Msk                      (0x1UL<<SYSCFG_CFGR2_SPL_Pos)                     /*!< 0x00000002 */
#define SYSCFG_CFGR2_SPL                          SYSCFG_CFGR2_SPL_Msk                              /*!< 